/*!
*******************************************************************************
* \file              arl_tclISourceFsm.cpp
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements the ISource FSM.
*  - COPYRIGHT:      &copy; 2012 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 27.09.2011 \version 1.4 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - GMNGA-16965 JIRA issue fix - Blockwidth which is expected to be allocated is
* returned based on number of audio channels (mono/stereo)
*
* \date 05.09.2011 \version 1.3 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - GMNGA-15409 JIRA issue fix - Number of audio channels to depend on the
* bandwidth which inturn depends on whether channel is mono or stereo.
* - GMNGA-15465 JIRA issue fix - Source activity error handling when source is
* not allocated.
*
* \date 25.08.2011 \version 1.2 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added handling for routing state and dynamic MOST channel deallocation &
* error handling.
* - GMNGA-14770 JIRA issue fix - Audio cue is no more relying on AV Manager
* service availability.
* - Additional error dispatching to application related to message post/service
* unavailability errors.
* - Correction in FSM for state machine mapping.
*
* \date 10.06.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Fixed for allocate/deallocate result to be send to client.
* - Added transitions from AV Request Activation/Deactivation states to Allocate
* & Deallocate respectively.
*
* \date 26.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include <generic_msgs_if.h>

#include "arl_tclISourceFsm.h"
#include "arl_tclBaseClienthandler.h"
#include "arl_tclISource.h"
#include "arl_LookupTable.h"
#include "arl_Trace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ARL_ISOURCE_FSM
#include "trcGenProj/Header/arl_tclISourceFsm.cpp.trc.h"
#endif

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/// State Name definitions.

#define ARL_STATE_INITIAL                    "State_Initial"
/// ISource ---- Allocate Route ----> Stream router.
#define ARL_STATE_STRROU_ALLOCATE            "State_StrRouAllocate"
/// ISource ---- DeAllocate Route ----> Stream router.
#define ARL_STATE_STRROU_DEALLOCATE          "State_StrRouDeAllocate"
/// ISource ---- SetPlay ----> Stream router.
#define ARL_STATE_STRROU_SETPLAY             "State_StrRouSetPlay"
/// ISource ---- SetStop ----> Stream router.
#define ARL_STATE_STRROU_SETSTOP             "State_StrRouSetStop"
/// ISource <---- Source Activity ---- AV Manager.
#define ARL_STATE_SRC_ACTIVITY               "State_SrcActivity"
/// ISource service unavailable.
#define ARL_STATE_SRV_UNAVAILABLE            "State_ISrcUnAvailable"
/// ISource service available.
#define ARL_STATE_SRV_AVAILABLE              "State_ISrcAvailable"
/// Wait confirmation of Application
#define ARL_STATE_APP_ACK_PLAY               "State_AppAckWait_Play"
/// Wait confirmation of Application
#define ARL_STATE_APP_ACK_STOPPAUSE          "State_AppAckWait_StopPause"


/// Trigger names

#define ARL_TRIG_SRC_ACTIVITY                "Trig_Activity"
#define ARL_TRIG_SRC_APP_ACK_PLAY            "Trig_AppAckPlay"
#define ARL_TRIG_SRC_APP_ACK_PAUSE           "Trig_AppAckPause"
#define ARL_TRIG_SRC_APP_ACK_STOP            "Trig_AppAckStop"

#define ARL_TRIG_STRROU_ALLOCATE             "Trig_StrRouAllocate"
#define ARL_TRIG_STRROU_ALLOCATE_MR          "Trig_StrRouAllocateMr"
#define ARL_TRIG_STRROU_ALLOCATE_ERR         "Trig_StrRouAllocateErr"

#define ARL_TRIG_STRROU_DEALLOCATE           "Trig_StrRouDeAllocate"
#define ARL_TRIG_STRROU_DEALLOCATE_MR        "Trig_StrRouDeAllocateMr"
#define ARL_TRIG_STRROU_DEALLOCATE_ERR       "Trig_StrRouDeAllocateErr"

#define ARL_TRIG_STRROU_SETPLAY              "Trig_StrRouSetPlay"
#define ARL_TRIG_STRROU_SETPLAY_MR           "Trig_StrRouSetPlayMr"
#define ARL_TRIG_STRROU_SETPLAY_ERR          "Trig_StrRouSetPlayErr"

#define ARL_TRIG_STRROU_SETSTOP              "Trig_StrRouSetStop"
#define ARL_TRIG_STRROU_SETSTOP_MR           "Trig_StrRouSetStopMr"
#define ARL_TRIG_STRROU_SETSTOP_ERR          "Trig_StrRouSetStopErr"

#define ARL_TRIG_MSG_POST_ERR                "Trig_MsgPostErr"
#define ARL_TRIG_MSG_POST_TIMEOUT            "Trig_MsgPostTimeOut"

#define ARL_TRIG_STRROU_AVAILABLE            "Trig_StrRouAvailable"
#define ARL_TRIG_STRROU_UNAVAILABLE          "Trig_StrRouUnavailable"

#define ARL_TRIG_APP_STATE_OFF               "Trig_AppState_Off"




/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

typedef arl_tclImplFunctor<arl_tclISourceFsm> arl_tISrcFsmFunctor;
typedef arl_tclImplFunctor<arl_tclISourceFsm> arl_tISrcFsmCondition;

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

/******************************************************************************
** FUNCTION:  arl_tclISourceFsm::arl_tclISourceFsm(arl_tclISource* const ...
******************************************************************************/

arl_tclISourceFsm::arl_tclISourceFsm // for ASF
(
   tU8 u8SrcNum
   , arl_tclISourceIF* const cpoISource
   , arl_tclStrClientHandler_ASF_IF* const cpoStrRouClient
   , arl_tclBaseClienthandler* const cpoAVManClient
):arl_tclFsm(), m_u8SrcNum(u8SrcNum), m_enAllocateOnStart(ARL_EN_ROUTE_STATE_NO_ALLOC), m_bAllocated(FALSE), m_enRouteState()
 , m_oSrcActivity(), m_hTimer(OSAL_C_INVALID_HANDLE),m_bLastAllocReq(FALSE), m_oLastSrcActivReq(), m_oISrcProp()
 , m_cpoISource(cpoISource)

 ,m_cpoStrRouClient(NULL)
 ,m_cpoStrRouClientAIL(NULL)
,m_cpoStrRouClientASF(cpoStrRouClient)
 ,m_cpoAVManClient(cpoAVManClient), m_oStateId(), m_oTrigId(), m_oFunctList(), m_u16SubSource(0), m_u16Sink(1), m_u8PosNextSrc(0)
{
   /// \note
   /// - Assert if ISource or Stream router client pointers are NULL.
   /// - AV Manager client can be NULL in case the ISource is not self triggered.
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoISource);
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoStrRouClientASF);

   // SourceActivity is initilized with ON as default, this need to be set explicitly
   m_oSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;
   m_oLastAppAckSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;

   if(m_cpoISource!=NULL)
   {
      ETG_TRACE_COMP(("[%d]Creating arl_tclISourceFsm for Source Num: %d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }

   if (OSAL_NULL != m_cpoISource)
   {
      // Get the Appid
      tU16 u16AppId  =  m_cpoISource->u16GetAppID();

      // Get the Source properties from ISource config table.
      if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcNum, m_oISrcProp))
      {
         // Source properties retrieved, setup the FSM.
         vSetupFsm();

         // Set Allocate on Startup
         if(TRUE == m_oISrcProp.m_bAllocOnStartUp)
         {
             m_enAllocateOnStart = ARL_EN_ROUTE_STATE_REQ_ALLOCATE;

         }//if(TRUE == m_oISrcProp.m_bAllocOnStartUp)

         // Finalize the FSM
         // First call to the FSM finalization, hence no neet to check return
         // value.
         (tVoid)bFsmFinalize();

      }  // if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcNum, m_oISrcProp))
      else
      {
         // 99.99% should not reach here.
         // If it does there is an ERROR in the Lookup Table logic.
         NORMAL_M_ASSERT_ALWAYS();

      }  // End of if-else; if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcN..

   }  // if (OSAL_NULL != m_cpoISource)

}  // arl_tclISourceFsm::arl_tclISourceFsm(tU8 u8SrcNum, arl_tclISource *c...


arl_tclISourceFsm::arl_tclISourceFsm //AHL
(
   tU8 u8SrcNum
   , arl_tclISourceIF* const cpoISource
   , arl_tclBaseClienthandler* const cpoStrRouClient
   , arl_tclBaseClienthandler* const cpoAVManClient
):arl_tclFsm(), m_u8SrcNum(u8SrcNum), m_enAllocateOnStart(ARL_EN_ROUTE_STATE_NO_ALLOC), m_bAllocated(FALSE), m_enRouteState()
, m_oSrcActivity(), m_hTimer(OSAL_C_INVALID_HANDLE),m_bLastAllocReq(FALSE), m_oLastSrcActivReq(), m_oISrcProp()
, m_cpoISource(cpoISource)
, m_cpoStrRouClient(cpoStrRouClient)
, m_cpoStrRouClientAIL(NULL)
, m_cpoStrRouClientASF(NULL)
, m_cpoAVManClient(cpoAVManClient), m_oStateId(), m_oTrigId(), m_oFunctList(), m_u16SubSource(0), m_u16Sink(1), m_u8PosNextSrc(0)
{
   /// \note
   /// - Assert if ISource or Stream router client pointers are NULL.
   /// - AV Manager client can be NULL in case the ISource is not self triggered.
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoISource);
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoStrRouClient);

   // SourceActivity is initilized with ON as default, this need to be set explicitly
   m_oSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;
   m_oLastAppAckSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;

   if(m_cpoISource!=NULL)
   {
      ETG_TRACE_COMP(("[%d]Creating arl_tclISourceFsm for Source Num: %d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }

   if (OSAL_NULL != m_cpoISource)
   {
      // Get the Appid
      tU16 u16AppId  =  m_cpoISource->u16GetAppID();

      // Get the Source properties from ISource config table.
      if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcNum, m_oISrcProp))
      {
         // Source properties retrieved, setup the FSM.
         vSetupFsm();

         // Set Allocate on Startup
         if(TRUE == m_oISrcProp.m_bAllocOnStartUp)
         {
             m_enAllocateOnStart = ARL_EN_ROUTE_STATE_REQ_ALLOCATE;

         }//if(TRUE == m_oISrcProp.m_bAllocOnStartUp)

         // Finalize the FSM
         // First call to the FSM finalization, hence no neet to check return
         // value.
         (tVoid)bFsmFinalize();

      }  // if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcNum, m_oISrcProp))
      else
      {
         // 99.99% should not reach here.
         // If it does there is an ERROR in the Lookup Table logic.
         NORMAL_M_ASSERT_ALWAYS();

      }  // End of if-else; if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcN..

   }  // if (OSAL_NULL != m_cpoISource)

}  // arl_tclISourceFsm::arl_tclISourceFsm(tU8 u8SrcNum, arl_tclISource *c...



arl_tclISourceFsm::arl_tclISourceFsm //AIL
(
   tU8 u8SrcNum
   , arl_tclISourceIF* const cpoISource
   , arl_tclBaseClienthandlerAIL* const cpoStrRouClient
   , arl_tclBaseClienthandler* const cpoAVManClient
):arl_tclFsm(), m_u8SrcNum(u8SrcNum), m_enAllocateOnStart(ARL_EN_ROUTE_STATE_NO_ALLOC), m_bAllocated(FALSE), m_enRouteState()
, m_oSrcActivity(), m_hTimer(OSAL_C_INVALID_HANDLE),m_bLastAllocReq(FALSE), m_oLastSrcActivReq(), m_oISrcProp()
, m_cpoISource(cpoISource)
, m_cpoStrRouClient(NULL)
, m_cpoStrRouClientAIL(cpoStrRouClient)
, m_cpoStrRouClientASF(NULL)
, m_cpoAVManClient(cpoAVManClient), m_oStateId(), m_oTrigId(), m_oFunctList(), m_u16SubSource(0), m_u16Sink(1), m_u8PosNextSrc(0)
{
   /// \note
   /// - Assert if ISource or Stream router client pointers are NULL.
   /// - AV Manager client can be NULL in case the ISource is not self triggered.
   NORMAL_M_ASSERT(OSAL_NULL != m_cpoISource);


   // SourceActivity is initilized with ON as default, this need to be set explicitly
   m_oSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;
   m_oLastAppAckSrcActivity.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;

   if(m_cpoISource != NULL)
   {
      ETG_TRACE_COMP(("[%d]Creating arl_tclISourceFsm for Source Num: %d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }

   if (OSAL_NULL != m_cpoISource)
   {
      // Get the Appid
      tU16 u16AppId  =  m_cpoISource->u16GetAppID();

      // Get the Source properties from ISource config table.
      if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcNum, m_oISrcProp))
      {
         // Source properties retrieved, setup the FSM.
         vSetupFsm();

         // Set Allocate on Startup
         if(TRUE == m_oISrcProp.m_bAllocOnStartUp)
         {
             m_enAllocateOnStart = ARL_EN_ROUTE_STATE_REQ_ALLOCATE;
             if(OSAL_OK != OSAL_s32TimerCreate((OSAL_tpfCallback)vHandleTimerCallBack,
                                               (tPVoid)this,&m_hTimer))
             {
                 ETG_TRACE_ERR(("ARL Timer Creation Failed."));
             }//if(OSAL_OK != OSAL_s32TimerCreate(...)
         }//if(TRUE == m_oISrcProp.m_bAllocOnStartUp)

         // Finalize the FSM
         // First call to the FSM finalization, hence no neet to check return
         // value.
         (tVoid)bFsmFinalize();

      }  // if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcNum, m_oISrcProp))
      else
      {
         // 99.99% should not reach here.
         // If it does there is an ERROR in the Lookup Table logic.
         NORMAL_M_ASSERT_ALWAYS();

      }  // End of if-else; if (FALSE != arl_bGetISrcProp(u16AppId, m_u8SrcN..

   }  // if (OSAL_NULL != m_cpoISource)

}  // arl_tclISourceFsm::arl_tclISourceFsm(tU8 u8SrcNum, arl_tclISource *c...

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnSrcActTimerExpiry()
******************************************************************************/
tVoid arl_tclISourceFsm::vOnSrcActTimerExpiry()
{
  //Get state
  tS16 actState = s16GetCurrentState();

  //take care for states individually
  if(m_cpoISource != NULL)
  {
     ETG_TRACE_FATAL(("[%d] vHandleTimerCallBack: Timeout for Source: %d, State: %d"
              ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
              ,ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(m_u8SrcNum))
              ,actState));
  }else{
     ETG_TRACE_FATAL(("[NULL] vHandleTimerCallBack: Timeout for Source: %d, State: %d"
                    ,ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(m_u8SrcNum))
                    ,actState));
    }

  if((actState == m_oStateId.m_s16AppAckActivityPlay)
           ||(actState == m_oStateId.m_s16AppAckActivityStopPause))
  {
     bOnSrcActivityErr();
  }
}  // tVoid arl_tclISourceFsm::vOnSrcActTimerExpiry()

/******************************************************************************
** FUNCTION:  arl_tclISourceFsm::vHandleTimerCallBack(tPVoid pvArg)
******************************************************************************/
tVoid arl_tclISourceFsm::vHandleTimerCallBack(tPVoid pvArg)
{
  arl_tclISourceFsm* pISource =  (arl_tclISourceFsm*)pvArg;
  if(pISource == NULL)
  {
    return;
  }


  //Get state
  tS16 actState = pISource->s16GetCurrentState();
  //take care for states individually
  if(pISource->m_cpoISource != NULL)
  {
    // Dispatch SrcActTimerExpiry function on timer call back
    // Fake the incoming service data.
    gm_tclEmptyMessage oFakeServiceData
    (
        pISource->m_cpoISource->u16GetAppID()             // Source App id -> From?
        , pISource->m_cpoISource->u16GetAppID()             // Target App Id -> Self
        , 0            // Register Id
        , 0            // Command counter
        , pISource->m_cpoISource->u16GetServiceID()         // Service ID
        , ARL_EN_SRCACTTIMEREXPIRY            // Function Id
        , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
    );

    // Send the response to the source activity.
    arl_tSrcActTimerExpiryStart oISrcActTimerExpiryStart;
    oISrcActTimerExpiryStart.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(pISource->m_u8SrcNum);

    ETG_TRACE_USR1(("[%d]Sending SrcActTimerExpiry method start for Source Num:%d."
        , ETG_ENUM(ail_u16AppId, pISource->m_cpoISource->u16GetAppID())
        , pISource->m_u8SrcNum));

    ETG_TRACE_FATAL(("[%d] vHandleTimerCallBack: Timeout for Source: %d, State: %d"
        ,ETG_ENUM(ail_u16AppId,pISource->m_cpoISource->u16GetAppID())
        ,ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(pISource->m_u8SrcNum))
        ,actState));

    pISource->m_cpoISource->bSendMessage(oFakeServiceData, oISrcActTimerExpiryStart);
    // Delete shared memory when going out of scope.
    (tVoid)oFakeServiceData.bDelete();
  }else{
    ETG_TRACE_FATAL(("[NULL] vHandleTimerCallBack: Timeout for Source: %d, State: %d"
        ,ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(pISource->m_u8SrcNum))
        ,actState));
  }
#if 0
  if((actState == pISource->m_oStateId.m_s16AppAckActivityPlay)
      ||(actState == pISource->m_oStateId.m_s16AppAckActivityStopPause))
  {
    pISource->bOnSrcActivityErr();
  }
#endif

}//tVoid arl_tclISourceFsm::vHandleTimerCallBack(tPVoid pvArg)

/******************************************************************************
** FUNCTION:  virtual arl_tclISourceFsm::~arl_tclISourceFsm()
******************************************************************************/

/*virtual*/
arl_tclISourceFsm::~arl_tclISourceFsm()
{
   try
   {
      if(m_cpoISource != NULL)
      {
         ETG_TRACE_COMP(("[%d]Destroying arl_tclISourceFsm for Source Num: %d"
               ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID()), m_u8SrcNum));
      }
      // if non empty, free the resources from the list and clear the list.
      for(arl_tFunctorListIter iterFuncList = m_oFunctList.begin();
         iterFuncList != m_oFunctList.end(); ++iterFuncList)
      {
         // Delete the object now.
         OSAL_DELETE *iterFuncList;
         *iterFuncList = OSAL_NULL;
      }  // for(uam_tDapiPoiMsgListIter iterFuncList = m_oFunctList.begin..
      m_oFunctList.clear();
      // Stop and delete all Timer
      if(m_cpoISource != NULL)
      {
         ETG_TRACE_COMP(("[%d]Stop and delete Timers"
               ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
      }
      bTimerStop();
      OSAL_s32TimerDelete(m_hTimer);
   }
   catch(...)
   {
   }
}  // arl_tclISourceFsm::~arl_tclISourceFsm()

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vSetupFsm()
******************************************************************************/

tVoid arl_tclISourceFsm::vSetupFsm()
{
   ETG_TRACE_USR1(("Setting up the FSM"));

   ///! \brief
   ///  Create the State machine
   ///  - Create the initial state or super state.

   m_oStateId.m_s16Super            =  s16AddState(ARL_STATE_INITIAL);

   /// \brief
   /// Register all the sub states to the FSM.

   m_oStateId.m_s16SrvUnavail       =  s16AddState(ARL_STATE_SRV_UNAVAILABLE
                                          , m_oStateId.m_s16Super);

   m_oStateId.m_s16SrvAvail         =  s16AddState(ARL_STATE_SRV_AVAILABLE
                                          , m_oStateId.m_s16Super);

   m_oStateId.m_s16StrRouAlloc      =  s16AddState(ARL_STATE_STRROU_ALLOCATE
                                          , m_oStateId.m_s16SrvAvail);

   m_oStateId.m_s16StrRouDeAlloc    =  s16AddState(ARL_STATE_STRROU_DEALLOCATE
                                          , m_oStateId.m_s16SrvAvail);

   m_oStateId.m_s16SrcAct           =  s16AddState(ARL_STATE_SRC_ACTIVITY
                                          , m_oStateId.m_s16SrvAvail);

   m_oStateId.m_s16StrRouSetPlay    =  s16AddState(ARL_STATE_STRROU_SETPLAY
                                          , m_oStateId.m_s16SrcAct);

   m_oStateId.m_s16StrRouSetStop    =  s16AddState(ARL_STATE_STRROU_SETSTOP
                                          , m_oStateId.m_s16SrcAct);

   m_oStateId.m_s16AppAckActivityPlay =  s16AddState(ARL_STATE_APP_ACK_PLAY
                                          , m_oStateId.m_s16SrcAct);

   m_oStateId.m_s16AppAckActivityStopPause =  s16AddState(ARL_STATE_APP_ACK_STOPPAUSE
                                          , m_oStateId.m_s16SrcAct);


   ///! \brief
   ///  Register Entry functions for each state.

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bOnStrRouUnavailable));
      (tVoid)bAdd(m_oStateId.m_s16SrvUnavail, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendAllocRouteOnStartUp));
   (tVoid)bAdd(m_oStateId.m_s16SrvAvail, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendAllocRoute));
   (tVoid)bAdd(m_oStateId.m_s16StrRouAlloc, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendDeAllocRoute));
   (tVoid)bAdd(m_oStateId.m_s16StrRouDeAlloc, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bOnSrcActivity));
   (tVoid)bAdd(m_oStateId.m_s16SrcAct, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendSetPlay));
   (tVoid)bAdd(m_oStateId.m_s16StrRouSetPlay, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendSetStop));
   (tVoid)bAdd(m_oStateId.m_s16StrRouSetStop, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendSrcActToApp));
      (tVoid)bAdd(m_oStateId.m_s16AppAckActivityPlay, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendSrcActToApp));
      (tVoid)bAdd(m_oStateId.m_s16AppAckActivityStopPause, m_oFunctList.back());

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendSrcActivResult));
   (tVoid)bAdd(m_oStateId.m_s16AppAckActivityPlay, m_oFunctList.back(), ARL_ON_STATE_EXIT);

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bSendSrcActivResult));
      (tVoid)bAdd(m_oStateId.m_s16StrRouSetStop, m_oFunctList.back(), ARL_ON_STATE_EXIT);

   m_oFunctList.push_back(OSAL_NEW arl_tISrcFsmFunctor(this, &arl_tclISourceFsm::bTimerStop));
         (tVoid)bAdd(m_oStateId.m_s16AppAckActivityStopPause, m_oFunctList.back(), ARL_ON_STATE_EXIT);



   ///! \brief
   ///  Register triggers from a state to different state.

   m_oTrigId.m_s16SrcActivity          =  s16AddTrigger(ARL_TRIG_SRC_ACTIVITY);
   m_oTrigId.m_s16AppAckActivityPause  =  s16AddTrigger(ARL_TRIG_SRC_APP_ACK_PAUSE);
   m_oTrigId.m_s16AppAckActivityStop   =  s16AddTrigger(ARL_TRIG_SRC_APP_ACK_STOP);
   m_oTrigId.m_s16AppAckActivityPlay   =  s16AddTrigger(ARL_TRIG_SRC_APP_ACK_PLAY);

   m_oTrigId.m_s16StrRouAvail          =  s16AddTrigger(ARL_TRIG_STRROU_AVAILABLE);
   m_oTrigId.m_s16StrRouUnavail        =  s16AddTrigger(ARL_TRIG_STRROU_UNAVAILABLE);

   m_oTrigId.m_s16StrRouAllocate       =  s16AddTrigger(ARL_TRIG_STRROU_ALLOCATE);
   m_oTrigId.m_s16StrRouAllocateMr     =  s16AddTrigger(ARL_TRIG_STRROU_ALLOCATE_MR);

   m_oTrigId.m_s16StrRouDeAllocate     =  s16AddTrigger(ARL_TRIG_STRROU_DEALLOCATE);
   m_oTrigId.m_s16StrRouDeAllocateMr   =  s16AddTrigger(ARL_TRIG_STRROU_DEALLOCATE_MR);

   m_oTrigId.m_s16StrRouSetPlay        =  s16AddTrigger(ARL_TRIG_STRROU_SETPLAY);
   m_oTrigId.m_s16StrRouSetPlayMr      =  s16AddTrigger(ARL_TRIG_STRROU_SETPLAY_MR);

   m_oTrigId.m_s16StrRouSetStop        =  s16AddTrigger(ARL_TRIG_STRROU_SETSTOP);
   m_oTrigId.m_s16StrRouSetStopMr      =  s16AddTrigger(ARL_TRIG_STRROU_SETSTOP_MR);

   m_oTrigId.m_s16StrRouAllocateErr    =  s16AddTrigger(ARL_TRIG_STRROU_ALLOCATE_ERR);
   m_oTrigId.m_s16StrRouDeAllocateErr  =  s16AddTrigger(ARL_TRIG_STRROU_DEALLOCATE_ERR);
   m_oTrigId.m_s16StrRouSetPlayErr     =  s16AddTrigger(ARL_TRIG_STRROU_SETPLAY_ERR);
   m_oTrigId.m_s16StrRouSetStopErr     =  s16AddTrigger(ARL_TRIG_STRROU_SETSTOP_ERR);
   m_oTrigId.m_s16MsgPostErr           =  s16AddTrigger(ARL_TRIG_MSG_POST_ERR);
   m_oTrigId.m_s16MsgPostTimeOut       =  s16AddTrigger(ARL_TRIG_MSG_POST_TIMEOUT);
   m_oTrigId.m_s16AppState_Off         =  s16AddTrigger(ARL_TRIG_APP_STATE_OFF);


   /// \brief
   /// Register transitions

   (tVoid)poAddTransition(m_oStateId.m_s16Super, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouAvail);

   (tVoid)poAddTransition(m_oStateId.m_s16Super, m_oStateId.m_s16SrvUnavail
      , m_oTrigId.m_s16StrRouUnavail);

   (tVoid)poAddTransition(m_oStateId.m_s16SrvAvail, m_oStateId.m_s16StrRouAlloc
      , m_oTrigId.m_s16StrRouAllocate);

   (tVoid)poAddTransition(m_oStateId.m_s16SrvAvail, m_oStateId.m_s16SrcAct
   , m_oTrigId.m_s16SrcActivity);

   (tVoid)poAddTransition(m_oStateId.m_s16SrvAvail, m_oStateId.m_s16StrRouDeAlloc
      , m_oTrigId.m_s16StrRouDeAllocate);

   (tVoid)poAddTransition(m_oStateId.m_s16StrRouAlloc, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouAllocateMr);

   (tVoid)poAddTransition(m_oStateId.m_s16StrRouDeAlloc, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouDeAllocateMr);

   (tVoid)poAddTransition(m_oStateId.m_s16SrcAct, m_oStateId.m_s16StrRouSetPlay
      , m_oTrigId.m_s16StrRouSetPlay);

   (tVoid)poAddTransition(m_oStateId.m_s16StrRouSetPlay, m_oStateId.m_s16AppAckActivityPlay
      , m_oTrigId.m_s16StrRouSetPlayMr);

   (tVoid)poAddTransition(m_oStateId.m_s16StrRouSetStop, m_oStateId.m_s16SrvAvail
         , m_oTrigId.m_s16StrRouSetStopMr);

   (tVoid)poAddTransition(m_oStateId.m_s16SrcAct, m_oStateId.m_s16AppAckActivityStopPause
      , m_oTrigId.m_s16StrRouSetStop);

   (tVoid)poAddTransition(m_oStateId.m_s16AppAckActivityPlay, m_oStateId.m_s16SrvAvail
               , m_oTrigId.m_s16AppAckActivityPlay);

   (tVoid)poAddTransition(m_oStateId.m_s16AppAckActivityPlay, m_oStateId.m_s16StrRouSetStop
                  , m_oTrigId.m_s16MsgPostTimeOut);

   arl_tclFsmTransition* poTransition = poAddTransition(m_oStateId.m_s16AppAckActivityStopPause, m_oStateId.m_s16StrRouSetStop
                     , m_oTrigId.m_s16AppAckActivityPause);
   ///Adding a condition to the above transition
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bValidateStateExit_AppAckActivityStopPause));

   poTransition = poAddTransition(m_oStateId.m_s16AppAckActivityStopPause, m_oStateId.m_s16StrRouSetStop
                  , m_oTrigId.m_s16AppAckActivityStop);
   ///Adding a condition to the above transition
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bValidateStateExit_AppAckActivityStopPause));


   (tVoid)poAddTransition(m_oStateId.m_s16AppAckActivityStopPause, m_oStateId.m_s16StrRouSetStop
                        , m_oTrigId.m_s16MsgPostTimeOut);
   //Condition not required for the above mentioned transition

   //Add conditional Transitions
   poTransition =
      poAddTransition(m_oStateId.m_s16StrRouAlloc, m_oStateId.m_s16SrvAvail
         , m_oTrigId.m_s16MsgPostErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnAllocErr));

   poTransition = poAddTransition(m_oStateId.m_s16SrvAvail, m_oStateId.m_s16SrvAvail
         , m_oTrigId.m_s16AppState_Off);
      (tVoid)bAddCondition(poTransition
            , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnAppStateOff));

//   poTransition = poAddTransition(m_oStateId.m_s16ISrcAlloc, m_oStateId.m_s16SrvAvail
//         , m_oTrigId.m_s16StrRouAllocateErr);
//   (tVoid)bAddCondition(poTransition
//         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnAllocErr));

   poTransition = poAddTransition(m_oStateId.m_s16SrcAct, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16MsgPostErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnSrcActivityErr));

   poTransition =  poAddTransition(m_oStateId.m_s16StrRouSetPlay, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouSetPlayErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnSrcActivityErr));

   poTransition =  poAddTransition(m_oStateId.m_s16StrRouSetStop, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouSetStopErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnSrcActivityErr));

   poTransition =  poAddTransition(m_oStateId.m_s16StrRouDeAlloc, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouDeAllocateErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnDeAllocErr));

   poTransition =  poAddTransition(m_oStateId.m_s16StrRouDeAlloc, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16MsgPostErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnDeAllocErr));

   poTransition =  poAddTransition(m_oStateId.m_s16StrRouAlloc, m_oStateId.m_s16SrvAvail
      , m_oTrigId.m_s16StrRouAllocateErr);
   (tVoid)bAddCondition(poTransition
         , OSAL_NEW arl_tISrcFsmCondition(this, &arl_tclISourceFsm::bOnAllocErr));

   (tVoid)bSetInitState(m_oStateId.m_s16Super);

}  // tVoid arl_tclISourceFsm::vSetupFsm()

/******************************************************************************
** FUNCTION:  tBool arl_tclISourceFsm::bSendStrRouMsg(const fi_tclMessageB..
******************************************************************************/

tBool arl_tclISourceFsm::bSendStrRouMsg(const fi_tclMessageBase &rfcoMsgBase)
{
   tBool bRetVal  =  FALSE;
   tU8 u8SrcNum = m_u8SrcNum;

   // For Route Allocation on Startup, use a mask to distinguish from the normal
   // Allocate command.This way, the response can be handled appropriately.
   if((ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)||
       (ARL_EN_ROUTE_STATE_REQ_DEALLOC == m_enAllocateOnStart))
   {
       u8SrcNum = (ARL_MASK_ROUALLOC_STARTUP | m_u8SrcNum);
   }//if(TRUE == m_enAllocateOnStart)

   if (OSAL_NULL != m_cpoStrRouClient)
   {
      // Send the message by embedding the source number in the command counter
      // This will be used to validate the message recieved from Stream router.
      bRetVal = m_cpoStrRouClient->bSendMessage(rfcoMsgBase, u8SrcNum);

      if (FALSE == bRetVal)
      {
         /// Trigger to FSM that Message post failed.
         (*this)(m_oTrigId.m_s16MsgPostErr);
      }
   }
   if (OSAL_NULL != m_cpoStrRouClientAIL)
   {
      // Send the message by embedding the source number in the command counter
      // This will be used to validate the message recieved from Stream router.
      bRetVal = m_cpoStrRouClientAIL->bSendMessage(rfcoMsgBase, u8SrcNum);

      if (FALSE == bRetVal)
      {
         /// Trigger to FSM that Message post failed.
         (*this)(m_oTrigId.m_s16MsgPostErr);
      }
   }

   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending stream router message returned status:%d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(ARL_STATUS, bRetVal)));
   }
   return bRetVal;
}  // tBool arl_tclISourceFsm::bSendStrRouMsg(const fi_tclMessageBase &rfco..


// For ASF we need to do it different
tBool arl_tclISourceFsm::bSendStrRouMsgAllocateRoute(tU16 ucid, tS32 STM_MsgBufID, tU16 u16Sink)
{
  // forward to ASH handler
  tBool bRetVal  =  FALSE;
  tU8 u8SrcNum = m_u8SrcNum;

  // For Route Allocation on Startup, use a mask to distinguish from the normal
  // Allocate command.This way, the response can be handled appropriately.
  if((ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)||
      (ARL_EN_ROUTE_STATE_REQ_DEALLOC == m_enAllocateOnStart))
  {
    u8SrcNum = (ARL_MASK_ROUALLOC_STARTUP | m_u8SrcNum);
  }//if(TRUE == m_enAllocateOnStart)

  if (OSAL_NULL != m_cpoStrRouClientASF)
  {
    // Send the message by embedding the source number in the command counter
    // This will be used to validate the message recieved from Stream router.
    //bRetVal = m_cpoStrRouClientASF->bSendMessage(rfcoMsgBase, u8SrcNum);
    bRetVal = m_cpoStrRouClientASF->bSendAllocateRoute(ucid, STM_MsgBufID, u8SrcNum, u16Sink);

    if (FALSE == bRetVal)
    {
      /// Trigger to FSM that Message post failed.
      (*this)(m_oTrigId.m_s16MsgPostErr);
    }
  }

  if(m_cpoISource != NULL)
  {
    ETG_TRACE_USR1(("[%d]Sending stream router Allocate message via ASF returned status:%d"
        , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
        , ETG_ENUM(ARL_STATUS, bRetVal)));
  }
  return bRetVal;

}

tBool arl_tclISourceFsm::bSendStrRouMsgDeAllocateRoute(tU16 ucid)
{
  // create the ASF and use ASF handler
  // forward to ASH handler
  tBool bRetVal  =  FALSE;
  tU8 u8SrcNum = m_u8SrcNum;

  // For Route Allocation on Startup, use a mask to distinguish from the normal
  // Allocate command.This way, the response can be handled appropriately.
  if((ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)||
      (ARL_EN_ROUTE_STATE_REQ_DEALLOC == m_enAllocateOnStart))
  {
    u8SrcNum = (ARL_MASK_ROUALLOC_STARTUP | m_u8SrcNum);
  }//if(TRUE == m_enAllocateOnStart)

  if (OSAL_NULL != m_cpoStrRouClientASF)
  {
    // Send the message by embedding the source number in the command counter
    // This will be used to validate the message recieved from Stream router.
    //bRetVal = m_cpoStrRouClientASF->bSendMessage(rfcoMsgBase, u8SrcNum);
    bRetVal = m_cpoStrRouClientASF->bSendDeAllocateRoute(ucid, u8SrcNum);

    if (FALSE == bRetVal)
    {
      /// Trigger to FSM that Message post failed.
      (*this)(m_oTrigId.m_s16MsgPostErr);
    }
  }

  if(m_cpoISource != NULL)
  {
    ETG_TRACE_USR1(("[%d]Sending stream router Deallocate message via ASF returned status:%d"
        , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
        , ETG_ENUM(ARL_STATUS, bRetVal)));
  }
  return bRetVal;

}

tBool arl_tclISourceFsm::bSendStrRouMsgSetPlay(tU16 ucid)
{
  // forward to ASH handler
  tBool bRetVal  =  FALSE;
  tU8 u8SrcNum = m_u8SrcNum;

  // For Route Allocation on Startup, use a mask to distinguish from the normal
  // Allocate command.This way, the response can be handled appropriately.
  if((ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)||
      (ARL_EN_ROUTE_STATE_REQ_DEALLOC == m_enAllocateOnStart))
  {
    u8SrcNum = (ARL_MASK_ROUALLOC_STARTUP | m_u8SrcNum);
  }//if(TRUE == m_enAllocateOnStart)

  if (OSAL_NULL != m_cpoStrRouClientASF)
  {
    // Send the message by embedding the source number in the command counter
    // This will be used to validate the message received from Stream router.
    //bRetVal = m_cpoStrRouClientASF->bSendMessage(rfcoMsgBase, u8SrcNum);
    bRetVal = m_cpoStrRouClientASF->bSendSetPlay(ucid, u8SrcNum);

    if (FALSE == bRetVal)
    {
      /// Trigger to FSM that Message post failed.
      (*this)(m_oTrigId.m_s16MsgPostErr);
    }
  }

  if(m_cpoISource != NULL)
  {
    ETG_TRACE_USR1(("[%d]Sending stream router Play message via ASF returned status:%d"
        , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
        , ETG_ENUM(ARL_STATUS, bRetVal)));
  }
  return bRetVal;
}

tBool arl_tclISourceFsm::bSendStrRouMsgSetStop(tU16 ucid)
{
  // forward to ASH handler
  tBool bRetVal  =  FALSE;
  tU8 u8SrcNum = m_u8SrcNum;

  // For Route Allocation on Startup, use a mask to distinguish from the normal
  // Allocate command.This way, the response can be handled appropriately.
  if((ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)||
      (ARL_EN_ROUTE_STATE_REQ_DEALLOC == m_enAllocateOnStart))
  {
    u8SrcNum = (ARL_MASK_ROUALLOC_STARTUP | m_u8SrcNum);
  }//if(TRUE == m_enAllocateOnStart)

  if (OSAL_NULL != m_cpoStrRouClientASF)
  {
    // Send the message by embedding the source number in the command counter
    // This will be used to validate the message recieved from Stream router.
    //bRetVal = m_cpoStrRouClientASF->bSendMessage(rfcoMsgBase, u8SrcNum);
    bRetVal = m_cpoStrRouClientASF->bSendSetStop(ucid, u8SrcNum);

    if (FALSE == bRetVal)
    {
      /// Trigger to FSM that Message post failed.
      (*this)(m_oTrigId.m_s16MsgPostErr);
    }
  }

  if(m_cpoISource != NULL)
  {
    ETG_TRACE_USR1(("[%d]Sending stream router Stop message via ASF returned status:%d"
        , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
        , ETG_ENUM(ARL_STATUS, bRetVal)));
  }
  return bRetVal;
}


/******************************************************************************
** FUNCTION:  tBool arl_tclISourceFsm::bSendAVManMsg(const fi_tclMessageB..
******************************************************************************/

tBool arl_tclISourceFsm::bSendAVManMsg(const fi_tclMessageBase &rfcoMsgBase)
{
   tBool bRetVal  =  FALSE;

   if (OSAL_NULL != m_cpoAVManClient)
   {
      // Send the message by embedding the source number in the command counter
      // This will be used to distinguish an error message recieved from AV Manager.
      bRetVal = m_cpoAVManClient->bSendMessage(rfcoMsgBase, m_u8SrcNum);

      if (FALSE == bRetVal)
      {
         /// Trigger to FSM that Message post failed.
         (*this)(m_oTrigId.m_s16MsgPostErr);
      }
   }
   else
   {
      // Not a self triggered source? If ya, fake the response to TRUE.
      bRetVal  =  (!m_oISrcProp.m_bIsSelfTrig);
   }

   ETG_TRACE_USR1(("Sending AV Manager message returned status:%d"
      , ETG_ENUM(ARL_STATUS, bRetVal)));

   return bRetVal;
}  // tBool arl_tclISourceFsm::bSendAVManMsg(const fi_tclMessageBase &rfcoMsgB..

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAllocate()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bOnStrRouUnavailable()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR4(("[%d] bOnStrRouUnavailable"
            ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
   }

   if(m_cpoISource != NULL)
   {
      m_oLastSrcActivReq.enType = arl_tSrcActivity::FI_EN_OFF;
      m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_oLastSrcActivReq);
      //daw2hi: second call was forgotten
      m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_u16SubSource, m_oLastSrcActivReq);

      m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_u16SubSource, static_cast<arl_tenSource>(m_u8PosNextSrc),  m_oLastSrcActivReq);

      m_bLastAllocReq = FALSE;
      m_bAllocated    = FALSE;
      m_cpoISource->bOnDeAllocate(static_cast<arl_tenSource>(m_u8SrcNum));
   }

   if(m_cpoISource != NULL)
   {
      m_cpoISource->vSetSrcAvailable(FALSE);
   }
   return TRUE;
}


/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAllocate()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bOnAllocate()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]OnAllocate/OnDeallocate, Last allocation request: %d"
            ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(ARL_ONALLOCATE, m_bLastAllocReq)));
   }

   // check for the latest request.
   if ((FALSE != m_bLastAllocReq) && (FALSE == m_bAllocated))
   {
      // If latest request is to do allocation trigger stream router allocation.
      (*this)(m_oTrigId.m_s16StrRouAllocate);
   }
   else if ((FALSE == m_bLastAllocReq) && (FALSE != m_bAllocated))
   {
      // If latest request is to deallocated trigger stream router deallocation.
      (*this)(m_oTrigId.m_s16StrRouDeAllocate);
   }
   else
   {
      // Nothing to do.
   }

   return TRUE;
}  // tBool arl_tclISourceFsm::bOnAllocate()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendAllocRouteOnStartUp()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendAllocRouteOnStartUp()
{
   if(m_cpoISource!= NULL){
      ETG_TRACE_USR4(("[%d]bSendAllocRouteOnStartUp(),Status: %d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            ,m_enAllocateOnStart));
   }
   else{
     ETG_TRACE_FATAL(("m_cpoISource is NULL"));
   }
    tBool bRetVal = TRUE;
    if(ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)
    {
        bRetVal = bSendAllocRoute();
        if((TRUE == bRetVal) && (OSAL_C_INVALID_HANDLE != m_hTimer))
        {
            if(OSAL_OK != OSAL_s32TimerSetTime(m_hTimer, ARL_STARTUP_DEALLOC_TIMER_DURATION, 0))
            {
                bRetVal = FALSE;
              if(m_cpoISource!= NULL) {
                ETG_TRACE_ERR(("[%d]Failed to Start Startup Route Allocate Timer"
                      , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
              }
            }//if(OSAL_OK != OSAL_s32TimerSetTime(...)
        }//if((TRUE == bRetVal) && (OSAL_C_INVALID_HANDLE != m_hTimer))
        else
        {
            m_enAllocateOnStart = ARL_EN_ROUTE_STATE_NO_ALLOC;
            if(OSAL_C_INVALID_HANDLE != m_hTimer)
            {
              if(m_cpoISource!= NULL) {
                ETG_TRACE_USR4(("[%d]Failed to Send Start Startup Route Allocate Message"
                    , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
              }
                OSAL_s32TimerDelete(m_hTimer);
                m_hTimer = OSAL_C_INVALID_HANDLE;
            }//if(OSAL_C_INVALID_HANDLE != m_hTimer)
        }// End of if-else; if((TRUE == bRetVal) && (OSAL_C_INVALID_HANDLE != m_hTimer))

    }//if(ARL_EN_ROUTE_STATE_REQ_ALLOCATE == m_enAllocateOnStart)
    return bRetVal;
}

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendAllocRoute()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendAllocRoute()
{
    tBool bRetVal = FALSE;

    if(ARL_EN_ROUTE_STATE_ALLOCATED == m_enAllocateOnStart)
    {
        m_enAllocateOnStart = ARL_EN_ROUTE_STATE_NO_ALLOC;
    }//if(ARL_EN_ROUTE_STATE_ALLOCATED == m_enAllocateOnStart)

   // If the application has setup the message buffer id, fill this from the
   // application using the callback functionality.
   if (OSAL_NULL != m_cpoISource)
   {
     tBool bServiceAvail =false;

      if(NULL != m_cpoStrRouClient)
      {
        bServiceAvail =m_cpoStrRouClient->bIfServiceAvailable();
      }
      if(NULL != m_cpoStrRouClientAIL)
      {
        bServiceAvail =m_cpoStrRouClientAIL->bIfServiceAvailable();
      }

      if(NULL != m_cpoStrRouClientASF)
      {
        bServiceAvail =m_cpoStrRouClientASF->bIfServiceAvailable();
      }

      if((true == bServiceAvail) &&(m_cpoStrRouClientASF != NULL))
      {
        // we use the ASF and provide the parameters UCID
        bRetVal = bSendStrRouMsgAllocateRoute(m_oISrcProp.m_u16Ucid,m_cpoISource->s32GetMsgBuffId(m_u8SrcNum),m_u16Sink);
      }
      else if (true == bServiceAvail)
      {
          // Prepare the Allocate route message.
          midw_streamrouterfi_tclMsgAllocateRouteMethodStart oMsgAllocRoute;
          oMsgAllocRoute.u16UCID  =  m_oISrcProp.m_u16Ucid;
          oMsgAllocRoute.u16Sink = m_u16Sink;  // ???? daw2hi 23.07.2018

          oMsgAllocRoute.s32STM_MsgBufID = m_cpoISource->s32GetMsgBuffId(m_u8SrcNum);
          ETG_TRACE_USR1(("[%d]Sending Allocate Route message to stream router\
          UCID:0x%x, SourceNum:%d, STM MsgBufId:%d, for Sink %d", ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
          , oMsgAllocRoute.u16UCID, m_u8SrcNum, oMsgAllocRoute.s32STM_MsgBufID,oMsgAllocRoute.u16Sink));

          // Trigger the sending of the stream router message.
          bRetVal = bSendStrRouMsg(oMsgAllocRoute);
      }
   }

   return bRetVal;
}  // tBool arl_tclISourceFsm::bSendAllocRoute()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendDeAllocRoute()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendDeAllocRoute()
{
   // If the source is still playing automatically trigger set stop & deallocate
   if (m_oSrcActivity.enType == arl_tSrcActivity::FI_EN_ON )
   {
      (tVoid)bSendSetStop();

      ETG_TRACE_USR1(("[%d]Sending Stop message to stream router before deallocate"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
   }

   // Prepare the deallocate route message
   arl_tDeAllocRouteStart oMsgDeAllocRoute;
   oMsgDeAllocRoute.u16UCID   =  m_oISrcProp.m_u16Ucid;

   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending Deallocate Route message to stream router\
      UCID:0x%x, SourceNum:%d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , oMsgDeAllocRoute.u16UCID
            , m_u8SrcNum));
   }

   tBool bRetVal;
   if(m_cpoStrRouClientASF != NULL)
   {
     // Trigger the sending of the stream router message via ASF.
     bRetVal = bSendStrRouMsgDeAllocateRoute(m_oISrcProp.m_u16Ucid);
   }
   else
   {
     // Trigger the sending of the stream router message.
     bRetVal = bSendStrRouMsg(oMsgDeAllocRoute);
   }
   if(TRUE == bRetVal)
   {
       m_enAllocateOnStart = ARL_EN_ROUTE_STATE_NO_ALLOC;
   }//if(TRUE == bRetVal)

   return bRetVal;
}  // tBool arl_tclISourceFsm::bSendDeAllocRoute()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendSetPlay()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendSetPlay()
{
   // Prepare the set play message
   arl_tSetPlayStart oMsgSetPlay;
   oMsgSetPlay.u16UCID  =  m_oISrcProp.m_u16Ucid;

   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending Set Play message to stream router\
      UCID:0x%x, SourceNum:%d"
            ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , oMsgSetPlay.u16UCID
            , m_u8SrcNum));
   }
   if(m_cpoStrRouClientASF != NULL)
   {
     return (bSendStrRouMsgSetPlay(m_oISrcProp.m_u16Ucid));
   }
   else
   {
     return (bSendStrRouMsg(oMsgSetPlay));
   }
}  // tBool arl_tclISourceFsm::bSendSetPlay()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendSetStop()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendSetStop()
{
   // Prepare the set play message
   arl_tSetStopStart oMsgSetStop;
   oMsgSetStop.u16UCID  =  m_oISrcProp.m_u16Ucid;

   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending Set Stop message to stream router\
      UCID:0x%x, SourceNum:%d"
            ,ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , oMsgSetStop.u16UCID, m_u8SrcNum));
   }
   if(m_cpoStrRouClientASF != NULL)
   {
     return (bSendStrRouMsgSetStop(m_oISrcProp.m_u16Ucid));
   }
   else
   {
     return (bSendStrRouMsg(oMsgSetStop));
   }
}  // tBool arl_tclISourceFsm::bSendSetStop()


tBool arl_tclISourceFsm::bSendSrcActToApp()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]bOnSendSrcActToApp: Request Application: source %d, activity %d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(m_u8SrcNum))
            , ETG_ENUM(arl_tenActivity, static_cast<arl_tenActivity>(m_oLastSrcActivReq.enType)) ));
   }
   //Start Timer (daw2hi: no, we just create the timer)
   bTimerStart((OSAL_tpfCallback)vHandleTimerCallBack, (tPVoid)this);

   if((OSAL_C_INVALID_HANDLE != m_hTimer))
   {
    //daw2hi: this will start the timer
      if(OSAL_OK != OSAL_s32TimerSetTime(m_hTimer, ARL_SOURCEACTIVITY_TIMER_DURATION, 0))
      {
          ETG_TRACE_ERR(("Failed to Start Startup Route Allocate Timer"));
      }//if(OSAL_OK != OSAL_s32TimerSetTime(...)
      ETG_TRACE_USR1(("[%d]bOnSendSrcActToApp: Set Timer %u ms"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ARL_SOURCEACTIVITY_TIMER_DURATION));
   }//if((TRUE == bRetVal) && (OSAL_C_INVALID_HANDLE != m_hTimer))
   else
   {
      ETG_TRACE_FATAL(("[%d]bOnSendSrcActToApp: OSAL_C_INVALID_HANDLE == m_hTimer"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
   }// End of if-else; if((TRUE == bRetVal) && (OSAL_C_INVALID_HANDLE != m_hTimer))



   tBool bRetVal  =  FALSE;
   tBool bRetValSubSource  =  FALSE;
   tBool bRetValSubSourcePosNext = FALSE;

   if (OSAL_NULL != m_cpoISource)
   {
      // Before sending the SA result, trigger the SourceActivity result to app
      bRetVal  =  m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_oLastSrcActivReq);
      bRetValSubSource  =  m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_u16SubSource, m_oLastSrcActivReq);
      bRetValSubSourcePosNext  =  m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_u16SubSource, static_cast<arl_tenSource>(m_u8PosNextSrc), m_oLastSrcActivReq);
      // Need to check both return values because it is not clear which call
      // is overridden and handled by the App. If one return is true then overall return is true
      bRetVal = bRetVal | bRetValSubSource | bRetValSubSourcePosNext;
   }else{
      ETG_TRACE_ERR(("[%d]bOnSendSrcActToApp: m_cpoISource == NULL", OSAL_ThreadWhoAmI()));
   }
   return bRetVal;
}

tVoid arl_tclISourceFsm::vActResult(arl_tenActivity act)
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]vActResult: App acknowledges: incoming activity %d of source %d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(arl_tenActivity, act)
            , ETG_ENUM(arl_tenSource, static_cast<arl_tenSource>(m_u8SrcNum))
      ));
   }
   ///Update the acknowledgement receive from application into the last acknowledged field
   m_oLastAppAckSrcActivity.enType = (midw_fi_tcl_e8_SrcActivity::tenType) act;

   switch(act)
   {
      case ARL_EN_ISRC_ACT_OFF:
         //Trigger state machine
         (*this)(m_oTrigId.m_s16AppAckActivityStop);
         break;
      case ARL_EN_ISRC_ACT_ON:
         //Trigger state machine
         (*this)(m_oTrigId.m_s16AppAckActivityPlay);
         break;
      case ARL_EN_ISRC_ACT_PAUSE:
         //Trigger state machine
         (*this)(m_oTrigId.m_s16AppAckActivityPause);
         break;
      default:
         break;
   }
}

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnSrcActivity()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bOnSrcActivity()
{
   tBool bRetVal  =  FALSE;


   // Set the trigger to the FSM.
   switch (m_oLastSrcActivReq.enType)
   {
      case arl_tSrcActivity::FI_EN_ON:
      {
         // Trigger stream router play.
         (*this)(m_oTrigId.m_s16StrRouSetPlay);
      }  // case ARL_SRCACT_EN_ON:
      break;
      case arl_tSrcActivity::FI_EN_PAUSE:
      case arl_tSrcActivity::FI_EN_OFF:
      {
         // Trigger stream router stop.
         (*this)(m_oTrigId.m_s16StrRouSetStop);
      }  // case ARL_SRCACT_EN_OFF:, case ARL_SRCACT_EN_PAUSE:
      break;

      default:
      {
         NORMAL_M_ASSERT_ALWAYS();
      }
      break;
   }  // switch (m_oLastSrcActivReq.enType)

   return bRetVal;
}  // tBool arl_tclISourceFsm::bOnSrcActivity()


/******************************************************************************
** FUNCTION:  tBool arl_tclISourceFsm::bServiceAvailable()
******************************************************************************/

tBool arl_tclISourceFsm::bServiceAvailable()
{
   tBool bRetVal  =  ((OSAL_NULL != m_cpoStrRouClient)
                        && (m_cpoStrRouClient->bIfServiceAvailable()));

   bRetVal  =  ((OSAL_NULL != m_cpoStrRouClientAIL)
                           && (m_cpoStrRouClientAIL->bIfServiceAvailable()));

   bRetVal  =  ((OSAL_NULL != m_cpoStrRouClientASF)
                           && (m_cpoStrRouClientASF->bIfServiceAvailable()));

   if(m_cpoISource != NULL)
   {
      if(bRetVal)
      {
         ETG_TRACE_USR1(("[%d]Source Num:%d service is available"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum));
      }else{
         ETG_TRACE_USR1(("[%d]Source Num:%d service is NOT available"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum));
      }
   }else{
      ETG_TRACE_USR1(("[Thread %p] ERROR m_cpoISource == NULL", OSAL_ThreadWhoAmI));
   }
   return bRetVal;

}  // tBool arl_tclISourceFsm::bServiceAvailable()

/******************************************************************************
** FUNCTION:  const arl_tISrcFsmTrigIds& arl_tclISourceFsm::rfcoGetTriggerId()..
******************************************************************************/

const arl_tclISourceFsm::arl_tISrcFsmTrigIds& arl_tclISourceFsm::rfcoGetTriggerId() const
{
   return (m_oTrigId);

}  // const arl_tISrcFsmTrigIds& arl_tclISourceFsm::rfcoGetTriggerId() const

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnAllocStart(const arl_tclISrcAllo..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnAllocStart(const arl_tISrcAllocStart &rfcoAllocateStart)
{
   ///! \note  Check if allocated or not?
   ///  - Allocate only if not allocated.
   /// - In case already allocated, reply with all the MOST allocation details.
   if (OSAL_NULL == m_cpoISource)
   {
      // Should not be here
      // If here the ISource pointer is NULL
      NORMAL_M_ASSERT_ALWAYS();

   }

   if (ARL_ROUTESTATE_EN_KILL != m_enRouteState.enType)
   {
      /// Store the last allocation request.
      m_bLastAllocReq   =  TRUE;

      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_USR1(("[%d]Allocate Start requested for Source Num:%d and Sink %d"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum
         , rfcoAllocateStart.u16Sink));
      }
      // store the SubSource. We need it for the reply.
      m_u16SubSource = rfcoAllocateStart.u16SubSource;
      m_u16Sink = rfcoAllocateStart.u16Sink; //daw2hi added
      /// Store the message as the tracker.
      m_oMsgTrackerList.push_back(rfcoAllocateStart);

      /// Trigger the state machine.
      (*this)(m_oTrigId.m_s16StrRouAllocate);

   }  // if (ARL_ROUTESTATE_EN_KILL != m_enRouteState.enType)
   else
   {
      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_USR1(("[%d]Allocate Start for Source Num:%d resulted in \
               error [ErrNoNotSupported] as route state is in KILL"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum ));
      }
   }  // else if (OSAL_NULL != m_cpoISource)

}  // tVoid arl_tclISourceFsm::vOnAllocStart(const midw_aud_src_routefi_tclMsgAllocateMethodStart &rfcoAl..

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnDeAllocStart(const arl_tISrcDeal..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnDeAllocStart(const arl_tISrcDeallocStart &rfcoDeallocStart)
{
   ///! \note  Just do the opposite of Allocation ;-)
//   if (FALSE != m_bAllocated)
   {
      /// Store the last deallocation request.
      m_bLastAllocReq   =  FALSE;
      if (OSAL_NULL != m_cpoISource)
      {
         ETG_TRACE_USR1(("[%d]DeAllocate Start requested for Source Num:%d"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum));
      }

      // store the SubSource. We need it for the reply.
      m_u16SubSource = rfcoDeallocStart.u16SubSource;
      /// Store the message as the tracker.
      m_oMsgTrackerList.push_back(rfcoDeallocStart);

      // Trigger the state machine.
      (*this)(m_oTrigId.m_s16StrRouDeAllocate);

   }  // if (FALSE != m_bAllocated)

}  // tVoid arl_tclISourceFsm::vOnDeAllocStart(const arl_tISrcDeallocStart &r..

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnSrcActStart(const arl_tISrcAct..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnSrcActStart(const arl_tISrcActivityStart &rfcoISrcActStart)
{
   // DO NOT INCLUDE LOGIC
   // check if already allocated.
   /*if (FALSE != m_bAllocated)
   {
   */
      // Store the latest source activity request.
      // Check whether action has to be taken/triggered based on current source
      // activity state.
      m_oLastSrcActivReq.enType   =  rfcoISrcActStart.e8Activity.enType;

      //if (!(rfcoISrcActStart.e8Activity == m_oSrcActivity))
      {
         m_oLastSrcActivReq.enType   =  rfcoISrcActStart.e8Activity.enType;

         if(m_cpoISource != NULL)
         {
            ETG_TRACE_USR1(("[%d]Source Activity:%d --> %d req for Source Num:%d"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                  , ETG_ENUM(arl_tSrcActivity , m_oSrcActivity.enType)
                  , ETG_ENUM(arl_tSrcActivity , m_oLastSrcActivReq.enType)
                  , m_u8SrcNum));
         }
         m_u16SubSource = rfcoISrcActStart.u16SubSource;
         m_u8PosNextSrc = rfcoISrcActStart.u8PosNextSrc.enType;

         ETG_TRACE_USR4(("SubSource=%d stored",rfcoISrcActStart.u16SubSource));

         // Store the request & trigger the statemachine.
         m_oMsgTrackerList.push_back(rfcoISrcActStart);

         // Trigger the FSM.
         (*this)(m_oTrigId.m_s16SrcActivity);

      /*
      }  // if (!(rfcoISrcActStart.e8Activity == m_oSrcActivity))
      else if (OSAL_NULL != m_cpoISource)
      {
         // Dispatch Source activity for all the pending requests.
         // Fake the incoming service data.
         gm_tclEmptyMessage oFakeServiceData
         (
            rfcoISrcActStart.u16GetSourceAppId()         // Source App id -> From?
            , m_cpoISource->u16GetAppID()                // Target App Id -> Self
            , rfcoISrcActStart.u16GetRegisterId()        // Register Id
            , rfcoISrcActStart.u16GetCmdCounter()        // Command counter
            , m_cpoISource->u16GetServiceID()            // Service ID
            , rfcoISrcActStart.u16GetFunctionId()        // Function Id
            , AMT_C_U8_CCAMSG_OPCODE_METHODSTART         // Opcode
         );

         //ensure there is no pending request


         // Do nothing reply client for source activity.
         arl_tISrcActivityResult oSrcActivityResult;

         oSrcActivityResult.u8Source.enType     =  static_cast<midw_fi_tcl_e8_AudSource::tenType> (m_u8SrcNum);
         oSrcActivityResult.e8Activity.enType   =  m_oSrcActivity.enType;

         ETG_TRACE_USR1(("[%d]Source Activity[%d] not changed for Source Num:%d.\
                         Sending default message with current source activity."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , ETG_ENUM(arl_tSrcActivity, m_oSrcActivity.enType)
                         , m_u8SrcNum));

         (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oSrcActivityResult);

         // Free the shared memory ;-)
         (tVoid)oFakeServiceData.bDelete();
      }

      else
      {
         // Should not be here
         // If here the ISource pointer is NULL
         NORMAL_M_ASSERT_ALWAYS();
      }  // End of if-else; if (!(rfcoISrcActStart.e8Activity == m_oSrcActivity))
      */
   }
   #if 0
   else
   {
      // Not allocated reply with error.

      // Dispatch Source activity for all the pending requests.
      // Fake the incoming service data.
      gm_tclEmptyMessage oFakeServiceData
      (
         rfcoISrcActStart.u16GetSourceAppId()         // Source App id -> From?
         , m_cpoISource->u16GetAppID()                // Target App Id -> Self
         , rfcoISrcActStart.u16GetRegisterId()        // Register Id
         , rfcoISrcActStart.u16GetCmdCounter()        // Command counter
         , m_cpoISource->u16GetServiceID()            // Service ID
         , rfcoISrcActStart.u16GetFunctionId()        // Function Id
         , AMT_C_U8_CCAMSG_OPCODE_METHODSTART         // Opcode
      );

      // Send error message ?
   }
   #endif
}  // tVoid arl_tclISourceFsm::vOnSrcActStart(const arl_tISrcActivityStart &rf..

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnStartUpAllocRouteResult(.u16SrcNum.)
******************************************************************************/
tVoid arl_tclISourceFsm::vOnStartUpAllocRouteResult(const tU16 u16SrcNum)
{
    m_enAllocateOnStart = ARL_EN_ROUTE_STATE_ALLOCATED;
    OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(u16SrcNum);
}//tVoid arl_tclISourceFsm::vOnStartUpAllocRouteResult(const tU16 u16SrcNum)

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnAllocRouteResult(const arl_tAllocR..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnAllocRouteResult
(
   const arl_tAllocRouteResult &rfcoAllocRouteResult
)
{
   m_bAllocated   =  TRUE;
   // Update the Source information, but clear the old contents first.
//   m_oSrcInfo.vDestroy();
//
//   m_oSrcInfo.u8SourceNr         =  m_u8SrcNum;
//   m_oSrcInfo.u16BlockWidth      =  rfcoAllocRouteResult.u8BlkWidth;
//   m_oSrcInfo.u16ConnectionLabel =  rfcoAllocRouteResult.u16MOSTConnectionLabel;

   // Trigger application activity on allocation result.
   if (OSAL_NULL != m_cpoISource)
   {
      tBool bRetVal  =  m_cpoISource->bOnAllocate(static_cast<arl_tenSource>(m_u8SrcNum), rfcoAllocRouteResult);

      ETG_TRACE_USR1(("[%d]Allocate Route result recieved for Source Num:%d.\
         Sending Allocation result to application returned status: %d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum, ETG_ENUM(ARL_STATUS, bRetVal)));
   }

   // Send Allocate result.
   (tVoid)bSendAllocateResult();

   // Set the trigger for state machine.
   (*this)(m_oTrigId.m_s16StrRouAllocateMr);
}

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnDeAllocRouteResult(const arl_tDeAl..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnDeAllocRouteResult
(
   const arl_tDeAllocRouteResult &rfcoDeAllocRouteResult
)
{
   m_bAllocated   =  FALSE;
   // Clear the Source Information, source activity flags etc.
//   m_oSrcInfo.vDestroy();

   // Trigger application activity on deallocation result.
   if (OSAL_NULL != m_cpoISource)
   {
      tBool bRetVal  =  m_cpoISource->bOnDeAllocate(static_cast<arl_tenSource>(m_u8SrcNum));

      ETG_TRACE_USR1(("[%d]DeAllocate Route result recieved for Source Num:%d.\
         Sending DeAllocation result to application returned status: %d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum, ETG_ENUM(ARL_STATUS, bRetVal)));
   }

   // Send DeAllocate result.
   (tVoid)bSendDeAllocateResult();

   // Set the trigger for state machine.
   (*this)(m_oTrigId.m_s16StrRouDeAllocateMr);

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(rfcoDeAllocRouteResult);
}


/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnSetPlayResult(const arl_tSetPlayR..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnSetPlayResult(const arl_tSetPlayResult& rfcoSetPlayResult)
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Set Play result recieved for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }
   // Trigger the statemachine on SetPlay result.
   // also set the internal state to SA ON.
   m_oSrcActivity.enType =  arl_tSrcActivity::FI_EN_ON;

   (*this)(m_oTrigId.m_s16StrRouSetPlayMr);

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(rfcoSetPlayResult);
}  // tVoid arl_tclISourceFsm::vOnSetPlayResult(const arl_tSetPlayResult& rfco..

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnSetStopResult(const arl_tSetStopRes..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnSetStopResult(const arl_tSetStopResult &rfcoSetStopResult)
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Set Stop result recieved for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }
   // Trigger the statemachine on SetPlay result.
   // also set the internal state to SA OFF.
   m_oSrcActivity.enType =  arl_tSrcActivity::FI_EN_OFF;

   (*this)(m_oTrigId.m_s16StrRouSetStopMr);

   OSAL_C_PARAMETER_INTENTIONALLY_UNUSED(rfcoSetStopResult);
}  // tVoid arl_tclISourceFsm::vOnSetStopResult(const arl_tSetStopResult &rfco..

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendSrcActivResult()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendSrcActivResult()
{

   tBool bRetVal = FALSE;
   bTimerStop();
   for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin();
      ((OSAL_NULL != m_cpoISource) && (iterMsgTrack != m_oMsgTrackerList.end()));)
   {
      const arl_tclMsgIdentifiers& rfcoMsgId =  (*iterMsgTrack);

      // Check if the FID is that of Source activity?
      if (ARL_EN_SOURCEACTIVITY == rfcoMsgId.u16GetFunctionId())
      {
         // Dispatch Source activity for all the pending requests.
         // Fake the incoming service data.
         gm_tclEmptyMessage oFakeServiceData
         (
            rfcoMsgId.u16GetSourceAppId()             // Source App id -> From?
            , m_cpoISource->u16GetAppID()             // Target App Id -> Self
            , rfcoMsgId.u16GetRegisterId()            // Register Id
            , rfcoMsgId.u16GetCmdCounter()            // Command counter
            , m_cpoISource->u16GetServiceID()         // Service ID
            , rfcoMsgId.u16GetFunctionId()            // Function Id
            , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
         );

         // Send the response to the source activity.
         midw_aud_src_routefi_tclMsgSourceActivityMethodResult oISrcActivResult;
         oISrcActivResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
         oISrcActivResult.e8Activity.enType  =  m_oLastSrcActivReq.enType;

         oISrcActivResult.u16SubSource = m_u16SubSource;
         ETG_TRACE_USR1(("[%d]Sending Source Activity[%d] result for Source Num:%d, SubSrc=%d"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(arl_tSrcActivity, m_oLastSrcActivReq.enType), m_u8SrcNum, m_u16SubSource));

         bRetVal = m_cpoISource->bSendMessage(oFakeServiceData, oISrcActivResult);

         // Delete shared memory when going out of scope.
         (tVoid)oFakeServiceData.bDelete();

         // Remove the message id from the list.
         iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         // If after sending request to SR another request was incoming
         // then retriggerStreamRouter
         tBool retriggerStreamRouter = FALSE;
         switch(m_oSrcActivity.enType)
         {
            // SrcAct request Pause & Off pointing to StreamRouter Off
            case midw_fi_tcl_e8_SrcActivity::FI_EN_OFF:
            case midw_fi_tcl_e8_SrcActivity::FI_EN_PAUSE:
               if(m_oLastSrcActivReq.enType == midw_fi_tcl_e8_SrcActivity::FI_EN_ON)
               {
                  retriggerStreamRouter = TRUE;
               }
               break;
            case midw_fi_tcl_e8_SrcActivity::FI_EN_ON:
            default:
               if(m_oLastSrcActivReq.enType != midw_fi_tcl_e8_SrcActivity::FI_EN_ON)
               {
                  retriggerStreamRouter = TRUE;
               }
               break;
         }

         if (retriggerStreamRouter)
         {
            ETG_TRACE_USR1(("[%d]Source activity is not same as last requested\
               Requesting source activity change [%d]-->[%d]"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , ETG_ENUM(arl_tSrcActivity, m_oSrcActivity.enType)
               , ETG_ENUM(arl_tSrcActivity, m_oLastSrcActivReq.enType)));

            // Trigger the source activity change.
            (*this)(m_oTrigId.m_s16SrcActivity);
            // Just exit after sending a message.
            break;
         }
         else
         {
            // Just continue looping.
         }
      }  // if (ARL_EN_SOURCEACTIVITY == rfcoMsgId.u16GetFunctionId())
      else
      {
         // Just increment the iterator.
         ++iterMsgTrack;
      }  // End of if-else; if (ARL_EN_SOURCEACTIVITY == rfcoMsgId.u16GetF..

   }  // for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.beg..

   return bRetVal;

}  // tBool arl_tclISourceFsm::bSendSrcActivResult()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendAllocateResult()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendAllocateResult()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending Allocate result for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }

   for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin();
      ((OSAL_NULL != m_cpoISource) && (iterMsgTrack != m_oMsgTrackerList.end()));)
   {
      const arl_tclMsgIdentifiers& rfcoMsgId =  (*iterMsgTrack);

      // Dispatch Source activity for all the pending requests.
      // Fake the incoming service data.
      gm_tclEmptyMessage oFakeServiceData
      (
         rfcoMsgId.u16GetSourceAppId()             // Source App id -> From?
         , m_cpoISource->u16GetAppID()             // Target App Id -> Self
         , rfcoMsgId.u16GetRegisterId()            // Register Id
         , rfcoMsgId.u16GetCmdCounter()            // Command counter
         , m_cpoISource->u16GetServiceID()         // Service ID
         , rfcoMsgId.u16GetFunctionId()            // Function Id
         , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
      );

      // Check if the FID is that of Source activity?
      if (ARL_EN_ALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
         midw_aud_src_routefi_tclMsgAllocateMethodResult oISrcAllocResult;
         oISrcAllocResult.u8Source.enType        =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
         oISrcAllocResult.u16BlockWidth      =  0;//m_oSrcInfo.u16BlockWidth;
         oISrcAllocResult.u16ConnectionLabel =  0;//m_oSrcInfo.u16ConnectionLabel;
         oISrcAllocResult.u16SubSource = m_u16SubSource;
         (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcAllocResult);

         ETG_TRACE_USR1(("[%d]Allocate result sent Source Num:%d, BlkWidth:%d, ConLabel:0x%x."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum, oISrcAllocResult.u16BlockWidth
            , oISrcAllocResult.u16ConnectionLabel));

         // Remove the message id from the list.
         iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         if (m_bLastAllocReq != m_bAllocated)
         {
            ETG_TRACE_USR1(("Allocation state is not same as last requested\
               Requesting allocation change [ALLOCATE]-->[DEALLOCATE]"));
            // Trigger the deallocation request, as already allocated and
            // Latest allocation request is to deallocate.
            (*this)(m_oTrigId.m_s16StrRouDeAllocate);

            // Just exit after sending a message.
            break;
         }
         else
         {
            // Just continue looping.
            ETG_TRACE_USR4(("[%d]Continue looping, till msg id list end."
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
         }
      }
      else if (ARL_EN_DEALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
         // If last request was allocate, it is not required to keep the
         // deallocation request in queue, just fake the response.

         if (m_bLastAllocReq == m_bAllocated)
         {
          midw_aud_src_routefi_tclMsgDeAllocateMethodResult oISrcDeallocResult;
            oISrcDeallocResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
            (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcDeallocResult);

            ETG_TRACE_USR1(("[%d]Faking DeAllocate result from Source Num:%d."
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                  , m_u8SrcNum));

            // Remove the message id from the list.
            iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         }  // if (m_bLastAllocReq == m_bAllocated)
         else
         {
            ETG_TRACE_USR4(("continue loop"));
            ++iterMsgTrack;
         }
      }
      else
      {
         // Just increment the iterator.
         ++iterMsgTrack;
         ETG_TRACE_USR4(("[%d]Incremented the msg id list iterator."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
      }
      // Delete shared memory when going out of scope.
      (tVoid)oFakeServiceData.bDelete();

   }  // for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin(..

   return TRUE;
}  // tBool arl_tclISourceFsm::bSendAllocateResult()

/******************************************************************************
** FUNCTION:  tBool arl_tclISourceFsm::bSendDeAllocateResult()
******************************************************************************/
/*virtual*/
tBool arl_tclISourceFsm::bSendDeAllocateResult()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending DeAllocate result for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }
   // Iterate through the message tracker list and send the deallocate result.
   for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin();
      ((OSAL_NULL != m_cpoISource) && (iterMsgTrack != m_oMsgTrackerList.end()));)
   {
      const arl_tclMsgIdentifiers& rfcoMsgId =  (*iterMsgTrack);
      ETG_TRACE_USR4(("bSendDeAllocateResult: rfcoMsgId.u16GetFunctionId: %d, m_enRouteState.enType %d, m_bLastAllocReq %d, m_bAllocated %d"
                     , rfcoMsgId.u16GetFunctionId()
                     , m_enRouteState.enType
                     , m_bLastAllocReq
                     , m_bAllocated ));
      // Dispatch Source activity for all the pending requests.
      // Fake the incoming service data.
      gm_tclEmptyMessage oFakeServiceData
      (
         rfcoMsgId.u16GetSourceAppId()             // Source App id -> From?
         , m_cpoISource->u16GetAppID()             // Target App Id -> Self
         , rfcoMsgId.u16GetRegisterId()            // Register Id
         , rfcoMsgId.u16GetCmdCounter()            // Command counter
         , m_cpoISource->u16GetServiceID()         // Service ID
         , rfcoMsgId.u16GetFunctionId()            // Function Id
         , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
      );

      // Check if the FID is that of Source activity?
      if (ARL_EN_DEALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
       midw_aud_src_routefi_tclMsgDeAllocateMethodResult oISrcDeallocResult;
         oISrcDeallocResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
         oISrcDeallocResult.u16SubSource = m_u16SubSource;
         (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcDeallocResult);

         ETG_TRACE_USR1(("[%d]DeAllocate result sent Source Num:%d."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum));

         // Remove the message id from the list.
         iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         if ((m_bLastAllocReq != m_bAllocated)
            && (ARL_ROUTESTATE_EN_KILL != m_enRouteState.enType))
         {
            ETG_TRACE_USR1(("[%d]Allocation state is not same as last requested\
               Requesting allocation change [DEALLOCATE]-->[ALLOCATE]"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));

            // Trigger the allocation request, as already deallocated and
            // Latest allocation request is to allocate.
            (*this)(m_oTrigId.m_s16StrRouAllocate);

            // Just exit after sending a message.
            break;
         }
         else if (ARL_ROUTESTATE_EN_KILL == m_enRouteState.enType)
         {
            ETG_TRACE_USR1(("[%d]Allocate Start for Source Num:%d resulted in \
                            error [ErrorNo. not supported] as route state is in KILL"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                  , m_u8SrcNum));

         }
         else
         {
            // Just continue looping.
            ETG_TRACE_USR4(("Continue looping, till msg id list end."));
         }
      }
      else if (ARL_EN_ALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
         // If last request was deallocate, it is not required to keep the
         // allocation request in queue, just fake the response, with
         // both connection label & Block width as "0"

         if (m_bLastAllocReq == m_bAllocated)
         {
          midw_aud_src_routefi_tclMsgAllocateMethodResult oISrcAllocResult;
            oISrcAllocResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
            oISrcAllocResult.u16BlockWidth      =  0;//m_oSrcInfo.u16BlockWidth;
            oISrcAllocResult.u16ConnectionLabel =  0;//m_oSrcInfo.u16ConnectionLabel;
            (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcAllocResult);

            ETG_TRACE_USR1(("[%d]Faking Allocate result from Source Num:%d"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                  , m_u8SrcNum));
//               , oISrcAllocResult.u16BlockWidth, oISrcAllocResult.u16ConnectionLabel));

            // Remove the message id from the list.
            iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         }  // if (m_bLastAllocReq == m_bAllocated)
         else
         {
            ETG_TRACE_USR4(("continue loop"));
            ++iterMsgTrack;
         }
      }
      else
      {
         // Just increment the iterator.
         ++iterMsgTrack;
         ETG_TRACE_USR4(("[%d]Incremented the msg id list iterator."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
      }

      // Delete shared memory when going out of scope.
      (tVoid)oFakeServiceData.bDelete();
   }

   return TRUE;
}  // tBool arl_tclISourceFsm::bSendDeAllocateResult()

/******************************************************************************
** FUNCTION:  tVoid arl_tclISourceFsm::vOnRouteState(const arl_tSe..
******************************************************************************/

tVoid arl_tclISourceFsm::vOnRouteState(const arl_tRouteState& rfcoRouteState)
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Route state processing for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(arl_tenSource, static_cast<arl_tenSource>(m_u8SrcNum))));
   }
   // if current source number is allocated? check this first - state handling
   if ((TRUE == m_bAllocated) || (ARL_ROUTESTATE_EN_KILL == m_enRouteState.enType))
   {
      tBool bFound   =  FALSE;

      // Iterate through the list to find the matching UCID. If found capture
      // the routing state
      for (arl_tRouteEntryIter iterRoute = rfcoRouteState.routing_list.sREntry.begin();
            iterRoute < rfcoRouteState.routing_list.sREntry.end(); ++iterRoute)
      {
         // Check the ucid match
         if (m_oISrcProp.m_u16Ucid == (tU16)iterRoute->u32UCID)
         {
            bFound   =  TRUE;
            if(m_cpoISource != NULL)
            {
               // Check for routing state. if routing state is KILL.
               // Set the state internally and report the MOST error to Application.
               ETG_TRACE_USR4(("[%d]    Found Source: %d UCID: 0x%x(%d) State:%d"
                     , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                     , ETG_ENUM(arl_tenSource, static_cast<arl_tenSource>(m_u8SrcNum))
                     , (tU16)iterRoute->u32UCID
                     , (tU16)iterRoute->u32UCID
                     , ETG_ENUM( midw_fi_tcl_e8_streamrouter_routestate.tenType, iterRoute->e8Routestate.enType)));
            }
            switch (iterRoute->e8Routestate.enType)
            {
            case ARL_ROUTESTATE_EN_KILL:
               {
                  // Route is killed. Set the route allocated to FALSE;
                  // Propogate error as a MOST Error to application.
                  m_bAllocated   =  FALSE;
                  // Clear the Source Information, source activity flags etc.
//                  m_oSrcInfo.vDestroy();

                  if ((ARL_ROUTESTATE_EN_KILL != m_enRouteState.enType)
                     && (OSAL_NULL != m_cpoISource))
                  {
                     m_enRouteState.enType   =  ARL_ROUTESTATE_EN_KILL;
                     m_cpoISource->vOnError(m_u8SrcNum, ARL_EN_ISRC_MOST_ERR);

                     // Send DeAllocate result.
                     (tVoid)bSendDeAllocateResult();

                     // Set the trigger for state machine.
                     (*this)(m_oTrigId.m_s16StrRouDeAllocateMr);
                  }

               }  // case ARL_ROUTESTATE_EN_KILL:
               break;

            case ARL_ROUTESTATE_EN_INIT:
            case ARL_ROUTESTATE_EN_STOP:
            case ARL_ROUTESTATE_EN_PLAY:
            default:
               {
                  m_enRouteState.enType   =  iterRoute->e8Routestate.enType;
               }  // case ARL_ROUTESTATE_EN_INIT:, case ARL_ROUTESTATE_EN_STOP..
               break;

            }  // switch (iterRoute.e8Routestate.enType)
         }  // if (m_oISrcProp.m_u16Ucid == (tU16)iterRoute.u32UCID)

      }  // for (arl_tRouteEntryIter iterRoute = rfcoRouteState.routing_list...

      if (FALSE == bFound)
      {
         // Entry not found in the list.
         // Route deallocation was successful. Only set the routing state to INIT.
         // all deallocation related handling would happen when the deallocation
         // result comes. ;-)
         m_enRouteState.enType   =  ARL_ROUTESTATE_EN_INIT;

      }  // if (FALSE == bFound)
   }
   else
   {
      // This source is not allocated, just discard. ;-)
   }

}  // tVoid arl_tclISourceFsm::vOnRouteState(const arl_tRouteState& rfcoRouteS..

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAppStateOff()
******************************************************************************/

tBool arl_tclISourceFsm::bOnAppStateOff()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]bOnAppStateOff for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }
   /***
   * Because of too many messages at stream router, we will have to deallocate and set stop, only when the route is allocated for the current source.
   * Fix for NCG3D-9715
   */
   if(m_bAllocated)
   {
     bSendSetStop();
     //Trigger Deallocation
     bSendDeAllocRoute();
   }
   //Stop SourceActivity Timer
   bTimerStop();
   return TRUE;
}

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAllocErr()
******************************************************************************/

/*virtual*/
tBool arl_tclISourceFsm::bOnAllocErr()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending Allocate error for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }
   for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin();
      ((OSAL_NULL != m_cpoISource) && (iterMsgTrack != m_oMsgTrackerList.end()));)
   {
      const arl_tclMsgIdentifiers& rfcoMsgId =  (*iterMsgTrack);

      // Dispatch Source activity for all the pending requests.
      // Fake the incoming service data.
      gm_tclEmptyMessage oFakeServiceData
      (
         rfcoMsgId.u16GetSourceAppId()             // Source App id -> From?
         , m_cpoISource->u16GetAppID()             // Target App Id -> Self
         , rfcoMsgId.u16GetRegisterId()            // Register Id
         , rfcoMsgId.u16GetCmdCounter()            // Command counter
         , m_cpoISource->u16GetServiceID()         // Service ID
         , rfcoMsgId.u16GetFunctionId()            // Function Id
         , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
      );

      // Check if the FID is that of Allocate?
      if (ARL_EN_ALLOCATE == rfcoMsgId.u16GetFunctionId())
      {

         ETG_TRACE_USR1(("[%d] Allocate error [info not supported], src %d"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , ETG_ENUM(arl_tenSource, m_u8SrcNum)));

     //Send error result
     midw_aud_src_routefi_tclMsgAllocateMethodResult oISrcAllocResult;
         oISrcAllocResult.u8Source.enType        =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
         oISrcAllocResult.u16BlockWidth      =  0;//m_oSrcInfo.u16BlockWidth;
         oISrcAllocResult.u16ConnectionLabel =  0;//m_oSrcInfo.u16ConnectionLabel;
         oISrcAllocResult.u16SubSource = m_u16SubSource;
     oISrcAllocResult.u16Error = ARL_ERROR_ALLOC;
         (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcAllocResult);

     ETG_TRACE_USR1(("[%d]Allocate result sent Source Num:%d, BlkWidth:%d, ConLabel:0x%x., Error:%d"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum, oISrcAllocResult.u16BlockWidth
            , oISrcAllocResult.u16ConnectionLabel, oISrcAllocResult.u16Error));


         // Remove the message id from the list.
         iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         if (m_bLastAllocReq != m_bAllocated)
         {
            ETG_TRACE_USR1(("[%d]Allocation state is not same as last requested\
               Requesting allocation change [ALLOCATE]-->[DEALLOCATE]"
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
            // Trigger the deallocation request, as already allocated and
            // Latest allocation request is to deallocate.
            (*this)(m_oTrigId.m_s16StrRouDeAllocate);

            // Just exit after sending a message.
            break;
         }
         else
         {
            // Just continue looping.
            ETG_TRACE_USR4(("[%d]Continue looping, till msg id list end."
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
         }
      }
      else if (ARL_EN_DEALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
         // If last request was allocate, it is not required to keep the
         // deallocation request in queue, just fake the response.

         if (m_bLastAllocReq == m_bAllocated)
         {
            midw_aud_src_routefi_tclMsgDeAllocateMethodResult oISrcDeallocResult;
            oISrcDeallocResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
            (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcDeallocResult);

            ETG_TRACE_USR1(("[%d]Faking DeAllocate result from Source Num:%d."
                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                  , m_u8SrcNum));

            // Remove the message id from the list.
            iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         }  // if (m_bLastAllocReq == m_bAllocated)
         else
         {
            ETG_TRACE_USR4(("continue loop"));
            ++iterMsgTrack;
         }
      }
      else
      {
         // Just increment the iterator.
         ++iterMsgTrack;
         ETG_TRACE_USR4(("[%d]Incremented the msg id list iterator."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
      }
      // Delete shared memory when going out of scope.
      (tVoid)oFakeServiceData.bDelete();

   }  // for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin(..

   return TRUE;
}  // tBool arl_tclISourceFsm::bOnAllocErr()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnDeAllocErr()
******************************************************************************/

/*virtual*/
tBool arl_tclISourceFsm::bOnDeAllocErr()
{
   if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending DeAllocate error for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }

   // Iterate through the message tracker list and send the deallocate result.
   for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin();
      ((OSAL_NULL != m_cpoISource) && (iterMsgTrack != m_oMsgTrackerList.end()));)
   {
      const arl_tclMsgIdentifiers& rfcoMsgId =  (*iterMsgTrack);

      // Dispatch Source activity for all the pending requests.
      // Fake the incoming service data.
      gm_tclEmptyMessage oFakeServiceData
      (
         rfcoMsgId.u16GetSourceAppId()             // Source App id -> From?
         , m_cpoISource->u16GetAppID()             // Target App Id -> Self
         , rfcoMsgId.u16GetRegisterId()            // Register Id
         , rfcoMsgId.u16GetCmdCounter()            // Command counter
         , m_cpoISource->u16GetServiceID()         // Service ID
         , rfcoMsgId.u16GetFunctionId()            // Function Id
         , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
      );

      // Check if the FID is that of deallocate?
      if (ARL_EN_DEALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
         ETG_TRACE_USR1(("[%d]DeAllocate error sent Source Num:%d."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum));



//         if ((m_bLastAllocReq != m_bAllocated)
//            && (ARL_ROUTESTATE_EN_KILL != m_enRouteState.enType))

         //Check if latest request was deallocate
         if (!m_bLastAllocReq)
         {
            ETG_TRACE_USR1(("[%d] Send Deallocate Error to requester. Source %d"
                 , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
                 , ETG_ENUM(arl_tenSource, m_u8SrcNum)));

      //Send error result
       midw_aud_src_routefi_tclMsgDeAllocateMethodResult oISrcDeallocResult;
       oISrcDeallocResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
       oISrcDeallocResult.u16SubSource = m_u16SubSource;
       oISrcDeallocResult.u16Error = ARL_ERROR_DEALLOC;
       (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcDeallocResult);

       ETG_TRACE_USR1(("[%d]DeAllocate result sent Source Num:%d, SubSource = %d,\
         Error:%d"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
               , m_u8SrcNum, m_u16SubSource, oISrcDeallocResult.u16Error));

            // Remove the message id from the list.
            iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

            // Trigger the allocation request, as already deallocated and
            // Latest allocation request is to allocate.
//            (*this)(m_oTrigId.m_s16StrRouAllocate);

            // Just exit after sending a message.
            break;
         }else{
            // Just continue looping.
            ETG_TRACE_USR4(("Continue looping, till msg id list end."));
            ++iterMsgTrack;
         }
      }
      else if (ARL_EN_ALLOCATE == rfcoMsgId.u16GetFunctionId())
      {
         // If last request was deallocate, it is not required to keep the
         // allocation request in queue, just fake the response, with
         // both connection label & Block width as "0"

//         if (m_bLastAllocReq == m_bAllocated)
//         {
//            arl_tISrcAllocResult oISrcAllocResult;
//            oISrcAllocResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
//            oISrcAllocResult.u16BlockWidth      =  0;//m_oSrcInfo.u16BlockWidth;
//            oISrcAllocResult.u16ConnectionLabel =  0;//m_oSrcInfo.u16ConnectionLabel;
//            (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcAllocResult);
//
//            ETG_TRACE_USR1(("[%d]Faking Allocate result from Source Num:%d"
//                  , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
//                  , m_u8SrcNum));
////               , oISrcAllocResult.u16BlockWidth, oISrcAllocResult.u16ConnectionLabel));
//
//            // Remove the message id from the list.
//            iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);
//
//         }  // if (m_bLastAllocReq == m_bAllocated)
          // Just continue looping.
          ETG_TRACE_USR4(("Continue looping, till msg id list end."));
          ++iterMsgTrack;
      }
      else
      {
         // Just increment the iterator.
         ++iterMsgTrack;
         ETG_TRACE_USR4(("[%d]Incremented the msg id list iterator."
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())));
      }

      // Delete shared memory when going out of scope.
//      (tVoid)oFakeServiceData.bDelete();
   }

   return TRUE;
}  // tBool arl_tclISourceFsm::bOnDeAllocErr()

/******************************************************************************
** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnSrcActivityErr()
******************************************************************************/

/*virtual*/
tBool arl_tclISourceFsm::bOnSrcActivityErr()
{
  if(m_cpoISource != NULL)
   {
      ETG_TRACE_USR1(("[%d]Sending Source Activity error for Source Num:%d."
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , m_u8SrcNum));
   }
   for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.begin();
         ((OSAL_NULL != m_cpoISource) && (iterMsgTrack != m_oMsgTrackerList.end()));)
   {
      const arl_tclMsgIdentifiers& rfcoMsgId =  (*iterMsgTrack);

      // Check if the FID is that of Source activity?
      if (ARL_EN_SOURCEACTIVITY == rfcoMsgId.u16GetFunctionId())
      {
         // Dispatch Source activity for all the pending requests.
         // Fake the incoming service data.
         gm_tclEmptyMessage oFakeServiceData
      (
         rfcoMsgId.u16GetSourceAppId()             // Source App id -> From?
         , m_cpoISource->u16GetAppID()             // Target App Id -> Self
         , rfcoMsgId.u16GetRegisterId()            // Register Id
         , rfcoMsgId.u16GetCmdCounter()            // Command counter
         , m_cpoISource->u16GetServiceID()         // Service ID
         , rfcoMsgId.u16GetFunctionId()            // Function Id
         , AMT_C_U8_CCAMSG_OPCODE_METHODSTART      // Opcode
      );

     // Send the response to the source activity.
         midw_aud_src_routefi_tclMsgSourceActivityMethodResult oISrcActivResult;
         oISrcActivResult.u8Source.enType         =  static_cast<midw_fi_tcl_e8_AudSource::tenType>(m_u8SrcNum);
         oISrcActivResult.e8Activity.enType  =  m_oLastSrcActivReq.enType;
     oISrcActivResult.u16SubSource = m_u16SubSource;
     oISrcActivResult.u16Error  =  ARL_ERROR_SRCACT;
     ETG_TRACE_USR1(("[%d]Sending Source Activity[%d] result for Source Num:%d, SubSource:%d, Error:%d"
               , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID())
            , ETG_ENUM(arl_tSrcActivity, m_oLastSrcActivReq.enType), m_u8SrcNum,m_u16SubSource,oISrcActivResult.u16Error));
      (tVoid)m_cpoISource->bSendMessage(oFakeServiceData, oISrcActivResult);

         // Remove the message id from the list.
         iterMsgTrack   =  m_oMsgTrackerList.erase(iterMsgTrack);

         //Send Player explicit SrcActivityOff command
         //trigger the SourceActivity Off
         m_oLastSrcActivReq.enType = midw_fi_tcl_e8_SrcActivity::FI_EN_OFF;
         m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_oLastSrcActivReq);
         // Some might support this with SubSource
         m_cpoISource->bOnSrcActivity(static_cast<arl_tenSource>(m_u8SrcNum), m_u16SubSource, m_oLastSrcActivReq);

         // Trigger
         (*this)(m_oTrigId.m_s16MsgPostTimeOut);
         // Just exit after sending a message.
         break;
      }  // if (ARL_EN_SOURCEACTIVITY == rfcoMsgId.u16GetFunctionId())
      else
      {
         // Just increment the iterator.
         ++iterMsgTrack;
      }  // End of if-else; if (ARL_EN_SOURCEACTIVITY == rfcoMsgId.u16GetF..

   }  // for (arl_tIMsgTrackerListIter iterMsgTrack = m_oMsgTrackerList.beg..

   return TRUE;
}  // tBool arl_tclISourceFsm::bOnSrcActivityErr()

tBool arl_tclISourceFsm::bTimerStop ()
{
   tBool bSuccess = FALSE;

   if (m_hTimer != OSAL_C_INVALID_HANDLE)
   {
      if(OSAL_s32TimerSetTime(m_hTimer, 0, 0) == OSAL_OK)
      {
         bSuccess = TRUE;
      }else{
         ETG_TRACE_ERR(("Stop Timer FAILED. Source: %d",
           ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(m_u8SrcNum))));
      }
   }
   return bSuccess;
}


/******************************************************************************
** FUNCTION:  tBool arl_tclISourceFsm::bValidateStateExit_AppAckActivityStopPause()
******************************************************************************/
tBool arl_tclISourceFsm::bValidateStateExit_AppAckActivityStopPause()
{
   tBool bSuccess = false;
   ///Exit state only when the requested source activity and acknowledged source activity are the same.
   if(m_oLastAppAckSrcActivity.enType != m_oLastSrcActivReq.enType)
   {
     ETG_TRACE_ERR(("State Change not allowed as Requested Src Activity : %d != Acknowledged Src Activity : %d"
       , ETG_ENUM(arl_tSrcActivity, m_oLastSrcActivReq.enType)
       , ETG_ENUM(arl_tSrcActivity, m_oLastAppAckSrcActivity.enType)
           ));
   }
   else
   {
     bSuccess = TRUE;
     if(m_cpoISource)
     {
       ETG_TRACE_USR1(("[%d] Allowed to Exit State AppAckActivityStopPause"
            , ETG_ENUM(ail_u16AppId,m_cpoISource->u16GetAppID()) ));
     }
   }
   return bSuccess;
}

tBool arl_tclISourceFsm::bTimerStart (OSAL_tpfCallback callback,tPVoid Parameter)
{
   if(OSAL_C_INVALID_HANDLE == m_hTimer)
   {
      if(OSAL_OK != OSAL_s32TimerCreate(callback,
                                        Parameter,&m_hTimer))
      {
         ETG_TRACE_ERR(("ARL Timer Creation Failed. Source: %d",
               ETG_ENUM(arl_tenSource ,static_cast<arl_tenSource>(m_u8SrcNum))));
         return FALSE;
      }//if(OSAL_OK != OSAL_s32TimerCreate(...)
   }
   return TRUE;
}

/***************************************************************************
* FUNCTION: arl_tclISourceFsm::vSetstatefsm (tS16 s16StateId)
*Function only used for the purpose of ARL UNIT testing to set ARL states.
*strictly recommended not to use for other purposes.
****************************************************************************/
tVoid arl_tclISourceFsm::vSetstatefsm(tS16 s16StateId)
{
  arl_tclFsm fsm;
  fsm.vSetState(s16StateId);
}
////////////////////////////////////////////////////////////////////////////////
// <EOF>
