/*
 * arl_tclISourceIF.h
 *
 *  Created on: Jul 6, 2012
 *      Author: vo84hi
 */

#ifndef ARL_TCLISOURCEIF_H_
#define ARL_TCLISOURCEIF_H_

/*!
*******************************************************************************
* \file              arl_tclISource.h
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements the ISource interface.
*  - COPYRIGHT:      &copy; 2010 - 2012 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 25.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - New Error enumerations for iSource errors.
* - Removed the overriden eUpdateClients() from FWK.
*
* \date 15.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/


/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

/// OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

/*!
 * \note
 *    - Interface: BASE_TYPES - Definition of the basic framework parts
 *    used by all FIs.
 *    - Interface: MESSAGE - Definition of an AMT class suitable but not
 *    exclusively usable together with the visitor and type framework.
 */
#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include <common_fi_if.h>

#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include <ahl_if.h>

//#define SYSTEM_S_IMPORT_INTERFACE_MAP     ///< Import STL IF
//#include <stl_pif.h>


#include <map>


#include "arl_ISourceTypes.h"
#include "arl_tclBitmask.h"

/*!
 * \typedef enum arl_tenISourceError
 * \brief ISource Error codes
 */

typedef enum
{
   ARL_EN_ISRC_REQAVACT_ERR = 0        ///< Request AV activation error.
   , ARL_EN_ISRC_REQAVDEACT_ERR        ///< Request AV deactivation error.
   , ARL_EN_ISRC_ALLOC_ERR             ///< Allocation error.
   , ARL_EN_ISRC_DEALLOC_ERR           ///< Deallocation error.
   , ARL_EN_ISRC_SRCACT_ON_ERR         ///< Source activity ON error.
   , ARL_EN_ISRC_SRCACT_OFF_ERR        ///< Source activity OFF error.
   , ARL_EN_ISRC_MOST_ERR              ///< Generic MOST error caused by loss of connection.
   , ARL_EN_ISRC_AVACTION_ERR          ///< Generic error for any AV action/trigger
   , ARL_EN_ISRC_INVALID_ERR  =  255   ///< Invalid error.
} arl_tenISourceError;




class arl_tclISourceVFsm;

/*!
 * \class arl_tclISource
 * \brief Base class which implements the iSource Interface.
 *
 */

class arl_tclISourceIF
{
public:
   virtual ~arl_tclISourceIF()
   {   }

   virtual tU16 u16GetAppID()
   {
      return _pAilAppInterface->u16GetAppId();
   }

   virtual tU16 u16GetServiceID()
   {
      return _u16ServiceID;
   }
   //AIL Variant
   arl_tclISourceIF(ail_tclAppInterfaceRestricted* pAilAppInterface,tU16 u16ServiceID);
   //AHL Variant
   arl_tclISourceIF(ahl_tclBaseOneThreadApp* pAhlAppInterface,tU16 u16ServiceID);


   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vOnTimer(tU16 u16TimerId)
   ***************************************************************************/
   /*!
    * \brief   Timer expiry handler.
    * \param   [u16TimerId]:    (I) Timer Id
    * \retval  NONE
    * \date 08.03.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnTimer(tU16 /*u16TimerId*/) = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tS32 arl_tclISource::s32GetMsgBuffId(tU8 u8SrcNum)const
   ***************************************************************************/
   /*!
    * \brief   Gets Message buffer id.
    * \param   [u8SrcNum]:  (I) Source Number.
    * \retval  [tBool]: TRUE, if source activity was successful, FALSE otherwise
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tS32 s32GetMsgBuffId(tU8 /*u8SrcNum*/) const  = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vOnError()
   ***************************************************************************/
   /*!
    * \brief   Function callback to trigger actions on request AV Activation
    *          error
    * \note    This function needs to be overloaded by the derived class only in
    *          case the source is self triggered & needs to send error message
    *          to clients based on source number.
    * \param   [u8SrcNum]:  (I) Source Number.
    * \param   [cenError]:  (I) ISource Error type.
    * \retval  NONE
    * \date 10.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnError(tU8 /*u8SrcNum*/
      , const arl_tenISourceError /*cenError*/)  = 0;

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vSendMessage( const amt_tclSe..
   ***************************************************************************/
   /*!
    * \brief   Generic Interface to send messages to next clients.
    * \param   [rfcoServiceData]: (I)   Service data to be sent.
    * \param   [rfcoMsgBase]:     (I)   Reference to the Fi object - Base
    *          class of the Fi object.
    * \retval  [tBool]: TRUE if message post successful, FALSE otherwise.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendMessage
   (
      const amt_tclServiceData &rfcoServiceData
      , const fi_tclMessageBase& rfcoMsgBase
   ) = 0;

   tBool bSendMessage(amt_tclServiceData &rfcoServiceData);

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceIF::bOnSrcActivity(const arl..
   ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Application specific function on Source Activity start.
    * \param   [enSrcNum]:  (I) Source Number.
    * \param   [rfcoSrcActivity]: (I) Source Activity
    * \retval  [tBool]: TRUE, if source activity was successful, FALSE otherwise
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 15.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnSrcActivity(arl_tenSource enSrcNum, const arl_tSrcActivity& rfcoSrcActivity) = 0;

   virtual tBool bOnSrcActivity(arl_tenSource enSrcNum, tU16 SubSource, const arl_tSrcActivity& rfcoSrcActivity) = 0;

   virtual tBool bOnSrcActivity(arl_tenSource enSrcNum, tU16 SubSource, arl_tenSource enPosNextSrc, const arl_tSrcActivity& rfcoSrcActivity) = 0;

   /******************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnFGGraphCmdResult(..
   ******************************************************************************/
   virtual tVoid vOnFGGraphCmdResult(arl_tenFGGraphCmd_Result enErrorCode) = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISource::bOnAllocate(
   ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Application specific function after Allocate is processed.
    * \param   [enSrcNum]:  (I) Source Number.
    * \param   [rfcoAllocRoute]: (I) Reference to Allocate route result
    * \retval  [tBool]: TRUE, if Application performed operations successfully,
    *          FALSE otherwise
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 15.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnAllocate(arl_tenSource enSrcNum, const arl_tAllocRouteResult& rfcoAllocRoute) = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceIF::bOnDeAllocate(..
   ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Application specific function after DeAllocate is processed.
    * \param   [enSrcNum]:  (I) Source Number.
    * \retval  [tBool]: TRUE, if Application performed operations successfully,
    *          FALSE otherwise
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 15.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnDeAllocate(arl_tenSource enSrcNum) = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vSetSrcAvailable(tBool bSrcAvail)
   ***************************************************************************/
   /*!
    * \brief   Interface can be used to set the Source availability
    * \param   [bSrcAvail]:   (I)   Source available or not
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vSetSrcAvailable(tBool bSrcAvail = TRUE) = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::bSetAudioRouteRequest(tBool bSrcAvail)
   ***************************************************************************/
   /*!
    * \brief   General implementation gets called by Interface frontend
    * \param   enSource Source the request is valid for
    * \param   enRequest the Requested state
    * \param   Reason
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    * \date 04.03.2014 \updated by daw2hi to use SourceID where subsource is included
    **************************************************************************/
   virtual tBool bSetAudioRouteRequest(SourceID SrcId, arl_tenActivity enRequest, audioroutinglib::arl_tclBitmask<arl_enRouteReqReason> Reason);


   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::bSetSourceAvailability(...
   ***************************************************************************/
   /*!
    * \brief   General implementation gets called by Interface frontend
    * \param   srcAvailability:   availability state of given source
    * \param   availabilityReason: reason for the change of availability
    * \param   enSource: source of that the available state changed
    * \param   u16SubSource: optional the subsource ID (may be identical to DeviceID)
    * \retval  NONE
    * \date 26.11.2013 \author Ruben Volkmer (mailto:R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tBool bSetSourceAvailability(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason availabilityReason, arl_tenSource enSource, tU16 u16SubSource = 0);


   virtual tVoid vCCA_OnUpRegRouteReq(amt_tclServiceData* poMessage);

   virtual tBool bSendFGGraphCmd(arl_tenSource enSrcNum, arl_tenFGCommand enFGCmd);
   virtual tVoid vCCA_OnUpRegMuteReq(amt_tclServiceData* poMessage);
   virtual tVoid vCCA_OnMuteState   (amt_tclServiceData* poMessage);
   virtual tBool bSetAudioProperty(arl_tenSource enSrcNum, arl_tenSourcePropType PropType, tS32 PropValue, tU16 SubSource=0);


   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /****************************************************************************
   ** FUNCTION:    amt_tclServiceData* arl_tclISource::pStatusMessageFactory(t..
   ****************************************************************************/
   /*!
    * \brief   This function shall create for every supported property a status
    *          message, which then is used by the framework to update registered
    *          clients. Typically this function is called when the service
    *          issues a function call like eUpdateClients(..) or
    *          eUpdateRequestingClient(..) or for framework related reasons
    *          (e.g. our service becomes available again).
    * \param   [u16Target]:      (I) Target App Id
    * \param   [u16FunctionId]:  (I) Function id
    * \param   [u16RegisterID]:  (I) Registration id
    * \param   [u16CmdCounter]:  (I) Command counter
    * \retval  [amt_tclServiceData*] : Pointer to service data object.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    ***************************************************************************/
   virtual amt_tclServiceData* pStatusMessageFactory(tU16 /*u16Target*/
      , tU16 u16FunctionId, tU16 /*u16RegisterID*/, tU16 /*u16CmdCounter*/);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnLoopback(tU16 u16ServiceID, amt_tc...
   ***************************************************************************/
   /*!
    * \brief   Loopback service ISource interface.
    * \param   [u16ServiceID]: (I) Service Id
    * \param   [poMessage]:    (->I) Pointer to incoming message.
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnLoopback(tU16 u16ServiceID, amt_tclServiceData* poMessage) = 0;

   virtual tBool bOnAudioRouteRequestReject(arl_tenSource enSource, tU16 SubSource) = 0;

   virtual tBool bOnAudioRouteRequestReject(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISource::bSrcAvailable() const
   ***************************************************************************/
   /*!
    * \brief   Interface provides whether source is available or not
    * \param   NONE
    * \retval  [tBool]: TRUE, if source is available, FALSE otherwise.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSrcAvailable() const = 0;

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnMsISource(amt_tclServiceD..
   ***************************************************************************/
   /*!
    * \brief   Method starts for the ISource based on MOST spec.
    * \param   [poMessage]:    (->I)   Pointer to service data.
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnMsISource(amt_tclServiceData* poMessage) = 0;

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnMrISource(amt_tclServiceD..
   ***************************************************************************/
   /*!
    * \brief   Method result for the ISource for handling loopbacks message
    *          results.
    * \param   [poMessage]:    (->I)   Pointer to service data.
    * \retval  NONE
    * \date 01.03.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnMrISource(amt_tclServiceData* poMessage) = 0;

   /***************************************************************************
    ** FUNCTION:  virtual tBool arl_tclISourceIF::bOnMuteState(..
    ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Interface gets triggered when AudioRouting changed Mute state
    *          for given Source
    * \param   [enSrcNum]: Source ID
    * \param   [enMuteState]: systems actual mute state
    * \retval  TRUE
    * \retval  FALSE
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tBool bOnMuteState(arl_tenSource enSrcNum, arl_tenMuteState enMuteState) = 0;

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::bSetMuteState(arl_tenSource ..
   ***************************************************************************/
   /***************************************************************************/
   /*!
    * \brief   CALLED BY PLAYER:
    *          Interface can be used to set a MuteRequest (this does not reflect
    *          the actual real state, only request system to change the state)
    * \param   [cu8SrcNum]:    Source ID
    * \param   [cu8MuteState]: wanted mute state
    * \retval  TRUE if command is being issued
    * \retval  FALSE when Parameter invalid or command couldn't be forwarded
    * \date 15.12.2010 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tBool bSetMuteState(arl_tenSource enSrcNum, arl_tenMuteAction enMuteAction, audioroutinglib::arl_tclBitmask<arl_enMuteReason> muteReason = ARL_EN_MUTE_REASON_OTHER);

   /**
    * OBSOLETE METHOD
    * This will cause EM Entry
    */
   virtual tBool bSetMuteState(arl_tenSource enSrcNum, arl_tenMuteState enMuteState);
   /***************************************************************************
    ** FUNCTION:  virtual tBool arl_tclISource::vSourceActivityResult(..
    ***************************************************************************/
   /*!
    * \brief   CALLED BY PLAYER:
    *          Interface notifies that AudioStreaming is ongoing now.
    *
    * \param   [enSrcNum]:  Source ID
    * \param   [enMuteState]:   actual source activity state
    * \retval  NONE
    * \date 15.12.2010 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tVoid vSourceActivityResult(arl_tenSource enSrcNum, const arl_tSrcActivity& enActivityAck, tU16 SubSource=0) = 0;

   virtual tVoid vOnServiceState( tU16 /*u16ServiceId*/, tU16 /*u16ServerId*/, tU16 /*u16RegisterId*/, tU8 /*u8ServiceState*/, tU16 /*u16SubId*/) {};

   virtual tVoid vOnNewAppStateNotify(tU32 u32OldAppState,tU32 u32AppState);

   virtual ail_tenCommunicationError eUpdateClients(tU16 u16FunctionId, amt_tclServiceData* poMessage = NULL) =0;

   //To remember the OSAL threadID
   OSAL_tThreadID m_s32MyThreadID;

protected:

   tVoid vRxFromOtherThreads (tU32 size, tPCUChar pcu8Data);

   tBool bIsArlServiceAvailable();


   tU16 _u16AppID;
   //@todo: Here FI ServiceID define should be inserted
   tU16 _u16ServiceID;
   tU16 _u16ServiceMajorVersion;
   tU16 _u16ServiceMinorVersion;
   tU16 _u16ServicePatchVersion;

   /// ISource Virtual FSM
  arl_tclISourceVFsm* m_poISourceVFsm;
  // Pointer to Application
  ail_tclAppInterfaceRestricted* _pAilAppInterface;
  // Pointer to Application
  ahl_tclBaseOneThreadApp* _pAhlAppInterface;


}; // class arl_tclISource : public ahl_tclBaseOneThreadService


#endif /* ARL_TCLISOURCEIF_H_ */
