/*!
*******************************************************************************
* \file              arl_tclISourceFsm.h
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements the ISource FSM.
*  - COPYRIGHT:      &copy; 2012 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 25.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added handling for routing state and dynamic MOST channel deallocation &
* error handling.
*
* \date 26.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef ARL_TCLISOURCEFSM_H_
#define ARL_TCLISOURCEFSM_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

/// OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

/// STL Includes
//#define SYSTEM_S_IMPORT_INTERFACE_EXH        ///< Import STL IF - Exception handler
//#define SYSTEM_S_IMPORT_INTERFACE_ITERATOR   ///< Import STL IF - Iterator
//#define SYSTEM_S_IMPORT_INTERFACE_LIST       ///< Import STL IF - List
//#include <stl_pif.h>


#include  <exception>
#include  <iterator>
#include  <list>
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"



#include "arl_tclFsm.h"
#include "arl_ISourceTypes.h"
#include "arl_tclISourceIF.h"
#include "arl_tclBaseClienthandler_AIL.h"

#include "arl_tclStrClientHandler_ASF_IF.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations (scope: global)
|----------------------------------------------------------------------------*/

class arl_tclISource;
class arl_tclBaseClienthandler;
class arl_tclStrClienthandler_ASF;
/*!
 * \class arl_tclISourceFsm
 * \brief iSource Finite state machine based on Serial Wait State Pattern.
 *
 * \defgroup SerialWaitStatePat Serial Wait State Pattern
 *
 * \par Intent:
 *
 * Initiating serial activity on multiple entities. Time taken to perform
 * the task is directly proportional to the number of entities involved. The
 * operation is time intensive when number of enties involved is too large.
 *
 * It give a better control over operations with multiple entities by allowing
 * only one operation at a time. The fault diagnosis is straightforward as well.
 *
 * It is also known as <b>Sequence Action State Pattern</b> and <b>Serial
 * Interaction</b>
 *
 * The pattern contains inbuilt event handlers:
 *    - Serial Message Sending Routine
 *    - Serial Response Handler
 *    - Timeout Handler
 *
 * References:
 * <A HREF="http://www.eventhelix.com/RealtimeMantra/PatternCatalog/SerialWaitStatePattern.htm">
 * Serial Wait State Pattern</A>
 *
 */

class arl_tclISourceFsm : protected arl_tclFsm
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /// Using the "Name injection" to make the operator() visible to external
   /// interface use.
   using arl_tclFsm::operator ();

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceFsm::arl_tclISourceFsm(arl_tclISource* const ...
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection
    *          Principle (DIP)
    * \param   [cpoISource]:        (->I)  Pointer to iSource Service
    * \param   [cpoStrRouClient]:   (->I)  Stream router client
    * \param   [cpoAVManClient]:    (->I)  AVManager client
    * \retval  NONE
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceFsm // for ASF
   (
      tU8 u8SrcNum
      , arl_tclISourceIF* const cpoISource
      , arl_tclStrClientHandler_ASF_IF* const cpoStrRouClient
      , arl_tclBaseClienthandler* const cpoAVManClient = OSAL_NULL
   );
   arl_tclISourceFsm // for AHL
   (
      tU8 u8SrcNum
      , arl_tclISourceIF* const cpoISource
      , arl_tclBaseClienthandler* const cpoStrRouClient
      , arl_tclBaseClienthandler* const cpoAVManClient = OSAL_NULL
   );

   arl_tclISourceFsm // for AIL
      (
         tU8 u8SrcNum
         , arl_tclISourceIF* const cpoISource
         , arl_tclBaseClienthandlerAIL* const cpoStrRouClient
         , arl_tclBaseClienthandler* const cpoAVManClient = OSAL_NULL
      );

   /***************************************************************************
   ** FUNCTION:  virtual arl_tclISourceFsm::~arl_tclISourceFsm()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~arl_tclISourceFsm();

   /***************************************************************************
   ** FUNCTION:  inline tBool arl_tclISourceFsm::bIsSelfTrig() const
   ***************************************************************************/
   /*!
    * \brief   Check whether an iSource is self triggered or not?
    * \param   NONE
    * \retval  [tBool]: TRUE if iSource is self triggered, FALSE otherwise.
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   inline tBool bIsSelfTrig() const { return m_oISrcProp.m_bIsSelfTrig; };

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnAllocStart(const arl_tclISrcAllo..
   ***************************************************************************/
   /*!
    * \brief   Function to handle Allocate.
    * \param   [rfcoAllocateStart]: (I)   Allocate Start object.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnAllocStart(const arl_tISrcAllocStart& rfcoAllocateStart);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnDeAllocStart(const arl_tISrcDeal..
   ***************************************************************************/
   /*!
    * \brief   Function to handle DeAllocate.
    * \param   [rfcoDeallocStart]:  (I)   Deallocate Start object.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnDeAllocStart(const arl_tISrcDeallocStart& rfcoDeallocStart);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnSrcActStart(const arl_tISrcAct..
   ***************************************************************************/
   /*!
    * \brief   Function to handle source activity.
    * \param   [rfcoISrcActStart]:  (I)   Pointer to incoming message.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnSrcActStart(const arl_tISrcActivityStart& rfcoISrcActStart);

   tVoid vOnSrcActTimerExpiry();
   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnAllocRouteResult(const arl_tAllo..
   ***************************************************************************/
   /*!
    * \brief   Function to handle allocate route result.
    * \param   [rfcoAllocRouteResult]: (I)   Allocate route result.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnAllocRouteResult(const arl_tAllocRouteResult& rfcoAllocRouteResult);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnStartUpAllocRouteResult(const tU8..)
   ***************************************************************************/
   /*!
    * \brief   Function to handle startup allocate route result.
    * \param   [u8SrcNum]: (I)   Source Number.
    * \retval  NONE
    * \date    31.01.2012 \author Raghavendra S (RBEI/ECP2)
    **************************************************************************/
   tVoid vOnStartUpAllocRouteResult(const tU16 u16SrcNum);


   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnDeAllocRouteResult(const arl_.
   ***************************************************************************/
   /*!
    * \brief   Function to handle deallocate route result.
    * \param   [rfcoDeAllocRouteResult]: (I)   Deallocate route result.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnDeAllocRouteResult(const arl_tDeAllocRouteResult& rfcoDeAllocRouteResult);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnSetPlayResult(const arl_tS..
   ***************************************************************************/
   /*!
    * \brief   Function to handle set play result.
    * \param   [rfcoSetPlayResult]:    (I)   Set Play result.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnSetPlayResult(const arl_tSetPlayResult& rfcoSetPlayResult);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnSetStopResult(const arl_tSe..
   ***************************************************************************/
   /*!
    * \brief   Function to handle set stop result.
    * \param   [rfcoSetStopResult]:    (I)   Set Stop result.
    * \retval  NONE
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnSetStopResult(const arl_tSetStopResult& rfcoSetStopResult);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vOnRouteState(const arl_tSe..
   ***************************************************************************/
   /*!
    * \brief   Function to handle route state status.
    * \param   [rfcoRouteState]:    (I)   Route State status
    * \retval  NONE
    * \date 12.08.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnRouteState(const arl_tRouteState& rfcoRouteState);

   tU8 u8GetSourceNumber(tVoid) {return m_u8SrcNum;}

   /*!
    * \typedef struct arl_rISrcFsmTrigIds arl_tISrcFsmTrigIds
    * Data structure for holding all the iSource FSM Trigger id's
    */

   typedef struct arl_rISrcFsmTrigIds
   {

      /// Source Activity request.
      tS16 m_s16SrcActivity;
      /// Request activation trigger id.
      tS16 m_s16ReqAvAct;
      /// Request activation error trigger id.
      tS16 m_s16ReqAvActErr;
      /// Request activation denied trigger id.
      tS16 m_s16ReqAvActDenied;
      /// Request activation granted trigger id.
      tS16 m_s16ReqAvActGranted;
      /// Request deactivation trigger id.
      tS16 m_s16ReqAvDeAct;
      /// Request deactivation error trigger id.
      tS16 m_s16ReqAvDeActErr;
      /// Request deactivation method result trigger id.
      tS16 m_s16ReqAvDeActMr;
      /// Request Stream router allocate trigger id.
      tS16 m_s16StrRouAllocate;
      /// Stream router allocate method result trigger id.
      tS16 m_s16StrRouAllocateMr;
      ///Stream router allocate error trigger id.
      tS16 m_s16StrRouAllocateErr;
      /// Request Stream router deallocate trigger id.
      tS16 m_s16StrRouDeAllocate;
      /// Stream router deallocate method result trigger id.
      tS16 m_s16StrRouDeAllocateMr;
      /// Stream router deallocate error trigger id.
      tS16 m_s16StrRouDeAllocateErr;
      /// Request Stream router set play trigger id.
      tS16 m_s16StrRouSetPlay;
      /// Stream router set play method result trigger id.
      tS16 m_s16StrRouSetPlayMr;
      /// Stream router set play error trigger id.
      tS16 m_s16StrRouSetPlayErr;
      /// Request Stream router set stop trigger id.
      tS16 m_s16StrRouSetStop;
      /// Stream router set stop method result trigger id.
      tS16 m_s16StrRouSetStopMr;
      /// Stream router set stop error trigger id.
      tS16 m_s16StrRouSetStopErr;
      /// Message post error trigger id.
      tS16 m_s16MsgPostErr;
      /// No result message received & a timed out is trigged. Time out trigger id.
      tS16 m_s16MsgPostTimeOut;
      /// AV Manager available trigger id.
      tS16 m_s16AVManAvail;
      /// AV Manager unavailable trigger id.
      tS16 m_s16AVManUnavail;
      /// Stream router available trigger id.
      tS16 m_s16StrRouAvail;
      /// Stream router unavailable trigger id.
      tS16 m_s16StrRouUnavail;
      /// Application Activity acknowledge
      tS16 m_s16AppAckActivityPlay;
      /// Application Activity acknowledge
      tS16 m_s16AppAckActivityStop;
      /// Application Activity acknowledge
      tS16 m_s16AppAckActivityPause;
      /// Application state change to Off
      tS16 m_s16AppState_Off;

      /************************************************************************
      ** FUNCTION:  arl_rISrcFsmTrigIds::arl_rISrcFsmTrigIds()
      ************************************************************************/
      /*!
       * \brief   Default Constructor
       * \param   NONE
       * \retval  NONE
       * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
       ***********************************************************************/
      arl_rISrcFsmTrigIds():
         m_s16SrcActivity(ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvAct(ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvActErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvActDenied(ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvActGranted(ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvDeAct (ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvDeActErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16ReqAvDeActMr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouAllocate(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouAllocateMr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouAllocateErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouDeAllocate(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouDeAllocateMr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouDeAllocateErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouSetPlay(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouSetPlayMr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouSetPlayErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouSetStop(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouSetStopMr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouSetStopErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16MsgPostErr(ARL_FSM_TRIG_ID_INVALID)
         , m_s16MsgPostTimeOut(ARL_FSM_TRIG_ID_INVALID)
         , m_s16AVManAvail(ARL_FSM_TRIG_ID_INVALID)
         , m_s16AVManUnavail(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouAvail(ARL_FSM_TRIG_ID_INVALID)
         , m_s16StrRouUnavail(ARL_FSM_TRIG_ID_INVALID)
         , m_s16AppAckActivityPlay(ARL_FSM_TRIG_ID_INVALID)
         , m_s16AppAckActivityStop(ARL_FSM_TRIG_ID_INVALID)
         , m_s16AppAckActivityPause(ARL_FSM_TRIG_ID_INVALID)
         , m_s16AppState_Off(ARL_FSM_TRIG_ID_INVALID)
      {}
   } arl_tISrcFsmTrigIds;

   /***************************************************************************
   ** FUNCTION:   const arl_tISrcFsmTrigIds& arl_tclISourceFsm::rfcoGetTrigge..
   ***************************************************************************/
   /*!
    * \brief   Function to get Trigger id's.
    * \param   NONE
    * \retval  [const arl_tISrcFsmTrigIds&]: Reference to trigger id's
    * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   const arl_tISrcFsmTrigIds& rfcoGetTriggerId() const;

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bSendStrRouMsg(const fi_tclMessageB..
   ***************************************************************************/
   /*!
    * \brief   Function to send a message to Stream router.
    * \param   [rfcoMsgBase]: (I) Message to be send.
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 01.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bSendStrRouMsg(const fi_tclMessageBase& rfcoMsgBase);

   // for ASF we need the parameters and not the message
   tBool bSendStrRouMsgAllocateRoute(tU16 ucid, tS32 STM_MsgBufID, tU16 u16Sink);
   tBool bSendStrRouMsgDeAllocateRoute(tU16 ucidD);
   tBool bSendStrRouMsgSetPlay(tU16 ucid);
   tBool bSendStrRouMsgSetStop(tU16 ucid);

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnDeAllocErr()
   ***************************************************************************/
   /*!
    * \brief   Notify Trigger the error message on receiving deallocate error.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 12.12.2012 \author Ruben Volkmer (mailto: R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tVoid vActResult(arl_tenActivity act);
   /***************************************************************************
   * FUNCTION: arl_tclISourceFsm::vSetstatefsm (tS16 s16StateId)
   *Function only used for the purpose of ARL UNIT testing to set ARL states.
   *strictly recommended not to use for other purposes.
   ****************************************************************************/
   tVoid vSetstatefsm(tS16 s16StateId);
   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/



   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bSendAVManMsg(const fi_tclMessageB..
   ***************************************************************************/
   /*!
    * \brief   Function to send a message to AV Manager.
    * \param   [rfcoMsgBase]: (I) Message to be send.
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 01.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bSendAVManMsg(const fi_tclMessageBase& rfcoMsgBase);

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceFsm::arl_tclISourceFsm()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So
    *          that any attempt to create without parameter will be caught by
    *          the compiler.
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceFsm();       // No definition exists.

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceFsm::arl_tclISourceFsm(const arl_tclISourceFsm&
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoISrcFsm]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So
    *          that any attempt to copy will be caught by the compiler.
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceFsm(const arl_tclISourceFsm& rfcoISrcFsm);    // No definition exists.

   /***************************************************************************
   ** FUNCTION:  arl_tclISourceFsm& arl_tclISourceFsm::operator=(const arl_t..
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoISrcFsm]: (I) Const reference to object to be copied
    * \retval  [arl_tclISourceFsm&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So
    *          that any attempt for assignment will be caught by the compiler.
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISourceFsm& operator= (const arl_tclISourceFsm& rfcoISrcFsm);    // No definition exists.

   /***************************************************************************
   ** FUNCTION:  tVoid vHandleTimerCallBack(tPVoid pvArg).
   ***************************************************************************/
   /*!
    * \brief   Timer Callback Function
    * \param   [pvArg]: (I) Timer callback argument
    * \retval  NONE.
    * \date    31.01.2012 \author Raghavendra S (RBEI/ECP2)
    **************************************************************************/
   static tVoid vHandleTimerCallBack(tPVoid pvArg);

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISourceFsm::vSetupFsm()
   ***************************************************************************/
   /*!
    * \brief   Setup the Finite state machine (FSM).
    * \param   NONE
    * \retval  NONE
    * \date 26.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vSetupFsm();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendAllocRoute()
   ***************************************************************************/
   /*!
    * \brief   Trigger the allocation of a route.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 01.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendAllocRoute();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendAllocRouteOnStartUp()
   ***************************************************************************/
   /*!
    * \brief   Trigger the early allocation of a route on Startup.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 12.12.2011 \author Raghavendra S (RBEI/ECV)
    **************************************************************************/
   virtual tBool bSendAllocRouteOnStartUp();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendDeAllocRoute()
   ***************************************************************************/
   /*!
    * \brief   Trigger the deallocation of a route.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 01.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendDeAllocRoute();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendSetPlay()
   ***************************************************************************/
   /*!
    * \brief   Trigger the set play for an allocated route.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 01.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendSetPlay();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendSetStop()
   ***************************************************************************/
   /*!
    * \brief   Trigger the set stop for an allocated route.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 01.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendSetStop();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bSendSrcActToApp()
   ***************************************************************************/
   /**
    * \brief Forwards the SourceActivity Request to Application
    *
    * \return return value of Application
    */
   virtual tBool bSendSrcActToApp();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnStrRouUnavailable()
   ***************************************************************************/
   /*!
    * \brief   When StreamRouterService got unavailable
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 24.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnStrRouUnavailable();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAllocate()
   ***************************************************************************/
   /*!
    * \brief   Set the appropriate trigger to the FSM on AppState
    *          change to Off.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 18.04.2013 \author Ruben Volkmer mailto:R.Volkmer@ITB-solutions.de
    **************************************************************************/
   virtual tBool bOnAppStateOff();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAllocate()
   ***************************************************************************/
   /*!
    * \brief   Set the appropriate trigger to the FSM on Allocate/Deallocate.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 24.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnAllocate();

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bOnSrcActivity()
   ***************************************************************************/
   /*!
    * \brief   Set the appropriate trigger to the FSM on source activity.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 24.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnSrcActivity();

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bSendSrcActivResult()
   ***************************************************************************/
   /*!
    * \brief   Set trigger to the FSM to send source activity result.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 24.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendSrcActivResult();

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bSendAllocateResult()
   ***************************************************************************/
   /*!
    * \brief   Set trigger to the FSM to send allocate result.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 24.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendAllocateResult();

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bSendDeAllocateResult()
   ***************************************************************************/
   /*!
    * \brief   Set trigger to the FSM to send deallocate result.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 24.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendDeAllocateResult();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnAllocErr()
   ***************************************************************************/
   /*!
    * \brief   Trigger the error message on receiving allocate error.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 23.08.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnAllocErr();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnDeAllocErr()
   ***************************************************************************/
   /*!
    * \brief   Trigger the error message on receiving deallocate error.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 23.08.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnDeAllocErr();

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISourceFsm::bOnSrcActivityErr()
   ***************************************************************************/
   /*!
    * \brief   Trigger the error message on receiving set play/stop error.
    * \param   NONE
    * \retval  [tBool], TRUE if successful, FALSE otherwise.
    * \date 23.08.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnSrcActivityErr();

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bServiceAvailable()
   ***************************************************************************/
   /*!
    * \brief   Check for service availability
    * \param   NONE
    * \retval  [tBool], TRUE if service available, FALSE otherwise.
    * \date 02.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bServiceAvailable();


   tBool bTimerStop ();
   tBool bTimerStart(OSAL_tpfCallback callback,tPVoid Parameter);
   /***************************************************************************
   ** FUNCTION:  tBool arl_tclISourceFsm::bValidateStateExit_AppAckActivityStopPause()
   ***************************************************************************/
   /*!
    * \brief   Check if exiting the state APP_ACK_STOPPAUSE is allowed on the current acknowledgement provided by application
    * \param   NONE
    * \retval  [tBool], TRUE if Allowed, FALSE otherwise.
    **************************************************************************/
   tBool bValidateStateExit_AppAckActivityStopPause();

   /*!
    * \typedef struct arl_rISrcFsmStateIds arl_tISrcFsmStateIds
    * Data structure for holding all the iSource FSM State id's
    */

   typedef struct arl_rISrcFsmStateIds
   {
      /// Super state for FSM
      tS16 m_s16Super;
      /// Wait state after request AV activation.
      tS16 m_s16ReqAVAct;
      /// Wait state after request AV deactivation.
      tS16 m_s16ReqAVDeAct;
      /// Wait state after request for Allocate route.
      tS16 m_s16StrRouAlloc;
      /// Wait state after request for Deallocate route.
      tS16 m_s16StrRouDeAlloc;
      /// Wait state after request for Set Play command.
      tS16 m_s16StrRouSetPlay;
      /// Wait state after request for Set Stop command.
      tS16 m_s16StrRouSetStop;
      /// Wait state after request for Set Stop command.
      tS16 m_s16AppAckActivityPlay;
      /// Processing state once requested for Source activity.
      tS16 m_s16AppAckActivityStopPause;
      /// Processing state once requested for Source activity.
      tS16 m_s16SrcAct;
      /// ISource unavailable state due to unavailability of any stake holder.
      tS16 m_s16SrvUnavail;
      /// ISource available state
      tS16 m_s16SrvAvail;

      /************************************************************************
      ** FUNCTION:  arl_rISrcFsmStateIds::arl_rISrcFsmStateIds()
      ************************************************************************/
      /*!
       * \brief   Default Constructor
       * \param   NONE
       * \retval  NONE
       * \date 11.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
       ***********************************************************************/
      arl_rISrcFsmStateIds():m_s16Super(ARL_FSM_STATE_ID_INVALID)
         , m_s16ReqAVAct(ARL_FSM_STATE_ID_INVALID)
         , m_s16ReqAVDeAct(ARL_FSM_STATE_ID_INVALID)
         , m_s16StrRouAlloc(ARL_FSM_STATE_ID_INVALID)
         , m_s16StrRouDeAlloc(ARL_FSM_STATE_ID_INVALID)
         , m_s16StrRouSetPlay(ARL_FSM_STATE_ID_INVALID)
         , m_s16StrRouSetStop(ARL_FSM_STATE_ID_INVALID)
         , m_s16AppAckActivityPlay(ARL_FSM_STATE_ID_INVALID)
         , m_s16AppAckActivityStopPause(ARL_FSM_STATE_ID_INVALID)
         , m_s16SrcAct(ARL_FSM_STATE_ID_INVALID)
         , m_s16SrvUnavail(ARL_FSM_STATE_ID_INVALID)
         , m_s16SrvAvail(ARL_FSM_STATE_ID_INVALID)
      {}

   } arl_tISrcFsmStateIds;

   /*!
    * \typedef list<arl_tclMsgIdentifiers> arl_tIMsgTrackerList
    * Data structure for holding ISource FSM's.
    */

   typedef std::list<arl_tclMsgIdentifiers> arl_tIMsgTrackerList;

   /*!
    * \typedef list<arl_tclMsgIdentifiers>::iterator arl_tIMsgTrackerListIter
    * Data structure for holding ISource FSM's.
    */

   typedef std::list<arl_tclMsgIdentifiers>::iterator arl_tIMsgTrackerListIter;

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Source Number.
   tU8 m_u8SrcNum;
   

   /// Allocate Route on Startup
   arl_tenStartUpRouState m_enAllocateOnStart;
   /// Source allocated flag
   tBool m_bAllocated;
   /// Source alloc state flag
   arl_tenRouteState m_enRouteState;
   /// Source Activity flag
   arl_tSrcActivity m_oSrcActivity;
   /// Timer Handle
   OSAL_tTimerHandle m_hTimer;

   /// Last source allocated request flag
   tBool m_bLastAllocReq;
   /// Last Source Activity request flag
   arl_tSrcActivity m_oLastSrcActivReq;
   /// Source Properties.
   arl_rISrcProp m_oISrcProp;
   /// Source information.
   //arl_tISourceInfo m_oSrcInfo;
   /// ISource Service pointer.
   arl_tclISourceIF* const m_cpoISource;
   /// Stream router client pointer.
   arl_tclBaseClienthandler*      const m_cpoStrRouClient;
   arl_tclBaseClienthandlerAIL*   const m_cpoStrRouClientAIL;
   arl_tclStrClientHandler_ASF_IF*  const m_cpoStrRouClientASF;

   /// AV Manager client pointer.
   arl_tclBaseClienthandler* const m_cpoAVManClient;
   /// State Ids
   arl_tISrcFsmStateIds m_oStateId;
   /// Trigger Ids
   arl_tISrcFsmTrigIds m_oTrigId;
   /// Functor lists to keep track of allocated functors to free at destruction
   arl_tFunctorList m_oFunctList;
   tU16 m_u16SubSource;
   tU16 m_u16Sink;
   tU8 m_u8PosNextSrc;
   /// Last Source Activity response received from application
   arl_tSrcActivity m_oLastAppAckSrcActivity;
   /// Message tracker list.
   arl_tIMsgTrackerList m_oMsgTrackerList;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class arl_tclISourceFsm : protected arl_tclFsm

#endif   // #ifndef ARL_TCLISOURCEFSM_H_

// <EOF>
