/*!
*******************************************************************************
* \file              arl_tclISource.h
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements the ISource interface.
*  - COPYRIGHT:      &copy; 2010 - 2012 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 25.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - New Error enumerations for iSource errors.
* - Removed the overriden eUpdateClients() from FWK.
*
* \date 15.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/
#ifndef ARL_TCLISOURCE_H_
#define ARL_TCLISOURCE_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

/// OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

/*!
 * \note
 *    - Interface: BASE_TYPES - Definition of the basic framework parts
 *    used by all FIs.
 *    - Interface: MESSAGE - Definition of an AMT class suitable but not
 *    exclusively usable together with the visitor and type framework.
 */
#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include <common_fi_if.h>

#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include <ahl_if.h>

#include "arl_ISourceTypes.h"
#include "arl_tclISourceIF.h"

#include "arl_tclStrClientHandler_ASF_IF.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

#if __cplusplus < 201103L
#define NULLPTR NULL //C++03 or below
#define OVERRIDE  //C++03 or below
#define FINAL
#else
#define NULLPTR nullptr //C++11 or above
#define OVERRIDE override //C++11 or above
#define FINAL final
#endif

/*!
 * \define  ARL_S32_DEFAULT_MSG_BUFFFER_ID
 * \description Default message buffer id, which might be required for playback
 * notification. (Applicable only for ISource applications on TEngine)
 */

#define ARL_S32_DEFAULT_MSG_BUFFFER_ID                ((tS32)-1)

/**
 * \Remove this
 */
#define ARL_SERVICE_ID  0x95

/*!
 * \define  BEGIN_ARL_MSG_MAP(theClass)
 * \description ARL Specific message map to overcome the CCA Message map bug (?).
 * This macro chains the message map from the derived class to the base class.
 *
 * \param   [theClass]:    (I)   class name
 */

#define BEGIN_ARL_MSG_MAP(theClass) \
   const mm_tMsgMap* theClass::GetMsgMap() const \
      { return &theClass::msgMyMap; } \
   const mm_tMsgMap theClass::msgMyMap = \
   { &arl_tclISource::msgMyMap, &theClass::_msgMyEntries[0] }; \
   const mm_tMsgMapEntry theClass::_msgMyEntries[] = \
   { \


/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/


/*!
 * \typedef enum arl_tenAVAction
 * \brief AV Manager Action.
 */

typedef enum
{
   ARL_EN_ISRC_REQACT   =  0           ///< [0] Request AV activation.
   , ARL_EN_ISRC_REQDEACT              ///< [1] Request AV deactivation.
} arl_tenAVAction;

/******************************************************************************
| forward declarations
|----------------------------------------------------------------------------*/

class arl_tclISourceVFsm;     ///< ISource Virtual Finite State machine

/*!
 * \class arl_tclISource
 * \brief Base class which implements the iSource Interface.
 *
 */

class arl_tclISource : public ahl_tclBaseOneThreadService , public arl_tclISourceIF
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/
   virtual tVoid vCCA_OnUpRegRouteReq(amt_tclServiceData* poMessage) OVERRIDE
   {
      arl_tclISourceIF::vCCA_OnUpRegRouteReq(poMessage);
   }

   virtual tVoid vCCA_OnMuteState(amt_tclServiceData* poMessage) OVERRIDE
   {
      arl_tclISourceIF::vCCA_OnMuteState(poMessage);
   }


   virtual tVoid vCCA_OnUpRegMuteReq(amt_tclServiceData* poMessage) OVERRIDE
   {
      arl_tclISourceIF::vCCA_OnUpRegMuteReq(poMessage);
   }

   /***************************************************************************
   ** FUNCTION:  arl_tclISource::arl_tclISource(ahl_tclBaseOneThreadApp* co...
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection
    *          Principle (DIP)
    * \param   cpoApp:            (->I)  Pointer to the main application
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISource(ahl_tclBaseOneThreadApp* const cpoApp);

   // here we expect to get a pointer to the real ASF extension
   arl_tclISource(ahl_tclBaseOneThreadApp* const cpoApp, arl_tclStrClientHandler_ASF_IF* poAsfStrClh);

   /***************************************************************************
   ** FUNCTION:  virtual arl_tclISource::~arl_tclISource()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~arl_tclISource();

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vOnTimer(tU16 u16TimerId)
   ***************************************************************************/
   /*!
    * \brief   Timer expiry handler.
    * \param   u16TimerId:    (I) Timer Id
    * \retval  NONE
    * \date 08.03.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnTimer(tU16 /*u16TimerId*/) OVERRIDE {};

   /***************************************************************************
   ** FUNCTION:  virtual tS32 arl_tclISource::s32GetMsgBuffId(tU8 u8SrcNum)const
   ***************************************************************************/
   /*!
    * \brief   Gets Message buffer id.
    * \param   u8SrcNum:  (I) Source Number.
    * \retval  tBool: TRUE, if source activity was successful, FALSE otherwise
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tS32 s32GetMsgBuffId(tU8 /*u8SrcNum*/) const OVERRIDE
      {return ARL_S32_DEFAULT_MSG_BUFFFER_ID;};

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vOnError()
   ***************************************************************************/
   /*!
    * \brief   Function callback to trigger actions on request AV Activation
    *          error
    * \note    This function needs to be overloaded by the derived class only in
    *          case the source is self triggered & needs to send error message
    *          to clients based on source number.
    * \param   u8SrcNum:  (I) Source Number.
    * \param   cenError:  (I) ISource Error type.
    * \retval  NONE
    * \date 10.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnError(tU8 /*u8SrcNum*/
      , const arl_tenISourceError /*cenError*/) OVERRIDE {};

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vSendMessage( const amt_tclSe..
   ***************************************************************************/
   /*!
    * \brief   Generic Interface to send messages to next clients.
    * \param   rfcoServiceData: (I)   Service data to be sent.
    * \param   rfcoMsgBase:     (I)   Reference to the Fi object - Base
    *          class of the Fi object.
    * \retval  tBool: TRUE if message post successful, FALSE otherwise.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSendMessage 
   (
      const amt_tclServiceData &rfcoServiceData
      , const fi_tclMessageBase& rfcoMsgBase
   ) OVERRIDE;



   /***************************************************************************
    ** FUNCTION:  virtual tBool arl_tclISource::bOnMuteState(..
    ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Interface gets triggered when AudioRouting changed Mute state
    *          for given Source
    * \param   enSrcNum: Source ID
    * \param   enMuteState: systems actual mute state
    * \retval  TRUE
    * \retval  FALSE
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tBool bOnMuteState(arl_tenSource /*enSrcNum*/, arl_tenMuteState /*enMuteState*/) OVERRIDE
   {
      return FALSE;
   };

   virtual tBool bOnMuteState(arl_tenSource /*enSrcNum*/, tU16 /*SubSource*/, arl_tenMuteState /*enMuteState*/)
   {
      return FALSE;
   };

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISource::bOnSrcActivity(const arl..
   ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Application specific function on Source Activity start.
    * \param   enSrcNum:  (I) Source Number.
    * \param   rfcoSrcActivity: (I) Source Activity
    * \retval  tBool: TRUE, if source activity was successful, FALSE otherwise
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 15.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnSrcActivity(arl_tenSource /*enSrcNum*/, const arl_tSrcActivity& /*rfcoSrcActivity*/) OVERRIDE
   {
      return FALSE;
   };

   virtual tBool bOnSrcActivity(arl_tenSource /*enSrcNum*/, tU16 /*SubSource*/, const arl_tSrcActivity& /*rfcoSrcActivity*/) OVERRIDE
   {
      return FALSE;
   };

   virtual tBool bOnSrcActivity(arl_tenSource /*enSrcNum*/, tU16 /*SubSource*/, arl_tenSource /*enPosNextSrc*/, const arl_tSrcActivity& /*rfcoSrcActivity*/) OVERRIDE
   {
      return FALSE;
   };

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISource::bOnAllocate(..
   ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          Application specific function after Allocate is processed.
    * \param   [enSrcNum]:  (I) Source Number.
    * \param   rfcoAllocRoute: (I) Reference to Allocate route result
    * \retval  tBool: TRUE, if Application performed operations successfully,
    *          FALSE otherwise
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 15.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnAllocate(arl_tenSource /*enSrcNum*/, const arl_tAllocRouteResult& /*rfcoAllocRoute*/) OVERRIDE
   {
      return FALSE;
   };

   virtual tBool bOnAllocate(arl_tenSource /*enSrcNum*/, tU16 /*sink*/, const arl_tAllocRouteResult& /*rfcoAllocRoute*/)
   {
      return FALSE;
   };
   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISource::bOnDeAllocate(
   ***************************************************************************/
   /*!
    * \brief   CALLED BY AUDIO-ROUTING-LIB:
    *          May be overridden by Player App to Release Resources.
    *          Application specific function after DeAllocate is processed.
    * \param   enSrcNum:  (I) Source Number.
    * \retval  tBool: TRUE, if Application performed operations successfully,
    *          FALSE otherwise
    * \date 25.07.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    * \date 15.02.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bOnDeAllocate(arl_tenSource /*enSrcNum*/) OVERRIDE
   {
      return FALSE;
   };

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::bSetMuteState(arl_tenSource ..
   ***************************************************************************/
   /***************************************************************************/
   /*!
    * \brief   CALLED BY PLAYER:
    *          Interface can be used to set a MuteRequest (this does not reflect
    *          the actual real state, only request system to change the state)
    * \param   cu8SrcNum:    Source ID
    * \param   cu8MuteState: wanted mute state
    * \param   [muteReason]: root cause of the request, by default ARL_EN_MUTE_REASON_OTHER is assumed
    * \retval  TRUE if command is being issued
    * \retval  FALSE when Parameter invalid or command couldn't be forwarded
    * \date 05.11.2012 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tBool bSetMuteState(arl_tenSource enSrcNum, arl_tenMuteAction enMuteAction, audioroutinglib::arl_tclBitmask<arl_enMuteReason> muteReason = ARL_EN_MUTE_REASON_OTHER) OVERRIDE
   { return arl_tclISourceIF::bSetMuteState(enSrcNum, enMuteAction, muteReason); }

   //OBSOLETE, due to task compiler warning removal this has to be here.
   virtual tBool bSetMuteState(arl_tenSource /*enSrcNum*/, arl_tenMuteState /*enMuteState*/) OVERRIDE
   { return FALSE; }

   /***************************************************************************
   ** FUNCTION:  virtual tBool bSendFGGraphCmd(arl_tenSource...
   ***************************************************************************/
   /***************************************************************************/
   /*!
    * \brief   CALLED BY PLAYER:
    *          Interface can be used to execute commands on the Filtergraph
    *          for given source.
    * @param enSrcNum command is executed on filtergraph of this source
    * @param enFGCmd command to be executed
    * @return TRUE when command was send, FALSE when command command could not be send
    * (e.g. StreamRouter Service not available)
    */
   virtual tBool bSendFGGraphCmd(arl_tenSource enSrcNum, arl_tenFGCommand enFGCmd) OVERRIDE
   {
      return arl_tclISourceIF::bSendFGGraphCmd(enSrcNum, enFGCmd);
   }

   /***************************************************************************
   ** FUNCTION:  virtual tBool vOnFGGraphCmdResult(arl_tenFGGra...
   ***************************************************************************/
   /***************************************************************************/
   /*!
   * \brief   CALLED BY AUDIO-ROUTING-LIB:
   *          Called when MethodResult from StreamRouter is incoming
   * @param enErrorCode Information if FilterGraph GraphCommand was executed
   * successfully
   */
   virtual tVoid vOnFGGraphCmdResult(arl_tenFGGraphCmd_Result enErrorCode) OVERRIDE;


   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::bSetAudioRouteRequest(...
   ***************************************************************************/
   /*!
   * \brief   CALLED BY PLAYER:
   *          Signals a route de-/allocate request,
   *          initially all requests are "off".
   *
   * \param   enSource Source the request is valid for
   * \param   enRequest the Requested state, this is initially OFF
   * \retval  NONE
   * \date 22.11.2012 \author Ruben Volkmer (mailto:R.Volkmer@ITB-Solutions.de)
   **************************************************************************/
   virtual tBool bSetAudioRouteRequest(arl_tenSource enSource, arl_tenActivity enRequest, audioroutinglib::arl_tclBitmask<arl_enRouteReqReason> Reason = ARL_EN_ROUTEREQ_REASON_OTHER)
   {
      //return arl_tclISourceIF::bSetAudioRouteRequest(enSource, enRequest, Reason);
    return arl_tclISourceIF::bSetAudioRouteRequest(SourceID(enSource,0), enRequest, Reason);
   }
   //daw2hi: overloaded with SourceID which has source and subsource
   virtual tBool bSetAudioRouteRequest(SourceID SrcId, arl_tenActivity enRequest, audioroutinglib::arl_tclBitmask<arl_enRouteReqReason> Reason = ARL_EN_ROUTEREQ_REASON_OTHER) OVERRIDE
   {
      return arl_tclISourceIF::bSetAudioRouteRequest(SrcId, enRequest, Reason);
   }

   virtual tBool bOnAudioRouteRequestReject(arl_tenSource /*enSource*/, tU16 /*SubSource*/) OVERRIDE
   {

     return FALSE;
   }
   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vSetSrcAvailable(tBool bSrcAvail)
   ***************************************************************************/
   /*!
    * \brief   Interface can be used to set the Source availability
    * \param   srcAvailability:   availability state of given source
    * \param   availabilityReason: reason for the change of availability
    * \param   enSource: source of that the available state changed
    * \param   u16SubSource: optional the subsource ID (may be identical to DeviceID)
    * \retval  NONE
    * \date 26.11.2013 \author Ruben Volkmer (mailto:R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tBool bSetSourceAvailability(arl_tenSrcAvailability srcAvailability, arl_tenAvailabilityReason availabilityReason, arl_tenSource enSource, tU16 u16SubSource = 0) OVERRIDE
   {
      return arl_tclISourceIF::bSetSourceAvailability(srcAvailability, availabilityReason, enSource, u16SubSource);
   }


   /***************************************************************************
    ** FUNCTION:  virtual tBool arl_tclISource::vSourceActivityResult(..
    ***************************************************************************/
   /*!
    * \brief   CALLED BY PLAYER:
    *          Interface notifies that AudioStreaming is ongoing now.
    *
    * \param   enSrcNum:  Source ID
    * \param   enActivityAck:   actual source activity state
    * \retval  NONE
    * \date 15.12.2010 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
    **************************************************************************/
   virtual tVoid vSourceActivityResult(arl_tenSource enSrcNum, const arl_tSrcActivity& enActivityAck, tU16 SubSource=0) OVERRIDE;
   virtual tVoid vSourceActivityResult(arl_tenSource enSrcNum, arl_tenActivity enActivityAck, tU16 SubSource=0);

   /***************************************************************************
   ** FUNCTION:  virtual tVoid arl_tclISource::vSetSrcAvailable(tBool bSrcAvail)
   ***************************************************************************/
   /*!
    * \brief   CALLED BY PLAYER:
    *          Interface can be used to set the Source availability
    * \param   [bSrcAvail]:   (I)   Source available or not
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vSetSrcAvailable(tBool bSrcAvail = TRUE) OVERRIDE;
   virtual tBool bSetAudioProperty(arl_tenSource enSrcNum, arl_tenSourcePropType PropType, tS32 PropValue, tU16 SubSource=0) OVERRIDE
   {
    return arl_tclISourceIF::bSetAudioProperty(enSrcNum, PropType, PropValue, SubSource);
   }

   virtual ail_tenCommunicationError eUpdateClients(tU16 u16FunctionId, amt_tclServiceData* poMessage = NULL) OVERRIDE
   {
      return ahl_tclBaseOneThreadService::eUpdateClients(u16FunctionId, poMessage);
   }


   virtual tVoid vOnNewAppStateNotify(tU32 u32OldAppState,tU32 u32AppState) OVERRIDE;

   virtual amt_tclServiceData* pStatusMessageFactory(tU16 /*u16Target*/, tU16 /*u16FunctionId*/, tU16 /*u16RegisterID*/, tU16 /*u16CmdCounter*/) OVERRIDE;

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/


   /// Using the "Name injection" to enable the compiler's overload resolution
   /// algorithm to see the base class methods.
   /// "Name injection" removes the compiler warning  #611-D: overloaded virtual
   /// function "ahl_tclBaseOneThreadService::vOnLoopback" is only partially
   /// overridden in class "audcu_tclMainApp"
   using ahl_tclBaseOneThreadService::vOnLoopback;




   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnLoopback(tU16 u16ServiceID, amt_tc...
   ***************************************************************************/
   /*!
    * \brief   Loopback service ISource interface.
    * \param   [u16ServiceID]: (I) Service Id
    * \param   [poMessage]:    (->I) Pointer to incoming message.
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnLoopback(tU16 u16ServiceID, amt_tclServiceData* poMessage) OVERRIDE;

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclISource::bSrcAvailable() const
   ***************************************************************************/
   /*!
    * \brief   Interface provides whether source is available or not
    * \param   NONE
    * \retval  [tBool]: TRUE, if source is available, FALSE otherwise.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bSrcAvailable() const OVERRIDE;

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnMsISource(amt_tclServiceD..
   ***************************************************************************/
   /*!
    * \brief   Method starts for the ISource based on MOST spec.
    * \param   [poMessage]:    (->I)   Pointer to service data.
    * \retval  NONE
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnMsISource(amt_tclServiceData* poMessage) OVERRIDE;

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclISource::vOnMrISource(amt_tclServiceD..
   ***************************************************************************/
   /*!
    * \brief   Method result for the ISource for handling loopbacks message
    *          results.
    * \param   [poMessage]:    (->I)   Pointer to service data.
    * \retval  NONE
    * \date 01.03.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vOnMrISource(amt_tclServiceData* poMessage) OVERRIDE;

   virtual tVoid vOnAudioRouteRequestReject(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  arl_tclISource::arl_tclISource()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So
    *          that any attempt to create without parameter will be caught by
    *          the compiler.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISource();     // No definition exists.

   /***************************************************************************
   ** FUNCTION:  arl_tclISource::arl_tclISource(const arl_tclISource&..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoISource]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So
    *          that any attempt to copy will be caught by the compiler.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISource(const arl_tclISource& rfcoISource); // No definition exists.

   /***************************************************************************
   ** FUNCTION:  arl_tclISource& arl_tclISource::operator=(co ..
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoISource]: (I) Const reference to object to be copied
    * \retval  [arl_tclISource&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So
    *          that any attempt for assignment will be caught by the compiler.
    * \date 15.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclISource& operator= (const arl_tclISource& rfcoISource); // No definition exists.

   virtual tU16 u16GetAppID() OVERRIDE {return ahl_tclBaseOneThreadService::u16GetAppID();};
   virtual tU16 u16GetServiceID() OVERRIDE {return ahl_tclBaseOneThreadService::u16GetServiceID();};

   tBool bTriggerSrcActivityLoopback(arl_tenSource enSrc, arl_tenActivity enAct);

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/
private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Source Availability
   tBool m_bSrcAvailable;


   /*! @}*/

   /***************************************************************************
   * Declare Msg Map - Utilizing the Framework for message map abstraction.
   ***************************************************************************/

   DECLARE_MSG_MAP(arl_tclISource)


   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class arl_tclISource : public ahl_tclBaseOneThreadService

#endif   // #ifndef ARL_TCLISOURCE_H_

// <EOF>
