/*!
********************************************************************************
* \file              arl_tclImplFunctor.h
********************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Generalized command functor - Functors are functions
*                    with a state.
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
********************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 07.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

#ifndef ARL_TCLIMPLFUNCTOR_H_
#define ARL_TCLIMPLFUNCTOR_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations (scope: global)
|----------------------------------------------------------------------------*/


/*!
 * \class arl_tclFunctor
 * \brief Functor - Abstract base class.
 *
 * Functors are functions with a state.Functors can encapsulate C and C++
 * function pointers employing the concepts templates and polymorphism.
 * Functors can be used to build up a list of pointers to member functions of
 * arbitrary classes and call them all through the same interface without
 * bothering about the need of a pointer to an instance or their class.
 * All the functions should have the same return-type and calling parameters.
 * Functors are also known as closures or Functionoids. Functors can be used to
 * implement callbacks.
 */

class arl_tclFunctor
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclFunctor::operator()()
   ***************************************************************************/
   /*!
    * \brief   The function executes the callback function.
    *          This is a pure virtual function which has to be overloaded by
    *          deriving, to provide an implementation.
    * \param   NONE
    * \retval  [tBool]: Status of callback mechanism.
    * \date 07.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool operator()() = 0;  // Pure virtual
   virtual ~arl_tclFunctor(){}
   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/
};

/*!
 * \class arl_tclImplFunctor
 * \brief Generic template class implementing the callback mechanism.
 *
 */

template<typename ill_tGeneric>
class arl_tclImplFunctor : public arl_tclFunctor
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /*!
    * \typedef  tBool (ill_tGeneric::*arl_pfFuncCallbk)()
    * \brief Callback function signature definition.
    */
   typedef  tBool (ill_tGeneric::*arl_pfFuncCallbk)();

   /***************************************************************************
   ** FUNCTION:  arl_tclImplFunctor(ill_tGeneric* poObj, arl_..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor - Dependency Injection Principle (DIP)
    * \param   [poObj]:       (->I) Command Object pointer.
    * \param   [pfCallbk]:    (->I) Function pointer for callback.
    * \retval  NONE
    * \date 07.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclImplFunctor(ill_tGeneric* poObj, arl_pfFuncCallbk pfCallbk)
      :m_poObj(poObj), m_pfCallbk(pfCallbk){};

   /***************************************************************************
   ** FUNCTION:  virtual tBool arl_tclImplFunctor::operator()()
   ***************************************************************************/
   /*!
    * \brief   The function executes the callback function.
    *          Callback function is called from here using the command object.
    * \param   NONE
    * \retval  [tBool]: Status of callback mechanism.
    * \date 07.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool operator()()
   {
      tBool bRetVal  = FALSE;
      if((OSAL_NULL != m_poObj) && (OSAL_NULL != m_pfCallbk))
      {
         bRetVal  =  (*m_poObj.*m_pfCallbk)();   // Function callback.
      }
      return bRetVal;
   };

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclImplFunctor::arl_tclImplFunctor()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So
    *          that any attempt to instantiate w/o parameter will be caught by
    *          the compiler.
    * \date 07.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclImplFunctor();      // No definition exists.

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Object pointer
   ill_tGeneric* m_poObj;

   /// Function pointer for callback
   arl_pfFuncCallbk m_pfCallbk;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class arl_tclImplFunctor : public arl_tclFunctor

#endif   // #ifndef ARL_TCLIMPLFUNCTOR_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
