/*!
*******************************************************************************
* \file              arl_tclFsmState.h
*******************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Implements FSM State
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 10.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef ARL_TCLFSMSTATE_H_
#define ARL_TCLFSMSTATE_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

/// OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

#include "arl_FsmTypes.h"
#include "arl_tclFsmTransition.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations
|----------------------------------------------------------------------------*/

/*!
 * \class arl_tclFsmState
 * \brief Implements FSM State
 *
 */

class arl_tclFsmState
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /*!
    * \enum arl_tenOnState
    * Enum definition to distinguish on entry & on exit in a state
    */
   typedef enum
   {
      ARL_EN_STATE_ENTRY =  0
      , ARL_EN_STATE_EXIT
   } arl_tenOnState;

   /***************************************************************************
   ** FUNCTION:  arl_tclFsmState::arl_tclFsmState(tS16 s16ParentStateId, tCSt..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection
    *          Principle (DIP)
    * \param   [s16ParentStateId]:  (I) Parent State ID
    * \param   [cszName]:           (->I) State Name
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclFsmState(tS16 s16ParentStateId, tCString cszName);

   /***************************************************************************
   ** FUNCTION:  virtual arl_tclFsmState::~arl_tclFsmState()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~arl_tclFsmState();

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclFsmState::operator()(const arl_tenOnState enOnS..
   ***************************************************************************/
   /*!
    * \brief   Perform operations on state change.
    * \param   [enOnState]: (I) Whether state is entered or exited.
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid operator()(const arl_tenOnState enOnState);

   /***************************************************************************
   ** FUNCTION:  arl_tclFsmTransition* arl_tclFsmState::poGetTransition..
   ***************************************************************************/
   /*!
    * \brief   Get FSM transition based on Trigger id.
    * \param   [s16TriggerId]: (I) Trigger id.
    * \retval  [arl_tclFsmTransition* const] Pointer to FSM transition.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclFsmTransition* poGetTransition(tS16 s16TriggerId);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclFsmState::vAdd(arl_tFsmFunctor* const cpoFsmFu..
   ***************************************************************************/
   /*!
    * \brief   Add a FSM functor for state changes (on entry or on exits.)
    * \param   [cpoFsmFunctor]:  (->I) Pointer to FSM functor
    * \param   [enOnState]:      (I) Entry or Exit?
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vAdd(arl_tFsmFunctor* const cpoFsmFunctor
      , arl_tenOnState enOnState = ARL_EN_STATE_ENTRY);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclFsmState::vAdd(arl_tclFsmTransition* const cpoF..
   ***************************************************************************/
   /*!
    * \brief   Add a FSM transition for state changes
    * \param   [cpoFsmTransition]:  (->I) Pointer to FSM transition
    * \retval  [tS16]: Transition id.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vAdd(arl_tclFsmTransition* const cpoFsmTransition);

   /***************************************************************************
   ** FUNCTION:  tCString arl_tclFsmState::cszGetStateName() const
   ***************************************************************************/
   /*!
    * \brief   Get State name
    * \param   NONE
    * \retval  [tCString]: State Name
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tCString cszGetStateName() const;

   /***************************************************************************
   ** FUNCTION:  tS16 arl_tclFsmState::s16GetParentStateId() const
   ***************************************************************************/
   /*!
    * \brief   Get Parent State id
    * \param   NONE
    * \retval  [tS16]: Parent State id
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tS16 s16GetParentStateId() const;

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclFsmState::arl_tclFsmState()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable it. So
    *          that any attempt to create without parameter will be caught by
    *          the compiler.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   arl_tclFsmState();   // No definition exists.

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Parent state id
   tS16 m_s16ParentStateId;
   /// State name
   tCString m_cszStateName;
   /// Entry functions
   arl_tFunctorList m_oEntryFunc;
   /// Exit functions
   arl_tFunctorList m_oExitFunc;
   /// Transitions
   arl_tTransitionList m_oTransitions;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/

}; // class arl_tclFsmState

/// On Entry State
#define ARL_ON_STATE_ENTRY  arl_tclFsmState::ARL_EN_STATE_ENTRY
/// On Exit State
#define ARL_ON_STATE_EXIT   arl_tclFsmState::ARL_EN_STATE_EXIT

#endif   // #ifndef ARL_TCLFSMSTATE_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
