/*!
********************************************************************************
* \file              arl_tclFsm.h
********************************************************************************
*  - PROJECT:        LCN2KAI
*  - SW-COMPONENT:   Audio Routing Library (ARL)
*  - DESCRIPTION:    Finite State Machine (FSM)
*  - COPYRIGHT:      &copy; 2011 Robert Bosch Car Multimedia Gmbh
********************************************************************************
* \date 26.06.2012 \version 2.0 \author Ruben Volkmer (R.Volkmer@ITB-Solutions.de)
* - Remove AHL inheritance to enable usage by object reference instead of inheritance
* - Remove MOST dependencies
* - Adapt to Project needs
*
* \date 10.01.2011 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
*******************************************************************************/

#ifndef ARL_TCLFSM_H_
#define ARL_TCLFSM_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

#include "arl_FsmTypes.h"
#include "arl_tclFsmState.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations (scope: global)
|----------------------------------------------------------------------------*/

/*!
 * \brief Invalid State Id
 */
const tS16 ARL_FSM_STATE_ID_INVALID = -1;

/*!
 * \brief Invalid Trigger Id
 */
const tS16 ARL_FSM_TRIG_ID_INVALID  = -1;

/*!
 * \class arl_tclFsm
 * \brief Finite State Machine (FSM)
 *
 */

class arl_tclFsm
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  arl_tclFsm::arl_tclFsm(tCString cszName = OSAL_NULL)
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   [cszName]: (->I)  FSM name
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   explicit arl_tclFsm(tCString cszName = OSAL_NULL);

   /***************************************************************************
   ** FUNCTION:  virtual arl_tclFsm::~arl_tclFsm()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~arl_tclFsm();

   /***************************************************************************
   ** FUNCTION:  tS16 arl_tclFsm::s16AddState(tCString cszStateName, tS16 s1..
   ***************************************************************************/
   /*!
    * \brief   Interface to add a new state to the FSM.
    * \param   [cszStateName]:      (->I) State Name
    * \param   [s16ParentStateId]   (I)   Parent State id
    * \retval  [tS16]: State Id.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tS16 s16AddState(tCString cszStateName
      , tS16 s16ParentStateId = ARL_FSM_STATE_ID_INVALID);

   /***************************************************************************
   ** FUNCTION:  tS16 arl_tclFsm::s16AddTrigger(tCString cszTrigName)
   ***************************************************************************/
   /*!
    * \brief   Interface to add a new state to the FSM.
    * \param   [cszTrigName]:      (->I) Trigger Name
    * \retval  [tS16]: Trigger Id.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tS16 s16AddTrigger(tCString cszTrigName);

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bAdd(tS16 s16StateId, arl_tFsmFunctor* co..
   ***************************************************************************/
   /*!
    * \brief   Add functors into State machine
    * \param   [s16StateId]:      (I) State Id
    * \param   [cpoFsmFunctor]:   (->I) FSM Functor
    * \param   [enOnState]:       (I) FSM On State exit/entry
    * \retval  [tBool]: TRUE if adding into FSM succeeded, FALSE otherwise.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bAdd(tS16 s16StateId, arl_tFsmFunctor* const cpoFsmFunctor
      , arl_tclFsmState::arl_tenOnState enOnState = ARL_ON_STATE_ENTRY);

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bSetInitState(tS16 s16StateId)
   ***************************************************************************/
   /*!
    * \brief   Function to set the initial State id for the FSM
    * \param   [s16StateId]:      (I) Initial State Id
    * \retval  [tBool]: TRUE if successful, FALSE otherwise.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bSetInitState(tS16 s16StateId);

   /***************************************************************************
   ** FUNCTION:  tVoid arl_tclFsm::operator()(tS16 s16TriggerId)
   ***************************************************************************/
   /*!
    * \brief   Function to trigger a state change in the FSM
    * \param   [s16TriggerId]:      (I) Trigger Id
    * \retval  NONE
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid operator()(tS16 s16TriggerId);

   /***************************************************************************
   ** FUNCTION:  tS16 arl_tclFsm::s16GetCurrentState() const
   ***************************************************************************/
   /*!
    * \brief   Function to get the retrieve current state.
    * \param   NONE
    * \retval  [tS16]: Current State id.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tS16 s16GetCurrentState() const;

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bFsmFinalize()
   ***************************************************************************/
   /*!
    * \brief   Function to FSM is finalize FSM
    * \param   NONE
    * \retval  [tBool]: TRUE if FSM is finalized, FALSE otherwise.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bFsmFinalize();

   /***************************************************************************
   ** FUNCTION:  arl_tclFsmTransition* arl_tclFsm::poAddTransition(tS..
   ***************************************************************************/
   /*!
    * \brief   Add a transition to the FSM
    * \param   [s16SourceStateId]:  (I) Source State Id
    * \param   [s16DestStateId]:    (I) Destination State Id
    * \param   [s16TriggerId]:      (I) Trigger Id
    * \param   [cpoFsmTransition]:  (->I) Pointer to transition Transition
    * \retval  [arl_tclFsmTransition*]: Pointer to FSM transition.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
    arl_tclFsmTransition* poAddTransition(tS16 s16SourceStateId
       , tS16 s16DestStateId, tS16 s16TriggerId);

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bAddTransitionFunc(arl_tclFsmTransition* c..
   ***************************************************************************/
   /*!
    * \brief   Add a functor to the transition.
    * \param   [cpoTransition]:     (->I) Pointer to transition.
    * \param   [cpoFunction]:       (->I) Functor pointer.
    * \retval  [tBool]: TRUE if functor was added to transition, FALSE otherwise
    * \date 11.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
    tBool bAddTransitionFunc
   (
      arl_tclFsmTransition* const cpoTransition
      , arl_tFsmFunctor* const cpoFunction
   ) const;

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bAddCondition(arl_tclFsmTransition* cons..
   ***************************************************************************/
   /*!
    * \brief   Add a condition to the transition.
    * \param   [cpoTransition]:     (->I) Pointer to transition.
    * \param   [cpoCond]:           (->I) Condition pointer.
    * \retval  [tBool]: TRUE if condition was added to transition, FALSE otherwise
    * \date 11.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bAddCondition
   (
      arl_tclFsmTransition* const cpoTransition
      , arl_tFsmCondition* const cpoCond
   ) const;

// Set state For ARL Utest only
  tVoid vSetState(tS16 s16StateId);
   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bSetState(tS16 s16StateId)
   ***************************************************************************/
   /*!
    * \brief   Function to set state
    * \param   [s16StateId]:     (I) State id to set.
    * \retval  [tBool]: TRUE if state was reached, FALSE otherwise.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bSetState(tS16 s16StateId);

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bIsValidState(tS16 s16StateId) const
   ***************************************************************************/
   /*!
    * \brief   Function to check whether state is valid
    * \param   [s16StateId]:     (I) State id to check.
    * \retval  [tBool]: TRUE if state is valid, FALSE otherwise.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bIsValidState(tS16 s16StateId) const;

   /***************************************************************************
   ** FUNCTION:  tBool arl_tclFsm::bIsValidTrigger(tS16 s16Trigger) const
   ***************************************************************************/
   /*!
    * \brief   Function to check whether trigger is valid
    * \param   [s16Trigger]:     (I) Trigger id to check.
    * \retval  [tBool]: TRUE if trigger is valid, FALSE otherwise.
    * \date 10.01.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bIsValidTrigger(tS16 s16Trigger) const;

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// FSM Finalized or not?
   tBool m_bFsmFinalized;
   /// Current State id.
   tS16 m_s16CurrentStateId;
   /// FSM Name
   tCString m_cszFsmName;
   /// FSM States
   arl_tclFsmStateList m_oStates;
   /// FSM Triggers
   arl_tclFsmTrigList m_oTriggers;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/
}; // class arl_tclFsm

#endif   // #ifndef ARL_TCLFSM_H_

////////////////////////////////////////////////////////////////////////////////
// <EOF>
