

#ifdef VARIANT_S_FTR_ENABLE_GM_STR
/*******************************************************************************
* FILE:          FC_StreamRouter_configMic.cpp
* PROJECT:
* SW-COMPONENT:  FC_StreamRouter application
*-------------------------------------------------------------------------------
*
* DESCRIPTION:   General class of microphone
*
*-------------------------------------------------------------------------------
*
* COPYRIGHT:    (c) 2011 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author                          | Modification
* 29.04.11  | TMS Neumann,Gaidatzi-Lken      | Inital version
*******************************************************************************/

/* OSAL Device header */
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "FC_StreamRouter_configMic.h"
#include "FC_StreamRouter_trace_def.h"
#include "EOLLib.h"


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_STREAMROUTER_APPLICATION
//#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_STREAMROUTER_MICROPHONE
#include "trcGenProj/Header/FC_StreamRouter_configMic.cpp.trc.h"
#endif



#define STR_MIC_STORAGE_FILE_PATH "/dev/ffs/strmic.conf"



/*******************************************************************************
                  FC_Streamrouter for microphone
  Read EOL Mic values and configures the Microphones
*******************************************************************************/
/*******************************************************************************
*
*   FUNCTION:
*   fc_streamrouter_tclConfigMic(tVoid)
*   Consructor
*
*******************************************************************************/
fc_streamrouter_tclConfigMic::fc_streamrouter_tclConfigMic()
:micPowerState(FALSE), micSelectState(FALSE), gpio_mask(0), mic1cfg(0), mic2cfg(0)
, m_pwmDevice(OSAL_ERROR)
{

  readEOL(&mic1cfg, &mic2cfg);

  if(bConfigureMic1(mic1cfg ,mic2cfg) != true)
  {
    ETG_TRACE_ERR(("ERR: Unable to set EOL gpio configuration for Mic1! Setting up default."));
    vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_HW_POWER_CTRL);
    vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_SELECT_CTRL);
    vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_DIAG1_ENABLE);
  }

  if(bConfigureMic2(mic2cfg, mic1cfg) != true)
  {
    ETG_TRACE_ERR(("ERR: Unable to set EOL gpio configuration for Mic2! Setting up default."));
    vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_HW_POWER_CTRL);
    vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_DIAG2_ENABLE);
  }


  ETG_TRACE_USR4(("DEBUG: MIC1_EOL: %d", (int)mic1cfg));
  ETG_TRACE_USR4(("DEBUG: MIC2_EOL: %d", (int)mic2cfg));

  //Read EOL Microphone Delay value
  u8mic1Delay = 0;
  u8mic2Delay = 0;
  readEOLMicDelay(&u8mic1Delay, &u8mic2Delay);

  ETG_TRACE_USR4(("DEBUG: MIC1_EOL_DELAY: %d", (int)u8mic1Delay));
  ETG_TRACE_USR4(("DEBUG: MIC2_EOL_DELAY: %d", (int)u8mic2Delay));

  //Set timer for mic adc pwm
  //Das machen wir unter Linux nicht, soll in die Alsa Konfiguration mitrein.
  vSetTimerMicPwm(PWM_HZ);
}



/*******************************************************************************
*
*   FUNCTION:
*   ~fc_streamrouter_tclConfigMic()
*   Desructor
*
*******************************************************************************/
fc_streamrouter_tclConfigMic::~fc_streamrouter_tclConfigMic()
{
   if (OSAL_ERROR != m_pwmDevice)
   {
      OSAL_s32IOClose(m_pwmDevice);
      m_pwmDevice =  OSAL_ERROR;
   }

  if(setGPIO(FALSE, STR_MIC_HW_POWER_CTRL, &micPowerState) != 0)
  {
    ETG_TRACE_USR4((">>> ! UNABLE TO SET MIC POWER ! <<<"));
  }

  micPowerState = FALSE;
  micSelectState = FALSE;
}



/*******************************************************************************
                  EOL AND CONFIGURE FUNCTIONS
*******************************************************************************/

/*******************************************************************************
*
*   FUNCTION: readEOL(tU8 *pu8Mic1cfg, tU8 *pu8Mic2cfg)
*
*   DESCRIPTION: This function read the EOL Parameter for the Microphone configuration
*
*******************************************************************************/
/*
#ifndef EOLLIB_OFFSET_MICROPHONE_1_CONFIG
  #define EOLLIB_OFFSET_MICROPHONE_1_CONFIG 0
#endif

#ifndef EOLLIB_OFFSET_MICROPHONE_2_CONFIG
  #define EOLLIB_OFFSET_MICROPHONE_2_CONFIG 0
#endif
*/
int fc_streamrouter_tclConfigMic::readEOL(tU8 *pu8Mic1cfg, tU8 *pu8Mic2cfg)
{
  int ret = 0;

  OSAL_tIODescriptor fd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_DIAG_EOL, OSAL_EN_READWRITE);
  if(fd != OSAL_ERROR)
  {
    OSAL_trDiagEOLEntry rEOLData;

    rEOLData.u8Table = EOLLIB_TABLE_ID_COUNTRY;
    rEOLData.u16Offset = EOLLIB_OFFSET_MICROPHONE_1_CONFIG;
    rEOLData.u16EntryLength = 1;
    rEOLData.pu8EntryData = pu8Mic1cfg;

    if(OSAL_s32IORead(fd, (tPS8)&rEOLData, sizeof(rEOLData)) == OSAL_ERROR)
    {
      ETG_TRACE_ERR(("ERR: Unable to read mic1 cfg!"));
      ret = 2;
    }

    rEOLData.u8Table = EOLLIB_TABLE_ID_DISPLAY_INTERFACE; //Moved from Country (19.07.12 Fofi Change)
  rEOLData.u16Offset = EOLLIB_OFFSET_MICROPHONE_2_CONFIG;
    rEOLData.pu8EntryData = pu8Mic2cfg;

    if(OSAL_s32IORead(fd, (tPS8)&rEOLData, sizeof(rEOLData)) == OSAL_ERROR)
    {
      ETG_TRACE_ERR(("ERR: Unable to read mic2 cfg!"));
      ret = 3;
    }
    else
    {
      // Mapping of EOL values of Mic2 ($0,$1,$2) to unique values of Mic1 ($0,$1,$2,$3,$4)
      if(1 == *pu8Mic2cfg)
      {
        *pu8Mic2cfg = STR_EOL_MIC_ANALOG_BALANCED_PHANTOMVOLTAGE;
      }
      else if(2 == *pu8Mic2cfg)
      {
        *pu8Mic2cfg = STR_EOL_MIC_MOST_PASSTHROUGH;
      }
      else
      {
        *pu8Mic2cfg = STR_EOL_MIC_NOMIC;
      }
    }

    OSAL_s32IOClose(fd);
  }
  else
  {
    ETG_TRACE_ERR(("ERR: Could not open EOL device!"));
    ret = 1;
  }

  return ret;
}



/*******************************************************************************
*
*   FUNCTION: readEOLMicDelay(tS8 *s8micDelay)
*
*   DESCRIPTION: This function read the EOL Parameter for the Microphone delay
*
*******************************************************************************/
/*
#ifndef EOLLIB_OFFSET_SUPPLEMENTAL_MICROPHONE_DELAY
  #define EOLLIB_OFFSET_SUPPLEMENTAL_MICROPHONE_DELAY 0
#endif
*/


int fc_streamrouter_tclConfigMic::readEOLMicDelay(tU8 *mic1Delay, tU8 *mic2Delay)
{
  int ret = 0;
  tU8 micDelayByte;

  OSAL_tIODescriptor fd = OSAL_IOOpen(OSAL_C_STRING_DEVICE_DIAG_EOL, OSAL_EN_READWRITE);

  if(fd != OSAL_ERROR)
  {
    OSAL_trDiagEOLEntry rEOLData;

    rEOLData.u8Table = EOLLIB_TABLE_ID_COUNTRY;
    rEOLData.u16Offset = EOLLIB_OFFSET_SUPPLEMENTAL_MICROPHONE_DELAY;
    rEOLData.u16EntryLength = 1;
    rEOLData.pu8EntryData = &micDelayByte;

    if(OSAL_s32IORead(fd, (tPS8)&rEOLData, sizeof(rEOLData)) == OSAL_ERROR)
    {
      ETG_TRACE_ERR(("ERR: Unable to read mic1 delay"));
      ret = 2;
    }

    if ( (micDelayByte & 0x80) == 0x80 ) //negative value => set Mic1 Delay value
    {
        *mic1Delay = 256 - micDelayByte;
        *mic2Delay = 0;
    }
    else //positive value => set Mic2 Delay value
    {
        *mic2Delay = micDelayByte;
        *mic1Delay = 0;
    }

    OSAL_s32IOClose(fd);
  }
  else
  {
    ETG_TRACE_ERR(("ERR: Could not open EOL device!"));
    ret = 1;
  }

  return ret;
}





/*******************************************************************************
*
*   FUNCTION: u8GetMic1cfg()
*
*   DESCRIPTION: This function return the EOL Microphone1 configuration
*
*******************************************************************************/
 tU8 fc_streamrouter_tclConfigMic::u8GetMic1cfg()
{
  return mic1cfg;
}



/*******************************************************************************
*
*   FUNCTION: tU8GetMic2cfg()
*
*   DESCRIPTION: This function return the EOL Microphone2 configuration
*
*******************************************************************************/
tU8 fc_streamrouter_tclConfigMic::u8GetMic2cfg()
{
  return mic2cfg;
}




/*******************************************************************************
*
*   FUNCTION: s8GetMicDelay()
*
*   DESCRIPTION: This function return the EOL Microphone Delay
*
*******************************************************************************/
void fc_streamrouter_tclConfigMic::vGetMicDelay(tU8 *mic1Delay, tU8 *mic2Delay)
{
    *mic1Delay = u8mic1Delay;
    *mic2Delay = u8mic2Delay;
}




/*******************************************************************************
*
*   FUNCTION: s8GetMicDelay()
*
*   DESCRIPTION: This function return the EOL Microphone Delay
*
*******************************************************************************/
void fc_streamrouter_tclConfigMic::vSetMicDelay(tU8 micDelay)
{
    if ( (micDelay & 0x80) == 0x80 ) //negative value => set Mic1 Delay value
    {
        u8mic1Delay = 256 - micDelay;
        u8mic2Delay = 0;
    }
    else //positive value => set Mic2 Delay value
    {
        u8mic2Delay = micDelay;
        u8mic1Delay = 0;
    }

}



/*******************************************************************************
*
*   FUNCTION: bConfigureMic1(int Mic1cfg, int Mic2cfg)
*
*   DESCRIPTION: This function configure the Microphone1 depend on Microphone2
*
*******************************************************************************/
tBool fc_streamrouter_tclConfigMic::bConfigureMic1(int Mic1cfg, int Mic2cfg)
{
  tBool bRet = TRUE;

  switch(Mic1cfg)
  {
    case STR_EOL_MIC_NOMIC:
    case STR_EOL_MIC_MOST_PASSTHROUGH:
    {
      vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_SELECT_CTRL);
      vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_DIAG1_ENABLE);

      if (  Mic2cfg == STR_EOL_MIC_NOMIC
        ||Mic2cfg == STR_EOL_MIC_MOST_PASSTHROUGH)
      {
        vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_HW_POWER_CTRL);
      }
      break;
    }
    case STR_EOL_MIC_ANALOG_PASSTHROUGH_SINGLEENDED:
    case STR_EOL_MIC_ANALOG_PASSTHROUGH_DIFFERENTIAL:
    {
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_HW_POWER_CTRL);
      vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_SELECT_CTRL);
      vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_DIAG1_ENABLE);
      break;
    }
    case STR_EOL_MIC_ANALOG_BALANCED_PHANTOMVOLTAGE:
    {
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_HW_POWER_CTRL);
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_SELECT_CTRL);
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_DIAG1_ENABLE);
      break;
    }

    default:
    {
      ETG_TRACE_ERR(("ERR: Unknow Mic1 configuration! is only 0-4 allowed!"));
      bRet = FALSE;
      break;
    }
  }

  mic1cfg = ((bRet)?Mic1cfg:mic1cfg);
  return bRet;
}



/*******************************************************************************
*
*   FUNCTION: bConfigureMic2(int Mic2cfg, int Mic1cfg)
*
*   DESCRIPTION: This function configure the Microphone2 in depend of Microphone1
*
*******************************************************************************/
tBool fc_streamrouter_tclConfigMic::bConfigureMic2(int Mic2cfg, int Mic1cfg)
{
  tBool bRet = TRUE;

  switch(Mic2cfg)
  {
    case STR_EOL_MIC_NOMIC:
    case STR_EOL_MIC_MOST_PASSTHROUGH:
    {
      vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_DIAG2_ENABLE);

      if (  Mic1cfg == STR_EOL_MIC_NOMIC
        ||Mic1cfg == STR_EOL_MIC_MOST_PASSTHROUGH)
      {
        vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_HW_POWER_CTRL);
      }
      break;
    }

    case STR_EOL_MIC_ANALOG_PASSTHROUGH_SINGLEENDED:
    case STR_EOL_MIC_ANALOG_PASSTHROUGH_DIFFERENTIAL:
    {
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_HW_POWER_CTRL);
      vSetGPIO((tBool)(FALSE),OSAL_EN_MIC_DIAG2_ENABLE);
      break;
    }
    case STR_EOL_MIC_ANALOG_BALANCED_PHANTOMVOLTAGE:
    {
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_HW_POWER_CTRL);
      vSetGPIO((tBool)(TRUE),OSAL_EN_MIC_DIAG2_ENABLE);
      break;
    }

    default:
    {
      ETG_TRACE_ERR(("ERR: Unknow Mic2 configuration, is only 0-4 allowed!"));
      bRet = FALSE;
      break;
    }
  }

  mic2cfg = ((bRet)?Mic2cfg:mic2cfg);
  return bRet;
}



/*******************************************************************************
                  GPIO FUNCTIONS
*******************************************************************************/

/*******************************************************************************
*
*  FUNCTION: fc_streamrouter_tclMicrophone()vSetGPIO(tBool bflag, OSAL_enGpioPins egpio)
*
*   DESCRIPTION: This function sets the GPIOs of Microphone
*
*******************************************************************************/
/*PR*/
void fc_streamrouter_tclConfigMic::vSetGPIO(tBool bflag, OSAL_enGpioPins egpio)
{
  ETG_TRACE_USR4(("vSetGPIO: egpio=%d, bflag=%d", (int)egpio, (int)bflag));

  //int ret = 0;
  OSAL_tIODescriptor gpiohandle = OSAL_ERROR;
  OSAL_tGPIODevID devid = egpio;

  gpiohandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
  if(gpiohandle != OSAL_ERROR)
  {

    if(OSAL_s32IOControl(gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT, (tS32)devid) != OSAL_ERROR)
    {

      if(bflag == TRUE)
      {
        OSAL_s32IOControl(gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT_ACTIVE, (tS32)devid);
      }
      else
      {
        OSAL_s32IOControl(gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT_INACTIVE, (tS32)devid);
    }

    }

    /* close the device only */
    (void)OSAL_s32IOClose(gpiohandle);

  }
  else
  {
      /* do nothing */
  }

  return;

}



/*******************************************************************************
*
*   FUNCTION: setGPIO(tBool flg, tU32 gpio, tBool *state)
*
*   DESCRIPTION: This function sets the GPIOs of Microphone and save the state????
*
*******************************************************************************/
int fc_streamrouter_tclConfigMic::setGPIO(tBool flg, tU32 gpio, tBool *state) // true = high, false = low
{
  ETG_TRACE_USR4(("setGPIO: gpio=%d, flg=%d", (int)gpio, (int)flg));

  int ret = 0;
  OSAL_tIODescriptor gpiohandle = OSAL_ERROR;
  OSAL_tGPIODevID devid = (OSAL_tGPIODevID)gpio;

  gpiohandle = OSAL_IOOpen(OSAL_C_STRING_DEVICE_GPIO, OSAL_EN_READWRITE);
  if(gpiohandle != OSAL_ERROR)
  {
    if(OSAL_s32IOControl(gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT, (tS32)devid) != OSAL_ERROR)
    {
      if(flg == TRUE)
      {

        if(OSAL_s32IOControl(gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT_ACTIVE, (tS32)devid) != OSAL_ERROR)
        {
          if(state != NULL) *state = TRUE;
        }
        else
        {
          ret = 6; // OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT_HIGH
        }
      }
      else
      {
        if(OSAL_s32IOControl(gpiohandle, OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT_INACTIVE, (tS32)devid) != OSAL_ERROR)
        {
          if(state != NULL) *state = FALSE;
        }
        else
        {
          ret = 7; // OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT_LOW
        }

      }

    }
    else
    {
      ret = 2; // OSAL_C_32_IOCTRL_GPIO_SET_OUTPUT
    }

    (void)OSAL_s32IOClose(gpiohandle);
  }
  else
  {
    ret = 1; // open
  }

  return ret;

}



/*******************************************************************************
*
*   FUNCTION: toggleMicPower()
*
*   DESCRIPTION: This testfunction toggle the HW-Power GPIO of Microphone
*
*******************************************************************************/
tBool fc_streamrouter_tclConfigMic::toggleMicPower()
{
  int ret = setGPIO((~micPowerState) & 1, STR_MIC_HW_POWER_CTRL, &micPowerState);

  if(ret != 0)
  {
    ETG_TRACE_USR4(("ERROR: mic power toggle: setGPIO() returned %d", ret));
    return FALSE;
  }

  return TRUE;
}



/*******************************************************************************
*
*   FUNCTION: toggleMicSelect()
*
*   DESCRIPTION: This testfunction toggle the Select GPIO of Microphone
*
*******************************************************************************/
tBool fc_streamrouter_tclConfigMic::toggleMicSelect()
{
  int ret = setGPIO((~micSelectState) & 1, STR_MIC_HW_SELECT, &micSelectState);

  if(ret != 0)
  {
    ETG_TRACE_USR4(("ERROR: mic select toggle: setGPIO() returned %d", ret));
    return FALSE;
  }

  return TRUE;
}



/*******************************************************************************
*
*   FUNCTION: toggleMicPower()
*
*   DESCRIPTION: This testfunction sets the HW-Power GPIO of Microphone
*
*******************************************************************************/
tBool fc_streamrouter_tclConfigMic::setMicPower(tBool flg)
{
  tBool ret = (tBool)setGPIO(flg, STR_MIC_HW_POWER_CTRL, &micPowerState);
  if(ret == 0)
  {
    storeMicSettings(micPowerState, micSelectState);
    ETG_TRACE_USR4(("Done."));
  }
  else
  {
    ETG_TRACE_USR4(("Failed."));
  }
  return ret;
}



/*******************************************************************************
*
*   FUNCTION: toggleMicSelect()
*
*   DESCRIPTION: This testfunction sets the Select GPIO of Microphone
*
*******************************************************************************/
tBool fc_streamrouter_tclConfigMic::setMicSelect(tBool flg)
{
  tBool ret = (tBool)setGPIO(flg, STR_MIC_HW_SELECT, &micSelectState);

  /* PR test */

  ETG_TRACE_USR4(("DEBUG: MIC1_EOL: %d", (int)mic1cfg));
  ETG_TRACE_USR4(("DEBUG: MIC1_EOL: %d", (int)mic2cfg));

  /*PR Test*/

  if(ret == 0)
  {
    storeMicSettings(micPowerState, micSelectState);
    ETG_TRACE_USR4(("Done."));
  }
  else
  {
    ETG_TRACE_USR4(("Failed."));
  }
  return ret;
}



/*******************************************************************************
*
*   FUNCTION: storeMicSettings(tBool statePower, tBool stateSelect)
*
*   DESCRIPTION: This testfunction save the GPIO of Microphone in a testfile
*
*******************************************************************************/
void fc_streamrouter_tclConfigMic::storeMicSettings(tBool statePower, tBool stateSelect)
{
  OSAL_tIODescriptor desc = OSAL_IOOpen(STR_MIC_STORAGE_FILE_PATH, OSAL_EN_READWRITE);
  if(desc == OSAL_ERROR)
  {
    desc = OSAL_IOCreate(STR_MIC_STORAGE_FILE_PATH, OSAL_EN_READWRITE);
  }

  if(desc != OSAL_ERROR)
  {
    OSAL_s32IOWrite(desc, (tPCS8)&statePower, 4);
    OSAL_s32IOWrite(desc, (tPCS8)&stateSelect, 4);
    OSAL_s32IOClose(desc);

    ETG_TRACE_USR4(("Done."));
  }
  else
  {
    ETG_TRACE_USR4(("ERR: Unable to store!"));
  }
}


/*******************************************************************************
*
*   FUNCTION: setMicGPIOs(unsigned int mask, unsigned int *store_mask)
*
*   DESCRIPTION: This function set mic gpios as defined in mask
*   (STR_MICCFG_POWER | STR_MICCFG_SELECT | STR_MICCFG_DIAG1 | STR_MICCFG_DIAG2)
*
*   RETURNVALUES: returns error mask with 1=failed, 0=ok
*
*******************************************************************************/
unsigned int fc_streamrouter_tclConfigMic::setMicGPIOs(unsigned int mask, unsigned int *store_mask)
{
  unsigned int gpiosetting = 0;
  unsigned int err_mask = 0;

  if(store_mask != NULL) gpiosetting = *store_mask;

  if(setGPIO((tBool)(mask & STR_MICCFG_POWER), OSAL_EN_MIC_HW_POWER_CTRL, NULL) == 0)
  {
    storeGPIOsetting(STR_MICCFG_POWER, (mask & STR_MICCFG_POWER), &gpiosetting);
  }
  else
  {
    err_mask |= STR_MICCFG_POWER;
  }

  if(setGPIO((tBool)(mask & STR_MICCFG_SELECT), OSAL_EN_MIC_SELECT_CTRL, NULL) == 0)
  {
    storeGPIOsetting(STR_MICCFG_SELECT, (mask & STR_MICCFG_SELECT), &gpiosetting);
  }
  else
  {
    err_mask |= STR_MICCFG_SELECT;
  }

  if(setGPIO((tBool)(mask & STR_MICCFG_DIAG1), OSAL_EN_MIC_DIAG1_ENABLE, NULL) == 0)
  {
    storeGPIOsetting(STR_MICCFG_DIAG1, (mask & STR_MICCFG_DIAG1), &gpiosetting);
  }
  else
  {
    err_mask |= STR_MICCFG_DIAG1;
  }

  if(setGPIO((tBool)(mask & STR_MICCFG_DIAG2), OSAL_EN_MIC_DIAG2_ENABLE, NULL) == 0)
  {
    storeGPIOsetting(STR_MICCFG_DIAG2, (mask & STR_MICCFG_DIAG2), &gpiosetting);
  }
  else
  {
    err_mask |= STR_MICCFG_DIAG2;
  }

  if(store_mask != NULL) *store_mask = gpiosetting;

  return err_mask;
}



/*******************************************************************************
*
*   FUNCTION: storeGPIOsetting(unsigned int gpioid, tBool value, unsigned int *store_mask)
*
*   DESCRIPTION: This function save the mask of Microphone ????
*
*******************************************************************************/
int fc_streamrouter_tclConfigMic::storeGPIOsetting(unsigned int gpioid, tBool value, unsigned int *store_mask)
{
  if(store_mask != NULL)
  {
    if(value != 0)
    {
      *store_mask |= gpioid;
    }
    else
    {
      *store_mask &= ~(unsigned int)gpioid;
    }
  }
  return 0;
}



/*******************************************************************************
                  TIMER MIC PWM FUNCTION
*******************************************************************************/

/*******************************************************************************
*
*   FUNCTION: vSetTimerMicPwm(tU32 pwmHz)
*
*   Set timer for mic adc pwm
*
*******************************************************************************/
#define OEDT_ACOUSTICIN_T001_DEVICE_NAME_IN \
              OSAL_C_STRING_DEVICE_ACOUSTICIN_IF_SPEECHRECO "/OedtWithoutRate"

#define OEDT_ACOUSTICIN_T001_SAMPLE_RATE_IN          16000
tVoid fc_streamrouter_tclConfigMic::vSetTimerMicPwm(tU32 pwmHz)
{
  (void)pwmHz;
   m_pwmDevice = OSAL_IOOpen(OEDT_ACOUSTICIN_T001_DEVICE_NAME_IN/*OSAL_C_STRING_DEVICE_PWM_MICADC_CLOCK*/, OSAL_EN_READONLY);
   if (OSAL_ERROR == m_pwmDevice)
   {
       NORMAL_M_ASSERT_ALWAYS();
       ETG_TRACE_USR4((">>> ! UNABLE TO OPEN ACOUSTIC DEVICE ! <<<"));
   }

   else
   {
       tS32 s32Ret = OSAL_ERROR;
       OSAL_trAcousticSampleRateCfg   rSampleRateCfg;
       rSampleRateCfg.enCodec     = OSAL_EN_ACOUSTIC_ENC_PCM;
       rSampleRateCfg.nSamplerate = OEDT_ACOUSTICIN_T001_SAMPLE_RATE_IN;
       s32Ret = OSAL_s32IOControl(m_pwmDevice, OSAL_C_S32_IOCTRL_ACOUSTICIN_SETSAMPLERATE, (tS32)&rSampleRateCfg);
       if(OSAL_OK != s32Ret)
       {
           ETG_TRACE_USR4(("OEDT_ACOUSTIC: ERROR SETSAMPLERATE IN: %u\n", (unsigned int)rSampleRateCfg.nSamplerate));

       }
       else
       {
           ETG_TRACE_USR4(("OEDT_ACOUSTIC:"
                            "SUCCESS SETSAMPLERATE IN: %u\n",
               (unsigned int)rSampleRateCfg.nSamplerate));
       }
       OSAL_s32IOClose(m_pwmDevice);
   }


/*
   if(m_pwmDevice != OSAL_ERROR)
   {
     ETG_TRACE_USR4(("Setting mic PWM: %d", pwmHz));

      OSAL_trPWM_SetPwm tSetPwm;
      tS32 s32RetVal = OSAL_ERROR;

      tSetPwm.hz = pwmHz;
      tSetPwm.percent_high = 0x7fff;      // duty cycle 50%
      s32RetVal = OSAL_s32IOControl(m_pwmDevice, OSAL_C_S32_IOCTRL_PWM_SET_PWM, (tS32)(&tSetPwm));

      if(s32RetVal == OSAL_ERROR)
      {
         NORMAL_M_ASSERT_ALWAYS();
         ETG_TRACE_USR4((">>> ! UNABLE TO SET PWM ! <<<"));
      }
   }
   else
   {
      NORMAL_M_ASSERT_ALWAYS();
      ETG_TRACE_USR4((">>> ! UNABLE TO OPEN PWM DEVICE ! <<<"));
   }

*/


}
#endif




