/******************************************************************************
 * FILE        : ecnrclient.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR Test Client
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2015 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 01.05.2015  | Patrick Rey            | initial version
 *             |                        | test client for ECNR daemon
 *****************************************************************************/




 /*****************************************************************************
 *  example of parameter set
 *
 * ./ecnrclient_out.out  -t 100 -l 20 -p 8 -a 2 -c 202 --spidev=AdevSPIInOut
 * ./ecnrclient_out.out  -t 1000 -l 666 -p 0 -a 2 -c 203 --usesync=1 //to use sync interface
 * ./ecnrclient_out.out  -i Get -n ecnrAsrMode
 * ./ecnrclient_out.out  -i Get -n ecnrAudioState
 * ./ecnrclient_out.out  -i Get -n ecnrAsrDataType
 * ./ecnrclient_out.out  -m ecnrSetMicGain -p 100
 * ./ecnrclient_out.out  -m ecnrGetEngineParameter -p 91
 *
 */


#include <stdio.h>
#include <locale.h>

#include <glib.h>
#include <glib-object.h>

#include <dbus/dbus-glib.h>

#include "ecnr-alsa.h"
#include "ecnr-error.h"
#include "ecnr-service.h"
#include "ecnr-common-defs.h"
#include "ecnr-introspection-client.h"



/* Well-known name for this service. */
#define VALUE_SERVICE_NAME        "org.bosch.ecnr.service"
/* Object path to the provided object. */
#define VALUE_SERVICE_OBJECT_PATH "/"
/* And we're interested in using it through this interface.
   This must match the entry in the interface definition XML. */
#define VALUE_SERVICE_INTERFACE   "org.bosch.ecnr.service"
#define VALUE_ALSA_INTERFACE      "org.bosch.ecnr.alsa"


#define CHANNEL_0 0
#define MAX_STREAMS_PER_DIR 10
#define MAX_RETRY   10


/*typedef*/ enum
{
    RTS_ADEV_SPI_IN  = 0,
    RTS_ADEV_SPI_OUT = 1,
    RTS_ADEV_MAX
}/*rtsd_testadev_t*/;



static trts_cfgadev* adevs = NULL;
static trts_cfgstream streams[] =
{
    [0] = {RTS_ADEV_SPI_IN, CHANNEL_0 },
    [1] = {RTS_ADEV_SPI_OUT, CHANNEL_0 },
};

static volatile gboolean stream_run = FALSE;
static volatile gboolean stream_exit = FALSE;
static GMutex   lock;
static GCond    update;
#ifdef ENABLE_ALSA_DEV_SETTING
const static char* ADEVECNRINOUT_16 = "AdevECNRInOut_16"; //"AdevECNRInOutExt:256";
#endif //ENABLE_ALSA_DEV_SETTING
static const char* ADEVSPIINOUT_16 = "AdevECNRInOut_16";  //"AdevECNRInOutExt:256";
static const char* ADEVSPIDEFAULT = "AdevSPIInOutExt:256"; //"AdevECNRInOutExt:256";
static const char* ADEVACOUSTICOUTSPEECH = "AdevAcousticoutSpeech";
static int  retry_wr = 0;
static int  retry_rd = 0;
static tRTS h;
static trts_cfg cfg;
static int stream_count = 0;
static GStrv deviceIn         = NULL;
static GStrv deviceOut         = NULL;
static GArray*    deviceProperties  = NULL;
static DBusGProxy *ecnr_proxy       = NULL;
static DBusGProxy *ecnr_alsa_proxy  = NULL;


/* main parameter variable */
static gint ecnr_client_streaming_rts_timeout_ms = 100;
static gint ecnr_client_streaming_rts_prefill_ms = 8;
static gint ecnr_client_streaming_loop = 33;
static gint ecnr_client_app_id = 2;
static gint ecnr_client_cfg_id = 204;
static gint ecnr_client_spi_use_sync_int = 0;
static char *ecnr_client_spi_dev_in = NULL;
static char *ecnr_client_spi_dev_out = NULL;

static char *ecnr_client_api_select = NULL;
static char *ecnr_client_api_select_name = NULL;
static char *ecnr_client_api_method_select = NULL;
static guint ecnr_client_api_method_param;


/* audio streaming thread buffer elements */
static  char* in_buffers[MAX_STREAMS_PER_DIR] = {NULL};
static  char* out_buffers[MAX_STREAMS_PER_DIR] = {NULL};
static  int in_size[MAX_STREAMS_PER_DIR] = {0};
static  int out_size[MAX_STREAMS_PER_DIR] = {0};
static  int in =0;
static  int out =0;

static gboolean  ecnr_client_unused_function_warning_suppressor(void)
{
    /* below functions are unused and remove from here when needs
     * to be used.
     */

    (void)&org_bosch_ecnr_service_ecnr_set_noise_reduction_async;
    (void)&org_bosch_ecnr_service_ecnr_get_engine_parameter_async;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_device;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_device_async;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_thread_priority;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_thread_priority_async;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_wide_band;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_wide_band_async;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_device_properties;
    (void)&org_bosch_ecnr_alsa_ecnr_alsa_set_device_properties_async;
    (void)&org_bosch_ecnr_dbug_ecnr_start_debug_data;
    (void)&org_bosch_ecnr_dbug_ecnr_start_debug_data_async;
    (void)&org_bosch_ecnr_dbug_ecnr_end_debug_data;
    (void)&org_bosch_ecnr_dbug_ecnr_end_debug_data_async;
    (void)&org_bosch_ecnr_service_ecnr_initialize_async;
    (void)&org_bosch_ecnr_service_ecnr_initialize_sync_async;
    (void)&org_bosch_ecnr_service_ecnr_reset;
    (void)&org_bosch_ecnr_service_ecnr_reset_async;
    (void)&org_bosch_ecnr_service_ecnr_destroy_async;
    (void)&org_bosch_ecnr_service_ecnr_terminate;
    (void)&org_bosch_ecnr_service_ecnr_terminate_async;
    (void)&org_bosch_ecnr_service_ecnr_set_configuration;
    (void)&org_bosch_ecnr_service_ecnr_get_version;
    (void)&org_bosch_ecnr_service_ecnr_set_configuration_async;
    (void)&org_bosch_ecnr_service_ecnr_set_configuration_sync_async;
    (void)&org_bosch_ecnr_service_ecnr_get_ecnr_configuration;
    (void)&org_bosch_ecnr_service_ecnr_get_ecnr_configuration_async;
    (void)&org_bosch_ecnr_service_ecnr_start_audio_async;
    (void)&org_bosch_ecnr_service_ecnr_start_audio_ext;
    (void)&org_bosch_ecnr_service_ecnr_start_audio_ext_async;
    (void)&org_bosch_ecnr_service_ecnr_stop_audio_async;
    (void)&org_bosch_ecnr_service_ecnr_frame_info;
    (void)&org_bosch_ecnr_service_ecnr_frame_info_async;
    (void)&org_bosch_ecnr_service_ecnr_get_version_async;
    (void)&org_bosch_ecnr_service_ecnr_mic_clip_stats;
    (void)&org_bosch_ecnr_service_ecnr_mic_clip_stats_async;
    (void)&org_bosch_ecnr_service_ecnr_ref_clip_stats;
    (void)&org_bosch_ecnr_service_ecnr_ref_clip_stats_async;
    (void)&org_bosch_ecnr_service_ecnr_recv_clip_stats;
    (void)&org_bosch_ecnr_service_ecnr_recv_clip_stats_async;
    (void)&org_bosch_ecnr_service_ecnr_mic_out_clip_stats;
    (void)&org_bosch_ecnr_service_ecnr_get_status_message;
    (void)&org_bosch_ecnr_service_ecnr_mic_out_clip_stats_async;
    (void)&org_bosch_ecnr_service_ecnr_recv_out_clip_stats;
    (void)&org_bosch_ecnr_service_ecnr_recv_out_clip_stats_async;
    (void)&org_bosch_ecnr_service_ecnr_set_send_mute_switch;
    (void)&org_bosch_ecnr_service_ecnr_set_send_mute_switch_async;
    (void)&org_bosch_ecnr_service_ecnr_set_send_gain_delta;
    (void)&org_bosch_ecnr_service_ecnr_set_send_gain_delta_async;
    (void)&org_bosch_ecnr_service_ecnr_set_mic_gain_async;
    (void)&org_bosch_ecnr_service_ecnr_set_pass_through_mode;
    (void)&org_bosch_ecnr_service_ecnr_set_pass_through_mode_async;
    (void)&org_bosch_ecnr_service_ecnr_set_nr_comb_floor;
    (void)&org_bosch_ecnr_service_ecnr_get_status_message_async;
    (void)&org_bosch_ecnr_service_ecnr_set_nr_comb_floor_async;
    (void)&org_bosch_ecnr_service_ecnr_set_echo_cancellation;
    (void)&org_bosch_ecnr_service_ecnr_set_echo_cancellation_async;
    (void)&org_bosch_ecnr_service_ecnr_set_noise_reduction;

    return TRUE;
}

static gboolean ecnr_client_initialize(DBusGProxy* remoteobj, guchar ecnrAppId,guint ecnrConfigurationId  )
{
    gboolean bret = TRUE;
    GError* error = NULL;

    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */
    bret = org_bosch_ecnr_service_ecnr_initialize (remoteobj, ecnrAppId, ecnrConfigurationId, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrInitialize failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}

static gboolean ecnr_client_initialize_sync
(
    DBusGProxy* remoteobj,
    guchar ecnrAppId,
    guint ecnrConfigurationId,
  GStrv *ecnrInDev,
  GStrv *ecnrOutDev,
    GArray** ecnrDeviceProperties
)
{
  gboolean bret = TRUE;
    GError* error = NULL;
    GStrv _ecnrInDev = NULL;
    GStrv _ecnrOutDev = NULL;

    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */
    bret = org_bosch_ecnr_service_ecnr_initialize_sync (remoteobj, ecnrAppId, ecnrConfigurationId, &_ecnrInDev, &_ecnrOutDev, ecnrDeviceProperties, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrInitializesync failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    *ecnrInDev = _ecnrInDev;
    *ecnrOutDev = _ecnrOutDev;

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}


static gboolean ecnr_client_set_configuration_sync
(
    DBusGProxy* remoteobj,
    guchar ecnrAppId,
    guint ecnrConfigurationId,
    GStrv *ecnrInDev,
    GStrv *ecnrOutDev,
    GArray** ecnrDeviceProperties
)
{
  gboolean bret = TRUE;
    GError* error = NULL;
    GStrv _ecnrInDev = NULL;
    GStrv _ecnrOutDev = NULL;

    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */
    bret = org_bosch_ecnr_service_ecnr_set_configuration_sync (remoteobj, ecnrAppId, ecnrConfigurationId, &_ecnrInDev, &_ecnrOutDev, ecnrDeviceProperties, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrSetConfigurationsync failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    *ecnrInDev = _ecnrInDev;
    *ecnrOutDev = _ecnrOutDev;

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}


static gboolean ecnr_cliemt_initialize(DBusGProxy* remoteobj, guchar ecnrAppId,guint ecnrConfigurationId  )
{
    gboolean bret = TRUE;
    GError* error = NULL;

    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */
    bret = org_bosch_ecnr_service_ecnr_initialize (remoteobj, ecnrAppId, ecnrConfigurationId, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrInitialize failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}


static gboolean ecnr_cliemt_start_audio(DBusGProxy* remoteobj, guchar ecnrAppId )
{
    gboolean bret = TRUE;
    GError* error = NULL;

    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */

    bret = org_bosch_ecnr_service_ecnr_start_audio (remoteobj, ecnrAppId, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrStartAudio failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}


static gboolean ecnr_cliemt_stop_audio(DBusGProxy* remoteobj, guchar ecnrAppId )
{
    gboolean bret = TRUE;
    GError* error = NULL;


    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */

    bret = org_bosch_ecnr_service_ecnr_stop_audio (remoteobj, ecnrAppId, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrStopAudio failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}


static void ecnr_client_process_buffers(char**o_buffers, char**i_buffers,int*in_sizes, int num_out, int num_in )
{
    int num;
    for(num=0;(num<num_in) && (num < num_out);num++)
        memcpy(o_buffers[num], i_buffers[num], (size_t)(in_sizes[num]));

    usleep(2000);//3000); //14000);
}


static gboolean ecnr_cliemt_destroy(DBusGProxy* remoteobj, guchar ecnrAppId )
{
    gboolean bret = TRUE;
    GError* error = NULL;


    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */

    bret = org_bosch_ecnr_service_ecnr_destroy (remoteobj, ecnrAppId, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrDestroy failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}

#ifdef ENABLE_ALSA_DEV_SETTING
static gboolean ecnr_cliemt_set_alsa_device(DBusGProxy* remoteobj, guchar ecnrAppId, guint AlsaDeviceSelect, char * AlsaDeviceName )
{
    gboolean bret = TRUE;
    GError* error = NULL;

    ECNR_PRINT_TIME_MSG("INIT");

    /* ecnr remote object */

    bret = org_bosch_ecnr_alsa_ecnr_alsa_set_device (remoteobj, ecnrAppId, AlsaDeviceSelect, AlsaDeviceName, &error);

    if (!bret && error != NULL)
    {
        pr_message("ecnrAlsaSetDevice failed with error code(%d) and message (%s)",error->code, error->message );
        g_clear_error(&error);
        bret = FALSE;
    }

    ECNR_PRINT_TIME_MSG("FINAL");

    return bret;
}
#endif //ENABLE_ALSA_DEV_SETTING

static void ecnr_client_audio_thread_set_prio(void)
{
    int err;
    struct sched_param sched_param;

    err = sched_getparam(0, &sched_param);
    if (err < 0)
        pr_message("sched_getparam error ");

    sched_param.sched_priority = sched_get_priority_max(SCHED_FIFO);
    err = sched_setscheduler(0, SCHED_FIFO, &sched_param);
    if(!err)
    {
        pr_message("Priority of ecnr_alsa_thread set to SCHED_FIFO %d", sched_param.sched_priority);
    }
    else
    {
        pr_warning("sched_setscheduler error %d when attempting to set priority SCHED_FIFO %d", err, sched_param.sched_priority);
    }
    return;
}


static int ecnr_client_retry(int err_in, int* retry_in)
{
    int err_out = ECNR_ERR_OK;
    int retry = *retry_in;

    if( err_in == -ENODATA || err_in == -EPIPE  )
    {

        if (retry < MAX_RETRY)
        {
            pr_message("rts streaming failed with err(%d) and current retry count is (%d)\n", err_in, retry);
            retry++;
            err_out = ECNR_ERR_OK;

            /* print static previous to destroy call */
            err_out = rts_destroy(h);
            if(err_out == ECNR_ERR_OK)
                err_out = rts_create( &h, &cfg);
            if(err_out)
            {
                pr_message("rts creation failed with err(%d) and current retry count is (%d)\n",err_out,retry);
            }
        }
    }
    else
        err_out =  err_in;


    *retry_in = retry;

    return err_out;
}


static gint ecnr_client_audio_thread_init()
{
  unsigned int    i = 0;
    gint    err = ECNR_ERR_OK;

    ECNR_PRINT_TIME_MSG("ENTER");

    in = 0;
    out = 0;

    /**************************************************
        setup spi devices with returned value of ecnr
    */

    if(ecnr_client_spi_use_sync_int)
    {
        guint8          dev_mask     = 0x00;
        gint            dev_prop = 0;
        unsigned int    rate;              /*any, e.g. 8000-192000*/
        unsigned int    period_time;      /*e.g. 64/128/256/512*/
        unsigned int    start_time_out;   /*e.g. 100 ms*/
        unsigned int    nb_period_Buffer;  /*e.g. 3*/
        gchar** tempDev    = NULL;
        gchar** devIN      = NULL;
        gchar** devOUT     = NULL;

        dev_prop = g_array_index (deviceProperties, gint, 0);

        dev_mask = (guint8)dev_prop;
        pr_message("dev_mask(%d)", dev_mask);


        /*******************************
         * capture / playback device configuration
         */

        if(adevs[RTS_ADEV_SPI_IN].pcmname)
        {
            g_free((gpointer)adevs[RTS_ADEV_SPI_IN].pcmname);
            adevs[RTS_ADEV_SPI_IN].pcmname = NULL;
        }
        tempDev = g_strsplit(*deviceIn," ",-1);
        devIN = g_strsplit(*tempDev,"=",-1);
        adevs[RTS_ADEV_SPI_IN].pcmname = g_strdup((gchar *) devIN[1]);
        pr_message("capture device name(%s)",  g_strdup((gchar *) devIN[1]));

        if(adevs[RTS_ADEV_SPI_OUT].pcmname)
        {
            g_free((gpointer)adevs[RTS_ADEV_SPI_OUT].pcmname);
            adevs[RTS_ADEV_SPI_OUT].pcmname = NULL;
        }
        tempDev = g_strsplit(*deviceOut," ",-1);
        devOUT = g_strsplit(*tempDev,"=",-1);
        adevs[RTS_ADEV_SPI_OUT].pcmname  = g_strdup((gchar *) devOUT[1]);
        pr_message("playback device name(%s)", g_strdup((gchar *) devOUT[1]));

        rate = (unsigned int)g_array_index (deviceProperties, gint, 1);
        adevs[RTS_ADEV_SPI_IN].rate             = (unsigned int)rate;
        adevs[RTS_ADEV_SPI_OUT].rate             = (unsigned int)rate;
        pr_message("first device rate(%d)", adevs[RTS_ADEV_SPI_IN].rate );

        period_time = (unsigned int)g_array_index (deviceProperties, gint, 3);

        adevs[RTS_ADEV_SPI_IN].period_frames    = (unsigned int)((period_time * rate) / 1000);
        adevs[RTS_ADEV_SPI_OUT].period_frames    = (unsigned int)((period_time * rate) / 1000);
        pr_message("first device period frames(%d)", adevs[RTS_ADEV_SPI_IN].period_frames );

        adevs[RTS_ADEV_SPI_IN].format           = SND_PCM_FORMAT_S16_LE;
        adevs[RTS_ADEV_SPI_OUT].format           = SND_PCM_FORMAT_S16_LE;
        pr_message("first device stream format(%d)", adevs[RTS_ADEV_SPI_IN].format );

        start_time_out = (unsigned int)g_array_index (deviceProperties, gint, 4);
        adevs[RTS_ADEV_SPI_IN].startup_tout     = (signed short)start_time_out;
        adevs[RTS_ADEV_SPI_OUT].startup_tout     = (signed short)start_time_out;
        pr_message("first device startup timeout(%d)", adevs[RTS_ADEV_SPI_IN].startup_tout );

        nb_period_Buffer = (unsigned int)g_array_index (deviceProperties, gint, 2);
        if(nb_period_Buffer <= 2)
        {
            cfg.prefill_ms = period_time;
            pr_message("all devices prefill(%d)", cfg.prefill_ms);
        }
    }


    for (i=0;i<cfg.num_streams;i++)
    {

        adevs[i].startup_tout = (signed short) ecnr_client_streaming_rts_timeout_ms;
        pr_message("set ecnr client streaming rts time out(%d) for device idx(%d)", adevs[i].startup_tout, i);

        if(!ecnr_client_spi_use_sync_int)
        {
            if(i == RTS_ADEV_SPI_IN)
            {
                if(adevs[i].pcmname)
                {
                    g_free((gpointer)adevs[i].pcmname);
                    adevs[i].pcmname = NULL;
                }
                adevs[i].pcmname = g_strdup(ecnr_client_spi_dev_in);
                if(adevs[i].pcmname)
                {
                    pr_message("assign ecnr client capture device (%s) for device idx(%d)", adevs[i].pcmname, i);
                }
            }
            else if(i == RTS_ADEV_SPI_OUT)
            {
                if(adevs[i].pcmname)
                {
                    g_free((gpointer)adevs[i].pcmname);
                    adevs[i].pcmname = NULL;
                }
                adevs[i].pcmname = g_strdup(ecnr_client_spi_dev_out);
                if(adevs[i].pcmname)
                {
                    pr_message("assign ecnr client playback device (%s) for device idx(%d)", adevs[i].pcmname, i);
                }
            }
            else
            {
                /* do nothing*/
            }
        }

        if (cfg.adevs[cfg.streams[i].adevidx].dir == SND_PCM_STREAM_PLAYBACK)
        {
            out_size[out]   = (int)(cfg.adevs[cfg.streams[i].adevidx].period_frames*2*1);
            out_buffers[out]=(char*)malloc((size_t)(out_size[out]));
            out++;

            pr_message("set playback device(%s) Nb(%d) ", adevs[i].pcmname, out);
        }
        else
        {
            in_size[in] = (int)(cfg.adevs[cfg.streams[i].adevidx].period_frames*2*1);
            in_buffers[in]=(char*)malloc((size_t)(in_size[in]));
            in++;
            pr_message("set capture device(%s) Nb(%d) ", adevs[i].pcmname, in);
        }
    }

    return err;
}


static void ecnr_client_audio_thread_finalize()
{
    int i = 0;

    ECNR_PRINT_TIME_MSG("ENTER");

    for (i = 0; i < MAX_STREAMS_PER_DIR; i++)
    {
        if(in_buffers[i])
            free(in_buffers[i]);
    }

    for (i = 0; i < MAX_STREAMS_PER_DIR; i++)
    {
        if(out_buffers[i])
            free(out_buffers[i]);
    }

    in = 0;
    out = 0;

}

gpointer ecnr_client_audio_thread(gpointer data)
{
    int err;
    int loop = 0;

    cfg.features = 0;
    cfg.adevs = adevs;
    cfg.num_adevs = RTS_ADEV_MAX; // sizeof(adevs)/sizeof(trts_cfgadev);
    cfg.streams = streams;
    cfg.num_streams = sizeof(streams)/sizeof(trts_cfgstream);
    cfg.prefill_ms = (unsigned int)ecnr_client_streaming_rts_prefill_ms;

    /* lint Info 715: prio2 */
    data = data;

    ECNR_PRINT_TIME_MSG("INIT");

    /* initalize device structures and rts structures*/
    ecnr_client_audio_thread_init();

    /* set priority of this thread */
    ecnr_client_audio_thread_set_prio();

    /* create rts instance */
    err = rts_create( &h, &cfg);
    if (err)
    {
        pr_message("rts create failed");
    }

    loop = 0;

    while (!err  && (loop < ecnr_client_streaming_loop))
    {

        ecnr_client_process_buffers(out_buffers, in_buffers, in_size, out, in);

        err = rts_write(h, (void**)out_buffers);
        if(err)
        {
            pr_message("write failed with err(%d)", err);
            err = ecnr_client_retry(err, &retry_wr);
            if (!err)
            {
              ECNR_PRINT_TIME_MSG("!ecnr_client_retry failed!\n");
            }
        }
        else
            retry_wr = 0;

        if(!loop)
        {
            g_mutex_lock(&lock);
            stream_run = TRUE;

            /* signal send to wake up from call within "ecnr_alsa_start()" function,
                   the return code is returned to object interface
            */
            g_cond_broadcast(&update);
            g_mutex_unlock(&lock);
            pr_message("client streaming is started");
        }

        err = rts_read(h, (void**)in_buffers);
        if(err)
        {
            pr_message("read failed with err(%d)", err);
            err = ecnr_client_retry(err, &retry_rd);
        }
        else
            retry_rd = 0;

        if(!err)
            loop++;
    }

    err = rts_destroy(h);

    ecnr_client_audio_thread_finalize();

    g_mutex_lock(&lock);

    stream_exit = TRUE;
    stream_run = FALSE;
    stream_count = loop;

    g_cond_broadcast(&update);
    g_mutex_unlock(&lock);

    ECNR_PRINT_TIME_MSG("FINAL");

    return NULL;// err;
}


static gboolean ecnr_client_wait_streaming_start(glong timeout_ms)
{
    gboolean success = FALSE ;

    ECNR_PRINT_TIME_MSG("INIT");

    gint64 tout = (gint64)(timeout_ms) ;
    gint64 end_time = g_get_monotonic_time () + tout * G_TIME_SPAN_MILLISECOND;

    /* wait the audio streaming thread has started streaming and
       updated state initialized to true after at least two R/W
       was processed successfully
    */
    g_mutex_lock(&lock);
    while (!stream_run)
    {
        //if (!g_cond_timed_wait(&update, &lock, &abs_timeout)) //Depricated
        if (!g_cond_wait_until(&update, &lock, end_time))
            break;
    }

    /* streaming was started successfully return TRUE */
    if(stream_run)
    {
        success = TRUE;
        ECNR_PRINT_TIME_MSG("streaming started successfully");
    }
    g_mutex_unlock(&lock);

    ECNR_PRINT_TIME_MSG("FINAL");

    return success;
}

static gboolean ecnr_client_wait_streaming_end()
{
    gboolean success = FALSE;

    /* wait release from ecnr_alsa_start() function */
    g_mutex_lock(&lock);

    while (stream_run && !stream_exit)
    {
        pr_message("wait streaming termination  \n");
        g_cond_wait(&update, &lock);
    }

    if (!stream_run && stream_exit)
        success = TRUE;

    g_mutex_unlock(&lock);

    return success;
}

static void ecnr_client_init()
{

    ECNR_PRINT_TIME_MSG("ENTER");

    g_mutex_init(&lock);
    g_cond_init(&update);

    stream_run = FALSE;
    stream_exit = FALSE;

    retry_wr= 0;
    retry_rd= 0;
    stream_count = 0;

    adevs = g_malloc0( RTS_ADEV_MAX * sizeof(trts_cfgadev));

   // static trts_cfgadev adevs[] = {

    adevs[RTS_ADEV_SPI_IN].pcmname          = g_strdup(ADEVSPIINOUT_16);
    adevs[RTS_ADEV_SPI_IN].dir              = SND_PCM_STREAM_CAPTURE;
    adevs[RTS_ADEV_SPI_IN].rate             = 16000;
    adevs[RTS_ADEV_SPI_IN].period_frames    = 128;  /* 8ms period */
    adevs[RTS_ADEV_SPI_IN].format           = SND_PCM_FORMAT_S16_LE;
    adevs[RTS_ADEV_SPI_IN].startup_tout     = 100; /* 100 ms timeout*/

    adevs[RTS_ADEV_SPI_OUT].pcmname          = g_strdup(ADEVACOUSTICOUTSPEECH);
    adevs[RTS_ADEV_SPI_OUT].dir              = SND_PCM_STREAM_PLAYBACK;
    adevs[RTS_ADEV_SPI_OUT].rate             = 16000;
    adevs[RTS_ADEV_SPI_OUT].period_frames    = 128;  /* 8ms period */
    adevs[RTS_ADEV_SPI_OUT].format           = SND_PCM_FORMAT_S16_LE;
    adevs[RTS_ADEV_SPI_OUT].startup_tout     = 100; /* 100 ms timeout*/

}

static void ecnr_client_finalize()
{
    ECNR_PRINT_TIME_MSG("ENTER");

    g_mutex_clear(&lock);
    g_cond_clear(&update);

    g_free (ecnr_client_spi_dev_in);
    g_free (ecnr_client_spi_dev_out);

    retry_wr= 0;
    retry_rd= 0;
    stream_count = 0;

    g_free(adevs);

}



/** main **/




static const GOptionEntry ecnr_client_opts[] =
{
    {
        .long_name = "timeout",
        .short_name = 't',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_streaming_rts_timeout_ms,
        .description = "set time out in rts for the streaming client  ",
        .arg_description = "T"
    },
    {
        .long_name = "prefill",
        .short_name = 'f',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_streaming_rts_prefill_ms,
        .description = "retrieve ecnr infos, options:[datapool, ecnr-config,tuning-trigger]",
        .arg_description = "F"
    },
    {
        .long_name = "loop",
        .short_name = 'l',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_streaming_loop,
        .description = "retrieve ecnr infos, options:[datapool, ecnr-config,tuning-trigger]",
        .arg_description = "L"
    },
    {
        .long_name = "appId",
        .short_name = 'a',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_app_id,
        .description = "rset application Id",
        .arg_description = "A"
    },
    {
        .long_name = "cfgId",
        .short_name = 'c',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_cfg_id,
        .description = "set ecnr configuration Id, dataset Id",
        .arg_description = "C"
    },
    {
        .long_name = "spidevin",
        .short_name = 's',
        .flags = 0,
        .arg = G_OPTION_ARG_STRING,
        .arg_data = &ecnr_client_spi_dev_in,
        .description = "set spi streaming capture device",
        .arg_description = "S"
    },
    {
        .long_name = "spidevout",
        .short_name = 's',
        .flags = 0,
        .arg = G_OPTION_ARG_STRING,
        .arg_data = &ecnr_client_spi_dev_out,
        .description = "set spi streaming playback device",
        .arg_description = "S"
    },
    {
        .long_name = "usesync",
        .short_name = 'y',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_spi_use_sync_int,
        .description = "select client use sync interface",
        .arg_description = "Y"
    },
    {
        .long_name = "interface",
        .short_name = 'i',
        .flags = 0,
        .arg = G_OPTION_ARG_STRING,
        .arg_data = &ecnr_client_api_select,
        .description = "ecnr client call api",
        .arg_description = "I"
    },
    {
        .long_name = "method call",
        .short_name = 'm',
        .flags = 0,
        .arg = G_OPTION_ARG_STRING,
        .arg_data = &ecnr_client_api_method_select,
        .description = "ecnr client call api method",
        .arg_description = "M"
    },
    {
        .long_name = "method call parameter",
        .short_name = 'p',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_client_api_method_param,
        .description = "ecnr client call api method parameter",
        .arg_description = "P"
    },
    {
        .long_name = "name",
        .short_name = 'n',
        .flags = 0,
        .arg = G_OPTION_ARG_STRING,
        .arg_data = &ecnr_client_api_select_name,
        .description = "select ecnr server api to call",
        .arg_description = "N"
    },
    { NULL }
};

/********************
 option function handler
*/

gint ecnr_client_use_sync_opt_handler(gint use_sync_option)
{
    gboolean bret = (gboolean)ECNR_ERR_OK;
    guint  i;
    guint32 dev_prop = 0;

    if(use_sync_option == 1)
    {
        bret = ecnr_client_initialize_sync
        (
            ecnr_proxy,
            (guchar)ecnr_client_app_id, /*ECNR_APPID_SPI*/
            (guint)ecnr_client_cfg_id, /*SPI_DATASET_4*/
            &deviceIn,
            &deviceOut,
            &deviceProperties
        );
    }
    else if (use_sync_option == 2)
    {
        /* call initialize for spi app and config Id 204 */
        bret = ecnr_client_initialize(ecnr_proxy,  /*ECNR_APPID_SPI*/(guchar)ecnr_client_app_id,  /*SPI_DATASET_4*/(guint)ecnr_client_cfg_id);

        if (!bret)
        {
            ECNR_PRINT_TIME_MSG("ecnr_client_initialize failed");
        }
        else
            ECNR_PRINT_TIME_MSG("set AppId(2) with ecnr configuration Id(204)");

        bret = ecnr_client_set_configuration_sync
        (
            ecnr_proxy,
            (guchar)ecnr_client_app_id, /*ECNR_APPID_SPI*/
            (guint)ecnr_client_cfg_id, /*SPI_DATASET_4*/
            &deviceIn,
            &deviceOut,
            &deviceProperties
        );
    }

    if (!bret)
    {
        ECNR_PRINT_TIME_MSG("ecnr_client_initialize_sync failed");
    }

    pr_message("number of properties[%d]", deviceProperties->len);

    for (i = 0; i < deviceProperties->len; i++)
    {
        dev_prop = (guint32)g_array_index (deviceProperties, gint, i);
        pr_message(" device_property[%d]: %d", i , dev_prop);
    }

    if(*deviceIn)
        pr_message("in device name(%s)", *deviceIn);

    if(*deviceOut)
        pr_message("out device name(%s)", *deviceOut);

    return (gint)bret;
}





int main(int argc, char **argv)
{
    DBusGConnection *bus;
    GError *error;
    GMainLoop *loop = NULL;
    guchar   *data_read = NULL;
    gboolean bret = TRUE;
    GThread   *thread;
    GOptionContext *context;
    guint i;


    ECNR_PRINT_TIME_MSG("Start ecnr client");

    loop = g_main_loop_new(NULL, FALSE);

    /***********************
      evaluation main keys
    ************************/

    setlocale(LC_ALL, "");
    context = g_option_context_new("");
    g_option_context_set_summary(context, "The ECNR client implements "
    "handsfree audio using to QNX/NUANCE libraries\n and ALSA audio "
    "devices.");
    g_option_context_add_main_entries(context, ecnr_client_opts, NULL);
    if (!g_option_context_parse(context, &argc, &argv, &error))
    {
        g_print("ecnr client: %s\n", error->message);
        g_error_free(error);
        g_option_context_free(context);
        _exit(1);
    }

    g_option_context_free(context);


    /* print main paramter */
    pr_message(" ecnr client rts time out                : %d", ecnr_client_streaming_rts_timeout_ms);
    pr_message(" ecnr client rts prefill in ms           : %d", ecnr_client_streaming_rts_prefill_ms);
    pr_message(" ecnr client streaming loop              : %d", ecnr_client_streaming_loop);
    pr_message(" ecnr client application Id              : %d", ecnr_client_app_id);
    pr_message(" ecnr client ecnr dataset Id             : %d", ecnr_client_cfg_id);
    pr_message(" ecnr client sync option                 : %d", ecnr_client_spi_use_sync_int);
    pr_message(" ecnr client spi capture  device         : %s", (ecnr_client_spi_dev_in)?ecnr_client_spi_dev_in:"NULL") ;
    pr_message(" ecnr client spi playback device         : %s", (ecnr_client_spi_dev_out)?ecnr_client_spi_dev_out:"NULL");
    pr_message(" ecnr client call ecnr server api only   : %s", (ecnr_client_api_select)?ecnr_client_api_select:"NULL") ;
    pr_message(" ecnr client selection of api object     : %s", (ecnr_client_api_select_name)?ecnr_client_api_select_name:"NULL");
    pr_message(" ecnr client call ecnr server api only   : %s", (ecnr_client_api_method_select)?ecnr_client_api_method_select:"NULL") ;



    if(!ecnr_client_spi_dev_in && !ecnr_client_spi_use_sync_int )
        ecnr_client_spi_dev_in = g_strdup(ADEVSPIDEFAULT);

    if(!ecnr_client_spi_dev_out && !ecnr_client_spi_use_sync_int )
        ecnr_client_spi_dev_out = g_strdup(ADEVSPIDEFAULT);

    /* init client */



    if(!ecnr_client_api_select && !ecnr_client_api_method_select)
        ecnr_client_init();

    error = NULL;
    bus = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (bus == NULL)
    {
        g_assert(error != NULL);
        if (error)
            pr_critical("Failed to open connection to bus: %s", error->message);
        g_error_free(error);
        _exit(1);
    }

    if(!ecnr_client_api_select && !ecnr_client_api_method_select)
    {
        ecnr_proxy =
        dbus_g_proxy_new_for_name(bus,
                  VALUE_SERVICE_NAME,        /* name */
                  VALUE_SERVICE_OBJECT_PATH, /* obj path */
                  VALUE_SERVICE_INTERFACE    /* interface */);
        if (ecnr_proxy == NULL)
        {
            g_assert(error != NULL);
            pr_critical("Couldn't create the proxy object");
            _exit(1);
        }

        ecnr_alsa_proxy =
        dbus_g_proxy_new_for_name(bus,
                  VALUE_SERVICE_NAME,        /* name */
                  VALUE_SERVICE_OBJECT_PATH, /* obj path */
                  VALUE_ALSA_INTERFACE    /* interface */);
        if (ecnr_alsa_proxy == NULL)
        {
            g_assert(error != NULL);
            pr_critical("Couldn't create the proxy to alsa object");
            _exit(1);
        }
    }


    ECNR_PRINT_TIME_MSG("main Starting main loop");

    if(ecnr_client_api_select)
    {
        gboolean isintparam = FALSE;
        gboolean isarrayparam = FALSE;
        GValue value_p = G_VALUE_INIT; //{ 0, };



        if (!g_strcmp0(ecnr_client_api_select, "Get"))
        {
            g_print("process Get property(%s) \n", ecnr_client_api_select_name);
            ecnr_proxy = dbus_g_proxy_new_for_name(  bus,
                          "org.bosch.ecnr.service",
                          "/",
                          "org.freedesktop.DBus.Properties");





            if (!g_strcmp0(ecnr_client_api_select_name, "ecnrAsrMode"))
            {
                g_value_init (&value_p, G_TYPE_INT);
                isintparam = TRUE;
            }
            else if (!g_strcmp0(ecnr_client_api_select_name, "ecnrAudioState"))
            {
                g_value_init (&value_p, G_TYPE_INT);
                isintparam = TRUE;
            }
            else if (!g_strcmp0(ecnr_client_api_select_name, "ecnrAsrDataType"))
            {
                //g_value_init (&value_p, G_TYPE_ARRAY);
                g_value_init (&value_p, DBUS_TYPE_G_UCHAR_ARRAY);
                //g_value_init(&value_p, dbus_g_type_get_collection("GArray", G_TYPE_UCHAR));//G_TYPE_UINT));
                isarrayparam = TRUE;
            }

            /* send set property message to audioprocess daemon */
            if (!dbus_g_proxy_call( ecnr_proxy,
                  "Get",
                  &error,
                  G_TYPE_STRING,"org.bosch.ecnr.service",
                  G_TYPE_STRING, ecnr_client_api_select_name,
                  G_TYPE_INVALID,
                  G_TYPE_VALUE, &value_p,
                  G_TYPE_INVALID))
            {

                g_print("ecnr_client_api_select_name(%s)\n", ecnr_client_api_select_name );
                g_error("Error setting property: \"%s\"", error->message);
                g_error_free (error);
            }

            if(isintparam)
            {
                gint v = g_value_get_int(&value_p);
                g_print("Get value:%d\n", v );
                g_value_unset (&value_p);
            }

            if(isarrayparam)
            {
                GArray*     array = NULL;

                /* Extract the active connections array from the GValue */
                array = g_value_get_boxed (&value_p);

                if (!array)
                {
                    g_warning ("Could not retrieve active connections property");
                }
                else
                {
                    /* And print out the details of each active connection */

                    data_read = &g_array_index(array, guchar, 0);
                    for (i = 0; i < array->len; i++)
                    {
                        g_print("array[%d]:%d\n", i, *data_read++);
                    }
                }
                g_value_unset (&value_p);
            }
        }
    }

    else if(ecnr_client_api_method_select)
    {
        g_print("call method(%s) with parameter value:%d\n", ecnr_client_api_method_select, ecnr_client_api_method_param);

        if (!g_strcmp0(ecnr_client_api_method_select, "ecnrSetMicGain"))
        {

            bret = TRUE; //for lint fix modified
            error = NULL;//for lint fix modified

            ecnr_proxy = dbus_g_proxy_new_for_name(bus,
                                                   VALUE_SERVICE_NAME,        /* name */
                                                   VALUE_SERVICE_OBJECT_PATH, /* obj path */
                                                   VALUE_SERVICE_INTERFACE    /* interface */);


            /* ./audproctestclient_out.out  Method AudprocInitialize 1 */
            g_print("call method(%s) with parameter value:%d\n", ecnr_client_api_method_select, ecnr_client_api_method_param);

            bret = org_bosch_ecnr_service_ecnr_set_mic_gain (ecnr_proxy, (gint) ecnr_client_api_method_param, &error);

            if (!bret && error != NULL)
                g_print("ecnrSetMicGain failed with error code(%d), domain(%d) and message(%s)\n",error->code, error->domain, error->message );


            g_clear_error(&error);
            bret = FALSE;

        }

        else if (!g_strcmp0(ecnr_client_api_method_select, "ecnrGetEngineParameter"))
        {
            GArray*     garead = NULL;
            GError*     err = NULL;


            bret = TRUE; //for lint fix modified
            ecnr_proxy = dbus_g_proxy_new_for_name(bus,
                                                   VALUE_SERVICE_NAME,        /* name */
                                                   VALUE_SERVICE_OBJECT_PATH, /* obj path */
                                                   VALUE_SERVICE_INTERFACE    /* interface */);


            /* ./audproctestclient_out.out  Method AudprocInitialize 1 */
            g_print("call method(%s) with parameter value:%d\n", ecnr_client_api_method_select, ecnr_client_api_method_param);


            bret = org_bosch_ecnr_service_ecnr_get_engine_parameter (ecnr_proxy, ecnr_client_api_method_param, &garead,  &err);

            if (!bret && err != NULL)
            {
                g_print("ecnrGetEngineParameter failed with error code(%d) and message (%s)",err->code, err->message );
                g_clear_error(&err);
                bret = FALSE;
            }


            /* print error code */
            if(garead)
            {
                g_print("probe_1_0, garead->len: %d\n", garead->len);
                if (garead->len == 0)
                {
                    _exit(0);
                }

                data_read = &g_array_index(garead, guchar, 0);
                for (i = 0; i < garead->len; i++)
                {
                    g_print("data[%d]:%d\n", i, *data_read++);
                }
                g_array_free(garead, TRUE);
            }
            else
                g_print("NULL return data buffer \n");


        }
    }
    else
    {

        if(ecnr_client_spi_use_sync_int)
        {
          /* call handler use sync option */
          bret = ecnr_client_use_sync_opt_handler(ecnr_client_spi_use_sync_int);
          if (!bret)
          {
            ECNR_PRINT_TIME_MSG("!ecnr_client_use_sync_opt_handler failed!\n");
          }

        }
        else
        {
          /* call initialize for spi app and config Id 204 */
          bret = ecnr_cliemt_initialize(ecnr_proxy,  /*ECNR_APPID_SPI*/(guchar)ecnr_client_app_id,  /*SPI_DATASET_4*/(guint)ecnr_client_cfg_id);

          if (!bret)
          {
            ECNR_PRINT_TIME_MSG("ecnr_cliemt_initialize failed");
          }
          else
            ECNR_PRINT_TIME_MSG("set AppId(2) with ecnr configuration Id(204)");
        }


      #ifdef ENABLE_ALSA_DEV_SETTING
        /* call alsa device setting  */
        bret = ecnr_cliemt_set_alsa_device(ecnr_alsa_proxy, /*ECNR_APPID_SPI*/ecnr_client_app_id , /*ECNR_MIC_OUT*/2,  ADEVECNRINOUT_16);

        if (!bret)
        {
          ECNR_PRINT_TIME_MSG("ecnr_cliemt_set_alsa_device failed");
        }
        else
        {
          pr_message("set alsa device(%s) for channel (%d)", ADEVECNRINOUT_16, 2 );
        }

        /* call ALSA device setting */
        bret = ecnr_cliemt_set_alsa_device(ecnr_alsa_proxy, /*ECNR_APPID_SPI*/ecnr_client_app_id , /*ECNR_RECV_IN*/3,  ADEVECNRINOUT_16);

         if (!bret)
        {
          ECNR_PRINT_TIME_MSG("ecnr_cliemt_set_alsa_device failed");
        }
        else
        {
            pr_message("set alsa device(%s) for channel (%d)", ADEVECNRINOUT_16, 3 );
        }
      #endif //ENABLE_ALSA_DEV_SETTING


        /******* start test streaming application ************/
        /*       start thread                                */
        /*                                                   */
        /*****************************************************/


        thread = g_thread_try_new ("ecnr_client_audio_thread", ecnr_client_audio_thread, NULL, &error);
        if (!thread)
        {
          pr_message("Can't create ecnr_client_audio_thread\n");
          return 0;
        }


        if(ecnr_client_wait_streaming_start(1000)) /* 1000 ms */
        {
          ECNR_PRINT_TIME_MSG("streaming thread was started");

          /* start audio for spi app and config Id 204 */
          bret = ecnr_cliemt_start_audio(ecnr_proxy, (guchar)ecnr_client_app_id);
          if (!bret)
          {
            ECNR_PRINT_TIME_MSG("ecnr_cliemt_start_audio failed ");
          }
          else
            ECNR_PRINT_TIME_MSG("ecnr streaming was started successfully");

        }


        /* wait until ecnr client streaming terminate */
        while(!ecnr_client_wait_streaming_end())
        {
          pr_message("stream_run (%d)", stream_run );
          ECNR_PRINT_TIME_MSG("streaming end failed");
        }


        ECNR_PRINT_TIME_MSG("client streaming is terminated");
        pr_message("end streaming after streaming count(%d)", stream_count);


        usleep(100000);

        /* call stop for spi app and config Id 204 */
        bret = ecnr_cliemt_stop_audio(ecnr_proxy, (guchar)ecnr_client_app_id);
        if (!bret)
        {
          ECNR_PRINT_TIME_MSG("ecnr_cliemt_stop_audio failed ");
        }
        else
          ECNR_PRINT_TIME_MSG("ecnr streaming was stopped successfully");

        /* call destroy for spi app and config Id 204 */
        bret = ecnr_cliemt_destroy(ecnr_proxy, (guchar)ecnr_client_app_id);
        if (!bret)
        {
          ECNR_PRINT_TIME_MSG("ecnr_cliemt_destroy failed ");
        }
        else
          ECNR_PRINT_TIME_MSG("ecnr instance was destroyed successfully");


        /* release all structure*/
        ecnr_client_finalize();

    }

    ECNR_PRINT_TIME_MSG("exit ecnr client ");

    /* function to suppress unused functions warning */
    (void)&ecnr_client_unused_function_warning_suppressor;

    /* end main loop thread */
    g_main_loop_quit(loop);
    if (loop)
        g_main_loop_unref (loop);

    _exit(0);
}
