/******************************************************************************
 * FILE        : ecnr-unitest-server.c
 * PROJECT     : Gen3
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2014 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 12.12.2014  | Patrick Rey            | initial version for Gen3
 *             |                        | unit testing for ECNR interfaces
 *****************************************************************************/


#include <stdlib.h>
#include <string.h>
#include <signal.h>

#include <glib.h>
#include <glib-unix.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>


#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-configuration.h"
#include "ecnr-engine-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-audio-rts-handler.h"


#if  !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX) &&  !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
#include "ecnr-datapool-access.h"
#endif

/*******************************************************************************
              Constant defintion
*******************************************************************************/


/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
/* command line options */
typedef enum _ecnr_unitest
{
    ECNR_UNITEST_SUITE_DEFAUT = 0,
    ECNR_UNITTEST_SUITE_1 = 1,
    ECNR_UNITTEST_SUITE_2 = 2,
    ECNR_UNITTEST_SUITE_3 = 3,          /*alsa ecnr test use cases*/
    ECNR_UNITTEST_SUITE_ALL = 255,
    ECNR_UNITTEST_SUITE_LAST
}ecnr_unitset;



/***********************************************

 UNITEST SUITE 1: Test function definitions

 - ecnr_unitest_s1_1
 - ecnr_unitest_s1_2
 - ecnr_unitest_s1_3
 - ecnr_unitest_s1_4
 - ecnr_unitest_s1_5
 - ecnr_unitest_s1_6
 - ecnr_unitest_s1_7
 - ecnr_unitest_s1_8
 - ecnr_unitest_s1_9
 - ecnr_unitest_s1_10

 ***********************************************/

static void ecnr_unitest_s1_1(void)
{
    int status = ECNR_ERR_OK;

    pr_message("START_UNITEST_S1_1");

    ecnr_service_init();

    status = ecnr_service_instance_create_new(1);
    g_assert(status == ECNR_ERR_OK);

    status = ecnr_service_do_action_initialize(1);
    g_assert(status == ECNR_ERR_OK);

    status = ecnr_service_do_action_destroy();
    g_assert(status == ECNR_ERR_OK);

    status = ecnr_service_instance_remove(1);
    g_assert(status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_1");

    return;
}

static void ecnr_unitest_s1_2(void)
{
    int status = ECNR_ERR_OK;

    pr_message("START_UNITEST_S1_2");

    ecnr_service_init();

    status = ecnr_service_instance_create_new(2);
    g_assert(status == ECNR_ERR_OK);

    status = ecnr_service_do_action_initialize(201);
    g_assert(status == ECNR_ERR_OK);

    status = ecnr_service_do_action_destroy();
    g_assert(status == ECNR_ERR_OK);

    status = ecnr_service_instance_remove(2);
    g_assert(status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_2");

    return;
}


static void ecnr_unitest_s1_3(void)
{
    pr_message("START_UNITEST_S1_3");

    ecnr_service_init();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    gint ecnr_service_check_AppId_ConfigId_Matching_status_NOK = ECNR_ERR_OK;
    ecnr_service_check_AppId_ConfigId_Matching_status_NOK = ecnr_service_check_AppId_ConfigId_Matching(1, 2);
    g_assert(ecnr_service_check_AppId_ConfigId_Matching_status_NOK != ECNR_ERR_OK);

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_3");

    return;
}


static void ecnr_unitest_s1_4(void)
{
    pr_message("START_UNITEST_S1_4");

    ecnr_service_init();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    gint ecnr_service_check_AppId_ConfigId_Matching_status_OK = ECNR_ERR_OK;
    ecnr_service_check_AppId_ConfigId_Matching_status_OK = ecnr_service_check_AppId_ConfigId_Matching(201, 2);
    g_assert(ecnr_service_check_AppId_ConfigId_Matching_status_OK == ECNR_ERR_OK);

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_4");

    return;
}

static void ecnr_unitest_s1_5(void)
{
    pr_message("START_UNITEST_S1_5");

    ecnr_service_init();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    /* check that this function do not take care for AppId versus CfgId check */
    gint ecnr_service_do_action_initialize_status_OK = ECNR_ERR_OK;
    ecnr_service_do_action_initialize_status_OK = ecnr_service_do_action_initialize(1);
    g_assert(ecnr_service_do_action_initialize_status_OK == ECNR_ERR_OK);

    gint ecnr_service_do_action_destroy_status_OK = ECNR_ERR_OK;
    ecnr_service_do_action_destroy_status_OK = ecnr_service_do_action_destroy();
    g_assert(ecnr_service_do_action_destroy_status_OK == ECNR_ERR_OK);

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_5");

    return;
}

static void ecnr_unitest_s1_6(void)
{
    pr_message("START_UNITEST_S1_6");

    ecnr_service_init();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);


    /* check that this function do not take care for AppId versus CfgId check */
    gint ecnr_service_do_action_initialize_status_OK = ECNR_ERR_OK;
    ecnr_service_do_action_initialize_status_OK = ecnr_service_do_action_initialize(1);
    g_assert(ecnr_service_do_action_initialize_status_OK == ECNR_ERR_OK);

    /* ecnr_service_do_action_destroy should be call previous to a new initilaization */
    gint ecnr_service_do_action_initialize_status_NOK = ECNR_ERR_OK;
    ecnr_service_do_action_initialize_status_NOK = ecnr_service_do_action_initialize(201);
    g_assert(ecnr_service_do_action_initialize_status_NOK != ECNR_ERR_OK);

    gint ecnr_service_do_action_destroy_status_OK = ECNR_ERR_OK;
    ecnr_service_do_action_destroy_status_OK = ecnr_service_do_action_destroy();
    g_assert(ecnr_service_do_action_destroy_status_OK == ECNR_ERR_OK);

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_6");

    return;
}


static void ecnr_unitest_s1_7(void)
{
    pr_message("START_UNITEST_S1_7");


    /* use service function previous to a call to ecnr_service_init */
    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status != ECNR_ERR_OK);


    pr_message("SUCCESS_UNITEST_S1_7");

    return;
}


static void ecnr_unitest_s1_8(void)
{
    pr_message("START_UNITEST_S1_8");

    ecnr_service_init();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    /* prior to new instance creation the last instance should be removed */
    gint ecnr_service_instance_create_new_status_NOK = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status_NOK = ecnr_service_instance_create_new(1);
    g_assert(ecnr_service_instance_create_new_status_NOK != ECNR_ERR_OK);

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_8");

    return;
}

static void ecnr_unitest_s1_9(void)
{
    pr_message("START_UNITEST_S1_9");

    ecnr_service_init();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    gint ecnr_service_alsa_thread_create_status = ECNR_ERR_OK;
    ecnr_service_alsa_thread_create_status = ecnr_service_audio_create();
    g_assert(ecnr_service_alsa_thread_create_status == ECNR_ERR_OK);

    /* wait thread is becoming in running state*/
    usleep(500000);

    /* check thread reference, should be 0, because the ecnr_alsa module was not initialized
       and the alsa thread cannot be created
     */
    struct ecnr_instance_object* my_inst;
    my_inst = ecnr_service_instance_get(2);

    g_assert(my_inst != NULL);

    if(my_inst)
    {
        g_assert(my_inst->ecnr_alsa_thread_ref == 0);
    }

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_9");

    return;
}

static void ecnr_unitest_s1_10(void)
{
    pr_message("START_UNITEST_S1_10");

    /* module initialization, service and alsa  */
    ecnr_service_init();
    ecnr_alsa_initialize();

    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    gint ecnr_service_alsa_thread_create_status = ECNR_ERR_OK;
    ecnr_service_alsa_thread_create_status = ecnr_service_audio_create();
    g_assert(ecnr_service_alsa_thread_create_status == ECNR_ERR_OK);

    /* wait thread is becoming in runing state*/
    usleep(500000);

    /* check thread reference, should be 1, because the alsa thread was created  */
    struct ecnr_instance_object* my_inst;
    my_inst = ecnr_service_instance_get(2);

    g_assert(my_inst != NULL);

    if(my_inst)
    g_assert(my_inst->ecnr_alsa_thread_ref == 1);

    /* initiate termination from the alsa thread */
    ecnr_alsa_exit_wait(ECNR_EXIT_TIMEOUT);

    /* wait thread is becoming terminated*/
    usleep(500000);

    /* check thread reference, should be 0, because the alsa thread was terminated  */
    if(my_inst)
    g_assert(my_inst->ecnr_alsa_thread_ref == 0);

    gint ecnr_service_remove_instance_status = ECNR_ERR_OK;
    ecnr_service_remove_instance_status = ecnr_service_instance_remove(2);
    g_assert(ecnr_service_remove_instance_status == ECNR_ERR_OK);

    /* close module */
    ecnr_service_finalize();
    ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S1_10");

    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_unitest_suite_1
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean ecnr_unitest_suite_1(void)
{
    int   succeed       = ECNR_ERR_OK;

    pr_message("START_UNITEST_SUITE_1");

    g_test_add_func ("/ecnr_suite_1/uc1", ecnr_unitest_s1_1);
    g_test_add_func ("/ecnr_suite_1/uc2", ecnr_unitest_s1_2);
    g_test_add_func ("/ecnr_suite_1/uc3", ecnr_unitest_s1_3);
    g_test_add_func ("/ecnr_suite_1/uc4", ecnr_unitest_s1_4);
    g_test_add_func ("/ecnr_suite_1/uc5", ecnr_unitest_s1_5);
    g_test_add_func ("/ecnr_suite_1/uc6", ecnr_unitest_s1_6);
    g_test_add_func ("/ecnr_suite_1/uc7", ecnr_unitest_s1_7);
    g_test_add_func ("/ecnr_suite_1/uc8", ecnr_unitest_s1_8);
    g_test_add_func ("/ecnr_suite_1/uc9", ecnr_unitest_s1_9);
    g_test_add_func ("/ecnr_suite_1/uc10", ecnr_unitest_s1_10);

    succeed = g_test_run ();


    pr_message("SUCCESS_UNITEST_SUITE_1");

    return succeed;
}

/***********************************************

 UNITEST SUITE 2: Test function definitions

 - ecnr_unitest_s2_1



 ***********************************************/
static void ecnr_unitest_s2_1(void)
{
    pr_message("START_UNITEST_S2_1");

#if  !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX) &&  !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)

    gboolean bRet = FALSE;


    /*initialize dp access */
    ecnr_datapool_access_init();

    /* go through all Datapool*/
    bRet = ecnr_configuration_ut_s2_1_1();
    g_assert(bRet == TRUE);
    bRet = ecnr_configuration_ut_s2_1_2();
    g_assert(bRet == TRUE);
    bRet = ecnr_configuration_ut_s2_1_3();
    g_assert(bRet == TRUE);
    bRet = ecnr_configuration_ut_s2_1_4();
    g_assert(bRet == TRUE);
    bRet = ecnr_configuration_ut_s2_1_5();
    g_assert(bRet == TRUE);
    bRet = ecnr_configuration_ut_s2_1_6();
    g_assert(bRet == TRUE);
    bRet = ecnr_configuration_ut_s2_1_7();
    g_assert(bRet == TRUE);

    //ecnr_configuration_check_config_files();
#endif
    pr_message("SUCCESS_UNITEST_S2_1");

    return;
}


/***********************************************

 UNITEST SUITE 3: Test function definitions

 - ecnr_unitest_s3_1



 ***********************************************/
static void ecnr_unitest_s3_1(void)
{
    pr_message("START_UNITEST_S3_1");


        /* module initialization, service and alsa  */
    ecnr_service_init();
    ecnr_alsa_initialize();

#if 0
    gint ecnr_service_instance_create_new_status = ECNR_ERR_OK;
    ecnr_service_instance_create_new_status = ecnr_service_instance_create_new(2);
    g_assert(ecnr_service_instance_create_new_status == ECNR_ERR_OK);

    gint ecnr_service_alsa_thread_create_status = ECNR_ERR_OK;
    ecnr_service_alsa_thread_create_status = ecnr_service_audio_create();
    g_assert(ecnr_service_alsa_thread_create_status == ECNR_ERR_OK);
#endif

    /* wait thread is becoming in runing state*/
    usleep(500000);

    /* Initialize SPI_App (2) and configuration 204 */
    EcnrObject obj;
    GError *error = NULL;
    gboolean ecnr_init_status;

    ecnr_init_status = ecnr_object_ecnr_initialize(&obj, 2, 204, &error);
    pr_message("ecnr_initialize call status(%d)\n",ecnr_init_status);
    //g_assert(ecnr_init_status == TRUE);


    /* start client */
    GString *alsa_client_spi_in_out = g_string_new(NULL); /* This will spawn an empty string */

    g_string_append(alsa_client_spi_in_out, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc_2.sh");
    g_string_append(alsa_client_spi_in_out, " ALSA_CLIENT_SIM START &");

    pr_message("call system(%s)", alsa_client_spi_in_out->str);
    system(alsa_client_spi_in_out->str);
    g_string_free(alsa_client_spi_in_out, TRUE);

#if 1

    /* start ecnr streaming  */
    gboolean ecnr_start_audio_status;
    int rstr, wstr;
    int rstr_curr, wstr_curr;
    g_clear_error (&error);
    ecnr_start_audio_status = ecnr_object_ecnr_start_audio(&obj, 2, &error);
    //g_assert(ecnr_start_audio_status == TRUE);

    rstr =  0; wstr = 0; rstr_curr = 0; wstr_curr = 0;

    int loop = 0;


    /* test streaming */
    if(ecnr_start_audio_status)
    {
        pr_message("audio is started ");

        while(loop < 5)
        {

            /* trigger update streaming statet*/
            ecnr_alsa_request_print_current_audio_state();

            /* wait thread trigger streaming update */
            usleep(50000);

            ecnr_audio_rts_handler_streaming_rw_state_get(&wstr, &rstr);
            g_assert(wstr > wstr_curr );
            g_assert(rstr > rstr_curr );

            rstr_curr = rstr;
            wstr_curr = wstr;

            /* wait streaming append*/
            usleep(500000);
            /* trigger update streaming statet*/
            ecnr_alsa_request_print_current_audio_state();

            /* wait thread trigger streaming update */
            usleep(50000);

            ecnr_audio_rts_handler_streaming_rw_state_get(&wstr, &rstr);
            g_assert(wstr > wstr_curr );
            g_assert(rstr > rstr_curr );

            /* wait thread trigger streaming update */
            usleep(50000);

            pr_message(" stream iteration count (%d)", loop++);
        }
    }


    /* stop client */
    alsa_client_spi_in_out = g_string_new(NULL);
    g_string_append(alsa_client_spi_in_out, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc_2.sh");
    g_string_append(alsa_client_spi_in_out, " ALSA_CLIENT_SIM STOP &");

    pr_message("call system(%s)", alsa_client_spi_in_out->str);
    system(alsa_client_spi_in_out->str);
    g_string_free(alsa_client_spi_in_out, TRUE);



    /* stop ecnr streaming  */
    gboolean ecnr_stop_audio_status;
    g_clear_error (&error);
    ecnr_stop_audio_status = ecnr_object_ecnr_stop_audio(&obj, 2, &error);
    pr_message("ecnr_stop_audio call status(%d)\n",ecnr_stop_audio_status);
    //g_assert(ecnr_start_audio_status == TRUE);
#endif

    /* destroy ecnr streaming  */
    g_clear_error (&error);
    gboolean ecnr_destroy_audio_status = ecnr_object_ecnr_destroy(&obj, 2, &error);
    pr_message("ecnr destroy audio status (%d)", ecnr_destroy_audio_status);
    //g_assert(ecnr_start_audio_status == TRUE);

    /* close module */
    //ecnr_service_finalize();
    //ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_S3_1");

    return;
}



/*******************************************************************************
*
* FUNCTION: ecnr_unitest_suite_2
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean ecnr_unitest_suite_2(void)
{
    int   succeed       = ECNR_ERR_OK;

    pr_message("START_UNITEST_SUITE_2");

    g_test_add_func ("/ecnr_suite_2/uc1", ecnr_unitest_s2_1);


    succeed = g_test_run ();


    pr_message("SUCCESS_UNITEST_SUITE_2");

    return succeed;
}

/*******************************************************************************
*
* FUNCTION: ecnr_unitest_suite_3
*
* DESCRIPTION: Alsa test
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean ecnr_unitest_suite_3(void)
{
    int   succeed       = ECNR_ERR_OK;

    pr_message("START_UNITEST_SUITE_3");

    g_test_add_func ("/ecnr_suite_3/uc1", ecnr_unitest_s3_1);


    succeed = g_test_run ();

    /* close module */
    ecnr_service_finalize();
    //ecnr_service_destroy();

    pr_message("SUCCESS_UNITEST_SUITE_3");

    return succeed;
}


/*******************************************************************************
*
* FUNCTION: ecnr_unitest_server_run
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gint ecnr_unitest_server_run(ecnr_unitset unitest_suite_selection)
{
    gint status = ECNR_ERR_OK;

    pr_message("init ecnr unitest\n");

    /* select test suite */
    switch (unitest_suite_selection)
    {
        case ECNR_UNITTEST_SUITE_1:
            status = ecnr_unitest_suite_1();
            break;
        case ECNR_UNITTEST_SUITE_2:
            status = ecnr_unitest_suite_2();
            break;
        case ECNR_UNITTEST_SUITE_3:
            status = ecnr_unitest_suite_3();
            break;
        case ECNR_UNITTEST_SUITE_ALL:
            status = ecnr_unitest_suite_1();
            status = ecnr_unitest_suite_2();
            break;
        case ECNR_UNITEST_SUITE_DEFAUT:
    case ECNR_UNITTEST_SUITE_LAST:
        default:
            pr_message("call default unitest suite\n");
            break;
    }

    return status; /* test succeed*/
}


