/******************************************************************************
 * qwa-service.c
 *----------------------------------------------------------------------------
 * Andy Lowe <alowe@mvista.com>
 * 2010 (c) MontaVista Software, LLC. This file is licensed under the terms of
 * the AFL.
 *****************************************************************************/

/******************************************************************************
 * FILE        : ecnr-service.c
 * PROJECT     : Gen3 & Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 15.08.2013  | Patrick Rey            | initial version for Gen3
 *             |                        | porting from QWA service module.
 *****************************************************************************/

#include <stdlib.h>
#include <locale.h>
#include <unistd.h>
#include <string.h>
#include <signal.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include <glib.h>
#include <glib-unix.h>
#include <glib/gi18n.h>
#include <glib-object.h>


#include "ecnr-error.h"
#include "ecnr-object.h"
#include "ecnr-common-defs.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-configuration.h"
#include "ecnr-engine-handler.h"
#include "ecnr-qwa-handler.h"
#include "ecnr-sse-handler.h"
#include "ecnr-datapool-access.h"
#include "ecnr-ioctrl.h"
#include "ecnr-audio-recording.h"
#include "ecnr-audio-rts-handler.h"


#include "../utest/ecnr-unitest-server.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR_SERVICE
#include "trcGenProj/Header/ecnr-service.c.trc.h"
#endif



#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"


/*******************************************************************************
              Constant defintion
*******************************************************************************/
#define ECNR_SERVICE_NAME "org.bosch.ecnr.service"

#define D_WATCH_RETRY_TIM_OUT_MS 100  //(100 ms)
#define D_WATCH_TIMER_ABORT_STREAMING_LIMIT 100
#define D_ECNR_SERVICE_FIRST_GPIO_CALL_DELAY 1000000 // 1 sec
#define D_ECNR_SERVICE_GPIO_CALL_RETRY_TIME_OUT 200000 // 200_msec
#define D_ECNR_SERVICE_MAX_GPIO_SET_RETRY 100

#define D_ECNR_SERVICE_MIC_LEVEL_NOTIFY_INTERVAL  100  /* in milliseconds */
/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
static struct ecnr_instance_object*     my_obj;
static GObject                          *obj;
GMainLoop                               *ecnr_main_loop;
GMutex                                  ecnr_service_lock;
static guint                            ecnr_service_sig_source_id[3];
static gboolean                         ecnr_service_init_state = FALSE;
static gboolean                         ecnr_service_info_select_flag = FALSE;
static gint                             ecnr_service_test_suite = 0;
static gchar*                           ecnr_service_info_select_param = NULL;
static gboolean                         ecnr_service_tuning_server_state = FALSE;
#ifdef D_TRIGGER_PERFORMANCE_TEST
static guint                            ecnr_service_performance_test_nb = 0;
#endif
static gboolean                         ecnr_service_pass_through_mode_default_state = FALSE;
static gint16                           ecnr_service_mic_gain_level = 0; //default MIC gain level is set 0 dB
static gboolean                         ecnr_service_use_default_dp = FALSE;
static guint                            ecnr_retry_tout_entry_count = 0;
static GThread*                         gpio_thread_hdl = (GThread*)NULL;
static gboolean                         gpio_proc_loop = FALSE;
static gboolean                         ecnr_service_feat_env_ret_fct_ok = FALSE;

static GSource*                         ecnr_mic_lvl_time_source;
static guint32                          ecnr_service_mic_lvl_notify_interval_in_ms = 0;// default timeout interval is set to 0ms; later updated with value from SDS for mic lvl notification

/* apl parameter table*/
static const struct ecnrParameter ecnr_parm_table[] =
{
    /* Property ID 0 is reserved by GLib, so don't use index 0 in the table. */
    {.ecnr_name = "",},
    ECNR_PARM_I32_R_ACCESS_ONLY("ecnr-asr-mode"                    , ecnrAsrMode                  , NULL),
    ECNR_PARM_I32_R_ACCESS_ONLY("ecnr-audio-state"                 , ecnrAudioState               , NULL),
    ECNR_PARM_U8_ARRAY_VAR_SIZE("ecnr-asr-data-type"               , ecnrAsrDataType              , NULL),
    ECNR_PARM_U32_RW_ACCESS("ecnr-mic-lvl-ntfy-interval"           , ecnrMicLvlNtfyInterval       , NULL),
};

#define ECNR_PARAM_TABLE_SIZE ARRAYSIZE(ecnr_parm_table)

static const AudioStateDesc ECNR_AUDIO_STATE_table[] = {

    /* ECNR audio stream specific states*/
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_UNDEFINED),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_SILENCE),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_ERROR),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_INIT),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_WAIT_DATA),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_RUNNING),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_WAIT_RECOVER),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_STOP),
    ECNR_AUDIO_STATE(ECNR_AS_STREAM_TERMINATED),
};


#define ECNR_AUDIO_STATE_MAX ARRAYSIZE(ECNR_AUDIO_STATE_table)


inline static int ECNR_SRV_FCT_RET(int err, const gchar*  fct)
//#define ECNR_SRV_FCT_RET(err,fct)
{
    if(!ecnr_service_feat_env_ret_fct_ok)
        return err;
    else
    {
        if(fct)
        {
            ETG_TRACE_USR3(("[%s]: EXIT, force function return with ECNR_ERR_OK", fct));
        }
        else
        {
            ETG_TRACE_USR3(("[ECNR_SRV_FCT_RET]: EXIT"));
        }
        return ECNR_ERR_OK;
    }
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_cb_info_select
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

static gboolean ecnr_service_cb_info_select
(
    const gchar *option_name,
    const gchar *value,
    gpointer data,
    GError **error
)
{
    ecnr_service_info_select_param = g_strdup (value);
    ecnr_service_info_select_flag = TRUE;

    if (!g_strcmp0(ecnr_service_info_select_param, "datapool"))
    {
    #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        g_print("*****************************************************\n");
        g_print("*             ecnr datapool info                     *\n");
        g_print("*****************************************************\n");
        ecnr_datapool_access_print_datapool_infos();
    #else
        pr_message("datapool not supported\n");
    #endif
    }
    else if (!g_strcmp0(ecnr_service_info_select_param, "ecnr-config"))
    {
        g_print("*****************************************************\n");
        g_print("*             ecnr config infos                     *\n");
        g_print("*****************************************************\n");
        ecnr_configuration_print_user_config();
    }
    else if (!g_strcmp0(ecnr_service_info_select_param, "tuning-trigger"))
    {
        g_print("*****************************************************\n");
        g_print("*             ecnr tuning trigger info              *\n");
        g_print("*****************************************************\n");
        g_print("* Activation of the ecnr tuning interface :\n");
        g_print("* 1 -->    apply command touch %s\n", DEBUG_TIGGER_FILE);
        g_print("* 1 --> or apply command touch %s\n", TUNING_ENABLE_FILE);
        g_print("* 2 --> reset the target\n");
        g_print("*****************************************************\n");
    }
    else
    {
        ecnr_service_info_select_flag = FALSE;
        pr_message("info paramter not available\n");
    }

    (void)option_name;
    (void)data;
    (void)error;
  return TRUE;
}



gpointer ecnr_service_gpio_thread(gpointer data)
{
    gboolean bret = FALSE;
    int retry = 0;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_gpio_thread]: ENTERED with initial delay of %d_us", D_ECNR_SERVICE_FIRST_GPIO_CALL_DELAY));

    /* initial delay previous to the first gpio call */
    usleep(D_ECNR_SERVICE_FIRST_GPIO_CALL_DELAY);

    while(gpio_proc_loop)
    {
        bret = ecnr_ioctrl_set_pin_state(IO_GPIO_SELECT_92, IO_LEVEL_OPEN);

        if(!bret && (retry < D_ECNR_SERVICE_MAX_GPIO_SET_RETRY))
        {
            /* The gpio set action has failed, we will retry later on
               suspend the execution of the thread during "D_ECNR_SERVICE_GPIO_CALL_RETRY_TIME_OUT"
               and retry to set the gpio after the thread has recovered
            */
            usleep(D_ECNR_SERVICE_GPIO_CALL_RETRY_TIME_OUT);
            retry++;
            ETG_TRACE_USR3(("[ecnr_service_gpio_thread]: retry to set the gpio_92 after %d_us -> %d_retry_count ", D_ECNR_SERVICE_GPIO_CALL_RETRY_TIME_OUT, retry));
            pr_message("retry to set the gpio_92 after %d_us -> %d_retry_count \n", D_ECNR_SERVICE_GPIO_CALL_RETRY_TIME_OUT, retry);
        }
        else
        {
            g_mutex_lock(&ecnr_service_lock);
            gpio_proc_loop = FALSE;
            g_mutex_unlock(&ecnr_service_lock);
        }
    }

    if(bret)
    {
        ETG_TRACE_USR3(("[ecnr_service_gpio_thread]: gpio is sucessfully set after %d retries", retry));
        pr_message("EXIT, gpio is sucessfully set after %d retries\n", retry);
    }
    else
    {
        ETG_TRACE_USR3(("[ecnr_service_gpio_thread]: EXIT, fails to set the gpio after %d retries", retry));
        pr_message("EXIT, fails to set the gpio after %d retries\n", retry);
    }
    (void)data;
    return NULL;
}

static void ecnr_service_process_gpio(void)
{
    gboolean started = FALSE;
    GError *error;
    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_process_gpio]: ENTERED"));

    g_mutex_lock(&ecnr_service_lock);
    if(gpio_proc_loop)
    {
        pr_message("the GPIO thread was already started\n");
        ETG_TRACE_USR3(("[ecnr_service_process_gpio]: the GPIO thread was already started"));
        started = TRUE;
    }
    else
        gpio_proc_loop = TRUE;
    g_mutex_unlock(&ecnr_service_lock);

    if(!started)
    {
        gpio_thread_hdl = g_thread_try_new ("ecnr_service_gpio_thread", ecnr_service_gpio_thread, NULL, &error);
        if(!gpio_thread_hdl)
        {
            gpio_proc_loop = FALSE;
            ETG_TRACE_ERR(("[ecnr_service_process_gpio]: Fails to create ecnr_service_gpio_thread "));
        }
    }

    pr_message("EXIT\n");
    return;
}

static void ecnr_service_process_gpio_termination(void)
{
    gboolean ended = FALSE;
    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_process_gpio_termination]: ENTERED"));


    g_mutex_lock(&ecnr_service_lock);
    if(!gpio_proc_loop)
    {
        pr_message("the GPIO thread was already terminated\n");
        ETG_TRACE_USR3(("[ecnr_service_process_gpio]: the GPIO thread was already terminated"));
        ended = TRUE;
    }
    else
        gpio_proc_loop = FALSE;
    g_mutex_unlock(&ecnr_service_lock);

    if(!ended)
    {
        if(gpio_thread_hdl)
        {
            /*(gint)*/ g_thread_join(gpio_thread_hdl);
        }
    }

    gpio_thread_hdl = (GThread*)NULL;
    pr_message("EXIT\n");

    ETG_TRACE_USR3(("[ecnr_service_process_gpio_termination]: EXIT"));
}


static const GOptionEntry ecnr_service_opts[] =
{
    {
        .long_name = "utest",
        .short_name = 'u',
        .flags = 0,
        .arg = G_OPTION_ARG_INT,
        .arg_data = &ecnr_service_test_suite,
        .description = "run ecnr unitest, option:[suite number 1(suite1),2(suite2),..,255(all suites)]",
        .arg_description = "U"
    },
    {
        .long_name = "infos",
        .short_name = 'i',
        .flags = 0,
        .arg = G_OPTION_ARG_CALLBACK,
        .arg_data = &ecnr_service_cb_info_select,
        .description = "retrieve ecnr infos, options:[datapool, ecnr-config,tuning-trigger]",
        .arg_description = "I"
    },
    { NULL }
};


static void ecnr_service_print_help (GOptionContext *context,
             gboolean        main_help,
             GOptionGroup   *group)
 {
   gchar *help;

   help = g_option_context_get_help (context, main_help, group);
   g_print ("%s", help);
   g_free (help);
   OSAL_vProcessExit();
   //exit (0);
 }

static gboolean ecnr_service_option_context_parse_help
(
    GOptionContext   *context,
    gint             *argc,
    gchar          ***argv,
    GError          **error
)
{
    gint i;
    gboolean parsed = FALSE;

    if (argc && argv)
    {
        gboolean stop_parsing = FALSE;

        for (i = 1; i < *argc; i++)
        {
            gchar *arg;

            if ((*argv)[i][0] == '-' && (*argv)[i][1] != '\0' && !stop_parsing)
            {

                if ((*argv)[i][1] == '-')
                {
                    /* -- option */
                    arg = (*argv)[i] + 2;

                    /* '--' terminates list of arguments */
                    if (*arg == 0)
                    {
                        stop_parsing = TRUE;
                        continue;
                    }

                    /* Handle help options */

                    if (strcmp (arg, "help") == 0)
                    {
                        ecnr_service_print_help (context, TRUE, NULL);
                        parsed = TRUE;
                    }
                    else if (strcmp (arg, "help-all") == 0)
                    {
                        ecnr_service_print_help (context, FALSE, NULL);
                        parsed = TRUE;
                    }
                    else
                    {
                        stop_parsing = TRUE;
                        continue;
                    }
                }
                else if ((*argv)[i][1] == 'h')
                {
                    ecnr_service_print_help (context, TRUE, NULL);
                    parsed = TRUE;
                }
                else
                {
                    stop_parsing = TRUE;
                    continue;
                }
            }
        }
    }
    
   (void)error;
   return parsed;
 }




/*******************************************************************************
*
* FUNCTION: ecnr_service_get_param_table_size
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
guint ecnr_service_get_param_table_size(void)
{
    pr_debug("number of configured parameter (%lu)\n", ECNR_PARAM_TABLE_SIZE);
    return (guint)ECNR_PARAM_TABLE_SIZE;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_get_param_table
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/

const struct ecnrParameter* ecnr_service_get_param_table(void)
{
    return (struct ecnrParameter*)ecnr_parm_table;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_get_param_infos
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
const struct ecnrParameter* ecnr_service_get_param_infos(guint16 param_id)
{
    const struct ecnrParameter *p;
    unsigned int param_idx = 0;


    for(param_idx = 0, p = ecnr_parm_table; param_idx < ECNR_PARAM_TABLE_SIZE; param_idx++, p++)
    {
        if(p->ecnr_data_id == param_id)
            return p;
    }

    return (struct ecnrParameter*)NULL;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_finalize
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_service_finalize(void)
{
    gboolean    ret = TRUE;
    int         err = ECNR_ERR_OK;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_service_finalize]: ENTERED"));

    /* stop streaming */
    ret =  ecnr_alsa_stop(ECNR_START_STOP_TIMEOUT);

    /* destroy engine instance */
    if(ret)
    {
        ret = ecnr_alsa_exit_wait(ECNR_EXIT_TIMEOUT);
        if(ret)
        {
            /* leave audio thread*/
            err = ecnr_service_do_action_destroy();

            if(err)
            {
                pr_critical("the ecnr destroyed failed with err %d", err);
                ETG_TRACE_ERR(("[ecnr_service_finalize]: the ecnr destroyed failed with err %d", err));
            }
        }
        else
        {
            /* write a critical log a error memory */
            pr_critical("the termination of the alsa thread has timed out\n");
            ETG_TRACE_ERR(("[ecnr_service_finalize]: the termination of the alsa thread has timed out"));
        }
    }
    else
    {
        /* write a critical log a error memory */
        pr_critical("the alsa thread stop failed with err  %d", err);
        ETG_TRACE_ERR(("[ecnr_service_finalize]: the alsa thread stop failed with err  %d", err));
    }

    /* remove current service instance */
    (void)ecnr_service_instance_current_appid_remove();

    /* remove all ALSA related resources */
    ecnr_alsa_finalize();

    /* terminate GPIO thread if not done automatically */
    ecnr_service_process_gpio_termination();


    pr_message("EXIT");
    ETG_TRACE_USR3(("[ecnr_service_finalize]: EXIT"));

    return TRUE;
}
/*******************************************************************************
*
* FUNCTION: ecnr_service_handle_quit_signal
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_service_handle_quit_signal (void)
{
    pr_message("Received quit signal");

    g_return_val_if_fail (ecnr_main_loop != NULL, FALSE);

    /* end main loop thread */
    g_main_loop_quit(ecnr_main_loop);

    return TRUE;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_install_signal_handler
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void ecnr_service_install_signal_handler (GMainLoop *main_loop)
{
    GSource *source = NULL;
    GMainContext *ctx = g_main_loop_get_context (main_loop);

    pr_message("install the signal cb for SIGTERM, SIGINT and SIGHUP\n");

    source = g_unix_signal_source_new (SIGTERM);
    g_source_set_callback (source,
                          (GSourceFunc)ecnr_service_handle_quit_signal,
                           NULL,
                           NULL);
    ecnr_service_sig_source_id[0] = g_source_attach (source, ctx);

    pr_debug("cb for signal SIGTERM was set");

    GSource *source_1 = NULL;
    source_1 = g_unix_signal_source_new (SIGINT);
    g_source_set_callback (source_1,
                           (GSourceFunc)ecnr_service_handle_quit_signal,
                           NULL,
                           NULL);
    ecnr_service_sig_source_id[1] = g_source_attach (source_1, ctx);
    pr_debug("cb for signal SIGINT was set");


    GSource *source_2 = NULL;
    source_2 = g_unix_signal_source_new (SIGHUP);
    g_source_set_callback (source_2,
                           (GSourceFunc)ecnr_service_handle_quit_signal,
                           NULL,
                           NULL);
    ecnr_service_sig_source_id[2] = g_source_attach (source_2, ctx);
    pr_debug("cb for signal SIGHUP was set");
}

/*******************************************************************************
*
* FUNCTION: main
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

int main(int argc, char *argv[])
{
    DBusGConnection *connection;
    GError          *error;
    DBusGProxy      *driver_proxy;
    guint32          request_name_ret;
    gint             test_succeed = ECNR_ERR_OK;
    GOptionContext  *context;

    pr_message("ecnr_main Entered \n");
    ETG_TRACE_COMP(("[main]: ENTERED"));

    dbus_g_thread_init();


    ecnr_main_loop = g_main_loop_new(NULL, FALSE);
    error = NULL;

    /***********************
      evaluation main keys
    ************************/

    setlocale(LC_ALL, "");
    context = g_option_context_new("");
    g_option_context_set_summary(context, "The ECNR daemon implements "
    "hands-free audio using to QNX/NUANCE libraries\n and ALSA audio "
    "devices.");

    g_option_context_add_main_entries(context, ecnr_service_opts, NULL);
    if(!ecnr_service_option_context_parse_help(context, &argc, &argv, &error))
    {
        if (!g_option_context_parse(context, &argc, &argv, &error))
        {
            g_print("ECNR Daemon: %s\n", error->message);
            g_error_free(error);
            g_option_context_free(context);
            OSAL_vProcessExit();
        }
    }

    g_option_context_free(context);


    /****************************************************************
      check for mode selection
      - execute unit-test if key 'u' is set and value is set to "1"
      - print ECNR information
      - otherwise run ECNR daemon
    *****************************************************************/

    if(ecnr_service_info_select_flag)
    {
        /* print ECNR information and exit immediately */
        ecnr_service_info_select_flag = FALSE;
        OSAL_vProcessExit();
    }

    /* execute unit-test */
    if (ecnr_service_test_suite)
    {
        pr_message("selection unit test ecnr_service_test_suite(%d)\n", ecnr_service_test_suite);

        g_test_init (&argc, &argv, NULL);
        test_succeed = ecnr_unitest_server_run((ecnr_unitset)ecnr_service_test_suite);
        if(test_succeed == 0)
        {
            pr_message("ecnr unit-test terminated successfully \n");
        }
        else
        {
            pr_message("ecnr unit-test failed \n");
        }

        OSAL_vProcessExit(); /* unit-test terminated */
    }

    /* run ECNR Daemon main program */
    connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (connection == NULL) {
        pr_critical("failed to open connection to bus: %s",error->message);
        g_error_free(error);
        OSAL_vProcessExit();
    }

    pr_debug("dbus_g_bus_get\n");

    obj = g_object_new(ECNR_TYPE_OBJECT, NULL);

    pr_debug("g_object_new\n");

    dbus_g_connection_register_g_object(connection, "/", obj);

    pr_debug("dbus_g_connection_register_g_object \n");

    driver_proxy = dbus_g_proxy_new_for_name(connection,
             DBUS_SERVICE_DBUS,
             DBUS_PATH_DBUS,
             DBUS_INTERFACE_DBUS);

    pr_debug("dbus_g_proxy_new_for_name \n");

    if (!org_freedesktop_DBus_request_name(driver_proxy,
                 ECNR_SERVICE_NAME,
                 0, &request_name_ret, &error))
    {
        g_assert(error != NULL);
        if (error)
            pr_critical("failed to get name: %s", error->message);
        g_error_free(error);
        OSAL_vProcessExit();
    }

    pr_debug("dbus_g_proxy_new_for_name \n");

    if (!(request_name_ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER))
    {
        pr_critical("Got result code %u from requesting name", request_name_ret);
        OSAL_vProcessExit();
    }

    /******************************************
    initialize ALSA module - audio streaming
    ******************************************/
    ecnr_alsa_initialize();

    pr_message("start ecnr after power cycle");

#ifdef D_SET_TUNING_PER_TRIGGER_FILE
    /* start debug interface handling to sse per trigger file  */
    if(ecnr_sse_debug_check_enable_file())
    {
        int err = ECNR_ERR_OK;
        err = ecnr_engine_start_debug_session();
        if(!err)
            ecnr_service_tuning_server_state = TRUE;
    }
#endif

    /* install callback for signal handler */
    ecnr_service_install_signal_handler(ecnr_main_loop);

   /* create a mainloop that runs/iterates the default GLib main context
    * (context NULL). when a message has been posted on the
    * bus or by systemd, the default main context will automatically dispatch the
    * associated resource and call the callback associated with the resource
    * The main loop remains in run state until g_main_loop_quit() is called
   */
    g_main_loop_run(ecnr_main_loop);

    pr_message("EXIT main loop\n");

    /* finalize ECNR */
    ecnr_service_finalize();
    ecnr_alsa_delete();

    pr_message("service was terminated\n");

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
    ecnr_datapool_access_finalize();
#endif

    if(ecnr_service_tuning_server_state)
    {
        ecnr_engine_end_debug_session();
    }

    if (ecnr_main_loop)
        g_main_loop_unref (ecnr_main_loop);

    OSAL_vProcessExit();
}



/*******************************************************************************
+ instance handler static functions
*******************************************************************************/

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_init
 *
 * DESCRIPTION:
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
void ecnr_service_init(void)
{
    pr_message("entered");
    ETG_TRACE_USR3(("[ecnr_service_init]: ENTERED"));

    g_mutex_init(&ecnr_service_lock);

    ecnr_service_init_state = TRUE;
    ecnr_service_set_pass_through_mode_default_state(FALSE);
    ecnr_service_set_mic_gain_level(0);

#if (defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2))
    /* set default state of HF_VR_MODE to open(Handsfree)  */
    ecnr_service_process_gpio();
    //ecnr_ioctrl_set_pin_state(IO_GPIO_SELECT_92, IO_LEVEL_OPEN);
#else
    /* to suppress unused function warning */
    (void)&ecnr_service_process_gpio;
#endif

    ecnr_service_feat_env_ret_fct_ok = FALSE;

    /* Read feature setup from service file as environment variable  */
    const char* secnr_srv_ret_fct_OK = getenv("ECNR_SRV_FCT_RETURN_ALWAYS_OK");

    if(secnr_srv_ret_fct_OK)
        pr_message("FEAT ECNR_SRV_FCT_RETURN_ALWAYS_OK = %s\n", secnr_srv_ret_fct_OK);

    if (!g_strcmp0(secnr_srv_ret_fct_OK, "TRUE"))  // record read stream only
    {
        ecnr_service_feat_env_ret_fct_ok = TRUE;
    }


    ecnr_engine_handler_performance_enable(FALSE);
    ecnr_audio_rts_handler_performance_enable(FALSE);

    /* performance test */
    const char* secnr_srv_performance_all = getenv("ECNR_SRV_PERFORMANCE_ALL_AVAIL");

    if(secnr_srv_performance_all)
        pr_message("FEAT ECNR_SRV_PERFORMANCE_ALL_AVAIL = %s\n", secnr_srv_performance_all);

    if (!g_strcmp0(secnr_srv_performance_all, "TRUE"))  // record read stream only
    {
        ecnr_engine_handler_performance_enable(TRUE);
        ecnr_audio_rts_handler_performance_enable(TRUE);
    }

    return;
}


/*******************************************************************************
 *
 * FUNCTION: ecnr_service_destroy
 *
 * DESCRIPTION:
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
void ecnr_service_destroy(void)
{
    pr_message("entered");
    ETG_TRACE_USR3(("[ecnr_service_destroy]: ENTERED"));

    g_mutex_clear(&ecnr_service_lock);

    ecnr_service_init_state = FALSE;

    ecnr_service_set_pass_through_mode_default_state(FALSE);
    ecnr_service_set_mic_gain_level(0);
    return;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_set_passthroughmode_default_state
 *
 * DESCRIPTION: This function set the default state for ecnr pass through mode
 *
 * PARAMETER: default state
 *
 * RETURNVALUE:
 *
 *******************************************************************************/
void ecnr_service_set_pass_through_mode_default_state(gboolean ptm_default_state)
{
  pr_message("entered");
  ecnr_service_pass_through_mode_default_state = ptm_default_state;
  pr_message("ecnr_service_pass_through_mode_default_state -> (%d)", ecnr_service_pass_through_mode_default_state);
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_get_passthroughmode_default_state
 *
 * DESCRIPTION: This function get's the default state for ecnr pass through mode
 *
 * PARAMETER:
 *
 * RETURNVALUE: default state
 *
 *******************************************************************************/
gboolean  ecnr_service_get_pass_through_mode_default_state(void)
{
  pr_message("entered");
  pr_message("ecnr_service_pass_through_mode_default_state -> (%d)", ecnr_service_pass_through_mode_default_state);
  return  ecnr_service_pass_through_mode_default_state;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_set_mic_gain_level
 *
 * DESCRIPTION: This function set the mic gain level
 *
 * PARAMETER: default state
 *
 * RETURNVALUE:
 *
 *******************************************************************************/
void ecnr_service_set_mic_gain_level(gint16 mic_gain_level)
{
    pr_message("entered");
    ecnr_service_mic_gain_level = mic_gain_level;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_service_get_mic_gain_level
 *
 * DESCRIPTION: This function get's the current mic gain level set
 *
 * PARAMETER:
 *
 * RETURNVALUE: default state
 *
 *******************************************************************************/
gint16  ecnr_service_get_mic_gain_level(void)
{
    pr_message("entered");
    return  ecnr_service_mic_gain_level;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_instance_create_new
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_instance_create_new(guchar app_id)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_service_instance_create_new]: ENTERED with Appid[%d]", app_id));

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(&ecnr_service_lock);

    if(my_obj)
    {
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
        goto error_out;
    }

    if(!app_id)
    {
        err = ECNR_ERR_APPID_NOT_AVAILABLE;
        goto error_out;
    }

    /* create new ecnr instamce */
    my_obj = g_new(struct ecnr_instance_object, 1);

    if(my_obj)
    {
        my_obj->ecnr_is_initialized = FALSE;
        my_obj->ecnr_Debug_intf_initialized = FALSE;
        my_obj->ecnr_current_config_id = 0;
        my_obj->ecnr_service_terminate_req = FALSE;
        my_obj->ecnr_current_app_id = app_id;
        my_obj->ecnr_alsa_thread_ref = 0;
        my_obj->ecnr_service_audio_state = (ecnr_audio_state)ECNR_AS_STREAM_TERMINATED; //ECNR_AS_STREAM_SILENCE;


        pr_message("create instance for application(%d)\n",app_id);
        ETG_TRACE_USR3(("[ecnr_service_instance_create_new]: new service inst for Appid[%d]", app_id));
    }
    else
        err = ECNR_ERR_NULL_POINTER;

error_out:
    g_mutex_unlock(&ecnr_service_lock);
    pr_message("EXIT with error(%d)\n", err );
    ETG_TRACE_USR3(("[ecnr_service_instance_create_new]: EXIT with ErrCode[%s]", ecnr_error_str(err)));
    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_audio_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_service_audio_create(void)
{
    int err = ECNR_ERR_OK;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_audio_create]: ENTERED"));

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */

        pr_warning("service module not initialized\n");
        ETG_TRACE_USR3(("[ecnr_service_audio_create]: service module not initialized"));
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(&ecnr_service_lock);
    if(!my_obj)
    {
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
        pr_debug("no service instance available\n");
    }
    g_mutex_unlock(&ecnr_service_lock);

    if(!err)
    {
        if(ECNR_AS_STREAM_TERMINATED != ecnr_service_audio_get_state())
        {
            pr_message("audio thread already in progress\n");
            ETG_TRACE_USR3(("[ecnr_service_audio_create]: audio thread already in progress"));
            err = ECNR_ERR_AUDIO_THREAD_ALREADY_ACTIVE;
        }
    }

    if(!err)
    {
        err = ecnr_alsa_audio_thread_new();
        if (err)
        {
            pr_warning("fails to create the audio thread\n");
            ETG_TRACE_USR3(("[ecnr_service_audio_create]: fails to create the audio thread"));
            err = ECNR_ERR_NOT_CREATED;
        }
        else
        {
            ecnr_service_audio_ref();

            pr_message("audio thread was started\n");
            ETG_TRACE_USR3(("[ecnr_service_audio_create]: audio thread was started "));
        }
    }

    return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_audio_terminate_cb
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_audio_terminate_cb(void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED\n");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        pr_message("EXIT with error code -> %d\n", ECNR_ERR_SERV_NOT_INITIALIZED);
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(&ecnr_service_lock);

    if(!my_obj)
    {
        pr_debug("no service instance available\n");
        err = ECNR_ERR_APPID_INSTANCE_ALREADY_ACTIVE;
    }

    if(!err)
    {
        my_obj->ecnr_service_audio_state        = (ecnr_audio_state)ECNR_AS_STREAM_TERMINATED; // ECNR_AS_STREAM_SILENCE;
        pr_debug("change state service audio thread to deactivated\n");
    }

    g_mutex_unlock(&ecnr_service_lock);

    pr_message("EXIT with error code -> %d\n", err);

    return err ;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_instance_remove
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_instance_remove(guchar app_id)
{
    pr_message("ENTERED");
    ETG_TRACE_COMP(("[ecnr_service_instance_remove]: ENTERED"));


    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        ETG_TRACE_ERR(("[ecnr_service_instance_remove]: EXIT with ErrCode[%s]", ecnr_error_str(ECNR_ERR_SERV_NOT_INITIALIZED)));
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        ETG_TRACE_ERR(("[ecnr_service_instance_remove]: EXIT with ErrCode[%s]", ecnr_error_str(ECNR_ERR_NO_INSTANCE_AVAILABLE)));
        return ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    if(app_id && my_obj->ecnr_current_app_id != app_id )
    {
        ETG_TRACE_ERR(("[ecnr_service_instance_remove]: no instance available for the AppId[%d]", app_id));
        pr_warning("no instance available for this appid(%d)\n", app_id);
        return ECNR_ERR_OK;
    }

    g_mutex_lock(&ecnr_service_lock);

    /* free ecnr instamce */
    if(my_obj)
        g_free(my_obj);
    my_obj = NULL;

    g_mutex_unlock(&ecnr_service_lock);

    ETG_TRACE_COMP(("[ecnr_service_instance_remove]: EXIT, service instance of appid -> %d is removed", app_id));
    pr_message("EXIT, service instance of appid -> %d is removed\n",app_id);

    return ECNR_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_instance_current_appid_remove
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_instance_current_appid_remove(void)
{
    pr_message("ENTERED");
    ETG_TRACE_COMP(("[ecnr_service_instance_current_appid_remove]: ENTERED"));


    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        ETG_TRACE_ERR(("[ecnr_service_instance_current_appid_remove]: EXIT with ErrCode[%s]", ecnr_error_str(ECNR_ERR_SERV_NOT_INITIALIZED)));
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        ETG_TRACE_ERR(("[ecnr_service_instance_current_appid_remove]: EXIT with ErrCode[%s]", ecnr_error_str(ECNR_ERR_NO_INSTANCE_AVAILABLE)));
        return ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    g_mutex_lock(&ecnr_service_lock);

    ETG_TRACE_COMP(("[ecnr_service_instance_current_appid_remove]: current  active AppId -> %d", my_obj->ecnr_current_app_id));
    pr_message("current  active AppId-> %d\n", my_obj->ecnr_current_app_id);

    /* free ecnr instamce */
    if(my_obj)
        g_free(my_obj);
    my_obj = NULL;

    g_mutex_unlock(&ecnr_service_lock);

    ETG_TRACE_COMP(("[ecnr_service_instance_current_appid_remove]: EXIT"));
    pr_message("EXIT\n");

    return ECNR_ERR_OK;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_instance_get
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct ecnr_instance_object* ecnr_service_instance_get(guchar app_id)
{
    pr_debug("ENTERED");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        pr_debug("service not initialized\n");
        return (struct ecnr_instance_object*)NULL;;
    }

    if(!my_obj)
    {
        pr_debug("no instance available\n");
        return (struct ecnr_instance_object*)NULL;
    }

    if((my_obj->ecnr_current_app_id != app_id) && (app_id != ECNR_APPID_ADMIN))
    {
        pr_warning("no instance available for the requested appid(%d)\n",app_id);
        return (struct ecnr_instance_object*)NULL; //return ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE;
    }

    return my_obj;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_instance_active_get
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct ecnr_instance_object* ecnr_service_instance_active_get(void)
{
    pr_debug("ENTERED");

    return my_obj;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_instance_active_is
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_instance_active_is(guchar app_id, gboolean* state)
{
    pr_debug("ENTERED");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        pr_debug("no instance available\n");
        *state = FALSE;
        return ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    if((my_obj->ecnr_current_app_id != app_id) && (app_id != ECNR_APPID_ADMIN))
    {
        pr_debug("no instance available for the requested appid(%d)\n",app_id);
        *state = FALSE;
        return ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE;
    }

    *state = TRUE;
    return ECNR_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_do_action_initialize
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_do_action_initialize(gint config_id)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED \n");
    ETG_TRACE_USR3(("[ecnr_service_do_action_initialize]: ENTERED"));

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    /*new*/
    g_mutex_lock(&ecnr_service_lock);
    if(!my_obj)
        err = ECNR_ERR_NO_INSTANCE_AVAILABLE;
    else if(my_obj->ecnr_is_initialized)
        err = ECNR_ERR_MODULE_ALREADY_INIT;
    else
    {
        pr_message("proceed to the initialization of the config_id -> %d\n", config_id);
        ETG_TRACE_USR3(("[ecnr_service_do_action_initialize]: proceed to the initialization of the config_id -> %d", config_id));
    }
    g_mutex_unlock(&ecnr_service_lock);

    if (!err)
    {
        /*re-initialize some ALSA specific variables */
        ecnr_alsa_reinitialize();

        /* reset the ECNR engine with a
         * default configuration file. If a configuration Id was given by the application
         * then this one it taken instead of.
             */


        err = ecnr_engine_handler_create_session((unsigned short)config_id);

        /* add support ASR */
        if(!err)
        {
            err = ecnr_alsa_notify_asr_mode_state();
            if(err)
            {
                (void)ecnr_engine_handler_destroy();
            }
            else
            {
                g_mutex_lock(&ecnr_service_lock);
                my_obj->ecnr_is_initialized = TRUE;
                my_obj->ecnr_current_config_id = config_id;
                g_mutex_unlock(&ecnr_service_lock);

                pr_message("successful initialization\n");
                ETG_TRACE_USR3(("[ecnr_service_do_action_initialize]: successful initialization "));
            }
        }
    }

    ETG_TRACE_USR3(("[ecnr_service_do_action_initialize]: EXIT with ErrCode[%s]", ecnr_error_str(err)));

    return err;
}


/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_frame_info
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_check_AppId_ConfigId_Matching(gint ecnrConfigurationId, guchar ecnrAppId)//, guchar CurrAppId )
{
    int ret = ECNR_ERR_OK;

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    /* check validity AppId*/
    ret = ecnr_configuration_check_AppId(ecnrAppId);

    /* check validity configuration Id*/
    if(!ret)
        ret = ecnr_configuration_check_ConfigId((short int)ecnrConfigurationId);

    if(!ret)
        ret = ecnr_configuration_check_match_AppId_and_configId((unsigned short int)ecnrConfigurationId,ecnrAppId);

    return ret;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_frame_info
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_do_action_destroy(void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[ecnr_service_do_action_destroy]: ENTERED"));

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    if(!my_obj)
    {
        pr_debug("no instance available\n");
        return err;
    }

    /*new*/
    g_mutex_lock(&ecnr_service_lock);

    if (my_obj->ecnr_is_initialized)
    {
        /* reset the ECNR engine with a
         * default configuration file. If a configuration Id was given by the application
         * then this one it taken instead of.
             */
        my_obj->ecnr_service_terminate_req = TRUE;
        (void)ecnr_engine_handler_destroy();

#if (defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2))
    /*****
     * - set default state of HF_VR_MODE to open(Handsfree) always prior to close ioctrl instance
     * - ecnr_ioctrl_finalize is call every time a ECNR instance is destroyed,
     *   so with the call of ecnr_destroy()
     */
    ecnr_ioctrl_set_pin_state(IO_GPIO_SELECT_92, IO_LEVEL_OPEN);
#endif

        /* close IO device and associated member variable */
        ecnr_ioctrl_finalize();

        /* reinitialize ALSA instance member */
        /*fix me*/
        //ecnr_alsa_reinitialize();
    }
//    else
//        err = ECNR_ERR_MODULE_NOT_INIT;


    my_obj->ecnr_is_initialized = FALSE;
    my_obj->ecnr_current_config_id = 0;
    my_obj->ecnr_service_terminate_req =FALSE;

    g_mutex_unlock(&ecnr_service_lock);

    pr_message("EXIT");
    ETG_TRACE_USR3(("[ecnr_service_do_action_destroy]: EXIT"));

    return err;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_audio_ref
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_audio_ref(void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(&ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    /* increment alsa thread reference */
    if(!err)
    {
        my_obj->ecnr_alsa_thread_ref = (guint8)(my_obj->ecnr_alsa_thread_ref + 1);
        pr_message("current ref count -> %d\n", my_obj->ecnr_alsa_thread_ref);
    }

    g_mutex_unlock(&ecnr_service_lock);

    return err;

}

/*******************************************************************************
*
* FUNCTION: ecnr_service_set_use_default_dp
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/

void  ecnr_service_set_use_default_dp(void)
{

    pr_message("ENTERED");

    g_mutex_lock(&ecnr_service_lock);
    ecnr_service_use_default_dp = TRUE;
    g_mutex_unlock(&ecnr_service_lock);

    return;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_set_use_default_dp_exit
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/

void  ecnr_service_set_use_default_dp_exit(void)
{

    pr_message("ENTERED");

    g_mutex_lock(&ecnr_service_lock);
    ecnr_service_use_default_dp = FALSE;
    g_mutex_unlock(&ecnr_service_lock);

    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_is_use_default_dp
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/

gboolean ecnr_service_is_use_default_dp(void)
{
    gboolean state = FALSE;

    state = ecnr_service_use_default_dp;

    pr_message("%s\n", state? "use default data set":" do not use default data set");

    return state;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_audio_unref
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_audio_unref(void)
{
    int err = ECNR_ERR_OK;

    pr_message("ENTERED");

    if(!ecnr_service_init_state)
    {
        /* the ecnr_service_init should be first call prior
           to any ecnr_service module function call
        */
        return ECNR_ERR_SERV_NOT_INITIALIZED;
    }

    g_mutex_lock(&ecnr_service_lock);

    if(!my_obj)
    {
        err = ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }

    /* increment alsa thread reference */
    if(!err)
    {
        my_obj->ecnr_alsa_thread_ref = 0;
        pr_message("remove alsa thread reference\n");
    }

    g_mutex_unlock(&ecnr_service_lock);

    pr_message("EXIT");

    return err;

}

int ecnr_service_audio_get_ref(void)
{

    int ref = 0;
    pr_message("ENTERED");

    g_mutex_lock(&ecnr_service_lock);

    if(my_obj)
    {
        pr_message("current thread ref ->%d\n", my_obj->ecnr_alsa_thread_ref);
        ref = my_obj->ecnr_alsa_thread_ref ;

    }
    else
    {
        pr_message("no ref available\n");
    }

    g_mutex_unlock(&ecnr_service_lock);


    return ref;

}

#ifdef D_TRIGGER_PERFORMANCE_TEST
/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_initialize
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_performance_initialize(void)
{

    pr_message("ENTERED");
    system("touch  /var/opt/bosch/dynamic/audiomanager/ecnr/EcnrPerformance.txt");
    ecnr_service_performance_test_nb++;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_start
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_performance_start(void)
{

    pr_message("ENTERED");

    GString *strbuf = g_string_new(NULL); /* This will spawn an empty string */
    g_string_append(strbuf, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc.sh");
    g_string_append(strbuf, " PERFORMANCE_TEST");
    g_string_append_printf(strbuf, " %d &", ecnr_service_performance_test_nb);

    pr_message("call system(%s)", strbuf->str);

    /*
     * Completely scratching the buffer, including its contents:
     */
    system(strbuf->str);
    g_string_free(strbuf, TRUE);
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_test_performance_stop
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_performance_stop(void)
{
    pr_message("ENTERED");
    system("rm  /var/opt/bosch/dynamic/audiomanager/ecnr/EcnrPerformance.txt");
    pr_message("EXIT");
}

#endif

#ifdef D_SIMULATE_SPI_CLIENT

/*******************************************************************************
*
* FUNCTION: ecnr_service_test_client_start
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_client_start(void)
{
    pr_message("ENTERED");

    GString *alsa_client_spi_in_out = g_string_new(NULL); /* This will spawn an empty string */

    g_string_append(alsa_client_spi_in_out, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc.sh");
    g_string_append(alsa_client_spi_in_out, " ALSA_CLIENT_SIM START");

    pr_message("call system(%s)", alsa_client_spi_in_out->str);
    system(alsa_client_spi_in_out->str);

    g_string_free(alsa_client_spi_in_out, TRUE);

    pr_message("EXIT");

}


/*******************************************************************************
*
* FUNCTION: ecnr_service_test_client_stop
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: void
*
*******************************************************************************/
void ecnr_service_test_client_stop(void)
{
    pr_message("ENTERED");

    GString *alsa_client_spi_in_out = g_string_new(NULL); /* This will spawn an empty string */

    g_string_append(alsa_client_spi_in_out, "/opt/bosch/audiomanager/test/ecnr-test-script/ecnr_stress_test_proc.sh");
    g_string_append(alsa_client_spi_in_out, " ALSA_CLIENT_SIM STOP");

    pr_message("call system(%s)", alsa_client_spi_in_out->str);
    system(alsa_client_spi_in_out->str);

    pr_message("EXIT");
}
#endif

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_get_engine_param
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_get_engine_param
(
    guint       param_id,
    GArray**    data
)
{
    int                         err = (int)ECNR_ERR_OK;
    const struct ecnrParameter*  p;
    GArray*                     ga;

    pr_message("ENTERED: param_id(%d)\n", param_id);


    /* parameter infos*/
    p = ecnr_service_get_param_infos((guint16)param_id);

    if(!p)
    {
        pr_debug("parameter Id(%d) not available\n", param_id);
        return ECNR_ERR_PARAM;
    }

    switch (param_id)
    {
        case ecnrAsrDataType:
        {
            stecnr_alsa_state* state = ecnr_alsa_get_instance();
            if(!state)
            {
                pr_debug("no alsa module instance available\n");
                err = ECNR_ERR_SERV_NOT_INITIALIZED;
                break;
            }

            if(!state->ecnr_asr_data_type)
            {
                pr_warning("no asr_data_type available\n");
                err = ECNR_ERR_INVALID_ASR_MODE;
                break;
            }

            if(!state->ecnr_asr_data_type_size)
            {
                pr_warning("size of the asr_data_type structure is NULL\n");
                err = ECNR_ERR_INVALID_ASR_MODE;
                break;
            }
#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)
            #if defined(SDS_DATASET_2)||defined(SDS_DATASET_4)
            err = ( ecnr_configuration_check_ConfigId(SDS_DATASET_2) & ecnr_configuration_check_ConfigId(SDS_DATASET_4) );
            if(!err)
            {
                if( (SDS_DATASET_2 != my_obj->ecnr_current_config_id) && (SDS_DATASET_4 != my_obj->ecnr_current_config_id) )
                {
                    err = ECNR_ERR_SERV_ASR_TYPE_UNSUPPORTED;
                    pr_warning("parameter(ecnrAsrDataType) only available with config_id (402) and (404) \n");
                }
            }
            #endif
#else
            #ifdef SDS_DATASET_2
            err = ecnr_configuration_check_ConfigId(SDS_DATASET_2);
            if(!err)
            {
                if(SDS_DATASET_2 != my_obj->ecnr_current_config_id)
                {
                   err = ECNR_ERR_SERV_ASR_TYPE_UNSUPPORTED;
                   pr_warning("parameter(ecnrAsrDataType) only available with config_id(402)\n");
                }
            }
            #endif
#endif

            ga = g_array_new(TRUE, TRUE, sizeof(gchar));

            /* append data structure */
            g_array_append_vals(ga, (gpointer)state->ecnr_asr_data_type, state->ecnr_asr_data_type_size);

            /* return Garray to caller  */
            *data = ga;

            break;
        }
        case ecnrAsrMode:
        case ecnrAudioState:
        default:
            pr_warning("no propoerty with this data type\n");
            break;

    }


    return err;

}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_get_param
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_get_param
(
    guint16             param_id,
    ecnr_type_defs      data_type,
    GValue*             param
)
{
    int                         err = (int)ECNR_ERR_OK;
    const struct ecnrParameter*  p;
    GArray*                     ga;
    guint                       i = 0;

    pr_message("ENTERED: paramId(%d), datatype(%d)\n", param_id, data_type);


    /* parameter infos*/
    p = ecnr_service_get_param_infos(param_id);

    if(!p)
    {
        pr_debug("parameter Id(%d) not available\n", param_id);
        return ECNR_ERR_PARAM;
    }

    ETG_TRACE_USR3(("[ecnr_service_fi_get_param]: requested property[%s]", (p->ecnr_name ? p->ecnr_name : "UNDEF")));

    switch (data_type)
    {
        case ECNR_TYPE_I16:
        case ECNR_TYPE_U16:
        case ECNR_TYPE_U32:
            pr_warning("no propoerty with this data type\n");
        break;
        case ECNR_TYPE_ARRAY_U8:
        case ECNR_TYPE_ARRAY_I8:
        {
            stecnr_alsa_state* state = ecnr_alsa_get_instance();
            if(!state)
            {
                pr_debug("no alsa module instance available\n");
                break;
            }

            if(!state->ecnr_asr_data_type)
            {
                pr_warning("no asr_data_type available\n");
                break;
            }

            if(!state->ecnr_asr_data_type_size)
            {
                pr_warning("size of the asr_data_type structure is NULL\n");
                break;
            }

            ga = g_array_new(FALSE, TRUE, sizeof(gchar));//state->ecnr_asr_data_type_size);

            /* append data structure */
            g_array_append_vals(ga, (gpointer)state->ecnr_asr_data_type, state->ecnr_asr_data_type_size);

            /* copy array to dbus output parameter  */
            g_value_set_boxed(param, ga);

            /* for trace */
            guchar *data_read = &g_array_index(ga, guchar, 0);
            for (i = 0; i < ga->len; i++)
            {
                pr_debug("array[%d]:%d\n", i, *data_read++);
                ETG_TRACE_USR4(("[ecnr_service_fi_get_param]: elem[%d]:%d",  i, *data_read++));
            }

            g_array_free(ga, TRUE);

            break;
        }
        case ECNR_TYPE_ARRAY_U16:
        case ECNR_TYPE_ARRAY_I16:
        case ECNR_TYPE_ARRAY_U32:
        case ECNR_TYPE_ARRAY_I32:
            pr_warning("no property with this data type\n");
            break;
        case ECNR_TYPE_I32:
        {
            gint32 p32=0;
            if (param_id == ecnrAsrMode)
                p32 = (gint32)ecnr_alsa_asr_mode_state();
            else if(param_id == ecnrAudioState )
            {
                if(my_obj)
                    p32 = (gint32)my_obj->ecnr_service_audio_state;
                else
                    p32 = -1;
            }
            else
            {
                /* do nothing */
            }

            pr_debug("%s = %d", p->ecnr_name, p32);

            g_value_set_int(param, p32);

            ETG_TRACE_USR4(("[ecnr_service_fi_get_param]: applied u32value[%d]", p32));

            break;
        }
        default:
            break;

    }

    ETG_TRACE_USR3(("[ecnr_service_fi_get_param]: EXIT with ErrCode[%s]", ecnr_error_str(err)));

    return err;
}






static gboolean ecnr_service_retry_time_out_analyse(gpointer   data)
{


    if(!data)
        return FALSE;

    stecnr_alsa_state*   myalsa = (stecnr_alsa_state*)data;

    if(++ecnr_retry_tout_entry_count > D_WATCH_TIMER_ABORT_STREAMING_LIMIT)
    {
        pr_critical("alsa streaming is terminated now after %d_sec expriration of retry awatch timer\n", D_WATCH_TIMER_ABORT_STREAMING_LIMIT);
        ETG_TRACE_ERR(("[ecnr_service_retry_time_out_analyse]: the alsa streaming is terminated, the retry timer has expired after %d_sec\n", D_WATCH_TIMER_ABORT_STREAMING_LIMIT));

        /* terminate alsa thread */
        ecnr_service_finalize();
        return FALSE;
    }

    if(!my_obj)
        return FALSE;

    if(   (my_obj->ecnr_service_audio_state == ECNR_AS_STREAM_RUNNING)
        ||!myalsa->retry)
    {
        ETG_TRACE_USR3(("[ecnr_service_retry_time_out_analyse]: streaming was successfully recovered"));
        pr_message("streaming was successfully recovered\n");
        return FALSE;
    }

    else if(   (my_obj->ecnr_service_audio_state == ECNR_AS_STREAM_ERROR)
            || (my_obj->ecnr_service_audio_state == ECNR_AS_STREAM_SILENCE)
            || (my_obj->ecnr_service_audio_state == ECNR_AS_STREAM_TERMINATED))
    {
        pr_message("watch timer closed with alsa streaming thread termination\n");
        return FALSE;
    }

    pr_message("retry watch timeout is retriggered, total retry time expired is : %d_ms", ecnr_retry_tout_entry_count * D_WATCH_RETRY_TIM_OUT_MS);
    ETG_TRACE_USR3(("[ecnr_service_retry_time_out_analyse]: retry watch timeout is retriggered, total retry time expired is : %d_ms", ecnr_retry_tout_entry_count * D_WATCH_RETRY_TIM_OUT_MS));
    return TRUE;
}
/************************************************************************
*
* FUNCTION: ecnr_service_start_retry_watch_timer
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static void ecnr_service_start_retry_watch_timer(void)
{
    GSource *retry_tout_source;
    stecnr_alsa_state*   data = ecnr_alsa_get_instance();

    if(!data)
        return;

    pr_message("ENTERED: retry time_out: %d_ms\n", D_WATCH_RETRY_TIM_OUT_MS);
    ETG_TRACE_USR3(("[ecnr_service_start_retry_watch_timer]: retry time_out: %d_ms",D_WATCH_RETRY_TIM_OUT_MS));

    ecnr_retry_tout_entry_count = 0;


    retry_tout_source = g_timeout_source_new ((guint)D_WATCH_RETRY_TIM_OUT_MS);
    g_source_set_priority (retry_tout_source, G_PRIORITY_DEFAULT);
    g_source_set_callback (retry_tout_source, (GSourceFunc)ecnr_service_retry_time_out_analyse, (gpointer) data, NULL);

    GMainContext* context = g_main_loop_get_context(ecnr_main_loop);
    g_source_attach(retry_tout_source, context);
    g_source_unref(retry_tout_source);


    return;
}



/*******************************************************************************
*
* FUNCTION: ecnr_error_str
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
const char *ecnr_audio_state_str(ecnr_audio_state code)
{
  guint i;
  const AudioStateDesc *as;
  static const char ecnr_unknown_audio_state[] = "Unknown ECNR audio state";

  for (i = 0, as = ECNR_AUDIO_STATE_table; i < ECNR_AUDIO_STATE_MAX; i++, as++) {
    if (as->code == code)
      return as->str;
  }
  return ecnr_unknown_audio_state;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_audio_change_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_audio_change_state
(
  ecnr_audio_state AudioState
)
{
    int  err = (int)ECNR_ERR_OK;
    gboolean doNewChange = TRUE;


    g_mutex_lock(&ecnr_service_lock);
    if(!my_obj)
    {
        pr_critical("ERROR: %s", ecnr_error_str(ECNR_ERR_SERV_NOT_INITIALIZED));
        ETG_TRACE_ERR(("[ecnr_service_audio_change_state]: ERROR: %s", ecnr_error_str(ECNR_ERR_SERV_NOT_INITIALIZED)));
        //return ECNR_ERR_SERV_NOT_INITIALIZED;
        err = ECNR_ERR_SERV_NOT_INITIALIZED;
        doNewChange = FALSE;
    }
    else if(my_obj->ecnr_service_audio_state == AudioState)
    {
        //pr_message("state %s is currently active\n", ecnr_audio_state_str(AudioState));
        //ETG_TRACE_ERR(("[ecnr_service_audio_change_state]: state %s is currently active", ecnr_audio_state_str(AudioState)));
        doNewChange = FALSE;
    }
    else
    {
        /*do nothing*/
        doNewChange = TRUE;
    }
    g_mutex_unlock(&ecnr_service_lock);

    if(doNewChange) //my_obj && (my_obj->ecnr_service_audio_state != AudioState))
    {
        ETG_TRACE_USR1(("[ecnr_service_audio_change_state]: prev state -> %s",ecnr_audio_state_str(my_obj->ecnr_service_audio_state)));

        /* try to change to new state */
        switch(AudioState)
        {
            case ECNR_AS_STREAM_SILENCE:
            {
#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)
          #if defined(SDS_DATASET_2)||defined(SDS_DATASET_4)
                if((SDS_DATASET_2 == my_obj->ecnr_current_config_id)||(SDS_DATASET_4 == my_obj->ecnr_current_config_id))
                {
                    /* Stream silence frames only in Silence state for Gen4 SDS and WuW to resolve NO_FRAME_SYNC error*/
                    ecnr_alsa_voice_stream_change_state(AudioState);
                }
          #endif
#endif

                break;
            }
            case ECNR_AS_STREAM_WAIT_DATA:
            case ECNR_AS_STREAM_WAIT_RECOVER:
            {
                /* start watch timer */
#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)
          #if defined(SDS_DATASET_2)||defined(SDS_DATASET_4)
                if((SDS_DATASET_2 != my_obj->ecnr_current_config_id)&&(SDS_DATASET_4 != my_obj->ecnr_current_config_id))
                {
                ecnr_service_start_retry_watch_timer();
                }
                else
                {
                    /* Enable silence write for gen4 sds usecase to resolve NO_FRAME_SYNC error*/
                    ecnr_alsa_voice_stream_change_state(AudioState);
                }
          #endif /* SDS_DATASET_2 */
#else
                ecnr_service_start_retry_watch_timer();
#endif /* VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2 */
                break;
            }
            case ECNR_AS_STREAM_INIT:
            {
                /* setup thread priority to real time*/
                ecnr_alsa_set_scheduling_priority();
                ecnr_audio_rts_handler_clear_xrun(); /*new*/

                /* initialize the ALSA device and RTS data structure */
                err = ecnr_alsa_audio_initialize();

#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)
           #if defined(SDS_DATASET_2)||defined(SDS_DATASET_4)
                if((SDS_DATASET_2 == my_obj->ecnr_current_config_id)||(SDS_DATASET_4 == my_obj->ecnr_current_config_id))
                {
                  /* To initialize silence frames for SDS in Gen4 to resolve NO_FRAME_SYNC*/
                  ecnr_alsa_voice_stream_change_state(AudioState);

                  ecnr_alsa_set_mic_level_watch(TRUE);

                    /* set MIC level notify interval with default value if
                     * its less than D_ECNR_SERVICE_MIC_LEVEL_NOTIFY_INTERVAL */
                    if(D_ECNR_SERVICE_MIC_LEVEL_NOTIFY_INTERVAL > ecnr_service_mic_lvl_notify_interval_in_ms)
                      ecnr_service_mic_lvl_notify_interval_in_ms = D_ECNR_SERVICE_MIC_LEVEL_NOTIFY_INTERVAL;

                    ecnr_service_do_action_micro_level_monitor(ecnr_service_mic_lvl_notify_interval_in_ms);
                }
          #endif /* SDS_DATASET_2 */
#endif /* VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2 */

                if(!err)
                    pr_message("audio streaming prepared\n");

                if(    ecnr_audio_recording_is_rec_active(ECNR_MULTI_CHAN_IN_DEV)
                    || ecnr_audio_recording_is_rec_active(ECNR_MIC_OUT)
                    || ecnr_audio_recording_is_rec_active(ECNR_REF_IN0)
                    || ecnr_audio_recording_is_rec_active(ECNR_REF_IN1))
                {
                   ecnr_audio_recording_enable_mic_rec();
                   ecnr_audio_recording_mic_rec_file_reopen();
                }

                if(    ecnr_audio_recording_is_rec_active(ECNR_RECV_OUT)
                    || ecnr_audio_recording_is_rec_active(ECNR_RECV_IN))
                {
                   ecnr_audio_recording_enable_rcv_rec();
                   ecnr_audio_recording_rcv_rec_file_reopen();
                }
                break;
            }
            case ECNR_AS_STREAM_RUNNING:
            {
#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)
          #if defined(SDS_DATASET_2)||defined(SDS_DATASET_4)
                if((SDS_DATASET_2 == my_obj->ecnr_current_config_id)||(SDS_DATASET_4 == my_obj->ecnr_current_config_id))
                {
                    /* switch streaming for sds gen4 from silence write to mic data*/
                    ecnr_alsa_voice_stream_change_state(AudioState);
                }
          #endif
#endif
                break;
            }
            case ECNR_AS_STREAM_TERMINATED:
            {
#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)
          #if defined(SDS_DATASET_2)||defined(SDS_DATASET_4)
                if((SDS_DATASET_2 == my_obj->ecnr_current_config_id)||(SDS_DATASET_4 == my_obj->ecnr_current_config_id))
                {
                  ecnr_alsa_voice_stream_change_state(AudioState);
                  ecnr_alsa_set_mic_level_watch(FALSE);
                }
         #endif /* SDS_DATASET_2*/
#endif /* VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2 */

                /* 20.08.2018: export recording here*/
                ecnr_audio_recording_export_rec(ECNR_MULTI_CHAN_IN_DEV);
                ecnr_audio_recording_export_rec(ECNR_REF_IN0);
                ecnr_audio_recording_export_rec(ECNR_REF_IN1);
                ecnr_audio_recording_export_rec(ECNR_RECV_IN);
                ecnr_audio_recording_export_rec(ECNR_RECV_OUT);
                ecnr_audio_recording_export_rec(ECNR_MIC_OUT);
                break;
            }
            case ECNR_AS_STREAM_ERROR:
            case ECNR_AS_STREAM_STOP:
            default:
                /* do nothing */
                break;

        }

        if(!err)
        {
            ecnr_object_emit_signal_ecnr_audio_status(AudioState);

            /* save new state */
            g_mutex_lock(&ecnr_service_lock);

            pr_message("change audio state from  %s to %s\n", ecnr_audio_state_str(my_obj->ecnr_service_audio_state), ecnr_audio_state_str(AudioState));
            ETG_TRACE_USR1(("[ecnr_service_audio_change_state]: new  state -> %s",ecnr_audio_state_str(AudioState)));

            my_obj->ecnr_service_audio_state = AudioState;
            g_mutex_unlock(&ecnr_service_lock);

        }
        else
        {
            pr_warning("fails to change state \"%s\" with err:%d \n", ecnr_audio_state_str(AudioState), err);
            ETG_TRACE_USR1(("[ecnr_service_audio_change_state]: fails to change state:%d with err:%d \n", AudioState, err));
        }
    }
    //g_mutex_unlock(&ecnr_service_lock);


  return err;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_audio_get_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
ecnr_audio_state ecnr_service_audio_get_state(void)
{
    ecnr_audio_state state;

    if(!my_obj)
    {
        pr_warning("no valide service instance \n");
        return (ecnr_audio_state)ECNR_AS_STREAM_SILENCE;
    }

    g_mutex_lock(&ecnr_service_lock);

    state = my_obj->ecnr_service_audio_state;

    pr_debug("current audio state: %s  \n", ecnr_audio_state_str(state));

    g_mutex_unlock(&ecnr_service_lock);

    return state;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_start_itctest
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_start_itctest
(
    gint32  troubleCode,
    gint32* rspTroubleCode,
    guchar* result,
    gint32* errorCode
)
{
    int status = (int)ECNR_ERR_OK;
    int err    = (int)ECNR_ERR_OK;
    int res    = (int)ECNR_ITC_PASSED;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

    enum eDataPool dpid = (enum eDataPool)ECNR_DP_NOT_DEF;
    unsigned char*   dpdata = (unsigned char*)NULL;
    guint16 dataset_size = 0;
    int     version[4]={0};
    guint16 crc = 0;
    guint32 u32dpRes = 0;
    int dpsize = 0;

    unsigned short  ConfigId = ECNR_CFG_ID_NA;
    const  char*    crc_log_file = NULL;
    const char* dp_export_file = NULL;


    ETG_TRACE_USR3(("[ecnr_service_fi_start_itctest]: with troubleCode[%d]", troubleCode));

    switch(troubleCode)
    {
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_01:
            dpid = ECNR_DP_BT_PHONE_NB;
            #ifdef PHONE_DATASET_1
            ConfigId = PHONE_DATASET_1;
            crc_log_file = CRC_DP_FILE_BT_PHONE_NB;
            dp_export_file = EXPORT_DP_FILE_BT_PHONE_NB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_02:
            dpid = ECNR_DP_BT_PHONE_WB;
            #ifdef PHONE_DATASET_2
            ConfigId = PHONE_DATASET_2;
            crc_log_file = CRC_DP_FILE_BT_PHONE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_03:
            dpid = ECNR_DP_INT_VOICE_WB;
            #ifdef SDS_DATASET_2
            ConfigId = SDS_DATASET_2;
            crc_log_file = CRC_DP_FILE_INT_VOICE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_04:
            dpid = ECNR_DP_CP_PHONE_NB;
            #ifdef SPI_DATASET_1
            ConfigId = SPI_DATASET_1;
            crc_log_file = CRC_DP_FILE_BT_PHONE_NB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_05:
            dpid = ECNR_DP_CP_PHONE_WB;
            #ifdef SPI_DATASET_2
            ConfigId = SPI_DATASET_2;
            crc_log_file = CRC_DP_FILE_BT_PHONE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_06:
            dpid = ECNR_DP_CP_FTIME_SWB;
            #ifdef SPI_DATASET_7
            ConfigId = SPI_DATASET_7;
            crc_log_file = CRC_DP_FILE_CP_FTIME_SWB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_09:
            dpid = ECNR_DP_BT_VOICE_NB;
            #ifdef PHONE_DATASET_3
            ConfigId = PHONE_DATASET_3;
            crc_log_file = CRC_DP_FILE_BT_VOICE_NB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_0A:
            dpid = ECNR_DP_BT_VOICE_WB;
            #ifdef PHONE_DATASET_4
            ConfigId = PHONE_DATASET_4;
            crc_log_file = CRC_DP_FILE_BT_VOICE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_0B:
            dpid = ECNR_DP_BT_SIRI_WB;
            #ifdef PHONE_DATASET_6
            ConfigId = PHONE_DATASET_6;
            crc_log_file = CRC_DP_FILE_BT_SIRI_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_0C:
            dpid = ECNR_DP_CP_VOICE_WB;
            #ifdef SPI_DATASET_4
            ConfigId = SPI_DATASET_4;
            crc_log_file = CRC_DP_FILE_CP_VOICE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_0D:
            dpid = ECNR_DP_AA_VOICE_WB;
            #ifdef SPI_DATASET_5
            ConfigId = SPI_DATASET_5;
            crc_log_file = CRC_DP_FILE_AA_VOICE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_0E:
            dpid = ECNR_DP_VPA_VOICE_WB;
            #ifdef WEB_DATASET_1
            ConfigId = WEB_DATASET_1;
            crc_log_file = CRC_DP_FILE_VPA_VOICE_WB;
            #endif
            break;
        case ITC_AIVI_CAL_ALGORITHM_BASED_FAILURE_LB_0F:
            dpid = ECNR_DP_WUW_VOICE_WB;
            #ifdef SDS_DATASET_4
            ConfigId = SDS_DATASET_4;
            crc_log_file = CRC_DP_FILE_WUW_VOICE_WB;
            #endif
            break;
        default:
        {
            pr_message("Invalid ITC value\n");
            ETG_TRACE_USR3(("[ecnr_service_fi_start_itctest]: Invalid ITC value"));
            res = ECNR_ITC_NORESULT;
            return ECNR_ERR_INVALID_INPUT;
            break;
        }
    }

    u32dpRes = ecnr_datapool_access_reload_dp_element(dpid);

    /*new*/
    if(u32dpRes)
    {
        /*new data pool */
         ecnr_configuration_set_new_datapool(dpid);

        /* remove from RFS
         *  - BSD/SCD file of the requested configuration
         *  - CRC locker file
         */
        const struct EcnrConfig *pdata = NULL;
        pdata = ecnr_configuration_get_config_data(ConfigId);
        if(pdata)
        {
            /* remove BSD/SCD file */
            (void)ecnr_configuration_remove_bsd_file(pdata->bsd_str);
            (void)ecnr_configuration_remove_crc_file(crc_log_file);

            ecnr_sse_handler_log_new_data_pool_info(ConfigId, pdata);
        }
    }
    pr_debug("datapool reload result for dp ID -> %d is %d\n",dpid, u32dpRes);
    ETG_TRACE_USR3(("[ecnr_service_fi_start_itctest]: data pool reload result for DP ID -> %d is %d", dpid,u32dpRes));

    dpdata = ecnr_datapool_access_get_bsd_dataset(dpid, &dataset_size, version, &crc);

    if(!dpdata)
    {
        pr_message("no data pool available\n");
        ETG_TRACE_ERR(("[ecnr_service_fi_start_itctest]: ERROR"));
        ETG_TRACE_ERR(("-> ErrorText[no data pool available]"));
        ETG_TRACE_ERR(("-> ErrorCode[ECNR_ERR_DP_NO_ELEMENT_AVAILABLE]: %d", ECNR_ERR_DP_NO_ELEMENT_AVAILABLE));
        err = ECNR_ERR_DP_NO_ELEMENT_AVAILABLE;
        res = ECNR_ITC_FAILED;
    }
    else
    {
        /* compare sse version in dataset from datapool with the one of the current sse version  */
        if(FALSE == ecnr_sse_handler_compare_version(version))
        {
            pr_message("the dp data set coded version do not match with current engine version\n");
            ETG_TRACE_ERR(("[ecnr_service_fi_start_itctest]: ERROR"));
            ETG_TRACE_ERR(("-> ErrorText[the DP data set coded version do not match with current engine version]"));
            ETG_TRACE_ERR(("-> ErrorCode[ECNR_ERR_VERSION_SSE_DP_DO_NOT_MATCH]: %d", ECNR_ERR_VERSION_SSE_DP_DO_NOT_MATCH));
            err = ECNR_ERR_VERSION_SSE_DP_DO_NOT_MATCH;
            res = ECNR_ITC_FAILED;
        }
        else
        {
            /*Retrieve DP size from data pool */
            (void)ecnr_datapool_access_get_dp_size_dyn(dpid, &dpsize);

            /*CRC bytes(2) are subtracted from retrieved DP size, since
            * CRC calculation done by excluding CRC stored bytes*/
            if(dpsize > D_NB_CHAR_CRC_DP)
                dpsize = dpsize - D_NB_CHAR_CRC_DP;

            if(FALSE == ecnr_datapool_access_calc_and_compare_crc_ccitt (dpdata,(guint16) dpsize, crc ))
            {
                /* exported data to a file */
                (void)ecnr_datapool_access_export_dp_to_file(dpdata, (guint16) (dpsize + D_NB_CHAR_CRC_DP), dp_export_file);

                ETG_TRACE_ERR(("[ecnr_service_fi_start_itctest]: ERROR"));
                ETG_TRACE_ERR(("-> ErrorText[CRC Mismatch ]"));
                ETG_TRACE_ERR(("-> ErrorCode[ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL]: %d", ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL));
                pr_message("!! CRC Mismatch !!\n");
                err = ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL;
                res = ECNR_ITC_FAILED;
            }
            else
            {
                /* here the check of the frame SSE is performed */
                (void)ecnr_sse_handler_sse_create_instance();
                status = ecnr_sse_handler_set_configuration(ConfigId, TRUE);

                if(status == ECNR_ERR_PERIODE_SIZE_INVALID)
                {
                    ETG_TRACE_ERR(("[ecnr_service_fi_start_itctest]: ERROR"));
                    ETG_TRACE_ERR(("-> ErrorText[invalid frame size]"));
                    ETG_TRACE_ERR(("-> ErrorCode[ECNR_ERR_PERIODE_SIZE_INVALID]: %d", ECNR_ERR_PERIODE_SIZE_INVALID));
                    pr_message("!! invalid frame size !!\n");
                    err = ECNR_ERR_PERIODE_SIZE_INVALID;
                    res = ECNR_ITC_FAILED;
                }
                ecnr_sse_handler_destroy();
            }
        }
    }
#endif //VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

    ETG_TRACE_USR3(("[ecnr_service_fi_start_itctest]: DONE"));
    ETG_TRACE_USR4(("-> troubleCode[%d]",troubleCode));
    ETG_TRACE_USR4(("-> errorCode[%d]",err));
    ETG_TRACE_USR4(("-> result[%d]",res));

    *rspTroubleCode = (gint32)troubleCode;
    *errorCode      = (gint32)err;
    *result      = (guchar)res;

    return ECNR_ERR_OK;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_initialize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_initialize(guchar ecnrAppId, gint ecnrConfigurationId)
{
    int status = (int)ECNR_ERR_OK;

    pr_message("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ETG_TRACE_USR3(("[ecnr_service_fi_initialize]: ENTERED with AppId[%d], CfgId[%d]", ecnrAppId, ecnrConfigurationId));

    /* create new ecnr instance*/
    status = ecnr_service_instance_create_new(ecnrAppId);
    if(status != ECNR_ERR_OK)
    {
        ETG_TRACE_ERR(("[ecnr_service_fi_initialize]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        return status;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        status = ecnr_service_do_action_initialize(ecnrConfigurationId);
    }

    /* try to remove instance */
    if(status != ECNR_ERR_OK)
        (void)ecnr_service_instance_remove(ecnrAppId);
    else
    {
        #ifdef D_TRIGGER_PERFORMANCE_TEST
        ecnr_service_test_performance_initialize();
        #endif

        if(ecnr_service_get_pass_through_mode_default_state())
            ecnr_service_fi_set_pass_through_mode(ECNR_APPID_TEST, TRUE);

#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2))  && !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
        if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
        {
            ecnr_service_fi_set_send_gain_delta(ecnrAppId, ecnr_service_get_mic_gain_level());
        }
#else
            ecnr_service_fi_set_send_gain_delta(ecnrAppId, ecnr_service_get_mic_gain_level());
#endif
        /*start audio thread */
       // (void)ecnr_service_audio_create();
    }

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_service_fi_initialize]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_destroy
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_destroy(guchar ecnrAppId)
{
    int status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;
    gboolean ret =TRUE;


    pr_message("ENTERED with appid(%d)\n", ecnrAppId );
    ETG_TRACE_USR3(("[ecnr_service_fi_destroy]: ENTERED with AppId[%d]", ecnrAppId));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_destroy]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }

    /* stop streaming */
    ret =  ecnr_alsa_stop(ECNR_START_STOP_TIMEOUT);

    /* destroy engine instance */
    if(ret)
    {
        ret = ecnr_alsa_exit_wait(ECNR_EXIT_TIMEOUT);
        if(ret)
        {
            /* leave audio thread*/
            status = ecnr_service_do_action_destroy();

            if(status)
            {
                pr_critical("the ecnr destroyed failed with err %d\n", status);
                ETG_TRACE_ERR(("[ecnr_service_fi_destroy]: the ecnr destroyed failed with err %d", status));
            }
        }
        else
        {
            /* write a critical log a error memory */
            pr_critical("the termination of the alsa thread has timed out\n");
            ETG_TRACE_ERR(("[ecnr_service_fi_destroy]: the termination of the alsa thread has timed out"));
        }
    }
    else
    {
        /* write a critical log a error memory */
        pr_critical("the termination of the alsa thread has timed out\n");
        ETG_TRACE_ERR(("[ecnr_service_fi_destroy]: the termination of the alsa thread has timed out"));
    }


    (void)ecnr_service_instance_remove(ecnrAppId);

    pr_message("EXIT with err %d\n", status );
    ETG_TRACE_USR3(("[ecnr_service_fi_destroy]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_start_audio
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_start_audio(guchar ecnrAppId, guchar StartMode)
{
    int         status = (int)ECNR_ERR_OK;
    struct      ecnr_instance_object* my_inst;
    gboolean    state = TRUE;
    glong       timeout = 0 ;

    pr_message("ENTERED with appid(%d) with StartMode(%d)\n", ecnrAppId, StartMode);
    ETG_TRACE_USR3(("[ecnr_service_fi_start_audio]: ENTERED with AppId[%d] and StartMode[%d]", ecnrAppId, StartMode));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_start_audio]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }

    my_inst = ecnr_service_instance_get(ecnrAppId);
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_INSTANCE_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_start_audio]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_INSTANCE_AVAILABLE"));
        return ECNR_ERR_NO_INSTANCE_AVAILABLE;
    }
    else
    {
        if (my_inst->ecnr_is_initialized)
        {

#ifdef D_SIMULATE_SPI_CLIENT
            //ecnr_service_test_client_start();
#endif

            gboolean started = TRUE;

            /*start audio thread*/
            (void)ecnr_service_audio_create(); /*fix me*/

            if(StartMode == START_MODE_WAIT_STREAMING_RESULT)
                timeout = ECNR_START_STOP_TIMEOUT;
            else if (StartMode == START_MODE_RETURN_IMMEDIATELY)
                timeout = 0;
            else // default
                 timeout = ECNR_START_STOP_TIMEOUT;

            started = ecnr_alsa_start(timeout);
            if (!started)
                status = ECNR_ERR_NO_AUDIO_STREAMING;
#ifdef D_TRIGGER_PERFORMANCE_TEST
            else
                ecnr_service_test_performance_start();
#endif
        }
        else
            status = ECNR_ERR_MODULE_NOT_INIT;
    }


    pr_message("EXIT with error code -> %d", status );
    //to remove prio2 lint warning (Info 715: Symbol not referenced)
    ETG_TRACE_USR3(("[ecnr_service_fi_start_audio]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    //return status;
    return ECNR_SRV_FCT_RET(status, "ecnr_service_fi_start_audio");
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_stop_audio
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_stop_audio(guchar ecnrAppId)
{
    gboolean stopped, state;
    int status = (int)ECNR_ERR_OK;

    pr_message("ENTERED with appid(%d)\n", ecnrAppId );
    ETG_TRACE_USR3(("[ecnr_service_fi_stop_audio]: ENTERED with AppId[%d]", ecnrAppId));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_stop_audio]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
#ifdef D_TRIGGER_PERFORMANCE_TEST
        ecnr_service_test_performance_stop();
#endif

#ifdef D_SIMULATE_SPI_CLIENT
        ecnr_service_test_client_stop();
#endif

        stopped = ecnr_alsa_stop(ECNR_START_STOP_TIMEOUT);
        if (!stopped)
        {
            /*new section */
            pr_warning("the stop audio function is timed out\n");

            /* leave audio thread*/
            ecnr_alsa_exit_wait(ECNR_EXIT_TIMEOUT);
            status = ECNR_ERR_OK;
        }
    }

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_service_fi_stop_audio]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_configuration
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_configuration(guchar ecnrAppId, gint ecnrConfigurationId)
{
    int status = ECNR_ERR_OK;
    gboolean state = TRUE;
    gboolean CreateNewSession = TRUE; //FALSE;
    struct ecnr_instance_object* my_inst;

    pr_message("ENTERED with appid(%d) and cfgid (%d)\n", ecnrAppId, ecnrConfigurationId  );
    ETG_TRACE_USR3(("[ecnr_service_fi_set_configuration]: ENTERED with AppId[%d], CfgId[%d]", ecnrAppId, ecnrConfigurationId));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_set_configuration]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }

    /* check first for AppId*/
    status = ecnr_service_check_AppId_ConfigId_Matching(ecnrConfigurationId, ecnrAppId);

    if (status == ECNR_ERR_OK)
    {
        /* check whether streaming is currently active */
        if(ecnr_alsa_get_audio_run_state())
        {
            pr_warning("audio streaming is active, ecnrStopAutio should be called before\n");
            status = ECNR_ERR_OBJECT_FI_AUDIO_IS_RUNNING;
        }
        else
        {
            /* retrieve current instance */
            my_inst = ecnr_service_instance_active_get();

            /* check whether config Id can be configured  */
            if(my_inst)
            {
                if (my_inst->ecnr_is_initialized)
                {
                    if(my_inst->ecnr_current_config_id == ecnrConfigurationId)
                    {
                        pr_warning("sse already initialized with Config %d", ecnrConfigurationId);
                        CreateNewSession = FALSE;
                        status = 0;
                    }
                    else
                        (void)ecnr_service_do_action_destroy();
                }
            }

            if (CreateNewSession)
            {
                status = ecnr_service_do_action_initialize(ecnrConfigurationId);

                /* try to remove instance */
                if(status != ECNR_ERR_OK)
                    (void)ecnr_service_instance_remove(ecnrAppId);
                else
                {
                    if(ecnr_service_get_pass_through_mode_default_state())
                        ecnr_service_fi_set_pass_through_mode(ECNR_APPID_TEST, TRUE);

#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2)) && !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
                    if(ecnrConfigurationId == PHONE_DATASET_1 || ecnrConfigurationId == PHONE_DATASET_2)
                    {
                        ecnr_service_fi_set_send_gain_delta(ecnrAppId, ecnr_service_get_mic_gain_level());
                    }
#else
                    ecnr_service_fi_set_send_gain_delta(ecnrAppId, ecnr_service_get_mic_gain_level());
#endif

                    /*start audio thread */
                    //(void)ecnr_service_audio_create();
                }
            }
        }
    }

    pr_message("EXIT with error (%d)", status );
    ETG_TRACE_USR3(("[ecnr_service_fi_set_configuration]: EXIT with ErrCode[%s]", ecnr_error_str(status)));

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_send_mute_switch
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_send_mute_switch(guchar ecnrAppId, guchar ecnrSendMuteSwitch)
{
    int     status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_set_send_mute_switch]: called with AppId[%d], MuteState[%d]", ecnrAppId, ecnrSendMuteSwitch));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_set_send_mute_switch]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        status = ecnr_engine_handler_set_send_mute_switch(ecnrSendMuteSwitch);
    }

    pr_message("EXIT with error (%d)\n", status );

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_send_mute_switch
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_send_gain_delta(guchar ecnrAppId, gint16 ecnrLevel)
{
    int     status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_set_send_gain_delta]: ENTERED with AppId[%d], level[%d]", ecnrAppId, ecnrLevel));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_set_send_gain_delta]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        status = ecnr_engine_handler_set_send_gain_delta(ecnrLevel);
    }

    pr_message("EXIT with error (%d)\n", status);

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_mic_gain
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_mic_gain(gint16 MicGainLevel)
{
    int status = ECNR_ERR_OK;
    struct ecnr_instance_object* my_inst = NULL;
    guchar ecnrAppId;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_set_mic_gain]: called with level[%d]", MicGainLevel));

    if(MicGainLevel < -30 || MicGainLevel > 30)
    {
        status = ECNR_ERR_ILLEGAL_SET_PARAM;
    }
    else
    {
        ecnr_service_set_mic_gain_level(MicGainLevel);

        /* retrieve current instance */
        my_inst = ecnr_service_instance_active_get();

        if(my_inst)
        {

            if(my_inst->ecnr_is_initialized)
            {
                ecnrAppId = my_inst->ecnr_current_app_id;


#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2))  && !defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM)
                if(my_inst->ecnr_current_config_id == PHONE_DATASET_1 || my_inst->ecnr_current_config_id == PHONE_DATASET_2)
                {
                    ecnr_service_fi_set_send_gain_delta(ecnrAppId, MicGainLevel);
                }
#else
                ecnr_service_fi_set_send_gain_delta(ecnrAppId, MicGainLevel);
#endif
            }
        }
    }

    pr_message("EXIT with error (%d)", status );

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_pass_through_mode
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_pass_through_mode(guchar ecnrAppId, guchar ecnrPassThroughMode)
{
    int   status = (int)ECNR_ERR_OK;
    struct ecnr_instance_object* my_inst = NULL;

    pr_message("ENTERED with ecnrPassThroughMode value -> (%d)\n", ecnrPassThroughMode );
    ETG_TRACE_USR3(("[ecnr_service_fi_set_pass_through_mode]: called with AppId[%d], Mode[%d]", ecnrAppId, ecnrPassThroughMode));

    if(ecnr_service_get_pass_through_mode_default_state() && (ecnrAppId != ECNR_APPID_TEST ))
        return ECNR_ERR_OK;

    if( ecnrAppId == ECNR_APPID_TEST )
    {
        ecnr_service_set_pass_through_mode_default_state(ecnrPassThroughMode);
    }

    /* retrieve current instance */
    my_inst = ecnr_service_instance_active_get();

    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_set_pass_through_mode]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    if(my_inst->ecnr_is_initialized)
    {
        status = ecnr_engine_handler_set_pass_through_mode(ecnrPassThroughMode);
    }

    pr_message("EXIT with error (%d)\n", status);
    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_nr_comb_floor
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_nr_comb_floor(guchar ecnrAppId, gint16 ecnrNRCombFloor)
{
    int     status = (int)ECNR_ERR_OK;
    gboolean state = TRUE;

    pr_message("ENTERED with ecnrNRCombFloor value %d\n", ecnrNRCombFloor);

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_set_nr_comb_floor]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        status = ecnr_engine_handler_set_nr_comb_floor(ecnrNRCombFloor);
    }

    pr_message("EXIT with error (%d)", status );

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_eecho_cancellation
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_echo_cancellation(guchar ecnrAppId, guchar EcnrECState)
{
    int         status  = ECNR_ERR_OK;
    gboolean    state   = TRUE;

    pr_message("ENTERED with EC state (%d)\n",EcnrECState);
    ETG_TRACE_USR3(("[ecnr_service_fi_set_eecho_cancellation]: called with AppId[%d], AECstate[%d]", ecnrAppId, EcnrECState));

    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_set_eecho_cancellation]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        status = ecnr_engine_handler_set_echo_cancellation(EcnrECState);
    }

    pr_message("EXIT with error (%d)\n", status);
    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_set_noise_reduction
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_set_noise_reduction(guchar ecnrAppId, guchar EcnrNRState)
{
    int         status  = ECNR_ERR_OK;
    gboolean    state   = TRUE;

    pr_message("ENTERED with NR state (%d)\n",EcnrNRState);
    ETG_TRACE_USR3(("[ecnr_service_fi_set_noise_reduction]: called with AppId[%d], NRstate[%d]", ecnrAppId, EcnrNRState));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_set_noise_reduction]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        status = ecnr_engine_handler_set_noise_reduction(EcnrNRState);
    }

    pr_message("EXIT with error (%d)\n", status);
    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_alsa_set_device
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_alsa_set_device (guchar ecnrAppId, guint AlsaDeviceSelect, gchar * AlsaDeviceName)
{
    int      status = ECNR_ERR_OK;
    gboolean state  = TRUE;
    gboolean set    = FALSE;
    enum ecnr_audio dev_id_rep = (enum ecnr_audio)ECNR_MIC_IN0;
    enum ecnr_audio dev_id_set = (enum ecnr_audio)ECNR_MIC_IN0;
    snd_pcm_stream_t sdir = SND_PCM_STREAM_PLAYBACK;

    pr_message("ENTERED App(%d) set device(%s) to channel(%d)\n", ecnrAppId, AlsaDeviceName,AlsaDeviceSelect);
    ETG_TRACE_USR3(("[ecnr_service_fi_alsa_set_device]: called with AppId[%d], DevType[%d], DevName[%s]", ecnrAppId, AlsaDeviceSelect, AlsaDeviceName));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_alsa_set_device]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        dev_id_set = (enum ecnr_audio)AlsaDeviceSelect;

        if(dev_id_set >= NUM_ECNR_AUDIO)
        {
            pr_message("dev_id (%d) out of range (max %d)",dev_id_set, NUM_ECNR_AUDIO);
            status = ECNR_ERR_DEV_ID_NOT_AVAILABLE;
        }

        if(!status)
        {
            set = TRUE;

            /*  set device per string */
            if(    dev_id_set == ECNR_SET_PLB_DEV_PER_STRING
                || dev_id_set == ECNR_SET_CAP_DEV_PER_STRING)
            {
                if (dev_id_set == ECNR_SET_PLB_DEV_PER_STRING)
                    sdir = SND_PCM_STREAM_PLAYBACK;
                else if (dev_id_set == ECNR_SET_CAP_DEV_PER_STRING)
                    sdir = SND_PCM_STREAM_CAPTURE;

                dev_id_rep = ecnr_configuration_get_dev_id(AlsaDeviceName,sdir);
                if(dev_id_rep != NUM_ECNR_AUDIO)
                {
                    dev_id_set = dev_id_rep;
                }
                else
                    set = FALSE;
            }

            /* the multi-channel input device ALSA device is set*/
            if(     dev_id_set == ECNR_MULTI_CHAN_IN_DEV
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_MIC_0
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_MIC_1
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_REF_0
                ||  dev_id_set == ECNR_MULTI_CHAN_IN_REF_1 )
            {
                ecnr_alsa_set_mutli_channel_in_stream_mode(TRUE);
            }

            else if ( dev_id_set == ECNR_MIC_IN0 )
            {
                ecnr_alsa_set_mutli_channel_in_stream_mode(FALSE);
            }

            if(set)
            {
                status = ecnr_configuration_check_match_dev_chid(dev_id_set, AlsaDeviceName);
                if(status == ECNR_ERR_OK)
                {
                    ecnr_alsa_set_dev_name(dev_id_set, AlsaDeviceName);
                    pr_debug("ecnr_alsa_set_dev_name to ecnr audio channel %d and device name %s \n" ,dev_id_set, AlsaDeviceName);
                }
            }
            else
                status = ECNR_ERR_DEV_ID_NOT_AVAILABLE;
        }
    }

    if(status == ECNR_ERR_OK)
        ecnr_alsa_disable_dev_configuration(dev_id_set);

    pr_message("EXIT with error (%d)", status );
    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_alsa_set_device_properties
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int  ecnr_service_fi_alsa_set_device_properties
(
    guchar  ecnrAppId,
    guint  AlsaDeviceSelect,
    guchar  AlsaDeviceProperty,
    guint  PropertyValue
)
{
    int         status  = ECNR_ERR_OK;;
    gboolean    state   = TRUE;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_alsa_set_device_properties]: ENTERED"));


    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_alsa_set_device_properties]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        switch(AlsaDeviceProperty)
        {
            case PROP_ALSA_DEV_OPEN_ORDER:
                ecnr_alsa_set_device_open_order((enum ecnr_audio)AlsaDeviceSelect, PropertyValue);
                ecnr_alsa_dev_order_ctrl(FALSE);
                break;
            case PROP_ALSA_DEV_PREFILL:
                ecnr_alsa_set_device_prefill_value(PropertyValue);
                ecnr_alsa_prefill_ctrl(FALSE);
                break;
            default:
                break;
        }
    }

    pr_message("EXIT with error (%d)", status );
    return status;
}



/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_get_ecnr_configuration
*
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_get_ecnr_configuration (gchar ** DatasetName, GArray ** EcnrProperties)
{
    int                             status = ECNR_ERR_OK;
    guint16                         crc = 0;
    guint32                         ecnr_properties[4];
    const struct EcnrConfig*        pdata = NULL;
    struct ecnr_instance_object*    my_inst = NULL;
    guint32                         val;
    gint                            i;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_get_ecnr_configuration]: ENTERED"));

    /* retrieve current instance */
    my_inst = ecnr_service_instance_active_get();

    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_get_ecnr_configuration]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    if(my_inst->ecnr_is_initialized)
    {
      /* retrieve configuration data for the requested copnfiguration Id */
      pdata = ecnr_configuration_get_config_data((unsigned short)my_inst->ecnr_current_config_id);

        if (!pdata)
        {
            pr_warning("no valid configuration for configId : % d", my_inst->ecnr_current_config_id);
            return ECNR_ERR_CONFIG_NOT_AVAILABLE;
        }

        if (pdata->opeth_str)
            *DatasetName = g_strdup((gchar *) pdata->opeth_str);
        else
            *DatasetName = g_strdup("");

        pr_debug("ecnr_dataset_name(%s)", pdata->opeth_str);

        if(!ecnr_configuration_get_crc_from_current_active_dataset(pdata->EcnrdPBaseKey, &crc))
            pr_message("value crc(0x%x) of dp(%d)", crc,pdata->EcnrdPBaseKey);

        ecnr_properties[0] = crc;
        ecnr_properties[1] = my_inst->ecnr_current_app_id;
        ecnr_properties[2] = (guint)my_inst->ecnr_current_config_id;
        ecnr_properties[3] = pdata->configId;

        *EcnrProperties = g_array_sized_new(FALSE, FALSE, sizeof(guint), 4);

        for (i = 0; i < 4 ; i++)
        {
            val = ecnr_properties[i];
            g_array_append_val(*EcnrProperties, val);

            pr_debug("EcnrProperties[%i] = %d", i, val);
        }
    }

    pr_message("EXIT with error (%d)", status );
    return status;
}


static void ecnr_object_print_all_debug_options(void)
{
    pr_message("************* Debug options ****************\n");
    pr_message(" 8  ---> print device properties \n");
    pr_message(" 9  ---> print rts configuration \n");
    pr_message(" 10 ---> print configuration data \n");
    pr_message(" 20 ---> print all datapool infos \n");
    pr_message(" 30 ---> activate audio recording with environment variable ECNR_REC_STREAM_TO_FILE \n");
    pr_message(" 31 ---> stop audio recording if activated per environment variable ECNR_REC_STREAM_TO_FILE \n");
    pr_message(" 32 ---> restart audio recording if activated per environment variable ECNR_REC_STREAM_TO_FILE \n");
    pr_message(" 33 ---> activate debug log level  \n");
    pr_message(" 34 ---> deactivate debug log level \n");
    pr_message(" 35 ---> terminate ecnr main  \n");
    pr_message(" 38 ---> enable performance test  \n");
    pr_message(" 39 ---> disable performance test  \n");
    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_start_debug_data
*
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int ecnr_service_fi_start_debug_data(guchar ecnrDebugMode)
{
    int status = ECNR_ERR_OK;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_start_debug_data]: ENTERED"));


    switch (ecnrDebugMode)
    {
        case ECNR_DBUG_PRINT_ALL_OPTIONS:
            ecnr_object_print_all_debug_options();
            break;
        case ECNR_DBUG_SET_TRIGGER_FILE:
            if(!ecnr_sse_set_debug_trigger_file())
                status=ECNR_ERR_DGB_SET_RMV_TRIGGER_FILE_FAILED;
            break;
        case ECNR_DBUG_FORCE_AUDIO_IO: //3
            break;
        case ECNR_DBUG_DATA_ALSA_AVAILABLE: //4
            status = ecnr_alsa_enable_sse_debug();
            break;
        case ECNR_DBUG_START_TUNING_SESSION: //6
            (void)ecnr_engine_start_debug_session(); //ecnr_sse_debug_interface_start_session();
            (void)ecnr_sse_set_debug_trigger_file();
            break;
        case ECNR_DBUG_STOP_TUNING_SESSION: //7
            (void)ecnr_engine_end_debug_session();
            (void)ecnr_sse_remove_debug_trigger_file(); //ecnr_sse_debug_interface_start_session();
            break;
        case ECNR_DBUG_PRINT_DEV_PROP: //8
            status = ecnr_alsa_print_dev_properties();
            break;
        case ECNR_DBUG_PRINT_RTS_CONFIG: //9
            ecnr_audio_rts_handler_print_config_data();
            break;
        case ECNR_DBUG_PRINT_ECNR_DATA_CONFIG:
            ecnr_configuration_print_config_data();
            break;
        case ECNR_DBUG_TEST_TRIGGER_NOTIFICATION_NEW_DATAPOOL_ALL:
    #ifdef D_DP_NOTIFICATION_SUPPORT
            ecnr_configuration_stub_notify_dp_change();
    #endif
            break;
        case ECNR_DBUG_TEST_TRIGGER_NOTIFICATION_NEW_DATAPOOL_1:
        #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            ecnr_datapool_access_trigger_new_datapool(ECNR_DP_BT_PHONE_NB);
        #endif
            break;
        case ECNR_DBUG_TEST_FORCE_EXIT_ALSA_THREAD:
            ecnr_alsa_exit_wait(ECNR_EXIT_TIMEOUT);
            break;
        case ECNR_DBUG_TEST_FORCE_CREATE_ALSA_THREAD:
            (void)ecnr_service_audio_create();
            break;
        case ECNR_DBUG_PRINT_CURRENT_AUDI0_STATE:
            ecnr_alsa_request_print_current_audio_state();
            break;
        case ECNR_DBUG_PRINT_RCS_CONF_FILES:
        #ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
            ecnr_qwa_handler_print_rcs_configuration();
        #endif
            break;
        case ECNR_CALL_NATIVE_START_AUDIO:
            ecnr_alsa_start(ECNR_START_STOP_TIMEOUT);
            break;
        case ECNR_DBUG_PRINT_DP_INFOS:
            ecnr_configuration_print_all_dp_infos();
            break;
        case ECNR_DBUG_SET_IO_CTRL_HF_VR_MODE:
            ecnr_ioctrl_set_pin_state(92, TRUE);
            break;
        case ECNR_DBUG_RESET_IO_CTRL_HF_VR_MODE:
            ecnr_ioctrl_set_pin_state(92, FALSE);
            break;
        case ECNR_DBUG_FORCE_USE_DEFAULT_DP:  //23
            ecnr_service_set_use_default_dp();
            break;
        case ECNR_DBUG_FORCE_USE_DEFAULT_DP_EXIT: //24
            ecnr_service_set_use_default_dp_exit();
            break;

        case ECNR_DBUG_SET_ACTIVATE_AUDIO_RECORD:
            (void)setenv("ECNR_REC_STREAM_TO_FILE"/*const char *name*/,
                        "7"/*const char *value*/,
                        1/*int overwrite*/);
            ecnr_audio_recording_init();
            break;
        case ECNR_DBUG_STOP_AUDIO_RECORD:
            ecnr_audio_recording_stop();
            break;
        case ECNR_DBUG_RESTART_AUDIO_RECORD:
            ecnr_audio_recording_restart();
            break;
        case ECNR_DBUG_SET_ACTIVATE_DEBUG_LOG_LEVEL:
            (void)setenv("G_MESSAGES_DEBUG"/*const char *name*/,
                         "all"/*const char *value*/,
                         1/*int overwrite*/);
            break;
        case ECNR_DBUG_SET_DEACTIVATE_DEBUG_LOG_LEVEL:
            (void)unsetenv("G_MESSAGES_DEBUG"/*const char *name*/);
            break;
        case ECNR_DBUG_FORCE_MAIN_TERMINATION:
            ecnr_service_handle_quit_signal();
            break;
#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
        case ECNR_DBUG_IMPORT_BIN_FILE_OPEN:
            ecnr_datapool_import_from_bin_file(TRUE);
            break;
        case ECNR_DBUG_IMPORT_BIN_FILE_CLOSE:
            ecnr_datapool_import_from_bin_file(FALSE);
            break;
#endif

        case ECNR_DBUG_ENABLE_PERFORMANCE_TEST:
            ecnr_engine_handler_performance_enable(TRUE);
            ecnr_audio_rts_handler_performance_enable(TRUE);
            break;
        case ECNR_DBUG_DISABLE_PERFORMANCE_TEST:
            ecnr_engine_handler_performance_enable(FALSE);
            ecnr_audio_rts_handler_performance_enable(FALSE);
            break;
        case ECNR_DBUG_SWITCH_OFF_CHECK_ALSA_DEV:
        case ECNR_DBUG_SWITCH_OFF_CHECK_APPID_CNFGID:
        default:
            status = ECNR_ERR_DEBUG_MODER_NOT_SUPPORTED;
            break;
    }

    pr_message("EXIT with error (%d)", status );
    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_mic_clip_stats
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_mic_clip_stats
(
       gint  Channel
     ,guint* ClipFrames
     ,guint* ClipCount
      ,gint* MaxAmp
      ,gint* MinAmp
)
{
    int           status = (int)ECNR_ERR_OK;
    struct ecnr_instance_object*    my_inst;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_mic_clip_stats]: ENTERED"));

    /* retrieve current instance */
    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_mic_clip_stats]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    status = ecnr_engine_handler_get_mic_in_clip_stats
            (
                     Channel
                    ,ClipFrames
                    ,ClipCount
                    ,MaxAmp
                    ,MinAmp
            );

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_ref_clip_stats
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_ref_clip_stats(guint * ClipFrames, guint * ClipCount, gint * MaxAmp, gint * MinAmp)
{
    int           status = (int)ECNR_ERR_OK;
    struct ecnr_instance_object*    my_inst;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_ref_clip_stats]: ENTERED"));


    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_ref_clip_stats]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    status = ecnr_engine_handler_get_ref_in_clip_stats
            (
                     0
                    ,ClipFrames
                    ,ClipCount
                    ,MaxAmp
                    ,MinAmp
            );

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_recv_clip_stats
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_recv_clip_stats
(
    guint* ClipFrames,
    guint* ClipCount,
     gint* MaxAmp,
     gint* MinAmp
)
{
    int                             status = (int)ECNR_ERR_OK;
    struct ecnr_instance_object*    my_inst;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_recv_clip_stats]: ENTERED"));


    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_recv_clip_stats]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    status = ecnr_engine_handler_get_rcv_in_clip_stats
            (
                     0
                    ,ClipFrames
                    ,ClipCount
                    ,MaxAmp
                    ,MinAmp
            );

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_mic_out_clip_stats
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_mic_out_clip_stats
(
    guint* ClipFrames,
    guint* ClipCount,
     gint* MaxAmp,
     gint* MinAmp
)
{
    int                             status = (int)ECNR_ERR_OK;
    struct ecnr_instance_object*    my_inst;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_mic_out_clip_stats]: ENTERED"));


    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_mic_out_clip_stats]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    status = ecnr_engine_handler_get_mic_out_clip_stats
            (
                     0
                    ,ClipFrames
                    ,ClipCount
                    ,MaxAmp
                    ,MinAmp
            );

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_recv_out_clip_stats
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_recv_out_clip_stats
(
    guint* ClipFrames,
    guint* ClipCount,
     gint* MaxAmp,
     gint* MinAmp
)
{
    int           status = (int)ECNR_ERR_OK;
    struct ecnr_instance_object*    my_inst;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_recv_out_clip_stats]: ENTERED"));


    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_recv_out_clip_stats]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    status = ecnr_engine_handler_get_rcv_out_clip_stats
            (
                     0
                    ,ClipFrames
                    ,ClipCount
                    ,MaxAmp
                    ,MinAmp
            );

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_alsa_set_thread_priority
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_alsa_set_thread_priority(guchar ecnrAppId, gint priority)
{
    int         status  = ECNR_ERR_OK;
    gboolean    state   = TRUE;
    int        thread_priority; //modified for lint fix lint574

    thread_priority =  (int)priority; // g_value_get_uint(value);

    pr_message("ENTERED, current Thread Prio is :%d, request Prio ist : %d\n", thread_priority, ecnr_alsa_get_thread_priority());
    ETG_TRACE_USR3(("[ecnr_service_fi_destroy]: ENTERED with AppId[%d]", ecnrAppId));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        ETG_TRACE_USR3(("[ecnr_service_fi_alsa_set_thread_priority]: EXIT with ErrCode[%s]", ecnr_error_str(status)));
        pr_message("EXIT with ErrCode[%s]\n", ecnr_error_str(status));
        return status;
    }
    else
    {
        if (thread_priority > sched_get_priority_max(SCHED_FIFO))
        {
            thread_priority = sched_get_priority_max(SCHED_FIFO);
            pr_debug("set Thread Prio is :%d", thread_priority);

            ecnr_alsa_set_thread_priority(thread_priority);
        }
    }

    return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_service_fi_set_param
 * DESCRIPTION  :
 * PARAMETER    :
 * RETURN VALUE :
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 *****************************************************************************/
int ecnr_service_fi_set_param(guint prop_id,ecnr_type_defs data_type, const GValue* value)
{
  int status = ECNR_ERR_OK;
  pr_message("ENTERED\n");

  switch (data_type)
  {
      case ECNR_TYPE_I16:
      case ECNR_TYPE_U16:
      case ECNR_TYPE_U32:
      {
          if(prop_id == ecnrMicLvlNtfyInterval)
          {
            guint32 notify_interval_in_ms = (guint32)g_value_get_uint(value);
            pr_message("property id(%d) -> notify interval (%u) ms",prop_id,notify_interval_in_ms);

            ecnr_service_set_micro_level_notify_interval(notify_interval_in_ms);
          }
          break;
      }
      default:
      {
        pr_critical("!! invalid parameter id (%d) !!",prop_id);
        status = ECNR_ERR_INVALID_PARAM_ID;
        break;
      }
  }

  pr_message("EXIT\n");
  return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_object_ecnr_get_version
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean ecnr_service_fi_get_version
(
    GArray ** version,
    gchar ** version_string,
    gchar ** version_comment
)
{
    unsigned int                    i;
    gint                            val;
    unsigned int                    ecnr_VersionArrayLen = 0;
    const int*                      ecnr_VersionArray = NULL;
    const char*                     ecnr_VersionString = NULL;
    const char*                     ecnr_VersionComment = NULL;
    struct ecnr_instance_object*    my_inst;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_object_ecnr_get_version]: ENTERED"));

    my_inst = ecnr_service_instance_active_get();
    if(!my_inst)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_get_version]: EXIT with ErrCode[%s]", "ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE"));
        return ECNR_ERR_NO_ECNR_OBJECT_AVAILABLE;
    }

    ecnr_engine_get_version( &ecnr_VersionArrayLen, &ecnr_VersionArray, &ecnr_VersionString, &ecnr_VersionComment);

    *version = g_array_sized_new(FALSE, FALSE, sizeof(guint), 4);
    if (ecnr_VersionArray)
    {
        for (i = 0; i < ecnr_VersionArrayLen; i++)
        {
            val = ecnr_VersionArray[i];
            g_array_append_val(*version, val);
        }
    }
    if (ecnr_VersionString)
        *version_string = g_strdup((gchar *) ecnr_VersionString);
    else
        *version_string = g_strdup("");
    if (ecnr_VersionComment)
        *version_comment = g_strdup((gchar *) ecnr_VersionComment);
    else
        *version_comment = g_strdup("");

    pr_debug("Version %s", *version_string);

   return ECNR_ERR_OK;
}

/******************************************************************************
 * FUNCTION     : ecnr_service_set_micro_level_notify_interval
 * DESCRIPTION  : set for microphone level notification frequency
 * PARAMETER    : notification frequency in milliseconds
 * RETURN VALUE : none
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
void ecnr_service_set_micro_level_notify_interval(guint32 interval_in_ms)
{
  pr_debug("ENTERED with MIC level notify interval (%u)ms\n",interval_in_ms);
  ETG_TRACE_USR4(("[ecnr_service_set_micro_level_notify_interval]:ENTERED with MIC level notify interval (%u)ms",interval_in_ms));

  ecnr_service_mic_lvl_notify_interval_in_ms = interval_in_ms;

  ETG_TRACE_USR4(("[ecnr_service_set_micro_level_notify_interval]:EXIT"));
  pr_debug("EXIT\n");
}

/******************************************************************************
 * FUNCTION     : ecnr_service_get_micro_level_notify_interval
 * DESCRIPTION  : get MIC level notify interval
 * PARAMETER    : none
 * RETURN VALUE : returns MIC level notify interval
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
guint32 ecnr_service_get_micro_level_notify_interval(void)
{
  return ecnr_service_mic_lvl_notify_interval_in_ms;
}

/******************************************************************************
 * FUNCTION     : ecnr_service_start_micro_level_notify
 * DESCRIPTION  : GSource timer calls this function on fixed timeout value
 *              : and here we retrieve the MIC level and emit the signal
 * PARAMETER    : none
 * RETURN VALUE : TRUE -> if MIC level monitor is active
 *              : FALSE -> if MIC level monitor is stopped
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 * 12.07.2019   | Ranjit Susal(RBEI/ECO)  | code optimization
 *****************************************************************************/
static gboolean  ecnr_service_start_micro_level_notify(void)
{
  gboolean status = FALSE;
  guint32 ecnr_mic_lvl = 0;
  pr_debug("ENTERED\n");

  status = ecnr_alsa_get_mic_level_watch();

  /* if returned FALSE, this function call timer stops */
  if(!status)
  {
    pr_message("terminating the MIC level monitoring -> status(%d)\n",status);
    ETG_TRACE_USR4(("[ecnr_service_start_micro_level_notify]: terminating the MIC level monitoring"));

    return status;
  }

  ecnr_mic_lvl = ecnr_alsa_get_mic_level();
  ecnr_object_emit_signal_ecnr_micro_level_status(ecnr_mic_lvl);

  pr_debug("EXIT\n");
  return status;
}

/******************************************************************************
 * FUNCTION     : ecnr_service_end_micro_level_notify
 * DESCRIPTION  : Triggered when the data to the call back function is no longer useful
 * PARAMETER    : none
 * RETURN VALUE : none
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 *****************************************************************************/
static void ecnr_service_end_micro_level_notify(void)
{
  pr_debug("ENTERED\n");
  pr_debug("EXIT\n");
}

/******************************************************************************
 * FUNCTION     : ecnr_service_do_action_micro_level_monitor
 * DESCRIPTION  : start the timer based on the notify frequency interval
 * PARAMETER    : timeout_ms -> notify frequency interval in milliseconds
 * RETURN VALUE : none
 * HISTORY      :
 * Date         | Author                  | Modification
 * 10.06.2019   | Raghunandan GK(RBEI/ECO)| Initial Revision
 *****************************************************************************/
void ecnr_service_do_action_micro_level_monitor(guint32 timeout_ms)
{
  pr_message("ENTERED with interval (%u)ms\n",timeout_ms);
  ETG_TRACE_USR4(("[ecnr_service_do_action_micro_level_monitor]: ENTERED with interval (%u)ms",timeout_ms));

  ecnr_mic_lvl_time_source = g_timeout_source_new ((guint)timeout_ms);
  g_source_set_priority (ecnr_mic_lvl_time_source, G_PRIORITY_DEFAULT);
  g_source_set_callback (ecnr_mic_lvl_time_source, (GSourceFunc)ecnr_service_start_micro_level_notify, NULL, (GDestroyNotify)ecnr_service_end_micro_level_notify);

  GMainContext* context = g_main_loop_get_context(ecnr_main_loop);
  g_source_attach(ecnr_mic_lvl_time_source, context);
  g_source_unref(ecnr_mic_lvl_time_source);

  ETG_TRACE_USR4(("[ecnr_service_do_action_micro_level_monitor]: EXIT"));
  pr_message("EXIT");
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_get_status_message
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_get_status_message
(
     guchar   ecnrAppId
     ,gchar** status_message
)
{

    int status = (int)ECNR_ERR_OK;
    gboolean  state = TRUE;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[ecnr_service_fi_get_status_message]: ENTERED"));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_get_status_message]: EXIT with ErrCode[%s]", "ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE"));
        return ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE;
    }
    else
    {
        status = ecnr_engine_handler_get_status_message(status_message);
    }

   return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_service_fi_ecnr_engine_reset
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int ecnr_service_fi_ecnr_engine_reset(guchar ecnrAppId)
{

    int status = (int)ECNR_ERR_OK;
    gboolean  state = TRUE;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_service_fi_ecnr_engine_reset]: ENTERED"));

    status = ecnr_service_instance_active_is(ecnrAppId, &state);
    if(!state)
    {
        pr_message("EXIT with ErrCode[%s]\n", "ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE");
        ETG_TRACE_USR3(("[ecnr_service_fi_ecnr_engine_reset]: EXIT with ErrCode[%s]", "ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE"));
        return ECNR_ERR_APPID_INSTANCE_NOT_AVAILABLE;
    }
    else
    {
        status = ecnr_engine_handler_ecnr_engine_reset();
    }

   return status;
}
