/******************************************************************************
 * FILE        : ecnr-ringbuffer.c
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 14.10.2013  | Patrick Rey            | initial version
 *             |                        | ring buffer feature handling
 *****************************************************************************/


#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-object.h"
#include "ecnr-service.h"
#include "ecnr-alsa.h"
#include "ecnr-ringbuffer.h"


/*******************************************************************************
*
* FUNCTION: ecnr_ring_buffer_get_free_bytes
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int ecnr_ring_buffer_get_free_bytes( tsRingbuffer * psRingbuffer )//FreeBytesRingbuffer( tsRingbuffer * psRingbuffer )
{
    int iFreeBytes = 0;

    if ( NULL == psRingbuffer)
    {
        return 0;
    }
    else
    {
        if ( psRingbuffer->iWriteIndex == psRingbuffer->iReadIndex )
        {
            iFreeBytes = psRingbuffer->iLength - 1;
        }
        else if ( psRingbuffer->iWriteIndex > psRingbuffer->iReadIndex )
        {
            iFreeBytes = ( psRingbuffer->iLength - 1 ) + ( psRingbuffer->iReadIndex - psRingbuffer->iWriteIndex );
        }
        else
        {
            iFreeBytes = (psRingbuffer->iReadIndex - psRingbuffer->iWriteIndex) - 1;
        }

        return iFreeBytes;
    }
}

/*******************************************************************************
*
* FUNCTION: ecnr_ring_buffer_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_ring_buffer_create(tsRingbuffer ** ppsRingbuffer, unsigned int uiSize) //buffer_create( tsRingbuffer ** ppsRingbuffer, unsigned int uiSize )
{
    pr_debug("Entered\n");

    if ( ( NULL == ppsRingbuffer)
        ||(NULL != *ppsRingbuffer)
        ||(0 == uiSize))
    {
        pr_debug("function parameter failure");
        return ECNR_ERR_ALLOC;
    }
    else
    {
        tsRingbuffer * psRingbuffer = (tsRingbuffer*)g_malloc0(sizeof(tsRingbuffer) ); // 1, sizeof(tsRingbuffer) );

        if ( NULL == psRingbuffer )
        {
            pr_debug("ringbuffer allocation failed");
            return ECNR_ERR_ALLOC;
        }
        else
        {
            psRingbuffer->pcHead = (char*)g_malloc0(uiSize +1); //calloc( uiSize +1, sizeof(char) );

            if ( NULL == psRingbuffer->pcHead )
            {
                pr_debug("ringbuffer allocation failed");

                free(psRingbuffer);
                psRingbuffer = NULL;

                return ECNR_ERR_ALLOC;
            }
            else
            {
                psRingbuffer->iLength = (int)(uiSize + 1); /* one byte more to detect a full ringbuffer */
                psRingbuffer->iReadIndex = 0;
                psRingbuffer->iWriteIndex = 0;

                *ppsRingbuffer = psRingbuffer;
            }
        }
    }

    pr_debug("EXIT\n");
    return ECNR_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: ecnr_ring_buffer_destroy
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_ring_buffer_destroy( tsRingbuffer ** ppsRingbuffer )
{
    pr_debug("Entered\n");

    if (  ( NULL == ppsRingbuffer)
        ||( NULL == *ppsRingbuffer))
    {
        pr_debug("exit with error %d\n", ECNR_ERR_ALLOC);
        return ECNR_ERR_ALLOC;
    }
    else
    {
        tsRingbuffer * psRingbuffer = *ppsRingbuffer;

        if(psRingbuffer->pcHead)
            g_free( psRingbuffer->pcHead );
        if(psRingbuffer)
            g_free( psRingbuffer );
        *ppsRingbuffer = NULL;
    }

    pr_debug("EXIT\n");
    return ECNR_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: ecnr_ring_buffer_data_insert
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_ring_buffer_data_insert( tsRingbuffer * psRingbuffer, const char * pcData, int iSize )
{
    int iFreeBytes = 0;
    int iWrittenBytes = 0;
    int i;


    if (  (NULL == psRingbuffer)
        ||(NULL == pcData)
        ||(0 > iSize ))
    {
        return 0;
    }
    else
    {

        iFreeBytes = ecnr_ring_buffer_get_free_bytes( psRingbuffer );

        if( iFreeBytes < iSize )
        {
            iWrittenBytes = iFreeBytes;
        }
        else
        {
            iWrittenBytes = iSize;
        }

        for ( i=0; i<iWrittenBytes; i++ )
        {
            psRingbuffer->pcHead[ psRingbuffer->iWriteIndex ] = pcData[i];

            /* mind the ringbuffer wrap */
            psRingbuffer->iWriteIndex++;
            if( psRingbuffer->iWriteIndex == psRingbuffer->iLength )
            {
                psRingbuffer->iWriteIndex = 0;
            }
        }
        return iWrittenBytes;
    }
}


/*******************************************************************************
*
* FUNCTION: ecnr_ringbuffer_extractFromRingbuffer
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
int ecnr_ring_buffer_data_extract(tsRingbuffer * psRingbuffer, char * pcData, int iSize)
{
    int iAvailableBytes = 0;
    int iExtractedBytes = 0;
    int i;


    if (  (NULL == psRingbuffer)
        ||(NULL == pcData )
        ||( 0 > iSize ) )
    {
        return 0;
    }
    else
    {
        iAvailableBytes = (psRingbuffer->iLength - 1) - ecnr_ring_buffer_get_free_bytes( psRingbuffer );

        if ( iAvailableBytes < iSize )
        {
            iExtractedBytes = iAvailableBytes;
        }
        else
        {
            iExtractedBytes = iSize;
        }

        for ( i=0; i<iExtractedBytes; i++ )
        {
            pcData[i] = psRingbuffer->pcHead[ psRingbuffer->iReadIndex ];

            /* mind the ringbuffer wrap */
            psRingbuffer->iReadIndex++;
            if( psRingbuffer->iReadIndex == psRingbuffer->iLength )
            {
                psRingbuffer->iReadIndex = 0;
            }
        }
        return iExtractedBytes;
    }
}


