/******************************************************************************
 * FILE        : ecnr-datapool-access.cpp
 * PROJECT     : Gen3 and Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2014 RBCM GMBH
 * HISTORY     :
 * Date        | Author                  | Modification
 * 14.04.2014  | Patrick Rey             | initial version
 *             |                         | data pool features handling
 *****************************************************************************/

#define SPM_UNIT_TESTS // solve warning

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX

#ifdef D_ECNR_DATAPOOL_SUPPORT
#include "dp/dp_def.h"
#include "ccaservice.h"
#define DP_DATAPOOL_ID =CCA_C_U16_APP_DIAGDEBUG
#endif

#define DP_S_IMPORT_INTERFACE_BASE
#include "dp_if.h"


#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#ifdef D_ECNR_DATAPOOL_SUPPORT
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_diagdebug_if.h"
#endif

#include <boost/crc.hpp>      // for boost::crc_basic, boost::crc_optimal

#ifdef __cplusplus
  extern "C" {
#endif
/*********************************************************************************/
/*                         define                                                */
/*********************************************************************************/

#include <glib.h>
#include <glib-unix.h>
#include <glib/gi18n.h>
#include <glib-object.h>

#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>

#include "ecnr-error.h"
#include "ecnr-common-defs.h"
#include "ecnr-service.h"
#include "ecnr-datapool-access.h"
#include "ecnr-alsa.h"
#include "ecnr-configuration.h"


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#include "ecnr-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_ECNR
#include "trcGenProj/Header/ecnr-datapool-access.cpp.trc.h"
#endif


#define ECNR_NUM_DP_INFO ARRAYSIZE(dp_info)


/*****************************************
 * constant definition and class symbols
 * for datapool access class
 */
#ifdef D_ECNR_DATAPOOL_SUPPORT

#define ECNR_S_IMPORT_INTERFACE_DP_ONLY
#define ECNR_S_IMPORT_INTERFACE_DP
#include "ecnr-datapool-default.h"
#include "ecnr-configuration-data.h"

#endif // D_ECNR_DATAPOOL_SUPPORT



//#define NB_ELEM_MAX 21
#if (defined (VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RIVIE) || defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI2))
#define D_SUPPORT_DP_RELOAD
#endif

/***********************
 * definition dp test class
 */

typedef struct _ecnr_dp_info
{
    enum eDataPool dp;
    int dp_size;
}ecnr_dp_info;

static int curr_dp_buffer_size = 0;
static unsigned char* dpReqBuf = (unsigned char*)NULL;
static struct ecnr_dp_reg ecnr_dp_reg_table[NB_ELEM_MAX];
static ecnr_dp_info ecnr_dp_access_info[NB_ELEM_MAX];
static guint8 ecnr_dp_dyn_nb;
static gboolean ecnr_datapool_access_enable_bin_file_import = FALSE;


/*********************************************************************************/
/*                         static function                                       */
/*********************************************************************************/


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_reg
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static struct ecnr_dp_reg* ecnr_datapool_access_get_reg(enum eDataPool dp)
{
    int i = 0;
    struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    for (i = 0; i < NB_ELEM_MAX ; i++)
    {
        ps = &ecnr_dp_reg_table[i];
        if(ps->dp == dp)
        {
            pr_debug("registration for element(%d) was found\n", dp);
            return ps;
        }
    }

    return (struct ecnr_dp_reg*)NULL;
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_nb_phone
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_nb_phone(void)
{
    struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_BT_PHONE_NB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_BT_PHONE_NB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_BT_PHONE_NB);

}
/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_wb_phone
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_wb_phone(void)
{
    struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_BT_PHONE_WB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_BT_PHONE_WB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_BT_PHONE_WB);

}


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_nb_voice
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_nb_voice(void)
{
    struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_BT_VOICE_NB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_BT_VOICE_NB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_BT_VOICE_NB);

}


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_wb_voice
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_wb_voice(void)
{
   struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_BT_VOICE_WB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_BT_VOICE_WB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_BT_VOICE_WB);
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_cp_nb_phone
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_cp_nb_phone(void)
{
   struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_CP_PHONE_NB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_CP_PHONE_NB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_CP_PHONE_NB);
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_cp_wb_phone
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_cp_wb_phone(void)
{
   struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_CP_PHONE_WB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_CP_PHONE_WB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_CP_PHONE_WB);
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_new_cp_wb_voice
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static tVoid ecnr_datapool_new_cp_wb_voice(void)
{
   struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;

    pr_debug("entered\n");

    ps = ecnr_datapool_access_get_reg(ECNR_DP_CP_VOICE_WB);

    if(ps)
    {
        if(ps->ecnrdpFunc)
        {
            /* call notification function */
            pr_debug("call notification function for element(%d)\n", ECNR_DP_CP_VOICE_WB);
            ps->ecnrdpFunc();
        }
        else
            pr_debug("no valid notification function for element(%d)\n",ps->dp);
    }
    else
        pr_debug("no registration for the dp element(%d)\n", ECNR_DP_CP_VOICE_WB);
}

#define  BIN_IMPORT_FILE


#if 0
static int ecnr_datapool_access_read_from_bin_file(unsigned char* pd, guint32 dpsize, const char* binfile)
{
  /*TODO*/

  (void)pd;
  (void)dpsize;
  (void)binfile;
  return 0;
}
#endif

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_bt_phone_nb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_bt_phone_nb(unsigned char* pd, guint32 dpsize)
{
    int    status = DP_S32_NO_ERR;
    long int read = 0;
    FILE *fhdl = NULL;

    pr_debug("ENTERED\n");

    dp_tclECNR_BT_PHONE_NB dpelem;
    if(pd)
    {
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);

        if(ecnr_datapool_access_enable_bin_file_import)
        {
            memset((void*)pd, 0, dpsize);
            fhdl = fopen(IMPORT_BIN_FILE_BT_PHONE_NB, "rb");
            if(fhdl)
            {
                if(!status)
                {
                    pr_debug("no datapool available \n");
                    fseek(fhdl, 0L, SEEK_END);
                    read = ftell(fhdl);
                    fseek(fhdl, 0L, SEEK_SET); /* go back to the beginning */
                    pr_debug("no datapool available, take file size -> %ld \n",read );
                    status = (int)read;/*TO Do in LP64*/
                }
                if(status > 0)
                	read = (long int)fread(pd, 1, (size_t)status, fhdl);/*TO Do in LP64*/
                fclose(fhdl);
                pr_debug("import %ld bytes from bin file -> %s\n",read, IMPORT_BIN_FILE_BT_PHONE_NB);
            }
        }
    }
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_bt_phone_wb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_bt_phone_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;
    FILE *fhdl = NULL;
    long int read = 0;

    dp_tclECNR_BT_PHONE_WB dpelem;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_bt_phone_wb]: ENTERED"));

    if(pd)
    {
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
        if(ecnr_datapool_access_enable_bin_file_import)
        {
            (void)memset((void*)pd, 0, dpsize);
            fhdl = fopen(IMPORT_BIN_FILE_BT_PHONE_WB, "r");
            if(fhdl)
            {
                if(!status)
                {
                    pr_debug("no datapool available \n");
                    fseek(fhdl, 0L, SEEK_END);
                    read = (long int)ftell(fhdl);
                    fseek(fhdl, 0L, SEEK_SET); /* go back to the beginning */
                    pr_debug("no datapool available, take file size -> %ld \n",read );
                    status = (int)read;
                }
                if(status > 0)
                	read = (long int)fread(pd, 1, (size_t)status, fhdl);
                fclose(fhdl);
                pr_debug("import %ld bytes from bin file -> %s\n",read, IMPORT_BIN_FILE_BT_PHONE_WB);
            }
        }
    }
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_bt_voice_nb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_bt_voice_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_BT_VOICE_NB dpelem;

    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_bt_voice_nb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_bt_voice_wb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_bt_voice_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_BT_VOICE_WB dpelem;

    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_bt_voice_wb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_bt_siri_nb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_bt_siri_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_BT_SIRI_NB dpelem;

    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_bt_siri_nb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_bt_siri_wb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_bt_siri_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_BT_SIRI_WB dpelem;

    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_bt_siri_wb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_voice_nb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_voice_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_VOICE_NB dpelem;
    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_cp_voice_nb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_voice_wb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_voice_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_VOICE_WB dpelem;
    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_cp_voice_wb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_voice_swb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_voice_swb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_VOICE_SWB dpelem;
    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_cp_voice_swb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_phone_nb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_phone_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_PHONE_NB dpelem;

    pr_debug("entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_dp_cp_phone_nb]: ENTERED"));

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_phone_wb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_phone_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_PHONE_WB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_phone_swb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_phone_swb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_PHONE_SWB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_cp_ftime_swb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_cp_ftime_swb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_CP_FTIME_SWB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_aa_voice_nb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_aa_voice_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_AA_VOICE_NB dpelem;
    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_aa_voice_wb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_aa_voice_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_AA_VOICE_WB dpelem;
    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_aa_voice_swb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_aa_voice_swb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_AA_VOICE_SWB dpelem;
    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_aa_phone_nb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_aa_phone_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_AA_PHONE_NB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_aa_phone_wb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_aa_phone_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_AA_PHONE_WB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_get_dp_aa_phone_swb
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static int  ecnr_datapool_access_get_dp_aa_phone_swb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_AA_PHONE_SWB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_int_voice_nb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_int_voice_nb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_INT_VOICE_NB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}


/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_get_dp_int_voice_wb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_int_voice_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_INT_VOICE_WB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
* FUNCTION: ecnr_datapool_access_get_dp_wuw_voice_wb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_wuw_voice_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_WUW_VOICE_WB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

/*******************************************************************************
* FUNCTION: ecnr_datapool_access_get_dp_vpa_voice_wb
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
static int  ecnr_datapool_access_get_dp_vpa_voice_wb(unsigned char* pd, guint32 dpsize)
{
    int status = DP_S32_NO_ERR;

    dp_tclECNR_VPA_VOICE_WB dpelem;

    pr_debug("entered");

    if(pd)
        status = dpelem.s32GetData((tU8*)pd,(tU32)dpsize);
    else
        status = DP_S32_ERR_UNKNOWN;

    pr_debug("datapool read returned status(%d)\n", status);

    return status;
}

#ifdef D_DP_NOTIFICATION_SUPPORT
/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_nb_phone
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_nb_phone(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif

    pr_debug("entered");

    dp_tclECNR_BT_PHONE_NB dpelem1;

    ret = dpelem1.bAddNotification((tVoid*)ecnr_datapool_new_nb_phone, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_nb_phone notification is done\n");

    dpelem1.~dp_tclECNR_BT_PHONE_NB();

    return ret;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_wb_phone
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_wb_phone(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif

    pr_debug("entered");

    dp_tclECNR_BT_PHONE_WB dpelem2;

    ret = dpelem2.bAddNotification((tVoid*)ecnr_datapool_new_wb_phone, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_wb_phone notification is done\n");

    dpelem2.~dp_tclECNR_BT_PHONE_WB();

    return ret;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_nb_voice
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_nb_voice(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif

    pr_debug("entered");

    dp_tclECNR_BT_VOICE_NB dpelem3;

    ret = dpelem3.bAddNotification((tVoid*)ecnr_datapool_new_nb_voice, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_nb_voice notification is done\n");

    dpelem3.~dp_tclECNR_BT_VOICE_NB();

    return ret;
}


/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_wb_voice
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_wb_voice(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif

    pr_debug("entered");

    dp_tclECNR_BT_VOICE_WB dpelem4;

    ret = dpelem4.bAddNotification((tVoid*)ecnr_datapool_new_wb_voice, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_wb_voice notification is done\n");

    dpelem4.~dp_tclECNR_BT_VOICE_WB();

    return ret;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_nb_cp_phone
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_nb_cp_phone(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif //D_ECNR_DATAPOOL_SUPPORT

    pr_debug("entered");

    dp_tclECNR_CP_PHONE_NB dpelem1;

    ret = dpelem1.bAddNotification((tVoid*)ecnr_datapool_new_cp_nb_phone, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_nb_phone notification is done\n");

    dpelem1.~dp_tclECNR_CP_PHONE_NB();

    return ret;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_wb_cp_phone
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_wb_cp_phone(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif

    pr_debug("entered");

    dp_tclECNR_CP_PHONE_WB dpelem1;

    ret = dpelem1.bAddNotification((tVoid*)ecnr_datapool_new_cp_wb_phone, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_nb_phone notification is done\n");

    dpelem1.~dp_tclECNR_CP_PHONE_WB();

    return ret;
}

/*******************************************************************************
 *
 * FUNCTION: ecnr_datapool_access_add_notification_dp_wb_cp_voice
 * DESCRIPTION: ..
 *
 *
 * PARAMETER: None.
 *
 * RETURNVALUE: ecnr error
 *
 *******************************************************************************/
static gboolean ecnr_datapool_access_add_notification_dp_wb_cp_voice(void)
{
    gboolean ret = FALSE;

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return TRUE;
#endif
    pr_debug("entered");

    dp_tclECNR_CP_VOICE_WB dpelem1;

    ret = dpelem1.bAddNotification((tVoid*)ecnr_datapool_new_cp_wb_voice, DP_DEFSET_USER|DP_DEFSET_TEF);

    if(ret)
        pr_debug("datapool_new_nb_phone notification is done\n");

    dpelem1.~dp_tclECNR_CP_VOICE_WB();

    return ret;
}
#endif //D_DP_NOTIFICATION_SUPPORT

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_create_dp
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ECNR error
*
*******************************************************************************/
static void ecnr_datapool_access_create_dp(void)
{
#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("dp not supported\n");
#else

    pr_debug("entered\n");

    /* release buffer*/
    curr_dp_buffer_size = 0;

    if(dpReqBuf)
        g_free(dpReqBuf);
    dpReqBuf = NULL;

    /* create a data pool for this process */
    DP_vCreateDatapool();

#endif
    return;
}

/*******************************************************************************
*
* FUNCTION: ecnr_datapool_access_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: ecnr error
*
*******************************************************************************/
void ecnr_datapool_access_init(void)
{

#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("data pool not supported");
    return;
#endif

    pr_debug("entered");

    /* create datapool for this process*/
    ecnr_datapool_access_create_dp();

    /* data module structure initialization  */
    memset(ecnr_dp_reg_table, 0 , NB_ELEM_MAX * sizeof(ecnr_dp_reg));

    /* data set crc table init*/
//    memset(ecnr_dataset_size_dyn_table, 0 , NB_ELEM_MAX * sizeof(int));
    memset(ecnr_dp_access_info, 0 , NB_ELEM_MAX * sizeof(ecnr_dp_info));

    ecnr_dp_dyn_nb = 0;
    ecnr_configuration_retrieve_all_dp_size();


#ifdef D_DP_NOTIFICATION_SUPPORT
    /* add notification function though all datapool*/
    ecnr_datapool_access_add_notification_dp_nb_phone();
    ecnr_datapool_access_add_notification_dp_wb_phone();
    ecnr_datapool_access_add_notification_dp_nb_voice();
    ecnr_datapool_access_add_notification_dp_wb_voice();
    ecnr_datapool_access_add_notification_dp_nb_cp_phone();
    ecnr_datapool_access_add_notification_dp_wb_cp_phone();
    ecnr_datapool_access_add_notification_dp_wb_cp_voice();
#endif

}



/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_finalize
* PARAMETER:    none
* RETURNVALUE:
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
void ecnr_datapool_access_finalize(void)
{
    if(dpReqBuf)
        g_free(dpReqBuf);
    dpReqBuf = NULL;
    curr_dp_buffer_size = 0;

    return;
}



/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_create_dp_buffer
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/



static int ecnr_datapool_access_create_dp_buffer(int* dpsize)
{
    int err = ECNR_ERR_OK;
    int size = *dpsize;

    //size = size + D_DP_ALL_INFO_ITEMS_SIZE;

    if(!size)
    {
        pr_debug("request dp buffer size(%d) is incorrect\n", size );
        return ECNR_ERR_DP_ERROR_SIZE_OF_DATAPOOL;
    }

    if(size == curr_dp_buffer_size)
    {
        /* do nothing*/
        pr_debug("Buffer with size(%d) still exist\n", curr_dp_buffer_size);
        memset(dpReqBuf, 0 , size * sizeof(unsigned char));

        /* return dp size with info items */
        *dpsize = curr_dp_buffer_size;

        return ECNR_ERR_OK;
    }

    /* first free current buffer and allocate a new one with new size */

    if(dpReqBuf)
        free(dpReqBuf);
    dpReqBuf= NULL;

    /* new allocate with new requested size */
    dpReqBuf = (unsigned char*)g_malloc0(size * sizeof(unsigned char));

    if(dpReqBuf)
    {
        pr_debug("new buffer from current size(%d) to new size(%d)\n", curr_dp_buffer_size, size);
        /* save cuuremt buffer size */
        curr_dp_buffer_size = size;

        /* return dp size with info items */
        *dpsize = size;

    }
    else
        err = ECNR_ERR_NULL_POINTER;

    return err;
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_dp_size_dyn
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/

int ecnr_datapool_access_get_dp_size_dyn(enum eDataPool dpkey, int* dpsize)
{
    int i =  0;
    ecnr_dp_info* pd = (ecnr_dp_info*) NULL;

    if(   (dpkey == 0)
        ||(dpsize == NULL))
    {
        pr_warning("parameter failure\n");
        return ECNR_ERR_NO_DEFAULT_CONFIGURATION;
    }

    for(i = 0, pd = ecnr_dp_access_info; i < NB_ELEM_MAX; i++, pd++)
    {
        if(pd->dp == dpkey)
        {
            pr_debug("dyn size of dp(%d) is : %d\n", dpkey, pd->dp_size);
            *dpsize = pd->dp_size;
            break;
        }
        else
        {
            // do nothing
        }
    }

    return ECNR_ERR_OK;
}

/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_set_dp_size_dyn
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/

int ecnr_datapool_access_set_dp_size_dyn(enum eDataPool dpkey, int dpsize)
{
    int i =  0;
    ecnr_dp_info* pd = (ecnr_dp_info*) NULL;

    if(    (dpkey == 0)
        || ((dpkey -1) > NB_ELEM_MAX))
    {
        return ECNR_ERR_NO_DEFAULT_CONFIGURATION;
    }

    for(i = 0, pd = ecnr_dp_access_info; i < NB_ELEM_MAX; i++, pd++)
    {
        if(pd->dp == dpkey)
        {
            if(pd->dp_size == dpsize)
            {
              pr_debug("size of dp(%d) remains same : %d\n", dpkey, pd->dp_size);
              ETG_TRACE_USR3(("[ecnr_datapool_access_set_dp_size_dyn]: size of dp(%d) remains same : %d",dpkey, pd->dp_size));
            }
            else
            {
              pr_debug("DP size changed for dp(%d) : from [old size : %d ] -> [new size : %d ]\n", dpkey, pd->dp_size,dpsize);
              ETG_TRACE_USR3(("[ecnr_datapool_access_set_dp_size_dyn]: DP size changed for dp(%d) : from [old size : %d ] -> [new size : %d ]",dpkey, pd->dp_size,dpsize));

              pd->dp_size = dpsize;
            }

            return ECNR_ERR_OK;
        }
        else if(pd->dp == 0)
        {
            pr_debug("set dyn dp size(%d) for dp(%d)\n", dpsize, dpkey);
            pd->dp = dpkey;
            pd->dp_size = dpsize;
            ecnr_dp_dyn_nb++;
            break;
        }
        else
        {
            // do nothing
        }
    }
    return ECNR_ERR_OK;
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_bsd_dataset
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
void   ecnr_datapool_import_from_bin_file(gboolean avail)
{
    ecnr_datapool_access_enable_bin_file_import = avail;
    pr_message("import bin file avail -> %d", ecnr_datapool_access_enable_bin_file_import);
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_bsd_dataset
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
unsigned char* ecnr_datapool_access_get_bsd_dataset(enum eDataPool dp, guint16* size_dataset, int* pversion, guint16* crc)
{
    int status = DP_S32_NO_ERR;
    int err = ECNR_ERR_OK;
    int dpsize = 0;
    int dssize = 0;

    pr_debug("retrieve dp for datapool element %d", dp);
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_bsd_dataset]: retrieve dp for datapool element %d", dp));

    /* init retrieve dp element buffer */
#ifdef D_ECNR_USE_DYN_DATASET_SIZE
    dpsize = (int)D_ECNR_USE_MAX_DP_SIZE;
    pr_debug("the max. size of datapool is: %d\n", dpsize);
#else
    (void)ecnr_configuration_get_dp_size(dp, &dpsize);
#endif

    /* request dp buffer, return bsd size extended with the bytes nb required to code item infos */
    err = ecnr_datapool_access_create_dp_buffer(&dpsize);

    if(err)
    {
        pr_warning("failed to allocate memory for dp buffer\n");
        goto err_label;
    }

    switch (dp)
    {
        case ECNR_DP_BT_PHONE_NB:
            status = ecnr_datapool_access_get_dp_bt_phone_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_PHONE_WB:
            status = ecnr_datapool_access_get_dp_bt_phone_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_VOICE_NB:
            status = ecnr_datapool_access_get_dp_bt_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_VOICE_WB:
            status = ecnr_datapool_access_get_dp_bt_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_SIRI_NB:
            status = ecnr_datapool_access_get_dp_bt_siri_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_SIRI_WB:
            status = ecnr_datapool_access_get_dp_bt_siri_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_VOICE_NB:
            status = ecnr_datapool_access_get_dp_cp_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_VOICE_WB:
            status = ecnr_datapool_access_get_dp_cp_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_VOICE_SWB:
            status = ecnr_datapool_access_get_dp_cp_voice_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_PHONE_NB:
            status = ecnr_datapool_access_get_dp_cp_phone_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_PHONE_WB:
            status = ecnr_datapool_access_get_dp_cp_phone_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_PHONE_SWB:
            status = ecnr_datapool_access_get_dp_cp_phone_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_FTIME_SWB:
            status = ecnr_datapool_access_get_dp_cp_ftime_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_VOICE_NB:
            status = ecnr_datapool_access_get_dp_aa_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_VOICE_WB:
            status = ecnr_datapool_access_get_dp_aa_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_VOICE_SWB:
            status = ecnr_datapool_access_get_dp_aa_voice_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_PHONE_NB:
            status = ecnr_datapool_access_get_dp_aa_phone_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_PHONE_WB:
            status = ecnr_datapool_access_get_dp_aa_phone_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_PHONE_SWB:
            status = ecnr_datapool_access_get_dp_aa_phone_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_INT_VOICE_NB:
            status = ecnr_datapool_access_get_dp_int_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_INT_VOICE_WB:
            status = ecnr_datapool_access_get_dp_int_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_WUW_VOICE_WB:
            status = ecnr_datapool_access_get_dp_wuw_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_VPA_VOICE_WB:
            status = ecnr_datapool_access_get_dp_vpa_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        default:
            break;
    }

    /* error while reading from dp */
    if(status < 0 ) //DP_S32_ERR_RD_BUF_LEN)
        goto err_label;


    /**********************************************/
    /* retrieve data set size and version as data */
    /* set row data                               */
    /**********************************************/

    /* the datappol was updated by the diagnose, status return the size of the datapool */
    else if(status > 0)
    {

        pr_debug("size of the dataset(%d) is: %d\n", dp, status);
        ETG_TRACE_USR3(("[ecnr_datapool_access_get_bsd_dataset]: size of the dataset(%d) is: %d", dp, status));

        /*If the size of dp gets changed on reload of DP element, then set new dp size from datapool */
        ecnr_datapool_access_set_dp_size_dyn(dp,status);


        /* size */
        if(size_dataset)
            err = ecnr_datapool_access_get_config_data_size(dpReqBuf, size_dataset);

        /* version */
        if(!err)
        {
            if(size_dataset)
            {
                pr_debug("size (%d) data set in dp element (%d)", *size_dataset, dp);
                ETG_TRACE_USR3(("[ecnr_datapool_access_get_bsd_dataset]: size (%d) data set in dp element (%d)", *size_dataset, dp));
            }
            if(pversion)
                err = ecnr_datapool_access_get_engine_version(dpReqBuf, pversion);

            if(!err && pversion)
            {
                pr_debug("version (%d:%d:%d:%d) engine in dp element (%d)", pversion[0], pversion[1], pversion[2],pversion[3], dp);
                ETG_TRACE_USR3(("[ecnr_datapool_access_get_bsd_dataset]: version (%d:%d:%d:%d) engine in dp element (%d)", pversion[0], pversion[1], pversion[2],pversion[3], dp));
            }
        }

        /* crc */
        if(!err)
        {
#ifdef D_ECNR_USE_DYN_DATASET_SIZE
            (void)ecnr_datapool_access_get_dp_size_dyn(dp, &dssize);
#else
            dssize = dpsize;
#endif

            /* set crc position with dp */
            if(dssize > D_NB_CHAR_CRC_DP)
                dssize = dssize - D_NB_CHAR_CRC_DP;

            if(crc)
                err = ecnr_datapool_access_get_config_data_crc(dssize, dpReqBuf, crc);

            if(!err && crc)
            {
                pr_debug("crc(0x%x)in dp element(%d)", *crc, dp);
                ETG_TRACE_USR3(("[ecnr_datapool_access_get_bsd_dataset]: crc(0x%x)in dp element(%d)", *crc, dp));
            }
//            else
//                goto err_label;
        }
        else
            goto err_label;
    }
    else // no data available in datapool
        goto err_label;

    return (unsigned char*)dpReqBuf;

err_label:
    if(size_dataset)
        *size_dataset = (guint16)NULL;
    if(pversion)
        memset(pversion,0, D_NB_CHAR_VERSION * sizeof(guint8));
    pr_debug("no datapool available\n");
    ETG_TRACE_USR3(("[ecnr_datapool_access_get_bsd_dataset]: no datapool available"));
    return (unsigned char*)NULL;
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_bsd_dataset_size
* PARAMETER:    none
* RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
int ecnr_datapool_access_get_bsd_dataset_size(enum eDataPool dp)
{
    int status = DP_S32_NO_ERR;
    int err = ECNR_ERR_OK;
    int dpsize = 0;


    pr_debug("retrieve dp for datapool element %d", dp);

    dpsize = (int)D_ECNR_USE_MAX_DP_SIZE;
    pr_debug("the max. size of datapool is: %d\n", dpsize);

    /* request dp buffer, return bsd size extended with the bytes nb required to code item infos */
    err = ecnr_datapool_access_create_dp_buffer(&dpsize);

    if(err)
    {
        pr_warning("failed to allocate memory for dp buffer\n");
        return ECNR_ERR_ALLOC;
    }

    switch (dp)
    {
        case ECNR_DP_BT_PHONE_NB:
            status = ecnr_datapool_access_get_dp_bt_phone_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_PHONE_WB:
            status = ecnr_datapool_access_get_dp_bt_phone_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_VOICE_NB:
            status = ecnr_datapool_access_get_dp_bt_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_VOICE_WB:
            status = ecnr_datapool_access_get_dp_bt_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_SIRI_NB:
            status = ecnr_datapool_access_get_dp_bt_siri_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_BT_SIRI_WB:
            status = ecnr_datapool_access_get_dp_bt_siri_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_VOICE_NB:
            status = ecnr_datapool_access_get_dp_cp_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_VOICE_WB:
            status = ecnr_datapool_access_get_dp_cp_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_VOICE_SWB:
            status = ecnr_datapool_access_get_dp_cp_voice_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_PHONE_NB:
            status = ecnr_datapool_access_get_dp_cp_phone_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_PHONE_WB:
            status = ecnr_datapool_access_get_dp_cp_phone_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_PHONE_SWB:
            status = ecnr_datapool_access_get_dp_cp_phone_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_CP_FTIME_SWB:
            status = ecnr_datapool_access_get_dp_cp_ftime_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_VOICE_NB:
            status = ecnr_datapool_access_get_dp_aa_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_VOICE_WB:
            status = ecnr_datapool_access_get_dp_aa_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_VOICE_SWB:
            status = ecnr_datapool_access_get_dp_aa_voice_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_PHONE_NB:
            status = ecnr_datapool_access_get_dp_aa_phone_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_PHONE_WB:
            status = ecnr_datapool_access_get_dp_aa_phone_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_AA_PHONE_SWB:
            status = ecnr_datapool_access_get_dp_aa_phone_swb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_INT_VOICE_NB:
            status = ecnr_datapool_access_get_dp_int_voice_nb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_INT_VOICE_WB:
            status = ecnr_datapool_access_get_dp_int_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_WUW_VOICE_WB:
            status = ecnr_datapool_access_get_dp_wuw_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        case ECNR_DP_VPA_VOICE_WB:
            status = ecnr_datapool_access_get_dp_vpa_voice_wb(dpReqBuf,(tU32)dpsize);
            break;
        default:
            break;
    }

    pr_debug("size of the dataset(%d) is: %d\n", dp, status);

    return status;
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_engine_version
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
int ecnr_datapool_access_get_engine_version(unsigned char* data, int* version)
{
    int err = ECNR_ERR_OK;

    if(!version)
        return ECNR_ERR_NULL_POINTER;

    if(!data)
        return ECNR_ERR_NULL_POINTER;

    memmove(version, &data[D_POS_VERSION] , D_NB_CHAR_VERSION * sizeof(guint8));

    pr_debug("engine version (%d:%d:%d:%d) \n", version[0], version[1], version[2], version[3]);

    return err;
}

/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_config_data_size
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
int ecnr_datapool_access_get_config_data_size(unsigned char* data, guint16* size)
{
    int err = ECNR_ERR_OK;

    if(!size)
        return ECNR_ERR_NULL_POINTER;

    if(!data)
        return ECNR_ERR_NULL_POINTER;

    memmove(size, &data[D_POS_SIZE_DP] , D_NB_CHAR_SIZE_DP * sizeof(guint8));

    pr_debug("data set size(%d) from bytes data[%d](%d) and data[%d](%d) \n",
        *size,
        D_POS_SIZE_DP,
        data[D_POS_SIZE_DP],
        (D_POS_SIZE_DP + 1),
        data[D_POS_SIZE_DP + 1] );

    return err;
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_config_data_crc
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
int ecnr_datapool_access_get_config_data_crc(int crc_pos, unsigned char* data, guint16* crc)
{
    int err = ECNR_ERR_OK;

    if(!crc)
        return ECNR_ERR_NULL_POINTER;

    if(!data)
        return ECNR_ERR_NULL_POINTER;

    //memmove(crc, &data[D_POS_CRC_DP] , D_NB_CHAR_CRC_DP * sizeof(guint8));
    memmove(crc, &data[crc_pos] , D_NB_CHAR_CRC_DP * sizeof(guint8));

    pr_debug("data set crc(0x%x) at position(%d)\n", *crc, crc_pos);

    return err;
}


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_get_config_row_data_start_pos
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
int ecnr_datapool_access_get_config_row_data_start_pos(unsigned char* data, guint16* size)
{
    int err = ECNR_ERR_OK;
    int pos = D_POS_DATASET_DP;

    if(!data)
    {
        pr_debug("datapool not available\n");
        *size = 0;
        return 0;
    }
    err = ecnr_datapool_access_get_config_data_size(data, size);

    if(err != ECNR_ERR_OK)
    {
        *size = 0;
        pos = D_POS_DATASET_DP;
    }

    pr_debug("data set size(%d), start position(%d) \n", *size, pos);

    return pos;
}


#ifdef D_DP_NOTIFICATION_SUPPORT
/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_reg_notification
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/

int ecnr_datapool_access_reg_notification(void* pFunc, enum eDataPool dp)
{
    int err = ECNR_ERR_OK;
    int i = 0;
    struct ecnr_dp_reg *ps = (struct ecnr_dp_reg*)NULL;


#ifndef D_ECNR_DATAPOOL_SUPPORT
    pr_debug("Datapool not supported");
    return ECNR_ERR_OK;
#endif

    if(!pFunc)
    {
        pr_debug("try to registry a callback function with NULL pointer");
        return ECNR_ERR_DP_ACCES_NULL_FCT_POINTER;
    }

    for (i = 0 ;i < NB_ELEM_MAX ; i++)
    {
        ps = &ecnr_dp_reg_table[i];

        if(ps->dp == ECNR_DP_NOT_DEF)
        {
            ps->ecnrdpFunc = (ecnrdpFuncCallback)pFunc;
            ps->active = TRUE;
            ps->dp = dp;

            pr_debug("registration of datapool element(%d) with callback function \n ", dp );

            break;

        }
        else if((ps->dp == dp) && ps->active)
        {
            pr_debug("registration of datapool element(%d) was already done\n ", dp );
            return  ECNR_ERR_DP_ACCES_DP_ALREADY_REGISTRIED;
        }
    }

    return err;
}
#endif //# D_DP_NOTIFICATION_SUPPORT

/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_compare_crc
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
gboolean ecnr_datapool_access_compare_crc(enum eDataPool dp, guint16 crc_in)
{
    gboolean        ret = FALSE;
    guint16         dataset_size = 0;
    int             version[4]={0};
    guint16         crc = 0;
    unsigned char*  data  = (unsigned char*)NULL;

    data = ecnr_datapool_access_get_bsd_dataset(dp, &dataset_size, version, &crc);

    if(data)
     {
        pr_debug("retrieve crc(0x%x) from dp(%d) to compare with crc_in(0x%x)\n", crc, dp, crc_in);

    if(crc == crc_in)
    {
        pr_debug("crc(0x%x) egal to crc_in(0x%x)\n", crc, crc_in);
        ret = TRUE;
    }
    else
        pr_debug("crc(0x%x) and crc_in(0x%x) are not matching\n", crc, crc_in);

     }



    return ret;
}

/*****************************************************************************/
/*                   Test function                                           */
/*****************************************************************************/


/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_trigger_new_datapool
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
void ecnr_datapool_access_trigger_new_datapool(enum eDataPool dp)
{

    pr_debug("entered");

    switch (dp)
    {
        case ECNR_DP_BT_PHONE_NB:
            ecnr_datapool_new_nb_phone();
            break;
        case ECNR_DP_BT_PHONE_WB:
            ecnr_datapool_new_wb_phone();
            break;
        case ECNR_DP_BT_VOICE_NB:
            ecnr_datapool_new_nb_voice();
            break;
        case ECNR_DP_BT_VOICE_WB:
            ecnr_datapool_new_wb_voice();
            break;
        case ECNR_DP_CP_PHONE_NB:
            ecnr_datapool_new_cp_nb_phone();
            break;
        case ECNR_DP_CP_PHONE_WB:
            ecnr_datapool_new_cp_wb_phone();
            break;
        case ECNR_DP_CP_VOICE_WB:
            ecnr_datapool_new_cp_wb_voice();
            break;
        default:
            pr_debug("no valid datapool");
            break;
    }

}

/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_print_datapool_infos
* PARAMETER:    none
* RETURNVALUE:  RETURNVALUE:  "0" on success  or "non-zero" value in case of error
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Patrick Rey
******************************************************************************/
void ecnr_datapool_access_print_datapool_infos(void)
{
    datapool_descriptor*    pdesc = (datapool_descriptor*)NULL;
    gboolean                loop = TRUE;
    guint                   i = 0;

    pr_debug("entered");

    while(loop  && (i < ECNR_NUM_DP_INFO) )
    {
        pdesc = &dp_info[i];

        if(   (pdesc->dp_class_name == NULL)
            && pdesc->dp_idx == (enum eDataPool)ECNR_DP_LAST)
        {
            loop = FALSE;
        }
        else
        {
            /* print datapool infos */
            pr_message ("datapool element(%d): dp_key(%d) , AccessClassName(%s)\n", i, pdesc->dp_idx, pdesc->dp_class_name);
        }
        i++;
    }

    return;
}

guint16 ecnr_datapool_access_calc_crc_ccitt ( const guint8 data[], guint16 length )
{
   boost::crc_ccitt_type crc_ccitt;
   crc_ccitt.process_bytes(data,length);
   return crc_ccitt.checksum();
}

//-----------------------------------------------------------------------------

gboolean ecnr_datapool_access_calc_and_compare_crc_ccitt ( const guint8 data[], guint16 length, guint16 crc )
{
   pr_debug(" ENTERED : data length -> %d, crc to compare -> %d \n", length, crc);
   tU16 crc_ccitt = ecnr_datapool_access_calc_crc_ccitt(data,length);
   pr_debug("calculated crc -> %d \n", crc_ccitt);

   return ( crc_ccitt == crc ) ? true : false;
}

size_t ecnr_datapool_access_export_dp_to_file(const guint8* data, guint16 length, const char* exportfile)
{
    FILE* fhdl = NULL;
    size_t wdata;

    if(!exportfile || !data || !length)
        return 0;

    if( remove(exportfile) != 0 )
    	pr_debug( "error deleting file\n");

    fhdl = fopen(exportfile, "w");
    if(!fhdl)
    {
        pr_warning("fails to open file to export the datapool\n");
        return 0;
    }

    wdata = fwrite((void*)data, (size_t)1, length, fhdl);
    if(wdata)
    {
        pr_debug("export %lu bytes from the datapool in the file -> %s \n", wdata,exportfile);
    }

    fclose(fhdl);

    return wdata;
}

/*****************************************************************************
* FUNCTION:    ecnr_datapool_access_reload_dp_element
* PARAMETER:   dpkey
* RETURNVALUE: dp reload result "1" on success and "0" on failure
* TEST CASE:
* DESCRIPTION:
* HISTORY:    Created Ranjit Susal
******************************************************************************/
guint32 ecnr_datapool_access_reload_dp_element(enum eDataPool dpkey)
{
    guint32 u32result  = 0;

    pr_debug("Entered");
    ETG_TRACE_USR3(("[ecnr_datapool_access_reload_dp_element]: with dp ID [%d]", dpkey));

#ifdef D_SUPPORT_DP_RELOAD
    switch(dpkey)
    {
        case ECNR_DP_BT_PHONE_NB:
        {
            dp_tclECNR_BT_PHONE_NB dpelem_BT_PHONE_NB;
            u32result = dpelem_BT_PHONE_NB.bReloadDpElement();
            break;
        }
        case ECNR_DP_BT_PHONE_WB:
        {
            dp_tclECNR_BT_PHONE_WB dpelem_BT_PHONE_WB;
            u32result = dpelem_BT_PHONE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_INT_VOICE_WB:
        {
            dp_tclECNR_INT_VOICE_WB dpelem_INT_VOICE_WB;
            u32result = dpelem_INT_VOICE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_WUW_VOICE_WB:
        {
            dp_tclECNR_WUW_VOICE_WB dpelem_WUW_VOICE_WB;
            u32result = dpelem_WUW_VOICE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_CP_PHONE_NB:
        {
            dp_tclECNR_CP_PHONE_NB dpelem_CP_PHONE_NB;
            u32result = dpelem_CP_PHONE_NB.bReloadDpElement();
            break;
        }
        case ECNR_DP_CP_PHONE_WB:
        {
            dp_tclECNR_CP_PHONE_WB dpelem_CP_PHONE_WB;
            u32result = dpelem_CP_PHONE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_CP_FTIME_SWB:
        {
            dp_tclECNR_CP_FTIME_SWB dpelem_CP_FTIME_SWB;
            u32result = dpelem_CP_FTIME_SWB.bReloadDpElement();
            break;
        }
        case ECNR_DP_BT_VOICE_NB:
        {
            dp_tclECNR_BT_VOICE_NB dpelem_BT_VOICE_NB;
            u32result = dpelem_BT_VOICE_NB.bReloadDpElement();
            break;
        }
        case ECNR_DP_BT_VOICE_WB:
        {
            dp_tclECNR_BT_VOICE_WB dpelem_BT_VOICE_WB;
            u32result = dpelem_BT_VOICE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_BT_SIRI_WB:
        {
            dp_tclECNR_BT_SIRI_NB dpelem_BT_SIRI_NB;
            u32result = dpelem_BT_SIRI_NB.bReloadDpElement();
            break;
        }
        case ECNR_DP_CP_VOICE_WB:
        {
            dp_tclECNR_CP_VOICE_WB dpelem_CP_VOICE_WB;
            u32result = dpelem_CP_VOICE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_AA_VOICE_WB:
        {
            dp_tclECNR_AA_VOICE_WB dpelem_AA_VOICE_WB;
            u32result = dpelem_AA_VOICE_WB.bReloadDpElement();
            break;
        }
        case ECNR_DP_VPA_VOICE_WB:
        {
            dp_tclECNR_VPA_VOICE_WB dpelem_VPA_VOICE_WB;
            u32result = dpelem_VPA_VOICE_WB.bReloadDpElement();
            break;
        }
        default:
        {
            pr_message("Invalid DP key value\n");
            ETG_TRACE_USR3(("[ecnr_datapool_access_reload_dp_element]: Invalid DP key value"));
            break;
        }
    }
#endif // D_SUPPORT_DP_RELOAD

       return u32result;
}

#ifdef __cplusplus
}
#endif


#endif // #ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_GMG3_LINUX
