/* =======================================================================================
*  Project          SSE (Speech Signal Enhancement)
*  (c) Copyright    2004-2013
*  Company          Nuance Communications
*                   All rights reserved
*  Secrecy Level    STRICTLY CONFIDENTIAL
* --------------------------------------------------------------------------------------*/
/**
*  @internal
*  @file            sse.h
*
*  Prototypes of the SSE API functions.
*
*  This header file contains all prototypes of the API functions of the SSE module.
*/
/*======================================================================================*/
/** @addtogroup DEF
*  @{*/
#ifndef __SSE_H__
#define __SSE_H__

#ifdef __cplusplus
extern "C" {
#endif



   /*--------------------------------------------------------------------------------------*/
   /*     Constant definitions                                                             */
   /*--------------------------------------------------------------------------------------*/

#define SSE_MAX_MEM_BUF_CNT 2   /**< Maximum number of memory buffers provided to sseCreate */

#define SSE_MAX_AUDIO_CNT   4   /**< Maximum number of audio channels    */

#define SSE_MAX_CONFIG_DATA_CNT 3     /**< Maximum number of config data buffers        */

#define SSE_MAX_STRING_SIZE   200     /**< Maximum length of a string in the SSE module */

#define SSE_PARAM_EQ_LEN       15     /**< (5 nodes * 3 values per node: center(Hz), width(Hz), gain(cdB)) */
#define SSE_PARAM_SENDEQ_LEN   15     /**< Number of send parametric EQ values (5 nodes * 3 values per node)                */
#define SSE_PARAM_RECVEQ_LEN   15     /**< Number of receive parametric EQ values (5 nodes * 3 values per node)                */
#define SSE_INPUTFILTER_HDR_LEN 3     /**< Header size + Filter data size + number of filter stages used */

   /* Signal types for generated diagnostic signals */
#define SSE_DIAG_WHITE_NOISE            0    /**< white noise */
#define SSE_DIAG_WHITE_BAND_NOISE       1    /**< pseudo white band-limited noise */
#define SSE_DIAG_PINK_NOISE             2    /**< pseudo pink noise */
#define SSE_DIAG_SINE                   3    /**< sine wave */
#define SSE_DIAG_SINE_AMP_SWEEP         4    /**< sine wave with amplitude sweep */
#define SSE_DIAG_SINE_FREQ_SWEEP        5    /**< sine wave with frequency sweep */
#define SSE_DIAG_SQUARE                 6    /**< square wave */
#define SSE_DIAG_ITU_NO_ECHO            7    /**< ITU */


   /** Default value for configuration values in tsseConfig and BSD SSE_MainConfig.
   All configuration enums/parameters are non-zero based.  A value of sseDefault (0)
   indicates 'use the default value' */
#define sseDefault 0

   /** @name Input Masks
   Masks used by RecvInputMask, DriverInputMask, CoDriverInputMask, etc
   to specify the inputs in use.  Masks can be 'OR'ed together.  An odd value is
   invalid for DriverInputMask/CoDriverBeamMask; an odd value > 1 is
   always invalid.
   * @{*/
#define sseNoInput 0x0001U    /**< No inputs enabled - cannot be used with other masks */
#define sseInput0  0x0002U    /**< Input 0 enabled */
#define sseInput1  0x0004U    /**< Input 1 enabled */
#define sseInput2  0x0008U    /**< Input 2 enabled */
#define sseInput3  0x0010U    /**< Input 3 enabled */
#define sseInput4  0x0020U    /**< Input 4 enabled */
#define sseInput5  0x0040U    /**< Input 5 enabled */
#define sseInput6  0x0080U    /**< Input 6 enabled */
#define sseInput7  0x0100U    /**< Input 7 enabled */
   /**@}*/

/*--------------------------------------------------------------------------------------*/
/*     Define module identification labels                                              */
/*--------------------------------------------------------------------------------------*/

/** @name Module Labels
   Module labels used by the sseRegisterCallback() and sseDeRegisterCallback() functions
   as well as the sseSetData() and sseGetData() functions.
* @{*/
#define SSE_ML_SSE      0x0400   /**<  Main SSE                          */
#define SSE_ML_ANA_MIC  0x0800   /**<  PCM input 1 analysis              */
#define SSE_ML_ANA_REF  0x0C00   /**<  PCM input 2 analysis              */
#define SSE_ML_SYN_MIC  0x1000   /**<  PCM output 1 synthesis            */
#define SSE_ML_RECV     0x1400   /**<  Receive side processing           */
#define SSE_ML_AEC      0x1800   /**<  Acoustic echo canceller           */
#define SSE_ML_FBF_DR   0x1C00   /**<  Fixed beamformer (driver)         */
#define SSE_ML_FBF_CO   0x2000   /**<  Fixed beamformer (co-driver)      */
#define SSE_ML_ABF_DR   0x2400   /**<  Adaptive beamformer (driver)      */
#define SSE_ML_ABF_CO   0x2800   /**<  Adaptive beamformer (co-driver)   */
#define SSE_ML_MIX      0x2C00   /**<  Mixer module                      */
#define SSE_ML_NR       0x3000   /**<  Noise reduction and residual acoustic echo suppression         */
   /** @cond NOT_IMPLEMENTED */
#define SSE_ML_MICN     0x3400   /**<  Microphone normalization          */
#define SSE_ML_LEC      0x3800   /**<  Line echo canceller               */
   /** @endcond */
#define SSE_ML_PEST     0x3C00   /**<  Pitch estimation                  */
#define SSE_ML_POW      0x4400   /**<  Power estimation                  */
#define SSE_ML_AGC      0x4800   /**<  Automatic gain control            */
   /** @cond OBSOLETE */
#define SSE_ML_SNR      0x5000   /**<  SNR estimates/vectors             */
#define SSE_ML_RECF     0x5400   /**<  no longer used                    */
   /** @endcond */
#define SSE_ML_NOISE    0x5800   /**<  Noise estimation                  */
#define SSE_ML_SPEC     0x5C00   /**<  Spectrum information              */
#define SSE_ML_HFEF     0x6000   /**<  HFE (frequency-domain)            */
   /** @cond OBSOLETE */
#define SSE_ML_HFET     0x6400   /**<  HFE (time-domain)                 */
   /** @endcond */
#define SSE_ML_DIAG     0x6800   /**<  Diagnostics                       */
#define SSE_ML_RAGC     0x7000   /**<  Receive-side agc                  */
#define SSE_ML_RBWE     0x7400   /**<  Receive-side bwe                  */
#define SSE_ML_RAEQ     0x7800   /**<  Receive-side adaptive equalization */
#define SSE_ML_TINF_RCV 0x7C00   /**<  Time-series info/voice detect for receive path */
#define SSE_ML_TINF_REF 0x8000   /**<  Time-series info/voice detect for reference path */
#define SSE_ML_RDTF     0x8400   /**<  Receive-side double talk filter   */
#define SSE_ML_RENS     0x8800   /**<  Receive-side electrical noise suppression */

#define SSE_ML_SCINFO   0x8C00   /**<  Spatial control information       */
#define SSE_ML_RLE      0x9000   /**<  Reference latency estimaton       */
#define SSE_ML_IC_DR    0x9400   /**<  Interference Canceller (driver)   */
/**@}*/

/*--------------------------------------------------------------------------------------*/
/*      Define sseSet/GetData identification labels/strings                             */
/*--------------------------------------------------------------------------------------*/

/* Labels used by the sseSetData() and sseGetData() functions to access extended data
   (String versions used by sseSetDataString() and sseGetDataString())

   RO - Read only
   WB - Data can only be written before initialization, read anytime
   WA - Data can only be written after initialization, read anytime
   RW - Data can be written, read anytime
   WO - Data can only be written after initialization, never read */

/** Data identification arguments used with sseSetData() and sseGetData().
    These values are stringified when used as the label string for sseSetDataString() and sseGetDataString().
    @sa @ref Parameters for a complete description of the available tsseDataID parameters. */
#define tsseDataID            int

#define sse_MicInCnt          3     /* WB count of microphone input channels               (StarRec_UInt16)*/
#define sse_RecvInCnt         4     /* WB count of receive signal input channels           (StarRec_UInt16)*/
#define sse_RefInCnt          5     /* WB count of reference signal input channels         (StarRec_UInt16)*/
#define sse_MicOutCnt         6     /* RO count of microphone output channels              (StarRec_UInt16)*/
#define sse_RecvOutCnt        7     /* RO count of receive signal output channels          (StarRec_UInt16)*/
#define sse_OpMode            8     /* RW operation mode (hands-free, voice recog., ...    (StarRec_Int32) */
#define sse_NRSwitch          9     /* RW on/off switch for noise reduction                (StarRec_Int32) */
#define sse_AECSwitch         10    /* RW on/off switch for acoustic echo cancellation     (StarRec_Int32) */
#define sse_RAESwitch         12    /* RW on/off residual acoustic echo suppression        (StarRec_Int32) */
#define sse_SpPfSwitch        13    /* RW on/off Spatial Postfilter                        (StarRec_Int32) */

#define sse_BFSwitchDR        14    /* WB on/off switch of driver beamformer               (StarRec_Int32) */
#define sse_BFModeDR          15    /* WB type of driver beamformer                        (StarRec_Int32) */
#define sse_BFCalSwitchDR     16    /* WB on/off switch for driver self-calibration        (StarRec_Int32) */
#define sse_BFCalModeDR       17    /* WB driver self-calibration mode                     (StarRec_Int32) */
#define sse_BFSwitchCD        18    /* WB on/off switch of co-driver beamformer            (StarRec_Int32) */
#define sse_BFModeCD          19    /* WB type of co-driver beamformer                     (StarRec_Int32) */
#define sse_BFCalSwitchCD     20    /* WB on/off switch for co-driver self-calibration     (StarRec_Int32) */
#define sse_BFCalModeCD       21    /* WB co-driver self-calibration mode                  (StarRec_Int32) */
#define sse_MIXSwitch         22    /* WB on/off switch for mixer                          (StarRec_Int32) */
#define sse_WNDBSwitch        24    /* RW on/off switch for windbuffet removal             (StarRec_Int32) */
#define sse_SendAGCSwitch     25    /* RW on/off switch for sending direction AGC          (StarRec_Int32) */
#define sse_SpeakerLocMode    26    /* WB on/off switch speaker localization               (StarRec_Int32) */
#define sse_MicCovDetSwitch   27    /* WB on/off switch microphone cover detection         (StarRec_Int32) */

#define sse_FiltInSwitch      28    /* RW on/off switch for input filter                   (StarRec_Int32) */
#define sse_FiltOutSwitch     29    /* RW on/off switch for output filter                  (StarRec_Int32) */
#define sse_FrameShift        30    /* WB input frame shift                                (StarRec_UInt16)*/
#define sse_SampleRate        31    /* WB input sample rate                                (StarRec_UInt16)*/
#define sse_MicInClipStats    32    /* RO microphone clipping statistics                   (tsseClipStats) */
#define sse_RefInClipStats    33    /* RO reference clipping statistics                    (tsseClipStats) */
#define sse_VersionNumber     34    /* RO SSE Version number                               (StarRec_Int32[4])*/
#define sse_BSDInfoStr        35    /* RO BSD header string                                (StarRec_Char[max 64])*/
#define sse_BSDCnt            36    /* RO count of loaded BSD files                        (StarRec_UInt16)*/
#define sse_RECVLimitSwitch   37    /* RW on/off switch for receive limiter                (StarRec_Int32) */
#define sse_RECVFiltSwitch    38    /* RW on/off switch for receive output filtering       (StarRec_Int32) */
#define sse_RECVAgcSwitch     39    /* RW on/off switch for receive agc                    (StarRec_Int32) */
#define sse_MIXMode           40    /* WB mixer mode                                       (StarRec_Int32) */
#define sse_VersionComment    41    /* RO SSE version comment string                       (StarRec_Char[max 128])*/
#define sse_WNDBMode          43    /* WB Wind buffet suppression mode                     (StarRec_Int32) */
#define sse_SendMuteSwitch    52    /* RW send mute switch                                 (StarRec_Int32) */
#define sse_RecvMuteSwitch    53    /* RW receive mute switch                              (StarRec_Int32) */
#define sse_HFESwitch         57    /* RW on/off switch for high frequency encoding        (StarRec_Int32) */
#define sse_PostConfig        59    /* WO post configuration data                          (StarRec_Byte[])*/
#define sse_RECVBweSwitch     60    /* RW on/off switch for receive bandwidth extension    (StarRec_Int32) */
#define sse_AECFiltLen        61    /* WB AEC filter length                                (StarRec_UInt16)*/
#define sse_PersistSize       63    /* RO persistent memory size                           (StarRec_UInt32)*/
#define sse_PassThroughSwitch 65    /* WA on/off switch for pass-through mode              (StarRec_Int32) */
#define sse_MaxFreqProc       66    /* WB maximum frequency processed                      (StarRec_UInt16)*/
#define sse_SendEQSwitch      67    /* RW on/off switch for send-side equalization         (StarRec_Int32) */
#define sse_AlignedPersistSize  69  /* RO persistent memory size for sseCreateExtMem()     (StarRec_UInt32)*/
#define sse_MemStatus         70    /* RO Memory status: available/used                    (StarRec_Int32[4])*/
#define sse_DebugIOBufferSizes 71   /* RO Sizes of the debug input/output buffers          (StarRec_Int32[4])*/
#define sse_DIAGSwitch        74    /* RW on/off switch for diagnostics                    (StarRec_Int32) */
#define sse_RECVAdaptEQSwitch 75    /* RW on/off switch for receive-side equalization      (StarRec_Int32) */
#define sse_RECVAdaptEQMode   76    /* RW receive-side equalization mode                   (StarRec_Int32) */
#define sse_RecvInClipStats   82    /* RO receive clipping statistics                      (tsseClipStats) */
#define sse_SendDynEQSwitch   83    /* RW on/off switch for dynamic send-side equalization (StarRec_Int32) */
#define sse_RECVDtfSwitch     85    /* RW on/off switch for receive-side double talk filter(StarRec_Int32) */
#define sse_RECVEnsSwitch     86    /* RW on/off switch for receive-side electrical noise suppression(StarRec_Int32) */
#define sse_ClipStatsMode     87    /* WA  clipping statistics mode      (StarRec_Int16) */
#define sse_PassThroughMode   88
#define sse_DebugMsg          89    /* WO Debug message                       (StarRec_Byte, variable length) */
#define sse_DebugMsgQueueSize 90    /* RO Debug message queue size            (StarRec_Byte, variable length) (obsolete)*/
#define sse_AudioTypeForASR   91    /* RO SSE Audiotype struct needed by the VoCon ASR                        */
#define sse_ASRMode           92    /* WB ASR mode                         */
#define sse_SaveCPUMode       93    /* RW Save CPU mode                         */
#define sse_PlaySignalSetupMicIn    94 /* WA Setup signal injection for microphone input    (StarRec_Int32[8]) */
#define sse_PlaySignalSetupMicOut   95 /* WA Setup signal injection for microphone output   (StarRec_Int32[8]) */
#define sse_PlaySignalSetupRecvIn   96 /* WA Setup signal injection for receive input       (StarRec_Int32[8]) */
#define sse_PlaySignalSetupRecvOut  97 /* WA Setup signal injection for receive output      (StarRec_Int32[8]) */
#define sse_PlaySignalSetupRefIn    98 /* WA Setup signal injection for reference input     (StarRec_Int32[8]) */
#define sse_PlaySignalWriteMicIn    99 /* WO Write signal injection for microphone input    (StarRec_Int32[8]) */
#define sse_PlaySignalWriteMicOut  100 /* WO Write signal injection for microphone output   (StarRec_Int32[])  */
#define sse_PlaySignalWriteRecvIn  101 /* WO Write signal injection for receive input       (StarRec_Int32[])  */
#define sse_PlaySignalWriteRecvOut 102 /* WO Write signal injection for receive output      (StarRec_Int32[])  */
#define sse_PlaySignalWriteRefIn   103 /* WO Write signal injection for reference input     (StarRec_Int32[])  */
#define sse_PlaySignalPlayState    104 /* WA Signal injection play state                    (StarRec_Int32)    */
#define sse_ASRInvalidFrameCnt     105 /* RO Number of invalid frames from SSE to the ASR   (StarRec_Int16)    */
#define sse_RefLatencyEstSwitch    106 /* RW on/off switch for reference latency estimation (StarRec_Int32)    */
#define sse_RoadBumpSupSwitch      107 /* RW on/off switch for road bump suppression        (StarRec_Int32)    */
#define sse_PhonemeSharpeningSwitch 109 /* RW on/off switch for phoneme sharpening          (StarRec_Int32)    */
#define sse_BFBypassEnable         110 /* WA True/False flag for enabling/disabling BF bypass and switch to single-channel (StarRec_Int16) */
#define sse_PicSwitch              112 /* WB on/off switch of passenger interference cancellation (PIC)  (StarRec_Int32) */
#define sse_BroadBeamSwitch        113 /* WB on/off switch of broad beam feature            (StarRec_Int32) */
#define sse_SelfSteeringBeamSwitch 114 /* WB on/off switch of self-steering beam            (StarRec_Int32) */

#define sse_AECEchoAlign           (SSE_ML_AEC | 0)     /* RO estimated echo-alignment, in ms   (StarRec_Int16) */
#define sse_AECInitEchoEst         (SSE_ML_AEC | 2)     /* WA initial echo estimate             (StarRec_Int16) */
#define sse_AECNlProcActivate      (SSE_ML_AEC | 4)     /* Switch nonlinearity                  (StarRec_Int32) */

#define sse_MicClipThreshold     (SSE_ML_ANA_MIC | 1) /* WA mic clipping threshold            (StarRec_Int16) */
#define sse_MicInRMS             (SSE_ML_ANA_MIC | 2) /* RO mic input RMS                     (StarRec_Int16) */
#define sse_MicLatency           (SSE_ML_ANA_MIC | 4) /* RW ms microphone latency             (StarRec_Int32) */
#define sse_MicMaxLatency        (SSE_ML_ANA_MIC | 5) /* WB ms maximum microphone latency     (StarRec_Int32) */
#define sse_MicInputFilterCh     (SSE_ML_ANA_MIC | 6) /* RW Channel-specific input filter parameters       (StarRec_Int16[]) */
#define sse_MicLatencyCh         (SSE_ML_ANA_MIC | 7) /* WA Channel-specific microphone delay (microsec)   (StarRec_Int16) */
#define sse_MicGainCh            (SSE_ML_ANA_MIC | 8) /* WA Channel-specific microphone gain  (cdB)        (StarRec_Int16) */

#define sse_RefClipThreshold     (SSE_ML_ANA_REF | 1) /* WA ref clipping threshold            (StarRec_Int16) */
#define sse_RefLatency           (SSE_ML_ANA_REF | 2) /* RW ms reference latency              (StarRec_Int32) */
#define sse_RefMaxLatency        (SSE_ML_ANA_REF | 3) /* WB ms maximum reference latency      (StarRec_Int32) */
#define sse_RefInRMS             (SSE_ML_ANA_REF | 4) /* RO ref input RMS                     (StarRec_Int16) */

#define sse_RecvPreGain          (SSE_ML_RECV |  0)   /* WA recv pre gain in cdB              (StarRec_Int32) */
#define sse_RecvInRMS            (SSE_ML_RECV |  3)   /* RO recv input RMS                    (StarRec_Int16) */
#define sse_RecvOutRMS           (SSE_ML_RECV |  4)   /* RO recv output RMS                   (StarRec_Int16) */
#define sse_RecvLimitThresh      (SSE_ML_RECV |  5)   /* WA recv limiter threshold            (StarRec_UInt16)*/
#define sse_RecvParamEQNodes     (SSE_ML_RECV |  6)   /* WA recv parametric EQ nodes          (StarRec_Int16[SSE_PARAM_RECVEQ_LEN])*/
#define sse_RecvEQCoeff          (SSE_ML_RECV |  7)   /* RO recv static EQ coefficients       (StarRec_Int16[])*/
#define sse_RecvPostGain         (SSE_ML_RECV |  8)   /* WA recv post gain in cdB             (StarRec_Int16) */
#define sse_RecvClipThreshold    (SSE_ML_RECV | 10)   /* WA recv clipping threshold           (StarRec_Int16) */
#define sse_RecvAEQBypass        (SSE_ML_RAEQ | 11)   /* WA Bypass receive-eq                 (StarRec_UInt16) */
#define sse_RecvPostGainDelta    (SSE_ML_RECV | 12)   /* WA recv post gain delta in dB (abs)  (StarRec_Byte)  */
#define sse_RecvOutClipStats     (SSE_ML_RECV | 13)   /* RO recv out clipping statistics      (tsseClipStats) */
#define sse_RecvPreGainDelta     (SSE_ML_RECV | 14)   /* WA recv pre gain delta in dB (abs)   (StarRec_Byte)  */

#define sse_RecvAGCGainCur       (SSE_ML_RAGC |  0)         /* WA recv current agc level            (StarRec_Int16) */
#define sse_RecvAGCGainVals      (SSE_ML_RAGC |  1)         /* WA recv agc min, start and max gain  (StarRec_Int16[3])*/
#define sse_RecvAGCAmpTgts       (SSE_ML_RAGC |  2)         /* WA recv agc amp target output range  (StarRec_Int16[2])*/
#define sse_RecvAGCMaxRates      (SSE_ML_RAGC |  3)         /* WA recv agc rise and fall rates      (StarRec_Int16[2])*/
#define sse_RecvAGCRisingPeakDelay     (SSE_ML_RAGC | 6)    /* WA delay for rising peak estimate in ms (StarRec_Int16) */
#define sse_RecvAGCFallingPeakDelay    (SSE_ML_RAGC | 7)    /* WA delay for falling peak estimate in ms (StarRec_Int16) */
#define sse_RecvAGCGainInc             (SSE_ML_RAGC | 10)   /* WA gain increasing in cdB/s (StarRec_Int16) */
#define sse_RecvAGCGainDec             (SSE_ML_RAGC | 11)   /* WA gain decreasing in cdB/s (StarRec_Int16) */

#define sse_RecvAEQEnrDes        (SSE_ML_RAEQ |  0)   /* WA Desired broadband ENR             (StarRec_Int16) */
#define sse_RecvAEQGainMax       (SSE_ML_RAEQ |  1)   /* WA Maximum allowed bradband gain     (StarRec_Int16) */
#define sse_RecvAeqGainAdapSpeed (SSE_ML_RAEQ |  2)   /* WA Adaption speed of the gain factor (CDBs) (StarRec_Int16) */
#define sse_RecvAEQEnrShapeStarts (SSE_ML_RAEQ | 3)   /* WA ENR below which shape is active   (StarRec_Int16) */
#define sse_RecvAEQEnrShapeIsMax (SSE_ML_RAEQ |  4)   /* WA ENR as of which shape is maximal  (StarRec_Int16) */
#define sse_RecvAEQShapeMax      (SSE_ML_RAEQ |  5)   /* WA Max of the des. freq. sel. ENR    (StarRec_Int16) */
#define sse_RecvAeqShapeAdapSpeed (SSE_ML_RAEQ |  6)  /* WA Adaption speed of the shape factor (CDBs) (StarRec_Int16) */
#define sse_RecvAEQEnrDesF       (SSE_ML_RAEQ |  30)  /* Unused dummy ID needed for backward compatibility of interface */

#define sse_RecvBWEBrightness    (SSE_ML_RBWE |  0)   /* WA recv bwe brightness               (StarRec_Int16) */
#define sse_RecvBWEWarmth        (SSE_ML_RBWE |  1)   /* WA recv bwe warmth                   (StarRec_Int16) */

#define sse_RecvDTFMaxAtten      (SSE_ML_RDTF |  0)   /* WA recv dtf maximum atten. in cdB    (StarRec_Int16) */
#define sse_RecvDTFMaxEchoRatio  (SSE_ML_RDTF |  1)   /* WA recv dtf max. echo ratio in cdB   (StarRec_Int16) */
#define sse_RecvDTFResponse      (SSE_ML_RDTF |  2)   /* WA recv dtf responsiveness           (StarRec_Int16) */

#define sse_RecvENSSuppress      (SSE_ML_RENS |  0)   /* WA recv ens threshold                (StarRec_Int16) */
#define sse_RecvENSActivityRatio (SSE_ML_RENS |  1)   /* WA recv ens speech activity ratio    (StarRec_Int16) */
#define sse_SendLimitThresh      (SSE_ML_SYN_MIC | 3) /* WA limiter threshold                 (StarRec_UInt16)*/
#define sse_MicOutRMS            (SSE_ML_SYN_MIC | 6) /* RO mic output RMS                    (StarRec_Int16) */
#define sse_SendParamEQNodes     (SSE_ML_SYN_MIC | 7) /* WA send parametric EQ nodes          (StarRec_Int16[SSE_PARAM_SENDEQ_LEN])*/
#define sse_SendGain             (SSE_ML_SYN_MIC | 9) /* WA fixed gain in cdB                 (StarRec_Int16) */
#define sse_MicOutClipStats      (SSE_ML_SYN_MIC |11) /* RO mic out clipping statistics       (tsseClipStats) */
#define sse_SendGainDelta        (SSE_ML_SYN_MIC |12) /* WA Send gain delta in dB (abs)      (StarRec_Byte)   */
#define sse_SendParamEQ2Nodes    (SSE_ML_SYN_MIC |14) /* WA send parametric EQ nodes, low SNR (StarRec_Int16[SSE_PARAM_SENDEQ_LEN])*/
#define sse_SendDynEQMinMaxSNR   (SSE_ML_SYN_MIC |17) /* WA send dynamic EQ SNR thresholds    (StarRec_Int16[2]) */
#define sse_SendDynEQSNRMaxAdaptCDB (SSE_ML_SYN_MIC |18) /* WA send dynamic EQ max downwards SBR adaptation (StarRec_Int16[2]) */

#define sse_RLEPowerSavingsMode    (SSE_ML_RLE | 0)    /* WA Delay estimation cpu power savings mode        */
#define sse_RLEAutoCompensate      (SSE_ML_RLE | 1)    /* WA Automatically compensate for referency latency */
#define sse_RLEOneTimeEstimation   (SSE_ML_RLE | 2)    /* WA Terminate After Successful Estimation          */
#define sse_RLEEstimatedLatency    (SSE_ML_RLE | 3)    /* RO Estimated latency in ms + confidence           */
#define sse_RLEConfidenceLevel     (SSE_ML_RLE | 4)    /* WA Confidence Level above which the estimate is accepted */

#define sse_NRFloor                (SSE_ML_NR | 0)     /* WA target NR floor in cdB               (StarRec_Int16)   */
#define sse_RAESuppress            (SSE_ML_NR | 1)     /* WA suppression of residual echo         (StarRec_Int16)   */
#define sse_RAEParams              (SSE_ML_NR | 2)     /* WA residual echo parameters             (StarRec_Int16[3])*/
#define sse_NRCombFloor            (SSE_ML_NR | 5)     /* WA tuned combined floor (cdB)           (StarRec_Int16)   */
#define sse_SpsActivateOffAxisSup  (SSE_ML_NR | 12)    /* WA On/Off Off-Axis-Suppression          (StarRec_UInt16)  */
#define sse_SpsOffAxisSupMax       (SSE_ML_NR | 13)    /* WA Maximum additional Attenuation       (StarRec_Int16)   */
#define sse_SpsOffAxisSupAggress   (SSE_ML_NR | 14)    /* WA Aggressivity Off-Axis Suppression    (StarRec_UInt16)  */
#define sse_SpsIncreaseDirectivity (SSE_ML_NR | 15)    /* WA Max add. Atten. (No longer used)     (StarRec_Int16)   */
#define sse_NRFormantBoostActivate (SSE_ML_NR | 16)    /* WA On/Off formant boosting              (StarRec_Int16)   */
#define sse_NRFormantMaxGain       (SSE_ML_NR | 17)    /* WA On/Off formant boosting              (StarRec_Int16)   */
#define sse_NRFormantINRThresh     (SSE_ML_NR | 18)    /* WA INR treshold for formant boosting    (StarRec_Int16)   */
#define sse_NRFreqDepSmFactor      (SSE_ML_NR | 19)    /* WA Smoothing factor for formants gains  (StarRec_Int16)   */
#define sse_NRMaxNumFormants       (SSE_ML_NR | 20)    /* WA Max number of formants               (StarRec_Int16)   */
#define sse_SpsWndbAggressivity    (SSE_ML_NR | 21)    /* WA Wind noise suppr. aggressivity       (StarRec_Int16)   */
#define sse_NRFloorFreq            (SSE_ML_NR | 22)    /* WA spectral floor freq. range (Hz)      (StarRec_UInt16)[3] */
#define sse_NRGainFricEnh          (SSE_ML_NR | 23)    /* WA Gain for boosting fricatives         (StarRec_Int16)   */
#define sse_NRFricInrRatioThresh   (SSE_ML_NR | 24)    /* WA Mean INR ratio thresholdin cdB       (StarRec_Int16)   */
#define sse_NRComfortNoiseAggr     (SSE_ML_NR | 25)    /* WA Comfort noise insertion level        (StarRec_Int16)   */
#define sse_NRFloorFluctExtraSuppr (SSE_ML_NR | 26)    /* WA Extra suppression of RES limitation  (StarRec_Int16)   */
#define sse_RAESuppressAdd         (SSE_ML_NR | 27)    /* WA suppr. of res. echo (additive aggr.) (StarRec_Int16)   */
#define sse_NRFloorSNR             (SSE_ML_NR | 33)     /* WA tuned SNR floor                     (StarRec_Int16[2])*/
#define sse_SpsSpfOverallAggr      (SSE_ML_NR | 37)    /* WA Overall Aggressivity Spatial PF      (StarRec_Int16)   */

#define sse_SpsRbsAggressivity     (SSE_ML_NR | 38)    /* WA RBS aggressivity in lower frequencies (StarRec_UInt16) */
#define sse_SpsRbsSpeechProtect    (SSE_ML_NR | 39)    /* WA RBS speech protection level (StarRec_UInt16) */
#define sse_RbsActivationSNR       (SSE_ML_NR | 54)    /* WA RBS SNR at which RBS is activated (StarRec_UInt16) */
#define sse_SpsTonalNoiseSupprLevel    (SSE_ML_NR | 58)     /* WA Tonal noise suppression level (StarRec_UInt16) */
#define sse_SpsNoiseShapingLevel       (SSE_ML_NR | 59)     /* WA Noise shaping level (StarRec_UInt16) */
#define sse_SpsSpeechProtectLevel      (SSE_ML_NR | 60)     /* WA Speech protection level (StarRec_UInt16) */
#define sse_SpsFrequencyContext        (SSE_ML_NR | 61)     /* WA Frequency context (StarRec_UInt16) */
#define sse_SpsDynamicNoiseOverestCDB  (SSE_ML_NR | 62)     /* WA Dynamic noise overestimation factor in CDB (StarRec_UInt16) */
#define sse_SpsFloorSNRCtrl            (SSE_ML_NR | 63)     /* WA SNR-controled spectral floor (StarRec_Int16[4])*/

#define sse_SendAGCGainCur             (SSE_ML_AGC | 0)     /* WA current cdB gain                  (StarRec_Int16) */
#define sse_SendAGCGainVals            (SSE_ML_AGC | 1)     /* WA min, start and max cdB gain       (StarRec_Int16[3])*/
#define sse_SendAGCAmpTgts             (SSE_ML_AGC | 2)     /* WA amp target cdB output range       (StarRec_Int16[2])*/
#define sse_SendAGCMaxRates            (SSE_ML_AGC | 3)     /* WA max cdB/s rise and fall rates     (StarRec_Int16[2])*/
#define sse_SendAGCRisingPeakDelay     (SSE_ML_AGC | 6)     /* WA delay for rising peak estimate in ms (StarRec_Int16) */
#define sse_SendAGCFallingPeakDelay    (SSE_ML_AGC | 7)     /* WA delay for falling peak estimate in ms (StarRec_Int16) */
#define sse_SendAGCGainInc             (SSE_ML_AGC | 10)     /* WA gain increasing in cdB/s (StarRec_Int16) */
#define sse_SendAGCGainDec             (SSE_ML_AGC | 11)    /* WA gain decreasing in cdB/s (StarRec_Int16) */

#define sse_DIAGActive           (SSE_ML_DIAG | 0)    /* RO diagnostic active flag            (StarRec_UInt16)*/
#define sse_DIAGStop             (SSE_ML_DIAG | 1)    /* WO stop active diagnostic            (StarRec_UInt16)*/
#define sse_DIAGPlaySignal       (SSE_ML_DIAG | 2)    /* WO play generated signal             (StarRec_Int32[6])*/
#define sse_DIAGLatencyTest      (SSE_ML_DIAG | 3)    /* WO start latency measurement         (StarRec_Int16[2])*/
#define sse_DIAGLatencyEst       (SSE_ML_DIAG | 4)    /* RO latency estimate                  (StarRec_Int16[3])*/
#define sse_DIAGLatencyTestRef   (SSE_ML_DIAG | 5)    /* WO start latency meas. for ref-in    (StarRec_Int16[2])*/

#define sse_MsSinceActiveRef    (SSE_ML_TINF_REF | 0) /* WA threshold in ms to detect ref inactivity (StarRec_UInt16)*/

#define sse_FBFSteeringAngleDR  (SSE_ML_FBF_DR | 0)   /* WA Azimut of the speaker position, driver BF    (StarRec_Int16) */
#define sse_FBFMicDistanceDR    (SSE_ML_FBF_DR | 1)   /* WB Microphone distance (in mm), driver BF    (StarRec_Int16) */

#define sse_FBFSteeringAngleCD  (SSE_ML_FBF_CO | 0)   /* WA Azimut of the speaker position, passenger BF    (StarRec_Int16) */
#define sse_FBFMicDistanceCD    (SSE_ML_FBF_CO | 1)   /* WB Microphone distance (in mm), passenger BF    (StarRec_Int16) */

#define sse_SplEnableLocalization      (SSE_ML_SCINFO | 0 ) /* WA Enable (1) or disable (0) localization    (StarRec_Int16) */
#define sse_SplEnableBeamSteering      (SSE_ML_SCINFO | 1 ) /* WA Enable (1) or disable (0) beamsteering    (StarRec_Int16) */
#define sse_SplMicDist                 (SSE_ML_SCINFO | 3)  /* WB Distance in mm of both microphones used for localization  */
#define sse_SplCenterDirectionZone1    (SSE_ML_SCINFO | 4)  /* WA Center direction of beam where speaker is expected during localization */
#define sse_SplCenterDirectionZone2    (SSE_ML_SCINFO | 5)  /* WA Center direction of beam where speaker is expected during localization */
#define sse_SplWidthZone1              (SSE_ML_SCINFO | 6)  /* WA Zone in degree around currently detected speaker location in zone 1 where a localization should be possible */
#define sse_SplWidthZone2              (SSE_ML_SCINFO | 7)  /* WA Zone in degree around currently detected speaker location in zone 2 where a localization should be possible */
#define sse_SplBfUpdateTolerance       (SSE_ML_SCINFO | 10) /* WA Tolerance for the beamformer that will not follow changes below this value */
#define sse_DcmMicCovered              (SSE_ML_SCINFO | 15) /* RO Detection of covered microphone (0: no mic covered,1:mic1 covered,2:mic2 covered,3: one of the mics covered) */

#define sse_SADbSensitivity            (SSE_ML_SCINFO | 18) /* WA Sensitivity for evaluation of the broadband speaker activity detection measure (StarRec_Int16) */
#define sse_SADDoubleTalkDetectSens    (SSE_ML_SCINFO | 19) /* WA Sensitivity representing threshold that has to be exceeded by the double-talk period measure to detect a double-talk region (StarRec_Int16) */
#define sse_SADLocalDistDetectSens     (SSE_ML_SCINFO | 20) /* WA Sensitivity representing threshold that has to be exceeded by the double-talk period measure to detect a double-talk region (StarRec_Int16) */

#define sse_DcmAutomaticBfBypass       (SSE_ML_SCINFO | 27) /* WA Set automatic beamformer bypass dependent on detection of covered microphones */
#define sse_SSBMode                    (SSE_ML_SCINFO | 38) /* WA Mode of the Self-Steering Beamformer */
#define sse_SSBCoherentSoundProtection (SSE_ML_SCINFO | 39) /* WA Control the strength of the Self-Steering Beam Coherent Sound Protection */
#define sse_SSBMicSpacingMm            (SSE_ML_SCINFO | 40) /* WB Microphone spacing for the Self-Steering Beam Control */
#define sse_MixerFadingPeriod          (SSE_ML_MIX | 0) /* WA Time period that defines the fading period during selective mixing (equals time that has to be been elapsed until a speaker reaches full dominance) (StarRec_Int16) */
#define sse_MixerDynFloorDeltaMax      (SSE_ML_MIX | 1) /* WA Delta to obtain maximum allowed dynamic spectral floor caused in the selective mixing process (StarRec_Int16) */
#define sse_MixerDynFloorDeltaMin      (SSE_ML_MIX | 2) /* WA Delta to obtain minimum allowed dynamic spectral floor caused in the selective mixing process (StarRec_Int16) */
#define sse_MixerSpeechGain            (SSE_ML_MIX | 3) /* WA Additional speech gain for amplification of speech active regions detected by the speaker activity detection (StarRec_Int16) */
#define sse_MixerVRChannelSelect       (SSE_ML_MIX | 4) /* WA Selected internal channel that is used to pass to the recognizer in ASR mode */
#define sse_SpatialSoftMixing          (SSE_ML_MIX | 6) /* WA Degree of spatial soft mixing if two beams are mixed using the selective mixer (StarRec_Int16) */

/*--------------------------------------------------------------------------------------*/
/*     Define SSE data types for switches, modes, etc. for the SSE module configuration */
/*--------------------------------------------------------------------------------------*/

/** On/off/disabled switch */
typedef enum tsseSwitch
{
   sseOff      = 1,                  /**< off                                          */
   sseOn       = 2,                  /**< on                                           */
   sseDisabled = 3,                  /**< initialized but disabled                     */
   SSE_ENFORCE_32BIT_15 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseSwitch;

/** Position of the acoustic echo canceller */
typedef enum tsseAcousticEchoPosition
{
   sseAcousticEchoPos1 = 1,        /**< acoustic echo cancellation before beamformer   */
   sseAcousticEchoPos2 = 2,        /**< acoustic echo canceller behind beamformer      */
   SSE_ENFORCE_32BIT_06 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tsseAcousticEchoPosition;

/**  Switch for the operation mode (hands-free and voice recognition).     */
typedef enum tsseOperationMode
{
   sseHandsFree       = 1,         /**< hands-free mode                                */
   sseVoiceRecog      = 2,         /**< voice recognition mode                         */
   ssePassThrough     = 4,         /**< \if CREATE_INTERNAL_DOC
                                   pass through mode: copy data from input to
                                   output buffer without processing (legacy)
                                   \endif */
   ssePassThroughHF   = 5,         /**< \if CREATE_INTERNAL_DOC
                                            pass through mode: perform hands-free
                                            processing, but copy data from input to output (legacy)
                                         \endif */
   ssePassThroughVR   = 6,         /**< \if CREATE_INTERNAL_DOC
                                           pass through mode: perform voice recognition
                                           processing, but copy data from input to output (legacy)
                                        \endif */
   SSE_ENFORCE_32BIT_05 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tsseOperationMode;

/**  Microphone self-calibration mode (what kind of microphone self-calibration)       */
typedef enum tsseSelfCalibMode
{
   sseMethodB         = 1,         /**< Element related (efficient): One single microphone is used as reference for matching the other microphones */
   sseMethodCc        = 2,         /**< Array related (efficient): The beamformer output is used as reference for matching all microphones. */
   sseMethodD         = 3,         /**< Array related (refined): The beamformer is used as reference for matching all microphones. */
   sseMethodE         = 4,         /**< Element related (refined): The microphones are matched pairwise. */
   SSE_ENFORCE_32BIT_08 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tsseSelfCalibMode;


/**  Beamformer mode (what type of beamformer)  */
typedef enum tsseBeamMode
{
   sseDelaySumBeam    = 1,         /**< Delay-and-Sum beamformer                       */
   sseFilterSumBeam   = 2,         /**< Filter-and-Sum beamformer                      */
   sseAdaptiveBeam1   = 3,         /**< Adaptive beamformer (type 1)                   */
   SSE_ENFORCE_32BIT_07 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tsseBeamMode;

/** Mixer modes */
typedef enum tsseMixerMode
{
   sseAgileMix          = 1,         /**< fast beam switching                            */
   sseStickyMix         = 2,         /**< saccadic switching                             */
   sseFixedMix          = 3,         /**< fixed mixing ratio                             */
   sseSelectiveMix      = 4,         /**< frequency-selective switching                  */
   sseSideInfoFixedMix  = 5,         /**< fixed predefined channel in the signal path, side info based on multiple signals */
   sseSingleSpeakerSelectiveMix  = 6,/**< frequency-selective switching for a single speaker */
   SSE_ENFORCE_32BIT_01 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseMixerMode;

/** ASR mode which defines what kind of data SSE provides to the VoCon ASR and how it
    is transferred. The complete audio/feature stream is created by the SSE function
    #ssePrepareDataForASR(). */
typedef enum
{
   sseAudioOnly          = 1,   /**< Only audio data is content of one frame. */
   sseSpeechProbAttached = 2,   /**< Speech probability is attached at the end of each audio frame. */
   sseSpeechProbLSB      = 3,    /**< Speech probability is encoded in the LSB of the audio frame. */
   sseSpeechProbAttachedLegacy = 4,   /**< Speech probability is attached at the end of each audio frame. */
   sseSpeechProbLSBLegacy      = 5,   /**< Speech probability is encoded in the LSB of the audio frame. */
   SSE_ENFORCE_32BIT_04 = 0x7FFFFFFF  /* force compiler to use four bytes for this enum */
} tsseASRMode;

/** Save CPU mode enabling CPU saving measures in SSE. The mode is only relevant for SSE operation mode #sseVoiceRecog.
    When set to a value unequal @c sseNoSaveCPU, SSE automatically deactivates some processing which is not relevant
    at the moment.*/
typedef enum
{
   sseNoSaveCPU          = 1,   /**< CPU saving mode is deactevated (default). */
   sseSaveCPULevel1      = 2,   /**< CPU saving level 1: When echo cancellation is activated for ASR preprocessing (for "barge-in"),
                                     SSE switches off the EC when there is no activity on the reference channel. */
   SSE_ENFORCE_32BIT_02 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseSaveCPUMode;


/** @cond CREATE_INTERNAL_DOC */
/** Limiter modes. The 'look-ahead' limiter has been set to default. The other mode is
no longer being used. */
typedef enum tsseLimiterMode
{
   sseHardLimit       = 1,         /**< hard threshold limiter                         */
   sseLookAheadLimit  = 2,         /**< 'look ahead' limiter with some memory          */
   SSE_ENFORCE_32BIT_11 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tsseLimiterMode;
/** @endcond*/

/** @cond OBSOLETE */
/** HFE modes */
typedef enum tsseHFEMode
{
   sseHFEFreq = 1,                 /**< frequency-domain HFE - 11k HF / 22k VR/BargeIn */
   sseHFETime = 2,                 /**< time-domain HFE - 22k VR mode - obsolete       */
   SSE_ENFORCE_32BIT_12 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseHFEMode;
/** @endcond*/

/** @cond CREATE_INTERNAL_DOC */
/** Adaptive equalization modes */
typedef enum tsseAdaptEQMode
{
   sseGainOnly        = 1,         /**< gain equalization only (ENR-based)             */
   sseGainAndShape    = 2,         /**< gain and frequency equalization (ENR-based)    */
   sseNBGain          = 3,         /**< gain equalization only (noise-based)           */
   sseShapeOnly       = 4,          /**< frequency equalization (ENR-based)             */
   SSE_ENFORCE_32BIT_03 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */

} tsseAdaptEQMode;
/** @endcond*/

/** Speaker localization mode */
typedef enum tsseSpeakerLocMode
{
   sseSpeakerLocOff    = 1,         /**< Speaker localization is switched off. */
   sseSpeakerLocOnly   = 2,         /**< Speaker localization is processed (Currently, the localization information is
                                         used only for speaker tracking within SSE. Therefore sseSpeakerTracking should be used.)
                                         This means only the localization information  */
   sseSpeakerTracking  = 3,          /**< Speaker localization and tracking (beam-steering)  */
   SSE_ENFORCE_32BIT_14 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseSpeakerLocMode;


/** Windbuffet removal modes. */
typedef enum tsseWindBufMode
{
   sseSingleChannel   = 1,           /**< single-channel windbuffet removal    */
   sseMultiChannel    = 2,           /**< multi-channel windbuffet removal     */
   sseCombined        = 3,           /**< not implemented */
   SSE_ENFORCE_32BIT_10 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseWindBufMode;


/** Type of external memory (used in #sseCreate) */
typedef enum tsseMemType
{
   ssePersistMem           = 0,         /**< persistent memory                    */
   sseFastPersistMem       = 1          /**< fast persistent memory               */
} tsseMemType;

/** Type of SSE optimization level (used in #sseCreate). It is used to internally control
the usage of different memory types. */
typedef enum tsseMemOptLevel
{
   sseMemNoOpt        = 0,    /**< Use only (slow) external memory                       */
   sseMemOptLevel1    = 1,    /**< Use internal memory only for most time critical parts */
   sseMemMaxOpt       = 9     /**< Maximum optimization: Use only (fast) internal memory */
} tsseMemOptLevel;

/**  Data type for the different states of the pass-through mode */
typedef enum tssePassThroughMode
{
   sseNoPassThrough      = 1,  /**< SSE works in regular mode and output is the processed signal. */
   ssePassThroughCopy    = 2,  /**< SSE does no processing at all but only copies the input signals to the output buffers. */
   ssePassThroughProcess = 3,  /**< SSE does regular processing but uses the input signal instead of the processed signals
                               for the output. */
   SSE_ENFORCE_32BIT_16 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tssePassThroughMode;

typedef enum tsseSelfSteeringBeamMode
{
   sseSmartPhone    = 1,                 /**< General Mode for Smart Phones */
   sseSmartPhoneEar = 2,                 /**< Specific Mode for Smart Phones when held at the ear */
   SSE_ENFORCE_32BIT_17 = 0x7FFFFFFF /* force compiler to use four bytes for this enum */
} tsseSelfSteeringBeamMode;
/** Clipping statistics structure.
Clipping statistics are continuously gathered and stored for each microphone and reference channel.
They can be retrieved by calling sseGetData() with #sse_MicInClipStats, #sse_MicOutClipStats,
#sse_RecvInClipStats, #sse_RecvOutClipStats or #sse_RefInClipStats.

With #sse_ClipStatsMode one can define if the statistics are reset after fetching them or not.

A microphone, receive or reference input sample is flagged as a clipping when it's absolute value
is greater than #sse_MicClipThreshold, #sse_RecvClipThreshold or #sse_RefClipThreshold respectively. */
typedef struct tsseClipStats
{
   unsigned int   ClipFrames;    /**< Number of frames over which statistics have been gathered, 0 on reset */
   unsigned int   ClipCount;     /**< Number of clipped samples in last ClipFrames, 0 on reset */
   short int      MaxAmp;        /**< Maximum amplitude for last ClipFrames, -32768 on reset */
   short int      MinAmp;        /**< Mininum amplitude for last ClipFrames, 32767 on reset */
} tsseClipStats;


/** Main SSE data structure (for SSE internal data)
   This structure contains all internally used data and memory of the SSE module.
   The main structure is known only within the SSE module and is not accessible to the
   calling process.                                                                    */
typedef struct sseMainStruct tsseMain;

/** Structure used to input configuration information with two-stage initialization */
typedef struct tsseConfigData
{
   /** Number of config data buffers to load, must be <= SSE_MAX_CONFIG_DATA_CNT   */
   unsigned short int ConfigBufferCnt;

   /** Array of pointers to config data buffers that contain parameters
       for the configuration of the SSE module. Unused pointers must be NULL     */
   char* ConfigBuffer[SSE_MAX_CONFIG_DATA_CNT];

   /** Array of BSD buffer sizes in bytes                                        */
   unsigned int ConfigBufferSize[SSE_MAX_CONFIG_DATA_CNT];
} tsseConfigData;



/** Pointer array for audio input / output buffers used in sseProcess(). */
typedef short int *tsseAudio[SSE_MAX_AUDIO_CNT];

/** Array of external memory buffer structures. They are used to provide the sseCreate()
    function external memory instead of letting SSE do the memory allocation. */
typedef struct sseExtMemStruct
{
   void         *pMem;      /** Pointer to the external memory buffer       */
   unsigned int  Size;      /** Size of the external memory buffer in bytes */
   tsseMemType   Type;      /** External memory type                        */
} tsseExtMem[SSE_MAX_MEM_BUF_CNT];


/** Data structure for input/output debug data used in the sseProcess() function */
typedef struct tsseDebug
{
   unsigned int    InBufferSize;      /**< Size of the debug input buffer in bytes                  */
   unsigned int    InBufferUsedSize;  /**< Number of bytes actually being used in the input buffer  */
   void*           InBuffer;          /**< Pointer to debug input buffer                            */
   unsigned int    OutBufferSize;     /**< Size of debug output buffer in bytes                     */
   unsigned int    OutBufferUsedSize; /**< Number of bytes actually being used in the output buffer */
   void*           OutBuffer;         /**< Pointer to debug output buffer                           */
} tsseDebug;

/*-----------------------------------------------------------------------------*/
/*       Function prototypes                                                   */
/*-----------------------------------------------------------------------------*/

/*------ first stage of two-stage initialization of the SSE module ---------------------*/
int sseCreate(       tsseMain      **Main,
               const tsseExtMem      ExtMem,
               const tsseMemOptLevel iOptLevel);

/*------ final stage of two-stage initialization of the SSE module ---------------------*/
int sseInitialize(       tsseMain*          pMain,
                   const tsseConfigData*    pConfigData);

/*------ de-initialization function of the SSE module ----------------------------------*/
int sseDestroy( tsseMain  **pMain );

/*------ reset function of the SSE module ----------------------------------------------*/
int sseReset( tsseMain* pMain );

/*------ signal processing function of the SSE module ----------------------------------*/
int sseProcess(      tsseMain*  pMain,
               const tsseAudio  MicIn,
               const tsseAudio  RecvIn,
               const tsseAudio  RefIn,
                     tsseAudio  MicOut,
                     tsseAudio  RecvOut,
                     tsseDebug* pDebugData );

/*------ post processing function for speech recognition ----------------------------------*/
int ssePrepareDataForASR (       tsseMain*     pMain,
                                 short int*    pi16InputBuffer,
                           const unsigned int  u32InputBufferSize,
                                 short int*    pi16OutputBuffer,
                           const unsigned int  u32OutputBufferSize,
                                 unsigned int* pu32WrittenElemCnt );

/*------ function to set extended data ------------------------------------------------*/
int sseSetData(       tsseMain*    pMain,
                const tsseDataID   DataID,
                const int          iChannel,
                const unsigned int iSize,
                const void*        pData );

/*------ function to get extended data ------------------------------------------------*/
int sseGetData(       tsseMain*     pMain,
                const tsseDataID    DataID,
                const unsigned int  iChannel,
                      unsigned int* pSize,
                      void*         pData );

int sseGetErrorMessage( const tsseMain* pMain,
                              char*     pErrorMessage);

void sseGetVersion(       unsigned int* pArrayLen,
                    const int**         pVersionArray,
                    const char**        pVersionString,
                    const char**        pVersionComment );

int sseGetDataString (        tsseMain*      pMain,
                        const char*          pLabelString,
                              char*          pParameterString,
                              unsigned int*  pStringSize );

int sseSetDataString (       tsseMain* pMain,
                       const char*     pLabelString,
                       const char*     pParameterString);



#ifdef __cplusplus
}
#endif


#endif
/**@}*/
