/******************************************************************************
 * qwa-object.h
 *----------------------------------------------------------------------------
 * Andy Lowe <alowe@mvista.com>
 * 2010 (c) MontaVista Software, LLC. This file is licensed under the terms of
 * the AFL.
 *****************************************************************************/

/******************************************************************************
 * FILE        : ecnr-object.h
 * PROJECT     : Gen3 & Gen4
 * SW-COMPONENT: ECNR
 *----------------------------------------------------------------------------
 *
 * DESCRIPTION : Echo Cancellation and Noise Reduction Engine
 *
 *----------------------------------------------------------------------------
 * COPYRIGHT   : (c) 2013 RBCM GMBH
 * HISTORY     :
 * Date        | Author                 | Modification
 * 15.08.2013  | Patrick Rey            | initial version for Gen3
 *             |                        | porting from QWA object module.
 *****************************************************************************/

#ifndef __ECNR_OBJECT_H__
#define __ECNR_OBJECT_H__

#include <glib-object.h>
#include <dbus/dbus-glib.h>


/**
 * Define enumerations for the different signals that we can generate
 * (so that we can refer to them within the signals-array [below]
 * using symbolic names). These are not the same as the signal name
 * strings.
 *
 * NOTE: E_SIGNAL_COUNT is NOT a signal enum. We use it as a
 *       convenient constant giving the number of signals defined so
 *       far. It needs to be listed last.
 */
typedef enum
{
    E_SIGNAL_ECNR_ASR_MODE_ENABLE,
    E_SIGNAL_ECNR_AUDIO_STATUS,
    E_SIGNAL_ECNR_MICRO_LEVEL_STATUS,
    E_SIGNAL_COUNT
} EcnrSignal;



#define SIGNAL_ECNR_ASR_MODE_ENABLE    "ecnr_asr_mode_enable"
#define SIGNAL_ECNR_AUDIO_STATUS       "ecnr_audio_status"
#define SIGNAL_ECNR_MICRO_LEVEL_STATUS "ecnr_micro_level_status"

typedef struct EcnrObject EcnrObject;
typedef struct EcnrObjectClass EcnrObjectClass;

GType ecnr_object_get_type(void);

struct EcnrObject {
  GObject parent;
  gboolean ecnr_is_initialized;
  gboolean ecnr_Debug_intf_initialized;
  gint ecnr_current_configid;
  guchar ecnr_current_AppId;
};

struct EcnrObjectClass {
  GObjectClass parent;
};

enum {
    START_MODE_WAIT_STREAMING_RESULT = 0,
    START_MODE_RETURN_IMMEDIATELY
};


#define ECNR_TYPE_OBJECT              (ecnr_object_get_type ())
#define ECNR_OBJECT(object)           (G_TYPE_CHECK_INSTANCE_CAST ((object), ECNR_TYPE_OBJECT, EcnrObject))
#define ECNR_OBJECT_CLASS(klass)      (G_TYPE_CHECK_CLASS_CAST ((klass), ECNR_TYPE_OBJECT, EcnrObjectClass))
#define ECNR_IS_OBJECT(object)        (G_TYPE_CHECK_INSTANCE_TYPE ((object), ECNR_TYPE_OBJECT))
#define ECNR_IS_OBJECT_CLASS(klass)   (G_TYPE_CHECK_CLASS_TYPE ((klass), ECNR_TYPE_OBJECT))
#define ECNR_OBJECT_GET_CLASS(obj)    (G_TYPE_INSTANCE_GET_CLASS ((obj), ECNR_TYPE_OBJECT, EcnrObjectClass))


extern   gboolean ecnr_object_ecnr_get_version(EcnrObject * qobj, GArray ** version,gchar ** version_string, gchar ** version_comment, GError ** error);
extern   gboolean ecnr_object_ecnr_get_status_message(EcnrObject * qobj, guchar ecnrAppId, gchar ** status_message, GError ** error);
extern   gboolean ecnr_object_ecnr_initialize(EcnrObject * qobj, guchar ecnrAppId, gint ecnrConfigurationId, GError ** error);
extern   gboolean ecnr_object_ecnr_initialize_sync(EcnrObject * qobj, guchar  ecnrAppId, gint  ecnrConfigurationId, gchar ** InDevices, gchar ** OutDevices, GArray ** DeviceProperties, GError ** error);
extern   gboolean ecnr_object_ecnr_reset(EcnrObject * qobj, guchar ecnrAppId, GError ** error);
extern   gboolean ecnr_object_ecnr_destroy(EcnrObject * qobj, guchar ecnrAppId, GError ** error);
extern   gboolean ecnr_object_ecnr_start_audio(EcnrObject * qobj, guchar ecnrAppId, GError ** error);
extern   gboolean ecnr_object_ecnr_start_audio_ext(EcnrObject * qobj, guchar ecnrAppId, guchar StartMode, GError ** error);
extern   gboolean ecnr_object_ecnr_stop_audio(EcnrObject * qobj, guchar ecnrAppId, GError ** error);
extern   gboolean ecnr_object_ecnr_frame_info(EcnrObject * qobj, guint * FrameCounter, gint * VoicePresent, gint * AboveBackgroundNoise, gint * PercentClipping, gint * InstantaneousSNR, gint * SmoothVNR, gint * ValidOutputCountdown, GError ** error);
extern   gboolean ecnr_object_ecnr_mic_clip_stats(EcnrObject * qobj, gint Channel, guint * ClipFrames, guint * ClipCount, gint * MaxAmp, gint * MinAmp, GError ** error);
extern   gboolean ecnr_object_ecnr_ref_clip_stats(EcnrObject * qobj, guint * ClipFrames, guint * ClipCount, gint * MaxAmp, gint * MinAmp, GError ** error);
extern   gboolean ecnr_object_ecnr_recv_clip_stats(EcnrObject * qobj, guint * ClipFrames, guint * ClipCount, gint * MaxAmp, gint * MinAmp, GError ** error);
extern   gboolean ecnr_object_ecnr_mic_out_clip_stats(EcnrObject * qobj, guint * ClipFrames, guint * ClipCount, gint * MaxAmp, gint * MinAmp, GError ** error);
extern   gboolean ecnr_object_ecnr_recv_out_clip_stats(EcnrObject * qobj, guint * ClipFrames, guint * ClipCount, gint * MaxAmp, gint * MinAmp, GError ** error);
extern   gboolean ecnr_object_ecnr_terminate(EcnrObject * qobj, GError ** error);
extern   gboolean ecnr_object_ecnr_set_configuration(EcnrObject * qobj, guchar ecnrAppId, gint ecnrConfigurationId, GError ** error);
extern   gboolean ecnr_object_ecnr_set_configuration_sync(EcnrObject * qobj, guchar  ecnrAppId, gint  ecnrConfigurationId, gchar ** InDevices, gchar ** OutDevices, GArray ** DeviceProperties, GError ** error);
extern   gboolean ecnr_object_ecnr_get_ecnr_configuration(EcnrObject * qobj, gchar ** DatasetName, GArray ** EcnrProperties, GError ** error);
extern   gboolean ecnr_object_ecnr_start_debug_data(EcnrObject * qobj,guchar ecnrDebugMode, GError ** error);
extern   gboolean ecnr_object_ecnr_end_debug_data(EcnrObject * qobj,guchar ecnrDebugMode, GError ** error);

/* new revision 1.1*/

extern   gboolean ecnr_object_ecnr_set_send_mute_switch(EcnrObject * qobj, guchar ecnrAppId, guchar ecnrSendMuteSwitch, GError ** error);
extern   gboolean ecnr_object_ecnr_set_send_gain_delta(EcnrObject * qobj, guchar ecnrAppId, gint16 ecnrLevel, GError ** error);
extern   gboolean ecnr_object_ecnr_set_mic_gain(EcnrObject * qobj, gint16 MicGainLevel, GError ** error);
extern   gboolean ecnr_object_ecnr_set_pass_through_mode(EcnrObject * qobj, guchar ecnrAppId, guchar ecnrPassThroughMode, GError ** error);
extern   gboolean ecnr_object_ecnr_set_nr_comb_floor(EcnrObject * qobj, guchar ecnrAppId, gint16 ecnrNRCombFloor, GError ** error);
extern   gboolean ecnr_object_ecnr_set_echo_cancellation(EcnrObject * qobj, guchar ecnrAppId, guchar EcnrECState, GError ** error);
extern   gboolean ecnr_object_ecnr_set_noise_reduction(EcnrObject * qobj, guchar ecnrAppId, guchar EcnrNRState, GError ** error);
extern   gboolean ecnr_object_ecnr_alsa_set_device (EcnrObject * qobj, guchar ecnrAppId, guint AlsaDeviceSelect, gchar * AlsaDeviceName, GError ** error);
extern   gboolean ecnr_object_ecnr_alsa_set_thread_priority(EcnrObject * qobj, guchar ecnrAppId, gint priority, GError ** error);
extern   gboolean ecnr_object_ecnr_alsa_set_wide_band(EcnrObject * qobj, guchar ecnrAppId, GError ** error);
extern   gboolean ecnr_object_ecnr_alsa_set_device_properties(EcnrObject * qobj, guchar  ecnrAppId, guint  AlsaDeviceSelect, guchar  AlsaDeviceProperty, guint  PropertyValue, GError  **  error);
extern   gboolean ecnr_object_ecnr_get_engine_parameter(EcnrObject * qobj, guint param_id, GArray **data, GError ** error);
extern   gboolean ecnr_object_ecnr_start_itc_test(EcnrObject * qobj, gint32 ITCValue, gint32* rspITCValue, guchar* TestResult, gint32* TestErrorCode, GError ** error);

/* send functions */
extern   int    ecnr_object_emit_signal_ecnr_asr_mode_enable       (gint EcnrAsrMode);
extern   int    ecnr_object_emit_signal_ecnr_audio_status          (gint AudioState);
extern   int    ecnr_object_emit_signal_ecnr_micro_level_status    (guint32 EcnrMicroLevel);


#endif /* __ECNR_OBJECT_H__ */
