//
// CCA_MessageTrackerBase.cpp
//
//  Created on: Jan 21, 2015
//      Author: Martin Koch, Fa. ESE
//


#include "CCA_Node.h"
#include "CCA_MessageTrackerBase.h"
// - - - - - - - - - - - - - - - - -

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include <etg_if.h>



#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_GENIVI_CCA_NODE
#include "trcGenProj/Header/CCA_MessageTrackerBase.cpp.trc.h"


namespace FIMessaging  { namespace CCA
{

   // ==========================================================================
   //
   //             M e s s a g e - T r a c k e r - B a s e   I t e m
   //

   /* constructor */ MessageTrackerBase::PendingCall:: PendingCall (const MessageInfo& info, uint16_t respID)
      : u16ResponseID(respID)
      , messageInfo(info)
      , invocationTime(OSAL_ClockGetElapsedTime())
   {
   }


   // ==========================================================================
   //
   //                M e s s a g e - T r a c k e r - B a s e
   //

   /* constructor */ MessageTrackerBase:: MessageTrackerBase (const ServiceInfo& svcInfo)
      : serviceInfo(svcInfo)
      , _list()
   {

   }

   // --------------------------------------------------------------------------

   /* destructor */ MessageTrackerBase:: ~MessageTrackerBase()
   {

   }

   // --------------------------------------------------------------------------

   void MessageTrackerBase:: vDumpEntries ()
   {
      // TTFis output of current list content
      //

      List copy;

      {  Node::AutoLock lock;

         copy = _list;

      }  // leave AutoLock scope

      if (copy.size())
         ETG_TRACE_USR4(("\t responseID: Service.Function.OpCode to Application/Sub"))
      for (Iterator it = copy.begin(); it != copy.end(); ++it)
      {
         const FunctionInfo& func = it->messageInfo.function;
         const RemoteAppInfo& remote = it->messageInfo.remoteApp;
         ETG_TRACE_USR4(("\t %u:\t %x.%x.%u\t at \t %x/%d"
               , it->messageInfo.u16CmdCounter
               , func.service.u16ServiceID, func.u16FunctionID, func.u8OpCode
               , remote.u16TargetAppID, remote.u16TargetSubID))
      }
   }

   // --------------------------------------------------------------------------

   tU16 MessageTrackerBase:: u16CreateResponseID ()
   {
      // Calculate a (temporary) unique response-ID
      //  - Normally increments with every call with turn-over
      //  - Never returns values 1 and 2, as these are reserved for
      //    UpReg / RelUpreg mechanism
      //

      static tU16 u16Latest = 2;

      // do not use 1 and 2 as they are reserved for UpReg / RelUpreg messages
      if (u16Latest < 0xffff)
         ++u16Latest;
      else
         u16Latest = 3;  // round robin re-use

      // simply use next value if all current response-IDs are lower than latest
      tU16 u16CurrentMax = u16Latest;
      for (Iterator it = _list.begin(); it != _list.end(); ++it)
         if (it->u16ResponseID > u16CurrentMax)
            u16CurrentMax = it->u16ResponseID;

      if (u16CurrentMax <= u16Latest)
         return u16Latest;  // normal return

      // - - - - - - -

      // round-robin turn-over condition with remaining high response-IDs
      // - scan list for lowest unused ID
      tU16 next = 3;
      for (  ; next < _list.size(); ++next)
      {
         bool used = false;
         Iterator it = _list.begin();
         while (it != _list.end())
         {
            if (it->u16ResponseID == next)
            {
               used = true;
               break;
            }
            ++it;
         }
         if ( ! used)
            break;  // for-loop - we can use this value
      }

      return next;
   }

   // --------------------------------------------------------------------------

   /* static */ bool MessageTrackerBase:: bResponseOpCodeMatchesRequest (uint8_t u8ResponseOpCode, uint8_t u8RequestOpCode)
   {
      // Determine, whether a (received or intended) response message is suitable
      // as answer to a previous request (command) message (sent or received)
      //

      switch (u8RequestOpCode)
      {
         // properties
         case CCA_C_U8_OPCODE_SET:
         case CCA_C_U8_OPCODE_GET:
         case CCA_C_U8_OPCODE_UPREG:
         case CCA_C_U8_OPCODE_RELUPREG:
         case CCA_C_U8_OPCODE_INCREMENT:
         case CCA_C_U8_OPCODE_DECREMENT:
         case CCA_C_U8_OPCODE_PURESET:
            switch (u8ResponseOpCode)
               {
                  case CCA_C_U8_OPCODE_STATUS:
                  case CCA_C_U8_OPCODE_ERROR:
                     return true;

                  default:
                     return false;
               }

         // methods
         case CCA_C_U8_OPCODE_METHODSTART:
            switch (u8ResponseOpCode)
               {
                  case CCA_C_U8_OPCODE_METHODRESULT:
                  case CCA_C_U8_OPCODE_METHODRESULTFIRST:
                  case CCA_C_U8_OPCODE_METHODRESULTMIDDLE:
                  case CCA_C_U8_OPCODE_METHODRESULTLAST:
                  case CCA_C_U8_OPCODE_ERROR:
                     return true;

                  default:
                     return false;
               }

         case CCA_C_U8_OPCODE_METHODABORT:
            switch (u8ResponseOpCode)
               {
                 case CCA_C_U8_OPCODE_ABORTRESULT:
                 case CCA_C_U8_OPCODE_ERROR:
                    return true;

                 default:
                    return false;
              }

         // illegal request codes
         case CCA_C_U8_OPCODE_STATUS:
         case CCA_C_U8_OPCODE_METHODRESULT:
         case CCA_C_U8_OPCODE_ERROR:
         case CCA_C_U8_OPCODE_ABORTRESULT:
         case CCA_C_U8_OPCODE_METHODRESULTFIRST:
         case CCA_C_U8_OPCODE_METHODRESULTMIDDLE:
         case CCA_C_U8_OPCODE_METHODRESULTLAST:
         case CCA_C_U8_OPCODE_CLIENT_PORT_ACK:
         case CCA_C_U8_OPCODE_SERVER_PORT_ACK:
         default:
            return false;
      }
   }

   // --------------------------------------------------------------------------

   void MessageTrackerBase:: vDropOverdues ()
   {
      // since some some requests intentionally do not have a response on success
      // (OpCodes PureSet and in MOST-compatible catalogs also some MethodStart)
      // we need to drop these requests from time to time
      //
      // Note: must be called within AutoLock protection
      //

      OSAL_tMSecond now = OSAL_ClockGetElapsedTime();
      for (Iterator it = _list.begin(); it != _list.end(); /* ++it */)
      {
         if (now > it->invocationTime + 10000) // consider as outdated after 10 seconds
         {
            FunctionInfo& func = it->messageInfo.function;
            switch ( func.u8OpCode)
            {
               // these might have no response by design, so give only silent notice
               case CCA_C_U8_OPCODE_METHODSTART:
               case CCA_C_U8_OPCODE_INCREMENT:
               case CCA_C_U8_OPCODE_DECREMENT:
               case CCA_C_U8_OPCODE_PURESET:
               case CCA_C_U8_OPCODE_RELUPREG:
                  ETG_TRACE_USR4(("MessageTracker::vDropOverdues() - dropping ID %d for request %x.%x.%u; new array size is %d"
                     , it->u16ResponseID
                     , func.service.u16ServiceID, func.u16FunctionID, func.u8OpCode
                     , _list.size()))
                  break;

               default:
                  // others should be responded in time or are invalid, so COMPLAIN
                  ETG_TRACE_ERR(("MessageTrackerBase::vDropOverdues() - W A R N I N G :"
                        "  dropping overdue ID %d for request %x.%x.%u; new array size is %d"
                     , it->u16ResponseID
                     , func.service.u16ServiceID, func.u16FunctionID, func.u8OpCode
                     , _list.size()))
                  break;
            }

            // drop it anyway
            it = _list.erase(it);
            continue;  // next item in for-loop - skip incrementing
         }

         ++it;  // can't increment in for-loop header due to intermediate erase()
      }

   }

   // --------------------------------------------------------------------------

}  }  // namespace FIMessaging::CCA
