//
// Client.cpp
//
//  Created on: Dec 18, 2014
//      Author: Martin Koch, Fa. ESE
//


#include "CCA_Client.h"
// - - - - - - - - - - - -

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include <etg_if.h>

#include "CCA_MessageTrackerBase.h"


#define ETG_DEFAULT_TRACE_CLASS  TR_CLASS_GENIVI_CCA_NODE
#include "trcGenProj/Header/CCA_Client.cpp.trc.h"



namespace FIMessaging  {  namespace CCA
{

   // ==========================================================================
   //
   //        private helper class   M e s s a g e - T r a  c k e r
   //

   class Client::MessageTracker : public MessageTrackerBase
   {
      public:
         inline /* constructor */ MessageTracker(const ServiceInfo& svcInfo)
            : MessageTrackerBase(svcInfo)   { }
         virtual /* destructor */ ~MessageTracker()  { }

         MessageInfo oAppend (const RemoteAppInfo& remote, const fi_tclMessageBase& fiCommand);
         // - return value can be used for posting the message.
         // - contained u16CmdCounter is used as u16ResponseID
         // - u16CmdCounter is zero, then OpCode of fiCommand indicates a server response,
         //   which is not valid for sending from client

         tU16 /* response-ID of matching entry */ u16RemoveEntry (const amt_tclServiceData& oSvcData);
   };

   // --------------------------------------------------------------------------

   MessageInfo Client::MessageTracker:: oAppend (const RemoteAppInfo& remote, const fi_tclMessageBase& fiCommand)
   {
      // - return value can be used for posting the message.
      // - contained u16CmdCounter is used as u16ResponseID
      // - u16CmdCounter is zero, then OpCode of fiCommand indicates a server response,
      //   which is not valid for sending from client

      FunctionInfo func(serviceInfo, fiCommand.u16GetFunctionID(), fiCommand.u8GetOpCode());
      MessageInfo info(remote, func, 0u, 0u);  // initialize with zero for CmdCounter and ACT
      unsigned int uSize;  // is initialized below

      {  Node::AutoLock lock;

         switch (info.function.u8OpCode)
         {
            case CCA_C_U8_OPCODE_UPREG:
               info.u16CmdCounter = 1; // like AHL, always use 1 for UpReg messages
               break;

            case CCA_C_U8_OPCODE_RELUPREG:
               info.u16CmdCounter = 2;  // like AHL, always use 2 for RelUpreg messages as AHL does
               break;

            case CCA_C_U8_OPCODE_SET:
            case CCA_C_U8_OPCODE_GET:
            case CCA_C_U8_OPCODE_METHODSTART:
            case CCA_C_U8_OPCODE_METHODABORT:
            case CCA_C_U8_OPCODE_INCREMENT:
            case CCA_C_U8_OPCODE_DECREMENT:
            case CCA_C_U8_OPCODE_PURESET:
               // scan list for unused entry
               info.u16CmdCounter = u16CreateResponseID();
               break;

            default:  // all others are server messages and invalid to be sent from clients
               info.u16CmdCounter = 0;
               break;
         }

         if (info.u16CmdCounter)
            _list.push_back(PendingCall(info, info.u16CmdCounter));

         uSize = (unsigned)_list.size();

      }  // end of AutoLock scope

      if (info.u16CmdCounter)
         ETG_TRACE_USR3(("CLient<%x>::oAppend() - successfully appended responseID %u for function %x.%x to Application %u, new size is %u"
               , serviceInfo.u16ServiceID, info.u16CmdCounter, func.u16FunctionID, func.u8OpCode, remote.u16TargetAppID, uSize))
      else
         ETG_TRACE_USR3(("CLient<%x>::oAppend() - E R R O R :  faield appending function %x.%x to Application %u, size remains %u"
               , serviceInfo.u16ServiceID, func.u16FunctionID, func.u8OpCode, remote.u16TargetAppID, uSize))

      vDumpEntries();

      return info;
   }

   // --------------------------------------------------------------------------

   tU16 Client::MessageTracker:: u16RemoveEntry (const amt_tclServiceData& oSvcData)
   {
      // will return found u16ResponseID

      tU16 u16CmdCounter = oSvcData.u16GetCmdCounter();
      tU8  u8OpCode      = oSvcData.u8GetOpCode();
      tU8  u8ReqOpCode   = 0xff;
      bool success      = false;
      unsigned int uSize       = 0;

      {
         Node::AutoLock lock;

         for (Iterator it = _list.begin(); it != _list.end(); ++it)
         {
            if (it->messageInfo.u16CmdCounter != u16CmdCounter)
               continue;
            else if (bResponseOpCodeMatchesRequest (u8OpCode, it->messageInfo.function.u8OpCode))
            {
               u8ReqOpCode = it->messageInfo.function.u8OpCode;
               (void) _list.erase(it);
               success = true;
               break;
            }
         }
         uSize = (unsigned)_list.size();

         // use this opportunity to get rid of old stuff
         vDropOverdues();

      }  // end of AutoLock scope

      tU16 u16ServiceID = oSvcData.u16GetServiceID();
      tU16 u16FunctionID = oSvcData.u16GetFunctionID();
      tU16 u16SourceAppID = oSvcData.u16GetSourceAppID();
      if (success)
         ETG_TRACE_USR3(("CCA_CLient<%x>::u16RemoveEntry() - successfully removed responseID %u for function %x.%x to Application %u, new size is %d"
               , u16ServiceID, u16CmdCounter, u16FunctionID, u8ReqOpCode, u16SourceAppID, uSize))
      else if ((CCA_C_U8_OPCODE_STATUS == u8OpCode) && (1 == u16CmdCounter))
         ETG_TRACE_USR3(("CCA_CLient<%x>::u16RemoveEntry() - forwarding spontaneous Status for function %x.%x to Application %u, size remains %d"
               , u16ServiceID, u16FunctionID, u8OpCode, u16SourceAppID, uSize))
      else
         ETG_TRACE_ERR(("CCA_CLient<%x>::u16RemoveEntry() - E R R O R :  failed removing function %x %x to Application %u from received OpCode, size remains %d"
               , u16ServiceID, u16FunctionID, u16SourceAppID, u8OpCode, uSize))

      vDumpEntries();

      return u16CmdCounter;
   }

   // ==========================================================================
   //
   //                     class   C C A :: C l i e n t
   //

   /* constructor */ Client:: Client (Node& node, ITarget& externalTarget)
      : TargetBase(node, externalTarget)
      , _u16ServerAppID(AMT_C_U16_APPID_INVALID)
      , _u16RegID(AMT_C_U16_REGID_INVALID)
      , _bServiceAvailable(false)
      , _pMessageTracker(new MessageTracker(externalTarget.GetServiceInfo()))
   {
      const tFIVersionInfo& ver = serviceInfo.fiVersion;
      ETG_TRACE_USR1(("CCA_Client<%x, Svc %x> with version {%u, %u} established"
            , u16OwnAppID, serviceInfo.u16ServiceID, ver.u16MajorVersion, ver.u16MinorVersion))
   }

   // --------------------------------------------------------------------------

   /* virtual destructor */ Client:: ~Client ()
   {

      ETG_TRACE_USR1(("CCA_Client<%x, Svc %x> terminating ... "
            , u16OwnAppID, serviceInfo.u16ServiceID))

      delete _pMessageTracker;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Client:: vOnNewAppState (tU32 u32NewAppState)
   {
      // callback function for incoming messages

      if (_u32AppState == u32NewAppState)
         return;

      // should protect these 2 lines with mutex
      tU32 u32LastAppState = _u32AppState;
      _u32AppState = u32NewAppState;

      ETG_TRACE_USR1(("CCA_Client<%x, Svc %x>:: vOnNewAppState() - changing state: 0x%08X --> 0x%08X"
            , u16OwnAppID, serviceInfo.u16ServiceID
            , ETG_ENUM(ail_u32CCAState, (tU32)u32LastAppState)
            , ETG_ENUM(ail_u32CCAState, (tU32)u32NewAppState) ))

      if (u32LastAppState == u32NewAppState)
         return;

      if (AMT_C_U32_STATE_NORMAL == u32NewAppState)
         vConnectToService();
   }

   // --------------------------------------------------------------------------

   tU16 Client:: u16GetRegisterID () const
   {
      return _u16RegID;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Client:: vOnAsyncRegisterConfExt(tU16 u16RegisterId, tU16 u16ServerAppId, tU16 u16ServiceId, tU16 u16TargetSubId, tU8 u8AsyncRegisterConfStatus)
   {
      switch (u8AsyncRegisterConfStatus)
      {
         // success
         case AMT_C_U8_REGCONF_SUCCESS :
            ETG_TRACE_USR1(("CCA_Client<%x, Svc %x>:: vOnAsyncRegisterConfExt(): Successfully registered for service %x at Application %x.%u Applied register-ID = %u"
                       , u16OwnAppID, serviceInfo.u16ServiceID
                       , ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId)
                       , ETG_ENUM(ail_u16AppId, (tU16)u16ServerAppId), u16TargetSubId, u16RegisterId))

            if (_u16RegID != AMT_C_U16_REGID_INVALID) // Warning => Already assigned register ID is overwritten
               ETG_TRACE_COMP(("CCA_Client<%x, Svc %x>:: vOnAsyncRegisterConfExt(): RegisterID %u overwrites old RegisterID %u of ServiceID %x"
                           , u16OwnAppID, serviceInfo.u16ServiceID, u16RegisterId, _u16RegID
                           , ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId) ))

            if (_u16ServerAppID == AMT_C_U16_APPID_INVALID)
               _u16ServerAppID = u16ServerAppId;
            _u16RegID = u16RegisterId;

            return;

         // error conditions
         case AMT_C_U8_REGCONF_SERVICE_ALREADY_REGISTERED :
            ETG_TRACE_ERRMEM(("CCA_Client<%x, Svc %x>:: vOnAsyncRegisterConfExt(): Registration for service %x at application %x,%u failed because this service is already registered"
                  , u16OwnAppID, serviceInfo.u16ServiceID
                  , ETG_ENUM(ail_u16ServiceId, u16ServiceId)
                  , ETG_ENUM(ail_u16AppId, u16ServerAppId), u16TargetSubId))
             return;

         case AMT_C_U8_REGCONF_SERVICE_DOES_NOT_EXIST :
             ETG_TRACE_SYS_MIN(("CCA_Client<%x, Svc %x>:: vOnAsyncRegisterConfExt(): Registration for service %x failed because there is no application which offers this service"
                   , u16OwnAppID, serviceInfo.u16ServiceID
                   , ETG_ENUM(ail_u16ServiceId, u16ServiceId)))
             return;

         case AMT_C_U8_REGCONF_SERVICE_VERSION_NOT_SUPPORTED :
            ETG_TRACE_SYS_MIN(("CCA_Client<%x, Svc %x>:: vOnAsyncRegisterConfExt(): Registration for service %x failed because application %x doesn't offer the requested service version"
                  , u16OwnAppID, serviceInfo.u16ServiceID
                  , ETG_ENUM(ail_u16ServiceId, u16ServiceId)
                  , ETG_ENUM(ail_u16AppId, u16ServerAppId)))
            return;

        case AMT_C_U8_REGCONF_SERVER_DOES_NOT_RESPOND_TIMEOUT :
            ETG_TRACE_FATAL(("CCA_Client<%x, Svc %x>:: vOnAsyncRegisterConfExt(): Registration for service %x failed because application %x.%u doesn't respond at all (timeout)"
                  , u16OwnAppID, serviceInfo.u16ServiceID
                  , ETG_ENUM(ail_u16ServiceId, u16ServiceId)
                  , ETG_ENUM(ail_u16AppId, u16ServerAppId), u16TargetSubId))
            return;

        case AMT_C_U8_REGCONF_SERVICE_REGISTRATION_PENDING :
            ETG_TRACE_FATAL(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Registration for service %x failed because a registration for this service is already pending"
                  , u16OwnAppID, serviceInfo.u16ServiceID
                  , ETG_ENUM(ail_u16ServiceId, (tU16)u16ServiceId)))
            return;

        default :
            ETG_TRACE_ERRMEM(("Application %x SubID 0x%04x vOnAsyncRegisterConfExt(): Parameter u8AsyncRegisterConfStatus = %u is unknown"
                  , u16OwnAppID, serviceInfo.u16ServiceID, u8AsyncRegisterConfStatus ))
            return;
      }
   }

   // --------------------------------------------------------------------------

   /* virtual */ tVoid Client:: vOnServiceState (tU8 u8ServiceState)
   {
      bool bNewAvailability = (AMT_C_U8_SVCSTATE_AVAILABLE == u8ServiceState);
      if (bNewAvailability == _bServiceAvailable)
         return;   // unchanged

      ETG_TRACE_USR1(("CCA_Client<%x, Svc %x>:: vOnServiceState(): Service availability changes %u -> %u"
                 , u16OwnAppID, serviceInfo.u16ServiceID
                 , _bServiceAvailable, bNewAvailability))

      _bServiceAvailable = bNewAvailability;

      // inform our external target
      if (_bServiceAvailable)
      {
         _externalTarget.vOnConnect();

         ITarget::AutoRegisterList list = _externalTarget.GetAutoRegisterList();
         ServiceInfo            svcInfo = _externalTarget.GetServiceInfo();
         if (list.pFunctionList)
            for (unsigned i = 0; i < list.u32ListSize; ++i)
            {
               fi_tclEmptyMsg msg(FunctionInfo(svcInfo, list.pFunctionList[i], (tU8)enUpreg));
               (void) u16PostFICommand(msg);
            }
      }
      else
         _externalTarget.vOnDisconnect();
   }

   // --------------------------------------------------------------------------

   void Client:: vConnectToService ()
   {
      /* ---  registration invalid ? --- */
      if (_u16RegID != AMT_C_U16_REGID_INVALID)
      {
        // it could be an error to call this function, if we are already registered
        ETG_TRACE_COMP(("CCA_Client<%x, Svc %x>:: vConnectToService() while system is not ready "
              , u16OwnAppID, serviceInfo.u16ServiceID))
        return;
      }

      // ---  try to register ---
      tBool bSuccess = _node.bRegisterAsync(serviceInfo.u16ServiceID
          , serviceInfo.fiVersion.u16MajorVersion, serviceInfo.fiVersion.u16MinorVersion, u16OwnSubID);

      // all ok, registering in process, now wait for answer
      ETG_TRACE_USR1(("CCA_Client<%x, Svc %x>:: vConnectToService() at Application %x SubID %d  returns %s "
                 , u16OwnAppID, serviceInfo.u16ServiceID
                 , ETG_ENUM(ail_u16AppId, u16OwnAppID), u16OwnSubID, (bSuccess ? "true" : "false")))
   }

   // --------------------------------------------------------------------------

   void Client:: vDisconnectFromService ()
   {
      ETG_TRACE_USR1(("CCA_Client<%x, Svc %x>:: vDisconnectFromService() SubID %d "
            , u16OwnAppID, serviceInfo.u16ServiceID, u16OwnSubID ))
      _node.vUnregisterService(serviceInfo.u16ServiceID, AMT_C_U16_APPID_INVALID, _u16RegID, u16OwnSubID);

      // Make register-ID invalid:
      _u16RegID = AMT_C_U16_REGID_INVALID;
   }

   // --------------------------------------------------------------------------

   /* virtual */ uint16_t Client:: u16PostFICommand (const fi_tclMessageBase& fiMsg)
   {
      if ( ! _bServiceAvailable)
      {
         ETG_TRACE_FATAL(("CCA_Client<%x, Svc %x>:: u16PostFICommand() - E R R O R  sending message for FID %x.%x : Service not available"
               , u16OwnAppID, serviceInfo.u16ServiceID, fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode()))
         return INVALID_RESPONSE_ID;
      }
      else if (NULL == _pMessageTracker)
      {
         ETG_TRACE_FATAL(("CCA_Client<%x, Svc %x>:: u16PostFICommand() - E R R O R  sending message for FID %x.%x :invalid pointer: _pMessageTracker"
               , u16OwnAppID, serviceInfo.u16ServiceID, fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode() ))
         return INVALID_RESPONSE_ID;
      }

      MessageInfo info = _pMessageTracker->oAppend(RemoteAppInfo(_u16ServerAppID, 0 /* subID */, _u16RegID), fiMsg);
      if (info.u16CmdCounter)
         _node.vPostFIMessage(info, fiMsg);

      return info.u16CmdCounter;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Client:: vSetServiceAvailable (bool /* bIsAvailable */)
   {
      ETG_TRACE_FATAL(("CCA_Client<%x, Svc %x>:: vSetServiceAvailable() - E R R O R - NOT a service"
            , u16OwnAppID, serviceInfo.u16ServiceID))
   }

   // --------------------------------------------------------------------------

   /* virtual */ bool Client:: bPostFIResult (const fi_tclMessageBase& fiMsg, uint16_t /* responseID */)
   {
      ETG_TRACE_FATAL(("CCA_Client<%x, Svc %x>:: bPostFIResult() - E R R O R  sending result message not allowed: FID %x.%x"
            , u16OwnAppID, serviceInfo.u16ServiceID, fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode() ))

      return false;
   }

   // --------------------------------------------------------------------------

   /* virtual */ bool Client:: bNotifyClients (const fi_tclMessageBase& fiMsg)
   {
      ETG_TRACE_FATAL(("Client<%x>::bNotifyClients() - E R R O R  sending status message not allowed: FID %x.%x"
            ,serviceInfo.u16ServiceID, fiMsg.u16GetFunctionID(), fiMsg.u8GetOpCode() ))

      return false;
   }

   // --------------------------------------------------------------------------

   /* virtual */ void Client:: vOnNewMessage (amt_tclServiceData& oSvcData)
   {
      // callback function for incoming messages
      ETG_TRACE_USR1(("Client<%x.%d>::vOnNewMessage(): ServiceData message received for FunctionID %x.%x"
                 , serviceInfo.u16ServiceID, oSvcData.u16GetSourceSubID()
                 , oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode()))

      // validate preconditions
      if (oSvcData.u16GetServiceID() != serviceInfo.u16ServiceID)
      {
         ETG_TRACE_FATAL(("Client<%x.%d>::vOnNewMessage - E R R O R : ServiceID of message does not match: is %x"
               , oSvcData.u16GetServiceID(), oSvcData.u16GetSourceSubID(), serviceInfo.u16ServiceID))
         return;
      }

      // transform and dispatch to external target
      uint32_t payloadOffset = AMT_C_U32_BASEMSG_ABSMSGSIZE + AMT_C_U32_SVCDATA_RELMSGSIZE; // skip 32 byte message header
      uint8_t* pPayload = oSvcData.pu8GetSharedMemBase() + payloadOffset;
      uint32_t length = oSvcData.u32GetSize();
      if (length >= payloadOffset)
         length -= payloadOffset;
      else
      {
         ETG_TRACE_FATAL(("Client<%x.%d::vOnNewMessage() - E R R O R : ServiceData message too short for FunctionID %x.%x"
                    , ETG_ENUM(ail_u16ServiceId, oSvcData.u16GetServiceID()), oSvcData.u16GetSourceSubID()
                    , oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode()))
         return;
      }

      FIMessage fiMsg(serviceInfo.u16ServiceID, serviceInfo.fiVersion, oSvcData.u16GetFunctionID(), oSvcData.u8GetOpCode(), oSvcData.u8GetACT()
            , pPayload, length);
      if (_pMessageTracker)
         fiMsg.u16ResponseID = _pMessageTracker->u16RemoveEntry(oSvcData);

      _externalTarget.vOnNewMessage(fiMsg);
   }

   // --------------------------------------------------------------------------

}  }  // namespace FIMessaging::CCA


