//
// MessageTypes.h
//
//  Created on: Jan 22, 2015
//      Author: Martin Koch, Fa. ESE
//

#ifndef MESSAGETYPES_H_
#define MESSAGETYPES_H_


namespace FIMessaging
{

   // --------------------------------------------------------------------------
   //
   //                              F I - M e s s a g e
   //
   // Transport-independent class for incoming messages,
   // interpretable with FI-based object
   //
   // Note: Payload must be copied if not read immediately!
   //

   class FIMessage
   {

      public:
         // member variables
         uint16_t        u16ServiceID;
         tFIVersionInfo  fiVersion;
         uint16_t        u16FunctionID;
         uint8_t         u8OpCode;
         uint16_t        u16ResponseID;
         uint8_t*        pPayload;
         uint32_t        u32PayloadLength;

         /* constructor */ FIMessage (uint16_t serviceId, const tFIVersionInfo& version
              , uint16_t functionId, uint8_t opCode, uint16_t responseID
              , uint8_t* pPayload, uint32_t length);
         /* constructor */ FIMessage (const FunctionInfo& fctInfo, uint16_t responseID
              , uint8_t* pPayload, uint32_t length);
         /* copy constructor */ FIMessage (const FIMessage& rhs);

         virtual /* inline destructor */ ~FIMessage ()  { }

         // use compiler-generated, all-member-copy assignment operator=()

         // create and destroy local payload copy
         FIMessage DeepCopy (void) const;
         void vDestroy (void);

         bool bPopulateFIData (fi_tclMessageBase& oFiData) const;

         inline ServiceInfo oServiceInfo (void) const
            {  return ServiceInfo(u16ServiceID, fiVersion);  }

         inline FunctionInfo oFunctionInfo (void) const
            {  return FunctionInfo(oServiceInfo(), u16FunctionID, u8OpCode);  }
   };

   // ==========================================================================
   //
   //                 E m p t y   fi_tclMessage   class
   //
   // helper class to generate any sendable FI message without payload
   //

   class fi_tclEmptyMsg  : public fi_tclTypeBase, public fi_tclMessageBase
   {
      public:
         const FunctionInfo _info;

         inline /* constructor */ fi_tclEmptyMsg (FunctionInfo info);
         virtual /* destructor */ ~fi_tclEmptyMsg ();

         // fi_tclMessageBase implementation
         virtual tU16 u16GetServiceID () const;
         virtual tU16 u16GetFunctionID () const;
         virtual tU8  u8GetOpCode () const;
         virtual const fi_tclTypeBase& corfoGetTypeBase () const;
         virtual fi_tclTypeBase& rfoGetTypeBase ();

         // fi_tclTypeBase implementation
         virtual tU32 u32GetSize (tU16 u16MajorVersion) const;
         virtual fi_tclOutContext& oWrite (fi_tclOutContext& oOut) const;
         virtual fi_tclInContext& oRead (fi_tclInContext& oIn);
         virtual tS32 s32GetTypeId () const;
         virtual fi_tclIOContextBase::tenByteOrder enGetByteOrder() const;
   };

   // - - - - - - - - - - - - - - -

   inline /* constructor */ fi_tclEmptyMsg:: fi_tclEmptyMsg (FunctionInfo info)
      : fi_tclTypeBase()
      , fi_tclMessageBase()
      , _info(info)
   {

   }

   // --------------------------------------------------------------------------
   //
   //              Binary   F I - M e s s a g e   class
   //
   // helper class to generate any sendable FI message with binary payload
   //

   class fi_tclBinaryMsg  : public fi_tclEmptyMsg
   {
      private:
         FunctionInfo _info;
         tU8*         _pLocalPayload;
         tU8*         _pPayload;
         uint32_t     _payloadSize;

      public:
         inline /* constructor */ fi_tclBinaryMsg (FunctionInfo info, tU8* pPayload, uint32_t payloadSize);
         virtual /* destructor */ ~fi_tclBinaryMsg ();

         // fi_tclTypeBase implementation
         virtual tU32 u32GetSize (tU16 u16MajorVersion) const;
         virtual tVoid vDestroy ();
         virtual fi_tclOutContext& oWrite (fi_tclOutContext& oOut) const;
         virtual fi_tclInContext& oRead (fi_tclInContext& oIn);
    };

   // - - - - - - - - - - - - - - -

   inline /* constructor */ fi_tclBinaryMsg:: fi_tclBinaryMsg (FunctionInfo info, tU8* pPayload, uint32_t payloadSize)
      : fi_tclEmptyMsg(info)
      , _pLocalPayload(NULL)
      , _pPayload(pPayload)
      , _payloadSize(payloadSize)
   {

   }

   // --------------------------------------------------------------------------
   //
   //                  E r r o r - M e s s a g e   class
   //
   // helper class to generate a sendable FI error message
   //
   // Note: Coding is little-endian (Intel format), as usually is in all fi's except most_fi
   //

   class fi_tclErrorMsg  : public fi_tclEmptyMsg
   {
      public:
         tU16  u16ErrorCode;

         inline /* constructor */ fi_tclErrorMsg (FunctionInfo info, tU16 u16ErrorCode);
         inline /* constructor */ fi_tclErrorMsg (FunctionInfo info);
         virtual /* destructor */ ~fi_tclErrorMsg ();

         // fi_tclTypeBase implementation
         virtual tU32 u32GetSize (tU16 u16MajorVersion) const;
         virtual fi_tclOutContext& oWrite (fi_tclOutContext& oOut) const;
         virtual fi_tclInContext& oRead (fi_tclInContext& oIn);
    };

   // - - - - - - - - - - - - - - -

   inline /* constructor */ fi_tclErrorMsg:: fi_tclErrorMsg (FunctionInfo info, tU16 errorCode)
      : fi_tclEmptyMsg(info)
      , u16ErrorCode(errorCode)
   {

   }

   // - - - - - - - - - - - - - - -

   inline /* constructor */ fi_tclErrorMsg:: fi_tclErrorMsg (FunctionInfo info)
      : fi_tclEmptyMsg(info)
      , u16ErrorCode(0)  /* CCA_C_U16_ERROR_UNSPECIFIED */
   {

   }

   // --------------------------------------------------------------------------

}  // namespace FIMessaging




#endif /* MESSAGETYPES_H_ */
