#include <glib/gi18n.h>
#include <glib.h>
#include <alsa/asoundlib.h>
#include "apl.h"
#include "apl-main.h"
#include "apl-internal.h"
#include "apl-pcm-copy.h"


#define APL_PCM_COPY_INSTANCE_NB_MAX 10


typedef struct _mydata
{
    tAplI32                 this_switch;
    tAplU8                  mod_inst;
    aplAudioSamples_desc*   buf_desc;
    gboolean                create_own_stream_buffer;
    gboolean                can_swap;

    /* module parameter */
    tAplI32                 mparam_aplPcmCopySwitch;            /*  0x0200   WB,(tAplI32)*/
    tAplBindingMatrix       mparam_aplAudioChannelBinding;
    tAplU16                 mparam_aplFrameShiftOut;            /*  20       WB,(tAplU16)*/
    tAplU16                 mparam_aplSampleRateOut;            /*  21       WB,(tAplU16)*/
    tAplU16                 mparam_aplChannelCntOut;            /*  22       WB,(tAplU16)*/
    tAplU16                 mparam_aplAudioPcmFormatOut;        /*  23       WB,(tAplU16)*/

    /* audio recording */
    aplrecord_desc          apl_rec_mod_desc;

}mydata;


LOCAL module_desc*      module_inst_list[APL_PCM_COPY_INSTANCE_NB_MAX] = {0};
LOCAL tAplU8            inst_count = 0;


LOCAL const char  *D_APL_RECORD_PROCESS_MODULE_FILE_OUT_PCM = "/tmp/apl_proc_rec_out_mod_pcm_copy.pcm";

/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_get_next_module_instance
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL module_desc* apl_pcmcopy_get_next_module_instance(void)
{
    module_desc*  this_modul_desc = (module_desc*)NULL ;

    pr_debug("ENTERED\n");


    if(inst_count < APL_PCM_COPY_INSTANCE_NB_MAX )
    {
       this_modul_desc = g_malloc0(sizeof(module_desc));

        if(this_modul_desc)
        {
            module_inst_list[inst_count] = this_modul_desc;
            inst_count++;

            this_modul_desc->inst_nb           = inst_count;
            this_modul_desc->associated_switch = aplPcmCopySwitch;

            pr_message("create a new instamce(%d) of the PCM_COPY module\n", inst_count);

            return this_modul_desc;
        }
    }

    return (module_desc*)NULL;
}

#if 0
/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_get_module_instance
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL module_desc* apl_pcmcopy_get_module_instance(tAplU8 inst)
{
    module_desc*  this_modul_desc = (module_desc*)NULL ;

    pr_debug("ENTERED\n");


    if(    (inst > 0)
        && (inst <= APL_PCM_COPY_INSTANCE_NB_MAX ))
    {

        this_modul_desc = module_inst_list[inst - 1];

        if(this_modul_desc)
        {
            pr_debug("found instamce(%d) of the PCM_COPY module", inst_count);
            return this_modul_desc;
        }
    }

    return (module_desc*)NULL;
}
#endif
/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_prepare
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_pcmcopy_prepare(void* data, void* mdata)//,  const tAplCfg* paplCfg)
{
    unsigned int        i = 0;

    pr_debug("ENTERED\n");

    struct aplMainStruct*   aplh = (struct aplMainStruct*)data;
    mydata*                 this_mdata = (mydata*)mdata;



    if(!aplh)
    {
        pr_debug("apl instance structure not available \n");
        return APL_ERR_NULL_POINTER;
    }

    if(!this_mdata)
    {
        pr_debug("no data available \n");
        return APL_ERR_NULL_POINTER;
    }


    if(!this_mdata->buf_desc)
    {
        pr_debug(" module stream buffer descriptor not available \n");
        return APL_ERR_NULL_POINTER;
    }
    /* initialize pcmcopy intern data  */


    /* initialize pcmcopy stream buffer */
    this_mdata->buf_desc->apl_stream_nb_ch    = this_mdata->mparam_aplChannelCntOut; //aplh->par_aplChannelCntOut;
    this_mdata->buf_desc->apl_audio_format    = (snd_pcm_format_t)this_mdata->mparam_aplAudioPcmFormatOut; //aplh->par_aplAudioPcmFormatOut;
    this_mdata->buf_desc->apl_sample_rate     = this_mdata->mparam_aplSampleRateOut; //aplh->par_aplSampleRateOut;
    this_mdata->buf_desc->apl_period_size     = this_mdata->mparam_aplFrameShiftOut; //aplh->par_aplFrameShiftOut;


    if(this_mdata->mparam_aplAudioChannelBinding.audbind)
    {
        pr_message("create own module buffer\n");
        this_mdata->create_own_stream_buffer = aplTRUE;
        this_mdata->buf_desc->apl_can_swap = FALSE;
    }

    if(this_mdata->create_own_stream_buffer)
    {
        if(!this_mdata->buf_desc->apl_stream_nb_ch)
            return APL_ERR_INVALID_CHANNEL_NUMBER;

        /* create streaming buffer */
        for(i = 0; i < this_mdata->buf_desc->apl_stream_nb_ch; i++)
        {
            if(this_mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE)
            {
                pr_debug("create s16 stream buffer[%d] with size(%d)\n", i, this_mdata->buf_desc->apl_period_size);
                this_mdata->buf_desc->s16audiosamples[i] = g_malloc0(this_mdata->buf_desc->apl_period_size * sizeof(short));
            }
            else if(this_mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            {
                pr_message("create s32 stream buffer[%d] with size(%d)\n", i, this_mdata->buf_desc->apl_period_size);
                this_mdata->buf_desc->s32audiosamples[i] = g_malloc0(this_mdata->buf_desc->apl_period_size * sizeof(long));
            }
            else
            {
                pr_debug("streaming format mismatch \n");
                return APL_ERR_AUDIO_STREAM_FORMAT_MISMATCH;
            }
        }
    }


    /* prepare audio recording */
    if(aplh->apl_internal_rec_mode_select == 4)
    {
        this_mdata->apl_rec_mod_desc.out_rec_active = aplTRUE;

        pr_debug("******* recorder descriptor module(%d) ********\n", aplPcmCopySwitch);
        pr_debug("create recording file : %s\n", D_APL_RECORD_PROCESS_MODULE_FILE_OUT_PCM);

        /* create output test file */
        remove(this_mdata->apl_rec_mod_desc.file_name);
        this_mdata->apl_rec_mod_desc.file_out_pcm = fopen (D_APL_RECORD_PROCESS_MODULE_FILE_OUT_PCM,"a");
        if(!this_mdata->apl_rec_mod_desc.file_out_pcm)
        {
            pr_warning("fails to create file %s\n", D_APL_RECORD_PROCESS_MODULE_FILE_OUT_PCM);
            return APL_ERR_NULL_POINTER;
        }

        fseek(this_mdata->apl_rec_mod_desc.file_out_pcm, 0, SEEK_SET);
    }

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_command
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_pcmcopy_command(void* data, tAplCmd cmd)
{
    pr_debug("ENTERED\n");

    if(!data)
    {
        pr_debug("module data pointer is NULL\n");
        return APL_ERR_NULL_POINTER;
    }

    mydata* mdata = (mydata*)data;

    if(    (cmd == aplCmdResetRecording)
        && mdata->apl_rec_mod_desc.out_rec_active)
    {
        /* close */
        remove(D_APL_RECORD_PROCESS_MODULE_FILE_OUT_PCM);
        if(mdata->apl_rec_mod_desc.file_out_pcm)
        {
            fclose(mdata->apl_rec_mod_desc.file_out_pcm);
            mdata->apl_rec_mod_desc.file_out_pcm = NULL;
        }

        /* recreate output test file */
        mdata->apl_rec_mod_desc.file_out_pcm = fopen (D_APL_RECORD_PROCESS_MODULE_FILE_OUT_PCM,"a");
        if(!mdata->apl_rec_mod_desc.file_out_pcm)
            return APL_ERR_NULL_POINTER;

        fseek(mdata->apl_rec_mod_desc.file_out_pcm, 0, SEEK_SET);
    }


    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_get_buffer
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL aplAudioSamples_desc* apl_pcmcopy_get_buffer(void* data)
{
    pr_debug("ENTERED\n");

    if(!data)
    {
        pr_debug("module data pointer is NULL\n");
        return (aplAudioSamples_desc*)NULL;
    }

    mydata* mdata = (mydata*)data;

  /* currently only one instance supported */
    if(mdata->this_switch != aplPcmCopySwitch)
    {
        pr_debug("copy module not active\n");
        return (aplAudioSamples_desc*)NULL;
    }
    return mdata->buf_desc;
}
/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_final
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_pcmcopy_finalize(void* data)
{
    unsigned int        i = 0;

    pr_debug("ENTERED\n");

    if(!data)
    {
        pr_debug("module data pointer is NULL\n");
        return APL_ERR_MODULE_NOT_INITIALIZED;
    }

    mydata*     mdata = (mydata*)data;

    if(mdata->buf_desc)
    {

        if(mdata->create_own_stream_buffer)
        {
            /* delete streaming buffer */
            for(i = 0; i < mdata->buf_desc->apl_stream_nb_ch; i++)
            {
                if(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE)
                {
                    short int* s16buf = mdata->buf_desc->s16audiosamples[i];
                    if(s16buf)
                        g_free(s16buf);

                }
                else if(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE)
                {
                    long int* s32buf = mdata->buf_desc->s32audiosamples[i];
                    if(s32buf)
                        g_free(s32buf);
                }
            }
        }

        g_free(mdata->buf_desc);
        mdata->buf_desc = NULL;
    }

    g_free(mdata);

    return APL_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_get_property
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS   apl_pcmcopy_get_property
(
    void* data,
    tAplDataID iDataID,
    tAplU32* pulSize,
    void* pData
)
{
    tAplI32 i32value;

    pr_debug("ENTERED\n");

    if(!data)
        return APL_ERR_MODULE_NOT_INITIALIZED;

    mydata* mdata = (mydata*)data;

    if(mdata->this_switch != aplPcmCopySwitch)
        return APL_ERR_MODULE_NOT_INITIALIZED;

    switch(iDataID)
    {
        case aplPcmCopySwitch:
        {
            *pulSize = sizeof(tAplI32);
            i32value = mdata->mparam_aplPcmCopySwitch;
            memcpy(pData, (void*)&i32value, *pulSize);
            break;
        }
        case aplAudioChannelBinding:
        default:
            break;
    }

    return APL_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_set_property
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL   APL_STATUS   apl_pcmcopy_set_property
(
    void* data,
    tAplInt iDataID,
    tAplU32 ulSize,
    const void* pData
)
{
    pr_debug("ENTERED\n");

    if(!data)
    {
        pr_debug("the PCM_COPY mod. is not initialized\n");
        return APL_ERR_MODULE_NOT_INITIALIZED;
    }

    mydata* mdata = (mydata*)data;
    if(mdata->this_switch != aplPcmCopySwitch)
    {
        pr_debug("the PCM_COPY mod. is not initialized\n");
        return APL_ERR_MODULE_NOT_INITIALIZED;
    }


    switch(iDataID)
    {
        case aplPcmCopySwitch:
        {
            pr_debug("set parameter aplPcmCopySwitch\n");
            memcpy((void*)&(mdata->mparam_aplPcmCopySwitch), pData , ulSize);
            break;
        }

        case aplPcmCopyFrameShiftOut:
        {
            pr_debug("set parameter aplPcmCopyFrameShiftOut\n");
            memcpy((void*)&mdata->mparam_aplFrameShiftOut, pData , ulSize);
            break;
        }
        case aplPcmCopyampleRateOut:
        {
            pr_debug("set parameter aplPcmCopyampleRateOut\n");
            memcpy((void*)&mdata->mparam_aplSampleRateOut, pData , ulSize);
            break;
        }
        case aplPcmCopyChannelCntOut:
        {
            pr_debug("set parameter aplPcmCopyChannelCntOut\n");
            memcpy((void*)&mdata->mparam_aplChannelCntOut, pData , ulSize);
            break;
        }
        case aplPcmCopyAudioPcmFormatOut:
        {
            pr_debug("set parameter aplPcmCopyAudioPcmFormatOut\n");
            memcpy((void*)&mdata->mparam_aplAudioPcmFormatOut, pData , ulSize);
            break;
        }

        case aplAudioChannelBinding:
        {
            if(mdata->mparam_aplAudioChannelBinding.audbind)
                g_free(mdata->mparam_aplAudioChannelBinding.audbind);

            mdata->mparam_aplAudioChannelBinding.audbind = g_malloc0(ulSize);
            if(mdata->mparam_aplAudioChannelBinding.audbind)
            {
                pr_debug("set parameter aplAudioChannelBinding\n");
                memcpy((void*)&mdata->mparam_aplAudioChannelBinding.audbind, pData , ulSize);

                if(ulSize >= 2)
                {
                  unsigned int bind_cnt = ulSize / 2;
                  unsigned int i;
                    for(i = 0; i <  bind_cnt; i++)
                    {
                        pr_debug("mdata->mparam_aplAudioChannelBinding.audbind[%d].in:%d\n", i, mdata->mparam_aplAudioChannelBinding.audbind[i].in);
                        pr_debug("mdata->mparam_aplAudioChannelBinding.audbind[%d].out:%d\n", i, mdata->mparam_aplAudioChannelBinding.audbind[i].out);
                    }
                }
            }
            break;
        }
        case aplInRoutingChannelCnt:
        {
            pr_debug("set parameter aplInRoutingChannelCnt\n");
            memcpy((void*)&mdata->mparam_aplAudioChannelBinding.inchannelnb, pData , ulSize);
            break;
        }
        case aplOutRoutingChannelCnt:
        {
            pr_debug("set parameter aplOutRoutingChannelCnt\n");
            memcpy((void*)&mdata->mparam_aplAudioChannelBinding.outchannelnb, pData , ulSize);
            break;
        }
        case aplNbBinding:
        {
            pr_debug("set parameter aplNbBinding\n");
            memcpy((void*)&mdata->mparam_aplAudioChannelBinding.bindingnb, pData , ulSize);
            break;
        }
        default:
            break;
    }

    return APL_ERR_OK;
}

/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_buffer_reconfig
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

LOCAL APL_STATUS apl_pcmcopy_buffer_reconfig
(
    const mydata*      data,
    unsigned int     new_size,
    snd_pcm_format_t    audio_format
)
{
    unsigned int        i = 0;
    unsigned int     old_buf_size = 0;

    pr_message("ENTERED\n");

    old_buf_size = data->buf_desc->apl_period_size;


    if(!data->buf_desc->apl_can_swap)
    {
        for(i = 0; i < data->buf_desc->apl_stream_nb_ch; i++)
        {
            if(data->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE)
            {
                pr_warning("recreate s16 stream buffer[%d] from size(%d) to size(%d)\n", i, old_buf_size, new_size);

                if(data->buf_desc->s16audiosamples[i])
                {
                    pr_message("free buffer channel -> %d \n", i);
                    pr_message("create_own_stream_buffer -> %d \n", data->create_own_stream_buffer);
                    pr_message("apl_can_swap -> %d \n", data->buf_desc->apl_can_swap);
                    g_free(data->buf_desc->s16audiosamples[i]);
                    data->buf_desc->s16audiosamples[i] = (short int*)NULL;
                }

                data->buf_desc->s16audiosamples[i] = g_malloc0(new_size * sizeof(short));

            }
            else if(data->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            {
                pr_warning("recreate s32 stream buffer[%d] from size(%d) to size(%d)\n", i, old_buf_size, new_size);

                if(data->buf_desc->s32audiosamples[i])
                {
                    pr_message("free buffer channel -> %d \n", i);
                    pr_message("create_own_stream_buffer -> %d \n", data->create_own_stream_buffer);
                    pr_message("apl_can_swap -> %d \n", data->buf_desc->apl_can_swap);
                    g_free(data->buf_desc->s32audiosamples[i]);
                    data->buf_desc->s32audiosamples[i] = (long int*)NULL;
                }

                data->buf_desc->s32audiosamples[i] = g_malloc0(new_size * sizeof(long));
                pr_message("create new buffer channel -> %d of size -> %d \n", i, new_size);

            }
            else
            {
                pr_warning("streaming format mismatch \n");
                return APL_ERR_AUDIO_STREAM_FORMAT_MISMATCH;
            }
        }
    }

    /* store new buffer size */
    data->buf_desc->apl_period_size = new_size;

    (void)audio_format;

    pr_message("EXIT\n");
    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_s16AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_pcmcopy_s16AudioTransfert
(
    mydata*                         data,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio
)
{

    pr_debug("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;

    if(!data)
        return APL_ERR_NULL_POINTER ;

    outAudio->err = APL_ERR_OK;

    if(data->mparam_aplAudioChannelBinding.audbind)
    {
        outAudio->isBinding = aplTRUE;
        pr_debug("binding is configured\n");
    }

    outAudio->s16ibuf = inAudio->s16audiosamples[0];
    if(!outAudio->s16ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    if(   (outAudio->apl_period_size <  inAudio->apl_period_size)
        ||(outAudio->apl_period_size >  inAudio->apl_period_size))
    {
        pr_warning("mismatch in size out_stream(%d) and in_stream(%d), reconfigure is called \n", outAudio->apl_period_size, inAudio->apl_period_size);
        outAudio->err = apl_pcmcopy_buffer_reconfig(data, inAudio->apl_period_size, SND_PCM_FORMAT_S16_LE );
        if(outAudio->err)
        {
          pr_debug("apl_pcmcopy_buffer_reconfig fails with %d\n", outAudio->err);
          return outAudio->err;
        }
    }

    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       outAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       outAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for(outAudio->i = 0; outAudio->i < outAudio->stream_cnt ; outAudio->i++)
    {
        if(outAudio->apl_period_size ==  inAudio->apl_period_size)
        {
            outAudio->ich = outAudio->i;
            outAudio->och = outAudio->i;

            /* here take care of binding */
            if(    outAudio->isBinding
                && data->mparam_aplAudioChannelBinding.bindingnb > outAudio->i)
            {
                outAudio->ich = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].in;
                if(outAudio->ich > (inAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit ich(%d) to idx(%d)",outAudio->ich, outAudio->i);
                    outAudio->ich = outAudio->i;
                }
                outAudio->och = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].out;
                if(outAudio->och > (outAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit och(%d) to idx(%d)",outAudio->och, outAudio->i);
                    outAudio->och = outAudio->i;
                }
                pr_debug("binding in_ch_%d with out_ch_%d\n", outAudio->ich, outAudio->och);
            }

            outAudio->sample_cnt = inAudio->apl_period_size;

            if(outAudio->apl_can_swap)
            {
                pr_debug ("swap stream buffer[%d]\n", outAudio->i);

                outAudio->s16ibuf = inAudio->s16audiosamples[outAudio->i];
                outAudio->s16audiosamples[outAudio->i] = outAudio->s16ibuf;
            }
            else
            {
                pr_debug("copy %d s16 sample to out stream s16 buffer[%d]\n", outAudio->sample_cnt, outAudio->i);

                outAudio->s16ibuf = inAudio->s16audiosamples[outAudio->i];
                outAudio->s16obuf = outAudio->s16audiosamples[outAudio->i];
                memcpy((void*)outAudio->s16obuf, (void*)outAudio->s16ibuf, outAudio->sample_cnt * sizeof(short int));
            }
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_is16os32AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

LOCAL APL_STATUS apl_pcmcopy_is16os32AudioTransfert
(
    mydata*                         data,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio
)
{
    pr_debug("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;

    if(!data)
        return APL_ERR_NULL_POINTER ;


    if(data->mparam_aplAudioChannelBinding.audbind)
    {
        outAudio->isBinding = aplTRUE;
        pr_debug("binding is configured\n");
    }

    outAudio->err = APL_ERR_OK;

    outAudio->s16ibuf = inAudio->s16audiosamples[0];
    if(!outAudio->s16ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    if(     (outAudio->apl_period_size <  inAudio->apl_period_size)
        ||  (outAudio->apl_period_size >  inAudio->apl_period_size))
    {
        pr_warning("mismatch in size out_stream(%d) and in_stream(%d), reconfigure is called \n", outAudio->apl_period_size, inAudio->apl_period_size);
        outAudio->err = apl_pcmcopy_buffer_reconfig(data, inAudio->apl_period_size, SND_PCM_FORMAT_S32_LE );
        if(outAudio->err)
        {
          pr_debug("apl_pcmcopy_buffer_reconfig fails with %d\n", outAudio->err);
          return outAudio->err;
        }
    }


    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       outAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       outAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( outAudio->i = 0; outAudio->i < outAudio->stream_cnt ; outAudio->i++)
    {
        if(outAudio->apl_period_size ==  inAudio->apl_period_size)
        {
            /* here take care of binding */
            outAudio->ich = outAudio->i;
            outAudio->och = outAudio->i;

            if(    outAudio->isBinding
                && data->mparam_aplAudioChannelBinding.bindingnb > outAudio->i)
            {
                outAudio->ich = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].in;
                if(outAudio->ich > (inAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit ich(%d) to idx(%d)",outAudio->ich, outAudio->i);
                    outAudio->ich = outAudio->i;
                }
                outAudio->och = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].out;
                if(outAudio->och > (outAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit och(%d) to idx(%d)",outAudio->och, outAudio->i);
                    outAudio->och = outAudio->i;
                }
                pr_debug("binding in_ch_%d with out_ch_%d\n", outAudio->ich, outAudio->och);
            }

            outAudio->sample_cnt = inAudio->apl_period_size;

            pr_debug("copy %d s16 sample to stream s32 buffer[%d]\n", outAudio->sample_cnt, outAudio->i);

            outAudio->s16ibuf = inAudio->s16audiosamples[outAudio->i];
            outAudio->s32obuf = outAudio->s32audiosamples[outAudio->i];

            for (outAudio->j = 0; outAudio->j < outAudio->sample_cnt; outAudio->j++)
                *outAudio->s32obuf++ = APL_PCM_COPY_CONVERT_S16_S32 * ((long int)*outAudio->s16ibuf++);
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_s32AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

LOCAL APL_STATUS apl_pcmcopy_s32AudioTransfert
(
    mydata*                         data,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio
)
{

    pr_debug("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;

    if(!data)
        return APL_ERR_NULL_POINTER ;


    if(data->mparam_aplAudioChannelBinding.audbind)
    {
        outAudio->isBinding = aplTRUE;
        pr_debug("binding is configured\n");
    }

    outAudio->err = APL_ERR_OK;

    outAudio->s32ibuf = inAudio->s32audiosamples[0];
    if(!outAudio->s32ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    if(    (outAudio->apl_period_size <  inAudio->apl_period_size)
        || (outAudio->apl_period_size >  inAudio->apl_period_size))
    {
        pr_warning("mismatch in size out_stream(%d) and in_stream(%d), reconfigure is called \n", outAudio->apl_period_size, inAudio->apl_period_size);
        outAudio->err = apl_pcmcopy_buffer_reconfig(data, inAudio->apl_period_size, SND_PCM_FORMAT_S32_LE );
        if(outAudio->err)
        {
            pr_debug("apl_pcmcopy_buffer_reconfig fails with %d\n", outAudio->err);
            return outAudio->err;
        }
    }


    /***
     * in stream and out stream are aligned
     * Considere only if no binding is configured
     */
    if(!outAudio->isBinding)
    {
        if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
            ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
        {
            outAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
        }
        else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
        {
            outAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
            pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
        }
    }

    for( outAudio->i = 0; outAudio->i < outAudio->stream_cnt ; outAudio->i++)
    {
        if( outAudio->apl_period_size ==  inAudio->apl_period_size)
        {
            outAudio->ich = outAudio->i;
            outAudio->och = outAudio->i;

            /* here take care of binding */
            if(    outAudio->isBinding
                && data->mparam_aplAudioChannelBinding.bindingnb > outAudio->i)
            {
                outAudio->ich = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].in;
                if(outAudio->ich > (inAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit ich(%d) to idx(%d)",outAudio->ich, outAudio->i);
                    outAudio->ich = outAudio->i;
                }
                outAudio->och = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].out;
                if(outAudio->och > (outAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit och(%d) to idx(%d)",outAudio->och, outAudio->i);
                    outAudio->och = outAudio->i;
                }
                pr_debug("binding in_ch_%d with out_ch_%d\n", outAudio->ich, outAudio->och);
            }

            outAudio->sample_cnt = inAudio->apl_period_size;

            if(outAudio->apl_can_swap)
            {
                pr_debug ("swap stream buffer[%d]\n", outAudio->ich);

                outAudio->s32ibuf = inAudio->s32audiosamples[outAudio->ich];
                outAudio->s32audiosamples[outAudio->och] = outAudio->s32ibuf;
            }
            else
            {
                pr_debug("copy %d s32 sample from in_stream_s32_buffer[%d] to out_stream_s32_buffer[%d]\n", outAudio->sample_cnt, outAudio->ich, outAudio->och);

                outAudio->s32ibuf = inAudio->s32audiosamples[outAudio->ich];
                outAudio->s32obuf = outAudio->s32audiosamples[outAudio->och];
                memcpy((void*)outAudio->s32obuf, (void*)outAudio->s32ibuf, outAudio->sample_cnt * sizeof(long int));
            }
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return APL_ERR_OK;
}


/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_is32os16AudioTransfert
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

LOCAL APL_STATUS apl_pcmcopy_is32os16AudioTransfert
(
    mydata*                         data,
    const aplAudioSamples_desc*     inAudio,
    aplAudioSamples_desc*           outAudio
)
{

    pr_debug("ENTERED\n");

    if(!inAudio)
        return APL_ERR_NULL_POINTER ;

    if(!outAudio)
        return APL_ERR_NULL_POINTER ;

    if(!data)
        return APL_ERR_NULL_POINTER ;

    outAudio->err = APL_ERR_OK;

    if(data->mparam_aplAudioChannelBinding.audbind)
    {
        outAudio->isBinding = aplTRUE;
        pr_debug("binding is configured\n");
    }

    if(!inAudio->s32audiosamples[0])
    {
        pr_debug("inAudio->s32audiosamples[0] is NULL \n");
        return APL_ERR_NULL_POINTER;
    }

    outAudio->s32ibuf = inAudio->s32audiosamples[0];
    if(!outAudio->s32ibuf)
    {
        pr_debug("No input data stream buffer available \n");
        return APL_ERR_NULL_POINTER;
    }

    /* in stream and out stream are aligned */
    if(   (outAudio->apl_stream_nb_ch == inAudio->apl_stream_nb_ch)
        ||(outAudio->apl_stream_nb_ch > inAudio->apl_stream_nb_ch ))
    {
       outAudio->stream_cnt =  inAudio->apl_stream_nb_ch;
    }
    else if(outAudio->apl_stream_nb_ch < inAudio->apl_stream_nb_ch)
    {
       outAudio->stream_cnt =  outAudio->apl_stream_nb_ch;
       pr_warning(" mismatch nb channel out_stream(%d) and in_stream(%d)\n",outAudio->apl_stream_nb_ch, inAudio->apl_stream_nb_ch );
    }

    for( outAudio->i = 0; outAudio->i < outAudio->stream_cnt ; outAudio->i++)
    {
        if(   (outAudio->apl_period_size ==  inAudio->apl_period_size)
            ||(outAudio->apl_period_size >  inAudio->apl_period_size))
        {
            /* here take care of binding */
            outAudio->ich = outAudio->i;
            outAudio->och = outAudio->i;

            /* here take care of binding */
            if(    outAudio->isBinding
                && data->mparam_aplAudioChannelBinding.bindingnb > outAudio->i)
            {
                outAudio->ich = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].in;
                if(outAudio->ich > (inAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit ich(%d) to idx(%d)",outAudio->ich, outAudio->i);
                    outAudio->ich = outAudio->i;
                }
                outAudio->och = data->mparam_aplAudioChannelBinding.audbind[outAudio->i].out;
                if(outAudio->och > (outAudio->apl_stream_nb_ch-1))
                {
                    pr_debug("limit och(%d) to idx(%d)",outAudio->och, outAudio->i);
                    outAudio->och = outAudio->i;
                }
                pr_debug("binding in_ch_%d with out_ch_%d\n", outAudio->ich, outAudio->och);
            }

            outAudio->sample_cnt = inAudio->apl_period_size;

            pr_debug("copy %d s32 sample to stream s16 buffer[%d]\n", outAudio->sample_cnt, outAudio->i);

            outAudio->s32ibuf = inAudio->s32audiosamples[outAudio->i];
            outAudio->s16obuf = outAudio->s16audiosamples[outAudio->i];
            for (outAudio->j = 0; outAudio->j < outAudio->sample_cnt; outAudio->j++)
                *outAudio->s16obuf++ = (short int)*outAudio->s32ibuf++;
        }
        else
        {
            pr_critical(" mismatch between nb of in_stream(%d) and out_strem(%d) samples \n", inAudio->apl_period_size, outAudio->apl_period_size);
        }
    }

    return APL_ERR_OK;
}



/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_process
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL APL_STATUS apl_pcmcopy_process
(
    void* data,
    const aplAudioSamples_desc* inAudio
)
{

    APL_STATUS status = APL_ERR_OK;
    gboolean can_swap  = TRUE;
    short int*  s16buf = (short int*)NULL;
    long int*  s32buf = (long int*)NULL;
    tAplUInt   data_size = 0;


    pr_debug("ENTERED\n");


    if(!data)
        return APL_ERR_MODULE_NOT_INITIALIZED;

    if(!inAudio)
    {
        pr_debug(" input buffer descriptor not available\n");
        return APL_ERR_NULL_POINTER;
    }


    module_desc*            this_module = (module_desc*)data;
    mydata*                 mdata = this_module->data;

    //if(mdata->create_own_stream_buffer)
    //    can_swap  = FALSE;

    if(   (inAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        &&(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE))
    {
        status = apl_pcmcopy_s16AudioTransfert( mdata, inAudio, mdata->buf_desc);
    }
    else if(  (inAudio->apl_audio_format == SND_PCM_FORMAT_S16_LE)
            &&(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE))
    {
        status = apl_pcmcopy_is16os32AudioTransfert(mdata, inAudio, mdata->buf_desc);
    }
    else if(  (inAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            &&(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE))
    {
        status = apl_pcmcopy_s32AudioTransfert(mdata, inAudio, mdata->buf_desc);
    }
    else if(  (inAudio->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            &&(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE))
    {
        pr_warning("there a lost of data, s32_in -> s16_out\n");
        status = apl_pcmcopy_is32os16AudioTransfert(mdata, inAudio, mdata->buf_desc);
    }
    else
    {
        /* do nothing */
        pr_warning("inAudio->apl_audio_format = %d, mdata->buf_desc->apl_audio_format = %d \n", inAudio->apl_audio_format, mdata->buf_desc->apl_audio_format);
    }

    /* audio recording */
    if(mdata->apl_rec_mod_desc.out_rec_active)
    {
        if(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE)
        {
            s16buf = mdata->buf_desc->s16audiosamples[0];
            data_size = mdata->buf_desc->apl_period_size * 2;
        }
        else if(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE)
        {
            s32buf = mdata->buf_desc->s32audiosamples[0];
            data_size = mdata->buf_desc->apl_period_size * 4;
        }
        else
        {
          /* do nothing */
        }

        if(!data_size)
        {
            pr_warning("number of byte to copy is NULL \n");
            return APL_ERR_OK;
        }


        if(mdata->apl_rec_mod_desc.file_out_pcm)
        {
            if(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S16_LE)
            {
                if(!s16buf)
                {
                    pr_warning("NULL POINTER audio daten buffer\n");
                    return APL_ERR_NULL_POINTER;
                }
                else
                    mdata->apl_rec_mod_desc.wdata = fwrite((void*)s16buf, (size_t)1, data_size, mdata->apl_rec_mod_desc.file_out_pcm);
            }

            else if(mdata->buf_desc->apl_audio_format == SND_PCM_FORMAT_S32_LE)
            {
                if(!s32buf)
                {
                    pr_warning("NULL POINTER audio daten buffer\n");
                    return APL_ERR_NULL_POINTER;
                }
                else
                    mdata->apl_rec_mod_desc.wdata = fwrite((void*)s32buf, (size_t)1, data_size, mdata->apl_rec_mod_desc.file_out_pcm);
            }
            else
            {
                pr_warning("data format(%d) not supported for recording\n", mdata->buf_desc->apl_audio_format );
            }
        }
    }

    (void)can_swap;

    return status;
}

/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_notification
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
LOCAL tAplBool apl_pcmcopy_notification
(
    void*       data,
    tAplInt*    piDataID,
    tAplAction* paplAct
)
{
    pr_debug("No implmentation available\n");

    (void)data;
    (void)piDataID;
    (void)paplAct;

    return aplFALSE;
}




/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_initialize
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT module_desc* apl_pcmcopy_initialize(void)
{
    module_desc*  this_modul_desc = (module_desc*)NULL ;

    pr_debug("ENTERED\n");


    this_modul_desc = apl_pcmcopy_get_next_module_instance();

    //this_modul_desc = g_malloc0(sizeof(module_desc));

    if(!this_modul_desc)
    {
        pr_critical("fails while allocating memory for new module descriptor\n");
        return (module_desc*)NULL;
    }

    //this_modul_desc->associated_switch          = aplPcmCopySwitch;
    this_modul_desc->state                      = apl_state_setup;
    this_modul_desc->aplopmode                  = APL_OM_PROD_CONS;
    this_modul_desc->apl_module_command         = apl_pcmcopy_command;
    this_modul_desc->apl_module_prepare         = apl_pcmcopy_prepare;
    this_modul_desc->apl_module_process         = apl_pcmcopy_process;
    this_modul_desc->apl_module_finalize        = apl_pcmcopy_finalize;
    this_modul_desc->apl_module_get_buffer      = apl_pcmcopy_get_buffer;
    this_modul_desc->apl_module_get_property    = apl_pcmcopy_get_property;
    this_modul_desc->apl_module_set_property    = apl_pcmcopy_set_property;
    this_modul_desc->apl_module_notification    = apl_pcmcopy_notification;

    this_modul_desc->next        = (void*)NULL;
    this_modul_desc->prev        = (void*)NULL;

    mydata*  module_data_inst = g_malloc0(sizeof(mydata));

    if(!module_data_inst)
    {
        pr_critical("fails while allocating memory for new buffer descriptor\n");
        return (module_desc*)NULL;
    }

    module_data_inst->this_switch   = aplPcmCopySwitch;
    module_data_inst->mod_inst      = this_modul_desc->inst_nb;
    module_data_inst->buf_desc = g_malloc0(sizeof(aplAudioSamples_desc));
    if(!module_data_inst->buf_desc)
    {
    	pr_critical("failed to allocate memory \n");
    	return (module_desc*)NULL;
    }
        module_data_inst->buf_desc->apl_can_swap = TRUE;
    module_data_inst->create_own_stream_buffer = FALSE;

    /* initialize binding structure */
    module_data_inst->mparam_aplAudioChannelBinding.audbind = (tAplBinding*)NULL;

    /* init buffer array */
    /**/
    for(int i = 0; i < APL_MAX_STREAM_CH; i++)
    {
        module_data_inst->buf_desc->s16audiosamples[i] = (short int*)NULL;
        module_data_inst->buf_desc->s32audiosamples[i] = (long int*)NULL;
    }

    /* setup stream audio buffer in buffer descriptor
     * not required in thsi module because not new audio data are processed
     * buffer pointer s16/s32audiosamplesinput hold a pointer of the input streaming buffer
     * so it can be return with the function call get_buffer()
     */
    this_modul_desc->data = (void*)module_data_inst;

    return this_modul_desc;
}

/*******************************************************************************
*
* FUNCTION: apl_pcmcopy_is_inst_available
*
* DESCRIPTION: function to set extended data
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
EXPORT tAplBool apl_pcmcopy_is_inst_available(tAplI32 aplswitch)
{
    tAplBool status = aplFALSE;

    pr_debug("ENTERED\n");

#ifndef APL_PCM_COPY_MULTIPLE_INST_ENABLE
    if(!module_data_inst)
        return status;

    if(module_data_inst->this_switch == aplswitch)
    {
        pr_debug(" the module pcm copy is available for switch: %d\n", aplswitch);
        status =  aplTRUE;
    }
#endif

    (void)aplswitch;

    return status;
}
