/*
 * audproc-state-machine.c
 *
 *
 * Author: Patrick Rey
 * Date: 03.11.2016
 * Version 0.1 : Init
 */



#include <stdlib.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <string.h>
#include <signal.h> //new
#include <glib/gi18n.h>
#include <glib-object.h>
#include <glib-unix.h> //new
#include <glib.h>
#include <locale.h>



#include "audproc-common-defs.h"
#include "audproc-state-machine.h"





/*******************************************************************************
              Constant defintion
*******************************************************************************/

#ifdef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
typedef struct _sm_request
{
    audproc_state_machine*  sm_new_req;
    audproc_audio_state     sm_req_state;
}audproc_sm_request;

#define AUPROC_SM_MAX_REQUEST 20

#endif
/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/
//static audproc_state_machine src_ent_sm;
//static audproc_state_machine src_voice_sm;
GMutex audproc_sm_lock;
static audproc_state_machine** sm_list = (audproc_state_machine**)NULL;
static guint8 sm_inst_count = 0;
#ifdef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
GCond  audproc_sm_update;
static audproc_sm_request* sm_request_list = (audproc_sm_request*)NULL;
static guint8 sm_request_count = 0;
static guint8 sm_request_rd_pos = 0;
static guint8 sm_request_wr_pos = 0;
static gboolean    sm_request_run = TRUE;
static GThread*  audproc_sm_thread;
#endif

static const audproc_as_desc AUDPROC_AUDIO_STATE_table[] = {

 /* SSE specifical error*/
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_SILENCE),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_INIT),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_WAIT_DATA),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_RUNNING),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_WAIT_RECOVER),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_ERROR),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_TERMINATED),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_STOP),
  AUDPROC_AUDIO_STATE(AUDPROC_AS_STREAM_SILENCE_PRELOAD)
};

#define AUDPROC_AUDIO_STATE_MAX ARRAYSIZE(AUDPROC_AUDIO_STATE_table)







static gboolean audproc_service_add_new_state_element
(
    guint8                  sm_inst,
    audproc_state_element*  new_s_e
)
{
    guint i = 0;
    audproc_state_element* p;
    audproc_state_machine* sm = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    sm = audproc_state_machine_get_instance(sm_inst);

    if(!sm)
        return FALSE;


    for(i = 0; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
    {
        p = sm->state_liste[i];

        if(!p)
        {

            if(new_s_e->this_state_elem_type == AUDPROC_STATE_TYPE_MAIN_STATE)
            {
                pr_debug(" new state element with main state:%d was added at index(%d)\n", new_s_e->this_main_state,i);
            }
            else if(new_s_e->this_state_elem_type == AUDPROC_STATE_TYPE_SUB_STATE)
            {
                pr_debug(" new state element with sub state:%d was added at index(%d)\n", new_s_e->this_sub_state,i);
            }
            else
            {
                /* do nothing */
            }

            sm->state_liste[i] = new_s_e;

            return TRUE;
        }
    }

    return FALSE;

}

static audproc_state_element* audproc_state_machine_get_associated_state_elem
(
    audproc_state_element* main_state_elem,
    audproc_audio_state new_astate
)
{
    guint i = 0;
    audproc_state_element* p;


    if(!main_state_elem)
        return (audproc_state_element*)NULL;

    if(!main_state_elem->associated_state_elem_list)
    {
        pr_warning("no next element list available for main state:%d\n", main_state_elem->this_main_state);
        return (audproc_state_element*)NULL;
    }

    for(i=0; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
    {
        p = main_state_elem->associated_state_elem_list[i];
        if(p)
        {
            if(p->this_main_state == new_astate)
            {
                pr_debug("found next main state:%d element of main state:%d at index:%d \n", new_astate, main_state_elem->this_main_state, i);
                return p;
            }
        }
        else
        {
            pr_debug("no next state element found in tne main state:%d\n",main_state_elem->this_main_state);
        }
    }

    return (audproc_state_element*)NULL;

}


static audproc_state_element* audproc_service_search_this_state_elem
(
    guint8              sm_inst,
    audproc_state_type  s_type,
    guint8              this_s
)
{
    guint i = 0;
    audproc_state_element* p;
    audproc_state_machine* sm = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    sm = audproc_state_machine_get_instance(sm_inst);
    if(!sm)
        return (audproc_state_element*)NULL;


    for(i=0; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
    {
        p = sm->state_liste[i];

        if(p)
        {
            if(p->this_state_elem_type == s_type)
            {
                if(    (p->this_state_elem_type  == AUDPROC_STATE_TYPE_MAIN_STATE)
                    && (p->this_main_state       == this_s ))
                {
                    pr_debug("found state element with main_state(%d)at index(%d)\n", this_s, i);
                    return p;
                }
                else if(    (p->this_state_elem_type  == AUDPROC_STATE_TYPE_SUB_STATE)
                        &&  (p->this_sub_state        == this_s ))
                {
                    pr_debug("found state element with sub_state(%d)at index(%d)\n", this_s, i);
                    return p;
                }
            }
        }
    }

    return (audproc_state_element*)NULL;
}

static audproc_state_element* audproc_service_search_this_state_elem_p
(
    audproc_state_machine* sm,
    audproc_state_type  s_type,
    guint8              this_s
)
{
    guint i = 0;
    audproc_state_element* p;

    pr_debug("ENTERED\n");


    if(!sm)
    {
        pr_warning("unvalide state machine\n");
        return (audproc_state_element*)NULL;
    }

    for(i=0; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
    {
        p = sm->state_liste[i];

        if(p)
        {
            if(p->this_state_elem_type == s_type)
            {
                if(    (p->this_state_elem_type  == AUDPROC_STATE_TYPE_MAIN_STATE)
                    && (p->this_main_state       == this_s ))
                {
                    pr_debug("found state element with main_state(%d)at index(%d)\n", this_s, i);
                    return p;
                }
                else if(    (p->this_state_elem_type  == AUDPROC_STATE_TYPE_SUB_STATE)
                        &&  (p->this_sub_state        == this_s ))
                {
                    pr_debug("found state element with sub_state(%d)at index(%d)\n", this_s, i);
                    return p;
                }
            }
        }
    }

    return (audproc_state_element*)NULL;
}


static gboolean audproc_service_is_this_state_defined
(
    guint8              sm_inst,
    audproc_state_type  s_type,
    guint8              this_s
)
{
    guint i = 0;
    audproc_state_element* p;
    audproc_state_machine* sm = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    sm = audproc_state_machine_get_instance(sm_inst);

    if(!sm)
        return AUDPROC_ERR_SM_INST_NAME_NOT_AVAIL;


    for(i=0; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
    {
        p = sm->state_liste[i];

        if(p)
        {
            if(p->this_state_elem_type == s_type)
            {
                if(    (p->this_state_elem_type  == AUDPROC_STATE_TYPE_MAIN_STATE)
                    && (p->this_main_state       == this_s ))
                {
                    pr_debug("found state element with main_state(%d)at index(%d)\n", this_s, i);
                    return TRUE;
                }
                else if(    (p->this_state_elem_type  == AUDPROC_STATE_TYPE_SUB_STATE)
                        &&  (p->this_sub_state        == this_s ))
                {
                    pr_debug("found state element with sub_state(%d)at index(%d)\n", this_s, i);
                    return TRUE;
                }
            }
        }
    }

    return FALSE;
}




static int audproc_state_machine_add_state
(
    guint8              sm_inst,
    audproc_state_type  this_state_type,
    gint8               this_state,
    audproc_state_type  associated_state_type,
    gint8               associated_state,
    void*               fct_state_action,
    void*               fct_chg_cond,
    void*               fct_tr_act
)
{
    int err = (int)AUDPROC_ERR_OK;
    guint i = 0;
    gboolean new_state_elem = FALSE;
    gboolean state_elem_already_defined = FALSE;
    audproc_state_element* this_new_state_elem = (audproc_state_element*)NULL;
    audproc_state_element* associated_state_elem = (audproc_state_element*)NULL;
    audproc_state_element* p =  (audproc_state_element*)NULL;
    audproc_state_machine* sm = (audproc_state_machine*)NULL;

    pr_debug(" add new main state:%d, with associated state:%d\n", this_state, (guint8)associated_state);

    sm = audproc_state_machine_get_instance(sm_inst);

    if(!sm)
        return AUDPROC_ERR_SM_INST_NAME_NOT_AVAIL;

    /* check if this state is new */
    state_elem_already_defined = audproc_service_is_this_state_defined(sm_inst, AUDPROC_STATE_TYPE_MAIN_STATE, (guint8)this_state);
    if(!state_elem_already_defined)
    {
        /* create a new state element */
        this_new_state_elem = g_malloc0(sizeof(audproc_state_element));
        if(this_new_state_elem)
        {
            this_new_state_elem->this_main_state      = (guint8)this_state;
            this_new_state_elem->this_state_elem_type = AUDPROC_STATE_TYPE_MAIN_STATE;

            /* create next_main_state structure*/
            this_new_state_elem->associated_state_elem_list = g_new0(audproc_state_element*, AUD_PROC_MAX_STATE_ELEMENT);

            if(fct_state_action)
                this_new_state_elem->audproc_service_this_state_action = fct_state_action;
            else
            {
                pr_warning("no action function for main state:%d was configured\n", this_state);
            }
        }
    }
    else
        this_new_state_elem = audproc_service_search_this_state_elem(sm_inst, AUDPROC_STATE_TYPE_MAIN_STATE, (guint8)this_state);


    if(!this_new_state_elem)
    {
        pr_warning("the main state:%d is not defined\n", this_state);
        return AUDPROC_ERR_NULL_POINTER;
    }

    /* configure associated state element */
    if(associated_state != AUDPROC_AS_STREAM_UNDEFINED)
    {
        if(this_new_state_elem->associated_state_elem_list)
        {
            for(i = 0 ; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
            {
                p = this_new_state_elem->associated_state_elem_list[i];
                if(p)
                {
                    if(p->this_main_state == associated_state)
                    {
                        pr_debug("next main state:%d of main state:%d still exist at index:%d \n", associated_state, this_state, i);
                        break;
                    }
                }
                else
                {
                    /*create a new state element */
                    associated_state_elem = g_malloc0(sizeof(audproc_state_element));
                    if(associated_state_elem)
                    {
                        associated_state_elem->this_main_state = (guint8)associated_state;

                        if(fct_chg_cond)
                            associated_state_elem->audproc_service_this_state_change_cond = fct_chg_cond;
                        if(fct_tr_act)
                            associated_state_elem->audproc_service_this_state_transition_action = fct_tr_act;
                        if(this_new_state_elem->audproc_service_this_state_action)
                            associated_state_elem->audproc_service_this_state_action = this_new_state_elem->audproc_service_this_state_action;

                        this_new_state_elem->associated_state_elem_list[i] = associated_state_elem;

                        pr_debug("store next main state:%d of main state:%d at index:%d \n",associated_state, this_state, i);
                        break;
                    }
                }
            }
        }
        else
        {
            pr_warning("a next state list is missing, should be created during instance creation of a new main state \n");
        }
    }
    else
    {
        /* do nothing */
    }


    /* store new state element  */
    if(!state_elem_already_defined)
        new_state_elem = audproc_service_add_new_state_element(sm_inst, this_new_state_elem);

    (void)this_state_type;
    (void)associated_state_type;
    (void)new_state_elem;
    return err;
}





static int audproc_state_machine_trigger_new_state_internal
(
    audproc_state_machine** p
)
{
    int err = (int)AUDPROC_ERR_OK;


    if(!(*p))
    {
        pr_warning("NULL pointer\n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    pr_debug("New state change request:%d, current state is:%d\n", (*p)->requested_new_state, (*p)->current_state);

    /*todo: add critical section */

    if((*p)->current_state_elem)
    {
        if((*p)->new_state_elem)
        {

            if((*p)->new_state_elem->audproc_service_this_state_change_cond)
                err = (*p)->new_state_elem->audproc_service_this_state_change_cond((guint8)(*p)->requested_new_state, (gint*)(*p)->args, (*p)->nargs);
            if(!err)
            {
                if((*p)->new_state_elem->audproc_service_this_state_transition_action)
                {
                    err = (*p)->new_state_elem->audproc_service_this_state_transition_action((guint8)(*p)->requested_new_state, (gint*)(*p)->args, (*p)->nargs);
                }

                if(!err)
                {
                    pr_debug("now change state from %d to %d\n", (*p)->current_state, (*p)->requested_new_state);

                    /* update state now */
                    (*p)->current_state = (*p)->requested_new_state;
                }
            }
            if(!err)
            {
                if((*p)->new_state_elem->audproc_service_this_state_action)
                    err = (*p)->new_state_elem->audproc_service_this_state_action((guint8)(*p)->requested_new_state, (gint*)(*p)->args, (*p)->nargs);
            }
            else
            {
                pr_warning("transition to requested state:%d failed with err:%d", (*p)->requested_new_state, err);
            }
        }
        else
        {
            pr_warning("no definition for requested state:%d", (*p)->requested_new_state);
        }
    }
    else
    {
        pr_warning("no definition for current state:%d", (*p)->current_state);
    }


    /* release always a pending request */
    (*p)->is_state_change_request_pending = FALSE;

    return err;
}

#ifdef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
static audproc_sm_request* audproc_sm_get_next_request(void)
{

    if(sm_request_count)
    {
        sm_request_count--;
        pr_debug("remaining request count:%d \n", sm_request_count);

        audproc_sm_request* next_req = &sm_request_list[sm_request_rd_pos];
        if(sm_request_rd_pos == (AUD_PROC_MAX_STATE_ELEMENT - 1))
            sm_request_rd_pos = 0;
        else
            sm_request_rd_pos++;

        return  next_req;
    }

    return (audproc_sm_request*)NULL;
}


/*******************************************************************************
*
* FUNCTION: audproc_state_machine_process_request
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
static gpointer audproc_state_machine_process_request(void)
{

    pr_message("ENTERED\n");
    audproc_sm_request* p = (audproc_sm_request*)NULL;
    audproc_state_machine* psm = (audproc_state_machine*)NULL;


    while (sm_request_run)
    {
        g_mutex_lock(&audproc_sm_lock);

        psm = NULL;


        /* wait release from audproc_alsa_start() function */
        while (!sm_request_count)
        {
            pr_debug("wait a new state change request\n");
            g_cond_wait(&audproc_sm_update, &audproc_sm_lock);
        }

        p = audproc_sm_get_next_request();
        if(p)
        {
            psm = p->sm_new_req;
            if(psm)
                pr_debug("process new request for state(%s) of instance_name(%s)\n", audproc_audio_state_str(p->sm_req_state), psm->this_inst_name);
        }

        g_mutex_unlock(&audproc_sm_lock);

        if(psm)
        {
            psm->err = audproc_state_machine_trigger_new_state_internal(&psm);
            if(psm->err && psm->this_inst_name)
            {
                pr_warning("error occured during processing state(%s) of source(%s)\n", audproc_audio_state_str(p->sm_req_state), psm->this_inst_name);
            }
            else
            {
                if(psm->this_inst_name)
                {
                    pr_message("the state of the source \"%s\" was changed to \"%s\" \n", psm->this_inst_name, audproc_audio_state_str(p->sm_req_state));
                }
            }

      if(psm->args)
      {
        g_free(psm->args);
        psm->args = (gint*)NULL;
      }
        }
    }

    pr_message("EXIT\n");
    return 0;
}

#endif
/*******************************************************************************
*
* FUNCTION: audproc_state_machine_create_new
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
audproc_state_machine*  audproc_state_machine_get_instance(guint8 sm_inst)
{
    if(!sm_list)
       return (audproc_state_machine*)NULL;

    if(!sm_inst)
        return (audproc_state_machine*)NULL;

    return  sm_list[sm_inst -1];
}



/*****************************************************************************
 *   Public functions definition area
 */



/*******************************************************************************
*
* FUNCTION: audproc_state_machine_init
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
void    audproc_state_machine_init(void)
{
    //GError *error;

    pr_message("ENTERED");

    g_mutex_init(&audproc_sm_lock);
    sm_list = g_malloc0(AUD_PROC_MAX_STATE_ELEMENT * sizeof(audproc_state_machine**));

    sm_inst_count = 0;

#ifdef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
    g_cond_init(&audproc_sm_update);

    /* initialize state machine worker requester-thread */
    sm_request_list = g_malloc0(AUPROC_SM_MAX_REQUEST * sizeof(audproc_sm_request));
    sm_request_count = 0;
    sm_request_rd_pos = 0;
    sm_request_wr_pos = 0;

    /* create a new thread only if not still exist */
    if (!audproc_sm_thread)
    {
        audproc_sm_thread = g_thread_try_new ("audproc_state_machine_process_request", audproc_state_machine_process_request, NULL, &error);
        if (!audproc_sm_thread)
        {
            pr_warning("fails to create audproc_state_machine_process_request with error code(%d) and message (%s)\n",error->code, error->message );
        }
    }
    else
    {
        pr_warning("thread already created\n");
    }

    /* start request processing thread */
    sm_request_run = TRUE;
#endif

    pr_message("EXIT");

    return;
}

/*******************************************************************************
*
* FUNCTION: audproc_state_machine_finalize
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
void    audproc_state_machine_finalize(void)
{
    audproc_state_element* p;
    struct _audproc_state_element* q;
    guint i, j , n;
    audproc_state_machine* sm = (audproc_state_machine*)NULL;

    pr_message("close all state machine and associated resources\n");

    if(!sm_list)
        return;

    /* check wether this state mache instance neme alreqdy exist */
    for( n = 0; n < sm_inst_count ; n++)
    {
        sm = sm_list[n];
        if(sm)
        {
            if(sm->this_inst_name)
            {
                pr_message("destruction of all resource of the state machine instance :%s", sm->this_inst_name);
            }

            for(i=0; i < AUD_PROC_MAX_STATE_ELEMENT; i++)
            {
                p = sm->state_liste[i];

                if(p)
                {
                    if(p->associated_state_elem_list)
                    {
                        for(j=0; j < AUD_PROC_MAX_STATE_ELEMENT; j++)
                        {
                            q = p->associated_state_elem_list[j];
                            if(q)
                            {
                               g_free(q);
                               p->associated_state_elem_list[j] = NULL;
                            }
                            else /* no instance */
                                break;
                        }
                    }
                    g_free(p);
                    sm->state_liste[i] = NULL;
                }
                else /* no instance */
                    break;
            }

            g_free(sm->this_inst_name);
            sm->this_inst_name = NULL;
            g_free(sm);
            sm_list[n] = NULL;
        }
    }

#ifdef  D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
    /* free request list   */
    sm_request_run = FALSE;
    g_cond_broadcast(&audproc_sm_update);

    if(sm_request_list)
    {
        g_free(sm_request_list);
        sm_request_list = NULL;
    }
    g_cond_clear(&audproc_sm_update);
#endif

    g_mutex_clear(&audproc_sm_lock);
    return;
}




/*******************************************************************************
*
* FUNCTION: audproc_state_machine_create_new
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
guint8  audproc_state_machine_create_new(const char* instance_name)
{
    audproc_state_machine* p = (audproc_state_machine*)NULL;
    guint i = 0;

    if(!instance_name)
    {
        pr_warning("a new state machine need a name instance to be created \n");
        return 0;
    }

    if(!sm_list)
        sm_list = g_malloc0(AUD_PROC_MAX_STATE_MACHINE * sizeof(audproc_state_machine*));

    if(AUD_PROC_MAX_STATE_MACHINE > sm_inst_count )
    {
        /* check wether this state mache instance neme alreqdy exist */
        for( i = 0; i < sm_inst_count ; i++)
        {
            p = sm_list[i];
            if(p)
            {
                if(p->this_inst_name)
                {
                    if(!g_strcmp0(p->this_inst_name, instance_name))
                    {
                        pr_warning("the state machine instance name: %s is already available\n", p->this_inst_name);
                        return p->this_inst;
                    }
                }
            }
            else
              break;
        }

        /* create new state machine */
        p = g_malloc0(sizeof(audproc_state_machine));
        if(!p)
        {
            pr_warning("fails to allocate memory for a new state machine\n");
            return 0;
        }

        p->this_inst_name = g_strdup(instance_name);
        p->this_inst = (guint8)(sm_inst_count + 1);  /* instance number is cuurent instance count + 1, mean always > 0*/
        p->config_lock = FALSE;
        p->state_liste = g_malloc0(AUD_PROC_MAX_STATE_ELEMENT * sizeof(audproc_state_element*));
        p->current_state = AUDPROC_AS_STREAM_SILENCE;
        p->requested_new_state = AUDPROC_AS_STREAM_UNDEFINED;
        p->is_state_change_request_pending =FALSE;
        p->err = (int)AUDPROC_ERR_OK;


        sm_list[sm_inst_count] = p;

        /* new state machine instance count */
        ++sm_inst_count;


        pr_message("*********** new state machine created *********\n");
        pr_message("sm instance                    :%d\n", p->this_inst);
        pr_message("sm instance name               :%s\n", p->this_inst_name?p->this_inst_name:NULL);
        pr_message("number of created state machine:%d\n", sm_inst_count);

        return p->this_inst;
    }


    return 0;
}



/*******************************************************************************
*
* FUNCTION: audproc_state_machine_config_begin
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_config_begin(guint8 sm_inst)
{
    int  err = (int)AUDPROC_ERR_OK;
    audproc_state_machine* p = (audproc_state_machine*)NULL;


    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
        return AUDPROC_ERR_NULL_POINTER;

    p->config_lock = TRUE;

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_state_machine_config_end
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_config_end(guint8 sm_inst)
{
    int  err = (int)AUDPROC_ERR_OK;
    audproc_state_machine* p = (audproc_state_machine*)NULL;


    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
        return AUDPROC_ERR_NULL_POINTER;

    p->config_lock = FALSE;

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_state_machine_add_main_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_add_main_state
(
    guint8 sm_inst,
    gint8 this_state,
    void* fct_state_action
)
{
    int  err = (int)AUDPROC_ERR_OK;
    audproc_state_machine* p = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
        return AUDPROC_ERR_NULL_POINTER;

    if(!p->config_lock)
    {
        pr_warning(" configuration of the state machine is terminated\n");
        return AUDPROC_ERR_SM_CONF_LOCK;
    }


    if(this_state > AUDPROC_AS_STREAM_LAST ||  this_state < AUDPROC_AS_STREAM_SILENCE)
    {
        pr_warning(" state undefined : %d\n", this_state);
        return AUDPROC_ERR_SM_UNDEFINED_STATE;
    }



    err = audproc_state_machine_add_state(  sm_inst
                                          , AUDPROC_STATE_TYPE_MAIN_STATE
                                          , this_state
                                          , AUDPROC_STATE_TYPE_UNDEFINED_STATE
                                          , AUDPROC_AS_STREAM_UNDEFINED
                                          , fct_state_action
                                          , NULL
                                          , NULL);


    return err;
}





/*******************************************************************************
*
* FUNCTION: audproc_state_machine_add_main_state_connections
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_add_main_state_connections
(
    guint8 sm_inst,
    gint8 this_state,
    gint8 associated_state,
    void* fct_chg_cond,
    void* fct_tr_act
)
{
    int  err = (int)AUDPROC_ERR_OK;
    audproc_state_machine* p = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
        return AUDPROC_ERR_NULL_POINTER;

    if(!p->config_lock)
    {
        pr_warning(" configuration of the state machine is terminated\n");
        return AUDPROC_ERR_SM_CONF_LOCK;
    }


    if(this_state > AUDPROC_AS_STREAM_LAST ||  this_state < AUDPROC_AS_STREAM_SILENCE)
    {
        pr_warning(" state undefined : %d\n", this_state);
        return AUDPROC_ERR_SM_UNDEFINED_STATE;
    }

    if(associated_state > AUDPROC_AS_STREAM_LAST ||  associated_state < AUDPROC_AS_STREAM_SILENCE)
    {
        pr_warning(" state undefined : %d\n", associated_state);
        return AUDPROC_ERR_SM_UNDEFINED_STATE;
    }


    err = audproc_state_machine_add_state(  sm_inst
                                          , AUDPROC_STATE_TYPE_MAIN_STATE
                                          , this_state
                                          , AUDPROC_STATE_TYPE_MAIN_STATE
                                          , associated_state
                                          , NULL
                                          , fct_chg_cond
                                          , fct_tr_act);


    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_state_machine_add_sub_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_add_sub_state
(
    guint8 sm_inst,
    gint8 this_state,
    gint8 associated_state,
    void* fct_state_action
)
{
    int  err = (int)AUDPROC_ERR_OK;
    audproc_state_machine* p = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
        return AUDPROC_ERR_NULL_POINTER;

    if(!p->config_lock)
    {
        pr_warning(" configuration of the state machine is terminated\n");
        return AUDPROC_ERR_SM_CONF_LOCK;
    }


    if(this_state > AUDPROC_AS_STREAM_LAST ||  this_state < AUDPROC_AS_STREAM_SILENCE)
    {
        pr_warning(" state undefined : %d\n", this_state);
        return AUDPROC_ERR_SM_UNDEFINED_STATE;
    }



    err = audproc_state_machine_add_state(  sm_inst
                                          , AUDPROC_STATE_TYPE_MAIN_STATE
                                          , this_state
                                          , AUDPROC_STATE_TYPE_SUB_STATE
                                          , associated_state
                                          , fct_state_action
                                          , NULL
                                          , NULL);

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_state_machine_add_sub_state_connections
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_add_sub_state_connections
(
    guint8 sm_inst,
    gint8 this_state,
    gint8 associated_state,
    void* fct_chg_cond,
    void* fct_tr_act
)
{
    int  err = (int)AUDPROC_ERR_OK;
    audproc_state_machine* p = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
        return AUDPROC_ERR_NULL_POINTER;

    if(!p->config_lock)
    {
        pr_warning(" configuration of the state machine is terminated\n");
        return AUDPROC_ERR_SM_CONF_LOCK;
    }


    if(this_state > AUDPROC_AS_STREAM_LAST ||  this_state < AUDPROC_AS_STREAM_SILENCE)
    {
        pr_warning(" state undefined : %d\n", this_state);
        return AUDPROC_ERR_SM_UNDEFINED_STATE;
    }

    if(associated_state > AUDPROC_AS_STREAM_LAST ||  associated_state < AUDPROC_AS_STREAM_SILENCE)
    {
        pr_warning(" state undefined : %d\n", associated_state);
        return AUDPROC_ERR_SM_UNDEFINED_STATE;
    }

    err = audproc_state_machine_add_state(  sm_inst
                                          , AUDPROC_STATE_TYPE_SUB_STATE
                                          , this_state
                                          , AUDPROC_STATE_TYPE_SUB_STATE
                                          , associated_state
                                          , NULL
                                          , fct_chg_cond
                                          , fct_tr_act);

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_state_machine_trigger_new_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_state_machine_trigger_new_state
(
    guint8              sm_inst,
    audproc_audio_state new_astate,
    gint**             args,
    gint              nargs
)
{

    pr_debug("ENTERED: new astate trigger with sm_inst:%d and astate:%d\n", sm_inst, new_astate);

    g_mutex_lock(&audproc_sm_lock);

    audproc_state_machine*  p = audproc_state_machine_get_instance(sm_inst);

    if(!p)
    {
        pr_warning(" no state machine available for the instance: %d\n", sm_inst);
        g_mutex_unlock(&audproc_sm_lock);
        return AUDPROC_ERR_SM_INST_NOT_AVAIL;
    }

    p->err = AUDPROC_ERR_OK;


    if(p->config_lock)
    {
        pr_warning("configuration of the state machine is not terminated\n");
        p->err = AUDPROC_ERR_SM_CONF_LOCK;
    }

    if(!p->err && p->is_state_change_request_pending)
    {
        pr_warning("a request change to the state \"%s\" is currently pending \n", audproc_audio_state_str(p->requested_new_state));
#ifndef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
        p->err = AUDPROC_ERR_SM_TRY_AGAIN;
#endif
    }

    if(!p->err)
    {
        p->requested_new_state = new_astate;
        p->current_state_elem = audproc_service_search_this_state_elem_p(p, AUDPROC_STATE_TYPE_MAIN_STATE, p->current_state);

        if(p->current_state_elem)
        {
            p->new_state_elem = audproc_state_machine_get_associated_state_elem(p->current_state_elem, p->requested_new_state);
            if(!p->new_state_elem)
            {
                p->err = AUDPROC_ERR_SM_UNDEFINED_STATE;
                pr_warning("no definition for requested state:%d", p->requested_new_state);
            }
        }
        else
        {
            p->err = AUDPROC_ERR_SM_UNDEFINED_STATE;
            pr_warning("no definition for current state:%d", p->current_state);
        }

        if(!p->err)
        {
            p->is_state_change_request_pending = TRUE;
            pr_debug("trigger now state change(%d)\n", p->requested_new_state);

            /* initializes the list of variable arguments*/
            if(*args && nargs)
            {
               p->args = (gint*)g_malloc0((gsize)nargs * sizeof(gint));
               if(p->args)
               {
                   for(int i = 0; i <nargs; i++)
                    p->args[i] = *args[i];
               }
            }
            else
                p->args = (gint*)NULL;

            //p->args = pargptr; //*pargptr;
            p->nargs = nargs;

        }
    }

#ifdef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
    /* free request list   */
    if(!p->err)
    {
        if(sm_request_count < AUD_PROC_MAX_STATE_ELEMENT)
        {
            audproc_sm_request* req = &sm_request_list[sm_request_wr_pos];

            req->sm_new_req = p;
            req->sm_req_state = new_astate;

            sm_request_count++;
            if(sm_request_wr_pos == (AUD_PROC_MAX_STATE_ELEMENT - 1))
                sm_request_wr_pos = 0;
            else
                sm_request_wr_pos++;

            g_cond_broadcast(&audproc_sm_update);
        }
    }

    g_mutex_unlock(&audproc_sm_lock);

#else
    g_mutex_unlock(&audproc_sm_lock);

    if(p->err)
    {
        pr_warning("state change trigger request terminated with err:%d \n", p->err);
    }
    else
    {
        p->err = audproc_state_machine_trigger_new_state_internal(&p);
        if(p->args)
        {
            g_free(p->args);
            p->args = (gint*)NULL;
        }
    }

#endif
    return p->err;
}



/*******************************************************************************
*
* FUNCTION: audproc_audio_state_str
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
const char *audproc_audio_state_str(audproc_audio_state code)
{
    guint i;
    const audproc_as_desc *as;
    static const char audproc_unknown_audio_state[] = "Unknown ECNR audio state";

    for (i = 0, as = AUDPROC_AUDIO_STATE_table; i < AUDPROC_AUDIO_STATE_MAX; i++, as++)
    {
        if (as->code == code)
            return as->str;
    }
    return audproc_unknown_audio_state;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_get_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
audproc_audio_state audproc_state_machine_get_audio_state(guint8 sm_inst)
{
    audproc_state_machine*              p = (audproc_state_machine*)NULL;

    pr_debug("ENTERED\n");

    p = audproc_state_machine_get_instance(sm_inst);
    if(!p)
    {
        pr_warning("no valide state machine \n");
        return AUDPROC_AS_STREAM_SILENCE;
    }

    pr_debug("current audio state: %s  \n", audproc_audio_state_str(p->current_state));

    return p->current_state;

}

/*******************************************************************************
*
* FUNCTION: audproc_state_machine_get_inst
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
guint8 audproc_state_machine_get_instance_per_name(char* instance_name)
{
    audproc_state_machine* p = (audproc_state_machine*)NULL;
    guint i = 0;

    if(!instance_name)
    {
        pr_warning("a new state machine need a name instance to be created \n");
        return 0;
    }

    if(!sm_list)
        return 0;


    /* check wether this state mache instance neme alreqdy exist */
    for( i = 0; i < sm_inst_count ; i++)
    {
        p = sm_list[i];
        if(p)
        {
            if(p->this_inst_name)
            {
                if(!g_strcmp0(p->this_inst_name, instance_name))
                {
                    pr_warning("the state machine instance name: %s is already available\n", p->this_inst_name);
                    return p->this_inst;
                }
            }
        }
        else
          break;
    }

    return 0;
}

