/*
 * audproc-service.c
 *
 *
 * Author: Patrick Rey
 * Date: 07.08.2015
 * Version 0.1 : Init
 */



#include <stdlib.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-bindings.h>
#include <dbus/dbus.h>
#include <string.h>
#include <signal.h> //new
#include <glib/gi18n.h>
#include <glib-object.h>
#include <glib-unix.h> //new
#include <glib.h>
#include <locale.h>


//#include <pthread.h>
#include <sys/types.h>
#include <unistd.h>
#include <sys/syscall.h>

/* alsa includes */
#include <alsa/asoundlib.h>


#include "audproc-common-defs.h"
#include "audproc-object.h"
#include "audproc-apl-handler.h"
#include "audproc-alsa.h"
#include "audproc-service.h"
#include "audproc-configuration.h"
#include "audproc-recording.h"
#include "audproc-audio-rts-handler.h"
#include "audproc-datapool-access.h"
#include "audproc-state-machine.h"


#ifdef D_SUPPORT_AUDROC_SERVER
#include "audproc-server-control.h"
#endif


/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "audproc-audioprocess-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AUDPROC_SERVICE
#include "trcGenProj/Header/audproc-service.c.trc.h"
#endif



/*******************************************************************************
              Constant defintion
*******************************************************************************/
#define AUDPROC_SERVICE_NAME "org.bosch.audproc.service"
#define AUDPROC_ERROR(name) { name, #name }
#define AUDPROC_SRC(name) { name, #name }
#define AUDPROC_ACTIVITY(name) { name, #name }


/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

static struct audproc_instance_object**     my_servs;
static GObject*                             obj;
GMainLoop*                                  audproc_main_loop;
GMutex                                      audproc_service_lock;
GCond                                       audproc_service_update;
static guint                                audproc_service_sig_source_id[3];
static tAplCfg*                             audproc_apl_init_data = (tAplCfg*)NULL;
static gboolean                             audproc_service_wait_notification = FALSE;
static guint8                               audproc_service_wait_notification_count = 0;
static guint8                               GetParamBuf[AUDPROC_SERVICE_MAX_READ_BUFFER_SIZE]; //500];
static gboolean                             audproc_service_select_loopback_route = FALSE;
#ifdef D_APL_CHECK_PARAMETER_CHANGE
static gint32                               audproc_last_input_level_deviation = -128;
#endif
inline int audproc_service_type_size(audproc_type apl_type)
{
    if( apl_type < AUDPROC_TYPE_ARRAY_U16)
        return ((((apl_type) < AUDPROC_TYPE_U32) ? 2 : 4));
    else
        return ((((apl_type) < AUDPROC_TYPE_ARRAY_U32) ? 2 : 4));
}



/* table of audproc_ error codes */
static const struct AudprocError AUDPROC_ERROR_table[] = {

    AUDPROC_ERROR(AUDPROC_ERR_ARKAMYS_NO_HANDLE_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_ARKAMYS_BLOCK_ID_ERROR),
   // AUDPROC_ERROR(AUDPROC_ERR_ARKAMYS_INSTANCE_ID_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_ARKAMYS_PARAMETER_ID_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_ARKAMYS_PARAMETER_VALUE_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_ARKAMYS_SIZE_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_OK),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_INPUT),
    AUDPROC_ERROR(AUDPROC_ERR_FAILED),
    AUDPROC_ERROR(AUDPROC_ERR_NO_AUDPROC_OBJECT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_CONFIG_FILE_OPEN_FAILED),
    AUDPROC_ERROR(AUDPROC_ERR_CONFIG_FILE_EMPTY),
    AUDPROC_ERROR(AUDPROC_ERR_CONFIG_FILE_READING_FAILED),
    AUDPROC_ERROR(AUDPROC_ERR_CONFIG_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_CONFIG_ALREADY_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_NO_RESULT),
    AUDPROC_ERROR(AUDPROC_ERR_SIZE_ONLY),
    AUDPROC_ERROR(AUDPROC_ERR_ALLOC),
    AUDPROC_ERROR(AUDPROC_ERR_NULL_POINTER),
    AUDPROC_ERROR(AUDPROC_ERR_MODULE_NOT_INIT),
    AUDPROC_ERROR(AUDPROC_ERR_NOT_CREATED),
    AUDPROC_ERROR(AUDPROC_ERR_MODULE_ALREADY_INIT),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_MAIN_STRUCT),
    AUDPROC_ERROR(AUDPROC_ERR_DEINIT_FAILED),
    AUDPROC_ERROR(AUDPROC_ERR_ILLEGAL_SET_PARAM),
    AUDPROC_ERROR(AUDPROC_ERR_ILLEGAL_SET_PARAM_BEFORE_INIT),
    AUDPROC_ERROR(AUDPROC_ERR_ILLEGAL_SET_PARAM_AFTER_INIT),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_PARAM_ID),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_CHANNEL_NUMBER),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_OPERATION_MODE),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_SIZE),
    AUDPROC_ERROR(AUDPROC_ERR_PARAM),
    AUDPROC_ERROR(AUDPROC_ERR_PARAM_CNT),
    AUDPROC_ERROR(AUDPROC_ERR_SAMPLE_RATE),
    AUDPROC_ERROR(AUDPROC_ERR_FRAMESHIFT),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_CHAN_CNT),
    AUDPROC_ERROR(AUDPROC_ERR_FFT_LENGTH),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_FILTER_LEN),
    AUDPROC_ERROR(AUDPROC_ERR_NULL_FILTER),
    AUDPROC_ERROR(AUDPROC_ERR_PARAMETER_TABLE),
    AUDPROC_ERROR(AUDPROC_ERR_NOT_IMP),
    AUDPROC_ERROR(AUDPROC_ERR_SUBMODULE_DISABLED),
    AUDPROC_ERROR(AUDPROC_ERR_INVALID_SET_PARAM_ENDIANNESS),
    AUDPROC_ERROR(AUDPROC_ERR_INIT_MMGR),
    AUDPROC_ERROR(AUDPROC_ERR_DGB_INIT_FAILED),
    AUDPROC_ERROR(AUDPROC_ERR_DGB_NO_DATA),
    AUDPROC_ERROR(AUDPROC_ERR_ALREADY_INITIALIZED),
    AUDPROC_ERROR(AUDPROC_ERR_ILLEGAL_SRC_NUMBER),

    /* error service module */
    AUDPROC_ERROR(AUDPROC_ERR_SERV_NOT_INITIALIZED),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_ALREADY_INSTANCED),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_NOT_INSTANCED),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_STREAMING_STILL_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_SRC_NOT_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_SRC_LOOPBACK_ALREADY_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_ROUTE_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_SERV_ROUTE_REQUEST_MISMATCH),

    /* error in configuration modul */
    AUDPROC_ERROR(AUDPROC_ERR_APPID_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_CONFID_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_NO_DEFAULT_CONFIGURATION),
    AUDPROC_ERROR(AUDPROC_ERR_APPID_INSTANCE_ALREADY_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_NO_INSTANCE_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_APPID_INSTANCE_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_APPID_CONFIGID_NOT_MATCHING),
    AUDPROC_ERROR(AUDPROC_ERR_NO_CONFIG_FILE_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_DEV_ID_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_DEV_CHNID_NOT_MATCHING),
    AUDPROC_ERROR(AUDPROC_ERR_CONFIG_EOL_IO_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_VERSION_SSE_DP_DO_NOT_MATCH),
    AUDPROC_ERROR(AUDPROC_ERR_NO_FILE_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_VALUE_RANGE_FAILURE),
    AUDPROC_ERROR(AUDPROC_ERR_TBL_IDX_NB_CFG_VALUE_TRANSGRESSION),
    AUDPROC_ERROR(AUDPROC_ERR_SRC_CFG_NOT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_APL_CFG_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_APL_NO_INSTANCE),


    /*-----------------------------*/

    AUDPROC_ERROR(AUDPROC_ERR_REQ_APPID_NOT_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_ALSA_NO_INSTANCE),
    AUDPROC_ERROR(AUDPROC_ERR_DEBUG_ALREADY_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_DEBUG_MODER_NOT_SUPPORTED),
    AUDPROC_ERROR(AUDPROC_ERR_DGB_SET_RMV_TRIGGER_FILE_FAILED),
    AUDPROC_ERROR(AUDPROC_ERR_OBJECT_FI_NOT_SUPPORTED),
    AUDPROC_ERROR(AUDPROC_ERR_OBJECT_FI_AUDIO_IS_RUNNING),
    AUDPROC_ERROR(AUDPROC_ERR_NO_AUDIO_STREAMING),
    AUDPROC_ERROR(AUDPROC_ERR_AUDIO_THREAD_ALREADY_ACTIVE),
    AUDPROC_ERROR(AUDPROC_ERR_OBJECT_FI_PARAM_OUT_OF_RANGE),

    /* error in rts handler  */
    AUDPROC_ERROR(AUDPROC_ERR_RTS_NOT_SUPPORTED),
    AUDPROC_ERROR(AUDPROC_ERR_RTS_STATE_ERROR),
    AUDPROC_ERROR(AUDPROC_ERR_RTS_HANDLE_INVALID),


    /* error in ecnr_datapool_access module*/
    AUDPROC_ERROR(AUDPROC_ERR_DP_ACCES_NULL_FCT_POINTER),
    AUDPROC_ERROR(AUDPROC_ERR_DP_ACCES_DP_ALREADY_REGISTRIED),
    AUDPROC_ERROR(AUDPROC_ERR_DP_NO_ELEMENT_AVAILABLE),
    AUDPROC_ERROR(AUDPROC_ERR_DP_INVALID_ELEMENT),
    AUDPROC_ERROR(AUDPROC_ERR_DP_ERROR_SIZE_OF_DATAPOOL),

   /* specifical error in  handler  */
    AUDPROC_ERROR(AUDPROC_ERR_OSAL_ERROR),

    /* apl specifical error*/
    AUDPROC_ERROR(APL_ERR_OK),
    AUDPROC_ERROR(APL_ERR_NO_RESULT),
    AUDPROC_ERROR(APL_ERR_SIZE_ONLY),
    AUDPROC_ERROR(APL_ERR_ALLOC),
    AUDPROC_ERROR(APL_ERR_NULL_POINTER),
    AUDPROC_ERROR(APL_ERR_MODULE_NOT_INIT)
};

#define AUDPROC_NUM_ERROR_CODES ARRAYSIZE(AUDPROC_ERROR_table)





/*******************************************************************************
*
* FUNCTION: audproc_service_error_str
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
const gchar *audproc_service_error_str(int code)
{
  unsigned int i;
  const struct AudprocError *err;
  static const gchar audproc_unknown_error[] = "Unknown AUDPROC error";

  for (i = 0, err = AUDPROC_ERROR_table; i < AUDPROC_NUM_ERROR_CODES; i++, err++) {
    if (err->code == code)
      return err->str;
  }
  return audproc_unknown_error;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_object_error_quark
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
GQuark audproc_service_object_error_quark(void)
{
  static GQuark quark = 0;

  if (!quark)
    quark = g_quark_from_static_string("audproc-error");

  //pr_message("equivalent quark to \"audproc-error\" : %d\n", quark);

  return quark;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_report_error
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_service_report_error
(
    int status,
    const gchar * dbus_msg,
    GError ** error
)
{
    gboolean ret = TRUE;
    GError *err;

    if(error == NULL)
        return FALSE;

    if (status != AUDPROC_ERR_OK)
    {
        audproc_warning(status, "%s", dbus_msg);
        ETG_TRACE_USR3(("[audproc_service_report_error]: %s", audproc_service_error_str(status)));



        g_set_error(error, audproc_service_object_error_quark(), status,
          "%s (%d): %s", audproc_service_error_str(status), status,
          dbus_msg);


        err = *error;
        pr_debug("err->domaine:%d, err->code=%d, err->message=%s\n",err->domain, err->code, err->message);

        ret = FALSE;
    }

    return ret;
}

/* table of audproc_ error codes */
static const struct AudprocSource AUDPROC_SOURCE_table[] = {

    /* apl specifical error*/
//    AUDPROC_SRC(AUD_PROC_SRC_DEFAULT),
    AUDPROC_SRC(AUD_PROC_SRC_NA),
    AUDPROC_SRC(AUD_PROC_SRC_ENT1),
    AUDPROC_SRC(AUD_PROC_SRC_ENT2),
    AUDPROC_SRC(AUD_PROC_SRC_ENT3),
    AUDPROC_SRC(AUD_PROC_SRC_ENT4),
    AUDPROC_SRC(AUD_PROC_SRC_ENT5),
    AUDPROC_SRC(AUD_PROC_SRC_ENT6),
    AUDPROC_SRC(AUD_PROC_SRC_ENT7),
    AUDPROC_SRC(AUD_PROC_SRC_ENT8),
    AUDPROC_SRC(AUD_PROC_SRC_ENT9),
    AUDPROC_SRC(AUD_PROC_SRC_ENT10),
    AUDPROC_SRC(AUD_PROC_SRC_ENT11),
    AUDPROC_SRC(AUD_PROC_SRC_ENT12),
    AUDPROC_SRC(AUD_PROC_SRC_ENT13),
    AUDPROC_SRC(AUD_PROC_SRC_ENT14),
    AUDPROC_SRC(AUD_PROC_SRC_ENT15),
    AUDPROC_SRC(AUD_PROC_SRC_ENT16),
    AUDPROC_SRC(AUD_PROC_SRC_ENT17),
    AUDPROC_SRC(AUD_PROC_SRC_MICIN_1),
    AUDPROC_SRC(AUD_PROC_SRC_MICIN_2),
    AUDPROC_SRC(AUD_PROC_SRC_VOICE1),
    AUDPROC_SRC(AUD_PROC_SRC_VOICE2),
    AUDPROC_SRC(AUD_PROC_SRC_LOOPBACK),
    AUDPROC_SRC(AUD_PROC_SRC_INFO1),
    AUDPROC_SRC(AUD_PROC_SRC_INFO2),
    AUDPROC_SRC(AUD_PROC_SRC_INFO3),
    AUDPROC_SRC(AUD_PROC_SRC_CLOCK_ON_OFF),
};


#define AUDPROC_NUM_SRC_CODES ARRAYSIZE(AUDPROC_SOURCE_table)


/*******************************************************************************
*
* FUNCTION: audproc_service_src_str
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
const gchar *audproc_service_src_str(guint8 src)
{
  unsigned int i;
  const struct AudprocSource *pd;
  static const gchar audproc_unknown_source[] = "Unknown AUDPROC source";

  for (i = 0, pd = AUDPROC_SOURCE_table; i < AUDPROC_NUM_SRC_CODES; i++, pd++) {
    if (pd->src == src)
      return pd->str;
  }
  return audproc_unknown_source;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_str_src
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
guint8  audproc_service_str_src(const gchar* src_str)
{
    if(!src_str)
    {
        pr_warning("route name is missing\n");
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT2"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT2;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT3")) // record capture stream only
    {
        return AUD_PROC_SRC_ENT3;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT4"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT4;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT5"))// record capture stream only
    {
       return AUD_PROC_SRC_ENT5;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT6"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT6;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT7"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT7;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT8"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT8;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT9"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT9;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT10"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT10;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT11"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT11;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT12"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT12;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT13"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT13;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT14"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT14;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT15"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT15;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT16"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT16;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_ENT17"))// record capture stream only
    {
        return AUD_PROC_SRC_ENT17;
    }
    else if (NULL != strstr(src_str,"AUD_PROC_SRC_ENT1"))   // record capture stream only
    {
        return AUD_PROC_SRC_ENT1;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_VOICE1"))// record capture stream only
    {
        return AUD_PROC_SRC_VOICE1;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_VOICE2"))// record capture stream only
    {
        return AUD_PROC_SRC_VOICE2;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_MICIN_1"))// record capture stream only
    {
        return AUD_PROC_SRC_MICIN_1;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_MICIN_2"))// record capture stream only
    {
        return AUD_PROC_SRC_MICIN_2;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_INFO1"))// record capture stream only
    {
        return AUD_PROC_SRC_INFO1;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_INFO2"))// record capture stream only
    {
        return AUD_PROC_SRC_INFO2;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_INFO3"))// record capture stream only
    {
        return AUD_PROC_SRC_INFO3;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_LOOPBACK"))// record capture stream only
    {
        return AUD_PROC_SRC_LOOPBACK;
    }
    else if (NULL != strstr(src_str, "AUD_PROC_SRC_CLOCK_ON_OFF"))// record capture stream only
    {
        return AUD_PROC_SRC_CLOCK_ON_OFF;
    }
    else
    {
        /* do nothing */
    }
    return AUD_PROC_SRC_NA;
}

/* table of audproc_ error codes */
static const struct AudprocActivity AUDPROC_SRC_ACTIVITY_table[] = {

    /* apl specifical error*/
    AUDPROC_ACTIVITY(AUDPROC_SA_UNDEF),
    AUDPROC_ACTIVITY(AUDPROC_SA_PREPARE),
    AUDPROC_ACTIVITY(AUDPROC_SA_ON),
    AUDPROC_ACTIVITY(AUDPROC_SA_OFF),
    AUDPROC_ACTIVITY(AUDPROC_SA_PAUSE),
    AUDPROC_ACTIVITY(AUDPROC_SA_ACTIVATE),
    AUDPROC_ACTIVITY(AUDPROC_SA_DEACTIVATE),
};

#define AUDPROC_NUM_ACTIVITY_CODES ARRAYSIZE(AUDPROC_SRC_ACTIVITY_table)

/*******************************************************************************
*
* FUNCTION: audproc_service_activity_str
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
const gchar *audproc_service_activity_str(audproc_source_activity activity)
{
  unsigned int i;
  const struct AudprocActivity *pd;
  static const gchar audproc_unknown_activity[] = "Unknown AUDPROC activity";

  for (i = 0, pd = AUDPROC_SRC_ACTIVITY_table; i < AUDPROC_NUM_ACTIVITY_CODES; i++, pd++) {
    if (pd->activity == activity)
      return pd->str;
  }
  return audproc_unknown_activity;
}

#define AUDPROC_NUM_SRC_CODES ARRAYSIZE(AUDPROC_SOURCE_table)


/*******************************************************************************
*
* FUNCTION: audproc_service_get_array_type
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
audproc_type    audproc_service_get_array_type(audproc_type in_type)
{

    switch(in_type)
    {
        case AUDPROC_TYPE_I16:
            return AUDPROC_TYPE_ARRAY_U16;
        case AUDPROC_TYPE_U16:
            return AUDPROC_TYPE_ARRAY_U32;
        case AUDPROC_TYPE_U32:
            return AUDPROC_TYPE_ARRAY_U32;
        case AUDPROC_TYPE_I32:
            return AUDPROC_TYPE_ARRAY_I32;
        default:
            return AUDPROC_TYPE_UNDEF;
    }

}


/*******************************************************************************
*
* FUNCTION: audproc_service_handle_quit_signal
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean audproc_service_handle_quit_signal (void)
{
    pr_message("Received quit signal");

    g_return_val_if_fail (audproc_main_loop != NULL, FALSE);

    /* end main loop thread */
    g_main_loop_quit(audproc_main_loop);

    return TRUE;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_install_signal_handler
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_service_install_signal_handler (GMainLoop *main_loop)
{
    GSource *source = NULL;
    GMainContext *ctx = g_main_loop_get_context (main_loop);

    pr_message("install the signal cb for SIGTERM, SIGINT and SIGHUP\n");

    source = g_unix_signal_source_new (SIGTERM);
    g_source_set_callback (source,
                          (GSourceFunc)audproc_service_handle_quit_signal,
                           NULL,
                           NULL);
    audproc_service_sig_source_id[0] = g_source_attach (source, ctx);

    pr_debug("cb for signal SIGTERM was set");

    GSource *source_1 = NULL;
    source_1 = g_unix_signal_source_new (SIGINT);
    g_source_set_callback (source_1,
                           (GSourceFunc)audproc_service_handle_quit_signal,
                           NULL,
                           NULL);
    audproc_service_sig_source_id[1] = g_source_attach (source_1, ctx);
    pr_debug("cb for signal SIGINT was set");


    GSource *source_2 = NULL;
    source_2 = g_unix_signal_source_new (SIGHUP);
    g_source_set_callback (source_2,
                           (GSourceFunc)audproc_service_handle_quit_signal,
                           NULL,
                           NULL);
    audproc_service_sig_source_id[2] = g_source_attach (source_2, ctx);
    pr_debug("cb for signal SIGHUP was set");
}

/*******************************************************************************
*
* FUNCTION: audproc_service_print_src_instance_state
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_service_print_src_instance_state(void)
{
    int                             instid = 0;
    struct audproc_instance_object* pd = (struct audproc_instance_object*)NULL;
    audproc_alsa_stream_cfg*        ps = (audproc_alsa_stream_cfg*)NULL;

    if(!my_servs)
        return;

    /* create new ecnr instamce */
    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    for(instid = 0; instid < nb_route_max; instid++)
    {

        pd = my_servs[instid];
        if(pd)
        {
            if(!pd->audproc_audio_instance)
                continue;
            if(!pd->audproc_audio_instance->audproc_stream_data)
                continue;

            ps = pd->audproc_audio_instance->audproc_stream_data;

            pr_message("------- source/route Id(%d): %s -------------\n",pd->audproc_this_src , audproc_service_src_str(pd->audproc_this_src));
            pr_message("audproc_this_src                    : %d\n", pd->audproc_this_src);
            pr_message("audproc_this_src_type               : %d\n", pd->audproc_this_src_type);
            pr_message("audproc_is_initialized              : %d\n", pd->audproc_is_initialized);
            pr_message("audproc_is_apl_instance_created     : %d\n", pd->audproc_is_apl_instance_created);
            pr_message("audproc_is_streaming_started        : %d\n", pd->audproc_is_streaming_started);
            pr_message("indev.audproc_dev_name              : %s\n", (ps->indev.audproc_dev_name!= NULL) ?\
                                                                      ps->indev.audproc_dev_name :          \
                                                                      NULL);
            pr_message("outdev.audproc_dev_name             : %s\n", (ps->outdev.audproc_dev_name!= NULL) ?\
                                                                      ps->outdev.audproc_dev_name: \
                                                                      NULL);
            pr_message("lpdev.audproc_dev_name              : %s\n", (ps->lpdev.audproc_dev_name!= NULL) ?\
                                    ps->lpdev.audproc_dev_name: \
                                                                      NULL);
        }
    }


    return ;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_get_instance
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static  struct audproc_instance_object* audproc_service_get_instance (guchar source)
{
    struct audproc_instance_object* pd = (struct audproc_instance_object*)NULL;
    audproc_alsa_stream_cfg*        ps = (audproc_alsa_stream_cfg*)NULL;
    gint instid;

    if(!my_servs)
        return (struct audproc_instance_object*)NULL;

    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    for(instid = 0; instid < nb_route_max; instid++)
    {
        pd = my_servs[instid];

        if(pd)
        {
            if(pd->audproc_this_src == source)
            {
                pr_debug("instance for source(%d) was found\n", source);
                return pd;
            }
        }
    }


    /*********************************
     * Solve startup issue : retry to initialize the service for the source "source"
     * During system start the audioprocess deamon is started prior to the loopback sound card
     * the initialization of the service instance implying the alsa device AdevLxcSrcSelect fails.
     * a initialzation retry is done here
     */

    /* create new service instance */
    ps = audproc_configuration_get_route_cfg(source);
    if(!ps)
    {
        pr_warning("no streaming configuration available for audpoc_source->%d)\n", source); //, audproc_service_src_str(source));

    }
    else
    {
        pd = audproc_service_create_new_instance(source);
        if(pd)
        {
            pr_message("create during run time a new service instance entry for(%d): %s\n", ps->source_id, audproc_service_src_str(ps->source_id));
            (void)audproc_service_add_new_instance(pd);

            return pd;
        }
        else
            pr_warning("no instance found for the source(%d)\n", source);
    }

    return (struct audproc_instance_object*)NULL;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_add_new_instance
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_service_add_new_instance (struct audproc_instance_object* new_inst)
{
    struct audproc_instance_object* pd = (struct audproc_instance_object*)NULL;
    gint instid;

    if(!my_servs)
        return FALSE;

    pr_debug("ENTERED");

    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    for(instid = 0; instid < nb_route_max; instid++)
    {
        pd = my_servs[instid];

        if(!pd)
        {
            /* add instance in the service list*/
            my_servs[instid] = new_inst;
            pr_debug("add new instance in the service list for the source(%d)\n", new_inst->audproc_this_src);
            return TRUE;
        }
    }

    pr_debug("instance for source(%d) was not added\n", new_inst->audproc_this_src);
    return FALSE;
}

#ifdef  D_USE_IT_AUDPROC_SERVICE_REMOVE_INSTANCE
/*******************************************************************************
*
* FUNCTION: audproc_service_remove_instance
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static gboolean audproc_service_remove_instance(guchar source)
{
    struct audproc_instance_object* pd = (struct audproc_instance_object*)NULL;
    gint                            instid;
    gboolean                        status = FALSE;

    if(!my_servs)
        return status;


    pr_message("ENTERED");


    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    for(instid = 0; instid < nb_route_max; instid++)
    {
        pd = my_servs[instid];

        if(pd)
        {
            if(pd->audproc_this_src == source)
            {
                (void)audproc_alsa_finalize_intance(pd->audproc_audio_instance);

                if(pd->audproc_audio_instance)
                {
                    g_free(pd->audproc_audio_instance);
                    pd->audproc_audio_instance = NULL;
                }

                g_free(my_servs[instid]);
                my_servs[instid] = NULL;

                pr_message("remove instance in the service list for the source(%d)\n", source);

                status = TRUE;
            }
        }
    }

    pr_message("EXIT with status(%d)", status);

    return status;
}

#endif // D_USE_IT_AUDPROC_SERVICE_REMOVE_INSTANCE



/*******************************************************************************
*
* FUNCTION: audproc_service_get_active_type_src
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct audproc_instance_object* audproc_service_get_active_type_src (audproc_source_type src_type)
{
    struct audproc_instance_object* pd = (struct audproc_instance_object*)NULL;
    gint i = 0;

    pr_debug("ENTERED: request for source type(%d)\n", src_type);

    if(!my_servs)
        return (struct audproc_instance_object*)NULL;


    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    for(i = 0; i < nb_route_max; i++)
    {
        pd = my_servs[i];
        if(    pd
            &&(pd->audproc_is_this_src_active )
            &&(pd->audproc_this_src_type == src_type ))
        {
            pr_debug("return active service: source(%s) , type(%d)\n", audproc_service_src_str(pd->audproc_this_src), src_type);
            return pd;
        }
    }

    if(i == nb_route_max)
        pr_debug("No active source found for the source type(%d)\n", src_type);

    return (struct audproc_instance_object*)NULL;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_get_first_type_src_occurence
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct audproc_instance_object* audproc_service_get_first_type_src_occurence (audproc_source_type src_type)
{
    struct audproc_instance_object* pd = (struct audproc_instance_object*)NULL;
    gint i = 0;

    pr_debug("ENTERED: request for source type(%d)\n", src_type);

    if(!my_servs)
        return (struct audproc_instance_object*)NULL;


    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    for(i = 0; i < nb_route_max; i++)
    {
        pd = my_servs[i];
        if(    pd
            &&(pd->audproc_this_src_type == src_type ))
        {
            pr_debug("return first configuration occurence: source(%s) , type(%d)\n", audproc_service_src_str(pd->audproc_this_src), src_type);
            return pd;
        }
    }

    if(i == nb_route_max)
        pr_debug("No active source found for the source type(%d)\n", src_type);

    return (struct audproc_instance_object*)NULL;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_create_new_instance
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
struct audproc_instance_object* audproc_service_create_new_instance (guchar source)
{
    audproc_alsa_stream_cfg*        ps = (audproc_alsa_stream_cfg*)NULL;
    audproc_alsa_state*             pa = (audproc_alsa_state*)NULL;


    pr_message("ENTERED: try to create a new service instance for src(%d): %s\n", source, audproc_service_src_str(source));

    ps = audproc_configuration_get_route_cfg(source);
    if(!ps)
    {
        pr_warning("no streaming configuration available for source(%d)\n", source);
        return (struct audproc_instance_object*)NULL;
    }

    struct audproc_instance_object* po = (struct audproc_instance_object*)NULL;

    po = g_new0(struct audproc_instance_object, 1);

    if(po)
    {
        po->audproc_this_src = source;
        po->audproc_this_src_type = ps->audcfg.audproc_this_src_type;
        po->audproc_is_initialized = FALSE;
        po->audproc_is_apl_instance_created = FALSE;
        po->audproc_is_streaming_started = FALSE;
        po->audproc_is_streaming_running_state = FALSE;
        po->audproc_astate = AUDPROC_AS_STREAM_SILENCE;

        pa = audproc_alsa_init_instance(ps);
        if(!pa)
        {
            pr_warning("fail to configure audio route for source(%d)\n", source);

            g_free(po);
            po = NULL;

            //(struct audproc_instance_object*)NULL;
        }
        else
        {
            po->audproc_audio_instance = pa;
            pr_debug("new service instance for source(%d) was created\n", source);

            /* create state machine */
            guint8 sm_inst = audproc_service_create_new_state_machine(po->audproc_this_src_type, audproc_service_src_str(source));
            po->audproc_sm_inst = sm_inst;
        }

        return po;
    }

    return (struct audproc_instance_object*)NULL;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_print_current_state
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_service_print_current_state(void)
{
    gint src_type = 0;
    struct audproc_instance_object*     act_obj;

    pr_message("print current streamings state\n");

    for (src_type = 0; src_type < (gint)AUD_PROC_SRC_TYPE_COUNT; src_type++)
    {
        act_obj = audproc_service_get_active_type_src((audproc_source_type)src_type);

        if(act_obj)
        {
            pr_message("---------- current service state src type: %d  -------------\n", src_type);
            ETG_TRACE_USR1(("---------- current service state src type: %d  -------------", src_type));
            pr_message("act_obj->audproc_this_src               : %s(%d), \n", audproc_service_src_str(act_obj->audproc_this_src), act_obj->audproc_this_src);
            ETG_TRACE_USR1(("act_obj->audproc_this_src               : %s", audproc_service_src_str(act_obj->audproc_this_src)));
            pr_message("act_obj->audproc_is_initialized         : %d\n", act_obj->audproc_is_initialized);
            ETG_TRACE_USR1(("act_obj->audproc_is_initialized         : %d", act_obj->audproc_is_initialized));
            pr_message("act_obj->audproc_is_streaming_started   : %d\n", act_obj->audproc_is_streaming_started);
            ETG_TRACE_USR1(("act_obj->audproc_is_streaming_started   : %d", act_obj->audproc_is_streaming_started));
            pr_message("source type current state               : %s\n", audproc_audio_state_str(audproc_service_get_audio_state((audproc_source_type)src_type))); //AUD_PROC_SRC_TYPE_ENT)));
            ETG_TRACE_USR1(("source type current state               : %s", audproc_audio_state_str(audproc_service_get_audio_state((audproc_source_type)src_type))));
        }
        else
        {
           pr_message("---------- source type(%d) not active -------------\n", src_type);
           ETG_TRACE_USR1(("************* Debug options ****************"));
        }
    }

    return;
}


/*******************************************************************************
*
* FUNCTION: main
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

int main(int argc, char *argv[])
{
    DBusGConnection*    connection;
    GError*             error;
    DBusGProxy*         driver_proxy;
    guint32             request_name_ret;
    char**              opt = &argv[0];
    int                 optcnt = argc;

    pr_message("audproc_main Entered \n");

    //g_type_init();
    //g_thread_init(NULL);
    //dbus_g_thread_init();
    dbus_threads_init (NULL);


    audproc_main_loop = g_main_loop_new(NULL, FALSE);
    error = NULL;

    /***********************
      evaluation main option
    ************************/
    if(*opt)
        pr_message("main entered with %d options\n", optcnt);

    /****************************************************************
      check for mode selection
      - execute unitest if key 'u' ist set and value is set to "1"
      - print audproc infos
      - otherwise run ecrn deamon
    *****************************************************************/


    /* run audproc deamon main program */
    connection = dbus_g_bus_get(DBUS_BUS_SYSTEM, &error);
    if (connection == NULL) {
        pr_critical("Failed to open connection to bus: %s",
          error->message);
        g_error_free(error);
        _exit(1);
    }

    pr_debug("dbus_g_bus_get\n");

    obj = g_object_new(AUDPROC_TYPE_OBJECT, NULL);

    pr_debug("g_object_new\n");

    dbus_g_connection_register_g_object(connection, "/", obj);

    pr_debug("dbus_g_connection_register_g_object \n");

    driver_proxy = dbus_g_proxy_new_for_name(connection,
             DBUS_SERVICE_DBUS,
             DBUS_PATH_DBUS,
             DBUS_INTERFACE_DBUS);

    if (!org_freedesktop_DBus_request_name(driver_proxy,
                 AUDPROC_SERVICE_NAME,
                 0, &request_name_ret, &error))
    {
        g_assert(error != NULL);
        if (error)
            pr_critical("Failed to get name: %s", error->message);
        g_error_free(error);
        _exit(1);
    }

    pr_debug("dbus_g_proxy_new_for_name \n");

    if (!(request_name_ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER))
    {
        pr_critical("Got result code %u from requesting name", request_name_ret);
        _exit(1);
    }


    pr_debug("Start audioproc after power cycle");

    /* install callback for signal handler */
    audproc_service_install_signal_handler(audproc_main_loop);

   /* create a mainloop that runs/iterates the default GLib main context
    * (context NULL). When a message has been posted on the
    * bus or bei systemd, the default main context will automatically dispatch the
    * associated resource and call the callback associated with the resource
    * The main loop remains in run state until g_main_loop_quit() is called
   */
    g_main_loop_run(audproc_main_loop);

    /* finalize audproc */
    audproc_service_finalize();

    if (audproc_main_loop)
        g_main_loop_unref (audproc_main_loop);


    _exit(0);
}



/*******************************************************************************
+ instance handler static functions
*******************************************************************************/

static void audproc_service_wait_apl_notification_start(void)
{

    pr_message("ENTERED, current notification wait count -> %d",audproc_service_wait_notification_count);
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification_start]: ENTERED,  current notificatin wait count -> %d", audproc_service_wait_notification_count));

#ifdef D_SAFE_NOTIFICATION
    /* wait till the parameter was updated  */
    g_mutex_lock(&audproc_service_lock);
    audproc_service_wait_notification = TRUE;
    audproc_service_wait_notification_count++;
    g_mutex_unlock(&audproc_service_lock);
#else
    audproc_service_wait_notification = TRUE;
    audproc_service_wait_notification_count++;
#endif
    pr_message("EXIT");
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification_start]: EXIT"));
    return;
}

static void audproc_service_wait_apl_notification_stop(void)
{
    pr_message("ENTERED, current notification wait count -> %d\n", audproc_service_wait_notification_count);
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification_stop]: ENTERED, current notification wait count -> %d ", audproc_service_wait_notification_count));

#ifdef D_SAFE_NOTIFICATION
    /* wait till the parameter was updated  */
    g_mutex_lock(&audproc_service_lock);
    audproc_service_wait_notification = FALSE;
    if(audproc_service_wait_notification_count)
        audproc_service_wait_notification_count--;
    g_cond_broadcast(&audproc_service_update);
    g_mutex_unlock(&audproc_service_lock);
#else
    audproc_service_wait_notification = FALSE;
    if(audproc_service_wait_notification_count)
        audproc_service_wait_notification_count--;
#endif
    pr_message("EXIT, current notification wait count -> %d\n", audproc_service_wait_notification_count);
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification_stop]: EXIT, current notification wait count -> %d ", audproc_service_wait_notification_count));
    return;
}

static int audproc_service_wait_apl_notification(void)
{
    int err = AUDPROC_ERR_OK;

    pr_message("ENTERED, current notification wait count -> %d , notification wait state -> %d", audproc_service_wait_notification_count , audproc_service_wait_notification);
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification]: ENTERED, current notification wait count -> %d , notification wait state -> %d\n",audproc_service_wait_notification_count , audproc_service_wait_notification));

    gint64 end_time = g_get_monotonic_time () + D_READ_DATA_TIME_OUT * G_TIME_SPAN_MILLISECOND;

    /* wait till the parameter was updated  */
    g_mutex_lock(&audproc_service_lock);

    while (audproc_service_wait_notification)
    {
        if (!g_cond_wait_until(&audproc_service_update, &audproc_service_lock, end_time))
        {
            // timeout has passed.
            pr_warning("apl notification time_out:%d_ms has expired", D_READ_DATA_TIME_OUT);
            ETG_TRACE_ERR(("[audproc_service_wait_apl_notification]: apl notification time_out:%d_ms has expired", D_READ_DATA_TIME_OUT));
            err = AUDPROC_ERR_PARAM;
            break;
        }
    }
    g_mutex_unlock(&audproc_service_lock);

    pr_message("EXIT, current notification wait count -> %d , notification wait state -> %d", audproc_service_wait_notification_count , audproc_service_wait_notification);
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification]: EXIT, current notification wait count -> %d , notification wait state -> %d\n",audproc_service_wait_notification_count , audproc_service_wait_notification));

    return err;
}

#if 0
static gboolean audproc_service_wait_apl_notification_is(void)
{
   // int err = AUDPROC_ERR_OK;
    gboolean bwaitenable = FALSE;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification_is]: ENTERED"));

#ifdef D_SAFE_NOTIFICATION
    /* wait till the parameter was updated  */
    g_mutex_lock(&audproc_service_lock);
    bwaitenable = audproc_service_wait_notification;
    g_mutex_unlock(&audproc_service_lock);
#else
    bwaitenable = audproc_service_wait_notification;
#endif

    pr_message("EXIT, current notification wait count -> %d\n", audproc_service_wait_notification_count);
    ETG_TRACE_USR3(("[audproc_service_wait_apl_notification_is]: EXIT, current notification wait count -> %d", audproc_service_wait_notification_count));
    return bwaitenable;
}
#endif

static void AplNotifyCb(tAplDataID param_Id, tAplAction param_action)
{
    int err = AUDPROC_ERR_OK;

    pr_message("ENTERED, , apl notification of  parameter -> %d", param_Id );
    ETG_TRACE_USR3(("[AplNotifyCb]: ENTERED, apl notification of  parameter -> %d", param_Id ));

    switch(param_Id)
    {
        case aplMicLvl:
        {
            if(param_action == APL_ACT_GET_PARAM)
            {
                guint32  MicroLevel = 0;
                guint32  size = 2;

                AplState* pi = audproc_apl_handler_get_instance(APL_INSTANCE_STR_MICRO_LEVEL_MONITORING_FCTS);
                err = audproc_apl_handler_get_data(pi, aplMicLvl, 0, &size, (void*)&MicroLevel);

                if(!err)
                {
                    /* dbus send signal with actual micro level */
                    pr_debug("notified value of current miclevel(%d)\n", MicroLevel);
                    ETG_TRACE_USR3(("[AplNotifyCb]:notified value of current Micro Level ->%d", MicroLevel));
                    err = audproc_object_emit_signal_micro_level(MicroLevel);
                }
                else
                {
                    ETG_TRACE_USR3(("[AplNotifyCb]:audproc_apl_handler_get_data() fails with err ->%d", err));
                    pr_warning("failure while trying to retrieve micro level at apl\n");
                }
            }
            break;
        }
        case aplArkamysLevelLoss:
        {
            if(param_action == APL_ACT_GET_PARAM)
            {
                gint     LevelLoss = 0;
                guint32  size = 4;

                AplState* pi = audproc_apl_handler_get_instance(APL_INSTANCE_STR_ARKAMYS_FCTS);
                err = audproc_apl_handler_get_data(pi, aplArkamysLevelLoss, 0, &size, (void*)&LevelLoss);

                if(!err)
                {
                    /* dbus send signal with actual micro level */
                    pr_debug("notified value of current  Arkamys LevelLoss -> %d\n", LevelLoss);
                    ETG_TRACE_USR3(("[AplNotifyCb]:notified value of current  Arkamys LevelLoss -> %d", LevelLoss));
                    err = audproc_object_emit_signal_arkamys_level_loss(LevelLoss);
                }
                else
                    pr_warning("failure while trying to retrieve micro level at apl\n");

            }
            else if(param_action == APL_ACT_SYNC_PARAM)
            {
                /* release mutex and allow read parameter update */
                pr_message("apl sync parameter -> %d\n", aplArkamysLevelLoss); /*new*/
                ETG_TRACE_USR3(("[AplNotifyCb]:apl sync parameter -> %d", aplArkamysLevelLoss));/*new*/
                #ifdef D_SAFE_NOTIFICATION
                audproc_service_wait_apl_notification_stop();
                #else
                audproc_service_wait_notification = FALSE;
                g_cond_broadcast(&audproc_service_update);
                #endif
            }
            else
            {
                /*tbd*/
            }

            break;
        }
        case aplArkamysReadData:
        case aplArkamysSetData:
        case aplArkamysWriteMultiFrame:
        case aplArkamysByPassActivate:
        case aplArkamysAvcActivate:
        case aplArkamysAmbienceSelect:
        case aplArkamysInputLevelDeviation:
        case aplArkamysCurrentSpeed:
        case aplArkamysCurrentVolumeStep:
        case aplArkamysCurrVolAttenuation:
        case aplArkamysLibraryVersion:
        case aplArkamysFlush:
        {
            if(param_action == APL_ACT_SYNC_PARAM)
            {
                /* release mutex and allow read parameter update */
                pr_message("apl sync for parameter -> %d\n", param_Id);
                ETG_TRACE_USR3(("[AplNotifyCb]: apl sync for parameter -> %d", param_Id));  /*changed*/

                #ifdef D_SAFE_NOTIFICATION
                audproc_service_wait_apl_notification_stop();
                #else
                audproc_service_wait_notification = FALSE;
                g_cond_broadcast(&audproc_service_update);
                #endif
            }
            else
            {
                /*tbd*/
            }
            break;
        }
        default:
            pr_debug("parameter id(%d) no supported\n", param_Id);
            break;
    }

    pr_message("EXIT");
    ETG_TRACE_USR3(("[AplNotifyCb]: EXIT"));
    return;

}

#if 0

static void AplTracecb(tAplU8 trace_lvl, char* trace_string)
{
    if(!trace_string)
    {
        ETG_TRACE_ERR(("[AUDPROC_LIB_ERR]: AplTracecb: invalid log descriptor"));
        pr_warning("invalid log descriptor\n");
        return;
    }
    if(trace_lvl >= 1)  /* pr_message*/
    {
        ETG_TRACE_USR3(("[AUDPROC_LIB_MSG]: %s", trace_string ));
    }
    else if(trace_lvl >= 2) /* pr_debug*/
    {
        ETG_TRACE_USR3(("[AUDPROC_LIB_DBG]: %s", trace_string ));
    }
    else if(trace_lvl >= 3) /* pr_debug_data*/
    {
        ETG_TRACE_USR3(("[AUDPROC_LIB_DBG]: %s", trace_string ));
    }
    else if(trace_lvl >= 4) /* pr_warning*/
    {
        ETG_TRACE_ERR(("[AUDPROC_LIB_WARN]: %s", trace_string ));
    }
    else
    {
        pr_warning("not support trace level \n");
    }
    return;
}

#endif

/*******************************************************************************
 *
 * FUNCTION: audproc_service_clear_notification
 *
 * DESCRIPTION:
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
static int audproc_service_clear_notification(tAplDataID param_Id, tAplAction param_action)
{
    int err = AUDPROC_ERR_OK;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_service_clear_notification]: ENTERED"));

    switch(param_Id)
    {
        case aplMicLvl:
        case aplArkamysLevelLoss:
        case aplArkamysReadData:
        case aplArkamysSetData:
        case aplArkamysWriteMultiFrame:
        case aplArkamysByPassActivate:
        case aplArkamysAvcActivate:
        case aplArkamysAmbienceSelect:
        case aplArkamysInputLevelDeviation:
        case aplArkamysCurrentSpeed:
        case aplArkamysCurrentVolumeStep:
        case aplArkamysCurrVolAttenuation:
        case aplArkamysLibraryVersion:
        case aplArkamysFlush:
        {
            /* release mutex and allow read parameter update */
            pr_message("apl sync for parameter -> %d\n", param_Id);
            ETG_TRACE_USR3(("[audproc_service_clear_notification]: apl sync for parameter -> %d", param_Id));   /*changed*/

            audproc_service_wait_notification = FALSE;
            if(audproc_service_wait_notification_count)
                audproc_service_wait_notification_count--;
            g_cond_broadcast(&audproc_service_update);

            break;
        }
        default:
            pr_debug("paramter id(%d) no supported\n", param_Id);
            break;
    }

    (void)param_action;

    return err;
}

/*******************************************************************************
 *
 * FUNCTION: audproc_service_send_notification
 *
 * DESCRIPTION:
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
int audproc_service_send_notification(tAplDataID param_Id, tAplAction param_action)
{
    int err = AUDPROC_ERR_OK;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_service_send_notification]: ENTERED with paramter -> %d and action -> %d",param_Id, param_action));

    switch(param_Id)
    {
        case aplMicLvl:
        {
            if(param_action == APL_ACT_GET_PARAM)
            {
                guint32  MicroLevel = 0;
                guint32  size = 2;

                AplState* pi = audproc_apl_handler_get_instance(APL_INSTANCE_STR_MICRO_LEVEL_MONITORING_FCTS);
                err = audproc_apl_handler_get_data(pi, aplMicLvl, 0, &size, (void*)&MicroLevel);

                if(!err)
                {
                    /* dbus send signal with actual micro level */
                    pr_debug("notified value of current miclevel(%d)\n", MicroLevel);
                    err = audproc_object_emit_signal_micro_level(MicroLevel);
                }
                else
                    pr_warning("failure while trying to retrieve micro level at apl\n");

            }
            break;
        }
        case aplArkamysLevelLoss:
        {
            if(param_action == APL_ACT_GET_PARAM)
            {
                gint     LevelLoss = 0;
                guint32  size = 4;

                AplState* pi = audproc_apl_handler_get_instance(APL_INSTANCE_STR_ARKAMYS_FCTS);
                err = audproc_apl_handler_get_data(pi, aplArkamysLevelLoss, 0, &size, (void*)&LevelLoss);

                if(!err)
                {
                    /* dbus send signal with actual micro level */
                    pr_debug("notified value of current  Arkamys LevelLoss(%d)\n", LevelLoss);
                    err = audproc_object_emit_signal_arkamys_level_loss(LevelLoss);
                }
                else
                    pr_warning("failure while trying to get Arkamys LevelLoss at apl\n");

            }
            break;
        }
        case aplArkamysReadData:
        case aplArkamysSetData:
        case aplArkamysWriteMultiFrame:
        case aplArkamysByPassActivate:
        case aplArkamysAvcActivate:
        case aplArkamysAmbienceSelect:
        case aplArkamysInputLevelDeviation:
        case aplArkamysCurrentSpeed:
        case aplArkamysCurrentVolumeStep:
        case aplArkamysCurrVolAttenuation:
        case aplArkamysLibraryVersion:
        case aplArkamysFlush:
        {
                /* release mutex and allow read parameter update */
            pr_message("apl sync for parameter -> %d\n", param_Id);
            ETG_TRACE_USR3(("[AplNotifyCb]: apl sync for parameter -> %d", param_Id));  /*changed*/

            #ifdef D_SAFE_NOTIFICATION
            audproc_service_wait_apl_notification_stop();
            #else
            audproc_service_wait_notification = FALSE;
            g_cond_broadcast(&audproc_service_update);
            #endif
            break;
        }
        default:
            pr_debug("paramter id(%d) no supported\n", param_Id);
            break;
    }


    ETG_TRACE_USR3(("[audproc_service_send_notification]: EXIT"));
    return err;

}


/*******************************************************************************
 *
 * FUNCTION: audproc_service_init
 *
 * DESCRIPTION:
 *
 * PARAMETER: [check xml file for parameter description]
 *
 * RETURNVALUE: int
 *
 *******************************************************************************/
void audproc_service_init(gboolean isReconfig, guint8 audioMode)
{
    int                             cfgidx;
    audproc_alsa_stream_cfg*        ps = (audproc_alsa_stream_cfg*)NULL;

    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_service_init]: ENTERED with audiomode-> %d", audioMode));

    g_mutex_init(&audproc_service_lock);
    g_mutex_lock(&audproc_service_lock);
    g_cond_init(&audproc_service_update);
    audproc_service_select_loopback_route = FALSE;


    /************************
     *   module initialization
     */

    /* init audioprocessing Library (apl)*/
    tAplStCfg papl;
    papl.AplNotifyfct = AplNotifyCb;
    papl.AplIsLoadArkamays = FALSE;
    //papl.AplTracecb = AplTracecb;

    (void)audproc_apl_handler_init(&papl);

    /* init derivated module */
    audproc_configuration_init(isReconfig, audioMode);
    audproc_alsa_init();
    audproc_audio_rts_handler_init();
    audproc_state_machine_init();

    /* walk through the configuration */
    gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

    my_servs = g_new0(struct audproc_instance_object*, nb_route_max);

    if(my_servs)
    {
        for(cfgidx = 0; cfgidx < nb_route_max; cfgidx++)
        {
            ps = audproc_configuration_get_route_cfg_list_idx((guint)cfgidx);
            if(!ps)
            {
                ETG_TRACE_ERR(("[audproc_service_init]: no streaming configuration available for cfgidx(%d)", cfgidx));
                pr_warning("no streaming configuration available for cfgidx(%d)\n", cfgidx);

                continue;
            }

            /* create this route on module initialization  */
            if(ps->create_route_on_init)
            {
                /* create new service instance */
                struct audproc_instance_object* pd = audproc_service_create_new_instance(ps->source_id);
                if(pd)
                {
                    pr_message("create new service instance entry for(%d): %s\n", ps->source_id, audproc_service_src_str(ps->source_id));
                    ETG_TRACE_USR4(("[audproc_service_init]: create new service instance entry for(%d): %s\n", ps->source_id, audproc_service_src_str(ps->source_id)));
                    (void)audproc_service_add_new_instance(pd);
                }
            }
        }
    }
    else
        pr_critical("memory failure while trying to allocate service object\n");


    /**/
    audproc_apl_init_data = (tAplCfg*)g_malloc0(sizeof(tAplCfg));

    /* print source state */
    audproc_service_print_src_instance_state();

    g_mutex_unlock(&audproc_service_lock);

    /*new*/
    audproc_alsa_init();

    ETG_TRACE_USR3(("[audproc_service_init]: EXIT"));
    pr_message("EXIT");
    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_do_source_activity
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_source_activity
(
    guchar      AudprocSource,
    guchar      AudprocSourceActivity
)
{
    int err = (int)AUDPROC_ERR_OK;


    /*********************************************************************
     * automatical source selectio sequence
     * 0. call lookup table to translate IN source to intern source defintion
     * 1. stop active source if same src type
     * 2. initialize source : apl
     * 3  start streaming
     */

    /*****************************************
     *  proceeed to step 0
     */
    guint8  audproc_internal_src_def = 0;

    err = audproc_configuration_look_up_src_table(AudprocSource, &audproc_internal_src_def, TRUE);


    pr_message("ENTERED: requested source(%d) with SA(%d)\n", audproc_internal_src_def, AudprocSourceActivity);
    ETG_TRACE_USR3(("[audproc_service_do_source_activity]: requested midw source id     -> %d", AudprocSource));
    ETG_TRACE_USR3(("[audproc_service_do_source_activity]: associated audproc source id -> %d", audproc_internal_src_def));
    ETG_TRACE_USR3(("[audproc_service_do_source_activity]: requested source activity    -> %d", AudprocSourceActivity));

    if(!err)
    {

        const gchar* str = audproc_configuration_get_inst_name(audproc_internal_src_def);

        if(str)
        {
            switch((audproc_source_activity)AudprocSourceActivity)
            {
                case AUDPROC_SA_PREPARE:
                case AUDPROC_SA_ON:
                {
                    /* 1. first flush memory item in the audio processing functions*/
                    err = audproc_apl_handler_command(str, aplCmdFlush);

                    if(!err)
                    {
                        gboolean isFlatReq;
                        tAplI32 aplswitch = aplSwitchON;
                        AplState* aplinst = audproc_apl_handler_get_instance(str);

                        err = audproc_configuration_look_up_flat_req(AudprocSource, &isFlatReq);
                        if(!err)
                        {
                            if(isFlatReq)
                            {
                                pr_message("set source(%d) to flat \n", AudprocSource);
                                (void)audproc_apl_handler_set_data(aplinst, aplArkamysByPassActivate, 0, sizeof(tAplI32),(void*)&aplswitch );
                            }
                            else
                            {
                                pr_message("remove always flat setting by source(%d)\n", AudprocSource);
                                aplswitch = aplSwitchOFF;
                                (void)audproc_apl_handler_set_data(aplinst, aplArkamysByPassActivate, 0, sizeof(tAplI32),(void*) &aplswitch );
                            }
                        }
                        else
                            pr_warning("audproc_configuration_look_up_flat_req call fails with err: %d\n", err);
                    }
                    else
                        pr_warning("audproc_apl_handler_command call fails with err: %d\n", err);


                    if(audproc_service_select_loopback_route)
                    {
                        struct audproc_instance_object* req_obj = audproc_service_get_instance(audproc_internal_src_def);
                        if(!req_obj)
                        {
                            pr_debug("no service instance for source(%d)\n", audproc_internal_src_def);
                            err =  AUDPROC_ERR_SERV_NOT_INSTANCED;
                        }
                        else
                        {
                            g_mutex_lock(&audproc_service_lock);
                            pr_debug("activate loopback\n");
                            req_obj->audproc_audio_instance->select_loopback = TRUE;
                            g_mutex_unlock(&audproc_service_lock);
                        }
                    }

                    break;
                }
                case AUDPROC_SA_OFF:
                {
                    /* do nothing */
                    break;
                }
                default:
                    break;
            }
        }
        else
        {
            pr_warning("the source id(%d) is not made available per configuration\n", audproc_internal_src_def);
            err = (int)AUDPROC_ERR_SRC_CFG_NOT_AVAILABLE;
        }
    }
    else
    {
        pr_debug("the midw source(%d) is disabled per configuration\n", AudprocSource);
        ETG_TRACE_USR3(("[audproc_service_do_source_activity}: the midw source(%d) is disabled per configuration", AudprocSource));
    }

    /* Now start streaming if required
    if(    !err
        && (AudprocSourceActivity == AUDPROC_SA_ON))
    {
        err = audproc_service_do_start_streaming(audproc_internal_src_def);
    }
    */
    pr_message("EXIT\n");

    return err;
}

//#define D_TEST__CLOSE_FORCE_DEV_FEAT

/*******************************************************************************
*
* FUNCTION: audproc_service_do_select_active_source
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_select_active_source
(
    audproc_source_select_Mode  AudprocSourceSelectMode,
    guchar                      AudprocSourceId,
    gchar*                      AudprocAlsaDev
)
{
    int                                 err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*     req_obj;
    struct audproc_instance_object*     act_obj;
    audproc_source_activity             req_activity = AUDPROC_SA_UNDEF;
    guchar                              src_id = 0;
    gchar*                              audprocADev = (gchar*)NULL;
    gboolean                            bparam3avail = TRUE;


    ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: ENTERED"));
    pr_message("ENTERED");


    /*********************************************************************
     * automatical source selectio sequence
     * 0. call lookup table to translate IN source to intern source defintion
     * 1. stop active source if same src type
     * 2. initialize source : apl
     * 3  start streaming
     */

    /*****************************************
     *  proceeed to step 0
     */
    guint8  audproc_internal_src_def = 0;

    /* copy input string */
    if(   (NULL == AudprocAlsaDev)
        ||('\0' == AudprocAlsaDev[0]))
    {
#ifdef D_TEST__CLOSE_FORCE_DEV_FEAT
        //audprocADev = (gchar*)g_malloc0(40);
        audprocADev = g_strdup("CloseEntSinkAdev");
        pr_message("create audprocADev and assign the string \"CloseEntSinkAdev\"\n");
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: create audprocADev and assign the string \"CloseEntSinkAdev\""));
#else
        bparam3avail = FALSE;
#endif
    }
    else
        audprocADev = g_strdup(AudprocAlsaDev);

    if(AudprocSourceSelectMode == AUDPROC_SELECT_SRC_PER_DEV)
    {

        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: AudprocSourceSelectMode -> %s", "AUDPROC_SELECT_SRC_PER_DEV"));
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Requested Activity      -> %s", audproc_service_activity_str(AudprocSourceId)));
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Requested input device  -> %s", AudprocAlsaDev ? AudprocAlsaDev : "UNDEFINED"));


        /* the audio route is setup, no audio processing is done */

        if(!bparam3avail)
        {
            ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: the select request is invalid, no route name was passed by the application"));
            ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: EXIT"));
            pr_message("the select request is invalid, no route name was passed by the application\n");
            pr_message("EXIT\n");
            return (int)AUDPROC_ERR_OK; //err = AUDPROC_ERR_NULL_POINTER;
        }
        else
        {
            /*****************************************
             * This parameter in this case is used
             * to setup  the connection state
             * - AUDPROC_SA_ON, AUDPROC_SA_ON
             */
            if(AudprocSourceId == AUDPROC_SA_ON)
            {
                req_activity = AUDPROC_SA_ON;
            }
            else if(AudprocSourceId == AUDPROC_SA_OFF)
            {
                req_activity = AUDPROC_SA_OFF;
            }
            else if(AudprocSourceId == AUDPROC_SA_ACTIVATE)
            {
                req_activity = AUDPROC_SA_ACTIVATE;
            }
            else if(AudprocSourceId == AUDPROC_SA_DEACTIVATE)
            {
                req_activity = AUDPROC_SA_DEACTIVATE;
            }
            else
            {
                req_activity = AUDPROC_SA_UNDEF;
            }

            /***************************************************
             * check device name
             * if device name == CloseEntSinkAdev
             * - the entertainemnt alsa device instance is release to avaoid a conflit with
             *   another application which uses a device that requires usage of the same resource.
             * - with req_activity == AUDPROC_SA_DEACTIVATE, the sink device instance is closed if already actived
             * - with req_activity == AUDPROC_SA_ACTIVATE, the sink device instance is restored if it was active while deactivation was closed
             */

            //CloseEntSinkAdev
            if(NULL != strstr(audprocADev, "Close"))
            {
                pr_message("request to close or restore a device instance only -> %s \n", audprocADev);
                ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: request to close a device instance only -> %s ", audprocADev));

                act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
                if(act_obj)
                {
                    pr_message("A entertainment route is currently active, the sink device \"%s\" cannot be closed", act_obj->audproc_audio_instance->audproc_stream_data->outdev.audproc_dev_name);
                    ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: A entertainment route is currently active, the sink device \"%s\" cannot be closed", act_obj->audproc_audio_instance->audproc_stream_data->outdev));
                    return (int)AUDPROC_ERR_OK;
                }

                if(NULL != strstr(audprocADev, "Ent"))
                {
                    if(NULL != strstr(audprocADev, "Sink"))
                    {
                        /* search for first AUD_PROC_SRC_TYPE_ENT configuration input */
                        struct audproc_instance_object* src_ent_type = audproc_service_get_first_type_src_occurence(AUD_PROC_SRC_TYPE_ENT);
                        if(src_ent_type)
                        {
                            audproc_alsa_adev_cfg* pb = (audproc_alsa_adev_cfg*)&src_ent_type->audproc_audio_instance->audproc_stream_data->outdev;
                            if(req_activity == AUDPROC_SA_ACTIVATE)
                            {
                                (void)audproc_audio_rts_handler_force_close((audproc_alsa_state*)src_ent_type->audproc_audio_instance,
                                                                   pb->audproc_dev_name,
                                                                   D_CONV_PLAYBACK_DEVICE);

                                pr_message("Entertainment sink device \"%s\" instance was closed", pb->audproc_dev_name);
                                ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Entertainment sink device \"%s\" instance was closed", pb->audproc_dev_name));
                                return (int)AUDPROC_ERR_OK;
                            }
                            else if(req_activity == AUDPROC_SA_DEACTIVATE)
                            {
                                (void)audproc_audio_rts_handler_restore_state((audproc_alsa_state*)src_ent_type->audproc_audio_instance,
                                                                          FALSE,
                                                                          D_CONV_PLAYBACK_DEVICE);
                                pr_message("Entertainment sink device \"%s\" instance is restored", pb->audproc_dev_name);
                                ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Entertainment sink device \"%s\" instance is restored", pb->audproc_dev_name));
                                return (int)AUDPROC_ERR_OK;
                            }
                        }
                    }
                }
            }


            err = audproc_configuration_get_src_id_from_capture_dev(audprocADev, &audproc_internal_src_def);

            /* the requested alsa device is not supported in the current active configuration  */
            if(err == AUDPROC_ERR_SRC_CFG_NOT_AVAILABLE)
            {
                /***************************************************
                * - special handling required if following conditions are fullfilled:
                * - an entertainment source/audio route is currently active?
                * - the present request refers to an "AdevEntADRIn" device
                * - the requested activity is AUDPROC_SA_ACTIVATE -> fix 26.09.17 (ticket NCG3D-63717)
                * - action to perform : stop the streaming to avoid undesirable process load
                */

                if (   (req_activity == AUDPROC_SA_ACTIVATE)
                    && !g_strcmp0(audprocADev, "AdevEntADRIn"))
                {
                    act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
                    if(act_obj)
                    {
                        err = audproc_service_do_stop_streaming(act_obj->audproc_this_src);

                        /* change state active source */
                        act_obj->audproc_is_initialized = FALSE;
                        act_obj->audproc_is_streaming_started = FALSE;
                        act_obj->audproc_is_streaming_running_state = FALSE;
                        act_obj->audproc_is_this_src_active = FALSE;
                        act_obj->audproc_is_apl_instance_created = FALSE;
                        act_obj->audproc_astate = AUDPROC_AS_STREAM_SILENCE;

                        pr_debug("stop entertainment route, the requested route %s is not configured as entertainemnt one\n", audprocADev);
                        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: stop entertainment route, the requested route %s is not configured as such", audprocADev));
                        return (int)AUDPROC_ERR_OK;
                    }
                }
            }
        }
    }
    else if (AudprocSourceSelectMode == AUDPROC_SELECT_SRC_PER_ID)
    {
        audproc_internal_src_def = AudprocSourceId;

        if (!g_strcmp0(audprocADev, "AUDPROC_SA_OFF"))
        {
            req_activity = AUDPROC_SA_OFF;
        }
        else if (!g_strcmp0(audprocADev, "AUDPROC_SA_ON"))
        {
            req_activity = AUDPROC_SA_ON;
        }
        else if (!g_strcmp0(audprocADev, "AUDPROC_SA_ACTIVATE"))
        {
            req_activity = AUDPROC_SA_ACTIVATE;
        }
        else if (!g_strcmp0(audprocADev, "AUDPROC_SA_DEACTIVATE"))
        {
            req_activity = AUDPROC_SA_DEACTIVATE;
        }
        else
        {
            req_activity    = AUDPROC_SA_UNDEF;
        }

        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: AudprocSourceSelectMode         -> %s", "AUDPROC_SELECT_SRC_PER_ID"));
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Requested audproc source/route  -> %s", audproc_service_src_str(AudprocSourceId)));
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Requested Activity              -> %s", audproc_service_activity_str(req_activity)));


    }
    else if(AudprocSourceSelectMode == AUDPROC_SELECT_SRC_PER_MDW_SRC_DEF)
    {
        err = audproc_configuration_look_up_src_table(AudprocSourceId, &audproc_internal_src_def, TRUE);

        if(!err)
        {
            if (!g_strcmp0(audprocADev, "AUDPROC_SA_OFF"))
            {
                req_activity = AUDPROC_SA_OFF;
            }
            else if (!g_strcmp0(audprocADev, "AUDPROC_SA_ON"))
            {
                req_activity = AUDPROC_SA_ON;
            }
            else if (!g_strcmp0(audprocADev, "AUDPROC_SA_ACTIVATE"))
            {
                req_activity = AUDPROC_SA_ACTIVATE;
            }
            else if (!g_strcmp0(audprocADev, "AUDPROC_SA_DEACTIVATE"))
            {
                req_activity = AUDPROC_SA_DEACTIVATE;

            }
            else
            {
                req_activity    = AUDPROC_SA_UNDEF;
            }
        }

        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: AudprocSourceSelectMode   -> %s", "AUDPROC_SELECT_SRC_PER_MDW_SRC_DEF"));
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Requested midw source     -> %s", AudprocSourceId));
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: Requested Activity        -> %s", audproc_service_activity_str(req_activity)));

    }
    else
    {
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: SourceSelectMode(%d) not supported, no action is considered", AudprocSourceSelectMode));
        pr_warning("SourceSelectMode(%d) not supported, no action is considered\n", AudprocSourceSelectMode);
        return (int)AUDPROC_ERR_SERV_ROUTE_NOT_AVAILABLE; // AUDPROC_ERR_OK;
    }

    /*****************************************
     * Source deactivation/activation is requested?
     * it should be executed only if the requested
     * source matches with the active one.
     */
    if(req_activity != AUDPROC_SA_UNDEF)
    {
        req_obj = audproc_service_get_instance(audproc_internal_src_def);
        if(req_obj)
        {
            act_obj = audproc_service_get_active_type_src(req_obj->audproc_this_src_type);
            if(act_obj)
            {
                if(act_obj->audproc_this_src != audproc_internal_src_def)
                {
                    pr_warning("The requested src(%d) does not match with the active one(%d), SA(%d) cannot be considered \n", audproc_internal_src_def, act_obj->audproc_this_src, req_activity);
                    ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: The requested src(%d) does not match with the active one(%d), SA(%d) cannot be considered", audproc_internal_src_def, act_obj->audproc_this_src, req_activity));
                    return (int)AUDPROC_ERR_SERV_ROUTE_REQUEST_MISMATCH; //AUDPROC_ERR_OK;
                }
            }
        }
        else
            return (int)AUDPROC_ERR_SERV_ROUTE_NOT_AVAILABLE; // AUDPROC_ERR_OK;
    }

    if(!err)
    {
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: process the request"));
        ETG_TRACE_USR3(("[lvar1]: new request         -> %s ", audproc_service_activity_str(req_activity)));
        ETG_TRACE_USR3(("[lvar2]: audproc source id   -> %s ", audproc_service_src_str(audproc_internal_src_def)));
        ETG_TRACE_USR3(("[lvar3]: capture device name -> %s ", audprocADev));

        pr_message("Req(%s) for source(%s) with capture device(%s) \n", audproc_service_activity_str(req_activity)
                                                                                ,audproc_service_src_str(audproc_internal_src_def)
                                                                                , audprocADev );

        /********************************************
         *  step 1n and 2 are done with the this call
         * Todo: is mute required ?
         */

        req_obj = audproc_service_get_instance(audproc_internal_src_def);

        /* first stop the current active route / source */
        if(req_obj)
            act_obj = audproc_service_get_active_type_src(req_obj->audproc_this_src_type);
        else
        {
            /* no source active from same type as the requested one */
            act_obj =  (struct audproc_instance_object*)NULL;
        }

        /* first stop current active source */
        if(act_obj)
        {
            /*is this route presently active then do nothing ?*/
            if(   (act_obj->audproc_this_src == audproc_internal_src_def)
               && (   ((req_activity != AUDPROC_SA_OFF)&&(req_activity != AUDPROC_SA_DEACTIVATE))
                    ||(req_activity == AUDPROC_SA_UNDEF)))
            {
                pr_warning(" the route %d is presently active \n", audproc_internal_src_def);
                ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: the route %d is presently active", audproc_internal_src_def));
                return (int)AUDPROC_ERR_OK;
            }

            /* change source activity only with request AUDPROC_SA_OFF */
            if(req_activity == AUDPROC_SA_OFF)
            {
                err = audproc_configuration_look_up_src_table(audproc_internal_src_def, &src_id, FALSE);
                if(!err)
                {
                    err = audproc_service_do_source_activity(src_id, AUDPROC_SA_OFF);
                }
                else
                {
                    pr_warning("No translation conf. found for the requested internal source(%d)\n", audproc_internal_src_def);
                    err = (int)AUDPROC_ERR_OK;
                }
            }

            if(!err)
            {
                err = audproc_service_do_stop_streaming(act_obj->audproc_this_src);

                /* change state active source */
                act_obj->audproc_is_initialized = FALSE;
                act_obj->audproc_is_streaming_started = FALSE;
                act_obj->audproc_is_streaming_running_state = FALSE;
                act_obj->audproc_is_this_src_active = FALSE;
                act_obj->audproc_is_apl_instance_created = FALSE;
                act_obj->audproc_astate = AUDPROC_AS_STREAM_SILENCE;

                /* send stop command */
                const gchar* str = audproc_configuration_get_inst_name(audproc_internal_src_def);
                if(str)
                {
                    err = audproc_apl_handler_command(str, aplCmdStopAudProc);
                }
            }
        }

        if(   (req_activity == AUDPROC_SA_ON)
            ||(req_activity == AUDPROC_SA_ACTIVATE)
            ||(req_activity == AUDPROC_SA_UNDEF))
        {
            err = audproc_service_do_initialize(audproc_internal_src_def, NULL);

            /* step 3*/
            if(!err)
            {
                err = audproc_service_do_start_streaming(audproc_internal_src_def);

                if(!err)
                {
                    if(req_activity == AUDPROC_SA_ON)
                    {
                        err = audproc_configuration_look_up_src_table(audproc_internal_src_def, &src_id, FALSE);
                        if(!err)
                        {
                            err = audproc_service_do_source_activity(src_id, AUDPROC_SA_ON);
                        }
                        else
                        {
                            pr_warning("No translation conf. found for the requested internal source(%d)\n", audproc_internal_src_def);
                            err = (int)AUDPROC_ERR_OK;
                        }
                    }

                    /* send start command */
                    const gchar* str = audproc_configuration_get_inst_name(audproc_internal_src_def);
                    if(str)
                    {
                        err = audproc_apl_handler_command(str, aplCmdStartAudProc);
                    }
                }
            }
        }
    }
    else
    {
        pr_debug("no translation to input source per configuration available\n");
        ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: no translation to input source per configuration available"));
    }

    if(audprocADev)
        g_free(audprocADev);


    ETG_TRACE_USR3(("[audproc_service_do_select_active_source]: EXIT"));
    pr_message("EXIT\n");

    return err;
}



/*******************************************************************************
*
* FUNCTION: audproc_service_setup_init_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
static gint audproc_service_setup_init_data (gchar* src_file, tAplCfg* data)
{
    gint      err = (int)AUDPROC_ERR_OK;
    FILE            *file_hdl = NULL;   /* pointer to config data file */
    long int    file_size = 0;   /* Config data buffer size */

    pr_debug("Entered");


    /* check for NULL pointer */
    if( NULL == data)
    {
        pr_warning("app config struct pointer NULL");
        return AUDPROC_ERR_NULL_POINTER;
    }

    /* File name is empty string or NULL pointer means we do not have a
     * config file (not really an error)
     */
    if((NULL != src_file) && ('\0' != src_file[0]))
    {
        file_hdl = fopen(src_file, "rb");

        if(NULL == file_hdl)
        {
            pr_warning("fail to open the configuration file:  %s", src_file);
            err = AUDPROC_ERR_NULL_POINTER; /* opening file failed */
        }
        else
        {
            /* find out the length of the file */
            fseek(file_hdl, 0L, SEEK_END);
            file_size = ftell(file_hdl);
            fseek(file_hdl, 0L, SEEK_SET); /* go back to the beginning */

            if (file_size == 0)
            {
                pr_warning("reading from file failed (empty file)");
                err = AUDPROC_ERR_CONFIG_FILE_READING_FAILED; /* memory allocation failed */
            }
            else
            {
                /* allocate memory for config data buffer */

                data->usSize = (tAplU16)file_size; /* size of config buffer,TODO: Gen4 LP64 */
                data->pData = (guchar *)g_malloc0((gsize)file_size);

                if(NULL == data->pData)
                {
                    pr_warning("fail to read the configuration file, memory allocation failed");
                    err = AUDPROC_ERR_CONFIG_FILE_READING_FAILED;
                }
                else
                {
                    if(file_size != (long int)fread(data->pData, (size_t)1, (size_t)file_size, file_hdl))
                    {
                        pr_warning("fail to read the configuration file");
                        err = AUDPROC_ERR_CONFIG_FILE_READING_FAILED;
                    }
                }
            }
        }
    }
    else
    {
        pr_warning("Filename is NULL");
        err = AUDPROC_ERR_CONFIG_FILE_EMPTY; /* memory allocation failed */
    }

    /* close file */
    if(NULL != file_hdl)
    {
        fclose(file_hdl);
    }

    pr_debug("Leaved with error: %d",err );

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_alsa_thread_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_initialize
(
    guchar                  AudprocSourceId,
    gchar*                  cfgfilesrc
)
{
    int err = AUDPROC_ERR_OK;
    struct audproc_instance_object*      req_obj;
    struct audproc_instance_object*      act_obj;
    gboolean                             is_new_instance_created = FALSE;

    pr_message("ENTERED: requested source(%d) and configuration file(%s)\n", AudprocSourceId,  (cfgfilesrc != NULL)? cfgfilesrc :"NULL");
    ETG_TRACE_USR3(("[audproc_service_do_initialize]: ENTERED with paramters"));
    ETG_TRACE_USR3(("[in_param1]: AudprocSourceId -> %d", AudprocSourceId));
    ETG_TRACE_USR3(("[in_param2]: cfgfilesrc      -> %s",(cfgfilesrc != NULL)? cfgfilesrc : "UNUSED" ));

    /*********************************************************************
     * first check availability of service instance
     * for the requested source
     */

    req_obj = audproc_service_get_instance(AudprocSourceId);

    if(!req_obj)
    {
        /*create new instance */
        req_obj = audproc_service_create_new_instance(AudprocSourceId);
        if(!req_obj)
            return AUDPROC_ERR_ALLOC;

        is_new_instance_created = TRUE;
    }
#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_LSIM //NCG3D-90366
    if(AudprocSourceId ==AUD_PROC_SRC_VOICE1)
    {
      ETG_TRACE_USR3(("[audproc_service_do_initialize][LSIM]: /usr/bin/amixer set cap unmute"));
      system("/usr/bin/amixer sset Line 100%,100% unmute cap");
    }
#endif
    /*********************************************************************
     * retrieve actual active source
     * - source is active
     * - type active source same as the requested one
     * - deactivate current source
     */
    act_obj = audproc_service_get_active_type_src(req_obj->audproc_this_src_type);


    /* do nothing requested audio source is already activated, return OK */
    if(    act_obj
        && !is_new_instance_created
        && (act_obj->audproc_this_src == AudprocSourceId))
    {
        pr_debug("requested source(%s) is already activated\n", audproc_service_src_str(AudprocSourceId));
        return AUDPROC_ERR_OK;
    }

    /* first stop current active source */
    if( act_obj)
        //&& !act_obj->audproc_audio_instance->isRouting)
    {
        err = audproc_service_do_stop_streaming(act_obj->audproc_this_src);

        /* change state active source */
        act_obj->audproc_is_initialized = FALSE;
        act_obj->audproc_is_streaming_started = FALSE;
        act_obj->audproc_is_streaming_running_state = FALSE;
        act_obj->audproc_is_this_src_active = FALSE;
        act_obj->audproc_is_apl_instance_created = FALSE;
        act_obj->audproc_astate = AUDPROC_AS_STREAM_SILENCE;
    }

    /* assigned active object with new instanced source */
    act_obj = req_obj;


    /*************************************
     * now initialize apl for the new source
     */

    if(err == AUDPROC_ERR_OK)
    {
        const gchar* str = audproc_configuration_get_inst_name(AudprocSourceId);

        if((NULL == cfgfilesrc) || ('\0' == cfgfilesrc[0]))
        {
            err = audproc_configuration_set_configuration(AudprocSourceId);
        }
        else
        {
            pr_debug("configuration file is: %s\n", cfgfilesrc);

            if(audproc_apl_init_data)
            {
                if(audproc_apl_init_data->pData)
                {
                    g_free(audproc_apl_init_data->pData);
                    audproc_apl_init_data->pData = NULL;
                    audproc_apl_init_data->usSize = 0;
                }

                err = audproc_service_setup_init_data(cfgfilesrc, audproc_apl_init_data);
                if(!err)
                {
                    err = audproc_apl_handler_initialize(str, audproc_apl_init_data);
                }
            }
        }
    }

    if (err == AUDPROC_ERR_OK)
    {
        g_mutex_lock(&audproc_service_lock);

        act_obj->audproc_is_initialized = TRUE;
        act_obj->audproc_is_this_src_active = TRUE;
        act_obj->audproc_is_apl_instance_created = TRUE;

        /* add new instance in thw service list */
        if(is_new_instance_created)
        {
            (void)audproc_service_add_new_instance(act_obj);
        }

        g_mutex_unlock(&audproc_service_lock);
    }

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_do_start_streaming
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_start_streaming(guchar AudprocSourceId)
{
    int err = AUDPROC_ERR_OK;
    struct audproc_instance_object*      req_obj;
    audproc_alsa_stream_cfg* data;

    pr_message("ENTERED: requested source -> %d\n", AudprocSourceId);

    ETG_TRACE_USR3(("[audproc_service_do_start_streaming]: ENTERED "));
    ETG_TRACE_USR3(("[in_param1]: AudprocSourceId -> %d", AudprocSourceId));



    req_obj = audproc_service_get_instance(AudprocSourceId);
    if(!req_obj)
    {
        pr_message("EXIT with invalide service instance\n");
        return AUDPROC_ERR_SERV_NOT_INSTANCED;
    }

    if(!req_obj->audproc_is_initialized)
    {
        pr_message("EXIT with uninitialized streaming instance\n");
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;
    }

    /******************************
     * check current streaming activity:
     * - requested source should be in stop state
     * - return failure is a source from same type is activated
     *   in this case the applicatino shoudl first stop teh cuurent active source
     */
    if(req_obj->audproc_is_streaming_started)
    {
        pr_message("EXIT with streaming already started\n");
        return AUDPROC_ERR_OK;
    }

    audproc_alsa_stream_cfg*  p = audproc_configuration_get_route_cfg(AudprocSourceId);
    audproc_source_type       src_type = p->audcfg.audproc_this_src_type;

    data =  req_obj->audproc_audio_instance->audproc_stream_data;

    if(   (src_type == AUD_PROC_SRC_TYPE_DEFAULT)
        ||(src_type == AUD_PROC_SRC_TYPE_ENT))
    {
        err = audproc_alsa_start_ent_instance(data->audproc_start_time_out_ms,
                                              req_obj->audproc_audio_instance);
    }
    else if(src_type == AUD_PROC_SRC_TYPE_VOICE)
    {
        err = audproc_alsa_start_voice_instance(data->audproc_start_time_out_ms,
                                                req_obj->audproc_audio_instance);
    }
    else if(src_type == AUD_PROC_SRC_TYPE_INFO)
    {
        err = audproc_alsa_start_info_instance(data->audproc_start_time_out_ms,
                                                req_obj->audproc_audio_instance);
    }
    else
    {
        /*tbd*/
    }

    if(err == AUDPROC_ERR_OK)
    {
        g_mutex_lock(&audproc_service_lock);
        req_obj->audproc_is_streaming_started = TRUE;
        pr_message("the streaming thread was started successfully\n");

        /* notify client source state change */
        //audproc_object_emit_signal_sources_state();

        g_mutex_unlock(&audproc_service_lock);
    }
    pr_message("EXIT\n");

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_do_stop_streaming
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_stop_streaming(guchar AudprocSourceId)
{
    int err = AUDPROC_ERR_OK;
    struct audproc_instance_object*      req_obj;


    pr_message("ENTERED: source requested(%d) \n", AudprocSourceId);
    ETG_TRACE_USR3(("[audproc_service_do_stop_streaming]: ENTERED"));
    ETG_TRACE_USR3(("[in_param1]: AudprocSourceId -> %d", AudprocSourceId));

    req_obj = audproc_service_get_instance(AudprocSourceId);
    if(!req_obj)
    {
        pr_debug("no service instance for source(%d)\n", AudprocSourceId);
        ETG_TRACE_USR1(("[audproc_service_do_stop_streaming]: no service instance for source(%d)", AudprocSourceId));
        return AUDPROC_ERR_SERV_NOT_INSTANCED;
    }

    if(!req_obj->audproc_is_initialized)
    {
        pr_warning("try to start audio streaming in uninitialized state\n");
        ETG_TRACE_USR1(("[audproc_service_do_stop_streaming]: try to start audio streaming in uninitialized state"));
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;
    }

    if(!req_obj->audproc_is_streaming_started)
    {
        pr_debug("steaming was not started, req_obj->audproc_is_streaming_stoped(%d)\n", req_obj->audproc_is_streaming_started);
        ETG_TRACE_USR1(("[audproc_service_do_stop_streaming]: steaming was not started, req_obj->audproc_is_streaming_stoped(%d)", req_obj->audproc_is_streaming_started));
        return AUDPROC_ERR_OK;
    }

    err = audproc_alsa_stop_instance(req_obj->audproc_audio_instance);

    if(err == AUDPROC_ERR_OK)
    {
        g_mutex_lock(&audproc_service_lock);

        /* print stream statistic */
        pr_message("--------- stream statistic after audio stop  ----------- \n");
        pr_message(" write count            : %d \n", req_obj->audproc_audio_instance->rts_write_cnt);
        pr_message(" read  count            : %d \n", req_obj->audproc_audio_instance->rts_read_cnt);
        pr_message(" xrun count pb device   : %d \n", req_obj->audproc_audio_instance->rts_xrun_pb_count);
        pr_message(" xrun count cap device  : %d \n", req_obj->audproc_audio_instance->rts_xrun_cap_count);

        req_obj->audproc_is_streaming_started = FALSE;
        req_obj->audproc_audio_instance->rts_write_cnt = 0;
        req_obj->audproc_audio_instance->rts_read_cnt = 0;
        req_obj->audproc_audio_instance->isAudioprocessing = FALSE;

        g_mutex_unlock(&audproc_service_lock);

        //pr_message("EXIT");
    }
    pr_message("EXIT");
    ETG_TRACE_USR3(("[audproc_service_do_stop_streaming]: EXIT"));
    return err;
}



static int   audproc_service_select_audio_mode(guint32 audiomode)
{
    int err = (int)AUDPROC_ERR_OK;

#ifndef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    pr_message("select audio mode:%d\n", audiomode);

    switch(audiomode)
    {
        case D_AUDIO_MODE_CLASSIC:
            err = audproc_configuration_set_audio_mode_classic();
            break;
        case D_AUDIO_MODE_AUDITORIUM:
            err = audproc_configuration_set_audio_mode_auditorium();
            break;
        case D_AUDIO_MODE_PREMIUM:
            err = audproc_configuration_set_audio_mode_premium();
            break;
        case D_AUDIO_MODE_EOL_RELOAD:
            err = audproc_configuration_set_audio_mode_eol_reload();
            break;
        default:
            pr_warning("audioMode(%d) not supported\n", audiomode);
            break;
    }
#else
     pr_message("select always D_AUDIO_MODE_WEB_CONTAINER with psarcc build variant\n");
#endif
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_do_set_param
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_set_param
(
    guchar              AudprocSourceId,
    guint32             param_id,
    audproc_type        data_type,
    const GValue*       param
)
{
    int                         err = (int)AUDPROC_ERR_OK;
    unsigned int                size, len;
    const struct aplParameter*  p;
    GArray*                     ga;
    gpointer                    a;
    AplState*                   aplinst =  (AplState*)NULL;


    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_service_do_set_param]: ENTERED"));

    /* parameter infos*/
    p = audproc_configuration_get_param_infos(param_id);

    if(!p)
    {
        pr_debug("parameter Id(%d) not available\n", param_id);
        return AUDPROC_ERR_PARAM;
    }

    pr_message("property_id(%d), property_name(\"%s\")\n", param_id, p->apl_name);
    ETG_TRACE_USR3(("[audproc_service_do_set_param}: ENTERED "));
    ETG_TRACE_USR3(("[in_param1]: property_id   -> %d", param_id));
    ETG_TRACE_USR3(("[in_param2]: property_name -> %s", p->apl_name ? p->apl_name :"UNDEFINED"));


    aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

    if(!aplinst)
    {
        pr_debug("apl instance(%s) is not available, shall be created\n", p->apl_instance_name);

        if(g_strcmp0(p->apl_instance_name, APL_INSTANCE_STR_ALL_FCTS))//p->apl_instance_name != APL_INSTANCE_STR_ALL_FCTS)
            err = audproc_configuration_set_configuration_by_apl_inst(p->apl_instance_name);
        else
        {
            aplinst = audproc_apl_handler_create_instance(APL_INSTANCE_STR_ALL_FCTS, APL_INIT_PREPARE_INST);
        }

        /* now retrieve apl instance */
        if(!err)
            aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

        if(!aplinst)
        {
            pr_warning("try to apl arkamys instance fails \n");
            return AUDPROC_ERR_PARAM;
        }
    }

    /* initialize async call */
    /* if streaming aud audio processinf are not cuurent active than send level loss value from here*/
    struct audproc_instance_object*     act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
    if(    act_obj
        && act_obj->audproc_is_streaming_running_state) //audproc_is_streaming_started)
    {
            pr_message("enable apl notification wait\n");
            audproc_service_wait_apl_notification_start();
    }

    /* handle here with command*/
    if((tAplDataID)param_id == aplArkamysFlush)
    {
        err = audproc_apl_handler_command(p->apl_instance_name, aplCmdFlush);
        if(!err)
        {
            /* initialize async call */
            /* if streaming aud audio processinf are not cuurent active than send level loss value from here*/
            err = audproc_service_wait_apl_notification();

            ETG_TRACE_USR3(("[audproc_service_do_set_param]: EXIT"));
            return err;
        }
    }

#ifdef D_APL_CHECK_PARAMETER_CHANGE
    else if((tAplDataID)param_id == aplArkamysInputLevelDeviation)
    {
        gint32 p32;
        p32 = g_value_get_int(param);

        ETG_TRACE_USR4(("[audproc_service_do_set_param]: requested value for the parameter aplArkamysInputLevelDeviation -> %d, return wi/o call to the arkamysy library ", p32));

        if(p32 == audproc_last_input_level_deviation)
        {
            ETG_TRACE_USR4(("[audproc_service_do_set_param]: aplArkamysInputLevelDeviation value unchanged -> %d, return wi/o call to the arkamysy library ", audproc_last_input_level_deviation));
            return AUDPROC_ERR_OK;
        }
        else
            audproc_last_input_level_deviation = p32;
    }
#endif
    else
    {
        /* do nothing */
    }


    switch (data_type)
    {
        case AUDPROC_TYPE_I16:
        {
            gint16 p16;
            size = 2;

            p16 = (gint16)g_value_get_int(param);

            ETG_TRACE_USR4(("[lvar1]: property_value -> %d", p16));

            err =  audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, size, (void*)&p16);
            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p16);
            }
            else
            {
                pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }
        case AUDPROC_TYPE_U16:
        {
            guint16 p16;
            size = 2;

            p16 = (guint16)g_value_get_uint(param);

            ETG_TRACE_USR4(("[lvar1]: property_value -> %d", p16));

            err =  audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id,(guint)p->apl_channel, size, (void*)&p16);
            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p16);
            }
            else
            {
                pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }
        case AUDPROC_TYPE_U32:
        {
            guint32 p32;
            size = 4;

            p32 = g_value_get_uint(param);
            ETG_TRACE_USR4(("[lvar1]: property_value -> %d", p32));

            err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, size, (void*)&p32);
            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p32);
            }
            else
            {
                pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }
        case AUDPROC_TYPE_I32:
        {
            gint32 p32;
            size = 4;

            p32 = g_value_get_int(param);
            ETG_TRACE_USR4(("[lvar1]: property_value -> %d", p32));

            if((tAplDataID)param_id == aplAudioMode)
            {
                err = audproc_service_select_audio_mode((guint)p32);
            }
            else
            {
                err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, size, (void*)&p32);
                if (!err)
                {
                    pr_debug("%s = %d", p->apl_name, p32);
                }
                else
                {
                    pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
                }
            }
            break;
        }
        case AUDPROC_TYPE_ARRAY_U16:
        {
            ga = g_value_get_boxed(param);

            if (ga->len == 0)
            {
                pr_warning("Zero-length array for %s", p->apl_name);
                break;
            }

            size = 2;
            len = size * ga->len;
            guint i;

            a = g_slice_alloc(len);
            guint16 *to16 = (guint16*)a;
            guint *from32 = &g_array_index(ga, guint, 0);

            for (i = 0; i < ga->len; i++)
                    *to16++ = (guint16)*from32++; /*TODO: Gen4 LP64 */

            err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, len, a);
            g_slice_free1(size * ga->len, a);

            if (!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
            }
            else
            {
                pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }
        case AUDPROC_TYPE_ARRAY_I16:
        {
            ga = g_value_get_boxed(param);

            if (ga->len == 0)
            {
                pr_warning("Zero-length array for %s", p->apl_name);
                break;
            }
            size = 2;
            len = size * ga->len;
            guint i;

            a = g_slice_alloc(len);
            gint16 *to16 = (gint16*)a;
            gint   *from32 = &g_array_index(ga, gint, 0);

            for (i = 0; i < ga->len; i++)
                    *to16++ = (gint16)*from32++; /*TODO: Gen4 LP64 */

            err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, len, a);
            g_slice_free1(size * ga->len, a);

            if (!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
            }
            else
            {
                pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }
        case AUDPROC_TYPE_ARRAY_U32:
        case AUDPROC_TYPE_ARRAY_I32:
        {
            ga = g_value_get_boxed(param);

            if (ga->len == 0)
            {
                pr_warning("Zero-length array for %s", p->apl_name);
                break;
            }
            size = 4;
            len = size * ga->len;
            err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id,(guint)p->apl_channel, len, (void*)ga->data);

            if (!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
            }
            else
            {
             pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }
        case AUDPROC_TYPE_ARRAY_U8:
        case AUDPROC_TYPE_ARRAY_I8:
        {
            ga = g_value_get_boxed(param);

            if (ga->len == 0)
            {
                pr_warning("Zero-length array for %s", p->apl_name);
                break;
            }

            err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id,(guint)p->apl_channel, ga->len, ga->data);

            if (!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
            }
            else
            {
                pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
            }
            break;
        }

        default:
            break;

    }

    audproc_service_wait_apl_notification_stop();
    ETG_TRACE_USR3(("[audproc_service_do_set_param]: EXIT"));

    (void)AudprocSourceId;

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_do_get_param
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_get_param
(
    guchar              AudprocSourceId,
    guint32             param_id,
    audproc_type        data_type,
    GValue*             param
)
{
    int                         err = (int)AUDPROC_ERR_OK;
    unsigned int                size, len;
    const struct aplParameter*  p;
    GArray*                     ga;
    gpointer                    a;
    AplState*                   aplinst =  (AplState*)NULL;


    pr_debug("ENTERED: paramId(%d), datatype(%d)\n", param_id, data_type);


    /* parameter infos*/
    p = audproc_configuration_get_param_infos(param_id);

    if(!p)
    {
        pr_debug("parameter Id(%d) not available\n", param_id);
        return AUDPROC_ERR_PARAM;
    }

    ETG_TRACE_USR3(("[audproc_service_do_get_param}: ENTERED with paramters"));
    ETG_TRACE_USR3(("-> property_id    : %d", param_id));
    ETG_TRACE_USR3(("-> property_name  : %s", p->apl_name ? p->apl_name :"UNDEFINED"));


    /* this is not an apl specifical parameter */
    if(param_id != aplAudioMode)
    {

        aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

        if(!aplinst)
        {
            pr_debug("apl instance(%s) is not available, shall be created\n", p->apl_instance_name);

            /* create apl instance for Arkamys use case */
            err = (int)audproc_configuration_set_configuration_by_apl_inst(p->apl_instance_name);

            /* now retrieve apl instance */
            if(!err)
                aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

            if(!aplinst && (param_id != aplArkamysLibraryVersion))
            {
                pr_warning("try to apl arkamys instance fails \n");
                return AUDPROC_ERR_PARAM;
            }
        }
    }


    /* initialize async call */
    /* if streaming aud audio processinf are not cuurent active than send level loss value from here*/
    struct audproc_instance_object*     act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
    if(    act_obj
        && act_obj->audproc_is_streaming_running_state) //audproc_is_streaming_started)
    {
        audproc_service_wait_apl_notification_start();
    }

    switch (data_type)
    {
        case AUDPROC_TYPE_I16:
        {
            gint p16;
            size = 2;

            err =  audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id,(guint)p->apl_channel, &size, (void*)&p16);
            if(!err && !size)
            {
                err = audproc_service_wait_apl_notification();
                size = 2;
                if(!err)
                    err =  audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &size, (void*)&p16);
            }

            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p16);
            }
            else
            {
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);
            }

            ETG_TRACE_USR4(("-> property_value  : %d", p16));
            g_value_set_int(param, p16);

            break;
        }
        case AUDPROC_TYPE_U16:
        {
            guint p16;
            size = 2;

            err =  audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id,(guint)p->apl_channel, &size, (void*)&p16);
            if(!err && !size)
            {
                err = audproc_service_wait_apl_notification();
                size = 2;
                if(!err)
                    err =  audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id,(guint)p->apl_channel, &size, (void*)&p16);
            }

            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p16);
            }
            else
            {
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);
            }
            ETG_TRACE_USR4(("-> property_value  : %d", p16));
            g_value_set_uint(param, p16);

            break;
        }
        case AUDPROC_TYPE_U32:
        {
            guint32 p32;
            size = 4;

            err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &size, (void*)&p32);
            if(!err && !size)
            {
                err = audproc_service_wait_apl_notification();
                size = 4;
                if(!err)
                    err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &size, (void*)&p32);
            }

            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p32);
            }
            else
            {
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);
            }
            ETG_TRACE_USR4(("-> property_value  : %d", p32));
            g_value_set_uint(param, p32);

            break;
        }
        case AUDPROC_TYPE_I32:
        {
            gint32 p32;
            size = 4;

            err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &size, (void*)&p32);
            if(!err && !size)
            {
                err = audproc_service_wait_apl_notification();
                size = 4;
                if(!err)
                    err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &size, (void*)&p32);
            }
            if (!err)
            {
                pr_debug("%s = %d", p->apl_name, p32);
            }
            else
            {
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);
            }
            ETG_TRACE_USR4(("-> property_value  : %d", p32));
            g_value_set_int(param, p32);

            break;
        }
        case AUDPROC_TYPE_ARRAY_U8:

        case AUDPROC_TYPE_ARRAY_I8:
        {
            size = 1;
            len = size * (guint)p->apl_num;

            a = g_slice_alloc0(len);
            ga = g_array_new(FALSE, FALSE, size);

            /* get parameter value */
            err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &len, a);
            if(!err && !len)
            {
                err = audproc_service_wait_apl_notification();
                len = size * (guint)p->apl_num;
                if(!err)
                    err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &len, a);
            }

            if(!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
                g_array_append_vals(ga, a, (guint)p->apl_num);
            }
            else
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);

            g_slice_free1((gsize)(size * (guint)p->apl_num), a);
            g_value_set_boxed(param, ga);
            g_array_free(ga, TRUE);

            break;
        }
        case AUDPROC_TYPE_ARRAY_U16:
        case AUDPROC_TYPE_ARRAY_I16:
        {
            size = 2;
            len = size * (guint)p->apl_num;

            a = g_slice_alloc0(len);
            ga = g_array_new(FALSE, FALSE, size);

            /* get parameter value */
            err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &len, a);
            if(!err && !len)
            {
                err = audproc_service_wait_apl_notification();
                len = size * (guint)p->apl_num;
                if(!err)
                    err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &len, a);
            }
            if(!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
                g_array_append_vals(ga, a,(guint)p->apl_num);
            }
            else
            {
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);
            }
            g_slice_free1((gsize)(size * (guint)p->apl_num), a);
            g_value_set_boxed(param, ga);
            g_array_free(ga, TRUE);

            break;
        }
        case AUDPROC_TYPE_ARRAY_U32:
        case AUDPROC_TYPE_ARRAY_I32:
        {
            size = 4;
            len = size * (guint)p->apl_num;

            a = g_slice_alloc0(len);
            ga = g_array_new(FALSE, FALSE, size);

            /* get parameter value */
            if(!aplinst && (param_id == aplArkamysLibraryVersion))
                err = audproc_apl_handler_get_ark_lib_ver(a);
            else
            {
                err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &len, a);
                if(!err && !len)
                {
                    err = audproc_service_wait_apl_notification();
                    len = size * (guint)p->apl_num;
                    if(!err)
                        err = audproc_apl_handler_get_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, &len, a);
                }
            }

            if(!err)
            {
                pr_debug("%s len %d", p->apl_name, ga->len);
                g_array_append_vals(ga, a,(guint)p->apl_num);
            }
            else
            {
                pr_warning("audproc_apl_handler_get_data error(%d) for %s", err, p->apl_name);
            }
            g_slice_free1((gsize)(size * (guint)p->apl_num), a);
            g_value_set_boxed(param, ga);
            g_array_free(ga, TRUE);

            break;
        }
        default:
            break;

    }

    audproc_service_wait_apl_notification_stop();
    ETG_TRACE_USR3(("[audproc_service_do_get_param]: EXIT"));

    (void)AudprocSourceId;

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_delete_all_apl_request
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
void  audproc_service_delete_all_apl_request(void)
{
   // int err                                   = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
    {
        pr_debug("service not available \n");
        ETG_TRACE_USR3(("[audproc_service_delete_all_apl_request]: service not available"));
        return;
    }

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_service_delete_all_apl_request]: ENTER"));

    (void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdDeleteAllRequest);

    pr_debug("EXIT\n");
    ETG_TRACE_USR3(("[audproc_service_delete_all_apl_request]: EXIT"));
    return;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_do_read_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_read_data
(
    guchar AudprocSourceId,
    guint32 param_id,
    GArray *ReadInfos,
    GArray **data
)
{
    int                         err = AUDPROC_ERR_OK;
    const struct aplParameter*  p;
    AplState*                   aplinst =  (AplState*)NULL;


    pr_message("ENTERED");


    if(!ReadInfos)
    {
        pr_warning("array is NULL\n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    /* parameter infos*/
    p = audproc_configuration_get_param_infos(aplArkamysReadData);

    if(!p)
    {
        pr_debug("parameter Id(%d) not available\n", param_id);
        return AUDPROC_ERR_PARAM;
    }

    ETG_TRACE_USR3(("[audproc_service_do_read_data}: ENTERED with paramters"));
    ETG_TRACE_USR3(("-> property_id    : %d", param_id));
    ETG_TRACE_USR3(("-> property_name  : %s", p->apl_name ? p->apl_name :"UNDEFINED"));

    aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

    if(!aplinst)
    {
        pr_debug("apl instance(%s) is not available, shall be created\n", p->apl_instance_name);

        /* create apl instance for Arkamys use case */
        err = audproc_configuration_set_configuration_by_cfg_id(AUDPROC_CFG_ARKAMYS_PROCESSING);

        /* now retrieve apl instance */
        aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

        if(!aplinst)
        {
            pr_warning("try to apl arkamys instance fails \n");
            return AUDPROC_ERR_PARAM;
        }
    }


    if (ReadInfos->len == 0)
    {
        pr_warning("Zero-length array for %s", p->apl_name);
        ETG_TRACE_ERR(("[audproc_service_do_read_data}: Zero-length array for %s", p->apl_name));
        return AUDPROC_ERR_PARAM;
    }

    /* initialize async call
     * fix: 28.03.18: enable the notification waiting here in order to cover the case where the apl notification
     * returns previous to the call of the function audproc_service_wait_apl_notification()
     */
    /* if streaming aud audio processinf are not cuurent active than send level loss value from here*/
    struct audproc_instance_object*     act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
    if(    act_obj
        && act_obj->audproc_is_streaming_running_state //audproc_is_streaming_started)
        &&!act_obj->audproc_audio_instance->isReadWriteErr )
    {
            audproc_service_wait_apl_notification_start();
    }
    else
    {
        ETG_TRACE_USR3(("[audproc_service_do_read_data}: synchrone call to apl"));
        err = AUDPROC_ERR_OK;
    }

    err = audproc_apl_handler_set_data(aplinst, (tAplDataID)aplArkamysReadData, (guint)p->apl_channel, ReadInfos->len, (void*)ReadInfos->data);

    if (!err)
    {
        pr_debug("set parameter %s with ReadInfos data length %d", p->apl_name, ReadInfos->len);
    }
    else
    {
        pr_warning("audproc_apl_handler_set_data error(%d) for %s", err, p->apl_name);
        ETG_TRACE_ERR(("[audproc_service_do_read_data}: audproc_apl_handler_set_data error(%d)", err));
    }


    err = audproc_service_wait_apl_notification();

    /* read parameter update */
    if(!err)
    {
        unsigned int    size = 0;
        GArray*         ga;

        /*******************************************************************************************
         *  - The read data method returns a byte stream which is based
         *  - either of an Error code only or an Status/Ctrl Code with a data field
         *
         *  Data container contains read a error code as integer, the datalength as integer
         *  and a byte array for the data field
         *  - Error code  (int)        :B0..B3
         *  - Data length (int)        :B4..B7
         *  - Data field  (Byte array) :B8..Bn
         *  -----------------------------------------------------------
         *  |    4 Bytes        |  4 bytes    | N Bytes               |
         *  |---------------------------------------------------------|
         *  | error/status/ctrl | data lenght | data_B0 ----- data_Bn |
         *  |----------------------------------------------------------
         *
         *  1/ in case of error : only B0..B3 are returned
         *  2/ in the case the read is sucessfull, an error code set to "0" or a "CtrL/Status", the
         *     data length is set as well as the data field bytes are filled with valid values
         *********************************************************************************************/

        memset((void*)GetParamBuf, 0, AUDPROC_SERVICE_MAX_READ_BUFFER_SIZE); /* I would prefer a dynamic allocation with g_malloc0 instaed of a static buffer */
        ga = g_array_new(TRUE, TRUE, sizeof(guint8));

        err = audproc_apl_handler_get_data(aplinst, aplArkamysReadData, 0, &size, (void*)&GetParamBuf);
        if(!err && size)
        {
            pr_message("append %d element in read return buffer", size + 8);
            ETG_TRACE_USR3(("[audproc_service_do_read_data}: append %d element in read return buffer", size + 8));
            g_array_append_vals(ga, &err, 4);
            g_array_append_vals(ga, &size, 4);
            g_array_append_vals(ga, &GetParamBuf, size);
        }
        else
        {
            if(!size)
            {
                unsigned int err_size = 4;
                int ark_err = 0;

                /* retrieve last arkamys error */
                err = audproc_apl_handler_get_data(aplinst, aplArkamysGetLastError, 0, &err_size, (void*)&ark_err);
                if(!err)
                {
                    pr_warning("aplArkamysReadData fails, arkamys library error : %d\n", ark_err);
                    ETG_TRACE_ERR(("[audproc_service_do_read_data}: aplArkamysReadData fails, arkamys library error: %d", ark_err));
                    err = AUDPROC_ERR_OK;

                    /* return error code to FI */
                    pr_message("append error code: %d in read return buffer", ark_err);
                    g_array_append_vals(ga, &ark_err, 4);

                    /* reset apl last error */
                    (void)audproc_apl_handler_set_data(aplinst, aplArkamysResetLastError, 0, err_size, (void*)&ark_err);
                }
                else
                {
                    pr_warning("apl fails with audproc_apl_handler_get_data,  err(%d) \n", err);
                }
            }
        }

        /* return Garray to caller */
        *data = ga;
    }
    else
    {
        /* here delete the apl pending request */
        audproc_service_delete_all_apl_request();
    }

    audproc_service_wait_apl_notification_stop();

    ETG_TRACE_USR3(("[audproc_service_do_read_data}: EXIT"));
    pr_message("EXIT\n");

    (void)AudprocSourceId;

    return err;
}



/*******************************************************************************
*
* FUNCTION: audproc_service_do_write_data
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_write_data
(
    guchar  AudprocSourceId,
    guint32 param_id,
    GArray  *data,
    GArray  **result
)
{
    int                         err = AUDPROC_ERR_OK;
    const struct aplParameter*  p;
    AplState*                   aplinst =  (AplState*)NULL;
    gboolean                    bsendnotification = FALSE;

    pr_message("ENTERED");


    if(!data)
    {
        pr_warning("array is NULL\n");
        return AUDPROC_ERR_NULL_POINTER;
    }

    /* parameter infos*/
    if(   (param_id == aplArkamysWriteMultiFrame)
        ||(param_id == aplArkamysSetData))
    {

        p = audproc_configuration_get_param_infos(param_id); //aplArkamysSetData);

        if(!p)
        {
            pr_warning("parameter Id(%d) not available\n", param_id);
            return AUDPROC_ERR_PARAM;
        }
    }
    else
    {
        pr_warning("parameter id -> %d not available\n", param_id);
        return AUDPROC_ERR_PARAM;
    }

    ETG_TRACE_USR3(("[audproc_service_do_write_data}: ENTERED with paramters"));
    ETG_TRACE_USR3(("-> property_id    : %d", param_id));
    ETG_TRACE_USR3(("-> property_name  : %s", p->apl_name ? p->apl_name :"UNDEFINED"));

    aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

    if(!aplinst)
    {
        pr_debug("apl instance(%s) is not available, shall be created\n", p->apl_instance_name);

        /* create apl instance for Arkamys use case */
        err = audproc_configuration_set_configuration_by_cfg_id(AUDPROC_CFG_ARKAMYS_PROCESSING);

        /* now retrieve apl instance */
        aplinst = audproc_apl_handler_get_instance(p->apl_instance_name);

        if(!aplinst)
        {
            pr_warning("try to apl arkamys instance fails \n");
            return AUDPROC_ERR_PARAM;
        }
    }

    if (data->len == 0)
    {
        pr_warning("Zero-length array for %s", p->apl_name);
        return AUDPROC_ERR_PARAM;
    }

    /* initialize async call */
    //audproc_service_wait_notification = TRUE;
    /* if streaming and audio processing are not currently active than send level loss value from here*/
    struct audproc_instance_object*     act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
    if(   !act_obj
        ||!act_obj->audproc_is_streaming_running_state //audproc_is_streaming_started)
        || act_obj->audproc_audio_instance->isReadWriteErr)
    {

        /* send level loss */
        pr_message("send level loss notification here\n");
        bsendnotification = TRUE;
    }
    else
    {
        (void)audproc_service_wait_apl_notification_start();
    }

    //err = audproc_apl_handler_set_data(aplinst, (tAplDataID)aplArkamysSetData, p->apl_channel, data->len, (void*)data->data);
    err = audproc_apl_handler_set_data(aplinst, (tAplDataID)param_id, (guint)p->apl_channel, data->len, (void*)data->data);
    if(err)
        pr_warning("audproc_apl_handler_set_data fails with err: %d \n", err);

    if(bsendnotification)
        (void)audproc_service_send_notification(aplArkamysLevelLoss, APL_ACT_GET_PARAM);
    (void)audproc_service_wait_apl_notification();

    /* check whether a failure due to the alp instance is pending */
    int ark_err  = 0;
    unsigned int err_size = 4;

    /* retrieve last arkamys error */
    err = audproc_apl_handler_get_data(aplinst, aplArkamysGetLastError, 0, &err_size, (void*)&ark_err);
    if(!err)
    {

        GArray* ga;
        ga = g_array_new(TRUE, TRUE, sizeof(guint8));

        err = AUDPROC_ERR_OK; //ark_err;

        /* return error code to FI */
        pr_message("append error code: %d in read return buffer", ark_err);
        g_array_append_vals(ga, &ark_err, 4);

        *result = ga;

        /* reset apl last error */
        (void)audproc_apl_handler_set_data(aplinst, aplArkamysResetLastError, 0, err_size, (void*)&ark_err);
    }
    else
    {
        pr_warning("apl fails with audproc_apl_handler_get_data,  err(%d) \n", err);
    }
    (void)AudprocSourceId;
    return err;
}


#define D_DESTROY_ALSA_INST

/*******************************************************************************
*
* FUNCTION: audproc_service_do_destroy
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_do_destroy(guchar AudprocSourceId)
{
    int err = AUDPROC_ERR_OK;
    struct audproc_instance_object*     req_obj;
    struct audproc_instance_object*     act_obj;
    audproc_alsa_stream_cfg*            pd;

    pr_message("ENTERED\n");

    ETG_TRACE_USR3(("[audproc_service_do_destroy}: ENTERED with paramter"));
    ETG_TRACE_USR3(("[in_param1]: AudprocSourceId -> %d", AudprocSourceId));


    req_obj = audproc_service_get_instance(AudprocSourceId);
    if(!req_obj)
        return AUDPROC_ERR_SERV_NOT_INSTANCED;

    act_obj = audproc_service_get_active_type_src(req_obj->audproc_this_src_type);

    /*********************
     * stop audio streaming
     */
    if(act_obj)
    {
        err = audproc_service_do_stop_streaming(AudprocSourceId);
    }
    else
    {
        pr_debug("no active source associated with this source type(%d) was found\n",req_obj->audproc_this_src_type);
        act_obj = req_obj;
    }

    /************************
     * destroy apl instance
     */
    pd = act_obj->audproc_audio_instance->audproc_stream_data;

    if(pd)
    {
        (void)audproc_apl_handler_destroy(pd->audcfg.audproc_apl_inst_name);

#ifdef D_DESTROY_ALSA_INST
        if(pd->audproc_close_on_stop)
        {
            //err = audproc_alsa_finalize_intance(act_obj->audproc_audio_instance);
            //(void)audproc_service_remove_instance(AudprocSourceId);
        }
#endif
    }


    /* release resources */
    //(void)audproc_configuration_finalize();

    act_obj->audproc_is_initialized = FALSE;
    act_obj->audproc_is_streaming_started = FALSE;
    act_obj->audproc_is_this_src_active = FALSE;
    act_obj->audproc_is_apl_instance_created = FALSE;
    act_obj->audproc_astate = AUDPROC_AS_STREAM_SILENCE;

    /* notify client source state change */
    //audproc_object_emit_signal_sources_state();

    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_do_create
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_service_do_create(void)
{
    int                 err     = AUDPROC_ERR_OK;
    AplState*           aplInst = (AplState*)NULL;


    pr_message("ENTERED\n");

    /* default default apl instance tha support all functions */
    aplInst = audproc_apl_handler_create_instance(APL_INSTANCE_STR_ALL_FCTS, TRUE);

    if(!aplInst)
        err = AUDPROC_ERR_APL_CFG_ERROR;

    return err;

}

/*******************************************************************************
*
* FUNCTION: audproc_service_do_mute
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_service_do_mute(void)
{
    int err = AUDPROC_ERR_OK;

    return err;

}


static void audproc_service_print_all_debug_options(void)
{

    pr_message("************* Debug options ****************\n");
    ETG_TRACE_USR1(("************* Debug options ****************"));
    pr_message(" 1  ---> print the current state  of all audioprocess sources \n");
    ETG_TRACE_USR1(("1  ---> print the current state  of all audioprocess sources"));
    pr_message(" 2  ---> print the audio statistic of all source type \n");
    ETG_TRACE_USR1(("2  ---> print the audio statistic of all source type"));
    pr_message(" 3  ---> print EOL data and infos on the current active audio mode \n");
    ETG_TRACE_USR1(("3  ---> print EOL data and infos on the current active audio mode"));
    pr_message(" 4  ---> print configuration infos of all sources/routes \n");
    ETG_TRACE_USR1(("4  ---> print configuration infos of all sources/routes"));
    pr_message(" 5  ---> print configuration infos of all available devices used by the audioporcess daemon \n");
    ETG_TRACE_USR1(("5  ---> print configuration infos of all available devices used by the audioporcess daemon"));
    pr_message(" 6  ---> print configuration infos of the translation table audproc_src <-> midw_src_id \n");
    ETG_TRACE_USR1(("6  ---> print configuration infos of the translation table audproc_src <-> midw_src_id"));
    pr_message(" 7  ---> restart audio recording if activated per environment variable AUDPROC_REC_ENT(MIC)_STREAM_TO_FILE \n");
    ETG_TRACE_USR1(("7  ---> restart audio recording if activated per environment variable AUDPROC_REC_ENT(MIC)_STREAM_TO_FILE"));
    pr_message(" 8  ---> stop audio recording if activated per environment variable AUDPROC_REC_ENT(MIC)_STREAM_TO_FILE \n");
    ETG_TRACE_USR1(("8  ---> stop audio recording if activated per environment variable AUDPROC_REC_ENT(MIC)_STREAM_TO_FILE"));
    pr_message(" 10 ---> activate audio recording with environment variable AUDPROC_REC_ENT(MIC)_STREAM_TO_FILE \n");
    ETG_TRACE_USR1(("10 ---> activate audio recording with environment variable AUDPROC_REC_ENT(MIC)_STREAM_TO_FILE"));
    pr_message(" 11 ---> activate debug log level  \n");
    ETG_TRACE_USR1(("11 ---> activate debug log level"));
    pr_message(" 12 ---> deactivate debug log level \n");
    ETG_TRACE_USR1(("12 ---> deactivate debug log level"));
    pr_message(" 13 ---> activate apl message log level  \n");
    ETG_TRACE_USR1(("13 ---> activate apl message log level"));
    pr_message(" 14 ---> activate apl debug log level  \n");
    ETG_TRACE_USR1(("14 ---> activate apl debug log level"));
    pr_message(" 22 ---> activate apl debug data log level  \n");
    ETG_TRACE_USR1(("22 ---> activate apl debug data log level"));
    pr_message(" 23 ---> activate apl journal log \n");
    ETG_TRACE_USR1(("23 ---> activate apl journal log"));
    pr_message(" 24 ---> deactivate apl journal log \n");
    ETG_TRACE_USR1(("24 ---> deactivate apl journal log"));
    pr_message(" 15 ---> deactivate apl log  \n");
    ETG_TRACE_USR1(("15 ---> deactivate apl log"));
    pr_message(" 16 ---> activate arkamys call logger  \n");
    ETG_TRACE_USR1(("16 ---> activate arkamys call logger"));
    pr_message(" 17 ---> deactivate arkamys call logger \n");
    ETG_TRACE_USR1(("17 ---> deactivate arkamys call logger"));
    pr_message(" 18 ---> activate performance test \n");
    ETG_TRACE_USR1(("18 ---> activate performance test"));
    pr_message(" 19 ---> deactivate performance test \n");
    ETG_TRACE_USR1(("19 ---> deactivate performance test"));
    pr_message(" 20 ---> activate performance apl \n");
    ETG_TRACE_USR1(("20 ---> activate performance apl"));
    pr_message(" 21 ---> deactivate performance apl \n");
    ETG_TRACE_USR1(("21 ---> deactivate performance apl"));
    return;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_do_set_debug_mode
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
int audproc_service_do_set_debug_mode(enum eAudprocDebugMode mode)
{
    int err = AUDPROC_ERR_OK;
    struct audproc_instance_object* act_obj = (struct audproc_instance_object*)NULL;

    pr_message("ENTERED with debug mode(%d)\n", mode);


    switch (mode)
    {
        case AUDPROC_DBUG_PRINT_ALL_OPTIONS:
            audproc_service_print_all_debug_options();
            break;
        case AUDPROC_DBUG_PRINT_CURRENT_SERV_STATE:
            audproc_service_print_current_state();
            break;
        case AUDPROC_DBUG_PRINT_AUDIO_STATISTIC:
        {
            act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_ENT);
            if(act_obj)
                audproc_alsa_print_current_audio_state(act_obj->audproc_audio_instance);
            else
            {
                pr_message("no entertainemnt source active\n");
                ETG_TRACE_USR1(("no entertainemnt source active"));
            }

            act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_VOICE);
            if(act_obj)
                audproc_alsa_print_current_audio_state(act_obj->audproc_audio_instance);
            else
            {
                pr_message("no voice source active\n");
                ETG_TRACE_USR1(("no voice source active"));
            }

            act_obj = audproc_service_get_active_type_src(AUD_PROC_SRC_TYPE_INFO);
            if(act_obj)
                audproc_alsa_print_current_audio_state(act_obj->audproc_audio_instance);
            else
            {
                pr_message("no info source active\n");
                ETG_TRACE_USR1(("no info source active"));
            }

            break;
        }
        case AUDPROC_DBUG_PRINT_AUDIO_PROC_EOL_CONFIG:
        {
            audproc_datapool_access_print_audio_mode_config();
            break;
        }
        case AUDPROC_DBUG_PRINT_AUDIO_ROUTE_CONFIG:
        {
            audproc_service_print_src_instance_state();
            break;
        }
        case AUDPROC_DBUG_PRINT_AUDIO_DEVICE_CONFIG:
        {
            audproc_audio_rts_handler_print_audio_route_config();
            break;
        }
        case AUDPROC_DBUG_PRINT_AUDIO_SRC_CONFIG:
        {
            audproc_configuration_print_configuration();
            break;
        }
        case AUDPROC_DBUG_STOP_RTS_RECORDING:
        {
            audproc_audio_recording_stop();
            break;
        }
        case AUDPROC_DBUG_RESTART_RTS_RECORDING:
        {
            audproc_audio_recording_restart();
            break;
        }

        case AUDPROC_DBUG_SET_MIC_LVL_300_MS:
            audproc_configuration_test_miclvl_300ms();
            break;
        case AUDPROC_DBUG_SET_ACTIVATE_AUDIO_RECORD:
            err = setenv("AUDPROC_REC_MIC_STREAM_TO_FILE"/*const char *name*/,
                        "3"/*const char *value*/,
                        1/*int overwrite*/);
            err = setenv("AUDPROC_REC_ENT_STREAM_TO_FILE"/*const char *name*/,
                        "3"/*const char *value*/,
                        1/*int overwrite*/);
            err = setenv("AUDPROC_REC_INFO_STREAM_TO_FILE"/*const char *name*/,
                        "3"/*const char *value*/,
                        1/*int overwrite*/);
            audproc_audio_recording_init();

            break;
        case AUDPROC_DBUG_SET_ACTIVATE_DEBUG_LOG_LEVEL:
            err = setenv("G_MESSAGES_DEBUG"/*const char *name*/,
                         "all"/*const char *value*/,
                         1/*int overwrite*/);
            break;
        case AUDPROC_DBUG_SET_DEACTIVATE_DEBUG_LOG_LEVEL:
            err = unsetenv("G_MESSAGES_DEBUG"/*const char *name*/);
            break;
        case AUDPROC_DBUG_SET_ACTIVATE_APL_MSG_LOG_LEVEL:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_log_level, 1);
            break;
        case AUDPROC_DBUG_SET_ACTIVATE_APL_DEBUG_LOG_LEVEL:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_log_level, 2);
            break;
        case AUDPROC_DBUG_SET_ACTIVATE_APL_DEBUG_DATA_LOG_LEVEL:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_log_level, 3);
            break;
        case AUDPROC_DBUG_SET_DEACTIVATE_APL_LOG:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_log_level, 0);
            break;
        case AUDPROC_DBUG_SET_ACTIVATE_APL_ARK_LOGGER:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_Ark_log, 1);
            break;
        case AUDPROC_DBUG_SET_DEACTIVATE_APL_ARK_LOGGER:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_Ark_log, 0);
            break;
        case AUDPROC_DBUG_SET_ACTIVATE_APL_JOURNAL_LOG:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_log_journal_enable, 1);
            break;
        case AUDPROC_DBUG_SET_DEACTIVATE_APL_JOURNAL_LOG:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_log_journal_enable, 0);
            break;
#ifdef D_SUPPORT_AUDROC_SERVER
        case AUDPROC_DBUG_SET_START_TEST_CASE_1:
            audproc_server_control_test_case_1_start();
            break;
        case AUDPROC_DBUG_SET_STOP_TEST_CASE_1:
            audproc_server_control_test_case_1_end();
            break;
#endif
        case AUDPROC_DBUG_ACTIVATE_PERFORMANCE_TEST:
            err = setenv("AUDPROC_RTS_PERFORMANCE_AVAILABLE"/*const char *name*/,
                        "3"/*const char *value*/,
                        1/*int overwrite*/);
            audproc_audio_rts_handler_performance_init();

            break;
        case AUDPROC_DBUG_DEACTIVATE_PERFORMANCE_TEST:
            err = unsetenv("AUDPROC_RTS_PERFORMANCE_AVAILABLE"/*const char *name*/);
            audproc_audio_rts_handler_performance_init();

            break;
        case AUDPROC_DBUG_ACTIVATE_APL_PERFORMANCE:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_performance, 1);
            break;
        case AUDPROC_DBUG_DEACTIVATE_APL_PERFORMANCE:
            audproc_apl_handler_analyze_setup(aplAnalyzeFeat_performance, 0);
            break;
        default:
            pr_debug("not supported debug mode\n");
            break;
    }


    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_loopback_activate
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE: sse error
*
*******************************************************************************/
void  audproc_service_loopback_activate(gboolean LoopbackAct)
{

  pr_debug("set activation state of the loopback route : %d \n", LoopbackAct);
  audproc_service_select_loopback_route  = LoopbackAct;
  return;
}



/******************************************************
 * state machine voice sources static functions
 */
static int  audproc_service_sm_voice_stream_terminate_audio_thread(guint8 state, gint* args, gint nargs)
{
    int                                 err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*     srv_inst = (struct audproc_instance_object*)NULL;
    tAplI32                             aplswitch = aplSwitchON;
    AplState*                           aplinst    = (AplState*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_VOICE);

    if(!srv_inst)
    {
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;
    }

    if(!srv_inst->audproc_audio_instance)
    {
       return AUDPROC_ERR_SERV_NOT_INITIALIZED;
    }

    if(!srv_inst->audproc_audio_instance->audproc_stream_data)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");


    /* end audio thread */
    audproc_alsa_adev_cfg* cap = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->indev;
    if(cap->audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
    {
        (void)audproc_audio_rts_handler_close((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               cap->audproc_dev_name,
                                               D_CONV_CAPTURE_DEVICE);
    }
    else if(cap->audproc_create_on_start == AUD_PROC_DEVICE_RECREATE_AT_STR_END_ONLY)
    {
        (void)audproc_audio_rts_handler_recreate((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               FALSE,
                                               D_CONV_CAPTURE_DEVICE);
    }

    audproc_alsa_adev_cfg* pb = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->outdev;
    if(pb->audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
    {
        (void)audproc_audio_rts_handler_close((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               pb->audproc_dev_name,
                                               D_CONV_PLAYBACK_DEVICE);
    }
    else if(pb->audproc_create_on_start == AUD_PROC_DEVICE_RECREATE_AT_STR_END_ONLY)
    {
        (void)audproc_audio_rts_handler_recreate((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               FALSE,
                                               D_CONV_PLAYBACK_DEVICE);
    }

    audproc_audio_rts_handler_performance_test_terminate(srv_inst->audproc_this_src, AUD_PROC_SRC_TYPE_VOICE, (char*)cap->audproc_dev_name, (char*)pb->audproc_dev_name);


    aplswitch = aplSwitchOFF;
    const gchar* aplStr     = srv_inst->audproc_audio_instance->audproc_stream_data->audcfg.audproc_apl_inst_name;

    /* retrieve apl instance name */
    aplinst = audproc_apl_handler_get_instance(aplStr);
    (void)audproc_apl_handler_set_data(aplinst, aplMicLvlWatchSwitch, 0, sizeof(tAplI32),(void*) &aplswitch );


    /* end audio thread */
    audproc_alsa_set_audio_thread_state(FALSE, srv_inst->audproc_audio_instance);
    audproc_audio_recording_export(AUD_PROC_SRC_TYPE_VOICE, SND_PCM_STREAM_CAPTURE);
    audproc_audio_recording_export(AUD_PROC_SRC_TYPE_VOICE, SND_PCM_STREAM_PLAYBACK);

    (void)state;
    (void)args;
    (void)nargs;


    return err;
}

static int  audproc_service_sm_voice_stream_initialize(guint8 state, gint* args, gint nargs)
{
    int                             err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object* srv_inst = (struct audproc_instance_object*)NULL;
    tAplU32                         REQ_MicLvlInt;
    tAplI32                         aplswitch = aplSwitchON;
    AplState*                       aplinst    = (AplState*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_VOICE);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_message("ENTERED\n");

    /* setup thread priority to real time*/
    audproc_alsa_set_scheduling_priority(srv_inst->audproc_audio_instance);
    audproc_alsa_set_audio_thread_state(TRUE, srv_inst->audproc_audio_instance);
    audproc_audio_rts_handler_voice_reset(srv_inst->audproc_audio_instance);


    const gchar* aplStr     = srv_inst->audproc_audio_instance->audproc_stream_data->audcfg.audproc_apl_inst_name;


    /* retrieve apl instance name */
    aplinst = audproc_apl_handler_get_instance(aplStr);

    guint32 size = 2;
    (void)audproc_apl_handler_set_data(aplinst, aplMicLvlWatchSwitch, 0, sizeof(tAplI32),(void*) &aplswitch );
    (void)audproc_apl_handler_get_data(aplinst, aplMicLvlWatchRefInt, 0, &size, (void*) &REQ_MicLvlInt);

    if(D_MIC_LEVEL_TIME_OUT_INTERVAL_MIN > REQ_MicLvlInt)
       REQ_MicLvlInt = D_MIC_LEVEL_TIME_OUT_INTERVAL_MIN;

    audproc_apl_handler_set_miclvl_timeout(REQ_MicLvlInt);
    audproc_audio_rts_handler_performance_test_init(50000);


    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->wrtsilence = TRUE;
        pr_message("pre_silence process is activated\n");
        ETG_TRACE_USR3(("[audproc_service_sm_voice_stream_initialize]: pre_silence process is activated"));
    }

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}



static int  audproc_service_sm_voice_stream_wait_recover(guint8 state, gint* args, gint nargs)
{
    int err                                   = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_VOICE);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");

    //(void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStopAudProc);

    srv_inst->audproc_audio_instance->isReadWriteErr = TRUE;
    srv_inst->audproc_audio_instance->recov_success_write_count = 0;

    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->wrtsilence = TRUE;
        srv_inst->audproc_audio_instance->silence_wrt_cnt_after_recover = 0;
        pr_message("pre_silence process is activated\n");
        ETG_TRACE_USR3(("[audproc_service_sm_voice_stream_wait_recover]: pre_silence process is activated"));
    }

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}

static int  audproc_service_sm_voice_stream_running(guint8 state, gint* args, gint nargs)
{
    int                              err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_VOICE);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");


    if(!nargs)
    {
         pr_debug("audio streaming running");
    }
    else
    {
        gint retry = args[0]; //
        pr_message("Streaming recovered after %d retries \n", retry);
    }

    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->wrtsilence = FALSE;
        srv_inst->audproc_audio_instance->silence_wrt_cnt_after_recover = 0;
        pr_message("pre_silence process is deactivated\n");
        ETG_TRACE_USR3(("[audproc_service_sm_voice_stream_running]: pre_silence process is deactivated"));
    }

    srv_inst->audproc_audio_instance->isReadWriteErr = FALSE;
    srv_inst->audproc_audio_instance->retry = 0;

    //(void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStartAudProc);
    (void)state;
    (void)args;
    (void)nargs;

    return err;
}

static int  audproc_service_sm_voice_stream_silence_preload(guint8 state, gint* args, gint nargs)
{
    int                              err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;
    gint retry = 0;

    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_VOICE);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");


    if(!nargs)
    {
         pr_debug("audio streaming running\n");
    }
    else
    {
        retry = args[0]; //va_arg(*args, gint);
    }

    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->silence_wrt_cnt_after_recover = 1;

        pr_message("preload with %d silence frames\n", retry);
        ETG_TRACE_USR3(("[audproc_service_sm_voice_stream_silence_preload]: preload with %d silence frames", retry));
    }


    //(void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStartAudProc);
    (void)state;


    return err;
}


/******************************************************
 * state machine info sources static functions
 */
static int  audproc_service_sm_info_terminate_audio_thread(guint8 state, gint* args, gint nargs)
{
    int err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_INFO);


    if(!srv_inst)
    {
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;
    }

    if(!srv_inst->audproc_audio_instance)
    {
       return AUDPROC_ERR_SERV_NOT_INITIALIZED;
    }

    if(!srv_inst->audproc_audio_instance->audproc_stream_data)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_message("ENTERED\n");


    /* end audio thread */
    audproc_alsa_adev_cfg* cap = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->indev;
    if(cap->audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
    {
        (void)audproc_audio_rts_handler_close((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               cap->audproc_dev_name,
                                               D_CONV_CAPTURE_DEVICE);
    }
    else if(cap->audproc_create_on_start == AUD_PROC_DEVICE_RECREATE_AT_STR_END_ONLY)
    {
        (void)audproc_audio_rts_handler_recreate((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               FALSE,
                                               D_CONV_CAPTURE_DEVICE);
    }

    audproc_alsa_adev_cfg* pb = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->outdev;
    if(pb->audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
    {
        (void)audproc_audio_rts_handler_close((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               pb->audproc_dev_name,
                                               D_CONV_PLAYBACK_DEVICE);
    }
    else if(pb->audproc_create_on_start == AUD_PROC_DEVICE_RECREATE_AT_STR_END_ONLY)
    {
        (void)audproc_audio_rts_handler_recreate((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               FALSE,
                                               D_CONV_PLAYBACK_DEVICE);
    }


    audproc_audio_rts_handler_performance_test_terminate(srv_inst->audproc_this_src, AUD_PROC_SRC_TYPE_INFO, (char*)cap->audproc_dev_name, (char*)pb->audproc_dev_name);

    audproc_alsa_set_audio_thread_state(FALSE, srv_inst->audproc_audio_instance);
    audproc_audio_recording_export(AUD_PROC_SRC_TYPE_INFO, SND_PCM_STREAM_CAPTURE);
    audproc_audio_recording_export(AUD_PROC_SRC_TYPE_INFO, SND_PCM_STREAM_PLAYBACK);

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}

static int  audproc_service_sm_info_stream_initialize(guint8 state, gint* args, gint nargs)
{
    int err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_INFO);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_message("ENTERED\n");

    /* setup thread priority to real time*/
    audproc_alsa_set_scheduling_priority(srv_inst->audproc_audio_instance);
    audproc_alsa_set_audio_thread_state(TRUE, srv_inst->audproc_audio_instance);
    (void)audproc_audio_rts_handler_info_reset(srv_inst->audproc_audio_instance);
    audproc_audio_rts_handler_performance_test_init(50000);

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}



static int  audproc_service_sm_info_stream_wait_recover(guint8 state, gint* args, gint nargs)
{
    int err                                   = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_INFO);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");

    //(void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStopAudProc);
    (void)state;
    (void)args;
    (void)nargs;

    return err;
}

static int  audproc_service_sm_info_stream_running(guint8 state, gint* args, gint nargs)
{
    int                              err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_INFO);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");


    if(!nargs)
    {
         pr_debug("audio streaming running");
    }
    else
    {
        gint retry = args[0]; //va_arg(*args, gint);
        pr_message("Streaming recovered after %d retries \n", retry);
    }


    //(void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStartAudProc);
    (void)state;

    return err;
}




/* state machine entertainement sources static functions */


static int  audproc_service_sm_ent_stream_terminate_audio_thread(guint8 state, gint* args, gint nargs)
{
    int err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_message("ENTERED\n");

#ifdef D_WRITE_SILENCE_DURING_PAUSE_TRACK_CHANGE
    /* stop silence process*/
    audproc_alsa_process_silence_terminated();
#endif
    audproc_alsa_adev_cfg* cap = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->indev;
    audproc_alsa_adev_cfg* pb = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->outdev;

    if(cap && pb)
        audproc_audio_rts_handler_performance_test_terminate(srv_inst->audproc_this_src, AUD_PROC_SRC_TYPE_ENT, (char*)cap->audproc_dev_name, (char*)pb->audproc_dev_name);
    else
        return AUDPROC_ERR_NULL_POINTER;

    //audproc_alsa_adev_cfg* cap = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->indev;
    if(cap->audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
    {
        (void)audproc_audio_rts_handler_close((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               cap->audproc_dev_name,
                                               D_CONV_CAPTURE_DEVICE);
    }
    else if(cap->audproc_create_on_start == AUD_PROC_DEVICE_RECREATE_AT_STR_END_ONLY)
    {
        (void)audproc_audio_rts_handler_recreate((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               FALSE,
                                               D_CONV_CAPTURE_DEVICE);
    }

    //audproc_alsa_adev_cfg* pb = (audproc_alsa_adev_cfg*)&srv_inst->audproc_audio_instance->audproc_stream_data->outdev;
    if(pb->audproc_create_on_start == AUD_PROC_DEVICE_OPEN_AT_STR_START_CLOSE_AT_STR_END)
    {
        (void)audproc_audio_rts_handler_close((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               pb->audproc_dev_name,
                                               D_CONV_PLAYBACK_DEVICE);
    }
    else if(pb->audproc_create_on_start == AUD_PROC_DEVICE_RECREATE_AT_STR_END_ONLY)
    {
        (void)audproc_audio_rts_handler_recreate((audproc_alsa_state*)srv_inst->audproc_audio_instance,
                                               FALSE,
                                               D_CONV_PLAYBACK_DEVICE);
    }


    //audproc_audio_rts_handler_performance_test_terminate(srv_inst->audproc_this_src, AUD_PROC_SRC_TYPE_ENT, (char*)cap->audproc_dev_name, (char*)pb->audproc_dev_name);

    /* end audio thread */
    audproc_alsa_set_audio_thread_state(FALSE, srv_inst->audproc_audio_instance);
    audproc_audio_recording_export(AUD_PROC_SRC_TYPE_ENT, SND_PCM_STREAM_CAPTURE);
    audproc_audio_recording_export(AUD_PROC_SRC_TYPE_ENT, SND_PCM_STREAM_PLAYBACK);

    /* new 21.11.2018*/
    //srv_inst->audproc_is_streaming_started = FALSE;
    srv_inst->audproc_is_streaming_running_state = FALSE;

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}

static int  audproc_service_sm_ent_stream_initialize(guint8 state, gint* args, gint nargs)
{
    int err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;


    if(!nargs)
    {
         pr_message("ENTERED\n");
         ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_initialize]: ENTERED"));
    }
    else
    {
        gint error = args[0]; //va_arg(*args, gint);
        pr_message("ENTERED with error code -> %d\n", error);
        ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_initialize]: ENTERED with error code -> %d", error));
    }


    //pr_message("ENTERED\n");

    /* setup thread priority to real time*/
    audproc_alsa_set_scheduling_priority(srv_inst->audproc_audio_instance);
    audproc_alsa_set_audio_thread_state(TRUE, srv_inst->audproc_audio_instance);
    (void)audproc_audio_rts_handler_ent_reset(srv_inst->audproc_audio_instance);
    (void)audproc_stream_buffer_desc_reinit(&srv_inst->audproc_audio_instance->PreProcess);
    audproc_audio_rts_handler_performance_test_init(50000);

    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->wrtsilence = TRUE;
        pr_message("pre_silence process is activated\n");
        ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_initialize]: pre_silence process is activated"));
    }

#ifdef D_WRITE_SILENCE_DURING_PAUSE_TRACK_CHANGE
    /* start silence process*/
    audproc_alsa_process_silence_start((gpointer)srv_inst->audproc_audio_instance);
#endif

    /* new 21.11.2018*/
    //srv_inst->audproc_is_streaming_started = FALSE;
    srv_inst->audproc_is_streaming_running_state = FALSE;

    (void)state;

    return err;
}



static int  audproc_service_sm_ent_stream_wait_recover(guint8 state, gint* args, gint nargs)
{
    int err                                   = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    if(!nargs)
    {
         pr_message("ENTERED\n");
         ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_wait_recover]: ENTERED"));
    }
    else
    {
        gint error = args[0]; //va_arg(*args, gint);
        pr_message("ENTERED with error code -> %d\n", error);
        ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_wait_recover]: ENTERED with error code -> %d", error));
    }


    //pr_debug("ENTERED\n");

    /* handle with playback pause */
    (void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStopAudProc);
    (void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdFlush);

    srv_inst->audproc_audio_instance->isReadWriteErr = TRUE;
    srv_inst->audproc_audio_instance->recov_success_write_count = 0;

    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->wrtsilence = TRUE;
        srv_inst->audproc_audio_instance->silence_wrt_cnt_after_recover = 0;    /*new*/
        pr_message("pre_silence process is activated\n");
        ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_wait_recover]: pre_silence process is activated"));
    }

    /* new : first send notification. delete all request  */
    (void)audproc_service_clear_notification(aplArkamysReadData, APL_ACT_GET_PARAM);
    audproc_service_delete_all_apl_request();

#ifdef D_WRITE_SILENCE_DURING_PAUSE_TRACK_CHANGE
    /* start silence process*/
    audproc_alsa_process_silence_start((gpointer)srv_inst->audproc_audio_instance);
#endif

    /* new 21.11.2018*/
    //srv_inst->audproc_is_streaming_started = FALSE;
    srv_inst->audproc_is_streaming_running_state = FALSE;

    (void)state;

    return err;
}

static int  audproc_service_sm_ent_stream_running(guint8 state, gint* args, gint nargs)
{
    int                              err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");


    if(!nargs)
    {
         pr_debug("audio streaming running");
    }
    else
    {
        gint retry = args[0]; //va_arg(*args, gint);
        pr_message("Streaming recovered after %d retries \n", retry);
    }


    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->wrtsilence = FALSE;
        srv_inst->audproc_audio_instance->silence_wrt_cnt_after_recover = 0;
        pr_message("pre_silence process is deactivated\n");
        ETG_TRACE_USR3(("[audproc_service_sm_ent_stream_running]: pre_silence process is deactivated"));
    }

    srv_inst->audproc_audio_instance->isReadWriteErr = FALSE;
    srv_inst->audproc_audio_instance->retry = 0;

#ifdef D_WRITE_SILENCE_DURING_PAUSE_TRACK_CHANGE
    /* stop silence process*/
    audproc_alsa_process_silence_terminated();
#endif

    /* requires wenn recover from pause*/
    (void)audproc_apl_handler_command(srv_inst->audproc_audio_instance->aplStr, aplCmdStartAudProc);

    /* new 21.11.2018*/
    //srv_inst->audproc_is_streaming_started = TRUE;
    srv_inst->audproc_is_streaming_running_state = TRUE;

    (void)state;

    return err;
}


static int  audproc_service_sm_ent_stream_silence_preload(guint8 state, gint* args, gint nargs)
{
    int                              err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;
    gint retry = 0;

    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_debug("ENTERED\n");


    if(!nargs)
    {
         pr_debug("audio streaming running\n");
    }
    else
    {
        retry = args[0]; //va_arg(*args, gint);
    }

    if(srv_inst->audproc_audio_instance->dosilence)
    {
        srv_inst->audproc_audio_instance->silence_wrt_cnt_after_recover = 1;

        pr_message("preload with %d silence frames\n", retry);
        ETG_TRACE_USR3(("[audproc_service_sm_voice_stream_silence_preload]: preload with %d silence frames", retry));
    }

    /* new 21.11.2018*/
    //srv_inst->audproc_is_streaming_started = FALSE;
    srv_inst->audproc_is_streaming_running_state = FALSE;

    (void)state;

    return err;
}


static int  audproc_service_sm_ent_stream_stop(guint8 state, gint* args, gint nargs)
{
    int err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;

    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    /* new 21.11.2018*/
    //srv_inst->audproc_is_streaming_started = FALSE;
    srv_inst->audproc_is_streaming_running_state = FALSE;


    pr_message("ENTERED\n");

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}


static int  audproc_service_sm_ent_stream_error(guint8 state, gint* args, gint nargs)
{
    int err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*  srv_inst = (struct audproc_instance_object*)NULL;


    srv_inst = audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);


    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;

    pr_message("ENTERED\n");

    (void)state;
    (void)args;
    (void)nargs;

    return err;
}
/*******************************************************************************
*
* FUNCTION: audproc_service_get_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
guint8 audproc_service_create_new_state_machine(audproc_source_type src_type, const char* sm_name)
{
    int  err = (int)AUDPROC_ERR_OK;
    guint8 sm_inst = 0;


    pr_message("ENTERED\n");



    sm_inst = audproc_state_machine_create_new(sm_name);

    if(!sm_inst)
        return 0;


    if(src_type == AUD_PROC_SRC_TYPE_ENT)
    {
        err = audproc_state_machine_config_begin(sm_inst);

        /* configure all main state  */
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_SILENCE           , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_INIT              , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_RUNNING           , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_STOP              , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_TERMINATED        , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_ERROR             , audproc_object_emit_signal_ent_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_SILENCE_PRELOAD   , audproc_object_emit_signal_ent_sources_state);

        /* configure next state*/
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE           , AUDPROC_AS_STREAM_INIT            , NULL, audproc_service_sm_ent_stream_initialize);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE           , AUDPROC_AS_STREAM_WAIT_DATA       , NULL, audproc_service_sm_ent_stream_initialize);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT              , AUDPROC_AS_STREAM_WAIT_DATA       , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT              , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_ent_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT              , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_ent_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_ent_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , AUDPROC_AS_STREAM_ERROR           , NULL, audproc_service_sm_ent_stream_error);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_ent_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_WAIT_RECOVER    , NULL, audproc_service_sm_ent_stream_wait_recover);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_ERROR           , NULL, audproc_service_sm_ent_stream_error);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_STOP            , NULL, audproc_service_sm_ent_stream_stop);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_ent_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_STOP              , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_ent_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_SILENCE_PRELOAD , NULL, audproc_service_sm_ent_stream_silence_preload);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE_PRELOAD   , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_ent_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE_PRELOAD   , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_ent_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_ent_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_ERROR           , NULL, audproc_service_sm_ent_stream_error);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_ent_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_TERMINATED        , AUDPROC_AS_STREAM_SILENCE         , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_TERMINATED        , AUDPROC_AS_STREAM_INIT            , NULL, audproc_service_sm_ent_stream_initialize);
        /* configure sub state : only for future use */
        //err = audproc_service_add_sub_state (AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_SILENCE       , AUDPROC_AS_STREAM_UNDEFINED    , NULL, NULL, NULL);

        err = audproc_state_machine_config_end(sm_inst);
    }
    else if(src_type == AUD_PROC_SRC_TYPE_INFO )
    {
        err = audproc_state_machine_config_begin(sm_inst);

        /* configure all main state  */
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_SILENCE       , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_INIT          , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA     , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_RUNNING       , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER  , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_STOP          , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_TERMINATED    , audproc_object_emit_signal_info_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_ERROR         , audproc_object_emit_signal_info_sources_state);

        /* configure next state*/
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE       , AUDPROC_AS_STREAM_INIT        , NULL, audproc_service_sm_info_stream_initialize);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE       , AUDPROC_AS_STREAM_WAIT_DATA   , NULL, audproc_service_sm_info_stream_initialize);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE       , AUDPROC_AS_STREAM_RUNNING     , audproc_service_sm_info_stream_initialize, audproc_service_sm_info_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT          , AUDPROC_AS_STREAM_WAIT_DATA   , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT          , AUDPROC_AS_STREAM_RUNNING     , NULL, audproc_service_sm_info_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT          , AUDPROC_AS_STREAM_TERMINATED  , NULL, audproc_service_sm_info_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA     , AUDPROC_AS_STREAM_RUNNING     , NULL, audproc_service_sm_info_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA     , AUDPROC_AS_STREAM_ERROR       , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA     , AUDPROC_AS_STREAM_TERMINATED  , NULL, audproc_service_sm_info_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING       , AUDPROC_AS_STREAM_WAIT_RECOVER, NULL, audproc_service_sm_info_stream_wait_recover);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING       , AUDPROC_AS_STREAM_ERROR       , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING       , AUDPROC_AS_STREAM_STOP        , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING       , AUDPROC_AS_STREAM_TERMINATED  , NULL, audproc_service_sm_info_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_STOP          , AUDPROC_AS_STREAM_TERMINATED  , NULL, audproc_service_sm_info_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER  , AUDPROC_AS_STREAM_RUNNING     , NULL, audproc_service_sm_info_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER  , AUDPROC_AS_STREAM_ERROR       , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER  , AUDPROC_AS_STREAM_TERMINATED  , NULL, audproc_service_sm_info_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_TERMINATED    , AUDPROC_AS_STREAM_SILENCE     , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_TERMINATED    , AUDPROC_AS_STREAM_INIT        , NULL, audproc_service_sm_info_stream_initialize);

        /* configure sub state : only for future use */
        //err = audproc_service_add_sub_state (AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_SILENCE       , AUDPROC_AS_STREAM_UNDEFINED    , NULL, NULL, NULL);

        err = audproc_state_machine_config_end(sm_inst);
    }
    else if(src_type == AUD_PROC_SRC_TYPE_VOICE )
    {
        err = audproc_state_machine_config_begin(sm_inst);

        /* configure all main state  */
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_SILENCE           , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_INIT              , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_RUNNING           , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_STOP              , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_TERMINATED        , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_ERROR             , audproc_object_emit_signal_voice_sources_state);
        err = audproc_state_machine_add_main_state(sm_inst, AUDPROC_AS_STREAM_SILENCE_PRELOAD   , audproc_object_emit_signal_voice_sources_state);


        /* configure next state*/
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE           , AUDPROC_AS_STREAM_INIT            , NULL, audproc_service_sm_voice_stream_initialize);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE           , AUDPROC_AS_STREAM_WAIT_DATA       , NULL, audproc_service_sm_voice_stream_initialize);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT              , AUDPROC_AS_STREAM_WAIT_DATA       , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT              , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_voice_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_INIT              , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_voice_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_voice_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , AUDPROC_AS_STREAM_ERROR           , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_DATA         , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_voice_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_WAIT_RECOVER    , NULL, audproc_service_sm_voice_stream_wait_recover);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_ERROR           , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_STOP            , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_RUNNING           , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_voice_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_STOP              , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_voice_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_SILENCE_PRELOAD , NULL, audproc_service_sm_voice_stream_silence_preload);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE_PRELOAD   , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_voice_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_SILENCE_PRELOAD   , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_voice_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_RUNNING         , NULL, audproc_service_sm_voice_stream_running);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_ERROR           , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_WAIT_RECOVER      , AUDPROC_AS_STREAM_TERMINATED      , NULL, audproc_service_sm_voice_stream_terminate_audio_thread);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_TERMINATED        , AUDPROC_AS_STREAM_SILENCE         , NULL, NULL);
        err = audproc_state_machine_add_main_state_connections(sm_inst, AUDPROC_AS_STREAM_TERMINATED        , AUDPROC_AS_STREAM_INIT            , NULL, audproc_service_sm_voice_stream_initialize);

        /* configure sub state : only for future use */
        //err = audproc_service_add_sub_state (AUD_PROC_SRC_TYPE_ENT, AUDPROC_AS_STREAM_SILENCE       , AUDPROC_AS_STREAM_UNDEFINED    , NULL, NULL, NULL);

        err = audproc_state_machine_config_end(sm_inst);
    }
    else
    {
        pr_warning("unknow source type: %d\n", src_type);
    }

    (void)err;
    return sm_inst;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_change_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
audproc_audio_state      audproc_service_get_audio_state
(
    audproc_source_type src_type
)
{
    struct audproc_instance_object*     srv_inst = (struct audproc_instance_object*)NULL;
    audproc_audio_state                 curr_state;

    srv_inst = audproc_service_get_active_type_src (src_type);
    if(!srv_inst)
        return AUDPROC_AS_STREAM_ERROR;

    g_mutex_lock(&audproc_service_lock);

    curr_state =  audproc_state_machine_get_audio_state(srv_inst->audproc_sm_inst); //audproc_astate;

    g_mutex_unlock(&audproc_service_lock);

    return curr_state;
}


/*******************************************************************************
*
* FUNCTION: audproc_service_change_audio_state
* DESCRIPTION: ..
*
*
* PARAMETER: None.
*
* RETURNVALUE:
*
*******************************************************************************/
int audproc_service_change_audio_state
(
    audproc_source_type src_type,
    audproc_audio_state astate,
    gint                nargs,
    ...
)
{
    int                                 err = (int)AUDPROC_ERR_OK;
    struct audproc_instance_object*     srv_inst = (struct audproc_instance_object*)NULL;
    audproc_audio_state                 curr_state;
    va_list                             args;


    srv_inst = audproc_service_get_active_type_src (src_type);
    if(!srv_inst)
        return AUDPROC_ERR_SERV_NOT_INITIALIZED;


    ETG_TRACE_USR4(("[audproc_service_change_audio_state]: request change state for src type -> %d within thread-id-> %d", src_type,  (unsigned int)pthread_self() )); //this_id ));

    g_mutex_lock(&audproc_service_lock);

    curr_state =  audproc_state_machine_get_audio_state(srv_inst->audproc_sm_inst); //audproc_astate;

    /* check transition condition */
    if(curr_state == astate)
    {
        pr_debug("new state request:%s is currently active, no new action will be performed \n", audproc_audio_state_str(astate) );
        //ETG_TRACE_USR3(("[audproc_service_change_audio_state]: requested state %s is currently active, no new action will be performed", audproc_audio_state_str(astate)));
        g_mutex_unlock(&audproc_service_lock);
        return (int)AUDPROC_ERR_OK;
    }

    pr_message("entry request change state %s for src %s \n", audproc_audio_state_str(astate), audproc_service_src_str(srv_inst->audproc_this_src));
    ETG_TRACE_USR3(("[audproc_service_change_audio_state]: entry request change state %s ", audproc_audio_state_str(astate)));

#ifdef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
    if(!err)
    {
        pr_message("\"%s\" request change of currently state \"%s\" to \"%s\"\n", audproc_service_src_str(srv_inst->audproc_this_src), audproc_audio_state_str(curr_state), audproc_audio_state_str(astate));
    }
#endif
    va_start (args, nargs);

    gint* args_int = (gint*)NULL;

    if(nargs)
    {
    args_int = (gint*)g_malloc0( (gsize)nargs * sizeof(gint));
      if(args_int)
        {
           for(int i = 0; i <nargs; i++)
                args_int[i] = va_arg(args, gint);
        }
    }

    err = audproc_state_machine_trigger_new_state(srv_inst->audproc_sm_inst, astate, &args_int, nargs); //&args, nargs);
    if(args_int)
        g_free(args_int);

    va_end (args);

#ifndef D_AUDPROC_SM_PROCESS_REQUEST_IN_OWN_THREAD
    if(!err)
    {
        pr_message("\"%s\" changes it state from \"%s\" to \"%s\"\n", audproc_service_src_str(srv_inst->audproc_this_src), audproc_audio_state_str(curr_state), audproc_audio_state_str(astate));
    }
#endif

    g_mutex_unlock(&audproc_service_lock);

#ifdef D_ONLY_FOR_DEBUG
    pr_message("state changed done\n");
#endif
    return err;
}

/*******************************************************************************
*
* FUNCTION: audproc_service_finalize
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_service_finalize(void)
{
    gint i;
    gint src_type;
    gint err = AUDPROC_ERR_OK;
    struct audproc_instance_object*     pd = (struct audproc_instance_object*)NULL;
    struct audproc_instance_object*     act_obj = (struct audproc_instance_object*)NULL;;


    pr_message("ENTERED");


    /*****
     * audprocess server section
     */

    /* destroy all active source types */
    for (src_type = 0; src_type < (gint)AUD_PROC_SRC_TYPE_COUNT; src_type++)
    {
        act_obj = audproc_service_get_active_type_src((audproc_source_type)src_type);

        if(act_obj)
            (void)audproc_service_do_destroy(act_obj->audproc_this_src);
    }


    /* remove all alsa related resources */
    if(my_servs)
    {
        gint nb_route_max = audproc_configuration_get_route_cfg_nb_max();

        if(nb_route_max)
        {
            for(i = 0; i < nb_route_max; i++)
            {
                pd = my_servs[i];
                if(pd)
                {
                    err = audproc_alsa_finalize_intance(pd->audproc_audio_instance);
                    if(err)
                        continue;

                    if(pd->audproc_audio_instance)
                    {
                        g_free(pd->audproc_audio_instance);
                        pd->audproc_audio_instance = NULL;
                    }

                    g_free(my_servs[i]);
                    my_servs[i] = NULL;
                }

                pr_message("next route_%d from total-> %d\n", i, nb_route_max);
            }
        }
    }

    (void)audproc_audio_rts_handler_finalize();
    audproc_datapool_access_finalize();
    (void)audproc_configuration_finalize();
    audproc_apl_handler_finalize();

    /* clean service module*/
    audproc_state_machine_finalize();

    if(my_servs)
        g_free(my_servs);
    my_servs = NULL;


    if(audproc_apl_init_data)
        g_free(audproc_apl_init_data);
    audproc_apl_init_data = NULL;

    g_mutex_clear(&audproc_service_lock);
    g_cond_clear(&audproc_service_update);

    /*new*/
    audproc_alsa_finalize();

    pr_message("EXIT");

    return TRUE;
}
