/*******************************************************************************
* FILE : audproc-object.c
*
* SW-COMPONENT :
*
* DESCRIPTION :
*
* AUTHOR : Patrick Rey
*
* COPYRIGHT : (C)
*
* HISTORY : 07.08.2015
* Initial version
*******************************************************************************/

#include <glib/gi18n.h>
#include <glib-object.h>
#include <alsa/asoundlib.h>
#include <sched.h>


#include "apl.h"
#include "audproc-common-defs.h"
#include "audproc-object.h"
#include "audproc-apl-handler.h"
#include "audproc-alsa.h"
#include "audproc-service.h"
#include "audproc-configuration.h"
#include "audproc-audio-rts-handler.h"
#include "audproc-datapool-access.h"
#include "audproc-introspection.h"
#include "audproc-state-machine.h"

/* Needed for Trace */
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define SPM_UNIT_TESTS // solve warning
#include "audproc-audioprocess-trace.h"
#include "etg_if.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_AUDPROC
#include "trcGenProj/Header/audproc-object.c.trc.h"
#endif


#define T_BUG

#ifndef T_BUG
G_DEFINE_TYPE(AudprocObject, audproc_object, G_TYPE_OBJECT)
#else
static void     audproc_object_init       (AudprocObject      *self);
static void     audproc_object_class_init (AudprocObjectClass *klass);
static gpointer audproc_object_parent_class = NULL;




struct audproc_signal
{
    const gchar *signal_name;
    AudProcSignal signal_enum;
    guint signal_id;
};

static  struct audproc_signal my_signal[E_SIGNAL_COUNT];
static AudprocObject* my_qobj;

/*******************************************************************************
*
* FUNCTION: audproc_object_class_intern_init
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void     audproc_object_class_intern_init (gpointer klass)
{
    pr_debug("ENTERED\n");

    audproc_object_parent_class = g_type_class_peek_parent (klass);
    audproc_object_class_init ((AudprocObjectClass*) klass);
}

/*******************************************************************************
*
* FUNCTION: audproc_object_get_type
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/

GType audproc_object_get_type (void)
{
    static volatile gsize g_define_type_id__volatile = 0;

    pr_debug("ENTERED\n");

    if (g_once_init_enter (&g_define_type_id__volatile))
    {
        GType g_define_type_id =
        g_type_register_static_simple (G_TYPE_OBJECT,
                                       g_intern_static_string ("AudprocObject"),
                                       sizeof (AudprocObjectClass),
                                       (GClassInitFunc) audproc_object_class_intern_init,
                                       sizeof (AudprocObject),
                                       (GInstanceInitFunc) audproc_object_init,
                                       (GTypeFlags)0);


        pr_debug("the type id is %lu",g_define_type_id);

        g_once_init_leave (&g_define_type_id__volatile, g_define_type_id);
    }

    pr_debug("EXIT \n");
    return g_define_type_id__volatile;
}

#endif




/*******************************************************************************
*
* FUNCTION: audproc_object_finalize
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_finalize(GObject * gobj)
{
    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_object_finalize]: ENTERED"));

    audproc_service_finalize();

    (G_OBJECT_CLASS(audproc_object_parent_class)->finalize) (gobj);

    my_qobj = NULL;

    ETG_TRACE_USR3(("[audproc_object_finalize]: EXIT"));
    pr_message("EXIT");
}



/*******************************************************************************
*
* FUNCTION: audproc_object_init
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_init(AudprocObject * qobj)
{
    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_init]: ENTERED"));

    /****************************************
     * initialize all module
     ****************************************/
    /**********
        -> called in audproc_object_init
        -> this functins is called by GObject framework after the new GObject was created
    */

#ifdef VARIANT_S_FTR_ENABLE_FEAT_AUDIO_PSA_LINUX
    audproc_service_init(TRUE, D_AUDIO_MODE_WEB_CONTAINER);
#else
    audproc_service_init(FALSE, D_AUDIO_MODE_UNDEFINED);
#endif
    my_qobj = qobj;

    ETG_TRACE_USR3(("[audproc_object_init]: EXIT"));
    pr_debug("EXIT\n");
}


/*******************************************************************************
*
* FUNCTION: audproc_object_install_signal_table
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_install_signal_table
(
    GObjectClass * gobj_class
)
{
    int i = 0;

    pr_debug("ENTERED");
    ETG_TRACE_USR3(("[audproc_object_install_signal_table]: ENTERED"));

    const gchar* signalNames[E_SIGNAL_COUNT] =
    {
            SIGNAL_AUDPROC_STATUS,
            SIGNAL_AUDPROC_MICRO_LEVEL_STATUS,
            SIGNAL_AUDPROC_ARKAMYS_LEVEL_LOSS
    };

    /* Create the signals in one loop, since they all are similar
     (except for the names). */
    for (i = 0; i <= (int)E_SIGNAL_COUNT; i++)
    {
        guint signalId;

        /* Most of the time you will encounter the following code without
         * comments. This is why all the parameters are documented
         * directly below.
         */

        if( i == (int)E_SIGNAL_AUDPROC_MICRO_LEVEL_STATUS)
        {
            signalId = g_signal_new(signalNames[i],
                                G_OBJECT_CLASS_TYPE(gobj_class),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL,
                                NULL,
                                g_cclosure_marshal_VOID__STRING,
                                G_TYPE_NONE,
                                1,
                                G_TYPE_LONG);

            /* Store the signal Id into the class state, so that we can use
             * it later.
             */
            my_signal[E_SIGNAL_AUDPROC_MICRO_LEVEL_STATUS].signal_id = signalId;

            pr_debug("signal(%s) with id(%d) was installed\n", signalNames[i], signalId);
        }

        else if(i == (int)E_SIGNAL_AUDPROC_STATUS)
        {
            signalId = g_signal_new(signalNames[i],
                                G_OBJECT_CLASS_TYPE(gobj_class),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL,
                                NULL,
                                g_cclosure_marshal_VOID__STRING,
                                G_TYPE_NONE,
                                6,
                                G_TYPE_UCHAR,                       /* GType(s),   <arg name="statusEntSrcEnum"          type="y"    direction="out"/> */
                                G_TYPE_UCHAR,                       /* GType(s),   <arg name="statusEntSrcActiveState"       type="y"    direction="out"/> */
                                G_TYPE_UCHAR,                       /* GType(s),   <arg name="statusVoiceSrcEnum"         type="y"    direction="out"/> */
                                G_TYPE_UCHAR,                       /* GType(s),   <arg name="statusVoiceSrcActiveState"      type="y"    direction="out"/> */
                                G_TYPE_UCHAR,                       /* GType(s),   <arg name="statusInfoSrcEnum"         type="y"    direction="out"/> */
                                G_TYPE_UCHAR);                      /* GType(s),   <arg name="statusInfoSrcActiveState"      type="y"    direction="out"/> */


            /* Store the signal Id into the class state, so that we can use
             * it later.
             */
            my_signal[E_SIGNAL_AUDPROC_STATUS].signal_id = signalId;

            pr_debug("signal(%s) with id(%d) was installed\n", signalNames[i], signalId);
        }
        else if(i == (int)E_SIGNAL_AUDPROC_ARKAMYS_LEVEL_LOSS)
        {
            signalId = g_signal_new(signalNames[i],
                                G_OBJECT_CLASS_TYPE(gobj_class),
                                G_SIGNAL_RUN_LAST,
                                0,
                                NULL,
                                NULL,
                                g_cclosure_marshal_VOID__STRING,
                                G_TYPE_NONE,
                                1,
                                G_TYPE_INT);                      /* GType(s),   <arg name="StatusArkamysLevelLoss"      type="i"    direction="out"/> */


            /* Store the signal Id into the class state, so that we can use
             * it later.
             */
            my_signal[E_SIGNAL_AUDPROC_ARKAMYS_LEVEL_LOSS].signal_id = signalId;

            pr_debug("signal(%s) with id(%d) was installed\n", signalNames[i], signalId);
        }
        else
        {
            pr_debug("signal not supported\n");
        }
    }

    ETG_TRACE_USR3(("[audproc_object_install_signal_table]: EXIT"));
    pr_debug("EXIT\n");
}

/*******************************************************************************
*
* FUNCTION: audproc_object_set_property
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_set_property
(
    GObject*        gobj,
    guint           prop_id,
    const GValue*   value,
    GParamSpec*     pspec
)
{
    gint status ;
    const struct aplParameter *p;
    guint table_size = audproc_configuration_get_param_table_size();
    const struct aplParameter *table = audproc_configuration_get_param_table();

    if(!table)
    {
        pr_debug("parameter table NULL");
        return;
    }

    if(!table_size)
    {
        pr_debug("parameter table size(%d)",table_size);
        return;
    }

    p = table + prop_id;


    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_object_set_property]: ENTERED"));

    if (prop_id < 1 || prop_id > table_size)
    {
        G_OBJECT_WARN_INVALID_PROPERTY_ID(gobj, prop_id, pspec);
        return;
    }

    status = audproc_service_do_set_param(AUD_PROC_SRC_ALL,(guint32)p->apl_data_id, p->apl_type, value);

    if(status)
        pr_debug("set of property(%d) fails with err(%d)", prop_id, status);

    ETG_TRACE_USR3(("[audproc_object_set_property]: EXIT"));
    pr_message("EXIT");

    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_get_property
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_get_property
(
    GObject*    gobj,
    guint       prop_id,
    GValue*     value,
    GParamSpec* pspec
)
{
    gint status ;
    const struct aplParameter *p;
    guint table_size = audproc_configuration_get_param_table_size();
    const struct aplParameter *table = audproc_configuration_get_param_table();

    if(!table)
    {
        pr_debug("parameter table NULL");
        return;
    }

    if(!table_size)
    {
        pr_debug("parameter table size(%d)",table_size);
        return;
    }

    p = table + prop_id;


    pr_message("ENTERED");
    ETG_TRACE_USR3(("[audproc_object_get_property]: ENTERED"));

    if (prop_id < 1 || prop_id > table_size)
    {
        G_OBJECT_WARN_INVALID_PROPERTY_ID(gobj, prop_id, pspec);
        return;
    }

    status = audproc_service_do_get_param(AUD_PROC_SRC_ALL, (guint32)p->apl_data_id, p->apl_type, value);

    if(status)
        pr_debug("Get of property(%d) fails with err(%d)", prop_id, status);


    ETG_TRACE_USR3(("[audproc_object_get_property]: EXIT"));
    pr_message("EXIT");

    return;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_install_property_table
*
* DESCRIPTION:
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_install_property_table
(
    GObjectClass*               gobj_class
)
{

    guint prop_id;
    GParamSpec *pspec;
    const struct aplParameter *p;
    const struct aplParameter *table = audproc_configuration_get_param_table();
    guint table_size = audproc_configuration_get_param_table_size();

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_install_property_table]: ENTERED"));

    if(!table)
    {
        pr_debug("parameter table NULL\n");
        return;
    }
    if(!table_size)
    {
        pr_debug("parameter table size(%d)\n",table_size);
        return;
    }

    pr_debug("%d properties\n", table_size - 1);
    ETG_TRACE_USR3(("[audproc_object_install_property_table]: %d properties", table_size - 1));

    for (prop_id = 1, p = table + 1; prop_id < table_size; prop_id++, p++)
    {
        pr_debug("installing property %d: %s\n", prop_id, p->apl_name);
        switch (p->parm_type)
        {
            case G_TYPE_INT:
            {
                gint pmin, pmax, pdefault;

                if (p->apl_type < AUDPROC_TYPE_I32)
                {
                    pmin = G_MINSHORT;
                    pmax = G_MAXSHORT;
                }
                else
                {
                    pmin = G_MININT;
                    pmax = G_MAXINT;
                }
                pdefault = 0;
                pspec = g_param_spec_int(p->prop_name, p->apl_name, p->apl_name, pmin, pmax, pdefault, p->flags);

                pr_debug("p->prop_name(%s),p->apl_name(%s), pmin(%d), pmax(%d), p->flags(%d), pspec(%p)\n", p->prop_name,
                                                                                               p->apl_name,
                                                                                               pmin,
                                                                                               pmax,
                                                                                               p->flags,
                                                                                               pspec);


                g_object_class_install_property(gobj_class, prop_id, pspec);

                break;
            }
            case G_TYPE_UINT:
            {
                guint pmin, pmax, pdefault;

                if (p->apl_type < AUDPROC_TYPE_U32)
                    pmax = G_MAXUSHORT;
                else
                    pmax = G_MAXUINT;
                pmin = 0;
                pdefault = 0;
                pspec = g_param_spec_uint(p->prop_name, p->apl_name, p->apl_name, pmin, pmax, pdefault, p->flags);
                pr_debug("p->prop_name(%s),p->apl_name(%s), pmin(%d), pmax(%d), p->flags(%d), pspec(%p)\n", p->prop_name,
                                                                                               p->apl_name,
                                                                                               pmin,
                                                                                               pmax,
                                                                                               p->flags,
                                                                                               pspec);
                g_object_class_install_property(gobj_class, prop_id, pspec);

                break;
            }
            case G_TYPE_BOXED:
            {
                GType etype;

                if (p->element_type == G_TYPE_INT)
                    etype = DBUS_TYPE_G_INT_ARRAY;
                else
                    etype = DBUS_TYPE_G_UINT_ARRAY;
                pspec = g_param_spec_boxed(p->prop_name, p->apl_name, p->apl_name, etype, p->flags);
                pr_debug("p->prop_name(%s),p->apl_name(%s), etype(%lu),p->flags(%d), pspec(%p)\n", p->prop_name,
                                                                                               p->apl_name,
                                                                                               etype,
                                                                                               p->flags,
                                                                                               pspec);
                g_object_class_install_property(gobj_class, prop_id, pspec);

                break;
            }
            default:
                pr_error("Unsupported GType %lu in property table\n", p->parm_type);
                break;
        }
    }

    ETG_TRACE_USR3(("[audproc_object_install_property_table]: EXIT"));
    pr_debug("EXIT\n");

    return;
}
/*******************************************************************************
*
* FUNCTION: audproc_object_class_init
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void audproc_object_class_init(AudprocObjectClass * qobj_class)
{
    GObjectClass *gobj_class    = G_OBJECT_CLASS(qobj_class);

    pr_debug("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_class_init]: ENTERED"));

    dbus_g_object_type_install_info(AUDPROC_TYPE_OBJECT, &dbus_glib_audproc_object_object_info);

    gobj_class->finalize = audproc_object_finalize;
    gobj_class->set_property = audproc_object_set_property;
    gobj_class->get_property = audproc_object_get_property;
    //gobj_class->notify = audproc_object_set_signal;
    audproc_object_install_signal_table(gobj_class);
    audproc_object_install_property_table(gobj_class);

    pr_debug("EXIT\n");
    ETG_TRACE_USR3(("[audproc_object_class_init]: EXIT"));
}


/*******************************************************************************
*
* FUNCTION: audproc_object_get_version
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_get_version
(
    AudprocObject*  qobj,
    GArray**        version,
    gchar**         version_string,
    gchar**         version_comment,
    GError**        error
)
{
  //To remove Gen4 Prio1 Compiler Warnings
  (void)qobj;
  (void)version;
  (void)version_string;
  (void)version_comment;
  (void)error;

    return TRUE;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_get_status
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_get_status
(
    AudprocObject*  qobj,
    GError**        error
)
{
    int status = AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_get_status]: ENTERED"));

    ret = audproc_service_report_error(status, "audproc_object_audproc_get_status", error);

    pr_message("EXIT\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_get_status]: EXIT"));
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_initialize
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_initialize
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    gchar*          AudprocConfigurationFile,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_initialize]: ENTERED"));

    status = audproc_service_do_initialize(AudprocSourceId, AudprocConfigurationFile);

    ret = audproc_service_report_error(status, "audproc_object_audproc_initialize", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_initialize]: EXIT"));
    pr_message("EXIT\n");

    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_select_source
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_source_select
(
    AudprocObject*  qobj,
    guchar          AudprocSourceSelectMode,
    guchar          AudprocSourceId,
    gchar*          AudprocAlsaDev,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_source_select]: ENTERED"));

    status = audproc_service_do_select_active_source((audproc_source_select_Mode)AudprocSourceSelectMode,
                                                      AudprocSourceId,
                                                      AudprocAlsaDev);

    ret = audproc_service_report_error(status, "audproc_object_audproc_source_select", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_source_select]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_source_activity
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_source_activity
(
    AudprocObject*      qobj,
    guchar              AudprocSourceMapId,
    guchar              AudprocSourceActivity,
    GError**            error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_source_activity]: ENTERED"));

    status = audproc_service_do_source_activity(AudprocSourceMapId, AudprocSourceActivity);

    ret = audproc_service_report_error(status, "audproc_object_audproc_source_activity", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_source_activity]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_set_configuration
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_set_configuration
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    gchar*          AudprocConfigurationFile,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_set_configuration]: ENTERED"));

    status = audproc_service_do_initialize(AudprocSourceId, AudprocConfigurationFile);

    ret = audproc_service_report_error(status, "audproc_object_audproc_set_configuration", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_set_configuration]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_destroy
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_destroy
(
    AudprocObject * qobj,
    guchar          AudprocSourceId,
    GError ** error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_destroy]: ENTERED"));

    status = audproc_service_do_destroy(AudprocSourceId);

    ret = audproc_service_report_error(status, "audproc_object_audproc_destroy", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_destroy]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}
/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_get_audproc_configuration
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_get_audproc_configuration
(
    AudprocObject*  qobj,
    gchar**         DatasetName,
    GArray**        AudprocProperties,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_get_audproc_configuration]: ENTERED"));

    ret = audproc_service_report_error(status, "audproc_object_audproc_get_audproc_configuration", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_get_audproc_configuration]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4  Compiler Warnings
    (void)qobj;
    (void)DatasetName;
    (void)AudprocProperties;

    return ret;
}
/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_start_audio
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_start_audio
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_start_audio]: ENTERED"));

    status = audproc_service_do_start_streaming(AudprocSourceId);

    ret = audproc_service_report_error(status, "audproc_object_audproc_start_audio", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_start_audio]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}
/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_stop_audio
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_stop_audio
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_stop_audio]: ENTERED"));

    status = audproc_service_do_stop_streaming(AudprocSourceId);

    ret = audproc_service_report_error(status, "audproc_object_audproc_stop_audio", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_stop_audio]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_mute
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_mute
(
    AudprocObject*  qobj,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_mute]: ENTERED"));

    status = audproc_service_do_mute();

    ret = audproc_service_report_error(status, "audproc_object_audproc_stop_audio", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_mute]: EXIT"));
    pr_message("EXIT\n");

    (void)qobj;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_create
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_create
(
    AudprocObject*  qobj,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    status = audproc_service_do_create();
    ETG_TRACE_USR3(("[audproc_object_audproc_create]: ENTERED"));

    ret = audproc_service_report_error(status, "audproc_object_audproc_stop_audio", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_create]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_set_param
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_set_param
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    guint           ParameterId,
    guchar          ParamType,
    guint32         uParameter,
    gint32          iParameter,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;
    audproc_type type = (audproc_type)ParamType;
    GValue value = { 0, };


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_set_param]: ENTERED"));

    if(   (type == AUDPROC_TYPE_U16)
        ||(type == AUDPROC_TYPE_U32))
    {
        //g_value_unset (&value);
        g_value_init (&value, G_TYPE_UINT);
        g_value_set_uint (&value, uParameter);
    }
    else
    {
        //g_value_unset (&value);
        g_value_init (&value, G_TYPE_INT);
        g_value_set_int (&value, (gint)uParameter);
    }

    status = audproc_service_do_set_param(AudprocSourceId, ParameterId, type, &value);

    ret = audproc_service_report_error(status, "audproc_object_audproc_set_param", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_set_param]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    (void)iParameter;
    return ret;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_write_data
*
* DESCRIPTION: This function initialize the  Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_write_data
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    guint           ParameterId,
    GArray*         data,
    GArray**        result,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_write_data]: ENTERED"));

    status = audproc_service_do_write_data(AudprocSourceId, ParameterId, data, result);

    ret = audproc_service_report_error(status, "audproc_object_audproc_write_data", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_write_data]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_read_data
*
* DESCRIPTION: This function initialize the  Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_read_data
(
    AudprocObject*  qobj,
    guchar          AudprocSourceId,
    guint           ParameterId,
    GArray*         ReadInfos,
    GArray**        data,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;

    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_read_data]: ENTERED"));

    status = audproc_service_do_read_data(AudprocSourceId, ParameterId, ReadInfos, data );

    ret = audproc_service_report_error(status, "audproc_object_audproc_read_data", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_read_data]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}




/*******************************************************************************
*
* FUNCTION: audproc_object_audproc_setup_debug
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_setup_debug
(
    AudprocObject*  qobj,
    guchar          AudprocSelectMode,
    GError**        error
)
{
    int status = (int)AUDPROC_ERR_OK;
    gboolean ret;


    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_setup_debug]: ENTERED"));

    status = audproc_service_do_set_debug_mode((enum eAudprocDebugMode)AudprocSelectMode);

    ret = audproc_service_report_error(status, "audproc_object_audproc_set_array_param", error);

    ETG_TRACE_USR3(("[audproc_object_audproc_setup_debug]: EXIT"));
    pr_message("EXIT\n");
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    return ret;
}



extern GMainLoop *audproc_main_loop;

/*******************************************************************************
*
* FUNCTION: audproc_object_terminate
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
gboolean audproc_object_audproc_terminate(AudprocObject * qobj, GError ** error)
{
    pr_message("ENTERED\n");
    ETG_TRACE_USR3(("[audproc_object_audproc_terminate]: ENTERED"));

    if(audproc_main_loop)
        g_main_loop_quit(audproc_main_loop);
    //To remove Gen4 Prio1 Compiler Warnings
    (void)qobj;
    (void)error;

    ETG_TRACE_USR3(("[audproc_object_audproc_terminate]: EXIT"));
    pr_message("EXIT\n");

    return TRUE;
}

/*******************************************************************************
*
* FUNCTION: audproc_object_emit_signal_micro_level
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_object_emit_signal_micro_level(guint32 MicroLevel)
{
    int             err = AUDPROC_ERR_OK;
    guint           signalId;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_object_emit_signal_micro_level]: ENTERED"));

    signalId =  my_signal[E_SIGNAL_AUDPROC_MICRO_LEVEL_STATUS].signal_id;

    /* This is the simplest way of emitting signals. */
    if(my_qobj)
        g_signal_emit( my_qobj, signalId, 0, (guint16)MicroLevel);

    ETG_TRACE_USR4(("[audproc_object_emit_signal_micro_level]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_emit_signal_arkamys_level_loss
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_object_emit_signal_arkamys_level_loss(gint ArkamysLevelLoss)
{
    int             err = AUDPROC_ERR_OK;
    guint           signalId;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_object_emit_signal_arkamys_level_loss]: ENTERED"));

    signalId =  my_signal[E_SIGNAL_AUDPROC_ARKAMYS_LEVEL_LOSS].signal_id;

    /* This is the simplest way of emitting signals. */
    if(my_qobj)
        g_signal_emit( my_qobj, signalId, 0, ArkamysLevelLoss);

    ETG_TRACE_USR4(("[audproc_object_emit_signal_arkamys_level_loss]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}


#define S_SERV_SRC_ON   (guint8)1
#define S_SERV_SRC_OFF  (guint8)0




static int audproc_object_emit_signal_sources_state(audproc_source_type src, guint8 state)
{
    int             err = AUDPROC_ERR_OK;
    guint           signalId;

    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_object_emit_signal_sources_state]: ENTERED"));

    guint8 src_voice_id = 0;
    guint8 src_voice_as = AUDPROC_AS_STREAM_SILENCE;
    guint8 src_info_id = 0;
    guint8 src_info_as = AUDPROC_AS_STREAM_SILENCE;
    guint8 src_ent_id   = 0;
    guint8 src_ent_as   = AUDPROC_AS_STREAM_SILENCE;


    signalId =  my_signal[E_SIGNAL_AUDPROC_STATUS].signal_id;

    /* This is the simplest way of emitting signals. */
    struct audproc_instance_object* obj_src_voice =  audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_VOICE);
    if(obj_src_voice)
    {
        src_voice_id    = (guint8)obj_src_voice->audproc_this_src;
        if(src == AUD_PROC_SRC_TYPE_VOICE )
            src_voice_as = (guint8)state;
        else
            src_voice_as = (guint8)audproc_state_machine_get_audio_state(obj_src_voice->audproc_sm_inst);
    }
    struct audproc_instance_object* obj_src_info =  audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_INFO);
    if(obj_src_info)
    {
        src_info_id    = (guint8)obj_src_info->audproc_this_src;
        if(src == AUD_PROC_SRC_TYPE_INFO )
            src_info_as = (guint8)state;
        else
            src_info_as = (guint8)audproc_state_machine_get_audio_state(obj_src_info->audproc_sm_inst);
    }

    struct audproc_instance_object* obj_src_ent =  audproc_service_get_active_type_src (AUD_PROC_SRC_TYPE_ENT);
    if(obj_src_ent)
    {
        src_ent_id      = (guint8)obj_src_ent->audproc_this_src;
        if(src == AUD_PROC_SRC_TYPE_ENT )
            src_ent_as = (guint8)state;
        else
            src_ent_as = (guint8)audproc_state_machine_get_audio_state(obj_src_ent->audproc_sm_inst);
    }

    pr_message("emit signal src_ent:%d, src_ent_value:%d, src_voice:%d, src_voice_value:%d, src_info:%d, src_info_value:%d\n", src_ent_id,src_ent_as,src_voice_id,src_voice_as,src_info_id,src_info_as);
    ETG_TRACE_USR3(("[audproc_object_emit_signal_sources_state]: emit signal src_ent:%d, src_ent_value:%d, src_voice:%d, src_voice_value:%d, src_info:%d, src_info_value:%d", src_ent_id,src_ent_as,src_voice_id,src_voice_as,src_info_id,src_info_as));

    ETG_TRACE_USR1(("[new audio routes state]"));
    ETG_TRACE_USR1(("[AUDPROC_AS][ENT_SRC_%d   : ENT_AS_%s]", src_ent_id, audproc_audio_state_str(src_ent_as)));
    ETG_TRACE_USR1(("[AUDPROC_AS][VOICE_SRC_%d : VOICE_AS_%s]", src_voice_id, audproc_audio_state_str(src_voice_as)));
    ETG_TRACE_USR1(("[AUDPROC_AS][INFO_SRC_%d  : INFO_AS_%s]", src_info_id, audproc_audio_state_str(src_info_as)));

    if(my_qobj)
        g_signal_emit( my_qobj, signalId, 0, src_ent_id, src_ent_as, src_voice_id, src_voice_as, src_info_id, src_info_as);


    ETG_TRACE_USR4(("[audproc_object_emit_signal_sources_state]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}



/*******************************************************************************
*
* FUNCTION: audproc_object_emit_signal_voice_sources_state
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_object_emit_signal_voice_sources_state(guint8 state)
{
    int             err = AUDPROC_ERR_OK;


    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_object_emit_signal_voice_sources_state]: ENTERED"));

    err = audproc_object_emit_signal_sources_state(AUD_PROC_SRC_TYPE_VOICE, state);

    ETG_TRACE_USR4(("[audproc_object_emit_signal_voice_sources_state]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_emit_signal_info_sources_state
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_object_emit_signal_info_sources_state(guint8 state)
{
    int             err = AUDPROC_ERR_OK;


    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_object_emit_signal_info_sources_state]: ENTERED"));

    err = audproc_object_emit_signal_sources_state(AUD_PROC_SRC_TYPE_INFO, state);

    ETG_TRACE_USR4(("[audproc_object_emit_signal_info_sources_state]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}


/*******************************************************************************
*
* FUNCTION: audproc_object_emit_signal_ent_sources_state
*
* DESCRIPTION: This function
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int audproc_object_emit_signal_ent_sources_state(guint8 state)
{
    int             err = AUDPROC_ERR_OK;


    pr_debug("ENTERED\n");
    ETG_TRACE_USR4(("[audproc_object_emit_signal_ent_sources_state]: ENTERED"));

    err = audproc_object_emit_signal_sources_state(AUD_PROC_SRC_TYPE_ENT, state);

    ETG_TRACE_USR4(("[audproc_object_emit_signal_ent_sources_state]: EXIT"));
    pr_debug("EXIT\n");

    return err;
}
