//
// VolumeManager/Configuration/StreamSet.cpp
//
// Collection of objects controlling audio streams
//
//  Created on: Jun 26, 2014
//      Author: Martin Koch, Fa. ESE.de
//


// framework
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include <etrace_if.h>

#include "Volume/Types.h"
#include "Volume/Utilities/Uncopyable.h"
#include "Volume/Utilities/Array.hpp"
#include "StreamSet.h"
// - - - - - - - - - - - - -

#include "Fading/Fading_IF.h"
#include "Volume/Configuration/Configuration.h"
#include "Volume/Configuration/ConfigDetails.hpp"
#include "Volume/Utilities/StringCopy.hpp"
#include "Volume/Engine/Functions/IFunction.h"
#include "Stream.h"

#include "../../../PostOfficeAdapter/InternalComponentCommunication/DataTypes/MessageDataTypes/VolumeData.h"

#include "controllerplugin_Trace.h"
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_CONTROLLER_VOLUME
#include "trcGenProj/Header/StreamSet.cpp.trc.h"



namespace VolumeManager
{

//   // --------------------------------------------------------------------------
//
//   template<typename T>
//   inline void swapMember (T& lhs, T& rhs)
//   {
//      T tmp = lhs;
//      lhs = rhs;
//      rhs = tmp;
//   };
//
   // -----------------------------------------------------------------------------

   /* constructor */ StreamSet :: StreamSet (const StreamSetConfig& setConfig, const Configuration& config, Fading_IF* pFading)
      : _pConfig(&setConfig)
      , _pFading(pFading)
      , _functions(setConfig.numberOfFunctions)
      , _streamCount(config.getStreamCount ())
      , _pStreams(new Stream[_streamCount])
      , _activeStreamID(EN_AUDIO_SOURCE_STREAM_DEFAULT)
   {
      if (NULL == _pStreams)   // shall never happen
      {
         _streamCount = 0;
         return;
      }

      ETG_TRACE_USR4(("StreamSet :: StreamSet _streamCount %d (_pConfig %p)",_streamCount,_pConfig));
    if((_pConfig) && (_pConfig->name))
    {
      ETG_TRACE_USR4(("StreamSet :: StreamSet (enResource %d) _pConfig->name %s",_pConfig->enResource,_pConfig->name));
    }

      size_t streamIndex = 0;
      for (size_t i = 0; i < _streamCount; ++i)
      {
         if (config.pGetStreamConfig(i) && (_pConfig == config.pGetStreamConfig(i)->pStreamSet))
         {
           ETG_TRACE_USR4((""));
            Stream s(*(config.pGetStreamConfig(i)));
            _pStreams[streamIndex].swap(s);
            ++streamIndex;
         }
      }
      _streamCount = streamIndex;

      ETG_TRACE_USR4(("StreamSet :: StreamSet _streamCount set to %d",_streamCount));

      if (_streamCount)
         _activeStreamID = _pStreams[0].enGetStreamID();

      // initialize empty array of functions
      for (unsigned i = 0; i < _functions.size(); ++i)
         _functions[i] = NULL;
      _functions.setSize(0);
   }

   // -----------------------------------------------------------------------------

   /* default constructor */ StreamSet :: StreamSet ()
      : _pConfig(NULL)
      , _pFading(NULL)
      , _functions(0)
      , _streamCount(0)
      , _pStreams(NULL)
      , _activeStreamID(EN_AUDIO_SOURCE_STREAM_DEFAULT)
   {

   }

   // -----------------------------------------------------------------------------

   /* virtual destructor */ StreamSet :: ~StreamSet()
   {
      // cleanup, but prevent throwing exceptions on failure
      delete[] _pStreams;
      _pConfig = NULL;
      _pFading = NULL;
   }

   // -----------------------------------------------------------------------------

   void StreamSet :: swap (StreamSet& rhs)
   {
      swapMember<const StreamSetConfig*>(_pConfig, rhs._pConfig);
      swapMember<Fading_IF*>(_pFading, rhs._pFading);
      swapMember<size_t>(_streamCount, rhs._streamCount);
      swapMember<Stream*>(_pStreams, rhs._pStreams);
      swapMember<tenStream>(_activeStreamID, rhs._activeStreamID);
      _functions.swap(rhs._functions);
   }

   // -----------------------------------------------------------------------------

   Stream* StreamSet:: pGetStream (tenStream enStream) const
   {
     ETG_TRACE_ERR(("StreamSet::pGetStream() called for index %u (_streamCount %d)",enStream,_streamCount));
      for (size_t i = 0; _pStreams && (i < _streamCount); ++i)
      {
       ETG_TRACE_ERR(("StreamSet::pGetStream() pStreams[%d].enGetResource() = %d",i,_pStreams[i].enGetResource()));
       ETG_TRACE_ERR(("StreamSet::pGetStream() pStreams[%d].enGetStreamID() = %d",i,_pStreams[i].enGetStreamID()));
         if ((enGetResource() == _pStreams[i].enGetResource()) && (_pStreams[i].enGetStreamID() == enStream))
         {
            ETG_TRACE_USR4(("StreamSet::pGetStream returned & _pStream[%i] %s "
               , i,_pStreams[i].getName()));
            return &(_pStreams[i]);
         }
      }
      // requested stream not in our set
      ETG_TRACE_ERR(("StreamSet::pGetStream() for index %u returned NULL ", enStream))
      return NULL;
   }

   // -----------------------------------------------------------------------------

   tenStream  StreamSet:: enGetActiveStreamID () const
   {
      return _activeStreamID;
   }

   // -----------------------------------------------------------------------------

   Stream* StreamSet:: pGetActiveStream ()
   {
      return pGetStream(_activeStreamID);
   }

   // -----------------------------------------------------------------------------

   tenResource  StreamSet:: enGetResource () const
   {
      if (_pConfig)
      {
        ETG_TRACE_ERR(("StreamSet::enGetResource() returning  _pConfig->enResource %d",  _pConfig->enResource));
         return _pConfig->enResource;
      }
      else
      {
        ETG_TRACE_ERR(("StreamSet::enGetResource() _pConfig is NULL returning UndefinedResource %d",UndefinedResource));
         return UndefinedResource;
      }
   }

   // -----------------------------------------------------------------------------

   tenVolumeMode StreamSet:: enGetVolumeMode ()
   {
      Stream* pActiveStream = pGetStream(_activeStreamID);
      if (pActiveStream)
         return pActiveStream->enGetCurrentMode();
      else
         return EN_INT_VOLMODE_NOT_DEF;
   }

   // -----------------------------------------------------------------------------

   tenInternalSource StreamSet:: enGetCurrentSource () const
   {
      const Stream* pActiveStream = pGetStream(_activeStreamID);
      ETG_TRACE_USR2(("StreamSet:: enGetCurrentSource() _activeStreamID=%d, pActiveStream %p",_activeStreamID,pActiveStream));
      if (pActiveStream)
      {
        ETG_TRACE_USR2(("StreamSet:: enGetCurrentSource() returning source %d", pActiveStream->enGetCurrentSource()));
         return pActiveStream->enGetCurrentSource();
      }
      else
         return AUD_INT_SRC_NOSOURCE;
   }

   // -----------------------------------------------------------------------------

   tenInternalSource StreamSet:: enGetRecentEntertainmentSource () const
   {
      // Assuming that every stream-set has only one stream dealing with entertainment sources,
      // we scan our streams and return the first hit.

      for (unsigned i = 0; _pStreams && (i < _streamCount); ++i)
         if (AUD_INT_SRC_NOSOURCE != _pStreams[i].enGetRecentEntertainmentSource ())
            return _pStreams[i].enGetRecentEntertainmentSource ();

      return AUD_INT_SRC_NOSOURCE;
   }

   // -----------------------------------------------------------------------------

   void StreamSet:: vSetActiveStream (tenStream enStreamID)
   {
      if (NULL == pGetStream (enStreamID))
      {
         ETG_TRACE_FATAL(("vSetActiveStream() - E R R O R :  attempt to activate non-existent Stream %u"
               , ETG_CENUM(tenStream, enStreamID)))
         return;
      }

      _activeStreamID = enStreamID;
      ETG_TRACE_USR3(("vSetActiveStream() - Stream %u now active"
            , ETG_CENUM(tenStream, enStreamID)))
   }

   // -----------------------------------------------------------------------------

   void StreamSet:: vAddFunction (IFunction* pFunction)
   {
      if (NULL == pFunction)
      {
         ETG_TRACE_FATAL(("vAddFunction() - E R R O R  : invalid pointer to function"))
         return;
      }

      unsigned i = _functions.size();
      _functions.setSize(1 + i);
      _functions[i] = pFunction;
   }

   // -----------------------------------------------------------------------------

   void StreamSet:: vApplyVolumeFilters (tenFadingReason enReason, VolumeData& volumeData)
   {
      // give our special functions the opportunity to apply their changes
      // on intended next volume user step

      for (unsigned i = 0; i < _functions.size(); ++i)
         if (_functions[i])
            _functions[i]->vAdjustVolume(enReason, volumeData);
   }

   // -----------------------------------------------------------------------------

   void StreamSet:: vApplyAmplifierFilters (tenFadingReason enReason, const VolumeData& volumeData, bool& useStandardAmplifierCommand)
   {
      // give our special functions the opportunity to manipulate
      // the amplifier command for intended next volume user step

      for (unsigned i = 0; i < _functions.size(); ++i)
         if (_functions[i])
            _functions[i]->vAdjustAmplifierVolume(enReason, volumeData, useStandardAmplifierCommand);
   }

   // -----------------------------------------------------------------------------

   Ramp  StreamSet:: queryRamp (tenActions enAction, tenFadingReason enReason, tenStream enStream) const
   {
      // obtain ramping parameters from Fading sub-component

      Ramp retVal(100, 0);  // fallback values
      if (_pFading)
      {
         retVal.lin = _pFading->fading_getRampLin (enAction, enReason, enStream);
         retVal.dB  = _pFading->fading_getRampdB (enAction, enReason, enStream);
      }
      else
         ETG_TRACE_COMP(("StreamSet::queryRamp() - WARNING : returning fallback ramp"))

      return retVal;
   }

   const dBCalculator* StreamSet::pGetCalculator() const
   {
     return _pConfig->pCalculator;
   }

   std::string StreamSet::getSinkName() const
   {
       if (_pConfig)
       {
           std::string name(_pConfig->name);
           return name;
       }
       std::string name("Speakers");
       return name;
   }

   std::string StreamSet::getSourceName() const
   {
       const Stream* pActiveStream = pGetStream(_activeStreamID);

        if (pActiveStream)
        {

            const char * c_name = pActiveStream->getName();
            std::string name(c_name);
            return name;
        }

        std::string name("NONE");
        return name;

   }

}  // namespace VolumeManager


