/*******************************************************************************
* FILE : aictrl.cpp
*
* SW-COMPONENT : aictrl
*
* DESCRIPTION : control ESAI interface during power On to setup int. clock
*
* AUTHOR : Patrick Rey (ESE)
*
* COPYRIGHT : (C)
*
* HISTORY : 08.07.2014
*
*
*
* date             item
* 08.07.2014       Initial version
*******************************************************************************/





/* c standard library */
#include <stdio.h>
#include <unistd.h>
#include <string.h>

/* linux kernel signal */
#include <signal.h>

/* alsa includes */
#include <alsa/asoundlib.h>

/* glib */
#include <glib/gi18n.h>
#include <glib.h>
#include "aictrl.h"





/*******************************************************************************
              VERSION COMPILER
*******************************************************************************/


/*******************************************************************************
              DEFINES - MACROS
*******************************************************************************/
#define RAISE(error, str) \
    if(error) \
    { \
        pr_warning("failed call (%s) with err(%d)\n", str, error); \
        pr_debug("exit with err(%d)\n", err); \
        return err; \
    } \
    else\
        pr_debug("success call (%s) \n", str); \

/*******************************************************************************
              Constant defintion
*******************************************************************************/



/*******************************************************************************
              STRUCTURE DEFINITION
*******************************************************************************/

/*******************************************************************************
              VARIABLE DECLARATIONS
*******************************************************************************/

static volatile int exit_flag = 0;
static snd_pcm_t* sndcard_handle = NULL;



/*******************************************************************************
              STATIC FUNCTION  DECLARATIONS
*******************************************************************************/

/*******************************************************************************
*
* FUNCTION: aictrl_signal_handler
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static void aictrl_signal_handler (int sig)
{
    pr_debug("received signal %d\n", sig);
    exit_flag = 1;
}

/*******************************************************************************
*
* FUNCTION: aictrl_switch_on_esai_clock
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
int aictrl_switch_on_esai_clock(void)
{
    snd_pcm_t *handle = (snd_pcm_t*)NULL;
    char snd_card_str[] = "AdevEsaiClockOnOff";

    pr_message("enter and set adev(%s)\n",snd_card_str );

    snd_pcm_hw_params_t *hw_params;
    unsigned int uval = 2;
    int dir = 0;

    int err = snd_pcm_open(&handle, snd_card_str, SND_PCM_STREAM_PLAYBACK, 0);

    if (err < 0)
    {
       pr_warning("can not open audio device: %s", snd_strerror(err));
    }
    else
    {
       /* save device handler */
       pr_debug("snd_pcm_open successfull with handel(%p) )\n", handle );
       sndcard_handle = handle;

       /* set HW parameter */
       pr_debug("Set hw_params\n");
       if((err = snd_pcm_hw_params_malloc(&hw_params)) < 0)
       {
           pr_warning("can not allocate hardware parameter structure (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_hw_params_malloc")

       if((err = snd_pcm_hw_params_any(handle, hw_params)) < 0)
       {
           pr_warning("can not initialize hardware parameter structure (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_hw_params_any")

       if((err = snd_pcm_hw_params_set_access(handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0)
       {
           pr_warning("can not set access type (%s)", snd_strerror (err));
       }

       RAISE(err,"snd_pcm_hw_params_set_access")

#if defined(VARIANT_S_FTR_ENABLE_FEAT_AUDIO_RNAIVI)
       if((err = snd_pcm_hw_params_set_format(handle, hw_params, SND_PCM_FORMAT_S32_LE)) < 0)
       {
           pr_warning("can not set sample format (%s)", snd_strerror (err));
       }
#else
       if((err = snd_pcm_hw_params_set_format(handle, hw_params, SND_PCM_FORMAT_S16_LE)) < 0)
       {
           pr_warning("can not set sample format (%s)", snd_strerror (err));
       }
#endif
       RAISE(err,"snd_pcm_hw_params_set_format")

       if((err = snd_pcm_hw_params_set_rate(handle, hw_params, 48000, 0)) < 0)
       {
           pr_warning("can not set sample rate (%s)", snd_strerror (err));
       }

       RAISE(err,"snd_pcm_hw_params_set_rate")

       if((err = snd_pcm_hw_params_set_channels(handle, hw_params, 1)) < 0)
       {
           pr_warning("can not set channel count (%s)", snd_strerror (err));
       }

       RAISE(err,"snd_pcm_hw_params_set_channels")

       if((err = snd_pcm_hw_params_set_periods_min(handle, hw_params, &uval, &dir)) < 0)
       {
           pr_warning("can not set periods (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_hw_params_set_periods_min")

       snd_pcm_uframes_t period_size = 0;
       if((err = snd_pcm_hw_params_get_period_size_max(hw_params, &period_size, &dir)) < 0)
       {
           pr_warning("can not get period_size max (%s)", snd_strerror (err));
       }

       RAISE(err,"snd_pcm_hw_params_get_period_size_max")

       pr_debug("Set period_size to %d\n", (int)period_size);

       if((err = snd_pcm_hw_params_set_period_size(handle, hw_params, period_size, dir)) < 0)
       {
           pr_warning("can not set period_size (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_hw_params_set_period_size")

       if((err = snd_pcm_hw_params(handle, hw_params)) < 0)
       {
           pr_warning("can not set parameters (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_hw_params")

       snd_pcm_uframes_t buffer_frames;
       if((err = snd_pcm_hw_params_get_buffer_size(hw_params, &buffer_frames)) < 0)
       {
           pr_warning("can not get buffer size (%s)", snd_strerror (err));
       }

       snd_pcm_hw_params_free(hw_params);

       RAISE(err,"snd_pcm_hw_params_get_buffer_size")

       /* set sw parameter*/
       snd_pcm_sw_params_t *sw_params;
       if((err = snd_pcm_sw_params_malloc(&sw_params)) < 0)
       {
           pr_warning("can not allocate software parameters structure (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_sw_params_malloc")

       if((err = snd_pcm_sw_params_current(handle, sw_params)) < 0)
       {
           pr_warning("can not initialize software parameters structure (%s)", snd_strerror(err));
       }

       RAISE(err,"snd_pcm_sw_params_current")
       if((err = snd_pcm_sw_params_set_avail_min(handle, sw_params, 0)) < 0)
       {
           pr_warning("can not set start mode (%s)", snd_strerror(err));
       }
       RAISE(err,"snd_pcm_sw_params_set_avail_min")

       snd_pcm_uframes_t boundary;
       if((err = snd_pcm_sw_params_get_boundary(sw_params, &boundary)) < 0)
       {
           pr_warning("can not get boundary (%s)", snd_strerror(err));
       }
       RAISE(err,"snd_pcm_sw_params_get_boundary")

       snd_pcm_uframes_t stop_threshold = boundary;
       if((err = snd_pcm_sw_params_set_stop_threshold(handle, sw_params, stop_threshold)) < 0)
       {
           pr_warning("can not set start mode (%s)", snd_strerror(err));
       }
       RAISE(err,"snd_pcm_sw_params_set_stop_threshold")

       if((err = snd_pcm_sw_params_set_start_threshold(handle, sw_params, 1)) < 0)
       {
           pr_warning("can not set start mode (%s)", snd_strerror(err));
       }
       RAISE(err,"snd_pcm_sw_params_set_start_threshold")

       if((err = snd_pcm_sw_params(handle, sw_params)) < 0)
       {
           pr_warning("can not set software parameters (%s)", snd_strerror(err));
       }
       RAISE(err,"snd_pcm_sw_params")

       if((err = snd_pcm_prepare(handle)) < 0)
       {
           pr_warning("can not prepare audio interface for use (%s)", snd_strerror(err));
       }
       RAISE(err,"snd_pcm_prepare")

       signed short buf[256] = {0};
       unsigned int bufsize = 256;
       /* start device */
       if((err = (int)snd_pcm_writei(handle, buf, bufsize)) < 0)
       {
           pr_warning("writei failed (%s)", snd_strerror(err));
       }

       if(err >= 0)
           err = 0;

       RAISE(err,"snd_pcm_writei")
    }
    return err;
}


/*******************************************************************************
*
* FUNCTION: aictrl_switch_off_esai_clock
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: [check xml file for parameter description]
*
* RETURNVALUE: int
*
*******************************************************************************/
static int aictrl_switch_off_esai_clock(void)
{
    snd_pcm_t *handle = (snd_pcm_t*)sndcard_handle;
    int err = 0;

    if(sndcard_handle != NULL)
    {
        if((err = snd_pcm_close(handle)) < 0)
        {
            pr_warning("can't close audio device %s", snd_strerror(err));
        }
        else
        {
            pr_debug("success close alsa device AdevEsaiClockOnOff\n");
            handle = NULL;
        }
    }
    else
    {
        pr_debug("device was closed already");
    }
    sndcard_handle = handle;

    return err;
}


 /*******************************************************************************
*
* FUNCTION: main
*
* DESCRIPTION: This function initialize the sse Engine
*
* PARAMETER: int argc,
             char *argv[]
*
* RETURNVALUE: int
*
*******************************************************************************/
int main (int /* argc */, char** /* argv */)
{
    struct sigaction act;

    pr_message("enter\n");

    memset (&act, '\0', sizeof(act));

    act.sa_handler = &aictrl_signal_handler;

    if (sigaction(SIGTERM, &act, NULL) < 0)
    {
        perror ("sigaction");
        return 1;
    }

    int err = aictrl_switch_on_esai_clock();
    if(err)
    {
        pr_critical("open alsa device for ESAI clock switch ON fail with err(%d)\n", err);
        return 2;
    }

    pr_message("ESAI clock is activated\n");

    while (!exit_flag)
        usleep(100000);

    /* close device */
    aictrl_switch_off_esai_clock();

    pr_message("aicrtl deamon terminated and ESAI clock is deactivated\n");

    return 0;
}
