/******************************************************************************

FILE:           etg_adit_linux_platform.c

PROJECT:        ADIT Gen2 Platform

  SW-COMPONENT:   general library

  DESCRIPTION:    interface to use generated traces

  AUTHOR:         CM-DI/PJ-CB36 - Hessling, Matthias

  COPYRIGHT:      (c) 2013 Robert Bosch Car Multimedia GmbH

  HISTORY:


  Date       | Rev. | Author                  | Modification
  ----------------------------------------------------------------------------
  27.01.2011 | 1.0  | CM-DI/PJ-CF32 - Resch   | Initial revision
  27.01.2013 | 2.0  | CM-DI/PJ-CB36 - Hessling| merge to linux

******************************************************************************/
#include <unistd.h>
#include <sys/fcntl.h>

#include <trace_interface.h>

#define ETG_S_IMPORT_INTERFACE_GENERIC
#include "etg_if.h"


/**************************************************************************
* FUNCTION       UTIL_ETG_DoTrace
* EXPLANATION    Wrapper from etg standard function to ADIT trace
* PARAMETER      tU16 u16TraceClass            - u16TraceClass
*                TR_tenTraceLevel enTraceLevel - enTraceLevel
*                const tS8* Buffer             - buffer
*                tU32 u32Length                - sizeof(buffer)
* RETURN VALUES  NONE
***************************************************************************/
etg_tVoid TRACE_ETG_DoTrace(etg_tU16 u16TraceClass, etg_tU16 enTraceLevel, const etg_tS8* pcos8Buffer, etg_tU32 u32Length)
{
   //U32 TR_core_uwTraceOut(U32 uwLen, U16 trClass, TR_tenTraceLevel trLevel, U8* p_ubTrData);
   U8* pData = (U8*)pcos8Buffer; //lint thie is based on existing interfaces
   (void)TR_core_uwTraceOut(u32Length, u16TraceClass, (TR_tenTraceLevel)enTraceLevel, pData);
}

/**************************************************************************
* FUNCTION       TRACE_ETG_bIsTraceActive
* EXPLANATION    Wrapper from etg standard function to ADIT trace
* PARAMETER      tU16 u16TraceClass            - trace class
*                TR_tenTraceLevel enTraceLevel - trace level
* RETURN VALUES  tBool TRUE  --> trace class is active
*                      FALSE --> trace class is not active
***************************************************************************/
etg_tBool TRACE_ETG_bIsTraceActive(etg_tU16 u16TraceClass, etg_tU16 enTraceLevel)
{
   // BOOL TR_core_bIsClassSelected(U16 trClass, TR_tenTraceLevel enLevel);
   etg_tBool bActive = (TR_core_bIsClassSelected(u16TraceClass, (TR_tenTraceLevel)enTraceLevel) != (BOOL)TR_ENABLED_CLASS_NONE)?TRUE:FALSE;
   return bActive;
}

/**************************************************************************
* FUNCTION       TRACE_ETG_s32RegTraceCallback
* EXPLANATION    Wrapper from etg standard function to ADIT Util
* PARAMETER      TR_tenTraceChan eChanId   - trace channek id
*                OSAL_tpfCallback pfvCb    - call beck function pointer
* RETURN VALUES  tS32 - result of the registration process
***************************************************************************/
etg_tS32 TRACE_ETG_s32RegTraceCallback(etg_tU16 eChanId, ETG_tpfCallback pfvCb)
{
   //BOOL TR_chan_acess_bRegChan(TR_tenTraceChan chan_id, TRACE_CALLBACK p_Func);
   etg_tS32 er = TR_chan_acess_bRegChan((TR_tenTraceChan)eChanId, (TRACE_CALLBACK)pfvCb)?ETG_RES_OK:ETG_RES_ERROR;
   return er; // 0 == E_OK and OSAL_OK other values see parts/include/tk/errno.h
}

/**************************************************************************
* FUNCTION       TRACE_ETG_s32UnregTraceCallback
* EXPLANATION    Wrapper from etg standard function to ADIT Util
* PARAMETER      TR_tenTraceChan eChanId   - trace channek id
*                OSAL_tpfCallback pfvCb    - call beck function pointer
* RETURN VALUES  tS32 - result of the registration process
***************************************************************************/
etg_tS32 TRACE_ETG_s32UnregTraceCallback(etg_tU16 eChanId, ETG_tpfCallback pfvCb)
{
   etg_tS32 er = TR_chan_acess_bUnRegChan((TR_tenTraceChan)eChanId, (TRACE_CALLBACK)pfvCb)?ETG_RES_OK:ETG_RES_ERROR;
   return er; // 0 == E_OK and OSAL_OK other values see parts/include/tk/errno.h
}

/**************************************************************************
* FUNCTION       lldErrMem_ETG_WriteFatalTrace
* EXPLANATION    Wrapper from etg stadart function to Error Memory
* PARAMETER      tU16 u16TraceClass            - U16CompId
*                const tU8* Buffer             - buffer
*                tU16 u16Length                - sizeof(buffer)
* RETURN VALUES  None
***************************************************************************/
etg_tVoid lldErrMem_ETG_WriteFatalTrace(etg_tU16 u16TraceClass, etg_tInt EntryFatal, etg_tU16 u16Length, const etg_tU8 *buffer)
{
   (void)u16TraceClass; (void)EntryFatal; // lint
   int errmem_fd = open("/dev/errmem", O_WRONLY);
   if(-1 != errmem_fd) {
      (void)write(errmem_fd, buffer, u16Length);
      (void)close(errmem_fd);
   }
}

etg_tS32 etgIF_s32RegisterEtgICmds(etg_i_tpfFileCallback callback, etg_tU32 length, etg_tU32* cmdIds)
{
  (void)callback;
  (void)length;
  (void)cmdIds;
  return ETG_RES_OK;
}

etg_tS32 etgIF_s32DeregisterEtgICmds(etg_tU32 length, etg_tU32* cmdIds)
{
  (void)length;
  (void)cmdIds;
  return ETG_RES_OK;
}

/**************************************************************************
* FUNCTION       vInitPlatformEtg
* EXPLANATION    Init ETG Function pointer
* PARAMETER      NONE
* RETURN VALUES  NONE
***************************************************************************/
void vInitPlatformEtg(void)
{
   if((ETG_pfvDoTrace       == NULL) &&
         (ETG_pfbIsTraceActive == NULL) &&
         (ETG_pfvDoErrMemTrace == NULL) &&
         (ETG_pfs32RegTraceCallback == NULL) &&
         (ETG_pfs32UnregTraceCallback == NULL) &&
         (ETG_pfs32RegisterEtgICmds == NULL) &&
         (ETG_pfs32DeregisterEtgICmds == NULL)){

      ETG_pfvDoTrace       = TRACE_ETG_DoTrace;
      ETG_pfbIsTraceActive = TRACE_ETG_bIsTraceActive;
      ETG_pfvDoErrMemTrace = lldErrMem_ETG_WriteFatalTrace;
      ETG_pfs32RegTraceCallback = TRACE_ETG_s32RegTraceCallback;
      ETG_pfs32UnregTraceCallback = TRACE_ETG_s32UnregTraceCallback;
      ETG_pfs32RegisterEtgICmds = etgIF_s32RegisterEtgICmds;
      ETG_pfs32DeregisterEtgICmds = etgIF_s32DeregisterEtgICmds;

   } /*if((ETG_pfvDoTrace       == NULL) && */
   
   return;
}

/**************************************************************************
* FUNCTION       vInitPlatformEtgAppId
* EXPLANATION    Init ETG Function pointer and DLT AppId
* PARAMETERS     onst char* dltAppId, const char* description
* RETURNS        NONE
***************************************************************************/
void vInitPlatformEtgAppId(const char* dltAppId, const char* description)
{
   (void)dltAppId;
   (void)description;
   /* in case on adit trace no use of app id */
   vInitPlatformEtg();
}


/**************************************************************************
* FUNCTION       vInitPlatformEtgAppIdOverrideIF
* EXPLANATION    Init ETG Function pointer and DLT AppId and overiding
*                and reset the backend function pointers
* PARAMETERS     onst char* dltAppId, const char* description
* RETURNS        NONE
***************************************************************************/
void vInitPlatformEtgAppIdOverrideIF(const char* dltAppId, const char* description)
{
   (void)dltAppId;
   (void)description;
   /* in case on adit trace no use of app id */
   vInitPlatformEtg();
}
