#ifndef _ET_ETG_H
#define _ET_ETG_H
/******************************************************************************
FILE:          etg.h
SW-COMPONENT:  general library for ETG (extended trace generator)

DESCRIPTION:   interface to use generated traces
Features:      VARIANT_S_FTR_ENABLE_TRC_GEN
                  General activation of ETG. The macros are translated into
                  certain hashes the strings are generated into a trcgen.tro
               VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING
                  Generate all macros with sprintf() (strings only) into the
                  generated header. The decoding rule in TTFis is simply the
                  string output.
               VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING_PURE
                  Same as before but even no header bytes for TTFis reception.
               VARIANT_S_FTR_ENABLE_ETG_PRINTF
                  Replace all macros by calls to printf()
               VARIANT_S_FTR_ENABLE_ETG_QTLOG
                  Replace all macros by calls to Qt specific log functions
               VARIANT_S_FTR_ENABLE_ETG_OSAL_THRCTRLBLOCK
                  Support OSAL based names for threads.
               VARIANT_S_FTR_ENABLE_ETG_ERRMEM_LF
                  Support for extra line feed at end of errmem messages.
               VARIANT_S_FTR_ENABLE_TRC_GEN_FORCE_UTF8
                  Support UTF8 encoded string decoding at log host enabled. (required for TTFis)
                  This feature is enabled by default, on current buildproduct environments.
                  Last is done with bracing macro ETG_UTF8(text) which makes this particular text decoded as utf8 only.
                  Supports enabling utf8 decoding everywhere; even in files w/o the UTF8 macro or variant switch being set
               VARIANT_S_FTR_ENABLE_TRC_GEN_DISABLE_UTF8
                  Supports in opposite to disable UTF8 decoding in all places except where ETG_UTF8() is used.
                  ETG_NO_UTF8() macro can be used to overrule the general enablement for only one statement.
               VARIANT_S_FTR_ENABLE_ETG_DLT_TRACING
                  Support fo DLT backend in the given process.
                  This switch enables dlt based string tracing ad requires linking against static lib trace_etg_wrapper-dlt.

Trace macro features:
               LVL       variable parameter for trace level
               DCL       variable parameter for double trace class
               CLS       for explicit trace class different to default
               THR       threading info to be added
               ERRMEM    trace data into error memory
               levels    ERRMEM, FATAL, ERR,  SYS_MIN, SYS,  COMP, USR1, USR2, USR3,  USR4
                         0+errmem. 0,   1,    2,       3,    4,    5,    6,    7,     8
               DLT-Lvls  OFF,    FATAL, ERROR,         WARN, INFO,             DEBUG, VERBOSE
                         0,      1,     2,             3,    4,                5,     6
               ETG_ENUM  printing a enum defined as ARRAY in trc file
               ETG_CENUM printing a cpp/h defined enum
               ETG_LIST  printing of an array of values (not in printf), (ETG_LIST_LEN and ETG_LIST_PTR_Tx are deprecated)
               ETG_NO_UTF8 and ETG_UTF8 flagging a particular parameter as ascii 8-bit encoded string or as utf8(default)
               ETG_INIT_PLATFORM()
                         This macros is to be called ONCE at begin of spawning a process to connect trace ETG frontend with
                         The DLT App ID will be set to default "HLOG"
                         platform specific ETG backend.
                         does not return a value
               ETG_INIT_PLATFORM_APPID(sDltAppId, sDescription)
                         Same as before but additionally setting the DLT Application ID for this process.
                         Additional description text can explain details for the application.
                         This is typically used in DLT but not in Adit trace and TTFis.
                         Both macros will work in any configuration.

Trace command input support ETG_I:
               ETG_I_REGISTER_CHN(pfDefProc) ETG_I_UNREGISTER_CHN()
                         Registering th trace channel (once per startup)
               ETG_I_REGISTER_FILE() ETG_I_UNREGISTER_FILE()
                         Adding this file into the command list chain (once or more) e.g.
                         in constructor.
               ETG_I_CMD_DEFINE(func, "scanf_string", parameter type descriptions)
                         Creating a trace command being connected to a given function.
               ETG_I_ENUM(a) ETG_I_CENUM(a) ETG_I_STRING ETG_I_ARRAY_U8 ETG_I_CONST_ARG
                         Special parameter type descriptors.
               ETG_I_bPROCESS(u8TtfisCmdBuf)
                         Optional call back handler for self defined threading support.

AUTHOR:        XC-CI1/ECA Peter Hassmann, Matthias Hessling
COPYRIGHT:     (c) 2010-2021 Robert Bosch Car Multimedia GmbH

HISTORY:
Date     | Rev. | Author                  | Modification
----------------------------------------------------------------------------
09-09-10 | 1.0  | CM-AI/PJ-VW36 Hassmann  | Initial revision
11-02-11 | 2.0  | CM-AI/PJ-VW36 Hessling  | Using function pointer to trace
11-04-11 | 2.1  | CM-AI/PJ-VW36 Hessling  | Adding errmem function pointer
24-09-12 | 2.2  | CM-AI/PJ-CB36 Hessling  | Adding ETG_I functions
19-12-13 | 2.3  | CM-AI/PJ-CB36 Hessling  | Make ETG independent from OSAL
01-03-14 | 2.4  | CM-AI/ECB4 Hessling     | Make ETG independent from trace.h
26-03-18 | 2.5  | CM-CI1/ESW4 Hessling    | Adding 64 bit / UTF8 and Qt tracing
16-04-21 | 2.6  | XC-CI1/ECA4 Hessling    | Adding DLT support (w/o) ttfis-plugin
******************************************************************************/
#include "etg_types.h" // ETG internal OSAL replacement
#if defined(VARIANT_S_FTR_ENABLE_ETG_QTLOG)
#include <QtCore/QDebug>
#endif


#define ETG_IF_VER_MAJ 2 // v2 was introduced with 64-bit support. several components cannot build due to strengthen the compiler check on register size
#define ETG_IF_VER_MIN 1

#ifdef VARIANT_S_FTR_ENABLE_ETG_DLT_TRACING
#ifndef VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING_PURE
#define VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING_PURE
#endif
#endif


#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING_PURE
// Add no "0xFF" after the trace-class to the ttfis-output
#ifndef VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING
#define VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING
#endif
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN_STRING_TRACING
#ifndef VARIANT_S_FTR_ENABLE_TRC_GEN
#define VARIANT_S_FTR_ENABLE_TRC_GEN
#endif
#endif

#if defined (VARIANT_S_FTR_ENABLE_ETG_PRINTF) && defined (VARIANT_S_FTR_ENABLE_TRC_GEN)
// _ETG_PRINTF is exclusive to TRC_GEN
#undef VARIANT_S_FTR_ENABLE_TRC_GEN
#endif

#if defined (VARIANT_S_FTR_ENABLE_ETG_QTLOG) && defined (VARIANT_S_FTR_ENABLE_TRC_GEN)
// _ETG_QTLOG is exclusive to TRC_GEN
#undef VARIANT_S_FTR_ENABLE_TRC_GEN
#endif

// define trace level names as typically used in DLT and map them to adit trace levels
// DLT macros ...WARN and ...DEBUG are mapped to the most verbose level from adit trace levels
// keep this is sync with trc_gen.pl and etg_linux_platform.c
#define ETG_TRACE_WARN             ETG_TRACE_SYS
#define ETG_TRACE_WARN_CLS         ETG_TRACE_SYS_CLS
#define ETG_TRACE_WARN_THR         ETG_TRACE_SYS_THR
#define ETG_TRACE_WARN_CLS_THR     ETG_TRACE_SYS_CLS_THR
#define ETG_TRACE_WARN_DCL         ETG_TRACE_SYS_DCL
#define ETG_TRACE_WARN_CLS_DCL     ETG_TRACE_SYS_CLS_DCL
#define ETG_TRACE_WARN_THR_DCL     ETG_TRACE_SYS_THR_DCL
#define ETG_TRACE_WARN_CLS_THR_DCL ETG_TRACE_SYS_CLS_THR_DCL

#define ETG_TRACE_INFO             ETG_TRACE_COMP
#define ETG_TRACE_INFO_CLS         ETG_TRACE_COMP_CLS
#define ETG_TRACE_INFO_THR         ETG_TRACE_COMP_THR
#define ETG_TRACE_INFO_CLS_THR     ETG_TRACE_COMP_CLS_THR
#define ETG_TRACE_INFO_DCL         ETG_TRACE_COMP_DCL
#define ETG_TRACE_INFO_CLS_DCL     ETG_TRACE_COMP_CLS_DCL
#define ETG_TRACE_INFO_THR_DCL     ETG_TRACE_COMP_THR_DCL
#define ETG_TRACE_INFO_CLS_THR_DCL ETG_TRACE_COMP_CLS_THR_DCL

#define ETG_TRACE_DEBUG             ETG_TRACE_USR3
#define ETG_TRACE_DEBUG_CLS         ETG_TRACE_USR3_CLS
#define ETG_TRACE_DEBUG_THR         ETG_TRACE_USR3_THR
#define ETG_TRACE_DEBUG_CLS_THR     ETG_TRACE_USR3_CLS_THR
#define ETG_TRACE_DEBUG_DCL         ETG_TRACE_USR3_DCL
#define ETG_TRACE_DEBUG_CLS_DCL     ETG_TRACE_USR3_CLS_DCL
#define ETG_TRACE_DEBUG_THR_DCL     ETG_TRACE_USR3_THR_DCL
#define ETG_TRACE_DEBUG_CLS_THR_DCL ETG_TRACE_USR3_CLS_THR_DCL

#define ETG_TRACE_VERBOSE             ETG_TRACE_USR4
#define ETG_TRACE_VERBOSE_CLS         ETG_TRACE_USR4_CLS
#define ETG_TRACE_VERBOSE_THR         ETG_TRACE_USR4_THR
#define ETG_TRACE_VERBOSE_CLS_THR     ETG_TRACE_USR4_CLS_THR
#define ETG_TRACE_VERBOSE_DCL         ETG_TRACE_USR4_DCL
#define ETG_TRACE_VERBOSE_CLS_DCL     ETG_TRACE_USR4_CLS_DCL
#define ETG_TRACE_VERBOSE_THR_DCL     ETG_TRACE_USR4_THR_DCL
#define ETG_TRACE_VERBOSE_CLS_THR_DCL ETG_TRACE_USR4_CLS_THR_DCL

#ifdef __cplusplus
extern "C"
{
#endif

#define ETG_LEVEL_ERRMEM 0x80 // we use an additional flag to signal error memory tracing

   typedef etg_tVoid (*ETG_tpfvDoTrace)(etg_tU16 u16TraceClass, etg_tU16 enTraceLevel, const etg_tS8* pcos8Buffer, etg_tU32 u32Length);
   typedef etg_tBool (*ETG_tpfbIsTraceActive)(etg_tU16 u16TraceClass, etg_tU16 enTraceLevel);
   // tenErrmemEntryType cannot be used as second parameter because having a ring dependency with osioctrl.h
   typedef etg_tVoid (*ETG_tpfvDoErrMemTrace)(etg_tU16 u16Entry, etg_tInt eEntryType, etg_tU16 u16EntryLength, const etg_tU8* pcou8Buffer);
   // this is the callback register function type to be defined by using component
   typedef etg_tS32 (*ETG_tpfs32RegTraceCallback)(etg_tU16 eChanId, ETG_tpfCallback pfvCb);
   // this is the callback un-register function type to be defined by using component
   typedef etg_tS32 (*ETG_tpfs32UnregTraceCallback)(etg_tU16 eChanId, ETG_tpfCallback pfvCb);
   // this is to support DLT mechanism to set four-char application identifier
   typedef etg_tS32 (*ETG_tpfs32SetEtgDltAppId)(const etg_tChar* dltAppId, const etg_tChar* description);

#ifndef ETG_IMPLEMENTS
   extern ETG_tpfvDoTrace ETG_pfvDoTrace;
   extern ETG_tpfbIsTraceActive ETG_pfbIsTraceActive;
   extern ETG_tpfvDoErrMemTrace ETG_pfvDoErrMemTrace;
   extern ETG_tpfs32RegTraceCallback ETG_pfs32RegTraceCallback;
   extern ETG_tpfs32UnregTraceCallback ETG_pfs32UnregTraceCallback;
   extern ETG_tpfs32SetEtgDltAppId ETG_pfs32SetEtgDltAppId;
#else
   ETG_tpfvDoTrace ETG_pfvDoTrace = NULL;
   ETG_tpfbIsTraceActive ETG_pfbIsTraceActive = NULL;
   ETG_tpfvDoErrMemTrace ETG_pfvDoErrMemTrace = NULL;
   ETG_tpfs32RegTraceCallback ETG_pfs32RegTraceCallback = NULL;
   ETG_tpfs32UnregTraceCallback ETG_pfs32UnregTraceCallback = NULL;
   ETG_tpfs32SetEtgDltAppId ETG_pfs32SetEtgDltAppId = NULL;
#endif

   /* macros to be used inside ETG_TRACE_XXX(generatedId, pcBuffer) */


   /* use ttfis utf8 string */
#define ETG_UTF8(utf8StringVar) utf8StringVar 

   /* Trace an enumeration, ttfisEnumName as in trc-file */
#define ETG_ENUM(ttfisEnumName,u8Val) u8Val
   /* Trace an enumeration, cEnumName as in declaration
   1.)
   typedef enum TAG? {
   ...
   } typeName;
   ---> use typeName as cEnumName

   2.) enum TAG {
   ...
   } varName?;
   ---> use TAG as cEnumName

   3.) enum {

   } varName;
   ---> use varName as cEnumName
   */
#define ETG_CENUM(cEnumName,u8Val) (u8Val)
   /* Trace a string-literal, the string will only be generated into the trc-file */
   /* If the List is not last argument of the trace-statement, its length must be a number (eg. 4 or 0xd) */



   typedef enum {
      ETG_TSTRING=0,
      ETG_T8=1,
      ETG_T16=2,
      ETG_T32=3,
      ETG_T64=4,
   } etg_tenListType;


   typedef struct  {
      etg_tU16 listLen;
      etg_tenListType listType;
      void const *listPtr;
   } etg_ListParams;

   // format-info is store as stacked bit list
   enum etg_enTraceType
   {
      ETG_EN_DONE = 0,
      ETG_EN_T8 = 1,
      ETG_EN_T16 = 2,
      ETG_EN_T32 = 3,
      ETG_EN_T64 = 4,
      ETG_EN_STRING = 5,
      ETG_EN_LIST = 6
   };
   // bit indicating that thread-info shall be traced
   #define ETG_THR_IND_MASK 0x40000000
   // bit indicating that more formats follow
   #define ETG_FORMAT_MORE_IND_MASK 0x80000000
   // maximum number of vargs (a string in the middle of a trace use 2 args)
   #define ETG_MAX_FORMATS 100
   // bit-mask to get ETG_NUM_FORMAT_BITS bits.
   #define ETG_FORMAT_MASK 7
   // number of bits used for coding format-info according to etg_enTraceType
   #define ETG_NUM_FORMAT_BITS 3

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#ifdef ETG_LIST
#error
#endif
#if defined(__cplusplus) || defined(c_plusplus) || !defined(_MSC_VER)
   #define ETG_INLINE inline
#else
   // needed, because VS2013 does not fully support C99 standard
   #define ETG_INLINE _inline
#endif

   // new format
      static ETG_INLINE etg_ListParams ETG_LIST(etg_tU32 listLen, etg_tenListType listType, void const *listPtr) {
         etg_ListParams res;
         res.listLen=(etg_tU16)(listLen);
         res.listType=listType;
         res.listPtr=listPtr;
         return res;
      }
   // deprecated format ETG_LIST_LEN, ETG_LIST_PTR
#define ETG_LIST_LEN(u16Entries) ETG_LIST(u16Entries
#define ETG_LIST_PTR_T8(pt8PointerToValues)   ETG_T8, (etg_tPCU8)pt8PointerToValues)
#define ETG_LIST_PTR_T16(pt16PointerToValues) ETG_T16, (etg_tPCU16)pt16PointerToValues)
#define ETG_LIST_PTR_T32(pt32PointerToValues) ETG_T32, (etg_tPCU32)pt32PointerToValues)
#define ETG_LIST_PTR_T64(pt64PointerToValues) ETG_T64, (etg_tPCU64)pt64PointerToValues)

#else
#define ETG_LIST(u16Entries, listType, listPtr) *(listPtr)[0]
   // deprecated format ETG_LIST_LEN, ETG_LIST_PTR
#define ETG_LIST_LEN(u16Entries) (u16Entries)
#define ETG_LIST_PTR_T8(pt8PointerToValues)   ((etg_tPCU8)pt8PointerToValues)
#define ETG_LIST_PTR_T16(pt16PointerToValues) ((etg_tPCU16)pt16PointerToValues)
#define ETG_LIST_PTR_T32(pt32PointerToValues) ((etg_tPCU32)pt32PointerToValues)
#define ETG_LIST_PTR_T64(pt64PointerToValues) ((etg_tPCU64)pt64PointerToValues)

#endif

   /* Maximum of chars per trace message */
#define ETG_C_TRACE_MAX 239

   /* Interface of generated trace-macros: */
   /*
   User-Interface for following macros of level 0-8:
   With double braces!!!
   ETG_TRACE_XXX((format, ...))
   ETG_TRACE_XXX_CLS((traceClass, format, ...))
   ETG_TRACE_XXX_THR((format, ...))
   ETG_TRACE_XXX_CLS_THR((traceClass, format, ...))
   */

   /* block for generator function */
#if defined(VARIANT_S_FTR_ENABLE_TRC_GEN)
#define ETG_INIT_PLATFORM() vInitPlatformEtg()
#define ETG_INIT_PLATFORM_APPID(sDltAppId, sDescription) vInitPlatformEtgAppId(sDltAppId, sDescription)

#define ETG_TRACE_LVL                 _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_CLS             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_THR             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_CLS_THR         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_DCL             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_CLS_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_THR_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_LVL_CLS_THR_DCL     _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_ERRMEM               _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_CLS           _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_THR           _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_CLS_THR       _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_DCL           _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_CLS_DCL       _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_THR_DCL       _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERRMEM_CLS_THR_DCL   _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_FATAL               _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_CLS           _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_THR           _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_CLS_THR       _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_DCL           _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_CLS_DCL       _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_THR_DCL       _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_FATAL_CLS_THR_DCL   _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_ERR                 _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_CLS             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_THR             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_CLS_THR         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_DCL             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_CLS_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_THR_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_ERR_CLS_THR_DCL     _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_SYS_MIN             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_CLS         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_THR         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_CLS_THR     _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_CLS_DCL     _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_THR_DCL     _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_MIN_CLS_THR_DCL _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_SYS                 _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_CLS             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_THR             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_CLS_THR         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_DCL             _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_CLS_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_THR_DCL         _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_SYS_CLS_THR_DCL     _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_COMP                _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_CLS            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_THR            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_CLS_THR        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_DCL            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_CLS_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_THR_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_COMP_CLS_THR_DCL    _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_USR1                _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_CLS            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_THR            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_CLS_THR        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_DCL            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_CLS_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_THR_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR1_CLS_THR_DCL    _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_USR2                _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_CLS            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_THR            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_CLS_THR        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_DCL            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_CLS_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_THR_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR2_CLS_THR_DCL    _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_USR3                _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_CLS            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_THR            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_CLS_THR        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_DCL            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_CLS_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_THR_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR3_CLS_THR_DCL    _ETG_TRACE_GEN(__LINE__)

#define ETG_TRACE_USR4                _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_CLS            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_THR            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_CLS_THR        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_DCL            _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_CLS_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_THR_DCL        _ETG_TRACE_GEN(__LINE__)
#define ETG_TRACE_USR4_CLS_THR_DCL    _ETG_TRACE_GEN(__LINE__)

#define _ETG_TRACE_GEN(_LINE_) _ETG_TRACE_GEN_(_LINE_)
#define _ETG_TRACE_GEN_(_LINE_) _ETG_TRACE_GEN_ ## _LINE_

/* block for printf function w/o generator */
#elif defined(VARIANT_S_FTR_ENABLE_ETG_PRINTF)

#define ETG_INIT_PLATFORM()
#define ETG_INIT_PLATFORM_APPID(sDltAppId, sDescription)

#ifdef ETG_DEFAULT_TRACE_CLASS
#define __ETG_TRACE_HEADER(lvl) printf("lvl:%d,cls:0x%04X: ",lvl,ETG_DEFAULT_TRACE_CLASS);
#else
#define __ETG_TRACE_HEADER(lvl) printf("lvl:%d: ",lvl);
#endif

#define __ETG_TRACE_FOOTER printf(" [%s(%d)]\n",__FILE__,__LINE__)
#define _ETG_TRACE_GEN(lvl,format) do {__ETG_TRACE_HEADER(lvl) printf format; __ETG_TRACE_FOOTER;} while (0);

#define ETG_TRACE_LVL(off)
#define ETG_TRACE_LVL_CLS(off)
#define ETG_TRACE_LVL_THR(off)
#define ETG_TRACE_LVL_CLS_THR(off)
#define ETG_TRACE_LVL_DCL(off)
#define ETG_TRACE_LVL_CLS_DCL(off)
#define ETG_TRACE_LVL_THR_DCL(off)
#define ETG_TRACE_LVL_CLS_THR_DCL(off)

#define ETG_TRACE_ERRMEM(format)  _ETG_TRACE_GEN(0,format)
#define ETG_TRACE_FATAL(format)   _ETG_TRACE_GEN(0,format)
#define ETG_TRACE_ERR(format)     _ETG_TRACE_GEN(1,format)
#define ETG_TRACE_SYS_MIN(format) _ETG_TRACE_GEN(2,format)
#define ETG_TRACE_SYS(format)     _ETG_TRACE_GEN(3,format)
#define ETG_TRACE_COMP(format)    _ETG_TRACE_GEN(4,format)
#define ETG_TRACE_USR1(format)    _ETG_TRACE_GEN(5,format)
#define ETG_TRACE_USR2(format)    _ETG_TRACE_GEN(6,format)
#define ETG_TRACE_USR3(format)    _ETG_TRACE_GEN(7,format)
#define ETG_TRACE_USR4(format)    _ETG_TRACE_GEN(8,format)

#define ETG_TRACE_ERRMEM_THR(format)  _ETG_TRACE_GEN(0,format)
#define ETG_TRACE_FATAL_THR(format)   _ETG_TRACE_GEN(0,format)
#define ETG_TRACE_ERR_THR(format)     _ETG_TRACE_GEN(1,format)
#define ETG_TRACE_SYS_MIN_THR(format) _ETG_TRACE_GEN(2,format)
#define ETG_TRACE_SYS_THR(format)     _ETG_TRACE_GEN(3,format)
#define ETG_TRACE_COMP_THR(format)    _ETG_TRACE_GEN(4,format)
#define ETG_TRACE_USR1_THR(format)    _ETG_TRACE_GEN(5,format)
#define ETG_TRACE_USR2_THR(format)    _ETG_TRACE_GEN(6,format)
#define ETG_TRACE_USR3_THR(format)    _ETG_TRACE_GEN(7,format)
#define ETG_TRACE_USR4_THR(format)    _ETG_TRACE_GEN(8,format)

#define ETG_TRACE_ERRMEM_CLS(off)
#define ETG_TRACE_ERRMEM_CLS_THR(off)
#define ETG_TRACE_ERRMEM_DCL(off)
#define ETG_TRACE_ERRMEM_CLS_DCL(off)
#define ETG_TRACE_ERRMEM_THR_DCL(off)
#define ETG_TRACE_ERRMEM_CLS_DCL(off)
#define ETG_TRACE_ERRMEM_CLS_THR_DCL(off)

#define ETG_TRACE_FATAL_CLS(off)
#define ETG_TRACE_FATAL_CLS_THR(off)
#define ETG_TRACE_FATAL_DCL(off)
#define ETG_TRACE_FATAL_CLS_DCL(off)
#define ETG_TRACE_FATAL_THR_DCL(off)
#define ETG_TRACE_FATAL_CLS_DCL(off)
#define ETG_TRACE_FATAL_CLS_THR_DCL(off)

#define ETG_TRACE_ERR_CLS(off)
#define ETG_TRACE_ERR_CLS_THR(off)
#define ETG_TRACE_ERR_DCL(off)
#define ETG_TRACE_ERR_CLS_DCL(off)
#define ETG_TRACE_ERR_THR_DCL(off)
#define ETG_TRACE_ERR_CLS_DCL(off)
#define ETG_TRACE_ERR_CLS_THR_DCL(off)

#define ETG_TRACE_SYS_MIN_CLS(off)
#define ETG_TRACE_SYS_MIN_CLS_THR(off)
#define ETG_TRACE_SYS_MIN_DCL(off)
#define ETG_TRACE_SYS_MIN_CLS_DCL(off)
#define ETG_TRACE_SYS_MIN_THR_DCL(off)
#define ETG_TRACE_SYS_MIN_CLS_DCL(off)
#define ETG_TRACE_SYS_MIN_CLS_THR_DCL(off)

#define ETG_TRACE_SYS_CLS(off)
#define ETG_TRACE_SYS_CLS_THR(off)
#define ETG_TRACE_SYS_DCL(off)
#define ETG_TRACE_SYS_CLS_DCL(off)
#define ETG_TRACE_SYS_THR_DCL(off)
#define ETG_TRACE_SYS_CLS_DCL(off)
#define ETG_TRACE_SYS_CLS_THR_DCL(off)

#define ETG_TRACE_COMP_CLS(off)
#define ETG_TRACE_COMP_CLS_THR(off)
#define ETG_TRACE_COMP_DCL(off)
#define ETG_TRACE_COMP_CLS_DCL(off)
#define ETG_TRACE_COMP_THR_DCL(off)
#define ETG_TRACE_COMP_CLS_DCL(off)
#define ETG_TRACE_COMP_CLS_THR_DCL(off)

#define ETG_TRACE_USR1_CLS(off)
#define ETG_TRACE_USR1_CLS_THR(off)
#define ETG_TRACE_USR1_DCL(off)
#define ETG_TRACE_USR1_CLS_DCL(off)
#define ETG_TRACE_USR1_THR_DCL(off)
#define ETG_TRACE_USR1_CLS_DCL(off)
#define ETG_TRACE_USR1_CLS_THR_DCL(off)

#define ETG_TRACE_USR2_CLS(off)
#define ETG_TRACE_USR2_CLS_THR(off)
#define ETG_TRACE_USR2_DCL(off)
#define ETG_TRACE_USR2_CLS_DCL(off)
#define ETG_TRACE_USR2_THR_DCL(off)
#define ETG_TRACE_USR2_CLS_DCL(off)
#define ETG_TRACE_USR2_CLS_THR_DCL(off)

#define ETG_TRACE_USR3_CLS(off)
#define ETG_TRACE_USR3_CLS_THR(off)
#define ETG_TRACE_USR3_DCL(off)
#define ETG_TRACE_USR3_CLS_DCL(off)
#define ETG_TRACE_USR3_THR_DCL(off)
#define ETG_TRACE_USR3_CLS_DCL(off)
#define ETG_TRACE_USR3_CLS_THR_DCL(off)

#define ETG_TRACE_USR4_CLS(off)
#define ETG_TRACE_USR4_CLS_THR(off)
#define ETG_TRACE_USR4_DCL(off)
#define ETG_TRACE_USR4_CLS_DCL(off)
#define ETG_TRACE_USR4_THR_DCL(off)
#define ETG_TRACE_USR4_CLS_DCL(off)
#define ETG_TRACE_USR4_CLS_THR_DCL(off)

#elif defined(VARIANT_S_FTR_ENABLE_ETG_QTLOG)
// redirect etg traces to qDebug, QtCreators log window without any filter / trc file
#define ETG_INIT_PLATFORM()
#define ETG_INIT_PLATFORM_APPID(sDltAppId, sDescription)

#define _ETG_TRACE_STD__DEBUG(args, ...)               qDebug(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_STD__INFO(args, ...)                qInfo(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_STD__WARN(args, ...)                qWarning(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_STD__FATAL(args, ...)               qFatal(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_STD__ERROR(args, ...)               qCritical(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_DCL__DEBUG(TraceClass,args, ...)    qDebug(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_DCL__INFO(TraceClass,args, ...)     qInfo(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_DCL__WARN(TraceClass,args, ...)     qWarning(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_DCL__ERROR(TraceClass,args, ...)    qCritical(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_DCL__FATAL(TraceClass,args, ...)    qFatal(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_CLS__DEBUG(TraceClass,args, ...)    qDebug(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_CLS__INFO(TraceClass,args, ...)     qInfo(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_CLS__WARN(TraceClass,args, ...)     qWarning(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_CLS__ERROR(TraceClass,args, ...)    qCritical(args "\n", ##__VA_ARGS__)
#define _ETG_TRACE_CLS__FATAL(TraceClass,args, ...)    qFatal(args "\n", ##__VA_ARGS__)


#define ETG_TRACE_ERRMEM(printfArgs)                                             _ETG_TRACE_STD__FATAL printfArgs
#define ETG_TRACE_ERRMEM_CLS(u16TraceClass_PrintfArgs)                           _ETG_TRACE_CLS__FATAL u16TraceClass_PrintfArgs
#define ETG_TRACE_ERRMEM_DCL(u16DerivedClass_printfArgs)                         _ETG_TRACE_DCL__FATAL u16DerivedClass_printfArgs
#define ETG_TRACE_ERRMEM_CLS_DCL(printfArgs)                                     _ETG_TRACE_CLS_DCL__FATAL printfArgs
#define ETG_TRACE_ERRMEM_CLS_THR(u16TraceClass_PrintfArgs)                       _ETG_TRACE_CLS__FATAL u16TraceClass_PrintfArgs
#define ETG_TRACE_ERRMEM_CLS_THR_DCL(printfArgs)                                 _ETG_TRACE_CLS_DCL__FATAL printfArgs
#define ETG_TRACE_ERRMEM_THR(printfArgs)                                         _ETG_TRACE_STD__FATAL printfArgs
#define ETG_TRACE_ERRMEM_THR_DCL(u16DerivedClass_printfArgs)                     _ETG_TRACE_DCL__FATAL u16DerivedClass_printfArgs

#define ETG_TRACE_FATAL(printfArgs)                                              _ETG_TRACE_STD__FATAL printfArgs
#define ETG_TRACE_FATAL_CLS(u16TraceClass_PrintfArgs)                            _ETG_TRACE_CLS__FATAL u16TraceClass_PrintfArgs
#define ETG_TRACE_FATAL_DCL(u16DerivedClass_printfArgs)                          _ETG_TRACE_DCL__FATAL u16DerivedClass_printfArgs
#define ETG_TRACE_FATAL_CLS_DCL(printfArgs)                                      _ETG_TRACE_CLS_DCL__FATAL printfArgs
#define ETG_TRACE_FATAL_CLS_THR(u16TraceClass_PrintfArgs)                        _ETG_TRACE_CLS__FATAL u16TraceClass_PrintfArgs
#define ETG_TRACE_FATAL_CLS_THR_DCL(printfArgs)                                  _ETG_TRACE_CLS_DCL__FATAL printfArgs
#define ETG_TRACE_FATAL_THR(printfArgs)                                          _ETG_TRACE_STD__FATAL printfArgs
#define ETG_TRACE_FATAL_THR_DCL(u16DerivedClass_printfArgs)                      _ETG_TRACE_DCL__FATAL u16DerivedClass_printfArgs

#define ETG_TRACE_ERR(printfArgs)                                                _ETG_TRACE_STD__ERROR printfArgs
#define ETG_TRACE_ERR_CLS(u16TraceClass_PrintfArgs)                              _ETG_TRACE_CLS__ERROR u16TraceClass_PrintfArgs
#define ETG_TRACE_ERR_DCL(u16DerivedClass_printfArgs)                            _ETG_TRACE_DCL__ERROR u16DerivedClass_printfArgs
#define ETG_TRACE_ERR_CLS_DCL(printfArgs)                                        _ETG_TRACE_CLS_DCL__ERROR printfArgs
#define ETG_TRACE_ERR_CLS_THR(u16TraceClass_PrintfArgs)                          _ETG_TRACE_CLS__ERROR u16TraceClass_PrintfArgs
#define ETG_TRACE_ERR_CLS_THR_DCL(printfArgs)                                    _ETG_TRACE_CLS_DCL__ERROR printfArgs
#define ETG_TRACE_ERR_THR(printfArgs)                                            _ETG_TRACE_STD__ERROR printfArgs
#define ETG_TRACE_ERR_THR_DCL(u16DerivedClass_printfArgs)                        _ETG_TRACE_DCL__ERROR u16DerivedClass_printfArgs

#define ETG_TRACE_SYS_MIN(printfArgs)                                            _ETG_TRACE_STD__WARN printfArgs
#define ETG_TRACE_SYS_MIN_CLS(u16TraceClass_PrintfArgs)                          _ETG_TRACE_CLS__WARN u16TraceClass_PrintfArgs
#define ETG_TRACE_SYS_MIN_DCL(u16DerivedClass_printfArgs)                        _ETG_TRACE_DCL__WARN u16DerivedClass_printfArgs
#define ETG_TRACE_SYS_MIN_CLS_DCL(printfArgs)                                    _ETG_TRACE_CLS_DCL__WARN printfArgs
#define ETG_TRACE_SYS_MIN_CLS_THR(u16TraceClass_PrintfArgs)                      _ETG_TRACE_CLS__WARN u16TraceClass_PrintfArgs
#define ETG_TRACE_SYS_MIN_CLS_THR_DCL(printfArgs)                                _ETG_TRACE_CLS_DCL__WARN printfArgs
#define ETG_TRACE_SYS_MIN_THR(printfArgs)                                        _ETG_TRACE_STD__WARN printfArgs
#define ETG_TRACE_SYS_MIN_THR_DCL(u16DerivedClass_printfArgs)                    _ETG_TRACE_DCL__WARN u16DerivedClass_printfArgs

#define ETG_TRACE_SYS(printfArgs)                                                _ETG_TRACE_STD__WARN printfArgs
#define ETG_TRACE_SYS_CLS(u16TraceClass_PrintfArgs)                              _ETG_TRACE_CLS__WARN u16TraceClass_PrintfArgs
#define ETG_TRACE_SYS_DCL(u16DerivedClass_printfArgs)                            _ETG_TRACE_DCL__WARN u16DerivedClass_printfArgs
#define ETG_TRACE_SYS_CLS_DCL(printfArgs)                                        _ETG_TRACE_CLS_DCL__WARN printfArgs
#define ETG_TRACE_SYS_CLS_THR(u16TraceClass_PrintfArgs)                          _ETG_TRACE_CLS__WARN u16TraceClass_PrintfArgs
#define ETG_TRACE_SYS_CLS_THR_DCL(printfArgs)                                    _ETG_TRACE_CLS_DCL__WARN printfArgs
#define ETG_TRACE_SYS_THR(printfArgs)                                            _ETG_TRACE_STD__WARN printfArgs
#define ETG_TRACE_SYS_THR_DCL(u16DerivedClass_printfArgs)                        _ETG_TRACE_DCL__WARN u16DerivedClass_printfArgs

#define ETG_TRACE_COMP(printfArgs)                                               _ETG_TRACE_STD__WARN printfArgs
#define ETG_TRACE_COMP_CLS(u16TraceClass_PrintfArgs)                             _ETG_TRACE_CLS__WARN u16TraceClass_PrintfArgs
#define ETG_TRACE_COMP_DCL(u16DerivedClass_printfArgs)                           _ETG_TRACE_DCL__WARN u16DerivedClass_printfArgs
#define ETG_TRACE_COMP_CLS_DCL(printfArgs)                                       _ETG_TRACE_CLS_DCL__WARN printfArgs
#define ETG_TRACE_COMP_CLS_THR(u16TraceClass_PrintfArgs)                         _ETG_TRACE_CLS__WARN u16TraceClass_PrintfArgs
#define ETG_TRACE_COMP_CLS_THR_DCL(printfArgs)                                   _ETG_TRACE_CLS_DCL__WARN printfArgs
#define ETG_TRACE_COMP_THR(printfArgs)                                           _ETG_TRACE_STD__WARN printfArgs
#define ETG_TRACE_COMP_THR_DCL(u16DerivedClass_printfArgs)                       _ETG_TRACE_DCL__WARN u16DerivedClass_printfArgs

#define ETG_TRACE_USR1(printfArgs)                                               _ETG_TRACE_STD__INFO printfArgs
#define ETG_TRACE_USR1_CLS(u16TraceClass_PrintfArgs)                             _ETG_TRACE_CLS__INFO u16TraceClass_PrintfArgs
#define ETG_TRACE_USR1_DCL(u16DerivedClass_printfArgs)                           _ETG_TRACE_DCL__INFO u16DerivedClass_printfArgs
#define ETG_TRACE_USR1_CLS_DCL(printfArgs)                                       _ETG_TRACE_CLS_DCL__INFO printfArgs
#define ETG_TRACE_USR1_CLS_THR(u16TraceClass_PrintfArgs)                         _ETG_TRACE_CLS__INFO u16TraceClass_PrintfArgs
#define ETG_TRACE_USR1_CLS_THR_DCL(printfArgs)                                   _ETG_TRACE_CLS_DCL__INFO printfArgs
#define ETG_TRACE_USR1_THR(printfArgs)                                           _ETG_TRACE_STD__INFO printfArgs
#define ETG_TRACE_USR1_THR_DCL(u16DerivedClass_printfArgs)                       _ETG_TRACE_DCL__INFO u16DerivedClass_printfArgs

#define ETG_TRACE_USR2(printfArgs)                                               _ETG_TRACE_STD__INFO printfArgs
#define ETG_TRACE_USR2_CLS(u16TraceClass_PrintfArgs)                             _ETG_TRACE_CLS__INFO u16TraceClass_PrintfArgs
#define ETG_TRACE_USR2_DCL(u16DerivedClass_printfArgs)                           _ETG_TRACE_DCL__INFO u16DerivedClass_printfArgs
#define ETG_TRACE_USR2_CLS_DCL(printfArgs)                                       _ETG_TRACE_CLS_DCL__INFO printfArgs
#define ETG_TRACE_USR2_CLS_THR(u16TraceClass_PrintfArgs)                         _ETG_TRACE_CLS__INFO u16TraceClass_PrintfArgs
#define ETG_TRACE_USR2_CLS_THR_DCL(printfArgs)                                   _ETG_TRACE_CLS_DCL__INFO printfArgs
#define ETG_TRACE_USR2_THR(printfArgs)                                           _ETG_TRACE_STD__INFO printfArgs
#define ETG_TRACE_USR2_THR_DCL(u16DerivedClass_printfArgs)                       _ETG_TRACE_DCL__INFO u16DerivedClass_printfArgs

#define ETG_TRACE_USR3(printfArgs)                                               _ETG_TRACE_STD__INFO printfArgs
#define ETG_TRACE_USR3_CLS(u16TraceClass_PrintfArgs)                             _ETG_TRACE_CLS__INFO u16TraceClass_PrintfArgs
#define ETG_TRACE_USR3_DCL(u16DerivedClass_printfArgs)                           _ETG_TRACE_DCL__INFO u16DerivedClass_printfArgs
#define ETG_TRACE_USR3_CLS_DCL(printfArgs)                                       _ETG_TRACE_CLS_DCL__INFO printfArgs
#define ETG_TRACE_USR3_CLS_THR(u16TraceClass_PrintfArgs)                         _ETG_TRACE_CLS__INFO u16TraceClass_PrintfArgs
#define ETG_TRACE_USR3_CLS_THR_DCL(printfArgs)                                   _ETG_TRACE_CLS_DCL__INFO printfArgs
#define ETG_TRACE_USR3_THR(printfArgs)                                           _ETG_TRACE_STD__INFO printfArgs
#define ETG_TRACE_USR3_THR_DCL(u16DerivedClass_printfArgs)                       _ETG_TRACE_DCL__INFO u16DerivedClass_printfArgs

#define ETG_TRACE_USR4(printfArgs)                                               _ETG_TRACE_STD__DEBUG printfArgs
#define ETG_TRACE_USR4_CLS(u16TraceClass_PrintfArgs)                             _ETG_TRACE_CLS__DEBUG u16TraceClass_PrintfArgs
#define ETG_TRACE_USR4_DCL(u16DerivedClass_printfArgs)                           _ETG_TRACE_DCL__DEBUG u16DerivedClass_printfArgs
#define ETG_TRACE_USR4_CLS_DCL(printfArgs)                                       _ETG_TRACE_CLS_DCL__DEBUG printfArgs
#define ETG_TRACE_USR4_CLS_THR(u16TraceClass_PrintfArgs)                         _ETG_TRACE_CLS__DEBUG u16TraceClass_PrintfArgs
#define ETG_TRACE_USR4_CLS_THR_DCL(printfArgs)                                   _ETG_TRACE_CLS_DCL__DEBUG printfArgs
#define ETG_TRACE_USR4_THR(printfArgs)                                           _ETG_TRACE_STD__DEBUG printfArgs
#define ETG_TRACE_USR4_THR_DCL(u16DerivedClass_printfArgs)                       _ETG_TRACE_DCL__DEBUG u16DerivedClass_printfArgs

#define ETG_TRACE_LVL(off)
#define ETG_TRACE_LVL_CLS(off)
#define ETG_TRACE_LVL_CLS_DCL(off)
#define ETG_TRACE_LVL_CLS_THR(off)
#define ETG_TRACE_LVL_CLS_THR_DCL(off)
#define ETG_TRACE_LVL_DCL(off)
#define ETG_TRACE_LVL_THR(off)
#define ETG_TRACE_LVL_THR_DCL(off)

   /* block for deactivation of trace macros */
#else // no trc-gen
#define ETG_INIT_PLATFORM()
#define ETG_INIT_PLATFORM_APPID(sDltAppId, sDescription)

#define ETG_TRACE_LVL(printfArgs)
#define ETG_TRACE_LVL_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_LVL_THR(printfArgs)
#define ETG_TRACE_LVL_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_LVL_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_LVL_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_LVL_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_LVL_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_ERRMEM(printfArgs)
#define ETG_TRACE_ERRMEM_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_ERRMEM_THR(printfArgs)
#define ETG_TRACE_ERRMEM_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_ERRMEM_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_ERRMEM_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_ERRMEM_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_ERRMEM_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)


#define ETG_TRACE_FATAL(printfArgs)
#define ETG_TRACE_FATAL_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_FATAL_THR(printfArgs)
#define ETG_TRACE_FATAL_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_FATAL_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_FATAL_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_FATAL_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_FATAL_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)


#define ETG_TRACE_ERR(printfArgs)
#define ETG_TRACE_ERR_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_ERR_THR(printfArgs)
#define ETG_TRACE_ERR_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_ERR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_ERR_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_ERR_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_ERR_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_SYS_MIN(printfArgs)
#define ETG_TRACE_SYS_MIN_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_SYS_MIN_THR(printfArgs)
#define ETG_TRACE_SYS_MIN_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_SYS_MIN_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_SYS_MIN_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_SYS_MIN_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_SYS_MIN_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_SYS(printfArgs)
#define ETG_TRACE_SYS_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_SYS_THR(printfArgs)
#define ETG_TRACE_SYS_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_SYS_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_SYS_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_SYS_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_SYS_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_COMP(printfArgs)
#define ETG_TRACE_COMP_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_COMP_THR(printfArgs)
#define ETG_TRACE_COMP_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_COMP_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_COMP_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_COMP_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_COMP_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)


#define ETG_TRACE_USR1(printfArgs)
#define ETG_TRACE_USR1_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR1_THR(printfArgs)
#define ETG_TRACE_USR1_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR1_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR1_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR1_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR1_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_USR2(printfArgs)
#define ETG_TRACE_USR2_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR2_THR(printfArgs)
#define ETG_TRACE_USR2_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR2_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR2_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR2_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR2_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_USR3(printfArgs)
#define ETG_TRACE_USR3_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR3_THR(printfArgs)
#define ETG_TRACE_USR3_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR3_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR3_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR3_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR3_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#define ETG_TRACE_USR4(printfArgs)
#define ETG_TRACE_USR4_CLS(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR4_THR(printfArgs)
#define ETG_TRACE_USR4_CLS_THR(u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR4_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR4_CLS_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)
#define ETG_TRACE_USR4_THR_DCL(u16DerivedClass_printfArgs)
#define ETG_TRACE_USR4_CLS_THR_DCL(u16DerivedClass_u16TraceClass_PrintfArgs)

#endif

   //lint -save -e773
#define ETG_MAKE_LIST_STRING_DECL(entryLen, UorS, StringOrStringSigned)                      \
   etg_tChar *etg_MakeT##entryLen##List##StringOrStringSigned(etg_tChar *coszOutBuf, etg_tCString coszFormat, etg_tU16 u16NumEntries, const etg_t##UorS##entryLen *p##UorS##entryLen##Buffer)
   //lint -restore

   ETG_MAKE_LIST_STRING_DECL(8, U, String);
   ETG_MAKE_LIST_STRING_DECL(16, U, String);
   ETG_MAKE_LIST_STRING_DECL(32, U, String);
   ETG_MAKE_LIST_STRING_DECL(64, U, String);

   ETG_MAKE_LIST_STRING_DECL(8, S, StringSigned);
   ETG_MAKE_LIST_STRING_DECL(16, S, StringSigned);
   ETG_MAKE_LIST_STRING_DECL(32, S, StringSigned);
   ETG_MAKE_LIST_STRING_DECL(64, S, StringSigned);


   etg_tU8 *etg_pu8MakeListStringBuf(etg_tU8 *pu8Dest, const etg_tChar *pcSrc, etg_tU8 u16MaxNumChars);
   etg_tVoid etg_vGetThreadData(etg_tThreadID *pId, etg_tChar *szName);

   etg_tVoid etg_vSetTraceDevice(etg_tIODescriptor ETGTrace);
   etg_tVoid etg_vTraceBinary(etg_tU32 u32CompAndLevelId, etg_tU32 u32TraceId, etg_tU16 u16LineNr, etg_tU32 u32EtgFormats, ...); //lint !e579  /*Review happened by Grosse-Pawig*/
   etg_tBool etg_bIsTraceActive(etg_tU16 u16CompID, etg_tU16 u16LevelId);
   etg_tBool etg_bIsTraceActiveShort(etg_tU32 u32CompAndLevelId);
   etg_tBool etg_bIsTraceActiveDouble(etg_tU32 u32CompAndLevelId, etg_tU16 u16LevleId2);


   // ------------------------------------------ INPUT ----------------------------------------------
   // macros to register a ETG_I_- channel
   // the pfDefProc is called if non of the ETG CB handler were taking the trace input (e.g. illformed or old traces)
   // the default process handler should contain at least a trace that the command was not processed
#define ETG_I_REGISTER_CHN(pfDefProc)
#define ETG_I_UNREGISTER_CHN()

   // macro to process ttfis-buffer via ETG_I wihout channel-registration
   // returns etg_tBool TRUE if ETG_I CMD has been processed FALSE otherwise
   // the default process handler should contain at least a trace that the command was not processed
#define ETG_I_bPROCESS(u8TtfisCmdBuf)

   // macro to (un)register ETG_I- commands of a file
   // this must be called once per file at very first time (e.g. constructor)
#define ETG_I_REGISTER_FILE()
#define ETG_I_UNREGISTER_FILE()

   // macro to generate an ETG_I_- command (USER)
   // SYNTAX ETG_I_CMD_DEFINE((<CallBackFuncPtr>, "<CmdName>[(<space><formatParam>)]+"[(, <type>)]+ ))
   // with CallBackFuncPtr = the function to be called having described parameters
   // CmdName = the string for the command in TTFis, will be prepended with the component name
   // formatParam = scanf() coded format parameter description
   // type = the data type to be used for this parameter
   // eg. ETG_I_CMD_DEFINE((fnToBeCalled, "ttfisCmdName %d %d", etg_tU8, ETG_I_CENUM(tenTypeOfEnum)));
   // the makro is to be defined (as #define) anywhere in the corresponding file
#define ETG_I_CMD_DEFINE(a)

   // macros to specify arguments inside ETG_I_CMD_DEFINE() (USER)
   // using an enum already defined in TTFis
#define ETG_I_ENUM(a)
   // using a enum generated by ETG from a given enum
#define ETG_I_CENUM(a)
   // using a string (zero terminated); max length is 118 chars
#define ETG_I_STRING
   // using a byte array (no termination); max length is 118 chars
#define ETG_I_ARRAY_U8
   // using the given text as is in the call. Usefull to define twos commands using the same CB function
#define ETG_I_CONST_ARG(a)

   // callback to be provided by each file to register its commands
   typedef etg_tBool (*etg_i_tpfFileCallback)(etg_tU16, etg_tU8, const etg_tUChar* const);
   // callback for trace-channel
   typedef etg_tVoid (*etg_i_tpfChnCallback)(const etg_tUChar* const);
   // (base-)node for intrusive list
   typedef struct _etg_i_tCmdInfos {
      struct _etg_i_tCmdInfos *prNext;
      etg_i_tpfFileCallback pfFileCallback;
      etg_tU16 u16FileId;
   } etg_i_tCmdInfos;

   // service function to maintain the interface
   etg_tS32 etg_i_s32RegisterTraceChannel(etg_tU16 enTraceChan, etg_i_tpfChnCallback pfChnCallback);
   etg_tS32 etg_i_s32UnRegisterTraceChannel(etg_tU16 enTraceChan, etg_i_tpfChnCallback pfChnCallback);
   etg_tVoid etg_i_vDefineDefProc(etg_i_tCmdInfos *poCmdInfos, ETG_tpfCallback pfDefProc);
   etg_tVoid etg_i_vAddCmdInfos(etg_i_tCmdInfos *prBase, etg_i_tCmdInfos *prNew);
   etg_tVoid etg_i_vRemoveCmdInfos(etg_i_tCmdInfos *prBase, const etg_i_tCmdInfos *prRemove);
   etg_tBool etg_i_bChnProcess(etg_i_tCmdInfos *prBase, const etg_tUChar* const pu8TtfisCmdBuf);

   // service functions to support DLT command injection
   etg_tS32 etg_i_s32RegisterEtgICmds(etg_i_tpfFileCallback callback, etg_tU32 length, etg_tU32* cmdIds);
   etg_tS32 etg_i_s32DeregisterEtgICmds(etg_tU32 length, etg_tU32* cmdIds);

   // some byte-utilities
   etg_tU32 etg_i_getU32(const etg_tU8* const pu8Ar);
   etg_tU16 etg_i_getU16(const etg_tU8* const pu8Ar);
   etg_tU8 etg_i_getU8(const etg_tU8* const pu8Ar);
   etg_tS32 etg_i_getS32(const etg_tU8* const pu8Ar);
   etg_tS16 etg_i_getS16(const etg_tU8* const pu8Ar);
   etg_tS8 etg_i_getS8(const etg_tU8* const pu8Ar);


   typedef etg_tS32 (*ETG_tpfs32RegisterEtgICmds)(etg_i_tpfFileCallback pfvCb, etg_tU32 length, etg_tU32* cmdIds);
   typedef etg_tS32 (*ETG_tpfs32DeregisterEtgICmds)(etg_tU32 length, etg_tU32* cmdIds);

#ifndef ETG_IMPLEMENTS
   extern ETG_tpfs32RegisterEtgICmds ETG_pfs32RegisterEtgICmds;
   extern ETG_tpfs32DeregisterEtgICmds ETG_pfs32DeregisterEtgICmds;
#else
   ETG_tpfs32RegisterEtgICmds ETG_pfs32RegisterEtgICmds = NULL;
   ETG_tpfs32DeregisterEtgICmds ETG_pfs32DeregisterEtgICmds = NULL;
#endif

#ifdef __cplusplus
}
#endif

   // ------------------------------------------ INPUT ----------------------------------------------


#endif
