/*
 * vd_clock_datetime.cpp
 *
 *  Created on: Apr 23, 2015
 *      Author: vee4kor
 */



#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#define ET_TRACE_INFO_ON
#include "etrace_if.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_VD_CLOCK_DATETIME
#include "trcGenProj/Header/vd_clock_datetime.cpp.trc.h"
#endif


#include "vd_clock_types.h"
#include "../framework/app/vd_clock_AppMain_Trace.h"
#include "../framework/app/vd_clock_AppMain.h"
#include "vd_clock_datetime.h"


#define _CLASS TR_CLASS_VD_CLOCK_DATETIME
#define ETG_TRACING

const tU32  cu32SecondsPerMinute   = 60L;
const tU32  cu32SecondsPerHour     = 60L * cu32SecondsPerMinute;
const tU32  cu32SecondsPerDay      = 24L * cu32SecondsPerHour;
const tU32  cu32SecondsPer365Days  = 365L * cu32SecondsPerDay;

// ****************************************************************************

tBool	b_IsLeapYear(tS16 s16Year)
{
	// Utility to distinguish leap years (having 29 days in february)
	// from non-leap years.
	// Leap years are years dividable by 4 with exception of 2100, 2200, 2300
	// and 2500, but regularly in 2000, 2400, 2800.
	//
	// Martin Koch / ESE, 2007-6-25

   // TSTS changed: Bugfix MMS 301182
   // For 100 years there is no exception and only 100 years will be used.
   // If changed the year 2100 --> 2000 leads to loss of one day (MMS 301182)

	if (( s16Year % 4) == 0 ) // && (((s16Year % 100) !=  0) || ((s16Year % 400) == 0)))
		{ return TRUE; }
	else
		{ return FALSE; }
};

// ****************************************************************************
// input (valid: 1 .. 12; returns 0 days otherwise)
tU8 u8_DaysPerMonth (tU8  u8Month, tS16 s16Year)
{
	// Utility to calculate month length
	//
	// Martin Koch / ESE, 2007-6-25

	switch (u8Month)
	{
		case 1: case 3: case 5: case 7: case 8: case 10: case 12:
			{ return 31; }
		case 4: case 6: case 9: case 11:
			{ return 30; }
		case 2:
			{
				if (b_IsLeapYear(s16Year))
					{ return 29; }
				else
					{ return 28; }
			}
		default:
			{ return 0; }
	}
};
// ****************************************************************************
// calculates the current day of the year
// Returns: tU8 day of year, used as helper for calculating the current weekday
// Monday = 1, Sunday = 7;
// author: Thomas Schurig, Brunel GmbH, 2011-24-02
tU16 u16_DayOfYear( tS16 s16Year,     // input, any valid tS16
                    tU8 u8Month,		// input, 1 .. 12
                    tU8 u8Day	)		// input, 1 .. number of days
{
   // initialize return value
   tU16 RetValue = 0;

   // if in January, just return the current date
   if(u8Month == 1)
      return u8Day;

   // if month February ... December
   else
   {
      // how many months have to be added? (From 1(january) to given month)
      for(tU8 i=1 ; i<u8Month ; i++)
      {
         // add the days of month value for each month for given year
         RetValue += u8_DaysPerMonth (i, s16Year);
      }

      // add current date
      RetValue += u8Day;
   }

   // return calculated value
   return (RetValue);
};

// ****************************************************************************
// calculates the current weekday
// Returns: 1=Monday ... 0=Sunday
// author: Thomas Schurig, Brunel GmbH, 2011-24-02
tU8 u8_Weekday ( tS16 s16Year,		   // input, any valid tS16
                 tU8 u8Month,			// input, 1 .. 12
                 tU8 u8Day				// input, 1 .. number of days
               )
{
   // initialize local variables
   tS32 RetValue = 0;

   // for all years after 2000 before current year
   for(tS16 i=0 ; i<s16Year ; i++)
   {
      if(b_IsLeapYear(i))
      {
         RetValue += 0x16E; // 366
      }
      else
      {
         RetValue += 0x16D; // 365
      }
   }

   // add the day of the current year
   RetValue += u16_DayOfYear(s16Year,u8Month,u8Day);
   //RetValue += u8_DayOfYear(s16Year,6,9);

   // 01.01.2000 = Saturday!
   RetValue += 5;

   // Return the modulo of 7 from calculated day from 01.01.2000 until the given date
   RetValue %= 7;

   return((tU8)RetValue);
};

// ****************************************************************************
tU32 u32_SecondsPerCentury (tS16 s16Year)
{
	// Utility to calculate century length; different due to leap year
	// differences in century base year
	//
	// Martin Koch / ESE, 2007-6-28


	tU32  u32RetVal  = 100L * cu32SecondsPer365Days
							 + 24L * cu32SecondsPerDay;
	if (b_IsLeapYear(s16Year - (s16Year % 100))) // reduce to century base
   {
      u32RetVal += cu32SecondsPerDay; // add one intercalary day
   }
	return u32RetVal;

};


// ****************************************************************************
tVoid	v_Decompose	( tU32  u32TimeDateValue,		// input (valid: 0 .. seconds per century)
		              tS16  s16CenturyBase,		// input (valid: any tS16 dividable by 100)
		              tS16*	rs16Year,				// location for result
		              tU8*	ru8Month,				//          "
		              tU8*	ru8Day, 					//          "
		              tU8*	ru8Hour, 				//          "
		              tU8*	ru8Minute, 				//          "
		              tU8*	ru8Second,				//          "
                    tU8*   ru8Weekday
                  )
{
	// Utility to decompose a given value (seconds elapsed since century base)
	// into single values for year, month, day, hour, minute second.
	// Return value: VDCLK_DateValid | VDCLK_TimeValid, if decomposition successful.
	//
	// Note: year will be returned between 0 and 100; so add century base externally.
	//
	// Martin Koch / ESE, 2007-6-26

	tU32  u32SecondsPerYear = cu32SecondsPer365Days;


	if (b_IsLeapYear(s16CenturyBase)) {
		u32SecondsPerYear += cu32SecondsPerDay;
	}
   tU32  u32TmpValue = u32TimeDateValue;

//	tU8   u8DateValid = 0;  // obsolete

	// extract # of years and decrease u32TmpValue correspondingly
   if(rs16Year) {
      *rs16Year = 0;
      while (u32TmpValue >= u32SecondsPerYear)
      {
         u32TmpValue -= u32SecondsPerYear;
         (*rs16Year)++;
         u32SecondsPerYear = cu32SecondsPer365Days;
         if (b_IsLeapYear(*rs16Year))
            u32SecondsPerYear += cu32SecondsPerDay;
      }
   } else {
      NORMAL_M_ASSERT_ALWAYS();
   }
	// Year is found, next extract Month and further decrease u32TmpValue
   if(ru8Month && rs16Year) {
	   *ru8Month = 1;
	   while (u32TmpValue >= u8_DaysPerMonth(*ru8Month, *rs16Year) * cu32SecondsPerDay)
	   {
		   u32TmpValue -= u8_DaysPerMonth(*ru8Month, *rs16Year) * cu32SecondsPerDay;
		   (*ru8Month)++;
	   }
   } else {
      NORMAL_M_ASSERT_ALWAYS();
   }
	// Month is found, just calculate Day
   if (ru8Day) {
	   *ru8Day = 1 + (tU8) (u32TmpValue / cu32SecondsPerDay);
   }
	u32TmpValue %= cu32SecondsPerDay;
	// calculate Hour, Minute, Second
   if(ru8Hour) {
	   *ru8Hour = (tU8) (u32TmpValue / cu32SecondsPerHour);
   }
	u32TmpValue %= cu32SecondsPerHour;
   if(ru8Minute) {
	   *ru8Minute = (tU8) (u32TmpValue / cu32SecondsPerMinute);
   }
	u32TmpValue %= cu32SecondsPerMinute;
   if(ru8Second) {
	   *ru8Second = (tU8) u32TmpValue;
   }
   if(ru8Weekday && rs16Year && ru8Month && ru8Day) {
      *ru8Weekday = u8_Weekday(*rs16Year,*ru8Month,*ru8Day);
   }
	return;
};

// ****************************************************************************

tU32 u32_Composition	( tS16 s16Year,	// input (valid: any tS16, only last 2 digits are relevant)
                       tU8 u8Month,	   // input (valid: 1 .. 12)
                       tU8 u8Day,		// input (valid: 1 .. number of days)
                       tU8 u8Hour,		// input (valid: 0 .. 23)
                       tU8 u8Minute,	// input (valid: 0 .. 59)
                       tU8 u8Second, 	// input (valid: 0 .. 59)
                       tU8 /*u8Weekday*/   // input (valid: 0 .. 6)
                     )
{
	// Utility to combine date/time components into one single value
	// Return value: Seconds elapsed since century base
	//
	// Note: Year will be forced into range 0 .. 99 by using modulo operation;
	// so handle century base externally.
	//
	// Note: No overflow checking, but misleading results if input is not valid
	//
	// Martin Koch / ESE, 2007-6-26

	// add components of time
   tU32 u32RetVal = u8Second
						+ u8Minute * cu32SecondsPerMinute
						+ u8Hour   * cu32SecondsPerHour;

	// add components of date
	u32RetVal += (u8Day - 1) * cu32SecondsPerDay;
	for (tU8 i = 1; i < u8Month; i++)
	{
		u32RetVal += u8_DaysPerMonth(i, s16Year) * cu32SecondsPerDay;
	}
	tS16 s16CenturyBase = s16Year - (s16Year % 100);

   for (tS16 y = s16CenturyBase; y < s16Year; y++)
   {
	   u32RetVal += cu32SecondsPer365Days;
	   if (b_IsLeapYear(y))  // is leap year, so add 1 day
		   u32RetVal += cu32SecondsPerDay;
   }
	return u32RetVal;
};

// ****************************************************************************
// ***			C l a s s   I n t e r f a c e   F u n c t i o n s				 ***
// ****************************************************************************

vdclk_tclDateTime::vdclk_tclDateTime (tVoid)
{
	// Standard constructor
	//
	// Martin Koch / ESE, 2007-6-26

	_u32Value         = 0UL;
	_s16CenturyBase   = 2000;
   _u8FieldsValid    = 0;

    ETG_TRACE_COMP(("Flow-Event: %d", ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_FlowEventCTOR)));

};

// ****************************************************************************

vdclk_tclDateTime::  vdclk_tclDateTime ( const vdclk_tclDateTime& roDT) :
      // initialize member variables
      _u32Value         (roDT._u32Value),
      _s16CenturyBase   (roDT._s16CenturyBase),
      _u8FieldsValid    (roDT._u8FieldsValid)

{
   // Copy constructor
	//
	// Martin Koch / ESE, 2007-7-30
    ETG_TRACE_COMP(("Flow-Event: %d", ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_FlowEventCTOR)));

};

// ****************************************************************************

vdclk_tclDateTime:: ~vdclk_tclDateTime (tVoid)
{
	// Standard destructor
	//
	// Martin Koch / ESE, 2007-6-26
    ETG_TRACE_COMP(("Flow-Event: %d", ETG_ENUM(VD_CLOCK_FlowEvents,TR_EN_FlowEventDTOR)));
};

// ****************************************************************************

tU8  vdclk_tclDateTime:: u8GetComponents ( tS16* ps16Year,			// location for result
		                                     tU8*	pu8Month,			//				"
		                                     tU8*	pu8Day, 				//				"
		                                     tU8*	pu8Hour, 			//				"
		                                     tU8*	pu8Minute, 			//				"
		                                     tU8*	pu8Second,			//				"
                                           tU8* pu8Weekday
	                                       )
{
	// Member function to decompose a given value (seconds elapsed since century base)
	// into single values for year, month, day, hour, minute second.
	// Return value: VDCLK_DateValid | VDCLK_TimeValid, if decomposition successful.
	//
	// Martin Koch / ESE, 2007-6-26

	v_Decompose(this->_u32Value, this->_s16CenturyBase,
									 ps16Year, pu8Month, pu8Day,
									 pu8Hour, pu8Minute, pu8Second, pu8Weekday);

	*ps16Year += _s16CenturyBase;
	return u8IsDateTimeValid();
};

// ****************************************************************************
tU32	vdclk_tclDateTime:: u32GetValue(	tVoid	) const
{
	// Member access function for internal value.
	// Return value: seconds elapsed since century base.
	//
	// Martin Koch / ESE, 2007-6-26

	return this->_u32Value;
};

// ****************************************************************************

tU8 vdclk_tclDateTime::u8Update (   tS16 s16Year,		// (input, valid: any tS16)
		                              tU8 u8Month,		// (input, valid: 1 .. 12)
		                              tU8 u8Day,			// (input, valid: 1 .. number of days in given month)
		                              tU8 u8Hour,			// (input, valid: 0 .. 23)
		                              tU8 u8Minute,		// (input, valid: 0 .. 59)
		                              tU8 u8Second, 		// (input, valid: 0 .. 59)
		                              tU8 enChangeMask	// (input, valid: any combination of <Change> values
								                              //						from VDCLK_TEN_TimeDateMask
                                )
{

	// Member function to update internal date and time value
	// from given composite data.
	// A bit field mask is used to indicate, which components are to be updated.
	//
	// Note: Dates before 1582 will be treated as extrapolated gregorian calendar
	// - No transformation to Julian calendar or so.
	//
	// Martin Koch / ESE, 2007-6-25

	tS16 s16YY = 0;		tU8  u8MM  = 0;		tU8  u8DD  = 0;		// Date locals
	tU8  u8HH  = 0;		tU8  u8NN  = 0;		tU8  u8SS  = 0;		// Time locals
   tU8  u8WD  = 0;
	tU8 u8RetVal = 0;

	// extract components
   v_Decompose(_u32Value, _s16CenturyBase, &s16YY, &u8MM, &u8DD, &u8HH, &u8NN, &u8SS, &u8WD);
	s16YY += _s16CenturyBase;
	// modify according to bit field
	if ((enChangeMask & (tU8) VDCLK_EN_YearChange) == (tU8) VDCLK_EN_YearChange)
	{
		//if (s16YY != s16Year)
		//	{ u8RetVal  |= VDCLK_EN_YearChange; }
		s16YY = s16Year;
		vMarkField (VDCLK_EN_YearChange, TRUE);
	}
	if (enChangeMask & (tU8) VDCLK_EN_MonthChange)
	{
		//if (u8MM != u8Month)
		//	{ u8RetVal  |= VDCLK_EN_MonthChange; }
		u8MM = u8Month;
		vMarkField (VDCLK_EN_MonthChange, (tBool) (u8MM <= 12));
	}
	if (enChangeMask & (tU8) VDCLK_EN_DayChange)
	{
		//if (u8DD != u8Day)
		//	{ u8RetVal |= VDCLK_EN_DayChange; }
		u8DD = u8Day;
		// check argument
		vMarkField (VDCLK_EN_DayChange, (tBool) (u8DD <= u8_DaysPerMonth(u8MM, s16YY)));
	}
	if (enChangeMask & (tU8) VDCLK_EN_HourChange)
	{
		//if (u8HH != u8Hour)
		//	{ u8RetVal |= VDCLK_EN_HourChange; }
		u8HH = u8Hour;
		vMarkField (VDCLK_EN_HourChange, (tBool) (u8HH < 24));
	}
	if (enChangeMask & (tU8) VDCLK_EN_MinuteChange)
	{
		//if (u8NN != u8Minute)
		//	{ u8RetVal |= VDCLK_EN_MinuteChange; }
		u8NN = u8Minute;
		vMarkField (VDCLK_EN_MinuteChange, (tBool) (u8NN < 60));
	}
	if (enChangeMask & (tU8) VDCLK_EN_SecondChange)
	{
		//if (u8SS != u8Second)
		//	{ u8RetVal |= VDCLK_EN_SecondChange; }
		u8SS = u8Second;
		vMarkField (VDCLK_EN_SecondChange, (tBool) (u8SS < 60));
	}

	// write back
	_u32Value = u32_Composition(s16YY, u8MM, u8DD, u8HH, u8NN, u8SS, u8WD);
	_s16CenturyBase = s16YY - (s16YY % 100);

   // TSTS
   //_s16CenturyBase = 2000;

   TRACE_DateTime("u8Update", (*this), _CLASS, __LINE__);
   u8RetVal |= u8IsDateTimeValid();
	return u8RetVal;
};

// ****************************************************************************

tU8 vdclk_tclDateTime::u8SetValue ( tU32  u32Value	)// (input, valid: 0 .. seconds per century)
{
   // Member function to update internal date and time value
	// from given composite value (seconds elapsed from century base).
	//
	// Return Value: combination of <Change> and <Valid> values
	//					from VDCLK_TEN_TimeDateMask
	//
	// Note:
   //   - According to definition in FI handling of u32Value distinguishes:
   //      normal   :  0x00000000 ... u32CurrentCenturyLength
   //      overflow :  u32CurrentCenturyLength + 1 ... 0xCEE90B00
   //      invalid  :  0xCEE90B01 ... 0xED33FCFF
   //      underflow:  0xED33FD00 ... 0xFFFFFFFF
   //
   //   - Input value only spans one century, so normally century base
	//     remains uneffected, except in case of over-/underflow.
	//
	// Martin Koch / ESE, 2007-6-25

	tU8  RetVal		= 0;
	tU32 val			= u32Value;
	tS16 bas			= _s16CenturyBase;
	tU32 u32CurrentCenturyLength = u32_SecondsPerCentury(_s16CenturyBase);
	//if (u32Value >= (0xFFFFFFFF - 10L * cu32SecondsPer365Days))  // underflow
	if (u32Value >= 0xED33FD00)  // underflow as defined in FI
	{
		bas -= 100;
 		val -= (0xFFFFFFFF - u32_SecondsPerCentury(bas)) + 1L;
	}
	//else if (u32Value >= u32CurrentCenturyLength  + 10L * cu32SecondsPer365Days)
	else if (u32Value > 0xCEE90B00)  // invalid as defined in FI
	{
      // set date and time invalid due to  ambiguity
		_u8FieldsValid &= ~((tU8) VDCLK_EN_AllChange);
		return RetVal | u8IsDateTimeValid();
	}
	//else if (u32Value >= u32CurrentCenturyLength)  // overflow
	else if (u32Value > u32CurrentCenturyLength)  // overflow
	{
		bas += 100;
		val -= u32CurrentCenturyLength;
	}

	// To Do:   enter critical
	_u32Value = val;
	_s16CenturyBase = bas;
	// To Do:   release critical

	return RetVal | u8IsDateTimeValid();
};

// ****************************************************************************
tS16	vdclk_tclDateTime::s16GetCenturyBase ( tVoid ) const
{
	// Member access function for internal value.

   return _s16CenturyBase;
};

// ****************************************************************************
tBool	 vdclk_tclDateTime:: bIsLeapYear	( tS16 s16Year	)
{
	// Static class member to distinguish leap years (having 29 days in february)
	// from non-leap years.
	// Leap years are years dividable by 4 with exception of 2100, 2200, 2300
	// and 2500, but regularly in 2000, 2400, 2800.
	//
	// Note: Dates before 1582 will be treated as extrapolated gregorian calendar
	// - No transformation to Julian calendar or so.
	//
	// Martin Koch / ESE, 2007-6-25

	return b_IsLeapYear(s16Year);
};

// ****************************************************************************

tU8  vdclk_tclDateTime::u8DaysPerMonth (tU8  u8Month,	tS16 s16Year )
{
	// Static class member to calculate the number of days within a given month
	// of a given year.
	//
	// Martin Koch / ESE, 2007-6-25

   return u8_DaysPerMonth (u8Month, s16Year);
};

// ****************************************************************************

tBool  vdclk_tclDateTime::bIsDateTimeValid (	tS16 s16Year,			// input, any valid tS16
		                                       tU8  u8Month,			// input, 1 .. 12
		                                       tU8  u8Day,				// input, 1 .. number of days
		                                       tU8  u8Hour,		   // input, 1 .. 24
		                                       tU8  u8Minute,			// input, 0 .. 59
		                                       tU8  u8Second,			// input, 0 .. 59
                                             tU8  /*u8weekday*/,      // input, 1 .. 7
		                                       tU8  enChangeMask		// input, any combination of VDCLK_EN_..Change values
									                                       //			 from VDCLK_TEN_TimeDateMask
	                                       )
   // Utility function to validate parameters intended for Update
   // - considers only components indicated by enChangeMask
   // - invalidates automatically, if trying to update day
   //   without giving valid month and year
	//
	// Return value is completely independent of internal state.
{
   tBool  bRetVal = TRUE;

   // validate seconds
   if (((enChangeMask & ((tU8) VDCLK_EN_SecondChange)) != 0) && (u8Second >= 60))
   {
      bRetVal = FALSE;
   }
   // validate minutes
   else if (((enChangeMask & ((tU8) VDCLK_EN_MinuteChange)) != 0) && (u8Minute >= 60))
   {
      bRetVal = FALSE;
   }
   // validate hours
   else if (((enChangeMask & ((tU8) VDCLK_EN_HourChange)) != 0) && (u8Hour >= 24))
   {
      bRetVal = FALSE;
   }
   // validate month
   else if (((enChangeMask & ((tU8) VDCLK_EN_MonthChange)) != 0) && ((u8Month == 0) || (u8Month > 12)))
   {
      bRetVal = FALSE;
   }
   // validate entire date
   else if (((enChangeMask & ((tU8) VDCLK_EN_DayChange)) != 0) && ( (u8Day == 0) || (u8Day > u8DaysPerMonth(u8Month, s16Year))))
   {
      bRetVal = FALSE;
   }
   return bRetVal;
};

// ****************************************************************************
   // alternative to using the == operator
tU8  vdclk_tclDateTime:: u8GetChanges (vdclk_tclDateTime* poPrevDT)
{
   // Returns and combination of <VDCLK_TEN_TimeDateMask> values
	//
	// Note: opposite logic: 0 means both operands are identical

    ETG_TRACE_USR3(("..."));

   tS16	y_, y2;				tU8	m1, m2;				tU8	d1, d2;
	tU8	h1, h2;				tU8	n1, n2;				tU8	s1, s2;
   tU8   w1, w2;
	tU8	RetVal;

	RetVal = this->u8GetComponents (&y_, &m1, &d1, &h1, &n1, &s1, &w1);
	RetVal &= poPrevDT->u8GetComponents (&y2, &m2, &d2, &h2, &n2, &s2, &w2);

	if (y_ != y2) RetVal  |=  (tU8) VDCLK_EN_YearChange;
	if (m1 != m2) RetVal  |=  (tU8) VDCLK_EN_MonthChange;
	if (d1 != d2) RetVal  |=  (tU8) VDCLK_EN_DayChange;      // day date change
	if (h1 != h2) RetVal  |=  (tU8) VDCLK_EN_HourChange;
	if (n1 != n2) RetVal  |=  (tU8) VDCLK_EN_MinuteChange;
	if (s1 != s2) RetVal  |=  (tU8) VDCLK_EN_SecondChange;
   if (w1 != w2) RetVal  |=  (tU8) VDCLK_EN_DayChange;      // weekday change
	return RetVal;
};

// ****************************************************************************

vdclk_tclDateTime&  vdclk_tclDateTime::   operator	+=
	(
		const tS64	s64Offset
	)

{
   // AddInPlace operator +=.
	//
	// used to add dynamic time offset (DTO) or duration values
    ETG_TRACE_USR3(("..."));

   this->u8SetValue((tU32) ((tS64) _u32Value + s64Offset));
//   _u32Value += s32Offset;
   return *this;
};

// ****************************************************************************

vdclk_tclDateTime&  vdclk_tclDateTime::   operator	=
	(
		const vdclk_tclDateTime& roDT
	)	//
	// Assignment operator =.
	//
{
   ETG_TRACE_USR3(("operator=()"));

   if (&roDT != this)
   {
      _u32Value       = roDT._u32Value;
      _s16CenturyBase = roDT._s16CenturyBase;
      _u8FieldsValid  = roDT._u8FieldsValid;
   }
   return *this;
};

// ****************************************************************************

tU8  vdclk_tclDateTime::operator ==	(vdclk_tclDateTime  oDT)
{

	// Comparison operator ==.
	// Returns and combination of <VDCLK_TEN_TimeDateMask> values
	//
	// Note: opposite logic: 0 means both operands are identical

    ETG_TRACE_USR3(("..."));

   tS16	y_, y2;				tU8	m1, m2;				tU8	d1, d2;
	tU8	h1, h2;				tU8	n1, n2;				tU8	s1, s2;
   tU8   w1, w2;
	tU8	RetVal;

	RetVal = this->u8GetComponents (&y_, &m1, &d1, &h1, &n1, &s1, &w1);
	RetVal &= oDT.u8GetComponents (&y2, &m2, &d2, &h2, &n2, &s2, &w2);

	if (y_ != y2) RetVal  |=  (tU8) VDCLK_EN_YearChange;
	if (m1 != m2) RetVal  |=  (tU8) VDCLK_EN_MonthChange;
	if (d1 != d2) RetVal  |=  (tU8) VDCLK_EN_DayChange;
	if (h1 != h2) RetVal  |=  (tU8) VDCLK_EN_HourChange;
	if (n1 != n2) RetVal  |=  (tU8) VDCLK_EN_MinuteChange;
	if (s1 != s2) RetVal  |=  (tU8) VDCLK_EN_SecondChange;
   if (w1 != w2) RetVal  |=  (tU8) VDCLK_EN_DayChange;
	return RetVal;
};
//
//// ****************************************************************************
//
//tU32		vdclk_tclDateTime::				u32NewValue
//
//   // utility for calculating the term
//   //
//   //   u32NewValue = u32Value + u32Summand - u32Subtrahend
//   //
//   // if all participients may use the full tU32 range
//   // (avoids overflow if possible).
//
//   (
//      tU32 u32Value,
//      tU32 u32Summand,
//      tU32 u32Subtrahend
//   )
//{
//    tU32 u32RetVal;
//    if (u32Value >= u32Subtrahend)
//    {
//       u32RetVal = (u32Value - u32Subtrahend) + u32Summand;
//    }
//    else if (u32Summand >= u32Subtrahend)
//    {
//       u32RetVal = u32Value + (u32Summand - u32Subtrahend);
//    }
//    else
//    {
//       u32RetVal = u32Value - (u32Subtrahend - u32Summand);
//    }
//    return u32RetVal;
//};

// ****************************************************************************
// ***				C l a s s   P r i v a t e   S e c t i o n						 ***
// ****************************************************************************

	// Member function to inform, whether date and time components
	// have been initalized by using u8Update().

tU8 vdclk_tclDateTime::u8IsDateTimeValid ( tVoid )  const
	// Returns:
	// - VDCLK_TimeValid if hour, minute and second have been initialized
	// - VDCLK_DateValid if year, month and day have been initialized
	// - VDCLK_AllValid  if both.
{
	tU8	u8RetVal = 0;
	if ((_u8FieldsValid & ((tU8) VDCLK_EN_TimeChange)) == ((tU8) VDCLK_EN_TimeChange))
		{ u8RetVal |= (tU8) VDCLK_EN_TimeValid; }
	if ((_u8FieldsValid & ((tU8) VDCLK_EN_DateChange)) == ((tU8) VDCLK_EN_DateChange))
		{ u8RetVal |= (tU8) VDCLK_EN_DateValid; }
	return u8RetVal;
};

// ****************************************************************************
// Function to set / reset the individual fields of _u8FieldsValid
tVoid	vdclk_tclDateTime::vMarkField	( VDCLK_TEN_TimeDateStatus	e8Field,	// input, any valid <Change> value
		                                  tBool	bIsValid	// input, condition
                                    )
{
	if (bIsValid)
		{ _u8FieldsValid |= (tU8) e8Field; }		// validate
	else
		{ _u8FieldsValid &= ~((tU8) e8Field); }		// invalidate
	return;
};

/************************************************************************
* FUNCTION:    vGetDateFormat
*
* DESCRIPTION: convert the u32 value into yy,mm,dd,hh,mm,ss format
* No. of years ,months calcualtion is based on UTC year(to find it is a leap year or not), UTC month( to find the length of the month)
*
* PARAMETER:   input: , Offset sign, u32value in seconds , UTC year, UTC month,
               output: no of.yy,mm,dd,hh,mm,ss
*
* RETURNVALUE: None
*************************************************************************/

tVoid vdclk_tclDateTime::vGetDateFormat(tBool  bIsPositive, tU32 u32Value, tS16 s16UTCYear, tU8 u8UTCMonth, tS16 *ps16Year, tU8 *pu8Month, tU8 *pu8Day, tU8 *pu8Hour, tU8 *pu8Minute, tU8 *pu8Second)
{
   tU32 u32tmpValue = u32Value;
   tS16 s16tempyear = s16UTCYear;
   tU32  u32SecondsPerYear = cu32SecondsPer365Days;
   bIsPositive ? s16tempyear++ : s16tempyear--;// increment or decrement year based on offset sign

   if(b_IsLeapYear(s16tempyear))
   {
      u32SecondsPerYear += cu32SecondsPerDay;
   }
   while(u32tmpValue >= u32SecondsPerYear)
   {
      u32tmpValue -= u32SecondsPerYear;
      (*ps16Year)++;
      bIsPositive ? s16tempyear++ : s16tempyear--;
      u32SecondsPerYear = cu32SecondsPer365Days;
      if(b_IsLeapYear(s16tempyear))
         u32SecondsPerYear += cu32SecondsPerDay;
      if((*ps16Year) > 99)
      {
         ETG_TRACE_USR3(("Error. Avoided EndlessLoop due to invalid year"));
         return;
      }
   }

   tU8 u8tmpMonth = ((bIsPositive ? u8UTCMonth+1 : u8UTCMonth-1) + 12) % 12; // increment or decrement the month based on the offset sign

   if(u8tmpMonth == 0) // 0 denotes december; make tmpMonth = 12
      u8tmpMonth = 12;

   s16tempyear = (bIsPositive ? s16UTCYear+(*ps16Year) : s16UTCYear-(*ps16Year));

   while(u32tmpValue >= u8_DaysPerMonth(u8tmpMonth, s16tempyear) * cu32SecondsPerDay)
   {
      u32tmpValue -= u8_DaysPerMonth(u8tmpMonth, s16tempyear) * cu32SecondsPerDay;
      (*pu8Month)++;

      u8tmpMonth = ((bIsPositive ? u8tmpMonth +1 : u8tmpMonth -1) + 12) % 12;
      if(u8tmpMonth == 0) // 0 denotes december; make tmpMonth = 12
         u8tmpMonth = 12;

      if((*pu8Month) > 12)
      {
         ETG_TRACE_USR3(("Error. Avoided EndlessLoop due to invalid Month"));
         return;
      }
   }
   if(u32tmpValue >= cu32SecondsPerDay)
   {
      *pu8Day = (tU8)(u32tmpValue / cu32SecondsPerDay);
      u32tmpValue = u32tmpValue % cu32SecondsPerDay;
   }
   if(u32tmpValue >= cu32SecondsPerHour)
   {
      *pu8Hour = (tU8)(u32tmpValue / cu32SecondsPerHour);
      u32tmpValue = u32tmpValue % cu32SecondsPerHour;
   }
   if(u32tmpValue >= cu32SecondsPerMinute)
   {
      *pu8Minute = (tU8)(u32tmpValue / cu32SecondsPerMinute);
      u32tmpValue = u32tmpValue % cu32SecondsPerMinute;
   }
   *pu8Second = (tU8)u32tmpValue;

}

/************************************************************************
* FUNCTION:    u8GetDayoftheWeek
*
* DESCRIPTION: Method to find the day of the week
* PARAMETER:   input: Year, month, Day of the month
               output: Day of the week
*
* RETURNVALUE: Day of the week
*************************************************************************/

tU8 vdclk_tclDateTime::u8GetDayoftheWeek(tU16 u16Year, tU8 u8Month, tU8 u8Day)
{
	tU8 u8Temp[] = {0, 3, 2, 5, 0, 3, 5, 1, 4, 6, 2, 4};
	//u16Year -= (u8Month < 3); //lint correction
	if(u8Month < 3)
	{
		u16Year = u16Year - 1;
	}
	return ((u16Year + (u16Year/4) - (u16Year/100) + (u16Year/400) + u8Temp[u8Month-1] + u8Day) % 7);
}

/************************************************************************
* FUNCTION:    vGetDateFormat
*
* DESCRIPTION:   Method to find the date for Nth day of the Nth Week of the Month
                 used to find the start of summer time and Winter time
       *
* PARAMETER:   input: Year, Month, Day of the week, nthweek of the month
               output: Date (where summer time or winter time starts)
*
* RETURNVALUE: Date (day of the month)
*************************************************************************/
tU8 vdclk_tclDateTime::u8GetNthDate_DST(tU16 u16Year, tU8 u8Month, tU8 DayofWeek, tU8 u8NthWeek)
{

	tU8 u8TargetDate = 1;
	tU8 u8First_DayofWeek = u8GetDayoftheWeek(u16Year, u8Month, u8TargetDate);
	while (u8First_DayofWeek != DayofWeek)
	{
		u8First_DayofWeek = ((u8First_DayofWeek+1) % 7);
		u8TargetDate++;
		if(u8TargetDate > 32)
		{
			ETG_TRACE_USR3(("Error. Avoided EndlessLoop due to invalid date in u8GetNthDate_DST"));
			break;
		}
	}
	  //calculate for Nth week(s)
	u8TargetDate += (u8NthWeek-1)*7;

	// for 5th sunday in a week
	if((u8NthWeek == 4) && ((u8TargetDate+7) <= 31))
	{
		u8TargetDate = u8TargetDate + 7;
	}

	return u8TargetDate;
}
