
#include "ipod_Utf8StringBase.h"
#include "ipod_Utf8String.h"
#include "ipod_Utf8StringComparator.h"

//#include "../ipod_trace.h"

// -- trace identifier ------- define this value for every class new
//#ifdef CLASS_NUMBER
//#undef CLASS_NUMBER
//#endif
//#define CLASS_NUMBER TRC::TR_CLASS_IPOD_REMOTECONTROLLER
// -- trace identifier ------- define this value for every file new
//#ifdef FILE_NUMBER
//#undef FILE_NUMBER
//#endif
//#define FILE_NUMBER TRC::ipod_remotecontroller_cpp
// -- trace identifier ------- end

tU32 ipod_tclUtf8StringBase::aru32SortValues[cu32MaxUTF8Value];
tBool ipod_tclUtf8StringBase::bInit = FALSE;

#define IPOD_NUM_SPCL_WORDS_TO_IGNORE   3

tCString u8spWordTable[IPOD_NUM_SPCL_WORDS_TO_IGNORE] =
{
   "A ",
   "An ",
   "The "
};

#define IPOD_NUM_SPCL_CHARS_TO_IGNORE   128

tU32 u32spCharTable[IPOD_NUM_SPCL_CHARS_TO_IGNORE] =
{
   0x0,
   0x020, // SPACE
   0x021,
   0x022,
   0x023,
   0x024,
   0x025,
   0x026,
   0x028,
   0x029,
   0x02a,
   0x02b,
   0x02c,
   0x02d,
   0x02e,
   0x02f,
   0x03a,
   0x03b,
   0x03c,
   0x03d,
   0x03e,
   0x03f,
   0x040,
   0x05b,
   0x05c,
   0x05d,
   0x05e,
   0x05f,
   0x060,
   0x07b,
   0x07c,
   0x07d,
   0x07e,
   0x0c2a2,
   0x0c2a3,
   0x0c2a4,
   0x0c2a5,
   0x0c2a6,
   0x0c2a7,
   0x0c2a8,
   0x0c2a9,
   0x0c2aa,
   0x0c2ab,
   0x0c2ac,
   0x0c2ae,
   0x0c2af,
   0x0c397,
   0x0c2b0,
   0x0c2b1,
   0x0c2b4,
   0x0c2b5,
   0x0c2b6,
   0x0c2b7,
   0x0c2b8,
   0x0c2bf,
   0x0c2b8,
   0x0c2ba,
   0x0c2bb,
   0x0c2bc,
   0x0c2bd,
   0x0c2be,
   0x0cdb4, //
   0x0cdb5,
   0x0cdba,
   0x0cdbe,
   0x0ce84,
   0x0ce85,
   0x0ce87,
   0x0cf97,//greek kai
   0x0cf8f, // greek kai
   0x0cf99, // archaic koppa
   0x0cf98,
   0x0e0b8b0,//thai
   0x0e0b8b1,
   0x0e0b8b2,
   0x0e0b8b3,
   0x0e0b8b4,
   0x0e0b8b5,
   0x0e0b8b6,
   0x0e0b8b7,
   0x0e0b8b8,
   0x0e0b8b9,
   0x0e0b8ba,
   0x0e0b8bf,
   0x0e0b980,
   0x0e0b981,
   0x0e0b982,
   0x0e0b983,
   0x0e0b984,
   0x0e0b985,
   0x0e0b986,
   0x0e0b987,
   0x0e0b988,
   0x0e0b989,
   0x0e0b98a,
   0x0e0b98b,
   0x0e0b98c,
   0x0e0b98d,
   0x0e0b98e,
   0x0e0b98f,
   0x0e0b99a,
   0x0e0b99b,
   0x0dba5,                //arabic
   0x0d88c,//
   0x0d88d,
   0x0d89b,
   0x0d89f,
   0x0d980,//
   0x0d9aa,//
   0x0d9ab,//
   0x0d9ac,//
   0x0d9ad,//
   0x0d9af,//
   0x0d9ae,//
   0x0db9d,//
   0x0db9e,//
   0x0db94,//
   0x0dba6,
   0x0dba9,//
   0x0dbbd,
   0x0dbbe,
   0x0d9ae,
   0x0d9af,
   0x0dbbb,
   0x0dbbc,
   0x0dbba,
   0x0dbae,
   0x0db9c,
};




/********************************************************************
*FUNCTION:     ipod_tclUtf8StringBase
*DESCRIPTION:  constructor
*PARAMETER:    pu8String, u32StringSize, bAlloc
*RETURNVALUE:  object of type ipod_tclUtf8StringBase
*******************************************************************/
ipod_tclUtf8StringBase::ipod_tclUtf8StringBase(tPCU8 pu8String, tU32 u32StringSize,tBool bAlloc)
   :
   _pu8String(pu8String)
   ,_pu8Data(OSAL_NULL)
   ,_u32StringSize(u32StringSize)
   ,_u32NumberValue(cu32InvalidNumberValue)
   ,_u32Offset(0)
   ,_bSZOnce(FALSE)
   ,_bAlloc(bAlloc)
{
   if(!u32StringSize && pu8String)
   {
      _u32StringSize = strlen((const char*)pu8String);
   }
   if(_bAlloc && _u32StringSize && pu8String)
   {
      _pu8Data = new tU8[_u32StringSize+1];
      if(_pu8Data)
      {
         (tVoid)OSAL_pvMemoryCopy(_pu8Data,pu8String,_u32StringSize);
         _pu8Data[_u32StringSize]=0;
         _pu8String = _pu8Data;
      }
   }
   else
   {
      _bAlloc = FALSE;
   }
   _u32NumberValue = u32CalculateNumberValue();
   if(!bInit)
   {
      vInitTable();
   }
};

/********************************************************************
*FUNCTION:     ipod_tclUtf8StringBase
*DESCRIPTION:  Constructor
*PARAMETER:    Constant reference to ipod_tclUtf8StringBase object
*RETURNVALUE:  None
*******************************************************************/
ipod_tclUtf8StringBase::ipod_tclUtf8StringBase( const ipod_tclUtf8StringBase& roTheOther)
{
   *this = roTheOther;
}

/********************************************************************
*FUNCTION:     ipod_tclUtf8StringBase
*DESCRIPTION:  Destructor
*PARAMETER:    None
*RETURNVALUE:  None
*******************************************************************/
ipod_tclUtf8StringBase::~ipod_tclUtf8StringBase()
{
   if(_bAlloc && _pu8Data)
   {
      delete [] _pu8Data;
   }
   _pu8String = OSAL_NULL;
};

/********************************************************************
*FUNCTION:     u32CalculateNumberValue ( currently this function is not used)
*DESCRIPTION:  Calculates the numeric value of the string litteral if it contains only numbers.
*PARAMETER:    none
*RETURNVALUE:  Numeric value.
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32CalculateNumberValue()
{
   if( (_u32NumberValue == cu32InvalidNumberValue) && (_pu8String != OSAL_NULL) )
   {
      tU32 u32Index   = 0;
      _u32NumberValue = 0;

      //calculate as long as we have charakter to check and the charkters are numbers
      for( ; (u32Index <= cu32NumOfDigits) && (u32Index < _u32StringSize) ; u32Index++)
      {
         tU8 u8Char = *(_pu8String+u32Index);
         if((u8Char < '0') || (u8Char > '9') )
         {
            break;   //not a number
         }
         _u32NumberValue = 10*_u32NumberValue + (u8Char-'0');
      }

      //no number found, set number value to default for a pure string
      if(u32Index == 0)
      {
         _u32NumberValue = cu32StringNumberValue;
      }
   }
   return _u32NumberValue;
}

/********************************************************************
*FUNCTION:     u32GetNextUtf8Value
*DESCRIPTION:  Returns the next utf8Value present in the string.
*PARAMETER:    None
*RETURNVALUE:  utf8Value returned in tU32
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32GetNextUtf8Value()
{
   return u32GetUtf8Value(_u32Offset, _bSZOnce);
}

/********************************************************************
*FUNCTION:     u32ignSplCharAtBegin
*DESCRIPTION:  Special chars present in the beginning of the string has to be ignored.
*              If the string contains only special characters then return the index as 0.
*
*PARAMETER:    pstr : String to search , u32BegIndex : start ignoring special characters from this index.
*
*RETURNVALUE:  Lowest Index where the character is not the special characters.
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32ignSplCharAtBegin(tPCU8 pstr, tU32 u32begIndex) const
{
   tU32 u32spCharIdx = 0;
   tU32 u32Utf8Value =0;

   u32Utf8Value = (tU32)pstr[u32begIndex];

   tU8 u8ByteCount = 0;
   if(u32Utf8Value> UTF_FOUR_BYTE_BOUNDARY)
   {
      u8ByteCount=4;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   else if(u32Utf8Value>UTF_THREE_BYTE_BOUNDARY && u32Utf8Value <= UTF_FOUR_BYTE_BOUNDARY)
   {
      u8ByteCount=3;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   else if(u32Utf8Value > UTF_TWO_BYTE_BOUNDARY && u32Utf8Value <= UTF_THREE_BYTE_BOUNDARY)
   {
      u8ByteCount=2;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   else
   {
      u8ByteCount=1;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   /*lint -e{850} u32idx is modified in the body of the for loop */
   for(tU32 u32idx = u32begIndex; u32idx <OSAL_u32StringLength(pstr) ; u32idx++)
   {
      u32Utf8Value = (tU32)pstr[u32idx];

      for( u32spCharIdx = 0; u32spCharIdx < IPOD_NUM_SPCL_CHARS_TO_IGNORE; u32spCharIdx++)
      {
         if(u8ByteCount == 2)//this utf8 char consists of two bytes
         {
            if( (u32idx + 1) >= OSAL_u32StringLength(pstr))
            {
               return 0;
            }

            u32Utf8Value = 0x00000100*u32Utf8Value + (tU32)pstr[u32idx+1]; //get second byte
            // //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"Entered u32ignSplCharAtBegin,u8ByteCount == 2 ,u32Utf8Value= :",u32Utf8Value);
         }
         else if(u8ByteCount == 3)
         {
            if( (u32idx + 2) >= OSAL_u32StringLength(pstr))
            {
               return 0;
            }
            u32Utf8Value = 0x00010000*u32Utf8Value + 0x00000100*((tU32)pstr[u32idx+1])+(tU32)pstr[u32idx+2]; //get second byte
            //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"Entered u32ignSplCharAtBegin,u8ByteCount == 3 ,u32Utf8Value= :",u32Utf8Value);
         }
         else
         {
            //Handle 4 bytes here
         }

         if( u32Utf8Value == u32spCharTable[u32spCharIdx])
         {
            break;
         }
      }
      if(u32spCharIdx == IPOD_NUM_SPCL_CHARS_TO_IGNORE)
      {
         //str[idx] is not a special character hence return the index;
         //if(u32idx)    //Trace only if there are special characters.
         //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"Special char ignored till index:",u32idx);

         return u32idx;
      }
      else
      {
         if(u8ByteCount == 2)
         {
            u32idx++;   //move past the second byte of UTF8 of special char.
         }
         else if(u8ByteCount == 3)
         {
            u32idx+=2;   //move past the third byte of UTF8 of special char.
         }
      }
   }

   //String contains only special characters
   return 0;
}


/********************************************************************
*FUNCTION:     u8GetUniqueChar
*DESCRIPTION:  This function returns the lower case letter for any
*              given input character
*PARAMETER:    input character
*RETURNVALUE:  lower case letter
*******************************************************************/
tU8 ipod_tclUtf8StringBase::u8GetUniqueChar(tU8 u8Char) const
{
   //This logic is considering input characters are only of ASCII letters.
   if(( u8Char - '0') > '0')
   {
      return u8Char;
   }
   return (u8Char + 0x20);
}


/********************************************************************
*FUNCTION:     u32ignSplWordAtBegin
*DESCRIPTION:  Ignore the special words present at the beginning of the string.
*
*PARAMETER:    pstr : string to consider u32begIndex: start index from which the words have to be ignored.
*RETURNVALUE:  tU32 : String position which is after the special word.
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32ignSplWordAtBegin(tPCU8 pstr, tU32 u32begIndex) const
{
   tU32 u32spWordIdx = 0, u32parseIndex = 0;
   tU32 u32idx = u32begIndex;
   for( u32spWordIdx = 0; u32spWordIdx < IPOD_NUM_SPCL_WORDS_TO_IGNORE; u32spWordIdx++)
   {
      for( u32idx = u32begIndex, u32parseIndex = 0; u32parseIndex < OSAL_u32StringLength(u8spWordTable[u32spWordIdx]); u32idx++,u32parseIndex++)
      {
         if( u8GetUniqueChar(pstr[u32idx]) != u8GetUniqueChar(u8spWordTable[u32spWordIdx][u32parseIndex]))
         {
            break;
         }
      }
      if(u32parseIndex >= OSAL_u32StringLength(u8spWordTable[u32spWordIdx]))
      {
         //search word found
         //TRACE_DBG_LS(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,u8spWordTable[u32spWordIdx]);
         //TRACE_DBG_LS(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"Special word present");
         return u32idx;
      }
   }
   if(u32spWordIdx == IPOD_NUM_SPCL_WORDS_TO_IGNORE )
   {
      //No special words found at the beginning of the string
      return u32begIndex;
   }

   //String contains only special word
   return 0;
}


/********************************************************************
*FUNCTION:     u32GetUtf8Value
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32GetUtf8Value(tU32& ru32OffSet, tBool& rbSZOnce) const
{
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"ru32OffSet:",(tS32)ru32OffSet);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"rbSZOnce:",(tS32)rbSZOnce);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"_pu8String:",(tS32)_pu8String);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"_u32StringSize:",(tS32)_u32StringSize);
   if(_pu8String == OSAL_NULL)
   {
      return cu32EndOfString;   //invalid pointer
   }

   if( ru32OffSet >= _u32StringSize )
   {
      return cu32EndOfString;   //found end of string
   }

   if(rbSZOnce)
   {
      rbSZOnce = FALSE;    // '' should be handeled a 'ss', this is the second 's'
      return (tU32)'s';
   }

   tU32 u32Utf8Value = *(_pu8String + ru32OffSet++);//get next byte from string
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value:",(tS32)u32Utf8Value);

   if( u32Utf8Value == 0)
   {
      return cu32EndOfString;   //end of string
   }

   tU8 u8ByteCount = 0;
   if(u32Utf8Value> UTF_FOUR_BYTE_BOUNDARY)
   {
      u8ByteCount=4;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   else if(u32Utf8Value>UTF_THREE_BYTE_BOUNDARY && u32Utf8Value <= UTF_FOUR_BYTE_BOUNDARY)
   {
      u8ByteCount=3;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   else if(u32Utf8Value > UTF_TWO_BYTE_BOUNDARY && u32Utf8Value <= UTF_THREE_BYTE_BOUNDARY)
   {
      u8ByteCount=2;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }
   else
   {
      u8ByteCount=1;

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8ByteCount:",(tS32)u8ByteCount);

   }

   if(u8ByteCount == 2)//this utf8 char consists of two bytes
   {
      //  //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value > cu32BiggestOneByteChar:",(tS32)u32Utf8Value);
      if( ru32OffSet >= _u32StringSize )
      {
         return cu32EndOfString;   //found end of string
      }

      tU32 u32Utf8Value2 = *(_pu8String + ru32OffSet++); //get second byte
      // //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value2:",(tS32)u32Utf8Value2);

      if( u32Utf8Value2 == 0)
      {
         return cu32EndOfString;   //end of string
      }

      u32Utf8Value = 0x00000100*u32Utf8Value + u32Utf8Value2; //calulate 2-Byte Utf8 Value
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"Effective 2 byte u32Utf8Value:",(tS32)u32Utf8Value);

   }
   else if(u8ByteCount == 3)
   {
      if( ru32OffSet >= _u32StringSize )
      {
         return cu32EndOfString;   //found end of string
      }
      tU32 u32Utf8Value2 = *(_pu8String + ru32OffSet++); //get second byte
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value2:",(tS32)u32Utf8Value2);
      tU32 u32Utf8Value3 = *(_pu8String + ru32OffSet++); //get second byte
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value3:",(tS32)u32Utf8Value3);
      if( u32Utf8Value3 == 0)
      {
         return cu32EndOfString;   //end of string
      }
      u32Utf8Value = 0x00010000*u32Utf8Value + u32Utf8Value2*0x00000100+u32Utf8Value3; //calulate 2-Byte Utf8 Value
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value:",(tS32)u32Utf8Value);
   }
   else if(u8ByteCount == 4)
   {
      // Whenever 4 byte utf 8 CHARACTER comes into picyre, handle it here
   }

   if(u32Utf8Value == 0xc39f )
   {
      rbSZOnce = TRUE;    //the character is a ''
      return (tU32)'s';
   }

   return u32Utf8Value;
}

/********************************************************************
*FUNCTION:     bThisIsEqual
*DESCRIPTION:  For the quicksearch feature the input search string contains single character
*              hence a string is equal to the search string if the beginning characters are same.
*PARAMETER:    search string.
*RETURNVALUE:  TRUE: if first characters match or belong to the same sort group.
*              FALSE: if they belong to different group.
*******************************************************************/
tBool ipod_tclUtf8StringBase::bThisIsEqual ( const ipod_tclUtf8StringBase& oOtherString) const
{

   if(_pu8String == OSAL_NULL)
   {
      return TRUE;
   }
   if(oOtherString._pu8String == OSAL_NULL)
   {
      return FALSE;
   }

   tU32 u32StringSize1=strlen((const char*)_pu8String);
   tU32 u32StringSize2=strlen((const char*)oOtherString._pu8String);

   tPU8 pu8ArabicString1 = new tU8[u32StringSize1+1];
   if(pu8ArabicString1 == OSAL_NULL)
   {
      return FALSE;
   }
   tPU8 pu8ArabicString2 = new tU8[u32StringSize2+1];
   if(pu8ArabicString2 == OSAL_NULL)
   {
      delete[] pu8ArabicString1;
      pu8ArabicString1 = 0;
      return FALSE;
   }
   tU32 u32Utf8Value1,u32Utf8Value2;

   tU32 u32SortValue1,u32SortValue2;

   tU32 u32OffSet1 = 0, u32OffSet2 = 0;

   tBool bSZOnce1  = FALSE;
   tBool bSZOnce2  = FALSE;

   (tVoid)OSAL_pvMemoryCopy(pu8ArabicString1,_pu8String,u32StringSize1);
   (tVoid)OSAL_pvMemoryCopy(pu8ArabicString2,oOtherString._pu8String,u32StringSize2);

   //if(pu8ArabicString1 != OSAL_NULL)
   //{
   pu8ArabicString1[u32StringSize1]='\0';
   //}
   //if(pu8ArabicString2 != OSAL_NULL)
   //{
   pu8ArabicString2[u32StringSize2]='\0';
   //}


   //if(pu8ArabicString1 != OSAL_NULL)
   //{
   if((pu8ArabicString1[0] >= UTF_ARABIC_LOWER_BOUNDARY)&&(pu8ArabicString1[0] < UTF_ARABIC_UPPER_BOUNDARY))
   {
      vReverseString(pu8ArabicString1,u32StringSize1);
   }
   //}
   //if(pu8ArabicString2 != OSAL_NULL)
   //{
   if((pu8ArabicString2[0] >= UTF_ARABIC_LOWER_BOUNDARY)&&(pu8ArabicString2[0] < UTF_ARABIC_UPPER_BOUNDARY))
   {
      vReverseString(pu8ArabicString2,u32StringSize2);
   }
   //}

   //if(pu8ArabicString1 != OSAL_NULL)
   //{
   if((pu8ArabicString1[0] >= UTF_ARABIC_LOWER_BOUNDARY)&&(pu8ArabicString1[0] < UTF_ARABIC_UPPER_BOUNDARY))
   {
      u32OffSet1 = u32ignSplCharAtBegin(pu8ArabicString1,u32OffSet1);
   }
   else
   {
      u32OffSet1 = u32ignSplCharAtBegin(_pu8String,u32OffSet1);
   }
   //}
   //if(pu8ArabicString2 != OSAL_NULL)
   //{
   if((pu8ArabicString2[0] >= UTF_ARABIC_LOWER_BOUNDARY)&&(pu8ArabicString2[0] < UTF_ARABIC_UPPER_BOUNDARY))
   {
      u32OffSet2 = u32ignSplCharAtBegin(pu8ArabicString2,u32OffSet2);
   }
   else
   {
      u32OffSet2 = u32ignSplCharAtBegin(oOtherString._pu8String,u32OffSet2);
   }
   //}
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32OffSet1:",(tS32)u32OffSet1);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32OffSet2:",(tS32)u32OffSet2);

   //if(pu8ArabicString1 != OSAL_NULL)
   //{
   if(pu8ArabicString1[0] < UTF_TWO_BYTE_BOUNDARY)
   {
      u32OffSet1 = u32ignSplWordAtBegin(_pu8String,u32OffSet1);
   }
   //}
   //if(pu8ArabicString2 != OSAL_NULL)
   //{
   if(pu8ArabicString2[0] < UTF_TWO_BYTE_BOUNDARY)
   {
      u32OffSet2 = u32ignSplWordAtBegin(oOtherString._pu8String,u32OffSet2);
   }
   //}

   u32Utf8Value1 = u32GetUtf8Value(u32OffSet1, bSZOnce1);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value1:",(tS32)u32Utf8Value1);
   u32Utf8Value2 = oOtherString.u32GetUtf8Value(u32OffSet2, bSZOnce2);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u32Utf8Value2:",(tS32)u32Utf8Value2);

   //if(pu8ArabicString1)
   //{
   delete[] pu8ArabicString1;
   pu8ArabicString1= OSAL_NULL;
   //}
   //if(pu8ArabicString2)
   //{
   delete[] pu8ArabicString2;
   pu8ArabicString2= OSAL_NULL;
   //}
   //end of this string reached, so it is shorter so this string < the other
   if( (u32Utf8Value1 == cu32EndOfString) && (u32Utf8Value2 != cu32EndOfString) )
   {
      return TRUE;
   }

   //end of string2 reached, so it is shorter so this string > the other
   if( (u32Utf8Value1 != cu32EndOfString) && (u32Utf8Value2 == cu32EndOfString) )
   {
      return FALSE;
   }


   //end of both strings reached, compare pointer addresses so (this string > the other) != (this string < the other), important for STL
   if( (u32Utf8Value1 == cu32EndOfString) && (u32Utf8Value2 == cu32EndOfString) )
   {
      return (this < &oOtherString);
   }

   u32SortValue1 = u32GetSortValue(u32Utf8Value1);
   u32SortValue2 = u32GetSortValue(u32Utf8Value2);

   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"SortValue1:",u32SortValue1);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"SortValue2:",u32SortValue2);

   if(u32SortValue1 == u32SortValue2)
   {
      return TRUE;
   }

   return FALSE;

}

/********************************************************************
*FUNCTION:     bThisIsLess ( currently this function  is not used in quicksearch)
*DESCRIPTION:  For the quicksearch feature, if the input search string contains a search string,
*              then this function is more suitable.
*
*PARAMETER:    Search string
*RETURNVALUE:  TRUE: if the search string is lexically lesser than the internal string.
*              FALSE: if greater or equal.
*              ( lexical ordering is based on the sort order defined elsewhere in the code)
*******************************************************************/
tBool ipod_tclUtf8StringBase::bThisIsLess ( const ipod_tclUtf8StringBase& oOtherString) const
{

   if(_pu8String == OSAL_NULL)
   {
      return TRUE;
   }
   if(oOtherString._pu8String == OSAL_NULL)
   {
      return FALSE;
   }

   tU32 u32Utf8Value1;
   tU32 u32Utf8Value2;

   tU32 u32SortValue1 =0;
   tU32 u32SortValue2=0;

   tU32 u32OffSet1 = 0;
   tU32 u32OffSet2 = 0;

   //Ignore the special characters and words like "a , an, the" before actually comaparing

   u32OffSet1 = u32ignSplCharAtBegin(_pu8String,u32OffSet1);
   u32OffSet1 = u32ignSplWordAtBegin(_pu8String,u32OffSet1);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,_pu8String,u32OffSet1);

   u32OffSet2 = u32ignSplCharAtBegin(oOtherString._pu8String,u32OffSet2);
   u32OffSet2 = u32ignSplWordAtBegin(oOtherString._pu8String,u32OffSet2);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,oOtherString._pu8String,u32OffSet2);

   tBool bSZOnce1  = FALSE;
   tBool bSZOnce2  = FALSE;

   for(;/*ever*/;)
   {
      u32Utf8Value1 = u32GetUtf8Value(u32OffSet1, bSZOnce1);
      u32Utf8Value2 = oOtherString.u32GetUtf8Value(u32OffSet2, bSZOnce2);

      //end of this string reached, so it is shorter so this string < the other
      if( (u32Utf8Value1 == cu32EndOfString) && (u32Utf8Value2 != cu32EndOfString) )
      {
         return TRUE;
      }

      //end of string2 reached, so it is shorter so this string > the other
      if( (u32Utf8Value1 != cu32EndOfString) && (u32Utf8Value2 == cu32EndOfString) )
      {
         return FALSE;
      }

      //end of both strings reached, compare pointer addresses so (this string > the other) != (this string < the other), important for STL
      if( (u32Utf8Value1 == cu32EndOfString) && (u32Utf8Value2 == cu32EndOfString) )
      {
         return (this < &oOtherString);
      }
      u32SortValue1 = u32GetSortValue(u32Utf8Value1);
      u32SortValue2 = u32GetSortValue(u32Utf8Value2);

      /*if(
      (u32OffSet1 == 1 ) && (u32OffSet2 == 1 )
      &&
      ( bIsANumber(u32Utf8Value1) || bIsANumber(u32Utf8Value2)  )
      &&
      ( u32GetNumberValue() != oOtherString.u32GetNumberValue() )
      &&
      ( u32SortValue1 >= u32GetSortValue('0') )
      &&
      ( u32SortValue2 >= u32GetSortValue('0') )
      )
      {
      return u32GetNumberValue() < oOtherString.u32GetNumberValue();
      }*/

      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"SortValue1:",u32SortValue1);
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"SortValue2:",u32SortValue2);

      if(u32SortValue1 != u32SortValue2)
      {
         return (u32SortValue1 < u32SortValue2);
      }
   }
}

/********************************************************************
*FUNCTION:
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*******************************************************************/
tVoid ipod_tclUtf8StringBase::vSetValue(tU32 u32Pos, tU32 u32Value)
{
   if(u32Pos >= cu32MaxUTF8Value)
   {
      return;   //not supported value
   }
   //all unknown values are special character, so they should be placed before the known character
   //that's why we add an offset for the known ones
   aru32SortValues[u32Pos] = u32Value + cu32ValidOffset;
}

/********************************************************************
*FUNCTION:
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*******************************************************************/
ipod_tclUtf8StringBase& ipod_tclUtf8StringBase::operator=( const ipod_tclUtf8StringBase& roTheOther)
{
   if(this != &roTheOther)
   {
      _u32StringSize  = roTheOther._u32StringSize;
      _u32NumberValue = roTheOther._u32NumberValue;
      _u32Offset      = 0;
      _bSZOnce        = FALSE;
      _bAlloc         = roTheOther._bAlloc;

      tPCU8 pu8Temp   = roTheOther._pu8String; //lint complains direct pointer copy ;)
      _pu8String      = pu8Temp;

      if(_bAlloc && _u32StringSize)
      {
         _pu8Data = new tU8[_u32StringSize+1];
         if(_pu8Data)
         {
            (tVoid)OSAL_pvMemoryCopy(_pu8Data,roTheOther._pu8String,_u32StringSize);
            _pu8Data[_u32StringSize]=0;
            _pu8String = _pu8Data;
         }
      }
   }
   return *this;
}

const tU32 ipod_tclUtf8StringBase::Table[ipod_tclUtf8StringBase::u32NumOfEntries][ipod_tclUtf8StringBase::u32EntrySize] =
{
   {0x0db9d,11},// arabic spc
   {0x0dbbb,2},
   {0x0d9ae,3},
   {0x0d9af,1},
   {0x0dbbc,5},
   {0x0dbba,6},
   {0x0dba9,4},
   {0x0dbbd,5},
   {0x0dbbe,6},
   {0x0dba5, 7},
   {0x0dba6,8},
   {0x0db9e,9},
   {0x0d980,10},
   {0x0e282ac , 19},//euro symbol
   {0x041 , 20},
   {0x061 , 20},
   {0x0c3a0 , 20},
   {0x0c380 , 20},
   {0x0c382 , 20},
   {0x0c3a2 , 20},
   {0x0c483 , 20},
   {0x0c482 , 20},
   {0x0c481 , 20},
   {0x0c480 , 20},
   {0x0c3a3 , 20},
   {0x0c383 , 20},
   {0x0c381 , 25},                 //
   {0x0c3a1 , 25},                  //
   {0x0c384 , 30},               //
   {0x0c3a4 , 30},                //
   {0x0c3a5 , 35},                 //
   {0x0c385 , 35},                 //  ----
   {0x0c484 , 40},                 //
   {0x0c485 , 40},                 //
   {0x0c386 , 41},            //
   {0x0c3a6 , 41},            //
   {0x062 , 45},
   {0x042 , 45},
   {0x0c387 , 50},
   {0x043 , 50},
   {0x063 , 50},
   {0x0c3a7 , 50},
   {0x0c486 , 55},                //
   {0x0c487 , 55},                //
   {0x0c48c , 60},                //-----
   {0x0c48d , 60},               //
   {0x064 , 65},
   {0x044 , 65},
   {0x0c491 , 65},
   {0x0c390 , 65},
   {0x0c3b0 , 65},
   {0x0c48e , 70},                    //
   {0x0c48f , 70},                    //
   {0x045 , 75},
   {0x065 , 75},
   {0x0c3a9 , 80},                   //
   {0x0c389 , 80},                   //
   {0x0c3a8 , 75},
   {0x0c388 , 75},
   {0x0c496 , 75},
   {0x0c497 , 75},
   {0x0c38a , 75},
   {0x0c3aa , 75},
   {0x0c38b , 75},
   {0x0c3ab , 75},
   {0x0c493 , 75},
   {0x0c492 , 75},
   {0x0c49b , 85},                 //
   {0x0c49a , 85},                 //
   {0x0c498 , 90},                       //
   {0x0c499 , 90},                       //
   {0x066 , 95},
   {0x046 , 95},
   {0x0c692 , 95},
   {0x047 , 100},
   {0x067 , 100},
   {0x0c49e , 100},
   {0x0c49f , 100},
   {0x0c4a3 , 100},
   {0x0c4a2 , 100},
   {0x068 , 105},
   {0x048 , 105},
   {0x0c4b2 , 110},
   {0x0c4b3 , 110},
   {0x069 , 110},
   {0x049 , 110},
   {0x0c4b1 , 110},
   {0x0c38c , 110},
   {0x0c3ac , 110},
   {0x0c4b0 , 110},
   {0x0c38e , 110},
   {0x0c3ae , 110},
   {0x0c38f , 110},
   {0x0c3af , 110},
   {0x0c4ac , 110},
   {0x0c4ad , 110},
   {0x0c4ab , 110},
   {0x0c4aa , 110},
   {0x0c4ae , 110},
   {0x0c4af , 110},
   {0x0c38d , 115},                 //
   {0x0c3ad , 115},                 //
   {0x06a , 120},
   {0x04a , 120},
   {0x04b , 125},
   {0x06b , 125},
   {0x0c4b6 , 125},
   {0x0c4b7 , 125},
   {0x06c , 130},
   {0x04c , 130},
   {0x0c4bb , 130},
   {0x0c4bc , 130},
   {0x0c4ba , 135},                                //
   {0x0c4b9 , 135},                               //------
   {0x0c4be , 140},                               //
   {0x0c4bd , 140},                               //
   {0x0c581 , 145},                               //
   {0x0c582 , 145},                               //
   {0x04d , 150},
   {0x06d , 150},
   {0x0c391 , 155},
   {0x06e , 155},
   {0x04e , 155},
   {0x0c3b1 , 155},
   {0x0c586 , 155},
   {0x0c585 , 155},
   {0x0c583 , 160},                               //
   {0x0c584 , 160},                               //
   {0x0c587 , 165},                           //---------
   {0x0c588 , 165},                                //
   {0x06f , 170},
   {0x04f , 170},
   {0x0c392 , 170},
   {0x0c3b2 , 170},
   {0x0c394 , 170},
   {0x0c3b4 , 170},
   {0x0c3b3 , 175},                              //
   {0x0c393 , 175},                              //
   {0x0c58d , 170},
   {0x0c58c , 170},
   {0x0c3b5 , 170},
   {0x0c395 , 170},
   {0x0c3b6 , 180},                //
   {0x0c396 , 180},                 //
   {0x0c591 , 185},                //
   {0x0c590 , 185},               //-----
   {0x0c398 , 190},               //
   {0x0c3b8 , 190},               //
   {0x070 , 195},
   {0x050 , 195},
   {0x071 , 200},
   {0x051 , 200},
   {0x052 , 205},
   {0x072 , 205},
   {0x0c597 , 205},
   {0x0c596 , 205},
   {0x0c595 , 210},            //-------------
   {0x0c594 , 210},            //-------------
   {0x0c598 , 215},           //
   {0x0c599 , 215},           //
   {0x0c39f , 220},
   {0x073 , 220},
   {0x053 , 220},
   {0x0c59e , 220},
   {0x0c59f , 220},
   {0x0c59a , 225},              //
   {0x0c59b , 225},              //
   {0x0c5a1 , 230},        // -----------
   {0x0c5a0 , 230},             //
   {0x0c39e , 235},
   {0x0c39e , 235},
   {0x054 , 240},
   {0x074 , 240},
   {0x0c5a3 , 240},
   {0x0c5a2 , 240},
   {0x0c5a4 , 245},                     //
   {0x0c5a5 , 245},                     //
   {0x055 , 250},
   {0x075 , 250},
   {0x0c3b9 , 250},
   {0x0c399 , 250},
   {0x0c3bb , 250},
   {0x0c39b , 250},
   {0x0c5aa , 250},
   {0x0c5ab , 250},
   {0x0c5a9 , 250},
   {0x0c5a8 , 250},
   {0x0c5b2 , 250},
   {0x0c5b3 , 250},
   {0x0c39a , 255},                      //
   {0x0c3ba , 255},                      //
   {0x0c39c , 260},                     //
   {0x0c3bc , 260},                     //
   {0x0c5ae , 265},                         //
   {0x0c5af , 265},                         //
   {0x0c5b1 , 270},                         //
   {0x0c5b0 , 270},                         //
   {0x056 , 275},
   {0x076 , 275},
   {0x057 , 280},
   {0x077 , 280},
   {0x058 , 285},
   {0x078 , 285},
   {0x079 , 290},
   {0x059 , 290},
   {0x0c5b8 , 290},
   {0x0c3bf , 290},
   {0x0c3bd , 295},                             //
   {0x0c39d , 295},                             //
   {0x0c2a1 , 300},
   {0x05a , 305},
   {0x07a , 305},
   {0x0c5ba , 310},                         //
   {0x0c5b9 , 310},                         //------
   {0x0c5bb , 315},                         //
   {0x0c5bc , 315},                         //
   {0x0c5bd , 320},                         //---------------
   {0x0c5be , 320},                         //
   {0x0d0b0 , 325},
   {0x0d090 , 325},
   {0x0d0b1 , 330},
   {0x0d091 , 330},
   {0x0d092 , 335},
   {0x0d0b2 , 335},
   {0x0d0b3 , 340},
   {0x0d093 , 340},
   {0x0d290 , 345},
   {0x0d291 , 345},
   {0x0d193 , 350},
   {0x0d083 , 355},
   {0x0d083 , 355},
   {0x0d094 , 360},
   {0x0d0b4 , 360},
   {0x0d082 , 365},
   {0x0d192 , 365},
   {0x0d082 , 365},
   {0x0d0b5 , 370},
   {0x0d095 , 370},
   {0x0d191 , 370},
   {0x0d081 , 373},
   {0x0d194 , 375},
   {0x0d084 , 375},
   {0x0d096 , 380},
   {0x0d0b6 , 380},
   {0x0d0b7 , 385},
   {0x0d097 , 385},
   {0x0d085 , 390},
   {0x0d195 , 390},
   {0x0d0b8 , 395},
   {0x0d098 , 395},
   {0x0d196 , 400},
   {0x0d086 , 400},
   {0x0d087 , 405},
   {0x0d197 , 405},
   {0x0d0b9 , 410},
   {0x0d099 , 410},
   {0x0d198 , 415},
   {0x0d088 , 415},
   {0x0d09a , 420},
   {0x0d0ba , 420},
   {0x0d08c , 425},
   {0x0d19c , 425},
   {0x0d0bb , 430},
   {0x0d09b , 430},
   {0x0d089 , 435},
   {0x0d199 , 435},
   {0x0d0bc , 440},
   {0x0d09c , 440},
   {0x0d09d , 445},
   {0x0d0bd , 445},
   {0x0d19a , 450},
   {0x0d08a , 450},
   {0x0d09e , 455},
   {0x0d0be , 455},
   {0x0d0bf , 460},
   {0x0d09f , 460},
   {0x0d180 , 465},
   {0x0d0a0 , 465},
   {0x0d0a1 , 470},
   {0x0d181 , 470},
   {0x0d182 , 475},
   {0x0d0a2 , 475},
   {0x0d08b , 480},
   {0x0d19b , 480},
   {0x0d183 , 485},
   {0x0d0a3 , 485},
   {0x0d08e , 490},
   {0x0d19e , 490},
   {0x0d0a4 , 495},
   {0x0d184 , 495},
   {0x0d185 , 500},
   {0x0d0a5 , 500},
   {0x0d0a6 , 505},
   {0x0d186 , 505},
   {0x0d0a7 , 510},
   {0x0d187 , 510},
   {0x0d08f , 515},
   {0x0d19f , 515},
   {0x0d0a8 , 520},
   {0x0d188 , 520},
   {0x0d189 , 525},
   {0x0d0a9 , 525},
   {0x0d0aa , 530},
   {0x0d18a , 530},
   {0x0d0ab , 535},
   {0x0d18b , 535},
   {0x0d18c , 540},
   {0x0d0ac , 540},
   {0x0d18d , 545},
   {0x0d0ad , 545},
   {0x0d0ae , 550},
   {0x0d18e , 550},
   {0x0d08d , 555},
   {0x0d19d , 555},
   {0x0d0af , 560},
   {0x0d18f , 560},
   {0x0ce91 , 1020},// Greek
   {0x0ceb1 , 1020},
   {0x0ce86 , 1020},
   {0x0ceac , 1020},
   {0x0ce92 , 1025},
   {0x0ceb2 , 1025},
   {0x0cf90 , 1025},
   {0x0ce93 , 1030},
   {0x0ceb3 , 1030},
   {0x0ce94 , 1035},
   {0x0ceb4 , 1035},
   {0x0ce95 , 1040},
   {0x0cead , 1040},
   {0x0ceb5 , 1040},
   {0x0cfb5 , 1040},
   {0x0ce88 , 1040},
   {0x0ce96 , 1045},
   {0x0ceb6 , 1045},
   {0x0ce97 , 1050},
   {0x0ceb7 , 1050},
   {0x0ceae , 1050},
   {0x0ce89 , 1050},
   {0x0ce98 , 1055},
   {0x0ceb8 , 1055},
   {0x0cfb4 , 1055},
   {0x0cf91 , 1055},
   {0x0ce99 , 1060},
   {0x0ceb9 , 1060},
   {0x0ceaa , 1060},
   {0x0cf8a , 1060},
   {0x0ce8a , 1060},
   {0x0ceaf , 1060},
   {0x0ce90 , 1060},
   {0x0ceb9 , 1060},
   {0x0ce9a , 1065},
   {0x0ceba , 1065},
   {0x0cfb0 , 1065},
   {0x0ce9b , 1070},
   {0x0cebb , 1070},
   {0x0ce9c , 1075},
   {0x0cebc , 1075},
   {0x0ce9d , 1080},
   {0x0cebd , 1080},
   {0x0ce9e , 1085},
   {0x0cebe , 1085},
   {0x0ce8c , 1090},
   {0x0cebf , 1090},
   {0x0ce9f , 1090},
   {0x0cf8c , 1090},
   {0x0cf80 , 1095},
   {0x0cf96 , 1095},
   {0x0cea0 , 1095},
   {0x0cea1 , 1100},
   {0x0cf81 , 1100},
   {0x0cfb1 , 1100},
   {0x0cfbc , 1100},
   {0x0cea3 , 1105},
   {0x0cf83 , 1105},
   {0x0cf82 , 1105},
   {0x0cfb2 , 1105},
   {0x0cfb9 , 1105},
   {0x0cfbe , 1105},
   {0x0cfbf , 1105},
   {0x0cdbc , 1105},
   {0x0cea4 , 1110},
   {0x0cf84 , 1110},
   {0x0cea5 , 1115},
   {0x0cf92 , 1115},
   {0x0cf93 , 1115},
   {0x0cf94 , 1115},
   {0x0ce8e , 1115},
   {0x0cea5 , 1115},
   {0x0ceab , 1115},
   {0x0ceb0 , 1115},
   {0x0cf85 , 1115},
   {0x0cf8b , 1115},
   {0x0cf8d , 1115},
   {0x0cea6 , 1120},
   {0x0cf86 , 1120},
   {0x0cf95 , 1120},
   {0x0cea7 , 1125},
   {0x0cf87 , 1125},
   {0x0cea8 , 1130},
   {0x0cf88 , 1130},
   {0x0ce8f , 1135},
   {0x0cea9 , 1135},
   {0x0cf89 , 1135},
   {0x0cf8e , 1135},
   {0x0cf9a , 1140},
   {0x0cf9b , 1140},
   {0x0cf9c , 1142},
   {0x0cf9d , 1142},
   {0x0cf9e , 1145},
   {0x0cf9f , 1145},
   {0x0cfa0 , 1150},
   {0x0cfa1 , 1150},
   {0x0d8a1,  1380},//arabic
   {0x0d8a4,  1380},
   {0x0d8a6,  1380},
   {0x0d8a7 , 1385},
   {0x0d8a2 , 1385},
   {0x0d8a3 , 1385},
   {0x0d8a5 , 1385},
   {0x0d9b1 , 1385},
   {0x0d9b2 , 1385},
   {0x0d9b3 , 1385},
   {0x0d9b5 , 1385},
   {0x0d8a8 , 1390},
   {0x0d9be , 1393},
   {0x0d8aa , 1395},
   {0x0d8a9 , 1398},
   {0x0d8ab , 1400},
   {0x0d8ac , 1405},
   {0x0d8ad , 1410},
   {0x0d8ae , 1415},
   {0x0d8af , 1420},
   {0x0d8b0 , 1425},
   {0x0d8b1 , 1430},
   {0x0d8b2 , 1435},
   {0x0da98 , 1443},
   {0x0d8b3 , 1444},
   {0x0d8b4 , 1445},
   {0x0d8b5 , 1450},
   {0x0d8b6 , 1455},
   {0x0d8b7 , 1460},
   {0x0d8b8 , 1462},
   {0x0d8b9 , 1463},
   {0x0d8ba , 1464},
   {0x0d981 , 1465},
   {0x0d982 , 1466},
   {0x0daa9 , 1467},
   {0x0daaa , 1468},
   {0x0d983 , 1468},
   {0x0daaf , 1469},
   {0x0d984 , 1470},
   {0x0d985 , 1471},
   {0x0d986 , 1472},
   {0x0d987 , 1473},
   {0x0d988 , 1474},
   {0x0d989 , 1475},
   {0x0d98a , 1476},
   {0x0d9b4 , 1477},
   {0x0d9b6 , 1477},
   {0x0d9b7 , 1477},
   {0x0d9b8 , 1477},
   {0x0d9b9 , 1478},
   {0x0d9ba , 1479},
   {0x0d9bb , 1480},
   {0x0d9bc , 1481},
   {0x0d9bd , 1481},
   {0x0d9bf , 1482},
   {0x0da80 , 1483},
   {0x0da81 , 1484},
   {0x0da82 , 1484},
   {0x0da83 , 1485},
   {0x0da84 , 1486},
   {0x0da85 , 1487},
   {0x0da87 , 1488},
   {0x0da88 , 1489},
   {0x0da89 , 1490},
   {0x0da8a , 1490},
   {0x0da8b , 1490},
   {0x0da8c , 1491},
   {0x0da8d , 1492},
   {0x0da8e , 1493},
   {0x0da8f , 1494},
   {0x0da90 , 1494},
   {0x0da91 , 1495},
   {0x0da92 , 1496},
   {0x0da93 , 1496},
   {0x0da94 , 1496},
   {0x0da95 , 1496},
   {0x0da96 , 1496},
   {0x0da97 , 1496},
   {0x0da99 , 1496},
   {0x0da9a , 1497},
   {0x0da9b , 1497},
   {0x0da9c , 1497},
   {0x0da9d , 1498},
   {0x0da9e , 1498},
   {0x0da9f , 1499},
   {0x0daa0 , 1500},
   {0x0daa1 , 1501},
   {0x0daa2 , 1501},
   {0x0daa3 , 1501},
   {0x0daa4 , 1501},
   {0x0daa5 , 1501},
   {0x0daa6 , 1502},
   {0x0daa7 , 1503},
   {0x0daa8 , 1503},
   {0x0daac , 1504},
   {0x0daab , 1504},
   {0x0daae , 1504},
   {0x0dab0 , 1505},
   {0x0dab1 , 1505},
   {0x0dab2 , 1505},
   {0x0dab3 , 1505},
   {0x0dab4 , 1505},
   {0x0dab5 , 1506},
   {0x0dab6 , 1506},
   {0x0dab7 , 1506},
   {0x0dab8 , 1506},
   {0x0dab9 , 1507},
   {0x0daba , 1507},
   {0x0dabb , 1507},
   {0x0dabc , 1507},
   {0x0dabd , 1507},
   {0x0dabe , 1508},
   {0x0dabf , 1509},
   {0x0db80 , 1510},
   {0x0db81 , 1510},
   {0x0db82 , 1510},
   {0x0db83 , 1511},
   {0x0db84 , 1512},
   {0x0db85 , 1513},
   {0x0db86 , 1513},
   {0x0db87 , 1514},
   {0x0db88 , 1515},
   {0x0db89 , 1515},
   {0x0db8a , 1516},
   {0x0db8d , 1517},
   {0x0db8c , 1517},
   {0x0db8e , 1517},
   {0x0db8f , 1518},
   {0x0db90 , 1519},
   {0x0db91 , 1520},
   {0x0db92 , 1520},
   {0x0db93 , 1520},
   {0x0db95 , 1525},
   {0x030 , 8565},                 // NUMBERS AND SPECIAL CHARACTERS AT THE END OF ALL LANGUAGE GROUPS
   {0x0c2b9 , 8570},
   {0x031 , 8570},
   {0x0c2b2 , 8575},
   {0x032 , 8575},
   {0x0c2b3 , 8580},
   {0x033 , 8580},
   {0x034 , 8585},
   {0x035 , 8590},
   {0x036 , 8595},
   {0x037 , 8600},
   {0x038 , 8605},
   {0x039 , 8610},
   {0x024 , 8700},
   {0x02b , 8705},
   {0x03b , 8710},
   {0x0c2a6 , 8715},
   {0x0c2a7 , 8720},
   {0x0c2a8 , 8725},
   {0x0c2a9 , 8730},
   {0x0c2aa , 8735},
   {0x0c2ab , 8740},
   {0x0c2ac , 8745},
   {0x0c2ae , 8750},
   {0x0c2af , 8755},
   {0x0c397 , 8760},
   {0x026 , 8765},
   {0x02d , 8770},
   {0x03d , 8775},
   {0x07b , 8780},
   {0x05b , 8785},
   {0x07c , 8790},
   {0x05c , 8795},
   {0x05d , 8800},
   {0x07d , 8805},
   {0x05e , 8810},
   {0x023 , 8815},
   {0x02a , 8820},
   {0x03a , 8825},
   {0x0c2b0 , 8830},
   {0x0c2b1 , 8835},
   {0x0c2b4 , 8840},
   {0x0c2b5 , 8845},
   {0x0c2b6 , 8850},
   {0x0c2b8 , 8860},
   {0x0 , 8865},
   {0x02e , 8870},
   {0x03e , 8875},
   {0x0c2bf , 8880},
   {0x021 , 8885},
   {0x029 , 8890},
   {0x022 , 8895},
   {0x040 , 8900},
   {0x0c2b7 , 8905},
   {0x0c2b8 , 8910},
   {0x0c2ba , 8915},
   {0x0c2bb , 8920},
   {0x0c2bc , 8925},
   {0x0c2bd , 8930},
   {0x0c2be , 8935},
   {0x028 , 8940},
   {0x02f , 8945},
   {0x03f , 8950},
   {0x025 , 8955},
   {0x02c , 8960},
   {0x03c , 8965},
   {0x05f , 8970},
   {0x060 , 8980},
   {0x07e , 8985},
   {0x0c2a2 , 8990},
   {0x0c2a3 , 8995},
   {0x0c2a4 , 9000},
   {0x0c2a5 , 9005},
   {0x02a , 9010},
   {0x03f , 9015},
   {0x0d9a0,8565},  // arabic numbers
   {0x0d9a1,8570},
   {0x0d9a2,8575},
   {0x0d9a3,8580},
   {0x0d9a4,8585},
   {0x0d9a5,8590},
   {0x0d9a6,8595},
   {0x0d9a7,8600},
   {0x0d9a8,8605},
   {0x0d9a9,8610},
   {0x0dbb0,8565},
   {0x0dbb1,8570},
   {0x0dbb2,8575},
   {0x0dbb3,8580},
   {0x0dbb4,8585},
   {0x0dbb5,8590},
   {0x0dbb6,8595},
   {0x0dbb7,8600},
   {0x0dbb8,8605},
   {0x0dbb9,8610},
   {0x0d9ad,8615},
   {0x0db94,8620},
   {0x0d9aa,8625},
   {0x0d9ac,8630},
   {0x0d89b, 8635},
   {0x0d89f,8640},
   {0x0d88c,8643},
   {0x0d9ab,8645}
};

#define TABLE_COL_SORT_POS  0
#define TABLE_COL_SORT_VAL  1


const tU32 ipod_tclUtf8StringBase::TableThai[ipod_tclUtf8StringBase::u32NumOfEntriesThai][ipod_tclUtf8StringBase::u32EntrySize] =
{
   {0x0e0b8bf,5150},
   {0x0e0b881 , 5155},//Thai
   {0x0e0b882 , 5160},
   {0x0e0b883 , 5165},
   {0x0e0b884 , 5170},
   {0x0e0b885 , 5175},
   {0x0e0b886 , 5180},
   {0x0e0b887 , 5185},
   {0x0e0b888 , 5190},
   {0x0e0b889 , 5195},
   {0x0e0b88a , 5200},
   {0x0e0b88b , 5205},
   {0x0e0b88c , 5210},
   {0x0e0b88d , 5215},
   {0x0e0b88e , 5220},
   {0x0e0b88f , 5225},
   {0x0e0b890 , 5230},
   {0x0e0b891 , 5235},
   {0x0e0b892 , 5240},
   {0x0e0b893 , 5245},
   {0x0e0b894 , 5250},
   {0x0e0b895 , 5255},
   {0x0e0b896 , 5260},
   {0x0e0b897 , 5265},
   {0x0e0b898 , 5270},
   {0x0e0b899 , 5275},
   {0x0e0b89a , 5280},
   {0x0e0b89b , 5285},
   {0x0e0b89c , 5290},
   {0x0e0b89d , 5295},
   {0x0e0b89e , 5300},
   {0x0e0b89f , 5305},
   {0x0e0b8a0 , 5310},
   {0x0e0b8a1 , 5315},
   {0x0e0b8a2 , 5320},
   {0x0e0b8a3 , 5325},
   {0x0e0b8a4 , 5330},
   {0x0e0b8a5 , 5335},
   {0x0e0b8a6 , 5340},
   {0x0e0b8a7 , 5345},
   {0x0e0b8a8 , 5350},
   {0x0e0b8a9 , 5355},
   {0x0e0b8aa , 5360},
   {0x0e0b8ab , 5365},
   {0x0e0b8ac , 5370},
   {0x0e0b8ad , 5375},
   {0x0e0b8ae , 5380},
   {0x0e0b8af ,5385},
   {0x0e0b8b0,9110},//special chars
   {0x0e0b8b1,9112},
   {0x0e0b8b2,9114},
   {0x0e0b8b3,9116},
   {0x0e0b8b4,9118},
   {0x0e0b8b5,9119},
   {0x0e0b8b6,9120},
   {0x0e0b8b7,9121},
   {0x0e0b8b8,9122},
   {0x0e0b8b9,9123},
   {0x0e0b8ba,9124},
   {0x0e0b980,9126},
   {0x0e0b981,9127},
   {0x0e0b982,9128},
   {0x0e0b983,9129},
   {0x0e0b984,9130},
   {0x0e0b985,9131},
   {0x0e0b986,9132},
   {0x0e0b987,9133},
   {0x0e0b988,9134},
   {0x0e0b989,9135},
   {0x0e0b98a,9136},
   {0x0e0b98b,9137},
   {0x0e0b98c,9138},
   {0x0e0b98d,9139},
   {0x0e0b98e,9140},
   {0x0e0b98f,9141},
   {0x0e0b990,8565},
   {0x0e0b991,8570},
   {0x0e0b992,8575},
   {0x0e0b993,8580},
   {0x0e0b994,8585},
   {0x0e0b995,8590},
   {0x0e0b996,8595},
   {0x0e0b997,8600},
   {0x0e0b998,8605},
   {0x0e0b999,8610},
   {0x0e0b99a,9142},
   {0x0e0b99b,9143}

};
/********************************************************************
*FUNCTION:     vInitTable
*DESCRIPTION:  Defines the sorting order
*              Sort order defined here is based on the ipod sorting.
*              This sort order may change if Apple changes its sorting order.
*PARAMETER:    None
*RETURNVALUE:  None
*******************************************************************/
tVoid ipod_tclUtf8StringBase::vInitTable()
{
   //init to default
   //all unknown characters are sorted by UFT8 Value
   for(tU32 u32Count = 0 ; u32Count < cu32MaxUTF8Value; u32Count++)
   {
      aru32SortValues[u32Count] = u32Count;
   }

   // UTF8 Value, SortIndex
   for( tU32 u32Index=0; u32Index < u32NumOfEntries; u32Index++)
   {
      vSetValue(Table[u32Index][TABLE_COL_SORT_POS] ,Table[u32Index][TABLE_COL_SORT_VAL]);
   }

   bInit = TRUE;
}

/********************************************************************
*FUNCTION:     vReverseString
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
******************************************************************/
tVoid ipod_tclUtf8StringBase::vReverseString(tPU8 pu8StringToBeReversed,tU32 u32StrLen) const
{
   tU8	u8Temp1=0;
   tU8	u8Temp2=0;
   if(pu8StringToBeReversed == OSAL_NULL)
   {
      return;
   }
   tPU8 pu8Begin1 = pu8StringToBeReversed;
   tPU8 pu8Begin2 = pu8StringToBeReversed+1;
   tPU8 pu8End1 = pu8StringToBeReversed+u32StrLen-2;
   tPU8 pu8End2 = pu8StringToBeReversed+u32StrLen-1;
   while((pu8Begin1 <= pu8End1)&&(pu8Begin2 <= pu8End2))
   {
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"pu8Begin1",pu8Begin1);
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"pu8Begin2",pu8Begin2);
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"pu8End1",pu8End1);
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"pu8End2",pu8End2);

      u8Temp1 = *pu8Begin1;
      u8Temp2 = *pu8Begin2;
      *pu8Begin1 = *pu8End1;
      *pu8Begin2 = *pu8End2;
      *pu8End1 = u8Temp1;
      *pu8End2 = u8Temp2;
      pu8Begin1+=2;
      pu8Begin2+=2;
      pu8End1-=2;
      pu8End2-=2;
   }
}

/********************************************************************
*FUNCTION:     u32GetSortValue
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32GetSortValue(tU32 u32Utf8Char) const
{
   if (u32Utf8Char <= cu32MaxUTF8Value - 1) // "cu32MaxUTF8Value - 1" to remove LINT
   {
      return aru32SortValues[(tU16)u32Utf8Char];
   }
   else
   {
      tU32 u32GroupValue = u32BinarySearch(u32Utf8Char);
      return (u32GroupValue + cu32ValidOffset);
   }
}
/********************************************************************
*FUNCTION:     u32BinarySearch
*DESCRIPTION:
*PARAMETER:
*RETURNVALUE:
*******************************************************************/
tU32 ipod_tclUtf8StringBase::u32BinarySearch(tU32 u32Utf8Char)const
{

   tU8 u8Low=0;
   tU8 u8High=ipod_tclUtf8StringBase::u32NumOfEntriesThai;
   tU8 u8Mid = (u8Low + u8High) / 2 ;
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8Low =",u8Low);
   //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8High =",u8High);
   do
   {
      u8Mid= (u8Low + u8High) / 2;
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8Mid =",u8Mid);
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"TableThai[u8Mid][TABLE_COL_SORT_POS] =",TableThai[u8Mid][TABLE_COL_SORT_POS]);
      if ( u32Utf8Char < TableThai[u8Mid][TABLE_COL_SORT_POS] )
      {
         u8High = u8Mid - 1;
         //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8High =",u8High);
      }
      else if ( u32Utf8Char > TableThai[u8Mid][TABLE_COL_SORT_POS])
      {
         u8Low = u8Mid + 1;
         //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"u8Low =",u8Low);
      }
   }
   while( u32Utf8Char!=TableThai[u8Mid][TABLE_COL_SORT_POS] && u8Low <= u8High);   /* End of do- while */

   if( u32Utf8Char == TableThai[u8Mid][TABLE_COL_SORT_POS] )
   {
      //TRACE_DBG_LSN(TRC::Fnu32ignSplWordAtBegin,TR_LEVEL_USER_3,"TableThai[u8Mid][TABLE_COL_SORT_VAL] =",TableThai[u8Mid][TABLE_COL_SORT_VAL]);
      return TableThai[u8Mid][TABLE_COL_SORT_VAL];
   }
   else
   {
      return 0;
   }
}
