/************************************************************************
* FILE:         midw_common_trace.cpp
* PROJECT:      VW_LL_NF
* SW-COMPONENT: DemoServer
*----------------------------------------------------------------------
*
* DESCRIPTION: handle trace outputs efficiently
* This class is used as template for all middleware server to provide
* trace output in an efficient way.
*
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2007 Robert Bosch GmbH, Hildesheim
*************************************************************************/

#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include "ahl_if.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "midw_common_trace_macros.h"
#include "midw_common_trace.h"
#include "midw_common_trace_input.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_MIDW_COMMON_TRACE
#include "trcGenProj/Header/midw_common_trace.cpp.trc.h"
#endif

OSAL_tIODescriptor midw_common_tclTraceChannel::_fdTrace = OSAL_ERROR;
midw_common_tclTraceInput* midw_common_tclTraceChannel::_paReceiver[midw_common_maxClassNames];

/******************************************************FunctionHeaderBegin******
 * CLASS       : midw_common_tclTrace
 * FUNCTION    : constructor
 * CREATED     : 2007-09-10
 * AUTHOR      : Matthias Hessling
 * DESCRIPTION : initialization to no action
 * SYNTAX      :
 * ARGUMENTS   : -
 * RETURN VALUE: -
 * NOTES       : -
 *******************************************************FunctionHeaderEnd******/
midw_common_tclTraceChannel::midw_common_tclTraceChannel(tVoid)
   :_eChannel(NO_CHAN_SELECTED)
{
   ETG_TRACE_USR4(("FnConstructor()"));
   // check if channel was opened already by another instance
   if(_fdTrace != OSAL_ERROR)
   {
      ETG_TRACE_USR3(("FnConstructor() registering call back"));
      vUnregisterTraceCallBack();
   }
   _fdTrace = OSAL_ERROR;
   for (tInt i = 0; i < (tInt)midw_common_maxClassNames; i++)
   {
      _paReceiver[i] =0;
   }
}

midw_common_tclTraceChannel::~midw_common_tclTraceChannel(tVoid)
{
   ETG_TRACE_USR4(("FnDestructor()"));

   try
   {
      if(_fdTrace != OSAL_ERROR)
      {
         ETG_TRACE_USR3(("FnDestructor() unregistering call back"));
         vUnregisterTraceCallBack();
      }
      _fdTrace = OSAL_ERROR;
   }
   catch (...)
   {
   }
   
}
/******************************************************FunctionHeaderBegin******
 * CLASS       : midw_common_tclTraceChannel::vTrace_Rx (tPCUChar pcu8Data)
 * FUNCTION    : vTrace_Rx
 * CREATED     : 2007-09-10
 * AUTHOR      : Matthias Hessling
 * DESCRIPTION : handles received messages from TTFIS
 * SYNTAX      : tVoid vTrace_vRx( tU8* pu8Data )
 * ARGUMENTS   : -
 *               pu8Data
 * RETURN VALUE: -
 *               none
 * NOTES       :   -
 *******************************************************FunctionHeaderEnd******/
tVoid  midw_common_tclTraceChannel::vTrace_Rx (tPCUChar pcu8Data)
{
   ETG_TRACE_USR4(("FnTraceRx()"));
   if(pcu8Data==0)
   {
      return;
   }
   // pu8Data[0] == length of bytes following
   if ( pcu8Data[0] > 0 )
   {
      // the router forwards the message to that call backs as defined by tclApp
      if(_paReceiver[pcu8Data[1]])
      {
         ETG_TRACE_USR4(("FnTraceRx() calling trace client.."));
         _paReceiver[pcu8Data[1]]->vTraceRx(pcu8Data);
      }
      else
      {
         ETG_TRACE_SYS(("FnTraceRx() Trace Channel this service is not initialized=%u" , (tU8)pcu8Data[1]));
      }
   }
   else
   {
      ETG_TRACE_SYS(("FnTraceRx() Trace Channel service number unknown=%u" , (tU8)pcu8Data[1]));
   }
   return;
}

/******************************************************FunctionHeaderBegin******
 * CLASS       : vRegisterTraceService
 * FUNCTION    : vRegisterTraceService
 * CREATED     : 2008-03-15
 * AUTHOR      : Matthias Hessling
 * DESCRIPTION : adds the given trace input interface into the function array
 * ARGUMENTS   : -
 * RETURN VALUE: -
 * NOTES       :   -
 *******************************************************FunctionHeaderEnd******/
tVoid midw_common_tclTraceChannel::vRegisterTraceService(midw_common_tenTrcClassName uInputNum,
      midw_common_tclTraceInput* pclInput)
{
   ETG_TRACE_USR4(("FnRegisterTraceService()"));
   if((int)uInputNum < (sizeof(_paReceiver)/sizeof(_paReceiver[0])))
   {
      _paReceiver[(int)uInputNum] = pclInput;
   }
}//lint !e1762 prio3 reviewed non-const RFU

/******************************************************FunctionHeaderBegin******
 * CLASS       : midw_common_tclTraceChannel
 * FUNCTION    : vRegisterTraceCallBack
 * CREATED     : 2008-03-15
 * AUTHOR      : Matthias Hessling
 * DESCRIPTION : register the call back function at the OSAL trace interface
 * ARGUMENTS   : -
 * RETURN VALUE: -
 * NOTES       :   -
 *******************************************************FunctionHeaderEnd******/
tVoid midw_common_tclTraceChannel::vRegisterTraceCallBack(TR_tenTraceChan eChan )
{
   ETG_TRACE_USR4(("FnRegisterTraceCallBack()"));
   tS32 s32Error = OSAL_ERROR;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;
   _eChannel = eChan;
   oTraceChannel.enTraceChannel = _eChannel;
   oTraceChannel.pCallback = (OSAL_tpfCallback)vTrace_Rx;
   _fdTrace = OSAL_IOOpen( OSAL_C_STRING_DEVICE_TRACE, OSAL_EN_READWRITE );

   if(_fdTrace != OSAL_ERROR)
   {
      s32Error = OSAL_s32IOControl( _fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_REG, (intptr_t)&oTraceChannel );
   }
   if( OSAL_OK != s32Error )
   {
      ETG_TRACE_ERR(("FnTraceRx() Couldn't register trace channel! "));
   }
   else
   {
      ETG_TRACE_USR1(("FnTraceRx() Trace callback successfully registered "));
   }
}

/******************************************************FunctionHeaderBegin******
 * CLASS       : rfd_tclTraceChannel
 * FUNCTION    : vUnregisterTraceCallBack
 * CREATED     : 2008-03-15
 * AUTHOR      : Matthias Hessling
 * DESCRIPTION : unregister the call back at trace interface
 * ARGUMENTS   : -
 * RETURN VALUE: -
 * NOTES       :   -
 *******************************************************FunctionHeaderEnd******/
tVoid midw_common_tclTraceChannel::vUnregisterTraceCallBack(tVoid)
{
   ETG_TRACE_USR4(("FnUnregisterTraceCallBack()"));
   tS32 s32Error = OSAL_ERROR;
   OSAL_trIOCtrlLaunchChannel  oTraceChannel;

   oTraceChannel.enTraceChannel = _eChannel;
   oTraceChannel.pCallback = (OSAL_tpfCallback)vTrace_Rx;

   if(_fdTrace != OSAL_ERROR)
   {
      s32Error = OSAL_s32IOControl( _fdTrace, OSAL_C_S32_IOCTRL_CALLBACK_UNREG, (intptr_t)&oTraceChannel );
      ETG_TRACE_USR3(("FnUnregisterTraceCallBack() OSAL_s32IOControl() returned %d",
                      s32Error ));
   }
   if( OSAL_OK != s32Error )
   {
      ETG_TRACE_ERR(("FnUnregisterTraceCallBack() Couldn't unregister trace channel" ));
   }
   OSAL_s32IOClose( _fdTrace );
}//lint !e1762 prio3 reviewed non-const RFU
