///////////////////////////////////////////////////////////
//  ConnectionConfigurator.cpp
//  Implementation of the Class ConnectionConfigurator
//  Created on:      12-Sep-2015 13:21:20
//  Original author: MOR2HI
///////////////////////////////////////////////////////////
#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "../../midw_common_trace.h"
#include "ConnectionConfigurator.h"

//For ETG tracing
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_MIDW_COMMON_AUTO_CONFIGURATION
#include "trcGenProj/Header/ConnectionConfigurator.cpp.trc.h"
#endif

/**
 * This constructor is giving the reference for storing the history.
 * This constructor is defensive. Means it'll configure the objects as auto configuration per default. This is because the configurator will be initialized in legacy components which don't initialize the configurator yet.
 * @param VehicleInfoByteContainingAvailabilityType: The vehicle
 * @param KDSEntityMask: bit mask for masking the bits the client is interested in
 * @param pHistoryReferenz
 */
ConnectionConfigurator::ConnectionConfigurator(ConnectionConfiguratorCallbackInterface* pCallbackInterface) :
      _AvailabilityType(MCAN_CHANNEL_AVAILABILITY_CHANNEL_AVAILABLE),
      _ChannelConnected(FALSE),
      oHistoryContainer(pCallbackInterface)
{
   ETG_TRACE_USR4(("ConnectionConfigurator::ConnectionConfigurator"));
}

/**
 * This constructor is defensive. Means it'll configure the objects as auto configuration per default. This is because the configurator will be initialized in legacy components which don't initialize the configurator yet.
 */
ConnectionConfigurator::ConnectionConfigurator() :
      _AvailabilityType(MCAN_CHANNEL_AVAILABILITY_CHANNEL_AVAILABLE),
      _ChannelConnected(FALSE)
{
   ETG_TRACE_USR4(("ConnectionConfigurator::ConnectionConfigurator: default"));
}

ConnectionConfigurator::~ConnectionConfigurator()
{

}

/**
 * This function checks whether it's been allowed to connect the MCAN channel.
 * @return: true: it's allowed to open the channel
 */
tBool ConnectionConfigurator::isCommunicationGranted()
{
   ETG_TRACE_USR4(("ConnectionConfigurator::isCommunicationGranted: enter"));
   tBool ConnectionGranted = FALSE;
   if (
         (MCAN_CHANNEL_AVAILABILITY_AUTO_CONFIGURATION == _AvailabilityType)
         ||
         (MCAN_CHANNEL_AVAILABILITY_CHANNEL_AVAILABLE == _AvailabilityType)
         )
   {
      ConnectionGranted = TRUE;
   }
   ETG_TRACE_USR4(("ConnectionConfigurator::isCommunicationGranted: exit: availability type: %d, communication granted: %d", _AvailabilityType, ConnectionGranted));
   return (ConnectionGranted);
}


/**
 * Checks whether the channel shall be connected. This depends on the strategy. For auto configuration history need to be
 * checked.
 * This function is needed to check in case of timeouts whether it's ok that the connection was not established.
 */
// function is currently not needed
tBool ConnectionConfigurator::isChannelConnectionExpected()
{
   ETG_TRACE_USR4(("ConnectionConfigurator::isChannelConnectionExpected: enter"));
   tBool ConnectionExpected = FALSE;

   if (
         (MCAN_CHANNEL_AVAILABILITY_AUTO_CONFIGURATION == _AvailabilityType)
         ||
         (MCAN_CHANNEL_AVAILABILITY_CHANNEL_AVAILABLE == _AvailabilityType)
         )
   {
      ConnectionExpected = TRUE;
   }
   ETG_TRACE_USR4(("ConnectionConfigurator::isChannelConnectionExpected: exit: Connection Expected: %d", ConnectionExpected));
   return (ConnectionExpected);
}

// interface
tVoid ConnectionConfigurator::markChannelAvailable()
{
   _AvailabilityType = MCAN_CHANNEL_AVAILABILITY_CHANNEL_AVAILABLE;
   oHistoryContainer.grantHistory(FALSE);
}

tVoid ConnectionConfigurator::markChannelAutoAvailable()
{
   ETG_TRACE_USR4(("ConnectionConfigurator::markChannelAutoAvailable: enter"));
   _AvailabilityType = MCAN_CHANNEL_AVAILABILITY_AUTO_CONFIGURATION;
   oHistoryContainer.grantHistory(TRUE);
}

tVoid ConnectionConfigurator::markChannelNotAvailable()
{
   ETG_TRACE_USR4(("ConnectionConfigurator::markChannelNotAvailable: enter"));
   _AvailabilityType = MCAN_CHANNEL_AVAILABILITY_CHANNEL_DISABLED;
   oHistoryContainer.grantHistory(FALSE);
}

tBool ConnectionConfigurator::isChannelConnected ()
{
   return (_ChannelConnected);
}

tVoid ConnectionConfigurator::connectionEstablished()
{
   ETG_TRACE_USR4(("ConnectionConfigurator::connectionEstablished: enter"));
   _ChannelConnected = TRUE;

   if (MCAN_CHANNEL_AVAILABILITY_AUTO_CONFIGURATION == _AvailabilityType)
   {
      oHistoryContainer.setHistory();
   }
}

/**
 * Check whether it's o.k. to tell the diagnosis that something is wrong.
 * We tell it the diagnosis only, if the channel is configured the way, that it's been expected to open the channel.
 * After that we must take the auto configuration into account. Here we check whether the history is set. History set means, that the channel was succesful connected before.
 * If history is false, we are not telling anything the diagnosis, because it means channel is not valid.
 * @return true: we should inform diagnosis about channel connection errors.
 */
tBool ConnectionConfigurator::isErrorDetectionGranted()
{
   ETG_TRACE_USR4(("ConnectionConfigurator::isErrorDetectionGranted: enter"));
   tBool ErrorDetectionGranted = FALSE;
   if (TRUE == isChannelConnectionExpected())
   {
      // after we get to know that the channel shall be connected, no check whether history comes into picture
      if (TRUE == oHistoryContainer.isHistoryGranted())
      {
         // auto configuration takes place. Thus we write only the ITC, if the channel was successful connected before.
         if (TRUE == oHistoryContainer.isHistorySet())
         {
            ErrorDetectionGranted = TRUE;
         }
      }
      else
      {
         ErrorDetectionGranted = TRUE; // History is not granted. This means it’s a normal channel connection, not the auto configuration. Here we write any connection failure
      }
   }
   ETG_TRACE_USR4(("ConnectionConfigurator::isErrorDetectionGranted: exit: error detection granted: %d", ErrorDetectionGranted));
   return (ErrorDetectionGranted);
}
