/************************************************************************
 * language_converter.h
 *
 * Contains function definitions for conversion of language codes to UTF8,
 * together with an enum defining the set of language conversions
 * supported by the CodeConversion component.
 *
 * component: midw_common\CodeConversion
 *
 * author: Ian Lisney
 *
 ***********************************************************************/

#ifndef __LANGUAGE_CONVERTER_H__
#define __LANGUAGE_CONVERTER_H__

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h" /*lint !e451 !e537 */
/* Warning 451: Header file '...' repeatedly included but does not have a standard include guard */
/* Warning 537: Repeated include file '...' */

#define CODECONV_MAX_INPUTLENGTH 512
#define CODECONV_MAX_OUTPUTLENGTH (CODECONV_MAX_INPUTLENGTH * 4)

/* Code Conversion Languages supported */
typedef enum
{
   CODECONV_NO_CHANGE              = 0,

   CODECONV_ISO8859_01             = 1,  /* Latin Alphabet No. 1 - most Western European languages:
                                            Danish (partial), Dutch (partial), English, Faeroese,
                                            Finnish (partial)[3], French (partial), German, Icelandic,
                                            Irish, Italian, Norwegian, Portuguese, Rhaeto-Romanic,
                                            Scottish Gaelic, Spanish, and Swedish.
                                            Languages from other parts of the world are also covered,
                                            including: Eastern European Albanian, Southeast Asian
                                            Indonesian, as well as the African languages Afrikaans
                                            and Swahili. */
   CODECONV_ISO8859_02             = 2,  /* Latin 2 / Central European - Bosnian, Polish, Croatian, Czech, Slovak,
                                            Slovene, Serbian and Hungarian */
   CODECONV_ISO8859_03             = 3,  /* Latin 3 / South European - Turkish, Maltese and Esperanto */
   CODECONV_ISO8859_04             = 4,  /* Latin 4 / Baltic - Estonian, Latvian, Lithuanian, Greenlandic
                                            and Sami */
   CODECONV_ISO8859_05             = 5,  /* Cyrillic - Covers mostly Slavic languages that use a Cyrillic
                                            alphabet, including Belarusian, Bulgarian, Macedonian, Russian,
                                            Serbian, and Ukrainian (partial). */
   CODECONV_ISO8859_06             = 6,  /* Latin / Arabic */
   CODECONV_ISO8859_07             = 7,  /* Latin / Greek */
   CODECONV_ISO8859_08             = 8,  /* Latin / Hebrew */
   CODECONV_ISO8859_09             = 9,  /* Latin 5 / Turkish */
   CODECONV_ISO8859_10             = 10, /* Latin 6 / Scandinavian */
   CODECONV_ISO8859_11             = 11, /* Thai */
   CODECONV_ISO8859_12             = 12, /* Devanagari */
   CODECONV_ISO8859_13             = 13, /* Latin 7 / Baltic Rim */
   CODECONV_ISO8859_14             = 14, /* Latin 8 / Celtic - Gaelic, Breton */
   CODECONV_ISO8859_15             = 15, /* Latin-9 / Western European - completes coverage of French, Finnish and Estonian */
   CODECONV_ISO8859_16             = 16, /* Latin / South-Eastern European */
   CODECONV_CHINESE_UNICODE_1_1    = 17, /* Chinese Unicode 1.1 - 2 byte codes supported - 4 byte codes are valid but not supported */

   /* The following 16 Windows Code Pages are identical to ISO8859_1 to ISO8859_16 */
   CODECONV_WINDOWS_CODEPAGE_28591 = 18, /* Latin Alphabet No. 1 */
   CODECONV_WINDOWS_CODEPAGE_28592 = 19, /* Latin 2 / Central European */
   CODECONV_WINDOWS_CODEPAGE_28593 = 20, /* Latin 3 / South European */
   CODECONV_WINDOWS_CODEPAGE_28594 = 21, /* Latin 4 / Baltic */
   CODECONV_WINDOWS_CODEPAGE_28595 = 22, /* Cyrillic */
   CODECONV_WINDOWS_CODEPAGE_28596 = 23, /* Latin / Arabic */
   CODECONV_WINDOWS_CODEPAGE_28597 = 24, /* Latin / Greek */
   CODECONV_WINDOWS_CODEPAGE_28598 = 25, /* Latin / Hebrew */
   CODECONV_WINDOWS_CODEPAGE_28599 = 26, /* Latin 5 / Turkish */
   CODECONV_WINDOWS_CODEPAGE_28600 = 27, /* Latin 6 / Scandinavian */
   CODECONV_WINDOWS_CODEPAGE_28601 = 28, /* Thai */
   CODECONV_WINDOWS_CODEPAGE_28602 = 29, /* Devanagari */
   CODECONV_WINDOWS_CODEPAGE_28603 = 30, /* Latin 7 / Baltic Rim */
   CODECONV_WINDOWS_CODEPAGE_28604 = 31, /* Latin 8 / Celtic - Gaelic, Breton */
   CODECONV_WINDOWS_CODEPAGE_28605 = 32, /* Latin-9 / Western European */
   CODECONV_WINDOWS_CODEPAGE_28606 = 33, /* Latin / South-Eastern European */

   /* The following Windows Code Pages are different to the ISO Code Pages */

   CODECONV_WINDOWS_CODEPAGE_874   = 34, /* Thai */
   CODECONV_WINDOWS_CODEPAGE_932   = 35, /* Japanese Shift-JIS - 2 byte codes */
   CODECONV_WINDOWS_CODEPAGE_936   = 36, /* Simplified Chinese GBK  - 2 byte codes */
   CODECONV_WINDOWS_CODEPAGE_949   = 37, /* Korean - 2 byte codes */
   CODECONV_WINDOWS_CODEPAGE_950   = 38, /* Traditional Chinese Big 5 - 2 byte codes */
   CODECONV_WINDOWS_CODEPAGE_951   = 39, /* Traditional Chinese Big 5 Extensions - 2 byte codes */
   CODECONV_WINDOWS_CODEPAGE_1250  = 40, /* Central European */
   CODECONV_WINDOWS_CODEPAGE_1251  = 41, /* Cyrillic */
   CODECONV_WINDOWS_CODEPAGE_1252  = 42, /* Latin 1 */
   CODECONV_WINDOWS_CODEPAGE_1253  = 43, /* Latin / Greek */
   CODECONV_WINDOWS_CODEPAGE_1254  = 44, /* Latin 5 / Turkish */
   CODECONV_WINDOWS_CODEPAGE_1255  = 45, /* Latin / Hebrew */
   CODECONV_WINDOWS_CODEPAGE_1256  = 46, /* Latin / Arabic */
   CODECONV_WINDOWS_CODEPAGE_1257  = 47, /* Latin 7 / Baltic Rim */
   CODECONV_WINDOWS_CODEPAGE_1258  = 48, /* Vietnam */

   CODECONV_LAST                   = 49

} tenCodeConvMapping;

#ifdef __cplusplus
extern "C" {
#endif

/***********************************************************************
 * Converts a string of native language codes to UTF8.
 *
 * The output array must have the same or greater size than the input array.
 *
 * in  tU8* const*         sourceStart  pointer to language codes to convert
 * in  const tU8*          sourceEnd    pointer to end of language codes
 * in  tU8* const*         targetStart  pointer to output array
 * in  const tU8*          targetEnd    pointer to end of output array
 * in  tenCodeConvMapping  language     language coding identifier
 * out tU32                             -1 if failure
 ***********************************************************************/
tU32 ConvertLanguagetoUTF8 (tU8* const*        sourceStart,
                            const tU8*         sourceEnd,
                            tU8* const*        targetStart,
                            const tU8*         targetEnd,
                            tenCodeConvMapping language);

/***********************************************************************
 * Checks whether a string is valid UTF8.
 *
 * Handles 1, 2, 3 and 4 byte sequences.
 * No recovery from errors in input string
 *
 * in  const tU8*          input_string   pointer to string to check
 * in  tU32                string_length  count of characters in string
 * out tBool               TRUE if string is UTF8, FALSE if not
 ***********************************************************************/
tBool CheckStringIsUTF8 (const tU8* input_string, tU32 string_length);

#ifdef __cplusplus
}
#endif

#endif /* __LANGUAGE_CONVERTER_H__ */
