/**************************************************************************//**
* \file     CharMap.c
*
*           Provides a generic mechanism for code maps. It can be used to
*           convert between different encodings, e.g. Unicode, ISO8859-x,
*           Shift-JIS, EBU, etc. The code tables can be non-continuous and
*           of (almost) any size.
*
*           Restrictions
*           - The function interface is based on tU32 to avoid any
*             artificial limitiations. However, the ROM-optimised data tables
*             currently only support 16-bit values, which is sufficient for
*             international character sets, including Asian encodings.
*           - Value 0 is returned by default if no mapping is found. It would
*             be cleaner if the mapping function would explicitly indicate
*             success or failure to the caller. The caller should then decide
*             about the default value. Note that even the data generator fills
*             small gaps with dummy mappings that have the result 0 to reduce
*             the number of ranges.
*
*           Data Generator
*           Any number of code maps can be defined in an Excel file. The Excel
*           file is parsed by a Python script, which generates a .dat file.
*           The .dat file is included by the CharMap implementation files.
*           See <t.b.d.> for working examples.
*
*           Performance
*           - The generated data is first optimised for ROM size.
*           - The execution speed is considered reasonable. It mainly depends
*             on the size of the range table in the .dat file. The search
*             mechanism is currently linear. It could be optimized by using
*             "binary search" in the sorted range table. However, you would
*             also need to consider how frequent certain characters are in
*             a random text. In a western text for instance, you would mainly
*             find ASCII characters which are located at the top of the range
*             table. Thus, linear search would find them very quickly.
*
*           Considerations
*           - rename to UTIL_CodeMap because it is actually not restricted to
*             character maps; not clear how to deal with the .dat files; they
*             would certainly have to be branched depending on project needs
*           - provide separate functions for selecting a code map and for
*             mapping a code to save call overhead and improve performance
*
*           You can find more details on character mapping in:
*           \\view\doc_view\cr_hmi_unit\_doc\Concepts\Concept_HMI_Fonts.doc
*
* \remark   Copyright: 2009 Robert Bosch Car Multimedia GmbH, Hildesheim
* \remark   Scope:     VAG HMI
*
* \author   - CM-AI/PJ-VW66 Janssen
******************************************************************************/
#include "CharMap.h"

enum
{
   RANGE_TYPE_OFFSET,
   RANGE_TYPE_WORD,
   RANGE_TYPE_BYTE
};


typedef struct _CharRange
{
   tU16 uwStart;
   tU16 uwSize    : 12;
   tU16 uwType    :  4;
   tU16 uwOffset;
} CharRange;


typedef struct _CharMap
{
   const CharRange* aRanges;
   const tU8* pubAttachments;
   const tU16 aSize;
} CharMap;


/**
 *  These macros generate the range arrays. Note - generated names need to be less than 31 charas to keep lint happy
 */
#define RANGE_LIST_BEGIN(map)                 static const CharRange R##map[] = {
#define RANGE(start, length, type, offset)    { start, length, type, offset },
#define RANGE_LIST_END                        { 0, 0, RANGE_TYPE_OFFSET, 0 } };  /* marks end of list */
#include "CharMapVar.h" /*lint !e451 !e537 */
/* Warning 451: Header file '...' repeatedly included but does not have a standard include guard */
/* Warning 537: Repeated include file '...' */
#undef  RANGE_LIST_BEGIN
#undef  RANGE
#undef  RANGE_LIST_END


/**
 *  These macros generate the arrays containing the attachment data. Note - generated names need to be less than 31 charas to keep lint happy
 */
#define RANGE_ATTACHMENTS_BEGIN(map)  static const tU8 A##map[] = {
#define RANGE_ATTACHMENTS_END         0 };
#include "CharMapVar.h" /*lint !e451 !e537 */
/* Warning 451: Header file '...' repeatedly included but does not have a standard include guard */
/* Warning 537: Repeated include file '...' */
#undef  RANGE_ATTACHMENTS_BEGIN
#undef  RANGE_ATTACHMENTS_END


/*
 *  An empty map to support NULL-Object-Pattern for maps.
 */
static const CharRange emptyRangeList[] =
{
   { 0, 0, RANGE_TYPE_OFFSET, 0 }
};


/**
 *  These macros generate a list of character maps.
 */
static const CharMap aCharMaps[] =
{
#define MAP(map)                { R##map, A##map, sizeof R##map },
#include "CharMapVar.h" /*lint !e451 !e537 */
   /* Warning 451: Header file '...' repeatedly included but does not have a standard include guard */
   /* Warning 537: Repeated include file '...' */
#undef  MAP
   { emptyRangeList, NULL, 0 }    /* for CHAR_MAP_NONE */
};


static const CharMap* pCurrentMap;


/**************************************************************************//**
* Selects a particular code map that will be used for any subsequent value
* conversion. The map selection has been separated from the value conversion
* so that the value conversion does not need to look-up the map for each value.
******************************************************************************/
void CharMap_vSelectMap(MapIdEnum enMapId)
{
   if (enMapId >= MAP_ID_LIMIT)
   {
      FATAL_M_ASSERT_ALWAYS();  /* invalid id */
      enMapId = MAP_NONE;
   }
   pCurrentMap = &aCharMaps[enMapId];
}


/**************************************************************************//**
* Converts an input value using the offset-range algorithm. The new value is
* calculated by adding the offset. The resulting value may even be smaller
* than the input value due to the wrap-around at 0x10000.
******************************************************************************/
static tU32 CharMap_ulwTransposeOffsetRange(const CharRange* pRange, tU32 ulwChar)
{
   return (ulwChar + pRange->uwOffset) & 0xFFFF;
}


/**************************************************************************//**
* Converts an input value using the word-range algorithm. A word range stores
* the entire target value in two subsequent bytes. Gaps in the range may be
* filled with 0, i.e. two zero-bytes.
******************************************************************************/
static tU32 CharMap_ulwTransposeWordRange(const CharRange* pRange, tU32 ulwChar)
{
   const tU32 ulwCharIndex = ulwChar - pRange->uwStart;
   const tU32 ulwAttachmentIndex = pRange->uwOffset + ((ulwCharIndex) * 2);
   const tU8 ubHigh = pCurrentMap->pubAttachments[ulwAttachmentIndex];
   const tU8 ubLow = pCurrentMap->pubAttachments[ulwAttachmentIndex + 1];
   return (ubHigh << 8) + ubLow;
}


/**************************************************************************//**
* Converts an input value using the byte-range algorithm. A byte range stores
* a target base value in the first two bytes of the attachment data. For each
* input value there is a byte offset that is added to the base value. Gaps in
* a byte range are marked with 0xFF. For such values 0 is returned.
******************************************************************************/
static tU32 CharMap_ulwTransposeByteRange(const CharRange* pRange, tU32 ulwChar)
{
   const tU32 ulwCharIndex = ulwChar - pRange->uwStart;
   const tU32 ulwAttachmentIndex = pRange->uwOffset + 2 + ulwCharIndex;
   const tU8 ubBaseOffset = pCurrentMap->pubAttachments[ulwAttachmentIndex];

   if (ubBaseOffset != 0xFF)
   {
      const tU8 ubBaseHigh = pCurrentMap->pubAttachments[pRange->uwOffset];
      const tU8 ubBaseLow = pCurrentMap->pubAttachments[pRange->uwOffset + 1];
      const tU32 ulwBase = (ubBaseHigh << 8) + ubBaseLow;
      return ulwBase + ubBaseOffset;
   }
   else
   {
      return 0;   /* it's a gap - no replacement for this value */
   }
}


/**************************************************************************//**
*
******************************************************************************/
static tU32 CharMap_ulwTransposeRange(const CharRange* pRange, tU32 ulwChar)
{
   switch (pRange->uwType)
   {
   case RANGE_TYPE_OFFSET:
      return CharMap_ulwTransposeOffsetRange(pRange, ulwChar);

   case RANGE_TYPE_WORD:
      return CharMap_ulwTransposeWordRange(pRange, ulwChar);

   case RANGE_TYPE_BYTE:
      return CharMap_ulwTransposeByteRange(pRange, ulwChar);

   default:
      FATAL_M_ASSERT_ALWAYS();  /* invalid range type */
      return 0;
   }
}


/**************************************************************************//**
* Returns the mapped code for "ulwChar" in the currently selected map. The map
* must have been selected with CharMap_vSelectMap() before. If "ulwChar"
* cannot be found in the map, 0 is returned.
******************************************************************************/
tU32 CharMap_ulwTranspose(tU32 ulwChar)
{
   const CharRange* pRange = pCurrentMap->aRanges;
   tU32 bottom = 0;
   tU32 median = 0;
   tU32 top = (pCurrentMap->aSize / (sizeof emptyRangeList)) - 1;

   while (bottom + 1 <= top)
   {
      median = (top + bottom) >> 1;

      /* Check if ulwChar is within the range pointed at by median */
      if (ulwChar < (tU32) ((pRange + median)->uwStart + (pRange + median)->uwSize) &&
            ulwChar >= (tU32) ((pRange + median)->uwStart))
      {
         /* Found it - convert ulwChar and return */
         return CharMap_ulwTransposeRange((pRange + median), ulwChar);
      }

      /* Had to put this additional check in to cater for unusual case of
         doing a binary search on a single entry table */
      if (top == bottom + 1)
      {
         return 0;
      }

      if (ulwChar < (tU32) ((pRange + median)->uwStart))
      {
         top = median;
      }
      else
      {
         bottom = median;
      }
   }
   return 0;     /* no mapping - value falls outside all ranges */
}

