/*!
*******************************************************************************
* \file              ihl_tclThreader.h
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    Thread Utility Class - Threads as objects
*                    Command Processor Model Thread class
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 30.06.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Updated event handling to separate out Application events.
*
* \date 06.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef IHL_TCLTHREADER_H_
#define IHL_TCLTHREADER_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

// Includes to utilize the OSAL Interface
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

// Includes to utilize the BP STL exceptional handler



#define IHL_S_IMPORT_INTERFACE_FI_HELPERS
#include <ihl_if.h>

#include "ihl_tclThreadable.h"

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

#define IHL_THREAD_TERMINATE_EVENT     (0x80000000)
#define IHL_THREAD_NEW_MSG_EVENT       (0x40000000)
#define IHL_THREAD_APP_EVENTS          (~(IHL_THREAD_TERMINATE_EVENT | IHL_THREAD_NEW_MSG_EVENT))
#define IHL_THREAD_ALL_EVENTS          ((~IHL_THREAD_APP_EVENTS)|IHL_THREAD_APP_EVENTS)

#define IHL_THREAD_SYNC_TIME           (1000)
#define IHL_THREAD_SYNC_TIME_CHECK(i1) ((i1<=2000)?i1:2000)
#define IHL_THREAD_STACK_SIZE          (4096)   // 4 KB

/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| forward declarations
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace thread {

      using namespace ihl::fi_helpers::midw_fi;

/*! 
 * \struct ihl_rThreadAttribute 
 * \brief This structure is similar to the OSAL Thread attribute struture
 * OSAL_trThreadAttribute, however is customized to fit the requirements of 
 * ihl thread attributes.
 */

typedef struct ihl_rThreadAttribute
{
   /// Thread Attributes
   OSAL_trThreadAttribute m_oOsalThreadAtt;
   /// Terminate Sync time in milliseconds should be <= 2 seconds
   tU32 m_u32SyncTime;

   ihl_rThreadAttribute():m_oOsalThreadAtt(), m_u32SyncTime(IHL_THREAD_SYNC_TIME)
   {
      tU8 u8Buffer[OSAL_C_U32_MAX_NAMELENGTH]  =  {0};
      // Autogenerate the thread name.
      OSAL_vRandomSeed(OSAL_ClockGetElapsedTime());  // Randomize
      //OSAL_s32PrintFormat((tChar*)&u8Buffer[0], "%x_%x_", OSAL_ThreadWhoAmI(), OSAL_s32Random()); 
      snprintf((tChar*)&u8Buffer[0], OSAL_C_U32_MAX_NAMELENGTH, "%x_%x_", OSAL_ThreadWhoAmI(), OSAL_s32Random()); 
      u8Buffer[OSAL_C_U32_MAX_NAMELENGTH - 1]   =  '\0';    // Null Termination protection
      ihl_tclFiString oThreadName((tCString)&u8Buffer[0]);
      oThreadName += "Worker";
      // Deep copy the Thread name
      m_oOsalThreadAtt.szName        =     OSAL_NULL;
      oThreadName.vClone(m_oOsalThreadAtt.szName);

      NORMAL_M_ASSERT(OSAL_NULL != m_oOsalThreadAtt.szName);
      m_oOsalThreadAtt.u32Priority   =     OSAL_C_U32_THREAD_PRIORITY_NORMAL;
      m_oOsalThreadAtt.s32StackSize  =     IHL_THREAD_STACK_SIZE;
      m_oOsalThreadAtt.pfEntry       =     OSAL_NULL;
      m_oOsalThreadAtt.pvArg         =     OSAL_NULL;
   }

   virtual ~ihl_rThreadAttribute()
   {
      OSAL_DELETE [] m_oOsalThreadAtt.szName;
      m_oOsalThreadAtt.szName  =  OSAL_NULL;
   }

   ihl_rThreadAttribute(OSAL_trThreadAttribute oThreadAttrib
      , tU32 u32SyncTime = IHL_THREAD_SYNC_TIME):m_oOsalThreadAtt(), m_u32SyncTime(u32SyncTime)
   {
      if (OSAL_NULL != oThreadAttrib.szName)
      {
         ihl_tclFiString oThreadName(oThreadAttrib.szName);
         // Deep copy the Thread name
         m_oOsalThreadAtt.szName     =  OSAL_NULL;
         oThreadName.vClone(m_oOsalThreadAtt.szName);
      }
      else
      {
         tU8 u8Buffer[OSAL_C_U32_MAX_NAMELENGTH]  =  {0};
         // Autogenerate the thread name.
         OSAL_vRandomSeed(OSAL_ClockGetElapsedTime());  // Randomize
         //OSAL_s32PrintFormat((tChar*)&u8Buffer[0], "%x_%x_", OSAL_ThreadWhoAmI(), OSAL_s32Random());
         snprintf((tChar*)&u8Buffer[0], OSAL_C_U32_MAX_NAMELENGTH, "%x_%x_", OSAL_ThreadWhoAmI(), OSAL_s32Random());
         u8Buffer[OSAL_C_U32_MAX_NAMELENGTH - 1]   =  '\0';    // Null Termination protection
         ihl_tclFiString oThreadName((tCString)&u8Buffer[0]);
         oThreadName += "Worker";
         // Deep copy the Thread name
         m_oOsalThreadAtt.szName     =    OSAL_NULL;
         oThreadName.vClone(m_oOsalThreadAtt.szName);
      }

      NORMAL_M_ASSERT(OSAL_NULL != m_oOsalThreadAtt.szName);

      m_oOsalThreadAtt.u32Priority   =     oThreadAttrib.u32Priority;
      m_oOsalThreadAtt.s32StackSize  =     oThreadAttrib.s32StackSize;
      m_oOsalThreadAtt.pfEntry       =     OSAL_NULL;
      m_oOsalThreadAtt.pvArg         =     OSAL_NULL;
      // Thread sync time correction.
      m_u32SyncTime = IHL_THREAD_SYNC_TIME_CHECK(m_u32SyncTime);
   }

   ihl_rThreadAttribute(const ihl_rThreadAttribute& rfcoThreadAttrib)
      :m_oOsalThreadAtt(), m_u32SyncTime(rfcoThreadAttrib.m_u32SyncTime)
   {
      ihl_tclFiString oThreadName(rfcoThreadAttrib.m_oOsalThreadAtt.szName);
      // Deep copy the Thread name
      m_oOsalThreadAtt.szName        =  OSAL_NULL;
      oThreadName.vClone(m_oOsalThreadAtt.szName);
      NORMAL_M_ASSERT(OSAL_NULL != m_oOsalThreadAtt.szName);

      m_oOsalThreadAtt.u32Priority   =  rfcoThreadAttrib.m_oOsalThreadAtt.u32Priority;
      m_oOsalThreadAtt.s32StackSize  =  rfcoThreadAttrib.m_oOsalThreadAtt.s32StackSize;
      m_oOsalThreadAtt.pfEntry       =  rfcoThreadAttrib.m_oOsalThreadAtt.pfEntry;
      m_oOsalThreadAtt.pvArg         =  rfcoThreadAttrib.m_oOsalThreadAtt.pvArg;
   }
   
   ihl_rThreadAttribute& operator=(const ihl_rThreadAttribute& rfcoThreadAttrib)
   {
      // Identity Test - Self assignment test.
      if (this != &rfcoThreadAttrib)
      {
         ihl_tclFiString oThreadName(rfcoThreadAttrib.m_oOsalThreadAtt.szName);
         // Deep copy the Thread name
         oThreadName.vClone(m_oOsalThreadAtt.szName);
         NORMAL_M_ASSERT(OSAL_NULL != m_oOsalThreadAtt.szName);

         m_oOsalThreadAtt.u32Priority  =  rfcoThreadAttrib.m_oOsalThreadAtt.u32Priority;
         m_oOsalThreadAtt.s32StackSize =  rfcoThreadAttrib.m_oOsalThreadAtt.s32StackSize;
         m_oOsalThreadAtt.pfEntry      =  rfcoThreadAttrib.m_oOsalThreadAtt.pfEntry;
         m_oOsalThreadAtt.pvArg        =  rfcoThreadAttrib.m_oOsalThreadAtt.pvArg;
         m_u32SyncTime                 =  rfcoThreadAttrib.m_u32SyncTime;
      }
      return *this;
   }
} ihl_tThreadAttribute;

/*! 
 * \struct ihl_rEventFieldAttribute 
 * \brief Event field attributes, especially event mask, mask flags and timeout
 */

typedef struct ihl_rEventFieldAttribute
{
   /// Event Mask
   OSAL_tEventMask m_evntMask;
   /// Event Mask Flags
   OSAL_tenEventMaskFlag m_enFlags;
   /// Event Field time out in milliseconds
   OSAL_tMSecond m_evntTimeOut;

   ihl_rEventFieldAttribute(OSAL_tenEventMaskFlag enFlags = OSAL_EN_EVENTMASK_OR
      , OSAL_tMSecond evntTimeOut = OSAL_C_TIMEOUT_FOREVER)
      :m_evntMask(IHL_THREAD_ALL_EVENTS), m_enFlags(enFlags)
      , m_evntTimeOut(evntTimeOut){}
   
   ihl_rEventFieldAttribute(const ihl_rEventFieldAttribute& rfcoEvtFieldAttrib)
      :m_evntMask(IHL_THREAD_ALL_EVENTS), m_enFlags(rfcoEvtFieldAttrib.m_enFlags)
      , m_evntTimeOut(rfcoEvtFieldAttrib.m_evntTimeOut){}

 } ihl_tEventFieldAttribute;


/*! 
 * \class ihl_tclThreader
 * \brief Command Processor Model Thread class
 *
 * The class design is based on Object oriented Threading based on "Delegation
 * or Forwarding based" Threading approach.
 * 
 * \addtogroup DelegateThreading
 * \image html Threading.jpg "Delegation based OO Threading Approach" 
 *
 * \par 
 * A thread is an identifiable flow of control within a process, with its own 
 * stack and thread execution dependent on platform and program priorities & 
 * policies. Thread execution is normally treated as the asynchronous execution
 * of a function; however threading is inherently task oriented.
 *
 * \par 
 * In order to overcome some of the drawbacks of the primitive free-threaded 
 * design, where threads are visible to other threads, along with their 
 * synchronization primitives, UAM implements thread wrapping. Thread wrapping 
 * is a powerful concept which brings in higher encapsulation, scalable, 
 * testable threading. 
 *
 * \par 
 * At the object level there are essentially two approaches for encapsulating 
 * threading APIs:
 *    -  The inheritance-based approach endows an active object with threadedness 
 *       as part of its type.
 *    -  The delegation-based approach endows an active object with threadedness  
 *       by association.
 *
 * \par 
 * IHL uses the delegation based approach to implement threading for the worker 
 * threads, as it is more loosely coupled than the inheritance-based approach. 
 * This is based on Command pattern, where each task can be thought of as a 
 * command object. 
 * 
 * \par 
 * Command pattern brings in the following advantages:
 *    -  Parameterize objects by an action to perform. Commands are an object-
 *       oriented replacement for callbacks. It decouples the object that invokes
 *       the operation from the one that knows how to perform it.
 *    -  Commands are first-class objects. They can be manipulated and extended 
 *       like any other object.
 *    -  Assembling commands into a composite command is possible.
 *    -  Specify, queue, and execute requests at different times. A Command object 
 *       can have a lifetime independent of the original request. If the receiver
 *       of a request can be represented in an address space independent way, 
 *       then you can transfer a command object for the request to a different  
 *       process and fulfill the request there.
 *    -  Structure a system around high-level operations built on primitives 
 *       operations. Such a structure is common in information systems that 
 *       support transactions. A transaction encapsulates a set of changes to 
 *       data. Command pattern offers a way to model transactions. Commands have
 *       a common interface,letting you invoke all transactions the same way. 
 *       The pattern also makes it easy to extend the system with new transactions.
 *    -  Easier to add new Commands, as existing classes don't have to be changed.
 * 
 */

class ihl_tclThreader
{
public:
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclThreader::ihl_tclThreader(const OSAL_trThre..
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor, based on Dependency Injection 
    *          Principle (DIP)
    * \param   [rfcoThreadAttrib]:     (I)   Thread attributes
    * \param   [rfcoEvtFieldAttrib]:   (I)   Event Field attributes
    * \param   [cpoThreadable]:        (->I) Command Object
    * \param   [bExecThread]:          (I)   Flag to signal whether to spawn the 
    *          thread immediately after creation or not?
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclThreader
   (
      const ihl_tThreadAttribute &rfcoThreadAttrib
      , const ihl_rEventFieldAttribute &rfcoEvtFieldAttrib
      , ihl_tclThreadable* const cpoThreadable
      , tBool bExecThread  =  TRUE
   );

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclThreader::~ihl_tclThreader()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclThreader();

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclThreader::bEventPost(OSAL_tEventMask ev..
   ***************************************************************************/
   /*!
    * \brief   External interface to post event to this thread.
    * \param   [evtEvent]:    (I)   Event 
    * \param   [enFlags]:     (I)   Event Flag for posting event
    * \retval  [tBool]: TRUE, if event post was successful, FALSE otherwise.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bEventPost
   (
      OSAL_tEventMask evtEvent 
      , OSAL_tenEventMaskFlag enFlags = OSAL_EN_EVENTMASK_OR
   );

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclThreader::bRunThread()
   ***************************************************************************/
   /*!
    * \brief   External interface to spawn the thread.
    * \param   NONE
    * \retval  TRUE, In case the thread is spawned, FALSE otherwise.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bRunThread();

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:
   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  virtual tVoid ihl_tclThreader::vExecute()
   ***************************************************************************/
   /*!
    * \brief   Command interface to execute the commands using command objects.
    * \param   NONE
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tVoid vExecute();

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclThreader::vEventClose()
   ***************************************************************************/
   /*!
    * \brief   Function to perform event handle cleanup.
    * \param   NONE
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vEventClose();

   /***************************************************************************
   ** FUNCTION:  tBool ihl_tclThreader::bAllocResrc(tCString cszName)
   ***************************************************************************/
   /*!
    * \brief   Function to create event fields and semaphores required for 
    *          synchronization between creator & worker threads.
    * \param   [cszName]: (->I) Base name for event name & semaphore name 
    *          generation
    * \retval  [tBool]: TRUE, if all the resources are created, FALSE otherwise.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tBool bAllocResrc(tCString cszName);

   /***************************************************************************
   ** FUNCTION:  ihl_tclThreader::ihl_tclThreader()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable default 
    *          construction.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclThreader();
   
   /***************************************************************************
   ** FUNCTION:  ihl_tclThreader::ihl_tclThreader(const ihl_tclThreader..
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoThreader]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So 
    *          that any attempt to copy will be caught by the compiler.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclThreader(const ihl_tclThreader &rfcoThreader);

   /***************************************************************************
   ** FUNCTION:  ihl_tclThreader& ihl_tclThreader::operator=(const ihl_tclThr..
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoThreader]: (I) Const reference to object to be copied
    * \retval  [ihl_tclThreader&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So 
    *          that any attempt for assignment will be caught by the compiler.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclThreader& operator=(const ihl_tclThreader &rfcoThreader);

   /*!
    * \addtogroup tclMem
    */
   /*! @{*/

   /// Event Field attributes
   const ihl_rEventFieldAttribute m_coEvtFieldAttrib;
   /// Flag to keep track of whether the thread is alive or not?
   tBool m_bThreadAlive;
   /// Thread id
   OSAL_tThreadID m_oThreadId;             
   /// Event Handle
   OSAL_tEventHandle m_hEvntHandle;
   /// Event field name
   ihl_tclFiString m_oEventName;             
   /// Semaphore Handle
   OSAL_tSemHandle m_hSemHandle;              
   /// Sempahore name
   ihl_tclFiString m_oSemName;               
   /// Reference to the Threadable.
   ihl_tclThreadable* const m_cpoThreadable;       
   /// Thread attributes
   ihl_tThreadAttribute m_oThreadAttrib;

   /*! @}*/

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  static tVoid ihl_tclThreader::vStartThread(tVoid *pvArg)
   ***************************************************************************/
   /*!
    * \brief   Thread Entry Function
    * \param   [pvArg]:    (->I) "this" pointer as void pointer
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   static tVoid vStartThread(tVoid *pvArg);

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/
}; // class ihl_tclThreader

   }  // namespace thread
}  // namespace ihl

#endif   // #ifndef IHL_TCLTHREADER_H_

// <EOF>
