/*!
*******************************************************************************
* \file              ihl_tclFileHandler.cpp
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    File Handler - Exception safe file handler
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 12.09.2011 \version 1.2 \author Raghavendra S (RBEI/ECV2)
* - Correction for renaming interface.
*
* \date 11.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added support for file renaming.
*
* \date 06.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "ihl_tclFileHandler.h"

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace io {

/******************************************************************************
** FUNCTION:  ihl_tclFileHandler::ihl_tclFileHandler(tCString csFile..)
******************************************************************************/

/*explicit*/
ihl_tclFileHandler::ihl_tclFileHandler
(
   tCString csFileName
   , ihl_enFileAccess enAccess
):m_ioFileDesc(OSAL_ERROR), m_u32ErrorCode(OSAL_E_NOERROR)
{
   // Assert if File name is NULL. 
   // Check is not required, as OSAL does this check
   NORMAL_M_ASSERT(OSAL_NULL != csFileName);
   
   // Initialize the File handler - Design Pattern RAII
   vOnInit(csFileName, enAccess);

}  // ihl_tclFileHandler::ihl_tclFileHandler(tCString csFileName, ihl_enFileA..

/******************************************************************************
** FUNCTION:  virtual ihl_tclFileHandler::~ihl_tclFileHandler()
******************************************************************************/

/*virtual*/
ihl_tclFileHandler::~ihl_tclFileHandler()
{
   try // try
   {
      // Close the file handle, if open.
      vFClose();
   }  // try
   catch(...) // catch (...)
   {
   }  // catch (...)
   // 
}  // ihl_tclFileHandler::~ihl_tclFileHandler()

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclFileHandler::bIsValid() const
******************************************************************************/

/*virtual*/
tBool ihl_tclFileHandler::bIsValid() const
{
   return (OSAL_ERROR != m_ioFileDesc);
}  // tBool ihl_tclFileHandler::bIsValid() const

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclFileHandler::bFRead(tPCS8 pcos8Buf ...)
******************************************************************************/

/*virtual*/
tBool ihl_tclFileHandler::bFRead(tPS8 ps8Buffer, tU32 u32MaxLength)
{
   tBool bRetVal  =  TRUE;
   
   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;

   tS32 s32ReadBytes  =  OSAL_s32IORead(m_ioFileDesc, ps8Buffer, u32MaxLength);

   if(OSAL_ERROR == s32ReadBytes)
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
      bRetVal  =  FALSE;
   }  // if(OSAL_ERROR == s32ReadBytes)
   else if(s32ReadBytes != ((tS32)u32MaxLength))
   {
      // Number of bytes read is less than expected. Hence return false.
      NORMAL_M_ASSERT_ALWAYS();
      m_u32ErrorCode    =  OSAL_E_UNKNOWN;   // set the error code as unknown.
      bRetVal  =  FALSE;
   }  // End of if-else; if(OSAL_ERROR == s32ReadBytes)

   return bRetVal;
}  // tBool ihl_tclFileHandler::bFRead(tPS8 ps8Buffer, tU32 u32MaxLength) const

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclFileHandler::bFWrite(tPCS8 pcos8Buf ...)
******************************************************************************/

/*virtual*/
tBool ihl_tclFileHandler::bFWrite(tPCS8 pcos8Buffer, tU32 u32Length)
{
   tBool bRetVal  =  TRUE;
   
   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;

   tS32 s32WroteBytes   =  OSAL_s32IOWrite(m_ioFileDesc, pcos8Buffer, u32Length);

   if(OSAL_ERROR == s32WroteBytes)
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
      bRetVal  =  FALSE;
   }  // if(OSAL_ERROR == s32WroteBytes)
   else if(s32WroteBytes != ((tS32)u32Length))
   {
      // Number of bytes wrote is less than expected. Hence return false.
      NORMAL_M_ASSERT_ALWAYS();
      m_u32ErrorCode    =  OSAL_E_UNKNOWN;   // set the error code as unknown.
      bRetVal  =  FALSE;
   }  // End of if-else; if(OSAL_ERROR == s32WroteBytes)
   
   return bRetVal;
}  // tBool ihl_tclFileHandler::bFWrite(tPCS8 pcos8Buffer, tU32 u32Length) const

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclFileHandler::bFSeek(tS32 s32Offset, t..)
******************************************************************************/
/*virtual*/
tBool ihl_tclFileHandler::bFSeek(tS32 s32Offset, tS32 s32Origin)
{
   tBool bRetVal  =  TRUE;
   
   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;
   
   if(OSAL_ERROR == OSALUTIL_s32FSeek(m_ioFileDesc, s32Offset, s32Origin))
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
      bRetVal  =  FALSE;
   }  // End of if-else; if(OSAL_ERROR == OSALUTIL_s32FSeek(m_ioFileDesc, s32O..
   
   return bRetVal;
}  // tBool ihl_tclFileHandler::bFSeek(tS32 s32Offset, tS32 s32Origin) const

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclFileHandler::bFGetPos(tPS32 ps32Ptr)
******************************************************************************/

/*virtual*/
tBool ihl_tclFileHandler::bFGetPos(tPS32 ps32Ptr)
{
   tBool bRetVal  =  TRUE;
   
   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;
   if(OSAL_ERROR == OSALUTIL_s32FGetpos(m_ioFileDesc, (intptr_t*)ps32Ptr))
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
      bRetVal  =  FALSE;
   }  // End of if-else; if(OSAL_ERROR == OSALUTIL_s32FGetpos(m_ioFileDesc, ps..
   
   return bRetVal;
}  // tBool ihl_tclFileHandler::bFGetPos(tPS32 ps32Ptr)

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclFileHandler::bFSetPos(tPS32 ps32Ptr)..
******************************************************************************/

/*virtual*/
tBool ihl_tclFileHandler::bFSetPos(tPS32 ps32Ptr)
{
   tBool bRetVal  =  TRUE;

   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;
   
   if(OSAL_ERROR == OSALUTIL_s32FSetpos(m_ioFileDesc, (intptr_t*)ps32Ptr))
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
      bRetVal  =  FALSE;
   }  // End of if-else; if(OSAL_ERROR == OSALUTIL_s32FSetpos(m_ioFileDesc, ps..
   
   return bRetVal;
}  // tBool ihl_tclFileHandler::bFSetPos(tPS32 ps32Ptr)

/******************************************************************************
** FUNCTION:  virtual tS32 ihl_tclFileHandler::s32GetSize()
******************************************************************************/

/*virtual*/
tS32 ihl_tclFileHandler::s32GetSize()
{
   tS32 s32Size   =  OSALUTIL_s32FGetSize(m_ioFileDesc);

   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;

   if(OSAL_ERROR == s32Size)
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
   }  // End of if-else; if(OSAL_ERROR == s32Size)

   return s32Size;
}  // tS32 ihl_tclFileHandler::s32GetSize()

/******************************************************************************
** FUNCTION:  virtual tS32 ihl_tclFileHandler::s32FTell()
******************************************************************************/

/*virtual*/
tS32 ihl_tclFileHandler::s32FTell()
{
   tS32 s32CurrPos = OSALUTIL_s32FTell(m_ioFileDesc);

   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;
   
   if(OSAL_ERROR == s32CurrPos)
   {
      // Query & Set the error code
      m_u32ErrorCode =  OSAL_u32ErrorCode();
   }  // End of if-else; if(OSAL_ERROR == s32CurrPos)

   return s32CurrPos;
}  // tS32 ihl_tclFileHandler::s32FTell()

/******************************************************************************
** FUNCTION:  virtual tS32 ihl_tclFileHandler::s32FPrintf(tCString szFo...
******************************************************************************/

/*virtual*/
tS32 ihl_tclFileHandler::s32FPrintf(tCString szFormat, ...)
{
   tS32 s32RetVal =  OSAL_ERROR;

   if(OSAL_ERROR != m_ioFileDesc)
   {
      // Modified version of implementation in OSALUTIL_s32FPrintf();

      tChar szBuffer[OSALUTIL_C_U32_MAX_BUFFERSIZE] = {0};
      tS32 s32Size = OSAL_ERROR;
      OSAL_tVarArgList argList;

      // argList is a built-in array, deLint warning
      OSAL_VarArgStart(argList, szFormat);         //lint !e530 !e1055 !e534 !e746

      s32Size = OSAL_s32VarNPrintFormat
         (szBuffer, OSALUTIL_C_U32_MAX_BUFFERSIZE, szFormat, argList); //lint !e530

      OSAL_VarArgEnd(argList);

      if (s32Size >= 0)
      {
         s32RetVal   =  
            OSAL_s32IOWrite(m_ioFileDesc, (tPCS8)szBuffer, (tU32)s32Size);
      }  // if (s32Size >= 0)

      // Reset the last error code to No error.
      m_u32ErrorCode =  OSAL_E_NOERROR;
      
      if(OSAL_ERROR == s32RetVal)
      {
         // Query & Set the error code
         m_u32ErrorCode =  OSAL_u32ErrorCode();
      }  // End of if-else; if(OSAL_ERROR == s32RetVal)
   }  // if(OSAL_ERROR != m_ioFileDesc)

   return (s32RetVal);
}  // tS32 ihl_tclFileHandler::s32FPrintf(tCString szFormat, ...)

/******************************************************************************
** FUNCTION: virtual tBool ihl_tclFileHandler::bFRename(tCString cszNewFileName)
******************************************************************************/

/*virtual*/
tBool ihl_tclFileHandler::bFRename(tCString cszNewFileName, tCString cszOldFileName)
{
   tBool bRetVal  =  FALSE;
   const tChar* csFileArray[2]   =  {OSAL_NULL};

   if((OSAL_NULL != cszNewFileName) && (OSAL_NULL != cszOldFileName))
   {
      csFileArray[0] = cszOldFileName;
      csFileArray[1] = cszNewFileName;

      // Reset the last error code to No error.
      m_u32ErrorCode =  OSAL_E_NOERROR;

      if (OSAL_ERROR == OSAL_s32IOControl(m_ioFileDesc, OSAL_C_S32_IOCTRL_FIORENAME, (intptr_t)csFileArray))
      {
         // Query & Set the error code
         m_u32ErrorCode =  OSAL_u32ErrorCode();
      }  // if (OSAL_ERROR == OSAL_s32IOControl(m_ioFileDesc, OSAL_C_S32_IOCTR..
      else
      {
         bRetVal = TRUE;
      }//End of if-else.if (OSAL_ERROR == OSAL_s32IOControl(m_ioFileDesc, ...)
   }//if((OSAL_NULL != cszNewFileName) && (OSAL_NULL != cszOldFileName))

   return bRetVal;
}  // tBool ihl_tclFileHandler::bFRename(tCString cszNewFileName)

/******************************************************************************
** FUNCTION:  tVoid ihl_tclFileHandler::vOnInit(tCString csFileName, enFi..
******************************************************************************/

tVoid ihl_tclFileHandler::vOnInit(tCString csFileName, ihl_enFileAccess enAcc)
{
   // Perform the appropriate actions.
   switch(enAcc)
   {
      case IHL_EN_RDONLY:
      case IHL_EN_TEXT:
      case IHL_EN_BINARY:
      {
         // Open the file in appropriate mode.
         vFOpen(csFileName, enAcc);
      }  // case IHL_EN_RDONLY:, IHL_EN_TEXT:, IHL_EN_BINARY:
      break;

      case IHL_EN_WRONLY:
      case IHL_EN_RDWR:
      case IHL_EN_APPEND:
      {
         // Try to open the file
         vFOpen(csFileName, enAcc);

         // USE CASE - First time or after FLASH formatting.
         // Check whether the Error was resource doesnot exist?
         if((OSAL_ERROR == m_ioFileDesc)
            &&(OSAL_E_DOESNOTEXIST == m_u32ErrorCode))
         {
            // Create the resource
            vFCreat(csFileName);
         }  // End of if-else; if(OSAL_E_DOESNOTEXIST != m_u32ErrorCode)

      }  // case IHL_EN_WRONLY:, IHL_EN_RDWR:, IHL_EN_APPEND:
      break;

      case IHL_EN_TRUNC:
      case IHL_EN_CREAT:
      {
         // Create or Truncate the file.
         vFCreat(csFileName);
      }  // case IHL_EN_TRUNC:, IHL_EN_CREAT:
      break;

      case IHL_EN_REMOVE:
      {
         // Remove the file from the file system.
         vFRemove(csFileName);
      }  // case IHL_EN_REMOVE:
      break;

      default:
      {
         // Nothing to do.
         // @TODO Error Trace
      }  // default:
      break;
   }  // switch(enAcc)

}  // tVoid ihl_tclFileHandler::vOnInit(tCString csFileName, ihl_enFileAccess enAc..

/******************************************************************************
** FUNCTION:  tVoid ihl_tclFileHandler::vFCreat(tCString ...)
******************************************************************************/

tVoid ihl_tclFileHandler::vFCreat(tCString csFileName, ihl_enFileAccess enAcc)
{
   // Create a file, if file exists this truncates the file and returns a 
   // file descriptor
   m_ioFileDesc   =  OSAL_IOCreate(csFileName, ((OSAL_tenAccess)enAcc));

   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;

   // Assert if Create fails.
   if(OSAL_ERROR == m_ioFileDesc)
   {
      m_u32ErrorCode =  OSAL_u32ErrorCode();

      vErrorCode(m_u32ErrorCode);
   }  // if(OSAL_ERROR != m_ioFileDesc)

}  // tVoid ihl_tclFileHandler::vFCreat(tCString csFileName, ihl_enFileAccess enAcc)

/******************************************************************************
** FUNCTION:  tVoid ihl_tclFileHandler::vFOpen(tCString ...)
******************************************************************************/

tVoid ihl_tclFileHandler::vFOpen(tCString csFileName, ihl_enFileAccess enAcc)
{
   // Open the file in the particular access mode.
   m_ioFileDesc   =  OSAL_IOOpen(csFileName, ((OSAL_tenAccess)enAcc));

   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;

   // Check if open failed.
   if(OSAL_ERROR == m_ioFileDesc)
   {
      m_u32ErrorCode =  OSAL_u32ErrorCode();

      // @TODO - Trace Error code.

      // Check whether the Error was resource doesnot exist?
      if(OSAL_E_DOESNOTEXIST != m_u32ErrorCode)
      {
         vErrorCode(m_u32ErrorCode);
      }  // if(OSAL_E_DOESNOTEXIST != m_u32ErrorCode)

   }  // if(OSAL_ERROR != m_ioFileDesc)
   
}  // tVoid ihl_tclFileHandler::vFOpen(tCString csFileName, ihl_enFileAccess enAcc)

/******************************************************************************
** FUNCTION:  tVoid ihl_tclFileHandler::vFClose()
******************************************************************************/

tVoid ihl_tclFileHandler::vFClose()
{
   // Check if the File handle is valid?
   if(OSAL_ERROR != m_ioFileDesc)
   {
      tS32 s32StatCode = OSAL_s32IOClose(m_ioFileDesc);
      NORMAL_M_ASSERT(OSAL_ERROR != s32StatCode);
      
      // Invalidate the File Handle
      m_ioFileDesc   =  OSAL_ERROR;
   }  // if(OSAL_ERROR != m_ioFileDesc)

}  // tVoid ihl_tclFileHandler::vFClose()

/******************************************************************************
** FUNCTION:  tVoid ihl_tclFileHandler::vFRemove(tCString csFileName)
******************************************************************************/

tVoid ihl_tclFileHandler::vFRemove(tCString csFileName)
{
   // Reset the last error code to No error.
   m_u32ErrorCode =  OSAL_E_NOERROR;

   if(OSAL_ERROR == OSAL_s32IORemove(csFileName))
   {
      m_u32ErrorCode =  OSAL_u32ErrorCode();

      // @TODO Error Trace.
   }  // if(OSAL_ERROR == OSAL_s32IORemove(csFileName))

}  // tVoid ihl_tclFileHandler::vFRemove(tCString csFileName)

/******************************************************************************
** FUNCTION:  tU32 ihl_tclFileHandler::u32ErrorCode() const
******************************************************************************/

tU32 ihl_tclFileHandler::u32ErrorCode() const
{
   return (m_u32ErrorCode);
}  // tU32 ihl_tclFileHandler::u32ErrorCode() const

/******************************************************************************
** FUNCTION:  tBool ihl_tclFileHandler::bValidateFile(tCString csFileName)
******************************************************************************/

/*static*/
tBool ihl_tclFileHandler::bValidateFile(tCString csFileName) 
{
    //Create an Object with Read-Only access.
    ihl_tclFileHandler oFile(csFileName,IHL_EN_RDONLY);
    //Check if File handle is valid
    tBool bRetVal = oFile.bIsValid();

    return bRetVal;

}  // tBool ihl_tclFileHandler::bValidateFile(tCString csFileName)

/******************************************************************************
** FUNCTION:  tBool ihl_tclFileHandler::bFSyncFile()
******************************************************************************/

tBool ihl_tclFileHandler::bFSyncFile()
{
    tBool bRetVal  =  TRUE;

    // Reset the last error code to No error.
    m_u32ErrorCode =  OSAL_E_NOERROR;

    if( OSAL_ERROR == OSAL_s32IOControl(m_ioFileDesc, OSAL_C_S32_IOCTRL_FIOFLUSH, 0))
    {
        // Query & Set the error code
        m_u32ErrorCode =  OSAL_u32ErrorCode();
        bRetVal  =  FALSE;

    }// if( OSAL_ERROR == OSAL_s32IOControl(m_ioFileDesc, OSAL_C_S32_IOCTRL_FIOFLUSH, 0))

    return bRetVal;

}  // tBool ihl_tclFileHandler::bFSyncFile()

/******************************************************************************
** FUNCTION:  tVoid ihl_tclFileHandler::vErrorCode(tCU32 cu32ErrorCode)
******************************************************************************/

tVoid ihl_tclFileHandler::vErrorCode(tCU32 cu32ErrorCode) const
{
   // Assert which is the error code returned during the Open Failure by OSAL.

   //--------------------------------------------------------------------------
   // :REF: POSIX Standards
   // <A HREF="http://www.opengroup.org/onlinepubs/000095399/functions/open.html">
   // :NOTE: OSAL specifies ANCI C open() as reference, so there can be minor
   // variations from POSIX standards.
   // But underlying File System which is LFS (Logical File System) on TEngine
   // platform follows the POSIX standards (exceptions exists).
   //--------------------------------------------------------------------------

   //--------------------------------------------------------------------------
   // Error Classification:
   // 1. Developer Errors     -  OSAL_E_INVALIDVALUE, OSAL_E_NAMETOOLONG, 
   // OSAL_E_BADFILEDESCRIPTOR
   // 2. Permission Errors    -  OSAL_E_NOACCESS, OSAL_E_NOPERMISSION
   // 3. System Errors        -  OSAL_E_DOESNOTEXIST, OSAL_E_ALREADYEXISTS, 
   // OSAL_E_NOTINTERRUPTCALLABLE, OSAL_E_NOSPACE, OSAL_E_BUSY, OSAL_E_IOERROR,
   // OSAL_E_IN_EXCL_ACCESS, , OSAL_E_INTERRUPT, OSAL_E_MAXFILES
   // 4. OSAL Specific Errors -  OSAL_E_CANCELED, OSAL_E_TIMEOUT, 
   // OSAL_E_UNKNOWN, OSAL_E_MEDIA_NOT_AVAILABLE, OSAL_E_NOTSUPPORTED
   //--------------------------------------------------------------------------
   
   // Invalid input to open()
   NORMAL_M_ASSERT(OSAL_E_INVALIDVALUE != cu32ErrorCode);
   // The length of the path argument exceeds {PATH_MAX} or a pathname
   // component is longer than {NAME_MAX}. 
   NORMAL_M_ASSERT(OSAL_E_NAMETOOLONG != cu32ErrorCode);
   // Is a directory
   NORMAL_M_ASSERT(OSAL_E_BADFILEDESCRIPTOR != cu32ErrorCode); 
   // Search permission is denied on a component of the path prefix
   NORMAL_M_ASSERT(OSAL_E_NOACCESS != cu32ErrorCode);
   // Read-only file system
   // The named file resides on a read-only file system and either 
   // O_WRONLY, O_RDWR, O_CREAT (if the file does not exist), or O_TRUNC
   // is set in the oflag argument. 
   NORMAL_M_ASSERT(OSAL_E_NOPERMISSION != cu32ErrorCode);
   // O_CREAT is not set and the named file does not exist; or O_CREAT 
   // is set and either the path prefix does not exist or the path 
   // argument points to an empty string. 
   NORMAL_M_ASSERT(OSAL_E_DOESNOTEXIST != cu32ErrorCode);
   // No such device or address
   // O_NONBLOCK is set, the named file is a FIFO, O_WRONLY is set, and 
   // no process has the file open for reading. 
   NORMAL_M_ASSERT(OSAL_E_NOTINTERRUPTCALLABLE != cu32ErrorCode);
   // The named file exists
   NORMAL_M_ASSERT(OSAL_E_ALREADYEXISTS != cu32ErrorCode); 
   // The directory or file system that would contain the new file cannot
   // be expanded, the file does not exist, and O_CREAT is specified. 
   NORMAL_M_ASSERT(OSAL_E_NOSPACE != cu32ErrorCode);
   // The file is a pure procedure (shared text) file that is being 
   // executed and oflag is O_WRONLY or O_RDWR.
   NORMAL_M_ASSERT(OSAL_E_BUSY != cu32ErrorCode);
   // The path argument names the slave side of a pseudo-terminal device
   // that is locked.
   NORMAL_M_ASSERT(OSAL_E_IN_EXCL_ACCESS != cu32ErrorCode);
   // An IO Error occured
   NORMAL_M_ASSERT(OSAL_E_IOERROR != cu32ErrorCode);
   // A signal was caught during open(). 
   NORMAL_M_ASSERT(OSAL_E_INTERRUPT != cu32ErrorCode);
   // Too many open files or Too many open files in system
   NORMAL_M_ASSERT(OSAL_E_MAXFILES != cu32ErrorCode);
   // Got cancelled, How - NO IDEA, ask OSAL experts? 
   NORMAL_M_ASSERT(OSAL_E_CANCELED != cu32ErrorCode);
   // IO operation timed out
   NORMAL_M_ASSERT(OSAL_E_TIMEOUT != cu32ErrorCode);
   // Usually should not happen with FFS, but may apply for SD card or USB
   NORMAL_M_ASSERT(OSAL_E_MEDIA_NOT_AVAILABLE != cu32ErrorCode);
   // Sorry-Strange? OSAL could not recognize the Error which occured.[:)]
   NORMAL_M_ASSERT(OSAL_E_UNKNOWN != cu32ErrorCode);
   // Open is not supported by OSAL.
   NORMAL_M_ASSERT(OSAL_E_NOTSUPPORTED != cu32ErrorCode);

}  // tVoid ihl_tclFileHandler::vErrorCode(tU32 cu32ErrorCode) const

   }  // namespace io
}  // namespace ihl

///////////////////////////////////////////////////////////////////////////////

// <EOF>
