/*!
*******************************************************************************
* \file              ihl_tclFileHandler.h
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library
*  - DESCRIPTION:    File Handler - Exception safe file handler
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 12.09.2011 \version 1.2 \author Raghavendra S (RBEI/ECV2)
* - Correction for renaming interface.
*
* \date 11.08.2011 \version 1.1 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* - Added support for file renaming.
*
* \date 06.12.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

#ifndef IHL_TCLFILEHANDLER_H_
#define IHL_TCLFILEHANDLER_H_

/******************************************************************************
| includes of component-internal interfaces, if necessary
| (scope: component-local)
|----------------------------------------------------------------------------*/

// Includes to utilize the OSAL Interface
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include <osal_if.h>

// Includes to utilize the BP STL exceptional handler



namespace ihl {
   namespace io {

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/*! \enum ihl_enFileAccess
 * File access type, extended access type of OSAL_tenAccess.
 * For additional details on OSAL_tenAccess, please refer ostypes.h
 */

enum ihl_enFileAccess
{
   IHL_EN_RDONLY   =  OSAL_EN_READONLY,   ///< Enum value for read only access.
   IHL_EN_WRONLY   =  OSAL_EN_WRITEONLY,  ///< Enum value for write only access.
   IHL_EN_RDWR     =  OSAL_EN_READWRITE,  ///< Enum value for RW access.
   IHL_EN_APPEND   =  OSAL_EN_APPEND,     ///< Enum value for Append.
   IHL_EN_TEXT     =  OSAL_EN_TEXT,       ///< Enum value for text mode open.
   IHL_EN_BINARY   =  OSAL_EN_BINARY,     ///< Enum value for binary mode open.
   IHL_EN_TRUNC,                          ///< Enum value for truncating.
   IHL_EN_CREAT,                          ///< Enum value for create a file.
   IHL_EN_REMOVE                          ///< Enum value to remove a file.
}; // enum ihl_enFileAccess


/*! 
 * \class ihl_tclFileHandler
 * \brief Exception safe file handler for file I/O.
 * 
 * \defgroup RAII RAII Design Pattern
 *
 * \par RAII - Resource Acquisition Is Initialization:
 *
 * The technique combines acquisition and release of resources with 
 * initialization and uninitialization of objects.
 * RAII involves assigning ownership of a resource to scoped objects for resource 
 * management. The acquisition is typically bound to the construction 
 * (initialization) and the automatic, deterministic destruction 
 * (uninitialization) is used to guarantee the resource is released. Since 
 * scoped objects are cleaned up regardless of whether the scope exits through 
 * normal use or through an exception, RAII is a key concept for writing 
 * exception-safe code. It&apos; s a technique which leverages the semantics of
 *  stack-based objects to tackle the resource management problem
 *
 * \par Advantages of RAII:
 * - Efficiency: Typically these classes should have little or no run-time 
 * disadvantages compared to alternative methods. 
 * - Transparency: Use of the class should generally be equivalent to, or a more
 * user-friendly version of, use of the underlying resource. 
 * - Reliability: The whole purpose of using a pure RAII class is to gain 
 * reliable control over some potentially dangerous aspect of the underlying 
 * resources behaviour. 
 * 
 * \par References:
 * Effective C++: 55 Specific Ways to Improve Your Programs and Designs - by 
 * Scott Meyers, ISBN: 0-321-33487-6, Publisher: Addison Wesley Professional
 * - Item 13: Use objects to manage resources,
 * - Item 14: Think carefully about copying behavior
 * 
 */

class ihl_tclFileHandler
{
public:
   
   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  ihl_tclFileHandler::ihl_tclFileHandler(tCString csFile..)
   ***************************************************************************/
   /*!
    * \brief   Parameterized Constructor
    * \param   [csFileName]:  (->I) File name (inclusive of file path)
    * \param   [enAccess]:    (I)   File Access type
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   explicit ihl_tclFileHandler(tCString csFileName, ihl_enFileAccess enAccess);

   /***************************************************************************
   ** FUNCTION:  virtual ihl_tclFileHandler::~ihl_tclFileHandler()
   ***************************************************************************/
   /*!
    * \brief   Destructor
    * \param   NONE
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual ~ihl_tclFileHandler();

  /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bIsValid() const
   ***************************************************************************/
   /*!
    * \brief   This function checks the validity of the file handle.
    * \param   NONE
    * \retval  [tBool]: TRUE if File handle is valid, FALSE otherwise.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bIsValid() const;

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bFRead(tPCS8 pcos8Buf ...)
   ***************************************************************************/
   /*!
    * \brief   This function reads a number of bytes from a data file. This 
    *          function corresponds to OSAL's \c OSAL_s32IORead().
    * \param   [ps8Buffer]:      (->O) Input Buffer
    * \param   [u32MaxLength]:   (I)   Maximal size of the buffer in bytes
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bFRead(tPS8 ps8Buffer, tU32 u32MaxLength);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bFWrite(tPCS8 pcos8Buf ...)
   ***************************************************************************/
   /*!
    * \brief   This function writes a number of bytes in to a data file. This 
    *          function corresponds to OSAL's \c OSAL_s32IOWrite().
    * \param   [pcos8Buffer]:  (->I) Output Buffer
    * \param   [u32Length]:    (I)   Size of the buffer in bytes
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tBool bFWrite(tPCS8 pcos8Buffer, tU32 u32Length);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bFSeek(tS32 s32Offset, t..)
   ***************************************************************************/
   /*!
    * \brief   This function sets the data file pointer for a following read-or
    *          write operation. The new position is obtained from offset 
    *          character relative to the origin Position. This function 
    *          corresponds to OSAL's \c OSALUTIL_s32FSeek().
    * \param   [s32Offset]:   (I) Offset
    * \param   [s32Origin]:   (I) Reference position
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/   
   virtual tBool bFSeek(tS32 s32Offset, tS32 s32Origin);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bFGetPos(tPS32 ps32Ptr) const
   ***************************************************************************/
   /*!
    * \brief   This function saves the current position of a data file in a 
    *          variable. This function corresponds to OSAL's 
    *          \c OSALUTIL_s32FGetpos().
    * \param   [ps32Ptr]:     (->O) Current position storage pointer
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/  
   virtual tBool bFGetPos(tPS32 ps32Ptr);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bFSetPos(tPS32 ps32Ptr)..
   ***************************************************************************/
   /*!
    * \brief   This function sets the position in an earlier opened file. This 
    *          function corresponds to OSAL's \c OSALUTIL_s32FSetpos().
    * \param   [ps32Ptr]:     (->I) Position to be set
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/ 
   virtual tBool bFSetPos(tPS32 ps32Ptr);
   
   /***************************************************************************
   ** FUNCTION:  virtual tS32 ihl_tclFileHandler::s32GetSize()
   ***************************************************************************/
   /*!
    * \brief   This function determines the size of a data file. This function 
    *          corresponds to OSAL's \c OSALUTIL_s32FGetSize().
    * \param   NONE
    * \retval  [tS32]: Size in bytes or OSAL_ERROR.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tS32 s32GetSize();

   /***************************************************************************
   ** FUNCTION:  virtual tS32 ihl_tclFileHandler::s32GetSize() const
   ***************************************************************************/
   /*!
    * \brief   This function returns the current data file position for an 
    *          opened data file. This function corresponds to OSAL's 
    *          \c OSALUTIL_s32FTell().
    * \param   NONE
    * \retval  [tS32]: Size in bytes or OSAL_ERROR.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tS32 s32FTell();

   /***************************************************************************
   ** FUNCTION:  virtual tS32 ihl_tclFileHandler::s32FPrintf(tCString szFo...
   ***************************************************************************/
   /*!
    * \brief   This function writes a formatted string in to an earlier opened 
    *          data file. The size of the expanded string should not exceed 
    *          OSALUTIL_C_U32_MAX_BUFFERSIZE. This function corresponds to 
    *          OSAL's \c OSALUTIL_s32FPrintf().
    * \param   [szFormat]: (->I) Format string
    * \param   [...]:      (->I) Variable parameterized list
    * \retval  [tS32]: Number of the written bytes or OSAL_ERROR.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   virtual tS32 s32FPrintf(tCString szFormat, ...);

   /***************************************************************************
   ** FUNCTION:  virtual tBool ihl_tclFileHandler::bFRename(tCString cszNewFi..
   ***************************************************************************/
   /*!
    * \brief   This function renames a file which was opened earlier. 
    * \param   [cszNewFileName]:     (->I) New File Name
    * \param   [cszOldFileName]:     (->I) Old File Name
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 11.08.2011 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/ 
   virtual tBool bFRename(tCString cszNewFileName, tCString cszOldFileName);

   /***************************************************************************
   ** FUNCTION:  tU32 ihl_tclFileHandler::u32ErrorCode() const
   ***************************************************************************/
   /*!
    * \brief   This function returns the last error code during the file 
    *          handling
    * \param   NONE
    * \retval  [tU32]: Error code
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tU32 u32ErrorCode() const;

     /***************************************************************************
   ** FUNCTION:  static tBool bValidateFile(tCString csFileName)
   ***************************************************************************/
   /*!
    * \brief   This function checks the validity of a file.
    * \param   [csFileName]:  (->I) File name (inclusive of file path)
    * \retval  [tBool]: TRUE if File is valid, FALSE otherwise.
    * \date 30.05.2012 \author Sonali Patil (RBEI/ECP2)
    **************************************************************************/
   static tBool bValidateFile(tCString csFileName);

   /***************************************************************************
   ** FUNCTION:  tBool bFSyncFile()
   ***************************************************************************/
   /*!
   * \brief   This function is used to synchronise the files to be copied from 
              cache to location specified(NAND).
   * \param   NONE
   * \retval  [tBool]: TRUE if successful, FALSE otherwise.
   * \date 11.07.2014 \author Sonali Patil (RBEI/ECP2)
   **************************************************************************/
   tBool bFSyncFile();

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:
   
   /***************************************************************************
   *******************************PROTECTED************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclFileHandler::vOnInit(tCString csFileName, enFi..
   ***************************************************************************/
   /*!
    * \brief   File handler initializer function.
    * \param   [csFileName]:  (->I) File name (inclusive of file path)
    * \param   [enAcc]:    (I)   File Access type   
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vOnInit(tCString csFileName, ihl_enFileAccess enAcc);
   
   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclFileHandler::vFOpen(tCString ...)
   ***************************************************************************/
   /*!
    * \brief   This function opens a data file with the specified options. 
    *          This function corresponds to OSAL's \c OSAL_IOOpen().
    * \param   [csFileName]:  (->I) File name (inclusive of file path)
    * \param   [enAcc]:    (I)   File Access type   
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vFOpen(tCString csFileName, ihl_enFileAccess enAcc);

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclFileHandler::vFCreat(tCString ...)
   ***************************************************************************/
   /*!
    * \brief   This function creates a new data file with the specified options. 
    *          This function corresponds to OSAL's \c OSAL_IOCreate() 
    * \note    Interpretation of File Access type:
    *          - IHL_EN_CREAT: Creates a new file if the file does not exists. 
    *          In case the file exists, the file is truncated and opened.
    *          - IHL_EN_TRUNC: Truncates a file to '0'. Similar to creating file
    * \param   [csFileName]:  (->I) File name (inclusive of file path)
    * \param   [enAcc]:    (I)   File Access type   
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vFCreat(tCString csFileName, ihl_enFileAccess enAcc = IHL_EN_RDWR);

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclFileHandler::vFClose()
   ***************************************************************************/
   /*!
    * \brief   This function closes an earlier opened file and shall deallocate 
    *          the file descriptor  This function corresponds to OSAL's 
    *          \c OSAL_s32IOClose().
    * 
    * \param   NONE
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vFClose();

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclFileHandler::vFRemove(tCString csFileName)
   ***************************************************************************/
   /*!
    * \brief   This function shall cause the file named by the pathname pointed 
    *          to by \c csFileName to be no longer accessible by that name. 
    *          A subsequent attempt to open the file using that name shall fail,
    *          unless it is created anew. This function corresponds to OSAL's
    *          \c OSAL_s32IORemove().
    * 
    * \param   [csFileName]: (->I) File name (inclusive of file path)
    * \retval  [tBool]: TRUE or FALSE in case of error.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vFRemove(tCString csFileName);

   /***************************************************************************
   ** FUNCTION:  tVoid ihl_tclFileHandler::vErrorCode(tCU32 cu32ErrorCode)
   ***************************************************************************/
   /*!
    * \brief   This function shall assert the failure reason for opening or 
    *          creating a file
    * \param   [cu32ErrorCode]: (I) Error code
    * \retval  NONE
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   tVoid vErrorCode(tCU32 cu32ErrorCode) const;

   /***************************************************************************
   ** FUNCTION:  ihl_tclFileHandler::ihl_tclFileHandler()
   ***************************************************************************/
   /*!
    * \brief   Default Constructor
    * \param   NONE
    * \retval  NONE
    * \note    Default constructor is declared protected to disable default 
    *          construction.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclFileHandler(); 

   /***************************************************************************
   ** FUNCTION:  ihl_tclFileHandler::ihl_tclFileHandler(const ihl_tclFile..)
   ***************************************************************************/
   /*!
    * \brief   Copy Constructor
    * \param   [rfcoFileHandler]: (I) Const reference to object to be copied
    * \retval  NONE
    * \note    Default copy constructor is declared protected to disable it. So 
    *          that any attempt to copy will be caught by the compiler.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclFileHandler(const ihl_tclFileHandler &rfcoFileHandler);

   /***************************************************************************
   ** FUNCTION:  ihl_tclFileHandler& ihl_tclFileHandler::operator=(const ..)
   ***************************************************************************/
   /*!
    * \brief   Assignment Operator
    * \param   [rfcoFileHandler]: (I) Const reference to object to be copied
    * \retval  [ihl_tclFileHandler&]: Reference to this pointer.
    * \note    Assignment operator is declared protected to disable it. So 
    *          that any attempt for assignment will be caught by the compiler.
    * \date 06.12.2010 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
    **************************************************************************/
   ihl_tclFileHandler& operator=(const ihl_tclFileHandler &rfcoFileHandler);

   /*!
    * \defgroup tclMem Member variables
    */
   /*! @{*/

   /// Resource : File handle to be acquired.
   OSAL_tIODescriptor m_ioFileDesc;          /// File Descriptor
   /// Last Error code
   tU32 m_u32ErrorCode;                      

   /*! @}*/

   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ****************************END OF PRIVATE**********************************
   ***************************************************************************/
}; // class ihl_tclFileHandler


/******************************************************************************
| typedefs (scope: global)
|----------------------------------------------------------------------------*/

/*! 
 * \typedef ihl_tclFileHandler ihl_tFileHandler
 * \brief Exception safe File handler
 *
 * The objects of this type will be registered for stack unwinding.
 * 
 * \note If you leave a function by exception, only destructors of registered 
 * stack objects will be called. That is contrary to C++ exception handling 
 * where the destructors will be called automatically by a mechanism called 
 * stack unwinding.
 *
 * Ref: Howto Using the Blaupunkt STL, v1.0, by Wiedemann Frank(CM-AI/PJ-CF15)
 * 
 */

typedef ihl_tclFileHandler ihl_tFileHandler;

   }  // namespace io
}  // namespace ihl

#endif   // #ifndef IHL_TCLFILEHANDLER_H_

// <EOF>
