/*!
*******************************************************************************
* \file              ihl_tclRawData.cpp
*******************************************************************************
*  - PROJECT:        GM Gen2
*  - SW-COMPONENT:   Infotainment Helper Library 
*  - DESCRIPTION:    Container class for Raw data handling
*  - COPYRIGHT:      &copy; 2010 Robert Bosch Car Multimedia Gmbh
*******************************************************************************
* \date 16.11.2010 \version 1.0 \author Pradeep Chand (CM-AI/PJ-GM55 RBEI)
* \bug No known bugs
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "ihl_tclRawData.h"

/******************************************************************************
| defines and macros (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

namespace ihl {
   namespace buffer {

static tU8 u8Null   =  '\0';

/******************************************************************************
| function prototype (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| function implementation (scope: external-interfaces)
|----------------------------------------------------------------------------*/


/******************************************************************************
** FUNCTION:  ihl_tclRawData::ihl_tclRawData()
******************************************************************************/
ihl_tclRawData::ihl_tclRawData()
:m_u32RawDataSize(0), m_pu8RawData(OSAL_NULL)
{
}  // ihl_tclRawData::ihl_tclRawData()

/******************************************************************************
** FUNCTION:  virtual ihl_tclRawData::~ihl_tclRawData()
******************************************************************************/

/*virtual*/
ihl_tclRawData::~ihl_tclRawData()
{
   try //try
   {
      /// Free the allocated memory
      vClear();
   }  // try
   catch(...) // catch (...)
   {
   }  // catch (...)
   // 

   // Pointers are actually freed in vClear()
   // Pointer explicitly made NULL or zeroed to solve Lint Info 1740
   m_pu8RawData   =  OSAL_NULL;
}  // ihl_tclRawData::~ihl_tclRawData()

/******************************************************************************
** FUNCTION:  ihl_tclRawData::ihl_tclRawData(tCU32 cu32Size, tU8 u8..
******************************************************************************/

/*explicit*/
ihl_tclRawData::ihl_tclRawData(tCU32 cu32Size, tU8 u8Value)
:m_u32RawDataSize(cu32Size), m_pu8RawData(pu8CreatBuffer(cu32Size))
{
   if(OSAL_NULL != m_pu8RawData)
   {
      vInit(u8Value);
   }  // if(OSAL_NULL != m_pu8RawData)
   else
   {
      // Buffer is NULL, so set size to 0.
      m_u32RawDataSize  =  0;
   }  // End of if-else; if(OSAL_NULL != m_pu8RawData)

}  // ihl_tclRawData::ihl_tclRawData(tCU32 cu32Size, tU8 u8Value)

/******************************************************************************
** FUNCTION:  ihl_tclRawData::ihl_tclRawData(tCU8 cu8Buffer[], tCU..)
******************************************************************************/

ihl_tclRawData::ihl_tclRawData(tCU8 cu8Buffer[], tCU32 cu32Size)
:m_u32RawDataSize(cu32Size), m_pu8RawData(pu8CreatBuffer(cu32Size))
{
   if(OSAL_NULL != m_pu8RawData)
   {
      /// Copy the contents of the memory
      /// Using the safe memory copy mechanism i.e. memmove 
      /// OSAL equivalent \c OSAL_pvMemoryMove() to copy instead of 
      /// \c OSAL_pvMemoryCopy()

      /// Though OSAL \c OSAL_pvMemoryMove() corresponds to ANSI C memmove, 
      /// it is similar to the POSIX std memmove.
      /// <A HREF="http://www.opengroup.org/onlinepubs/000095399/functions/memmove.html"/>

      (tVoid)OSAL_pvMemoryMove
         (
            static_cast<tPVoid>(m_pu8RawData) 
            , static_cast<tPCVoid>(&cu8Buffer[0])
            , cu32Size
         ); 

   }  // if(OSAL_NULL != m_pu8RawData)
   else
   {
      // Buffer is NULL, so set size to 0.
      m_u32RawDataSize  =  0;
   }  // End of if-else; if(OSAL_NULL != m_pu8RawData)
}  // ihl_tclRawData::ihl_tclRawData(tCU8 u8Buffer[], tCU32 cu32Size)

/******************************************************************************
** FUNCTION:  ihl_tclRawData::ihl_tclRawData(const ihl_tclRaw..)
******************************************************************************/

ihl_tclRawData::ihl_tclRawData(const ihl_tclRawData &rfcoRawData)
:m_u32RawDataSize(0), m_pu8RawData(OSAL_NULL)
{   
   // Clone the object
   vClone(rfcoRawData);
}  // ihl_tclRawData::ihl_tclRawData(const ihl_tclRawData &rfcoRawData)

/******************************************************************************
** FUNCTION:  ihl_tclRawData& ihl_tclRawData::operator=(const ..)
******************************************************************************/

ihl_tclRawData& ihl_tclRawData::operator =(const ihl_tclRawData &rfcoRawData)
{
   // Identity Test - Self assignment test.
   if (this != &rfcoRawData)
   {
      // Clone the object
      vClone(rfcoRawData);
   }  // if (this == &rfcoRawData) 

   return (*this);
}  // ihl_tclRawData& ihl_tclRawData::operator =(const ihl_tcl..

/******************************************************************************
** FUNCTION:  tU8& ihl_tclRawData::operator[](tCU32 cu32Index)
******************************************************************************/

tU8& ihl_tclRawData::operator [](tCU32 cu32Index)
{
   if((OSAL_NULL != m_pu8RawData) && (0 != m_u32RawDataSize) 
      && (cu32Index < m_u32RawDataSize))
   {
      return m_pu8RawData[cu32Index];
   }  // if((OSAL_NULL != m_pu8RawData)&&(0 != m_u32RawDataSize)&&(cu32Index <..
   
   return u8Null;
}  // tU8& ihl_tclRawData::operator [](tCU32 cu32Index)

/******************************************************************************
** FUNCTION:  tCU8& ihl_tclRawData::operator[](tCU32 cu32Index) const
******************************************************************************/

tCU8& ihl_tclRawData::operator [](tCU32 cu32Index) const
{
   if((OSAL_NULL != m_pu8RawData) && (0 != m_u32RawDataSize) 
      && (cu32Index < m_u32RawDataSize))
   {
      return m_pu8RawData[cu32Index];
   }  // if((OSAL_NULL != m_pu8RawData)&&(0 != m_u32RawDataSize)&&(cu32Index <..
   
   return u8Null;
}  // const tU8& ihl_tclRawData::operator [](tCU32 cu32Index) const

/******************************************************************************
** FUNCTION:  tBool ihl_tclRawData::operator==(const ihl_tclRawDat...
******************************************************************************/

tBool ihl_tclRawData::operator ==(const ihl_tclRawData &rfcoRawDataRhs) const
{
   tBool bRetVal  =  FALSE;
   
   if((OSAL_NULL == m_pu8RawData) || (OSAL_NULL == rfcoRawDataRhs.m_pu8RawData))
   {
      if(rfcoRawDataRhs.m_pu8RawData == m_pu8RawData)
      {
         bRetVal  =  TRUE;
      }  // if(rfcoRawDataRhs.m_pu8RawData == m_pu8RawData)
      else if(OSAL_NULL == m_pu8RawData)
      {
         bRetVal  =  (0 == rfcoRawDataRhs.m_u32RawDataSize);
      }  // else if(OSAL_NULL == m_pu8RawData)
      else
      {
         bRetVal  =  (0 == m_u32RawDataSize);
      }  // End of if-else; if(rfcoRawDataRhs.m_pu8RawData == m_pu8RawData)
   }  // if((OSAL_NULL == m_pu8RawData) || (OSAL_NULL == rfcoRawDataRhs.m_pu8Ra..
   else if(rfcoRawDataRhs.m_u32RawDataSize == m_u32RawDataSize)
   {
      bRetVal  =  (0 == OSAL_s32MemoryCompare
                  (m_pu8RawData, rfcoRawDataRhs.m_pu8RawData, m_u32RawDataSize));
   }  // else if(rfcoRawDataRhs.m_u32RawDataSize == m_u32RawDataSize)
   else
   {
      // Nothing to do. Already return value is false.
   }  // End of if-else; if((OSAL_NULL == m_pu8RawData) || (OSAL_NULL == rf..

   return bRetVal;
}  // tBool ihl_tclRawData::operator ==(const ihl_tclRawData &rfco ..

/******************************************************************************
** FUNCTION:  tBool ihl_tclRawData::operator!=(const ihl_tclRawDat..
******************************************************************************/

tBool ihl_tclRawData::operator !=(const ihl_tclRawData &rfcoRawDataRhs) const
{
   return (!(*this == rfcoRawDataRhs));
}  // tBool ihl_tclRawData::operator !=(const ihl_tclRawData &rfco..

/******************************************************************************
** FUNCTION:  ihl_tclRawData ihl_tclRawData::operator+(const ipod...
******************************************************************************/

ihl_tclRawData ihl_tclRawData::operator +
(
   const ihl_tclRawData &rfcoRawData
) const
{
   ihl_tclRawData oRawDataTemp(*this);
   return (oRawDataTemp += rfcoRawData);
}  // ihl_tclRawData ihl_tclRawData::operator +(const ihl_tcl..

/******************************************************************************
** FUNCTION:  ihl_tclRawData& ihl_tclRawData::operator+=(const ipo..
******************************************************************************/

ihl_tclRawData& ihl_tclRawData::operator +=(const ihl_tclRawData &rfcoRawData)
{
   // Calculate the buffer size.
   tU32 u32BufferSize   =  (m_u32RawDataSize + rfcoRawData.m_u32RawDataSize);

   // Create a local buffer
   tPU8 pu8Buffer       =  pu8CreatBuffer(u32BufferSize);

   if((OSAL_NULL != pu8Buffer) && (OSAL_NULL != m_pu8RawData) 
      && (OSAL_NULL != rfcoRawData.m_pu8RawData))
   {
      // Copy the existing contents of "this" object to buffer.
      (tVoid)OSAL_pvMemoryMove
         (
            static_cast<tPVoid>(&pu8Buffer[0]) 
            , static_cast<tPCVoid>(m_pu8RawData) 
            , m_u32RawDataSize
         );  

      // Copy the contents of the object added to the buffer
      (tVoid)OSAL_pvMemoryMove
         (
            static_cast<tPVoid>(&pu8Buffer[m_u32RawDataSize])
            , static_cast<tPCVoid>(rfcoRawData.m_pu8RawData) 
            , rfcoRawData.m_u32RawDataSize
         );  

      // Clear the contents of "this" object to copy the new contents
      vClear();

      // Assign the buffer pointer and the buffer size to 'this' object
      m_pu8RawData      =  pu8Buffer;
      m_u32RawDataSize  =  u32BufferSize;

   }  // if((OSAL_NULL != pu8Buffer) && (OSAL_NULL != m_pu8RawData) && (OSA..
   else
   {
      NORMAL_M_ASSERT(OSAL_NULL != pu8Buffer);
      NORMAL_M_ASSERT(OSAL_NULL != m_pu8RawData);
      NORMAL_M_ASSERT(OSAL_NULL != rfcoRawData.m_pu8RawData);
   }  // End of if-else; if((OSAL_NULL != pu8Buffer) && (OSAL_NULL != m_pu8..

   return (*this);
}  // ihl_tclRawData& ihl_tclRawData::operator +=(const ihl_tc..

/******************************************************************************
** FUNCTION:  tVoid ihl_tclRawData::vClear()
******************************************************************************/

tVoid ihl_tclRawData::vClear()
{
   // Data really exists? Check if the Raw data pointer is NULL.
   OSAL_DELETE [] m_pu8RawData;
   m_pu8RawData      =  OSAL_NULL;
   
   m_u32RawDataSize  =  0;
}  // tVoid ihl_tclRawData::vClear()

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclRawData::bIsEmpty() const
******************************************************************************/

/*virtual*/
tBool ihl_tclRawData::bIsEmpty() const
{
   return (OSAL_NULL == m_pu8RawData);
}  // tBool ihl_tclRawData::bIsEmpty() const

/******************************************************************************
** FUNCTION:  virtual tU32 ihl_tclRawData::u32GetDataSize() const
******************************************************************************/

/*virtual*/
tU32 ihl_tclRawData::u32GetDataSize() const
{
   return m_u32RawDataSize;
}  // tU32 ihl_tclRawData::u32GetDataSize() const

/******************************************************************************
** FUNCTION:  virtual tVoid ihl_tclRawData::vResize(tU32 u32Size)
******************************************************************************/

/*virtual*/
tVoid ihl_tclRawData::vResize(tCU32 cu32Size)
{
   if(0 == cu32Size)
   {
      // Clear the raw data buffer.
      vClear();
   }  // if(0 == cu32Size)
   else if(0 == m_u32RawDataSize)
   {
      // If the current buffer if NULL, create a raw data buffer and assign to 
      // current one.
      ihl_tclRawData oRawDataTemp(cu32Size);
      *this = oRawDataTemp;
   }  // else if(0 == m_u32RawDataSize)
   else if(m_u32RawDataSize == cu32Size)
   {
      // Nothing to do. in this case.
   }  // else if(m_u32RawDataSize == cu32Size)
   else if(cu32Size < m_u32RawDataSize)
   {
      ihl_tclRawData oRawDataTemp(cu32Size);
      
      if(OSAL_NULL != m_pu8RawData)
      {
         // Copy the contents uptil the (size -1)
         (tVoid)OSAL_pvMemoryMove
            (
               static_cast<tPVoid>(&oRawDataTemp[0])
               , static_cast<tPCVoid>(&m_pu8RawData[0]) 
               , cu32Size
            ); 
      }  // if(OSAL_NULL != m_pu8RawData)

      *this = oRawDataTemp;
   }  // else if(cu32Size < m_u32RawDataSize)
   else
   {
      ihl_tclRawData oRawDataTemp(cu32Size);
      tBool bError = oRawDataTemp.bInsert(0, *this);
      NORMAL_M_ASSERT(TRUE == bError);
      *this = oRawDataTemp;
   }  // End of if-else; if(0 == cu32Size)
}  // tVoid ihl_tclRawData::vResize(tU32 cu32Size)

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclRawData::bInsert(tCU32 cu32Index, c..
******************************************************************************/

/*virtual*/
tBool ihl_tclRawData::bInsert(tCU32 cu32Index, const ihl_tclRawData &rfcoRawData)
{
   tBool bRetVal  =  TRUE;
   if(0 == (cu32Index + rfcoRawData.m_u32RawDataSize))
   {
      // Nothing to insert, there is no data
   }  // if(0 == (cu32Index + rfcoRawData.m_u32RawDataSize))
   else 
   {
      tU32 u32Size   =  0;

      if(cu32Index <= m_u32RawDataSize)
      {
         u32Size   =  (m_u32RawDataSize + rfcoRawData.m_u32RawDataSize);
         ihl_tclRawData oRawDataTemp(u32Size);

         // Copy the contents uptil index
         // If index is 0, directly the data of incoming object can be inserted.
         if((0 != cu32Index) && (OSAL_NULL != m_pu8RawData))
         {
            (tVoid)OSAL_pvMemoryMove
               (
                  static_cast<tPVoid>(&oRawDataTemp.m_pu8RawData[0])
                  , static_cast<tPCVoid>(&m_pu8RawData[0]) 
                  , cu32Index
               ); 
         }  // if(0 != cu32Index)

         // Copy contents of the inserted object
         (tVoid)OSAL_pvMemoryMove
            (
               static_cast<tPVoid>(&oRawDataTemp[cu32Index])
               , static_cast<tPCVoid>(&rfcoRawData[0]) 
               , rfcoRawData.m_u32RawDataSize
            ); 

         tU32 u32NextIndex =  cu32Index + rfcoRawData.m_u32RawDataSize;

         // Calculate amount of data left to be copied from actual object, 
         tU32 u32RemSize   =  (m_u32RawDataSize - cu32Index);

         // Copy contents of the after index
         if((0 != u32RemSize)&& (OSAL_NULL != m_pu8RawData))
         {
            (tVoid)OSAL_pvMemoryMove
               (
                  static_cast<tPVoid>(&oRawDataTemp[u32NextIndex])
                  , static_cast<tPCVoid>(&m_pu8RawData[cu32Index]) 
                  , u32RemSize
               ); 
         }  // if(0 != u32RemSize)

         *this    =  oRawDataTemp;

      }  // if(cu32Index <= m_u32RawDataSize)
      else
      {
         // Insertion index is beyond range.
         bRetVal  =  FALSE;
      }  // End of if-else; if(cu32Index <= m_u32RawDataSize)
   }  // End of if-else; if(0 == (cu32Index + rfcoRawData.m_u32RawDataSize))

   return bRetVal;
}  // tBool ihl_tclRawData::bInsert(tCU32 cu32Index, const ihl_tcl..

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclRawData::bInsert(tCU32 cu32Index, tCU..
******************************************************************************/

/*virtual*/
tBool ihl_tclRawData::bInsert(tCU32 cu32Index, tCU8 cu8Buffer[], tCU32 cu32Size)
{
   ihl_tclRawData oRawDataTemp(cu8Buffer, cu32Size);
   return (bInsert(cu32Index, oRawDataTemp));
}  // tBool ihl_tclRawData::bInsert(tCU32 cu32Index, tCU8 cu8Buffer[], tU32..

/******************************************************************************
** FUNCTION:  virtual tBool ihl_tclRawData::bGetRawData(tU8 u8Buffer[], ..
******************************************************************************/

/*virtual*/
tBool ihl_tclRawData::bGetRawData(tU8 u8Buffer[], tCU32 cu32MaxSize) const
{
   // Return value, defaulted to TRUE as 99% return status is true ;-)
   tBool bRetVal     =  TRUE;

   // Check if the Raw data size is zero.
   if(0 == m_u32RawDataSize)
   {
      // No data to copy, nothing to do.
   }  // if(0 == m_u32RawDataSize)
   // Check if the Size of the data to be copied is gtr than maximum size.
   else if(cu32MaxSize >= m_u32RawDataSize)
   {
      if(OSAL_NULL != m_pu8RawData)
      {
         (tVoid)OSAL_pvMemoryMove
         (
            static_cast<tPVoid>(&u8Buffer[0]) 
            , static_cast<tPCVoid>(&m_pu8RawData[0])
            , m_u32RawDataSize
         );
      }
   }  // if(cu32MaxSize >= m_u32RawDataSize)
   else
   {
      // Should never reach here, ideally, but we have to cover the 1% chance
      // ;-)

      // Assert for debugging purpose, size > max size?
      NORMAL_M_ASSERT_ALWAYS();

      // Copying failed due to size exceeding buffer size.
      bRetVal     =     FALSE;
   }  // End of if-else; if(cu32MaxSize >= m_u32RawDataSize)

   return bRetVal;

}  // tBool ihl_tclRawData::bGetRawData(tU8 u8Buffer[], tCU32 cu32MaxSize)

/******************************************************************************
** FUNCTION:  tVoid ihl_tclRawData::vClone(const ihl_tclRawData &r..
******************************************************************************/

tVoid ihl_tclRawData::vClone(const ihl_tclRawData &rfcoRawData)
{
   // Check if the Raw data pointer is valid
   if(OSAL_NULL != rfcoRawData.m_pu8RawData)
   {
      // Exception safety mechanism.
      // Create a local buffer and copy the contents instead of clearing
      // existing contents and then copying
      tPU8 pu8Buffer   = pu8CreatBuffer(rfcoRawData.m_u32RawDataSize);

      if(OSAL_NULL != pu8Buffer)
      {
         /// Copy the contents of the memory
         (tVoid)OSAL_pvMemoryMove
            (
               static_cast<tPVoid>(pu8Buffer)
               , static_cast<tPCVoid>(&rfcoRawData[0])
               , rfcoRawData.m_u32RawDataSize
            );   

         // Clear the contents of "this" object to copy the new contents.
         vClear();

         m_pu8RawData      =  pu8Buffer;
         m_u32RawDataSize  =  rfcoRawData.m_u32RawDataSize;

      }  // if(OSAL_NULL != pu8Buffer)

   }  // if (OSAL_NULL != rfcoRawData.m_pu8RawData)
   else
   {
      // Nothing to do here. 
      // Clear the contents of 'this' object. ;-)
      vClear();

   }  // End of if-else; if((OSAL_NULL != rfcoRawData.m_pu8RawData)

}  // tVoid ihl_tclRawData::vClone(const ihl_tclRawData &rfcoRawData)

/******************************************************************************
** FUNCTION:  tPU8 ihl_tclRawData::pu8CreatBuffer(tU32 u32Size) const
******************************************************************************/

/*virtual*/
tPU8 ihl_tclRawData::pu8CreatBuffer(tU32 u32Size) const
{
   tPU8 pu8Buffer =  OSAL_NULL;
  
   // Check the size of the required buffer; it has to be > 0
   if(0 < u32Size)
   {
      // Create the buffer
      pu8Buffer   = OSAL_NEW tU8[u32Size];
   }  // if(0 < u32Size)

   return pu8Buffer;
}  // tPU8 ihl_tclRawData::pu8CreatBuffer(tU32 u32Size) const

/******************************************************************************
** FUNCTION:  tVoid ihl_tclRawData::vInitRawData(tCU8 cu8Value)
******************************************************************************/

tVoid ihl_tclRawData::vInit(tU8 u8Value)
{
   if(OSAL_NULL != m_pu8RawData)
   {
      (tVoid)OSAL_pvMemorySet
      (
         static_cast<tPVoid>(m_pu8RawData)
         , static_cast<tU32>(u8Value)
         , m_u32RawDataSize
      );

   }  // if(OSAL_NULL != m_pu8RawData)
}  // tVoid ihl_tclRawData::vInitRawData(tU8 u8Value)

   }  // namespace buffer
}  // namespace ihl

///////////////////////////////////////////////////////////////////////////////

// <EOF>
