/************************************************************************
* FILE:        hc_tclThermalSensor_SM.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: base class of thermal sensor state machine
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 11.03.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclThermalSensor_SM.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclThermalSensor_SM.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclThermalSensor_SM()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_SM::hc_tclThermalSensor_SM()
: hc_tclThermalSensor_Config()
{
   ETG_TRACE_USR4(("hc_tclThermalSensor_SM() entered."));
   m_enSensorState = enThermalState_INIT;
   m_enSensorStateLastValid = enThermalState_INIT;
   m_tTemperature = 0;
   m_enTemperatureQualifier = enTemperatureQualifier_UNSUPPORTED;
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclThermalSensor_SM()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_SM::~hc_tclThermalSensor_SM()
{
   ETG_TRACE_USR4(("~hc_tclThermalSensor_SM() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	vThermalSensor_SM()
*
* DESCRIPTION: 	update of thermal state depending of the new thermal information
*
* PARAMETER:   	tThermalInfo_Standard &ThermalInfo
* 				tBool &bTemperatureChanged
* 				tBool &bSensorStateChanged
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensor_SM::vThermalSensor_SM(const tThermalInfo_Internal &ThermalInfo,
		tBool &bTemperatureChanged, tBool &bSensorStateChanged)
{
   ETG_TRACE_USR4(("vThermalSensor_SM() entered."));

   tS16Temperature _tTemperature_Old = m_tTemperature;
   tS16Temperature _tTemperature_New = ThermalInfo.tTemperature;
   tenThermalState _enSensorState_Old = m_enSensorState;
   tenThermalState _enSensorState_New = m_enSensorState;
   // configuration item of thermal sensor
   tBool _bSensorEnabled = bGetConfigEnable();
   tBool _bIfAvailable = ThermalInfo.bIfAvailable;
   //tBool _bTemperatureSupported = ThermalInfo.bTemperatureSupported;
   tBool _bTemperatureReceived = ThermalInfo.bTemperatureReceived;
   tBool _bSensorStateSupported = ThermalInfo.bSensorStateSupported;
   tBool _bSensorStateReceived = ThermalInfo.bSensorStateReceived;
   tBool _bConfigValid = bIsConfigValid();
   // check sensor availability
   if (FALSE == _bSensorEnabled)
   {
      _enSensorState_New = enThermalState_UNAVAILABLE;
      ETG_TRACE_USR3(("vThermalSensor_SM: sensor %d unavailable", ThermalInfo.enSensorID ));
   }
   // following thermal state of external thermal sensor
   else if (TRUE == _bSensorStateSupported && TRUE == _bSensorStateReceived)
   {
      _enSensorState_New = ThermalInfo.uSensorState.enHC;
   }
   // check sensor configuration
   else if (FALSE == _bConfigValid)
   {
      _enSensorState_New = enThermalState_CONFIGFAILURE;
      ETG_TRACE_USR3(("vThermalSensor_SM: invalid configuration!"));
   }
   // check the interface availability and temperature receiving state during the startup (INIT)
   else if ((FALSE == _bIfAvailable || FALSE == _bTemperatureReceived)
            && (enThermalState_INIT == _enSensorState_Old))
   {
      _enSensorState_New = enThermalState_INIT;
      ETG_TRACE_USR3(("vThermalSensor_SM: sensor interface doesn't receive a temperature and stay in init"));
   }
   // check the interface availability except during the startup (INIT)
   else if (FALSE == _bIfAvailable && enThermalState_INIT < _enSensorState_Old)
   {
      _enSensorState_New = enThermalState_TEMPUNAVAIL;
      ETG_TRACE_USR3(("vThermalSensor_SM: sensor interface unavailable and not init state"));
   }
   // check the interface availability and temperature receiving state
   else if ((TRUE == _bIfAvailable && FALSE == _bTemperatureReceived)
            && (enThermalState_CRITICALLOW <= _enSensorState_Old
                || enThermalState_TEMPUNAVAIL == _enSensorState_Old))
   {
      _enSensorState_New = enThermalState_TEMPUNAVAIL;
      ETG_TRACE_USR3(("vThermalSensor_SM: sensor interface available and no temperature received after valid temperature"));
   }
   // calculate new sensor state depending of new temperature
   else if(_tTemperature_Old != _tTemperature_New
           || enThermalState_CRITICALLOW <= _enSensorState_Old)
   {
	   // restore last known valid thermal state
	   if (TRUE == _bIfAvailable && TRUE == _bTemperatureReceived && enThermalState_CRITICALLOW > _enSensorState_Old)
	   {
		   _enSensorState_Old = m_enSensorStateLastValid;
	   }
	   // update current thermal state
	   if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_MIN) > _tTemperature_New
		   || s16GetConfigTemp(enThermalSensorConfigItem_TEMP_MAX) < _tTemperature_New)
		   _enSensorState_New = enThermalState_TEMPUNAVAIL;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_EH) < _tTemperature_New)
		   _enSensorState_New = enThermalState_EMERGENCYHIGH;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_EH2H) <= _tTemperature_New)
	   {
		   _enSensorState_New = enThermalState_HIGH;
		   if (enThermalState_EMERGENCYHIGH == _enSensorState_Old
			   /*|| enThermalState_INIT == _enSensorState_Old*/)
			   _enSensorState_New = enThermalState_EMERGENCYHIGH;
	   }
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_H) < _tTemperature_New)
		   _enSensorState_New = enThermalState_HIGH;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_H2W) <= _tTemperature_New)
	   {
		   _enSensorState_New = enThermalState_WARM;
		   if (enThermalState_HIGH <= _enSensorState_Old
			   || enThermalState_INIT == _enSensorState_Old)
			   _enSensorState_New = enThermalState_HIGH;
	   }
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W) < _tTemperature_New)
		   _enSensorState_New = enThermalState_WARM;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W2N) <= _tTemperature_New)
	   {
		   _enSensorState_New = enThermalState_NORMAL;
		   if (enThermalState_WARM <= _enSensorState_Old
			   || enThermalState_INIT == _enSensorState_Old)
			   _enSensorState_New = enThermalState_WARM;
	   }
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_CL) > _tTemperature_New)
		   _enSensorState_New = enThermalState_CRITICALLOW;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_CL2L) >= _tTemperature_New)
	   {
		   _enSensorState_New = enThermalState_LOW;
		   if (enThermalState_CRITICALLOW == _enSensorState_Old
			   /*|| enThermalState_INIT == _enSensorState_Old*/)
			   _enSensorState_New = enThermalState_CRITICALLOW;
	   }
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_L) > _tTemperature_New)
		   _enSensorState_New = enThermalState_LOW;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_L2C) >= _tTemperature_New)
	   {
		   _enSensorState_New = enThermalState_COLD;
		   if (enThermalState_LOW == _enSensorState_Old
			   || enThermalState_INIT == _enSensorState_Old)
			   _enSensorState_New = enThermalState_LOW;
	   }
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C) > _tTemperature_New)
		   _enSensorState_New = enThermalState_COLD;
	   else if (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C2N) >= _tTemperature_New)
	   {
		   _enSensorState_New = enThermalState_NORMAL;
		   if (enThermalState_COLD == _enSensorState_Old
			   || enThermalState_INIT == _enSensorState_Old)
			   _enSensorState_New = enThermalState_COLD;
	   }
	   else if (TRUE == _bIfAvailable && TRUE == _bTemperatureReceived)
	   {
		   _enSensorState_New = enThermalState_NORMAL;
	   }
   }
   // save temperature, current and last valid thermal state, update temperature qualifier
   m_tTemperature = _tTemperature_New;
   m_enSensorState = _enSensorState_New;
   if (enThermalState_CRITICALLOW <= _enSensorState_New && enThermalState_MAX > _enSensorState_New)
	   m_enSensorStateLastValid = _enSensorState_New;
   if (_tTemperature_Old != _tTemperature_New)
	   bTemperatureChanged = TRUE;
   if (_enSensorState_Old != _enSensorState_New)
	   bSensorStateChanged = TRUE;
   vUpdate_TemperatureQualifier(ThermalInfo);
}

/*******************************************************************************
*
* FUNCTION: 	vUpdate_TemperatureQualifier()
*
* DESCRIPTION: 	update temperature qualifier
*
* PARAMETER:   	const tThermalInfo_Internal &ThermalInfo
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_SM::vUpdate_TemperatureQualifier(const tThermalInfo_Internal &ThermalInfo)
{
   ETG_TRACE_USR4(("vUpdate_TemperatureQualifier() entered."));
	m_enTemperatureQualifier = enTemperatureQualifier_INVALID;
	if (TRUE == ThermalInfo.bTemperatureSupported)
		if (TRUE == ThermalInfo.bTemperatureReceived && enThermalState_CRITICALLOW <= enGetSensorState())
			if (TRUE == ThermalInfo.bTemperatureEstimated)
				m_enTemperatureQualifier = enTemperatureQualifier_ESTIMATED;
			else
				m_enTemperatureQualifier = enTemperatureQualifier_REAL;
		else
			m_enTemperatureQualifier = enTemperatureQualifier_INVALID;
	else
		m_enTemperatureQualifier = enTemperatureQualifier_UNSUPPORTED;
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_SM()
*
* DESCRIPTION: 	trace state machine status information
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_SM::vTraceInfo_SM(const char* strIdentifier)
{
   ETG_TRACE_USR4(("vTraceInfo_SM() entered." ));

   ETG_TRACE_USR2(("%26s: %30s=%20s ", strIdentifier, "               STATE",
		   strGetThermalState(enGetSensorState()) ));
   ETG_TRACE_USR2(("%26s: %30s=%20s ", strIdentifier, "         VALID_STATE",
		   strGetThermalState(enGetSensorStateLastValid()) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentifier, "                TEMP",
		   tGetTemperature() ));
   ETG_TRACE_USR2(("%26s: %30s=%20s ", strIdentifier, "      TEMP_QUALIFIER",
		   strGetTemperatureQualifier(enGetTemperatureQualifier()) ));
}
