/************************************************************************
* FILE:        hc_tclThermalSensor_ITC.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: class of thermal sensor specific internal trouble code
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 11.02.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclThermalSensor_ITC.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclThermalSensor_ITC.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclThermalSensor_ITC()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_ITC::hc_tclThermalSensor_ITC()
{
   ETG_TRACE_USR4(("hc_tclThermalSensor_ITC() entered."));
   (tVoid)memset((tVoid*)m_atITCParameter, OSAL_NULL, sizeof(m_atITCParameter));
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclThermalSensor_ITC()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_ITC::~hc_tclThermalSensor_ITC()
{
   ETG_TRACE_USR4(("~hc_tclThermalSensor_ITC() entered."));
   (tVoid)memset((tVoid*)m_atITCParameter, OSAL_NULL, sizeof(m_atITCParameter));
}

/*******************************************************************************
*
* FUNCTION:    bITCNotify()
*
* DESCRIPTION: Notification for emergency function
*
* PARAMETER:   tenITC_ID enITC_ID:           ITC ID
*              tenThermalState enUpperState: upper state to activate ITC
*              tenThermalState enLowerState: lower state to activate ITC
*
* RETURNVALUE: tBool {successful or failed}
*
*******************************************************************************/
tBool hc_tclThermalSensor_ITC::bITCNotify(tenITC_ID enITC_ID,
		tenThermalState enUpperState, tenThermalState enLowerState)
{
   ETG_TRACE_USR4(("bITCNotify(itc: %d, upper state: %d, lower state: %d) entered.",
		   enITC_ID, enUpperState, enLowerState ));

   if( enITC_ID_MAX > enITC_ID)
   {
	   m_atITCParameter[enITC_ID].enUpperState = enThermalState_UNAVAILABLE;
	   if (enThermalState_NORMAL < enUpperState && enThermalState_MAX > enUpperState)
		   m_atITCParameter[enITC_ID].enUpperState = enUpperState;
	   m_atITCParameter[enITC_ID].enLowerState = enThermalState_UNAVAILABLE;
	   if (enThermalState_NORMAL > enLowerState && enThermalState_CRITICALLOW <= enLowerState)
		   m_atITCParameter[enITC_ID].enLowerState = enLowerState;
	   return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    vUpdate_ITCState()
*
* DESCRIPTION: update all ITC states depending of last valid state
*
* PARAMETER:   tBool &bChanged: one of ITC state has been changed
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_ITC::vUpdate_ITCState(tBool &bChanged)
{
   ETG_TRACE_USR4(("vUpdate_ITCState() entered."));

   tenThermalState _enCurState = enGetSensorState();
   tenThermalState _enLastValidState = enGetSensorStateLastValid();
   for (tU8 _u8Index = 0; (tU8)enITC_ID_MAX > _u8Index; ++_u8Index)
   {
	   tenITC_State _enState_New = enITC_State_DISABLED;
	   if (enThermalState_UNAVAILABLE != _enCurState)
	   {
		   if ((enThermalState_UNAVAILABLE != m_atITCParameter[_u8Index].enUpperState)
			   || (enThermalState_UNAVAILABLE != m_atITCParameter[_u8Index].enLowerState))
		   {
			   ETG_TRACE_USR4(("vUpdate_ITCState() itc id :%d , state : %d ", ETG_CENUM ( tenITC_State , _u8Index ), ETG_CENUM( tenITC_State , m_atITCParameter[_u8Index].enUpperState ) ));
			   _enState_New = enITC_State_NoResult;
			   if (enThermalState_TEMPUNAVAIL < _enLastValidState && enThermalState_MAX > _enLastValidState)
			   {
				   _enState_New = enITC_State_Passed;
				   // check upper state
				   if (enThermalState_UNAVAILABLE != m_atITCParameter[_u8Index].enUpperState)
					   if (m_atITCParameter[_u8Index].enUpperState <= _enLastValidState)
						   _enState_New = enITC_State_Failed;
				   // check lower state
				   if (enThermalState_UNAVAILABLE != m_atITCParameter[_u8Index].enLowerState)
					   if (m_atITCParameter[_u8Index].enLowerState >= _enLastValidState)
						   _enState_New = enITC_State_Failed;
			   }
		   }
	   }
	   if (_enState_New != m_atITCParameter[_u8Index].enITCState)
		   bChanged = TRUE;
	   m_atITCParameter[_u8Index].enITCState = _enState_New;
   }
   ETG_TRACE_USR4(("vUpdate_ITCState(changed: %d) exit.", bChanged ));
}

/*******************************************************************************
*
* FUNCTION:    bGetNewITCState()
*
* DESCRIPTION: getting the new emergency function state information
*
* PARAMETER:   tITCInfo_Internal &ITCStates
*
* RETURNVALUE: tBool		{successful or failed}
*
*******************************************************************************/
tBool hc_tclThermalSensor_ITC::bGetNewITCState(tITCInfo_Internal &ITCStates) const
{
   ETG_TRACE_USR4(("bGetNewITCState() entered."));

   for (tU8 _u8Index = 0; (tU8)enITC_ID_MAX > _u8Index; ++ _u8Index)
	   ITCStates.aenITCState[_u8Index] = m_atITCParameter[_u8Index].enITCState;
   return TRUE;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_ITC()
*
* DESCRIPTION: trace status info
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_ITC::vTraceInfo_ITC(const char* strIdentifier)
{
   ETG_TRACE_USR4(("vTraceInfo_ITC() entered."));

   for (tU8 _u8Index = 0; (tU8)enITC_ID_MAX > _u8Index; ++ _u8Index)
   {
	   if (enThermalState_UNAVAILABLE != m_atITCParameter[_u8Index].enUpperState
		   || enThermalState_UNAVAILABLE != m_atITCParameter[_u8Index].enLowerState
		   || enITC_State_DISABLED != m_atITCParameter[_u8Index].enITCState )
	   {
		   tChar _strAttrName[20] = "";
		   (tVoid)sprintf(_strAttrName, "     ITC_%5s_", strGetITC_Abbreviation((tenITC_ID)_u8Index));

		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "___ID",
				   strGetITC_ID((tenITC_ID)_u8Index) ));
		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "UPPER",
				   strGetThermalState(m_atITCParameter[_u8Index].enUpperState) ));
		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "LOWER",
				   strGetThermalState(m_atITCParameter[_u8Index].enLowerState) ));
		   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ", strIdentifier, _strAttrName, "STATE",
				   strGetITC_State(m_atITCParameter[_u8Index].enITCState) ));
	   }
   }
}
