/************************************************************************
* FILE:        hc_tclThermalSensor_Config.h
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: base class of thermal sensors configuration
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 05.02.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

#ifndef HC_TCLTHERMALSENSOR_CONFIG_H_
#define HC_TCLTHERMALSENSOR_CONFIG_H_

// Basic OSAL includes
#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"
#endif //#ifndef OSAL_S_IMPORT_INTERFACE_GENERIC

#include "hc_AppMain.h"
#include "I_hc_tclCfgMgr.h"
#include "hc_tclThermalSensor_Types.h"
#include "hc_tclFANControl_Types.h"

// thermal sensor config base class
class hc_tclThermalSensor_Config
{
public:
	hc_tclThermalSensor_Config();
	virtual ~hc_tclThermalSensor_Config();

	enum tenTempConfigItem
	{
		enThermalSensorConfigItem_TEMP_MIN,
		enThermalSensorConfigItem_TEMP_CL,
		enThermalSensorConfigItem_TEMP_CL2L,
		enThermalSensorConfigItem_TEMP_L,
		enThermalSensorConfigItem_TEMP_L2C,
		enThermalSensorConfigItem_TEMP_C,
		enThermalSensorConfigItem_TEMP_C2N,
		enThermalSensorConfigItem_TEMP_W2N,
		enThermalSensorConfigItem_TEMP_W,
		enThermalSensorConfigItem_TEMP_H2W,
		enThermalSensorConfigItem_TEMP_H,
		enThermalSensorConfigItem_TEMP_EH2H,
		enThermalSensorConfigItem_TEMP_EH,
		enThermalSensorConfigItem_TEMP_MAX,
		enThermalSensorConfigItem_MAX
	};
	// access for configured temperature thresholds
	inline tS16 s16GetConfigTemp(tenTempConfigItem enItem) const
	{
		if(enThermalSensorConfigItem_MAX > enItem)
			return m_as16ConfigTemp[enItem];
		return HC_TEMP_INVALID_VALUE;
	};
	inline tenConfigSource enGetConfigTempSource(tenTempConfigItem enItem) const
	{
		if(enThermalSensorConfigItem_MAX > enItem)
			return m_aenConfigTempSource[enItem];
		return enConfigSource_UNDEF;
	};
	inline tBool bSetConfigTemp(tenTempConfigItem enItem, tS16 s16Temperature)
	{
		if(enThermalSensorConfigItem_MAX > enItem)
		{
			m_as16ConfigTemp[enItem] = s16Temperature;
			(tVoid)bSetConfigTempSource(enItem, enConfigSource_Developer_Dynamic);
			return TRUE;
		}
		return FALSE;
	};
	inline tBool bSetConfigTempSource(tenTempConfigItem enItem, tenConfigSource enSource)
	{
		if(enThermalSensorConfigItem_MAX > enItem)
		{
			m_aenConfigTempSource[enItem] = enSource;
			return TRUE;
		}
		return FALSE;
	};
	virtual tBool bSetConfigTempPers(tenTempConfigItem enItem, tS16 s16Temperature) = 0;
	virtual tBool bResetConfigTempPers(tenTempConfigItem enItem){
		return bSetConfigTempPers(enItem, HC_TEMP_INVALID_VALUE);
	};
	// access for configured sensor fan thresholds
	inline tS16 s16GetConfigFanTemp(tenFANID enFan, tenFANSpeedLevel enSpeedLevel) const
	{
		if ((HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed) > (tU16)enFan)
			&& (HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0]) > (tU16)enSpeedLevel))
		{
			return m_as16ConfigTempFanSpeed[enFan][enSpeedLevel];
		}
		return HC_TEMP_INVALID_VALUE;
	};
	inline tBool bSetConfigFanTemp(tenFANID enFan, tenFANSpeedLevel enSpeedLevel, tS16 s16Temperature)
	{
		if ((HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed) > (tU16)enFan)
			&& (HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0]) > (tU16)enSpeedLevel))
		{
			m_as16ConfigTempFanSpeed[enFan][enSpeedLevel] = s16Temperature;
			(tVoid)bSetConfigFanTempSource(enFan, enSpeedLevel, enConfigSource_Developer_Dynamic);
			return TRUE;
		}
		return FALSE;
	};
	inline tenConfigSource enGetConfigFanTempSource(tenFANID enFan, tenFANSpeedLevel enSpeedLevel) const
	{
		if((HC_C_U16_ARRAYELEMENTS(m_aenConfigTempFanSpeed_Source) > (tU16)enFan)
		   && (HC_C_U16_ARRAYELEMENTS(m_aenConfigTempFanSpeed_Source[0]) > (tU16)enSpeedLevel))
		{
			return m_aenConfigTempFanSpeed_Source[enFan][enSpeedLevel];
		}
		return enConfigSource_UNDEF;
	};
	inline tBool bSetConfigFanTempSource(tenFANID enFan, tenFANSpeedLevel enSpeedLevel, tenConfigSource enSource)
	{
		if((HC_C_U16_ARRAYELEMENTS(m_aenConfigTempFanSpeed_Source) > (tU16)enFan)
		   && (HC_C_U16_ARRAYELEMENTS(m_aenConfigTempFanSpeed_Source[0]) > (tU16)enSpeedLevel))
		{
			m_aenConfigTempFanSpeed_Source[enFan][enSpeedLevel] = enSource;
			return TRUE;
		}
		return FALSE;
	};
	virtual tBool bSetConfigFanTempPers(tenFANID enFan, tenFANSpeedLevel enSpeedLevel, tS16 s16Temperature) = 0;
	virtual tBool bResetConfigFanTempPers(tenFANID enFan, tenFANSpeedLevel enSpeedLevel){
		return bSetConfigFanTempPers(enFan, enSpeedLevel, HC_UNCONFIGURED_TEMP);
	};
	// access for configured sensor availability
	inline tBool bGetConfigEnable(tVoid) const {return m_bConfigEnable;};
	inline tenConfigSource enGetConfigEnableSource(tVoid) const {return m_enConfigEnableSource;};
	inline tBool bSetConfigEnable(tBool bEnable) {
		m_bConfigEnable = bEnable;
		(tVoid)bSetConfigEnableSource(enConfigSource_Developer_Dynamic);
		return TRUE;
	};
	inline tBool bSetConfigEnableSource(tenConfigSource enSource) {m_enConfigEnableSource = enSource;return TRUE;};
	virtual tBool bSetConfigEnablePers(tU8 u8Enable) = 0;
	virtual tBool bResetConfigEnablePers(tVoid){return bSetConfigEnablePers(HC_UNCONFIGURED_BOOL);};
	// access for configured attenuation cycle time
	inline tU8 u8GetConfigAttenuationCycleTime(tVoid) const {return m_u8CycleTimeAttenuation;};
	inline tenConfigSource enGetConfigAttenuationCycleTimeSource(tVoid) const {return m_enCycleTimeAttenuationSource;};
	inline tBool bSetConfigAttenuationCycleTime(tU8 u8CycleTime)
	{
		if (HC_UNCONFIGURED_CYCLETIME != u8CycleTime)
		{
			m_u8CycleTimeAttenuation = u8CycleTime;
			(tVoid)bSetConfigAttenuationCycleTimeSource(enConfigSource_Developer_Dynamic);
		}
		return (m_u8CycleTimeAttenuation == u8CycleTime);
	}
	inline tBool bSetConfigAttenuationCycleTimeSource(tenConfigSource enSource){m_enCycleTimeAttenuationSource=enSource;return TRUE;};
	virtual tBool bSetConfigAttenuationCycleTimePers(tU8 u8CycleTime) = 0;
	virtual tBool bResetConfigAttenuationCycleTimePers(tVoid){return bSetConfigAttenuationCycleTimePers(HC_UNCONFIGURED_CYCLETIME);};
	// access for configured fan speeds
	inline tU8 u8GetConfigFANSpeed(tenThermalState enState) {
		if (enThermalState_MAX > enState && enThermalState_NORMAL < enState)
			return m_au8ConfigFANSpeed[enState];
		return 0;
	};
	virtual tU8 u8GetConfigFANSpeed4Temp(tenFANID enFANID, tS16 s16Temp) const;
	// validation of thermal sensor configuration
	virtual tBool bIsConfigValid(tVoid) const;
	// loading configuration
	virtual tVoid vLoadConfig(tenThermalSensorID enSensorID){(tVoid)enSensorID;};
protected:
	// loading fan configuration
	virtual tVoid vLoadConfig_FanSpeed(tVoid);
	// validation of thermal sensor fan configuration
	virtual tBool bIsFanConfigValid(tenFANID enFANID) const;
    // trace status info
    virtual tVoid vTraceInfo_Config(const char* strIdentifier);
	// temperature thresholds
	tS16 m_as16ConfigTemp[enThermalSensorConfigItem_MAX];
	tenConfigSource m_aenConfigTempSource[enThermalSensorConfigItem_MAX];
	tBool m_bConfigEnable;
	tenConfigSource m_enConfigEnableSource;
	tU8 m_u8CycleTimeAttenuation;
	tenConfigSource m_enCycleTimeAttenuationSource;
	// fan speeds depending of last valid state
	tU8 m_au8ConfigFANSpeed[enThermalState_MAX];
	// fan speed depending of last valid high temperature ... 100%, 75%, 50%, 25% and 0%
	tS16 m_as16ConfigTempFanSpeed[enFANID_MAX][enFANSpeedLevel_MAX];
	tenConfigSource m_aenConfigTempFanSpeed_Source[enFANID_MAX][enFANSpeedLevel_MAX];
	tS16 m_s16ConfigTempFanSpeedOffset;
};

// template class for sensor specific config
template <class TRegEnable, class TDevelopEnable,
class TRegConfMinTemp,
class TRegConfMaxTemp,
class TRegConfEmeregencyHigh, class TDevelopConfEmeregencyHigh,
class TRegConfEmeregencyHigh2High, class TDevelopConfEmeregencyHigh2High,
class TRegConfHigh, class TDevelopConfHigh,
class TRegConfHigh2Warm, class TDevelopConfHigh2Warm,
class TRegConfWarm, class TDevelopConfWarm,
class TRegConfWarm2Normal, class TDevelopConfWarm2Normal,
class TRegConfCold2Normal, class TDevelopConfCold2Normal,
class TRegConfCold, class TDevelopConfCold,
class TRegConfLow2Cold, class TDevelopConfLow2Cold,
class TRegConfLow, class TDevelopConfLow,
class TRegConfCriticalLow2Low, class TDevelopConfCriticalLow2Low,
class TRegConfCriticalLow, class TDevelopConfCriticalLow,
class TRegConfCycleTimeAttenuation, class TDevelopConfCycleTimeAttenuation,
class TRegConfTemp4SystemFanSpeed_000, class TDevelopConfTemp4SystemFanSpeed_000,
class TRegConfTemp4SystemFanSpeed_025, class TDevelopConfTemp4SystemFanSpeed_025,
class TRegConfTemp4SystemFanSpeed_050, class TDevelopConfTemp4SystemFanSpeed_050,
class TRegConfTemp4SystemFanSpeed_075, class TDevelopConfTemp4SystemFanSpeed_075,
class TRegConfTemp4SystemFanSpeed_MAX, class TDevelopConfTemp4SystemFanSpeed_MAX>
class fc_heatctrl_ThermalSensor_ConfigTemplate
: public virtual hc_tclThermalSensor_Config
{
public:
   // constructor
   fc_heatctrl_ThermalSensor_ConfigTemplate() : hc_tclThermalSensor_Config()
   {;}
   // destructor
   virtual ~fc_heatctrl_ThermalSensor_ConfigTemplate() {;};
	// loading configuration
	virtual tVoid vLoadConfig(tenThermalSensorID enSensorID)
	{
	  I_hc_tclCfgMgr* _poCfgMgr = OSAL_NULL;
	  if (OSAL_NULL != hc_tclAppMain::theServer())
		  _poCfgMgr = dynamic_cast<I_hc_tclCfgMgr*>(hc_tclAppMain::theServer()->getHandler("I_hc_tclCfgMgr"));
      // sensor enable
	  TRegEnable _oEnable;
	  TDevelopEnable _oEnable_Develop;
      m_bConfigEnable = (TRUE == _oEnable_Develop.tGetData());
      (tVoid)bSetConfigEnableSource(enConfigSource_Developer);
      if(TRUE < _oEnable_Develop.tGetData())
      {
          if((OSAL_NULL != _poCfgMgr) && (TRUE >= _poCfgMgr->u8GetKDSSensorEnable(enSensorID)))
          {
        	  m_bConfigEnable = (TRUE == _poCfgMgr->u8GetKDSSensorEnable(enSensorID));
              (tVoid)bSetConfigEnableSource(enConfigSource_KDS);
          }
          else
          {
        	  m_bConfigEnable = _oEnable.tGetData();
              (tVoid)bSetConfigEnableSource(enConfigSource_Registry);
          }
      }
      // lowest valid temperature
      tenTempConfigItem _enTempConfigItem = enThermalSensorConfigItem_TEMP_MIN;
      TRegConfMinTemp _oMinTemp;
      m_as16ConfigTemp[_enTempConfigItem] = HC_TEMP_INVALID_VALUE;
      if (OSAL_NULL != _poCfgMgr)
    	  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorMinTemp(enSensorID);
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
          m_as16ConfigTemp[_enTempConfigItem] = _oMinTemp.tGetData();
          (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
      }
      // highest valid temperature
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_MAX;
      TRegConfMaxTemp _oMaxTemp;
      m_as16ConfigTemp[_enTempConfigItem] = HC_TEMP_INVALID_VALUE;
      if (OSAL_NULL != _poCfgMgr)
    	  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorMaxTemp(enSensorID);
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
          m_as16ConfigTemp[_enTempConfigItem] = _oMaxTemp.tGetData();
          (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
      }
      // temperatur configuration
      // ... EMERGENCY_HIGH Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_EH;
	  tenKDSDataSensorThreshold _enKDSTempConfigItem = enKDSDataSensorThreshold_CriticalHigh;
	  tBool _bIsHysteresis = FALSE;
      TRegConfEmeregencyHigh _oTempEH;
      TDevelopConfEmeregencyHigh _oTempEH_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempEH_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempEH.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... EMERGENCY_HIGH to HIGH Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_EH2H;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_CriticalHigh;
	  _bIsHysteresis = TRUE;
      TRegConfEmeregencyHigh2High _oTempEH2H;
      TDevelopConfEmeregencyHigh2High _oTempEH2H_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempEH2H_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempEH2H.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... HIGH Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_H;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_High;
	  _bIsHysteresis = FALSE;
      TRegConfHigh _oTempH;
      TDevelopConfHigh _oTempH_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempH_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempH.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... HIGH to WARM Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_H2W;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_High;
	  _bIsHysteresis = TRUE;
      TRegConfHigh2Warm _oTempH2W;
      TDevelopConfHigh2Warm _oTempH2W_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempH2W_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempH2W.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... WARM Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_W;
      TRegConfWarm _oTempW;
      TDevelopConfWarm _oTempW_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempW_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
          m_as16ConfigTemp[_enTempConfigItem] = _oTempW.tGetData();
          (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
      }
      tenTempConfigItem _enAltTempConfigItem = enThermalSensorConfigItem_TEMP_H2W;
      if(m_as16ConfigTemp[_enTempConfigItem] > m_as16ConfigTemp[_enAltTempConfigItem])
      {
    	  if(enConfigSource_PDD == enGetConfigTempSource(_enAltTempConfigItem))
    	  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD_Default);
    	  }
    	  else if(enConfigSource_KDS == enGetConfigTempSource(_enAltTempConfigItem))
    	  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_KDS_Default);
    	  }
      }
      // ... WARM to Normal Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_W2N;
      TRegConfWarm2Normal _oTempW2N;
      TDevelopConfWarm2Normal _oTempW2N_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempW2N_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
          m_as16ConfigTemp[_enTempConfigItem] = _oTempW2N.tGetData();
          (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
      }
      _enAltTempConfigItem = enThermalSensorConfigItem_TEMP_W;
      if(m_as16ConfigTemp[_enTempConfigItem] > m_as16ConfigTemp[_enAltTempConfigItem])
      {
		  if(enConfigSource_PDD_Default == enGetConfigTempSource(_enAltTempConfigItem))
		  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD_Default);
		  }
		  else if(enConfigSource_KDS_Default == enGetConfigTempSource(_enAltTempConfigItem))
		  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_KDS_Default);
		  }
      }
      // ... CRITICAL_LOW Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_CL;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_CriticalLow;
	  _bIsHysteresis = FALSE;
      TRegConfCriticalLow _oTempCL;
      TDevelopConfCriticalLow _oTempCL_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempCL_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempCL.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... CRITICAL_LOW to LOW Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_CL2L;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_CriticalLow;
	  _bIsHysteresis = TRUE;
      TRegConfCriticalLow2Low _oTempCL2L;
      TDevelopConfCriticalLow2Low _oTempCL2L_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempCL2L_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempCL2L.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... LOW Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_L;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_Low;
	  _bIsHysteresis = FALSE;
      TRegConfLow _oTempL;
      TDevelopConfLow _oTempL_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempL_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempL.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... LOW to COLD Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_L2C;
	  _enKDSTempConfigItem = enKDSDataSensorThreshold_Low;
	  _bIsHysteresis = TRUE;
      TRegConfLow2Cold _oTempL2C;
      TDevelopConfLow2Cold _oTempL2C_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempL2C_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if (OSAL_NULL != _poCfgMgr)
    		  m_as16ConfigTemp[_enTempConfigItem] = _poCfgMgr->s16GetPDDSensorThreshold(enSensorID,_enKDSTempConfigItem,_bIsHysteresis);
		  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD);
	      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
	      {
			  m_as16ConfigTemp[_enTempConfigItem] = _oTempL2C.tGetData();
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
	      }
      }
      // ... COLD Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_C;
      TRegConfCold _oTempC;
      TDevelopConfCold _oTempC_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempC_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
          m_as16ConfigTemp[_enTempConfigItem] = _oTempC.tGetData();
          (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
      }
      _enAltTempConfigItem = enThermalSensorConfigItem_TEMP_L2C;
      if(m_as16ConfigTemp[_enAltTempConfigItem] > m_as16ConfigTemp[_enTempConfigItem])
      {
    	  if(enConfigSource_PDD == enGetConfigTempSource(_enAltTempConfigItem))
    	  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD_Default);
    	  }
    	  else if(enConfigSource_KDS == enGetConfigTempSource(_enAltTempConfigItem))
    	  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_KDS_Default);
    	  }
      }
      // ... COLD to Normal Temperature Threshold
      _enTempConfigItem = enThermalSensorConfigItem_TEMP_C2N;
      TRegConfCold2Normal _oTempC2N;
      TDevelopConfCold2Normal _oTempC2N_Develop;
	  m_as16ConfigTemp[_enTempConfigItem] = _oTempC2N_Develop.tGetData();
      (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Developer);
      if(HC_TEMP_INVALID_VALUE >= m_as16ConfigTemp[_enTempConfigItem])
      {
          m_as16ConfigTemp[_enTempConfigItem] = _oTempC2N.tGetData();
          (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_Registry);
      }
      _enAltTempConfigItem = enThermalSensorConfigItem_TEMP_C;
      if(m_as16ConfigTemp[_enAltTempConfigItem] > m_as16ConfigTemp[_enTempConfigItem])
      {
		  if(enConfigSource_PDD_Default == enGetConfigTempSource(_enAltTempConfigItem))
		  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_PDD_Default);
		  }
		  else if(enConfigSource_KDS_Default == enGetConfigTempSource(_enAltTempConfigItem))
		  {
			  m_as16ConfigTemp[_enTempConfigItem] = m_as16ConfigTemp[_enAltTempConfigItem];
			  (tVoid)bSetConfigTempSource(_enTempConfigItem,enConfigSource_KDS_Default);
		  }
      }
      // attenuation cycle time
      TRegConfCycleTimeAttenuation _oCycleTimeAttenuation;
      TDevelopConfCycleTimeAttenuation _oCycleTimeAttenuation_Develop;
	  m_u8CycleTimeAttenuation = _oCycleTimeAttenuation_Develop.tGetData();
      (tVoid)bSetConfigAttenuationCycleTimeSource(enConfigSource_Developer);
      if(HC_UNCONFIGURED_CYCLETIME == m_u8CycleTimeAttenuation)
      {
          m_u8CycleTimeAttenuation = _oCycleTimeAttenuation.tGetData();
          (tVoid)bSetConfigAttenuationCycleTimeSource(enConfigSource_Registry);
      }
      // load configured fan speeds
      vLoadConfig_FanSpeed();
   };
   // loading fan configuration
   virtual tVoid vLoadConfig_FanSpeed(tVoid)
   {
	    tenFANID _enFANID = enFANID_SYSTEM;
		// temperature threshold for system fan speed 0 percent
	    {
			tenFANSpeedLevel _enSpeedLevel = enFANSpeedLevel_0;
			TRegConfTemp4SystemFanSpeed_000 _oTemp4SystemFanSpeed;
			TDevelopConfTemp4SystemFanSpeed_000 _oTemp4SystemFanSpeed_Develop;
			m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed_Develop.tGetData();
			m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Developer;
			if(HC_UNCONFIGURED_TEMP == m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel])
			{
				m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed.tGetData();
				m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Registry;
			}
	    }
		// temperature threshold for system fan speed 25 percent
	    {
			tenFANSpeedLevel _enSpeedLevel = enFANSpeedLevel_25;
			TRegConfTemp4SystemFanSpeed_025 _oTemp4SystemFanSpeed;
			TDevelopConfTemp4SystemFanSpeed_025 _oTemp4SystemFanSpeed_Develop;
			m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed_Develop.tGetData();
			m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Developer;
			if(HC_UNCONFIGURED_TEMP == m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel])
			{
				m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed.tGetData();
				m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Registry;
			}
	    }
		// temperature threshold for system fan speed 50 percent
	    {
			tenFANSpeedLevel _enSpeedLevel = enFANSpeedLevel_50;
			TRegConfTemp4SystemFanSpeed_050 _oTemp4SystemFanSpeed;
			TDevelopConfTemp4SystemFanSpeed_050 _oTemp4SystemFanSpeed_Develop;
			m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed_Develop.tGetData();
			m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Developer;
			if(HC_UNCONFIGURED_TEMP == m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel])
			{
				m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed.tGetData();
				m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Registry;
			}
	    }
		// temperature threshold for system fan speed 75 percent
	    {
			tenFANSpeedLevel _enSpeedLevel = enFANSpeedLevel_75;
			TRegConfTemp4SystemFanSpeed_075 _oTemp4SystemFanSpeed;
			TDevelopConfTemp4SystemFanSpeed_075 _oTemp4SystemFanSpeed_Develop;
			m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed_Develop.tGetData();
			m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Developer;
			if(HC_UNCONFIGURED_TEMP == m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel])
			{
				m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed.tGetData();
				m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Registry;
			}
	    }
		// temperature threshold for system fan speed 100 percent
	    {
			tenFANSpeedLevel _enSpeedLevel = enFANSpeedLevel_100;
			TRegConfTemp4SystemFanSpeed_MAX _oTemp4SystemFanSpeed;
			TDevelopConfTemp4SystemFanSpeed_MAX _oTemp4SystemFanSpeed_Develop;
			m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed_Develop.tGetData();
			m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Developer;
			if(HC_UNCONFIGURED_TEMP == m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel])
			{
				m_as16ConfigTempFanSpeed[_enFANID][_enSpeedLevel] = _oTemp4SystemFanSpeed.tGetData();
				m_aenConfigTempFanSpeed_Source[_enFANID][_enSpeedLevel] = enConfigSource_Registry;
			}
	    }
   };
   // set sensor availability persistent
   virtual tBool bSetConfigEnablePers(tU8 u8Enable)
   {
      if(bSetConfigEnable( (0 < u8Enable) ))
      {
         TDevelopEnable _oDbConfEnable;
         if(_oDbConfEnable.tGetData() != u8Enable)
        	 _oDbConfEnable.vSetData(u8Enable);
         if(TRUE < u8Enable)
        	 (tVoid)bSetConfigEnableSource(enConfigSource_Default);
         else
        	 (tVoid)bSetConfigEnableSource(enConfigSource_Developer);
         return TRUE;
      }
      return FALSE;
   };
   // set sensor temperature threshold persistent
   virtual tBool bSetConfigTempPers(tenTempConfigItem enItem, tS16 s16Temperature)
   {
      if(bSetConfigTemp(enItem,s16Temperature))
      {
         switch(enItem)
         {
         case enThermalSensorConfigItem_TEMP_EH:
            {
               TDevelopConfEmeregencyHigh _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_EH2H:
            {
               TDevelopConfEmeregencyHigh2High _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_H:
            {
               TDevelopConfHigh _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_H2W:
            {
               TDevelopConfHigh2Warm _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_W:
            {
               TDevelopConfWarm _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_W2N:
            {
               TDevelopConfWarm2Normal _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_C2N:
            {
               TDevelopConfCold2Normal _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_C:
            {
               TDevelopConfCold _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_L2C:
            {
               TDevelopConfLow2Cold _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_L:
            {
               TDevelopConfLow _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_CL2L:
            {
               TDevelopConfCriticalLow2Low _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         case enThermalSensorConfigItem_TEMP_CL:
            {
               TDevelopConfCriticalLow _oDbConfTemp;
               if(_oDbConfTemp.tGetData() != s16Temperature)
                  _oDbConfTemp.vSetData(s16Temperature);
            }
            break;
         default:
            {
               return FALSE;
            }
         }
         if(HC_TEMP_INVALID_VALUE < s16Temperature)
        	 bSetConfigTempSource(enItem, enConfigSource_Developer);
         else
        	 bSetConfigTempSource(enItem, enConfigSource_Default);
         return TRUE;
      }
      return FALSE;
   };
   // set fan temperature threshold persistent
   virtual tBool bSetConfigFanTempPers(tenFANID enFan, tenFANSpeedLevel enSpeedLevel, tS16 s16Temperature)
   {
       if(bSetConfigFanTemp(enFan,enSpeedLevel,s16Temperature))
	   {
    	   if (enFANID_SYSTEM == enFan)
    	   {
    		   switch (enSpeedLevel)
    		   {
    		   case enFANSpeedLevel_0:
				   {
					   TDevelopConfTemp4SystemFanSpeed_000 _oDpConfFanTemp;
		               if(_oDpConfFanTemp.tGetData() != s16Temperature)
		            	   _oDpConfFanTemp.vSetData(s16Temperature);
				   }
				   break;
    		   case enFANSpeedLevel_25:
				   {
					   TDevelopConfTemp4SystemFanSpeed_025 _oDpConfFanTemp;
		               if(_oDpConfFanTemp.tGetData() != s16Temperature)
		            	   _oDpConfFanTemp.vSetData(s16Temperature);
				   }
				   break;
    		   case enFANSpeedLevel_50:
				   {
					   TDevelopConfTemp4SystemFanSpeed_050 _oDpConfFanTemp;
		               if(_oDpConfFanTemp.tGetData() != s16Temperature)
		            	   _oDpConfFanTemp.vSetData(s16Temperature);
				   }
				   break;
    		   case enFANSpeedLevel_75:
				   {
					   TDevelopConfTemp4SystemFanSpeed_075 _oDpConfFanTemp;
		               if(_oDpConfFanTemp.tGetData() != s16Temperature)
		            	   _oDpConfFanTemp.vSetData(s16Temperature);
				   }
				   break;
    		   case enFANSpeedLevel_100:
				   {
					   TDevelopConfTemp4SystemFanSpeed_MAX _oDpConfFanTemp;
		               if(_oDpConfFanTemp.tGetData() != s16Temperature)
		            	   _oDpConfFanTemp.vSetData(s16Temperature);
				   }
				   break;
    		   default:
				   {
					   return FALSE;
				   }
				   break;
    		   }
    		   return TRUE;
    	   }
	   }
	   return FALSE;
   };
   // set sensor specific cycle time of attenuation persistent
   virtual tBool bSetConfigAttenuationCycleTimePers(tU8 u8CycleTime)
   {
      if(bSetConfigAttenuationCycleTime(u8CycleTime))
      {
         TDevelopConfCycleTimeAttenuation _oCycleTimeAttenuation_Develop;
         if(_oCycleTimeAttenuation_Develop.tGetData() != u8CycleTime)
            _oCycleTimeAttenuation_Develop.vSetData(u8CycleTime);
         if(HC_UNCONFIGURED_CYCLETIME > u8CycleTime)
        	 bSetConfigAttenuationCycleTimeSource(enConfigSource_Developer);
         else
        	 bSetConfigAttenuationCycleTimeSource(enConfigSource_Default);
         return TRUE;
      }
      return FALSE;
   };
};

#endif /* HC_TCLTHERMALSENSOR_CONFIG_H_ */
