/************************************************************************
* FILE:        hc_tclThermalSensor_Config.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: base class of thermal sensors configuration
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 05.02.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclThermalSensor_Config.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclThermalSensor_Config.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclThermalSensor_Config()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_Config::hc_tclThermalSensor_Config()
{
   ETG_TRACE_USR4(("hc_tclThermalSensor_Config() entered."));
   m_bConfigEnable = FALSE;
   m_u8CycleTimeAttenuation = 0;
   m_enConfigEnableSource = enConfigSource_UNDEF;
   m_enCycleTimeAttenuationSource = enConfigSource_UNDEF;
   (tVoid)memset((tVoid*)m_as16ConfigTemp,OSAL_NULL,sizeof(m_as16ConfigTemp));
   (tVoid)memset((tVoid*)m_aenConfigTempSource,OSAL_NULL,sizeof(m_aenConfigTempSource));
   // setup fan speeds
   (tVoid)memset((tVoid*)m_au8ConfigFANSpeed, OSAL_NULL, sizeof(m_au8ConfigFANSpeed));
   m_au8ConfigFANSpeed[enThermalState_EMERGENCYHIGH] = 100;
   m_au8ConfigFANSpeed[enThermalState_HIGH] = 75;
   m_au8ConfigFANSpeed[enThermalState_WARM] = 25;
   // temperature level
   m_s16ConfigTempFanSpeedOffset = -200;
   (tVoid)memset((tVoid*)m_as16ConfigTempFanSpeed,OSAL_NULL,sizeof(m_as16ConfigTempFanSpeed));
   (tVoid)memset((tVoid*)m_aenConfigTempFanSpeed_Source,OSAL_NULL,sizeof(m_aenConfigTempFanSpeed_Source));
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclThermalSensor_Config()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclThermalSensor_Config::~hc_tclThermalSensor_Config()
{
   ETG_TRACE_USR4(("~hc_tclThermalSensor_Config() entered."));
}

/*******************************************************************************
*
* FUNCTION:    bIsConfigValid()
*
* DESCRIPTION: validation of configuration
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
tBool hc_tclThermalSensor_Config::bIsConfigValid() const
{
	tBool _bReturn = TRUE;
	for(tU8 _u8Index = (tU8)enThermalSensorConfigItem_TEMP_MIN;
			((tU8)enThermalSensorConfigItem_MAX-1) > _u8Index; ++_u8Index)
	{
		if(s16GetConfigTemp((tenTempConfigItem)_u8Index) > s16GetConfigTemp((tenTempConfigItem)(_u8Index+1)))
			_bReturn = FALSE;
	}
	return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    vLoadConfig_FanSpeed()
*
* DESCRIPTION: setup fan speed temperature thresholds
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_Config::vLoadConfig_FanSpeed(tVoid)
{
	(tVoid)memset(m_as16ConfigTempFanSpeed, OSAL_NULL, sizeof(m_as16ConfigTempFanSpeed));

	m_s16ConfigTempFanSpeedOffset = -200;
	// configure system fan
	tenFANID _enFanID = enFANID_SYSTEM;
	// ... temperature of fan speed 0%
	tenFANSpeedLevel _enFanSpeedLevel = enFANSpeedLevel_0;
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] = (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W2N) - s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C2N))/2;
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] += m_s16ConfigTempFanSpeedOffset;
	m_aenConfigTempFanSpeed_Source[_enFanID][_enFanSpeedLevel] = enConfigSource_Default;
	// ... temperature of fan speed 25%
	_enFanSpeedLevel = enFANSpeedLevel_25;
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] = s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W);
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] += m_s16ConfigTempFanSpeedOffset;
	m_aenConfigTempFanSpeed_Source[_enFanID][_enFanSpeedLevel] = enConfigSource_Default;
	// ... temperature of fan speed 50%
	_enFanSpeedLevel = enFANSpeedLevel_50;
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] = (s16GetConfigTemp(enThermalSensorConfigItem_TEMP_H) - s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W)) + s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W);
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] += m_s16ConfigTempFanSpeedOffset;
	m_aenConfigTempFanSpeed_Source[_enFanID][_enFanSpeedLevel] = enConfigSource_Default;
	// ... temperature of fan speed 75%
	_enFanSpeedLevel = enFANSpeedLevel_75;
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] = s16GetConfigTemp(enThermalSensorConfigItem_TEMP_H);
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] += m_s16ConfigTempFanSpeedOffset;
	m_aenConfigTempFanSpeed_Source[_enFanID][_enFanSpeedLevel] = enConfigSource_Default;
	// ... temperature of fan speed 100%
	_enFanSpeedLevel = enFANSpeedLevel_100;
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] = s16GetConfigTemp(enThermalSensorConfigItem_TEMP_EH);
	m_as16ConfigTempFanSpeed[_enFanID][_enFanSpeedLevel] += m_s16ConfigTempFanSpeedOffset;
	m_aenConfigTempFanSpeed_Source[_enFanID][_enFanSpeedLevel] = enConfigSource_Default;
}

/*******************************************************************************
*
* FUNCTION:    bIsFanConfigValid()
*
* DESCRIPTION: validation of fan configuration
*
* PARAMETER:   tenFANID enFANID			specified fan
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclThermalSensor_Config::bIsFanConfigValid(tenFANID enFANID) const
{
	if (HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed) > (tU16)enFANID)
	{
		tS16 _s16LowerTemp = s16GetConfigFanTemp(enFANID,enFANSpeedLevel_0);
		for (tU16 _u16Idx = 1; HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0]) > _u16Idx; ++_u16Idx)
		{
			if (_s16LowerTemp > s16GetConfigFanTemp(enFANID,(tenFANSpeedLevel)_u16Idx))
			{
			    ETG_TRACE_ERR(("bIsFanConfigValid: invalid sensor fan speed configuration! (fan: %u, temp[%u]: %ddC  temp[%u]: %ddC) ",
			    		ETG_ENUM(HC_FANID, enFANID),
			    		ETG_ENUM(HC_FANTHRESHOLDID, _u16Idx-1),
			    		s16GetConfigFanTemp(enFANID,(tenFANSpeedLevel)(_u16Idx-1)),
			    		ETG_ENUM(HC_FANTHRESHOLDID, _u16Idx),
			    		s16GetConfigFanTemp(enFANID,(tenFANSpeedLevel)(_u16Idx)) ));
			    return FALSE;
			}
		}
		return TRUE;
	}
	return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u8GetConfigFANSpeed4Temp()
*
* DESCRIPTION: constructor
*
* PARAMETER:   tenFANID enFANID    fan id
*              tS16 s16Temp        temperature
*
* RETURNVALUE: tU8                 fan speed
*
*******************************************************************************/
tU8 hc_tclThermalSensor_Config::u8GetConfigFANSpeed4Temp(tenFANID enFANID, tS16 s16Temp) const
{
    ETG_TRACE_USR2(("u8GetConfigFANSpeed4Temp(fan: %u, temp: %d) entered.", enFANID, s16Temp ));

	tU8 _u8Speed = 0;
    if ((HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed) > (tU16)enFANID) && (TRUE == bIsFanConfigValid(enFANID)))
    {
    	tU16 _u16SpeedIdx = 0;
    	for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0]) > _u16Idx; ++_u16Idx)
    	{
    		if (m_as16ConfigTempFanSpeed[enFANID][_u16Idx] <= s16Temp)
    			_u16SpeedIdx = _u16Idx;
    	}
		if ((HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0])-1) <= _u16SpeedIdx)
		{
			// temperature grater or equal maximum speed threshold temperature (speed 100%)
			_u8Speed = 100;
		}
		else if (m_as16ConfigTempFanSpeed[enFANID][0] < s16Temp)
		{
			// temperature grater or equal minimum speed threshold temperature (speed 0%)
			tS16 _s16LowerThreshold = m_as16ConfigTempFanSpeed[enFANID][_u16SpeedIdx];
			tS16 _s16UpperThreshold = m_as16ConfigTempFanSpeed[enFANID][_u16SpeedIdx + 1];
			tS16 _s16Speed = (s16Temp - _s16LowerThreshold) * 25;
			_s16Speed /= (_s16UpperThreshold - _s16LowerThreshold);
			_s16Speed += _u16SpeedIdx * 25;
			_u8Speed = (tU8)_s16Speed;
		}
		if (100 < _u8Speed)
			_u8Speed = 100;
    }
    ETG_TRACE_USR2(("u8GetConfigFANSpeed4Temp: fan id = %u, speed = %u percent, temp = %d dC ", (tU16)enFANID, _u8Speed, s16Temp ));
	return _u8Speed;
}


/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_Config()
*
* DESCRIPTION: 	trace configuration status information
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensor_Config::vTraceInfo_Config(const char* strIdentifier)
{
   ETG_TRACE_USR4(( "vTraceInfo_Config() entered." ));

   char _strAttrName[60] = {0};

   ETG_TRACE_USR2(("%26s: %30s=%u {src=%u} ", strIdentifier, 		"          CFG_ENABLE",
		   ETG_ENUM(HC_Bool, bGetConfigEnable()),
		   ETG_ENUM(HC_CFGSRC, enGetConfigEnableSource()) ));
   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, 				"           CFG_VALID",
		   ETG_ENUM(HC_Bool, bIsConfigValid()) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"        CFG_TEMP_MAX",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_MAX),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_MAX)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"         CFG_TEMP_EH",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_EH),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_EH)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"       CFG_TEMP_EH2H",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_EH2H),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_EH2H)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"          CFG_TEMP_H",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_H),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_H)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier,	"        CFG_TEMP_H2W",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_H2W),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_H2W)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"          CFG_TEMP_W",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_W)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"        CFG_TEMP_W2N",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_W2N),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_W2N)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"        CFG_TEMP_C2N",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C2N),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_C2N)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"          CFG_TEMP_C",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_C),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_C)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"        CFG_TEMP_L2C",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_L2C),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_L2C)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"          CFG_TEMP_L",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_L),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_L)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"       CFG_TEMP_CL2L",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_CL2L),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_CL2L)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"         CFG_TEMP_CL",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_CL),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_CL)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u} ", strIdentifier, 	"       CFG_TEMP__MIN",
		   s16GetConfigTemp(enThermalSensorConfigItem_TEMP_MIN),
		   ETG_ENUM(HC_CFGSRC, enGetConfigTempSource(enThermalSensorConfigItem_TEMP_MIN)) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [s] {src=%u} ", strIdentifier, 	"       ATT_CYCLETIME",
		   u8GetConfigAttenuationCycleTime(),
		   ETG_ENUM(HC_CFGSRC, enGetConfigAttenuationCycleTimeSource()) ));
   // fan speed
   /* TODO: sensor state based fan speed config
   ETG_TRACE_USR2(("%26s: %30s=%d [perc] ", strIdentifier, "    CFG_FAN_SPEED_EH",
		   u8GetConfigFANSpeed(enThermalState_EMERGENCYHIGH) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [perc] ", strIdentifier, "     CFG_FAN_SPEED_H",
		   u8GetConfigFANSpeed(enThermalState_HIGH) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [perc] ", strIdentifier, "     CFG_FAN_SPEED_W",
		   u8GetConfigFANSpeed(enThermalState_WARM) ));
   ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentifier,   "CFG_FAN_SPEED_OFFSET",
		   m_s16ConfigTempFanSpeedOffset ));
   */
   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed) > _u16Idx; ++_u16Idx)
   {
	   sprintf(_strAttrName, " CFG_FAN_SPEED_%01u_NAM", _u16Idx);
	   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, _strAttrName,
			   ETG_ENUM(HC_FANID, _u16Idx) ));
	   sprintf(_strAttrName, " CFG_FAN_SPEED_%01u_VAL", _u16Idx);
	   ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier, _strAttrName,
			   ETG_ENUM(HC_Bool, bIsFanConfigValid((tenFANID)_u16Idx)) ));
	   tU16 _u16LevPerc = (100 / (HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0])-1));
	   for (tU16 _u16Lev = 0; HC_C_U16_ARRAYELEMENTS(m_as16ConfigTempFanSpeed[0]) > _u16Lev; ++_u16Lev)
	   {
		   sprintf(_strAttrName, " CFG_FAN_SPEED_%01u_%03u", _u16Idx, (_u16Lev * _u16LevPerc) );
		   ETG_TRACE_USR2(("%26s: %30s=%d [dC] {src=%u}", strIdentifier, _strAttrName,
				   s16GetConfigFanTemp((tenFANID)_u16Idx, (tenFANSpeedLevel)_u16Lev),
				   ETG_ENUM(HC_CFGSRC, enGetConfigFanTempSource((tenFANID)_u16Idx, (tenFANSpeedLevel)_u16Lev)) ));
	   }
   }
}
