/************************************************************************
* FILE:        hc_tclThermalSensorControl.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: classes to control all thermal sensors
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 02.03.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "I_hc_tclSimuMode.h"
#include "hc_tclThermalSensorControl.h"
#include "hc_tclThermalSensor.h"
#include "hc_tclThermalSensorV850.h"
#include "hc_tclThermalSensorPanel.h"
#include "hc_tclThermalSensorCD.h"
#include "hc_tclThermalSensorDVD.h"
#include "hc_tclThermalSensorGPS.h"
#include "hc_tclThermalSensorGYRO.h"
#include "hc_tclThermalSensorEMMC.h"
#include "hc_tclThermalSensorAMP.h"
#include "hc_tclThermalSensorIMX.h"
#include "hc_tclThermalSensorHEATSINK.h"
#include "hc_tclThermalSensorBLUERAY.h"
#include "hc_tclThermalSensorXMMODULE.h"
#include "hc_tclThermalSensorTVMODULE.h"
#include "hc_tclThermalSensorTVMODULESLAVE.h"
#include "hc_tclThermalSensorPanel2.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclThermalSensorControl.cpp.trc.h"
#endif

hc_tclThermalSensor* hc_tclThermalSensorControl::m_poThermalSensors[enThermalSensorID_MAX];

/*************************************************************************
*
* FUNCTION:    hc_tclThermalSensorControl()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain* poMainAppl: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclThermalSensorControl::hc_tclThermalSensorControl(const hc_tclAppMain* poMainAppl)
: I_hc_tclThermalSensorControl(poMainAppl)
{
   ETG_TRACE_USR4(("hc_tclThermalSensorControl() entered."));
   // simulation mode
   m_poSimuMode = OSAL_NULL;
   // main thermal state
   m_enMainThermalState = enThermalState_INIT;
   // prepare internal messages to own service handler
   (tVoid)memset((tVoid*)&m_msgSensorUpdate_SrvHdl,OSAL_NULL,sizeof(m_msgSensorUpdate_SrvHdl));
   (tVoid)OSAL_szStringNCopy(m_msgSensorUpdate_SrvHdl.strClassName,
		   "Ihc_tclServiceHeatIf", sizeof(m_msgSensorUpdate_SrvHdl.strClassName)-1);
   // prepare internal messages to service handler AUTOSAR
   (tVoid)memset((tVoid*)&m_msgSensorUpdate_SrvHdl_AUTOSAR,OSAL_NULL,sizeof(m_msgSensorUpdate_SrvHdl_AUTOSAR));
   (tVoid)OSAL_szStringNCopy(m_msgSensorUpdate_SrvHdl_AUTOSAR.strClassName,
		   "I_hc_tclClientThermal", sizeof(m_msgSensorUpdate_SrvHdl_AUTOSAR.strClassName)-1);
   // prepare interanl message to emergency function controller
   (tVoid)memset((tVoid*)&m_msgEmergencyFunctionStates,OSAL_NULL,sizeof(m_msgEmergencyFunctionStates));
   m_msgEmergencyFunctionStates.eCmd = hc_tclBaseIf::eNewEmergencyFunctionState;
   (tVoid)OSAL_szStringNCopy(m_msgEmergencyFunctionStates.strClassName,
		   "I_hc_tclEmergencyFunctionControl", sizeof(m_msgEmergencyFunctionStates.strClassName)-1);
   // prepare interanl message to ITC controller
   (tVoid)memset((tVoid*)&m_msgITCStates,OSAL_NULL,sizeof(m_msgITCStates));
   m_msgITCStates.eCmd = hc_tclBaseIf::eNewITCState;
   (tVoid)OSAL_szStringNCopy(m_msgITCStates.strClassName,
		   "I_hc_tclITCControl", sizeof(m_msgITCStates.strClassName)-1);
   // prepare internal message to FAN controller
   (tVoid)memset((tVoid*)&m_msgFANSpeed,OSAL_NULL,sizeof(m_msgFANSpeed));
   m_msgFANSpeed.eCmd = hc_tclBaseIf::eNewSensorFanSpeed;
   (tVoid)OSAL_szStringNCopy(m_msgFANSpeed.strClassName,
		   "I_hc_tclFANControl", sizeof(m_msgFANSpeed.strClassName)-1);
   // Instantiate all thermal sensor objects
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorV850())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorV850!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_V850))
   {
	   poGetThermalSensor(enThermalSensorID_V850)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorPanel())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorPanel!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_Panel))
   {
	   poGetThermalSensor(enThermalSensorID_Panel)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorCD())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorCD!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_CDDRIVE))
   {
	   poGetThermalSensor(enThermalSensorID_CDDRIVE)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorDVD())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorDVD!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_DVDDRIVE))
   {
	   poGetThermalSensor(enThermalSensorID_DVDDRIVE)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorGPS())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorGPS!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_GPS))
   {
	   poGetThermalSensor(enThermalSensorID_GPS)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorGYRO())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorGYRO!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_GYRO))
   {
	   poGetThermalSensor(enThermalSensorID_GYRO)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorEMMC())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorEMMC!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_EMMC))
   {
	   poGetThermalSensor(enThermalSensorID_EMMC)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorAMP())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorAMP!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_AMP))
   {
	   poGetThermalSensor(enThermalSensorID_AMP)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorIMX())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorIMX!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_IMX))
   {
	   poGetThermalSensor(enThermalSensorID_IMX)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorHEATSINK())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorHEATSINK!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_HEATSINK))
   {
	   poGetThermalSensor(enThermalSensorID_HEATSINK)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorBLUERAY())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorBLUERAY!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_BLUERAY))
   {
	   poGetThermalSensor(enThermalSensorID_BLUERAY)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorXMMODULE())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorXMMODULE!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_XMModule))
   {
	   poGetThermalSensor(enThermalSensorID_XMModule)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorTVMODULE())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorTVMODULE!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_TVModule))
   {
	   poGetThermalSensor(enThermalSensorID_TVModule)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorTVMODULESLAVE())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorTVMODULESLAVE!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_TVModuleSlave))
   {
	   poGetThermalSensor(enThermalSensorID_TVModuleSlave)->vGetReferences();
   }
   if (OSAL_NULL == OSAL_NEW hc_tclThermalSensorPanel2())
   {
	   ETG_TRACE_FATAL(("vGetReferences: couldn't instantiate hc_tclThermalSensorPanel2!"));
   }
   else if (OSAL_NULL != poGetThermalSensor(enThermalSensorID_Panel2))
   {
	   poGetThermalSensor(enThermalSensorID_Panel2)->vGetReferences();
   }
}


/*************************************************************************
*
* FUNCTION:    ~hc_tclThermalSensorControl()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclThermalSensorControl::~hc_tclThermalSensorControl()
{
   ETG_TRACE_USR4(("~hc_tclThermalSensorControl() entered."));
   m_poSimuMode = OSAL_NULL;
   m_enMainThermalState = enThermalState_UNAVAILABLE;
   for (tU8 _u8Index = 0; (tU8)enThermalSensorID_MAX > _u8Index; ++_u8Index)
   {
	   if(OSAL_NULL != m_poThermalSensors[_u8Index])
		   OSAL_DELETE m_poThermalSensors[_u8Index];
	   m_poThermalSensors[_u8Index] = OSAL_NULL;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage()
*
* DESCRIPTION: 	handler for internal messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received messagde
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
   HC_NULL_POINTER_CHECK(pMsg);
   switch (pMsg->eCmd)
   {
   case hc_tclBaseIf::eNewThermalInfo_STANDARD:
   case hc_tclBaseIf::eNewThermalInfo_AUTOSAR:
	   {
		   vHandleMessage_NewThermalInfo(*pMsg);
	   }
	   break;
   case hc_tclBaseIf::eTriggerLoadSettings:
	   {
		   vHandleMessage_LoadSettings(*pMsg);
	   }
	   break;
   default:
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command messages
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
   switch (u32MsgCode) {

   case FC_HEATCTRL_HC_GETSENSORSTATE:
       {
    	   vHandleTraceCmd_GetThermSensorState(puchData);
       }
       break;

   case FC_HEATCTRL_HC_SIMUSENSOR_IF_FLAG:
       {
    	   vHandleTraceCmd_Simu_IF_FLAG(puchData);
       }
       break;

   case FC_HEATCTRL_HC_SIMUSENSOR_IF_TEMP:
       {
    	   vHandleTraceCmd_Simu_IF_TEMP(puchData);
       }
       break;

   case FC_HEATCTRL_HC_SIMUSENSOR_IF_STATE:
       {
    	   vHandleTraceCmd_Simu_IF_STATE(puchData);
       }
       break;

   case FC_HEATCTRL_HC_SETSENSOR_STATE:
       {
    	   vHandleTraceCmd_Set_STATE(puchData);
       }
       break;

   case FC_HEATCTRL_HC_SETSENSOR_LASTVALIDSTATE:
       {
    	   vHandleTraceCmd_Set_LASTVALIDSTATE(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSOR_ENABLE:
       {
    	   vHandleTraceCmd_Config_SensorEnable(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSOR_ENABLE_RESET:
       {
    	   vHandleTraceCmd_ConfigReset_SensorEnable(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSOR_THRESHOLD:
       {
    	   vHandleTraceCmd_Config_Threshold(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSOR_THRESHOLD_RESET:
       {
    	   vHandleTraceCmd_Config_Threshold_Reset(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSOR_ATTENUATIONCYCLETIME:
       {
    	   vHandleTraceCmd_Config_Attenuation_Cycletime(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSOR_ATTENUATIONCYCLETIME_RESET:
       {
    	   vHandleTraceCmd_Config_Attenuation_Cycletime_Reset(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSORFAN_THRESHOLD:
       {
    	   vHandleTraceCmd_Config_FanThreshold(puchData);
       }
       break;

   case FC_HEATCTRL_HC_CONFIGSENSORFAN_THRESHOLD_RESET:
       {
    	   vHandleTraceCmd_Config_FanThreshold_Reset(puchData);
       }
       break;

   case FC_HEATCTRL_HC_GETABSTRACTSENSORSTATE:
       {
    	   vHandleTraceCmd_GetAbstractSensorState(puchData);
       }
       break;

   default:
       ETG_TRACE_USR4(("hc_tclThermalSensorControl::vHandleTraceMessage(): unhandled command %d.", u32MsgCode));
       break;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));

   // reference of simulation mode interface
   m_poSimuMode = dynamic_cast<I_hc_tclSimuMode*>(_cpoMain->getHandler("I_hc_tclSimuMode"));
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   for( tU8 _u8SensorID = (tU8)enThermalSensorID_V850; (tU8)enThermalSensorID_MAX > _u8SensorID; ++_u8SensorID)
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)_u8SensorID);
	   if (OSAL_NULL != _poSensor)
	   {
		   _poSensor->vTraceInfo();
		   // collect emergency function state information of specified sensor
		   m_msgEmergencyFunctionStates.u.tEmergencyFunctionInfo.enSensorID = _poSensor->enGetSensorID();
		   if (TRUE == _poSensor->bGetNewEmergencyFunctionState(m_msgEmergencyFunctionStates.u.tEmergencyFunctionInfo))
			   // send message to emergency function controller
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgEmergencyFunctionStates);
		   // collect ITC state information of specified sensor
		   m_msgITCStates.u.tITCInfo.enSensorID = _poSensor->enGetSensorID();
		   if (TRUE == _poSensor->bGetNewITCState(m_msgITCStates.u.tITCInfo))
			   // send message to ITC controller
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgITCStates);
	   }
   }
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	Trace information
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vTraceInfo()
{
}

/*******************************************************************************
*
* FUNCTION: 	bNotify()
*
* DESCRIPTION: 	notification method for thermal sensor objects
*
* PARAMETER: 	hc_tclThermalSensor* poSensor		reference of thermal sensor
* 				tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tBool ... successful
*
*******************************************************************************/
tBool hc_tclThermalSensorControl::bNotify(hc_tclThermalSensor* poSensor, tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("bNotify(sensor: 0x%08x, sensor id: %d) entered.", poSensor, enSensorID ));
   if ((OSAL_NULL != poSensor) && (enThermalSensorID_MAX > enSensorID))
   {
	   m_poThermalSensors[enSensorID] = poSensor;
	   return TRUE;
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION: 	vDenotify()
*
* DESCRIPTION: 	denotification method for thermal sensor objects
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vDenotify(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("vDenotify(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID)
   {
	   m_poThermalSensors[enSensorID] = OSAL_NULL;
   }
}

/*******************************************************************************
*
* FUNCTION: 	enGetSensorState()
*
* DESCRIPTION: 	get thermal state of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tenThermalState
*
*******************************************************************************/
tenThermalState hc_tclThermalSensorControl::enGetSensorState(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("enGetSensorState(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->enGetSensorState();
   return enThermalState_UNAVAILABLE;
}

/*******************************************************************************
*
* FUNCTION: 	enGetSensorStateLastValid()
*
* DESCRIPTION: 	get last valid thermal state of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tenThermalState
*
*******************************************************************************/
tenThermalState hc_tclThermalSensorControl::enGetSensorStateLastValid(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("enGetSensorStateLastValid(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->enGetSensorStateLastValid();
   return enThermalState_INIT;
}

/*******************************************************************************
*
* FUNCTION: 	tGetTemperature()
*
* DESCRIPTION: 	get temperature of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tS16Temperature
*
*******************************************************************************/
tS16Temperature hc_tclThermalSensorControl::tGetTemperature(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("tGetTemperature(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->tGetTemperature();
   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	enGetTemperatureQualifier()
*
* DESCRIPTION: 	get temperature quality of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tenTemperatureQualifier
*
*******************************************************************************/
tenTemperatureQualifier hc_tclThermalSensorControl::enGetTemperatureQualifier(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("enGetTemperatureQualifier(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->enGetTemperatureQualifier();
   return enTemperatureQualifier_UNSUPPORTED;
}

/*******************************************************************************
*
* FUNCTION: 	tGetTemperatureMinStatistics()
*
* DESCRIPTION: 	get lowest measured temperature of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tS16Temperature
*
*******************************************************************************/
tS16Temperature hc_tclThermalSensorControl::tGetTemperatureMinStatistics(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("tGetTemperatureMinStatistics(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->tGetTemperatureMinStatistics();
   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	tGetTemperatureMaxStatistics()
*
* DESCRIPTION: 	get highest measured temperature of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tS16Temperature
*
*******************************************************************************/
tS16Temperature hc_tclThermalSensorControl::tGetTemperatureMaxStatistics(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("tGetTemperatureMaxStatistics(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->tGetTemperatureMaxStatistics();
   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	u8GetConfigAttenuationCycleTime()
*
* DESCRIPTION: 	get configured attenuation cycle time of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tU8
*
*******************************************************************************/
tU8 hc_tclThermalSensorControl::u8GetConfigAttenuationCycleTime(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("u8GetConfigAttenuationCycleTime(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->u8GetConfigAttenuationCycleTime();
   return HC_UNCONFIGURED_CYCLETIME;
}

/*******************************************************************************
*
* FUNCTION: 	u8GetFANSpeed()
*
* DESCRIPTION: 	get current fan speed
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tU8
*
*******************************************************************************/
tU8 hc_tclThermalSensorControl::u8GetFANSpeed(tenThermalSensorID enSensorID)
{
   ETG_TRACE_USR4(("u8GetFANSpeed(sensor id: %d) entered.", enSensorID ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
	   return m_poThermalSensors[enSensorID]->u8GetFANSpeed();
   return 0;
}

/*******************************************************************************
*
* FUNCTION: 	bIsEmergencyFunctionActive()
*
* DESCRIPTION: 	returns TRUE if the specified emergency function of specified
*               thermal sensor active
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*               tenEmergencyFunctionID enFunction   emergency function
*
* RETURNVALUE: 	tBool
*
*******************************************************************************/
tBool hc_tclThermalSensorControl::bIsEmergencyFunctionActive(tenThermalSensorID enSensorID, tenEmergencyFunctionID enFunction)
{
   ETG_TRACE_USR4(("bIsEmergencyFunctionActive(sensor id: %d, function id: %d) entered.", enSensorID, enFunction ));
   if (enThermalSensorID_MAX > enSensorID && OSAL_NULL != m_poThermalSensors[enSensorID])
   	   return m_poThermalSensors[enSensorID]->bIsEmergencyFunctionActive(enFunction);
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_NewThermalInfo()
*
* DESCRIPTION: 	handle new thermal information for specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleMessage_NewThermalInfo(hc_tclBaseIf::TMsg &Msg)
{
   ETG_TRACE_USR4(("vHandleMessage_NewThermalInfo(pMsg: 0x%08x) entered.", &Msg));
   hc_tclThermalSensor* _poThermalSensor = poGetThermalSensor(Msg.u.tThermalInfo.enSensorID);
   tBool _bTemperatureChanged = FALSE;
   tBool _bSensorStateChanged = FALSE;
   tBool _bMainThermalStateChanged = FALSE;
   tBool _bEmergencyFunctionStateChanged = FALSE;
   tBool _bITCStateChanged = FALSE;
   tBool _bFANSpeedChanged = FALSE;
   tBool _bAUTOSAR = FALSE;
   if (OSAL_NULL != _poThermalSensor)
   {
	   switch(Msg.eCmd)
	   {
	   case hc_tclBaseIf::eNewThermalInfo_STANDARD:
		   _poThermalSensor->vHandleMessage_ThermalInfoStandard(Msg, _bTemperatureChanged, _bSensorStateChanged);
		   break;
	   case hc_tclBaseIf::eNewThermalInfo_AUTOSAR:
		   _poThermalSensor->vHandleMessage_ThermalInfoAUTOSAR(Msg, _bTemperatureChanged, _bSensorStateChanged);
		   _bAUTOSAR = TRUE;
		   break;
	   default:
		   return;
	   }
	   // update MAIN thermal state
	   if (TRUE == _bSensorStateChanged)
		   vUpdate_MainThermalState(_bMainThermalStateChanged);
	   // inform all service handler ... own service, AUTOSAR::ThermalManagementProxy
	   m_msgSensorUpdate_SrvHdl.u.tThermalSensorInfo.enSensorID = Msg.u.tThermalInfo.enSensorID;
	   m_msgSensorUpdate_SrvHdl_AUTOSAR.u.tThermalSensorInfo.enSensorID = Msg.u.tThermalInfo.enSensorID;
	   if (TRUE == _bTemperatureChanged)
	   {
		   m_msgSensorUpdate_SrvHdl.u.tThermalSensorInfo.tTemperature = _poThermalSensor->tGetTemperature();
		   m_msgSensorUpdate_SrvHdl_AUTOSAR.u.tThermalSensorInfo.tTemperature = _poThermalSensor->tGetTemperature();
		   m_msgSensorUpdate_SrvHdl.u.tThermalSensorInfo.bTemperatureValid =
				   ((enTemperatureQualifier_ESTIMATED <= _poThermalSensor->enGetTemperatureQualifier())
					&& (enTemperatureQualifier_REAL >= _poThermalSensor->enGetTemperatureQualifier()));
		   m_msgSensorUpdate_SrvHdl_AUTOSAR.u.tThermalSensorInfo.bTemperatureValid = m_msgSensorUpdate_SrvHdl.u.tThermalSensorInfo.bTemperatureValid;
		   m_msgSensorUpdate_SrvHdl.eCmd = eUpdateSensorTemperature;
		   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgSensorUpdate_SrvHdl);
		   if( TRUE == _poThermalSensor->bGetNotifyAbstractSensor())
		   {
			   m_msgSensorUpdate_SrvHdl.eCmd = eUpdateAbstractSensorTemperature;
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgSensorUpdate_SrvHdl);
		   }
		   // inform AUTOSAR::ThermalManagementProxy
		   if (FALSE == _bAUTOSAR)
		   {
			   m_msgSensorUpdate_SrvHdl_AUTOSAR.eCmd = eUpdateSensorTemperature;
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgSensorUpdate_SrvHdl_AUTOSAR);
		   }
		   // update FAN speed
		   _poThermalSensor->vUpdate_FANSpeed(_bFANSpeedChanged);
	   }
	   if (TRUE == _bSensorStateChanged || TRUE == _bMainThermalStateChanged)
	   {
		   m_msgSensorUpdate_SrvHdl.u.tThermalSensorInfo.enSensorState = _poThermalSensor->enGetSensorState();
		   m_msgSensorUpdate_SrvHdl.u.tThermalSensorInfo.enSensorStateLastValid = _poThermalSensor->enGetSensorStateLastValid();
		   m_msgSensorUpdate_SrvHdl_AUTOSAR.u.tThermalSensorInfo.enSensorState = _poThermalSensor->enGetSensorState();
		   m_msgSensorUpdate_SrvHdl_AUTOSAR.u.tThermalSensorInfo.enSensorStateLastValid = _poThermalSensor->enGetSensorStateLastValid();
		   m_msgSensorUpdate_SrvHdl.eCmd = eUpdateSensorState;
		   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgSensorUpdate_SrvHdl);
		   if( TRUE == _poThermalSensor->bGetNotifyAbstractSensor())
		   {
			   m_msgSensorUpdate_SrvHdl.eCmd = eUpdateAbstractSensorState;
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgSensorUpdate_SrvHdl);
		   }
		   // inform AUTOSAR::ThermalManagementProxy
		   if (FALSE == _bAUTOSAR)
		   {
			   m_msgSensorUpdate_SrvHdl_AUTOSAR.eCmd = eUpdateSensorState;
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgSensorUpdate_SrvHdl_AUTOSAR);
		   }
		   // update all emergency function states
		   _poThermalSensor->vUpdate_EmergencyFunctionState(_bEmergencyFunctionStateChanged);
		   // update all ITC states
		   _poThermalSensor->vUpdate_ITCState(_bITCStateChanged);
	   }
	   // trace status info
	   if (TRUE == _bSensorStateChanged || TRUE == _bTemperatureChanged
		   || TRUE == _bEmergencyFunctionStateChanged || TRUE == _bITCStateChanged
		   || TRUE == _bFANSpeedChanged)
	   {
		   _poThermalSensor->vTraceInfo();
	   }
	   // trigger emergency function state update
	   if (TRUE == _bEmergencyFunctionStateChanged)
	   {
		   // collect emergency function state information of specified sensor
		   m_msgEmergencyFunctionStates.u.tEmergencyFunctionInfo.enSensorID = _poThermalSensor->enGetSensorID();
		   if (TRUE == _poThermalSensor->bGetNewEmergencyFunctionState(m_msgEmergencyFunctionStates.u.tEmergencyFunctionInfo))
			   // send message to emergency function controller
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgEmergencyFunctionStates);
	   }
	   // trigger ITC state update
	   if (TRUE == _bITCStateChanged)
	   {
		   // collect ITC state information of specified sensor
		   m_msgITCStates.u.tITCInfo.enSensorID = _poThermalSensor->enGetSensorID();
		   if (TRUE == _poThermalSensor->bGetNewITCState(m_msgITCStates.u.tITCInfo))
			   // send message to ITC controller
			   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgITCStates);
	   }
	   // trigger FAN speed update
	   if (TRUE == _bFANSpeedChanged)
	   {
		   // collect fan speed information of specified sensor
		   (tVoid)memset((tVoid*)&m_msgFANSpeed.u.tFANSpeedReq, OSAL_NULL, sizeof(m_msgFANSpeed.u.tFANSpeedReq));
		   m_msgFANSpeed.u.tFANSpeedReq.enSensorID = _poThermalSensor->enGetSensorID();
		   for (tU16 _u16Idx = 0; (tU16)enFANID_MAX > _u16Idx; ++_u16Idx)
		   {
			   m_msgFANSpeed.u.tFANSpeedReq.au8Speed[_u16Idx] = _poThermalSensor->u8GetFANSpeed((tenFANID)_u16Idx);
		   }
		   // send message to fan controller
		   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgFANSpeed);
	   }
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_LoadSettings()
*
* DESCRIPTION: 	handle trigger to load settings for thermal sensors
*
* PARAMETER: 	message
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleMessage_LoadSettings(hc_tclBaseIf::TMsg &Msg)
{
   ETG_TRACE_USR4(("vHandleMessage_LoadSettings(pMsg: 0x%08x) entered.", &Msg ));
   for (tU32 _u32Idx = (tU32)enThermalSensorID_V850; (tU32)enThermalSensorID_MAX > _u32Idx; ++_u32Idx)
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)_u32Idx);
	   if (OSAL_NULL != _poSensor)
	   {
		   _poSensor->vLoadConfig((tenThermalSensorID)_u32Idx);
	   }
   }
}

/*******************************************************************************
*
* FUNCTION: 	vUpdate_MainThermalState()
*
* DESCRIPTION: 	update MAIN thermal state
*
* PARAMETER: 	tBool &bMainThermalStateChanged
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vUpdate_MainThermalState(tBool &bMainThermalStateChanged)
{
   ETG_TRACE_USR4(("vUpdate_MainThermalState() entered."));
   tenThermalState _enOldState = enGetMainState();
   tenThermalState _enMinState = enThermalState_UNAVAILABLE;
   tenThermalState _enMaxState = enThermalState_UNAVAILABLE;
   for (tU8 _u8Index = 0; (tU8)enThermalSensorID_MAX > _u8Index; ++_u8Index)
   {
	   tenThermalSensorID _enSensorID = (tenThermalSensorID)_u8Index;
	   tenThermalState _enSensorState = enThermalState_UNAVAILABLE;
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor(_enSensorID);
	   if (OSAL_NULL != _poSensor)
	   {
		   _enSensorState = _poSensor->enGetSensorStateLastValid();
		   if (_enMaxState < _enSensorState)
			   _enMaxState = _enSensorState;
		   if ((enThermalState_CRITICALLOW <= _enSensorState)
               && ((enThermalState_CRITICALLOW > _enMinState)
                   || (_enMinState > _enSensorState)))
			   _enMinState = _enSensorState;
	   }
   }
   if (enThermalState_NORMAL < _enMaxState)
	   m_enMainThermalState = _enMaxState;
   else if (enThermalState_NORMAL > _enMinState)
	   m_enMainThermalState = _enMinState;
   else if (enThermalState_NORMAL == _enMaxState)
	   m_enMainThermalState = _enMaxState;
   else if (enThermalState_INIT != _enOldState)
	   m_enMainThermalState = enThermalState_TEMPUNAVAIL;
   else
	   m_enMainThermalState = enThermalState_INIT;
   bMainThermalStateChanged = (_enOldState != m_enMainThermalState);
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetThermSensorState()
*
* DESCRIPTION: 	handler for trace command HC_GetThermSensorState
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_GetThermSensorState(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_GetThermSensorState() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];

   ETG_TRACE_FATAL(("%26s: HC_GetThermSensorState %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1) ));

   tenThermalSensorID _enSensorID = (tenThermalSensorID)param1;
   hc_tclThermalSensor* _poSensor = poGetThermalSensor(_enSensorID);
   if (enThermalSensorID_MAX == _enSensorID || enThermalSensorID_UNDEF == _enSensorID)
   {
	   for (tU8 _u8SensorID = (tU8)enThermalSensorID_V850; (tU8)enThermalSensorID_MAX > _u8SensorID; ++_u8SensorID)
	   {
		   _enSensorID = (tenThermalSensorID)_u8SensorID;
		   _poSensor = poGetThermalSensor(_enSensorID);
		   if (OSAL_NULL != _poSensor)
			   _poSensor->vTraceInfo();
	   }
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
   else if (OSAL_NULL != _poSensor)
   {
	   _poSensor->vTraceInfo();
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Simu_IF_FLAG()
*
* DESCRIPTION: 	handler for trace command HC_SimuSensor_IF_FLAG
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Simu_IF_FLAG(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Simu_IF_FLAG() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];
   tU32 param3 = puchData[5];
   tU32 param4 = puchData[6];

   ETG_TRACE_FATAL(("%26s: HC_SimuSensor_IF_FLAG %u %u %u %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_IFFLAG, param2),
		   ETG_ENUM(HC_Bool, param3),
		   ETG_ENUM(HC_Update, param4) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   hc_tclBaseIf::TMsg _oMsg;
		   if (TRUE == (tBool)param4)
		   {
			   _poSensor->vGetLastIfMessage(_oMsg);
			   switch (param2)
			   {
			   case enThermalInfoFlags_AVAILABLE:
				   _oMsg.u.tThermalInfo.bIfAvailable = (TRUE == (tBool)param3);
				   break;
			   case enThermalInfoFlags_TEMP_SUPPORTED:
				   _oMsg.u.tThermalInfo.bTemperatureSupported = (TRUE == (tBool)param3);
				   break;
			   case enThermalInfoFlags_TEMP_RECEIVED:
				   _oMsg.u.tThermalInfo.bTemperatureReceived = (TRUE == (tBool)param3);
				   break;
			   case enThermalInfoFlags_TEMP_ESTIMATED:
				   _oMsg.u.tThermalInfo.bTemperatureEstimated = (TRUE == (tBool)param3);
				   break;
			   case enThermalInfoFlags_STATE_SUPPORTED:
				   _oMsg.u.tThermalInfo.bSensorStateSupported = (TRUE == (tBool)param3);
				   break;
			   case enThermalInfoFlags_STATE_RECEIVED:
				   _oMsg.u.tThermalInfo.bSensorStateReceived = (TRUE == (tBool)param3);
				   break;
			   default:
				   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid interface flag %d!",
						   "HC.APPL.TSEN.CMD__.RES__", param2 ));
				   return;
			   }
			   vHandleMessage_NewThermalInfo(_oMsg);
		   }
		   else
		   {
			   switch (param2)
			   {
			   case enThermalInfoFlags_AVAILABLE:
				   _poSensor->vSet_IF_AVAILABLE( (TRUE == (tBool)param3) );
				   break;
			   case enThermalInfoFlags_TEMP_SUPPORTED:
				   _poSensor->vSet_IF_TEMP_SUPPORTED( (TRUE == (tBool)param3) );
				   break;
			   case enThermalInfoFlags_TEMP_RECEIVED:
				   _poSensor->vSet_IF_TEMP_RECEIVED( (TRUE == (tBool)param3) );
				   break;
			   case enThermalInfoFlags_TEMP_ESTIMATED:
				   _poSensor->vSet_IF_TEMP_ESTIMATED( (TRUE == (tBool)param3) );
				   break;
			   case enThermalInfoFlags_STATE_SUPPORTED:
				   _poSensor->vSet_IF_STATE_SUPPORTED( (TRUE == (tBool)param3) );
				   break;
			   case enThermalInfoFlags_STATE_RECEIVED:
				   _poSensor->vSet_IF_STATE_RECEIVED( (TRUE == (tBool)param3) );
				   break;
			   default:
				   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid interface flag %d!",
						   "HC.APPL.TSEN.CMD__.RES__", param2 ));
				   return;
			   }
		   }
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.TSEN.CMD__.RES__" ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Simu_IF_TEMP()
*
* DESCRIPTION: 	handler for trace command HC_SimuThermSensor_IF_TEMP
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Simu_IF_TEMP(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Simu_IF_TEMP() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tS16 param2 = ((tS16)puchData[4] << 8) | (tS16)puchData[5];
   tU32 param3 = puchData[6];

   ETG_TRACE_FATAL(("%26s: HC_SimuThermSensor_IF_TEMP %u %d %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   param2,
		   ETG_ENUM(HC_Update, param3) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   hc_tclBaseIf::TMsg _oMsg;
		   if (TRUE == (tBool)param3)
		   {
			   _poSensor->vGetLastIfMessage(_oMsg);
			   _oMsg.u.tThermalInfo.tTemperature = (tS16Temperature)param2;
			   vHandleMessage_NewThermalInfo(_oMsg);
		   }
		   else
		   {
			   _poSensor->vSet_IF_TEMP( (tS16Temperature)param2 );
		   }
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.TSEN.CMD__.RES__" ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Simu_IF_STATE()
*
* DESCRIPTION: 	handler for trace command HC_SimuThermSensor_IF_STATE
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Simu_IF_STATE(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Simu_IF_STATE() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];
   tU32 param3 = puchData[5];

   ETG_TRACE_FATAL(("%26s: HC_SimuThermSensor_IF_STATE %u %u %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_SensorState, param2),
		   ETG_ENUM(HC_Update, param3) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   hc_tclBaseIf::TMsg _oMsg;
		   if (TRUE == (tBool)param3)
		   {
			   _poSensor->vGetLastIfMessage(_oMsg);
			   _oMsg.u.tThermalInfo.uSensorState.enHC = (tenThermalState)param2;
			   vHandleMessage_NewThermalInfo(_oMsg);
		   }
		   else
		   {
			   _poSensor->vSet_IF_STATE( (tenThermalState)param2 );
		   }
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.TSEN.CMD__.RES__" ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Simu_IF_STATE()
*
* DESCRIPTION: 	handler for trace command HC_SetThermSensor_STATE
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Set_STATE(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Set_STATE() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];

   ETG_TRACE_FATAL(("%26s: HC_SetThermSensor_STATE %u %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_SensorState, param2) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   _poSensor->vSetSensorState((tenThermalState)param2);
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.TSEN.CMD__.RES__" ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Set_LASTVALIDSTATE()
*
* DESCRIPTION: 	handler for trace command HC_SetThermSensor_LASTVALIDSTATE
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Set_LASTVALIDSTATE(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Set_LASTVALIDSTATE() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];

   ETG_TRACE_FATAL(("%26s: HC_SetThermSensor_LASTVALIDSTATE %u %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_SensorState, param2) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   _poSensor->vSetSensorStateLastValid((tenThermalState)param2);
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.TSEN.CMD__.RES__" ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_SensorEnable()
*
* DESCRIPTION: 	handler for trace command HC_ConfigThermSensor_Enable
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_SensorEnable(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_SensorEnable() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];
   tU32 param3 = puchData[5];

   ETG_TRACE_FATAL(("%26s: HC_ConfigThermSensor_Enable %u %u %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_Persist, param2),
		   ETG_ENUM(HC_Bool, param3) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   tBool _bResult = TRUE;
		   if (TRUE == (tBool)param2)
			   _bResult = _poSensor->bSetConfigEnablePers((tBool)param3);
		   else
			   _bResult = _poSensor->bSetConfigEnable((tBool)param3);
		   if (TRUE == _bResult)
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigReset_SensorEnable()
*
* DESCRIPTION: 	handler for trace command HC_ConfigThermSensor_Enable_Reset
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_ConfigReset_SensorEnable(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigReset_SensorEnable() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];

   ETG_TRACE_FATAL(("%26s: HC_ConfigThermSensor_Enable_Reset %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   if (TRUE == _poSensor->bResetConfigEnablePers())
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_Threshold()
*
* DESCRIPTION: 	handler for trace command HC_ConfigThermSensor_Threshold
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_Threshold(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_Threshold() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];
   tU32 param3 = puchData[5];
   tS16 param4 = ((tS16)puchData[6] << 8) | (tS16)puchData[7];

   ETG_TRACE_FATAL(("%26s: HC_ConfigThermSensor_Threshold %u %u %u %d",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_Persist, param2),
		   ETG_ENUM(HC_ThresholdID, param3),
		   param4 ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   tBool _bResult = TRUE;
		   if (TRUE == (tBool)param2)
			   _bResult = _poSensor->bSetConfigTempPers((hc_tclThermalSensor_Config::tenTempConfigItem)param3, param4);
		   else
			   _bResult = _poSensor->bSetConfigTemp((hc_tclThermalSensor_Config::tenTempConfigItem)param3, param4);
		   if (TRUE == _bResult)
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_Threshold_Reset()
*
* DESCRIPTION: 	handler for trace command HC_ConfigThermSensor_Threshold_Reset
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_Threshold_Reset(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_Threshold_Reset() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];

   ETG_TRACE_FATAL(("%26s: HC_ConfigThermSensor_Threshold_Reset %u %u ",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_ThresholdID, param2) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   if (TRUE == _poSensor->bResetConfigTempPers((hc_tclThermalSensor_Config::tenTempConfigItem)param2))
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_Attenuation_Cycletime()
*
* DESCRIPTION: 	handler for trace command HC_ConfigThermSensor_AttenuationCycleTime
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_Attenuation_Cycletime(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_Attenuation_Cycletime() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];
   tU32 param2 = puchData[4];
   tU32 param3 = puchData[5];

   ETG_TRACE_FATAL(("%26s: HC_ConfigThermSensor_AttenuationCycleTime %u %u %d ",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_Persist, param2),
		   param3 ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   tBool _bResult = FALSE;
		   if (TRUE == (tBool)param2)
			   _bResult = _poSensor->bSetConfigAttenuationCycleTimePers((tU8)param3);
		   else
			   _bResult = _poSensor->bSetConfigAttenuationCycleTime((tU8)param3);
		   if (TRUE == _bResult)
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_Attenuation_Cycletime_Reset()
*
* DESCRIPTION: 	handler for trace command HC_ConfigThermSensor_AttenuationCycleTime_Reset
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_Attenuation_Cycletime_Reset(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_Attenuation_Cycletime_Reset() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];

   ETG_TRACE_FATAL(("%26s: HC_ConfigThermSensor_AttenuationCycleTime_Reset %u ",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   if (TRUE == _poSensor->bResetConfigAttenuationCycleTimePers())
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_FanThreshold()
*
* DESCRIPTION: 	handler for trace command HC_ConfigSensorFan_Threshold
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_FanThreshold(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_FanThreshold() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3]; 	// sensor id
   tU32 param2 = puchData[4];	// fan id
   tU32 param3 = puchData[5];	// persistant
   tU32 param4 = puchData[6];	// speed level id
   tS16 param5 = ((tS16)puchData[7] << 8) | (tS16)puchData[8];

   ETG_TRACE_FATAL(("%26s: HC_ConfigSensorFan_Threshold %u %u %u %u %d",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_FANID, param2),
		   ETG_ENUM(HC_Persist, param3),
		   ETG_ENUM(HC_FANTHRESHOLDID, param4),
		   param5 ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   tBool _bResult = TRUE;
		   if (TRUE == (tBool)param3)
			   _bResult = _poSensor->bSetConfigFanTempPers((tenFANID)param2, (tenFANSpeedLevel)param4, param5);
		   else
			   _bResult = _poSensor->bSetConfigFanTemp((tenFANID)param2, (tenFANSpeedLevel)param4, param5);
		   if (TRUE == _bResult)
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_Config_FanThreshold_Reset()
*
* DESCRIPTION: 	handler for trace command HC_ConfigSensorFan_Threshold_Reset
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_Config_FanThreshold_Reset(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_Config_FanThreshold_Reset() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 param1 = puchData[3];	// sensor id
   tU32 param2 = puchData[4];	// fan id
   tU32 param3 = puchData[5];	// fan threshold id

   ETG_TRACE_FATAL(("%26s: HC_ConfigSensorFan_Threshold_Reset %u %u %u ",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_SensorID, param1),
		   ETG_ENUM(HC_FANID, param2),
		   ETG_ENUM(HC_FANTHRESHOLDID, param3) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   hc_tclThermalSensor* _poSensor = poGetThermalSensor((tenThermalSensorID)param1);
	   if (OSAL_NULL != _poSensor)
	   {
		   if (TRUE == _poSensor->bResetConfigFanTempPers( (tenFANID)param2, (tenFANSpeedLevel)param3))
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.TSEN.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid sensor id %d!",
				   "HC.APPL.TSEN.CMD__.RES__", param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetAbstractSensorState()
*
* DESCRIPTION: 	handler for trace command HC_GetThermSensorState
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclThermalSensorControl::vHandleTraceCmd_GetAbstractSensorState(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_GetThermSensorState() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];

   if (enAbstractSensorID_MAIN == (tenAbstractSensorID)param1)
   {
	   ETG_TRACE_FATAL(("%26s: HC_GetAbstractSensorState %u",
			   "HC.APPL.TSEN.CMD__.START",
			   ETG_ENUM(HC_AbstractSensorID, param1) ));

	   ETG_TRACE_USR2(("%26s: ",
			   "HC.APPL.ASEN.MAIN_.START" ));
	   ETG_TRACE_USR2(("%26s: %30s=%25s ",
			   "HC.APPL.ASEN.MAIN_.ATTR_",
			   "               STATE",
			   strGetThermalState(enGetMainState()) ));
	   ETG_TRACE_USR2(("%26s: ",
			   "HC.APPL.ASEN.MAIN_.END__" ));

	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}
