/************************************************************************
* FILE:        hc_tclAbstractSensorControl.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: classes to control all abstract sensors
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 05.03.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclAbstractSensorControl.h"
#include "hc_tclThermalSensor.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclAbstractSensorControl.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/
hc_tclThermalSensor* hc_tclAbstractSensorControl::m_apoThermalSensor[enAbstractSensorID_MAX];

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*************************************************************************
*
* FUNCTION:    hc_tclAbstractSensorControl()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain* poMainAppl: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclAbstractSensorControl::hc_tclAbstractSensorControl(const hc_tclAppMain* poMainAppl)
: I_hc_tclAbstractSensorControl(poMainAppl)
{
   ETG_TRACE_USR4(("hc_tclAbstractSensorControl() entered."));
}


/*************************************************************************
*
* FUNCTION:    ~hc_tclAbstractSensorControl()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclAbstractSensorControl::~hc_tclAbstractSensorControl()
{
   ETG_TRACE_USR4(("~hc_tclAbstractSensorControl() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage()
*
* DESCRIPTION: 	handler for internal messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received messagde
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
   HC_NULL_POINTER_CHECK(pMsg);
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vHandleTraceMessage(const tUChar* puchData)
{
	ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
	HC_NULL_POINTER_CHECK(puchData);

	tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
	switch (u32MsgCode) {
	case FC_HEATCTRL_HC_GETABSTRACTSENSORSTATE:
	   {
		   vHandleTraceCmd_GetAbstractSensorState(puchData);
	   }
	   break;

	default:
	   ETG_TRACE_USR4(("hc_tclAbstractSensorControl::vHandleTraceMessage(): unhandled command %d.", u32MsgCode));
	   break;
	}
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	Trace information
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vTraceInfo()
{
}

/*******************************************************************************
*
* FUNCTION: 	bNotify()
*
* DESCRIPTION: 	notification method for thermal sensor objects
*
* PARAMETER: 	hc_tclThermalSensor* poSensor		reference of thermal sensor
* 				tenAbstractSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tBool ... successful
*
*******************************************************************************/
tBool hc_tclAbstractSensorControl::bNotify(hc_tclThermalSensor* poSensor, tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("bTSNotify(sensor: 0x%08x, sensor id: %d) entered.", poSensor, enSensorID ));
   if ((OSAL_NULL != poSensor) && (enAbstractSensorID_MAX > enSensorID) && (OSAL_NULL == m_apoThermalSensor[enSensorID]))
      m_apoThermalSensor[enSensorID] = poSensor;
   return ((enAbstractSensorID_MAX > enSensorID) && (poSensor == m_apoThermalSensor[enSensorID]));
}

/*******************************************************************************
*
* FUNCTION: 	vDenotify()
*
* DESCRIPTION: 	denotification method for thermal sensor objects
*
* PARAMETER: 	tenAbstractSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	none
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vDenotify(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("vTSDenotify(sensor id: %d) entered.", enSensorID ));
   if (enAbstractSensorID_MAX > enSensorID)
	   m_apoThermalSensor[enSensorID] = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	enGetSensorState()
*
* DESCRIPTION: 	get thermal state of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tenThermalState
*
*******************************************************************************/
tenThermalState hc_tclAbstractSensorControl::enGetSensorState(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("enGetSensorState(sensor id: %d) entered.", enSensorID ));
   if (OSAL_NULL != poGetThermalSensor(enSensorID))
	   return poGetThermalSensor(enSensorID)->enGetSensorState();
   return enThermalState_UNAVAILABLE;
}

/*******************************************************************************
*
* FUNCTION: 	enGetSensorStateLastValid()
*
* DESCRIPTION: 	get last valid thermal state of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tenThermalState
*
*******************************************************************************/
tenThermalState hc_tclAbstractSensorControl::enGetSensorStateLastValid(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("enGetSensorStateLastValid(sensor id: %d) entered.", enSensorID ));
   if (OSAL_NULL != poGetThermalSensor(enSensorID))
	   return poGetThermalSensor(enSensorID)->enGetSensorStateLastValid();
   return enThermalState_INIT;
}

/*******************************************************************************
*
* FUNCTION: 	tGetTemperature()
*
* DESCRIPTION: 	get temperature of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tS16Temperature
*
*******************************************************************************/
tS16Temperature hc_tclAbstractSensorControl::tGetTemperature(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("tGetTemperature(sensor id: %d) entered.", enSensorID ));
   if (OSAL_NULL != poGetThermalSensor(enSensorID))
	   return poGetThermalSensor(enSensorID)->tGetTemperature();
   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	enGetTemperatureQualifier()
*
* DESCRIPTION: 	get temperature quality of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tenTemperatureQualifier
*
*******************************************************************************/
tenTemperatureQualifier hc_tclAbstractSensorControl::enGetTemperatureQualifier(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("enGetTemperatureQualifier(sensor id: %d) entered.", enSensorID ));
   if (OSAL_NULL != poGetThermalSensor(enSensorID))
	   return poGetThermalSensor(enSensorID)->enGetTemperatureQualifier();
   return enTemperatureQualifier_UNSUPPORTED;
}

/*******************************************************************************
*
* FUNCTION: 	tGetTemperatureMinStatistics()
*
* DESCRIPTION: 	get lowest measured temperature of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tS16Temperature
*
*******************************************************************************/
tS16Temperature hc_tclAbstractSensorControl::tGetTemperatureMinStatistics(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("tGetTemperatureMinStatistics(sensor id: %d) entered.", enSensorID ));
   if (OSAL_NULL != poGetThermalSensor(enSensorID))
	   return poGetThermalSensor(enSensorID)->tGetTemperatureMinStatistics();
   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	tGetTemperatureMaxStatistics()
*
* DESCRIPTION: 	get highest measured temperature of specified thermal sensor
*
* PARAMETER: 	tenThermalSensorID enSensorID		thermal sensor ID
*
* RETURNVALUE: 	tS16Temperature
*
*******************************************************************************/
tS16Temperature hc_tclAbstractSensorControl::tGetTemperatureMaxStatistics(tenAbstractSensorID enSensorID)
{
   ETG_TRACE_USR4(("tGetTemperatureMaxStatistics(sensor id: %d) entered.", enSensorID ));
   if (OSAL_NULL != poGetThermalSensor(enSensorID))
	   return poGetThermalSensor(enSensorID)->tGetTemperatureMaxStatistics();
   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetAbstractSensorState()
*
* DESCRIPTION: 	handler for trace command HC_GetAbstractSensorState
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclAbstractSensorControl::vHandleTraceCmd_GetAbstractSensorState(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_GetAbstractSensorState() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];

   if (enAbstractSensorID_MAIN == (tenAbstractSensorID)param1)
	   return;

   ETG_TRACE_FATAL(("%26s: HC_GetAbstractSensorState %u",
		   "HC.APPL.TSEN.CMD__.START",
		   ETG_ENUM(HC_AbstractSensorID, param1) ));

   tenAbstractSensorID _enSensorID = (tenAbstractSensorID)param1;
   hc_tclThermalSensor* _poSensor = poGetThermalSensor(_enSensorID);
   if (OSAL_NULL != _poSensor)
   {
	   _poSensor->vTraceInfo();
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors",
			   "HC.APPL.TSEN.CMD__.RES__" ));
   }
}
