/*******************************************************************************
* FILE:        hc_tclFAN_Statistics.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*-------------------------------------------------------------------------------
*
* DESCRIPTION: class to manage statistical information of fan
*
*-------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2017 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 24.01.2017| CM-AI/EPB2 Bernard | init
*
*******************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

// datapool access
#ifndef DP_S_IMPORT_INTERFACE_FI
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_fc_heatctrl_if.h"
#endif //#ifndef DP_S_IMPORT_INTERFACE_FI

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include "hc_AppMain.h"
#include "hc_tclFAN_Statistics.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclFAN_Statistics.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Statistics()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Statistics::hc_tclFAN_Statistics()
{
   ETG_TRACE_USR4(("hc_tclFAN_Statistics() entered."));

   // init last information
   (tVoid)memset((tVoid*)&m_tLastInfo, OSAL_NULL, sizeof(m_tLastInfo));
   // init runtime statistic info
   (tVoid)memset((tVoid*)&m_atStatisticInfo, OSAL_NULL, sizeof(m_atStatisticInfo));
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclFAN_Statistics()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Statistics::~hc_tclFAN_Statistics()
{
   ETG_TRACE_USR4(("~hc_tclFAN_Statistics() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_FAN_Statistics()
*
* DESCRIPTION: trace attributes of FAN Statistics
*
* PARAMETER:   const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Statistics::vTraceInfo_FAN_Statistics(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceInfo_FAN() entered."));

	ETG_TRACE_USR2(("%26s: <<< FAN Statistics >>> ", strIdentifier ));
	// common fan state information
	ETG_TRACE_USR2(("%26s: %30s=%d ms (%u, live time: %d ms, run time: %d ms)",       strIdentifier,
			"STAT_ACTIVE_DURATION", u32GetActivationDuration(enFANCtrlMode_AUTO, enOperationtime),
			ETG_ENUM(HC_FANMODE, enFANCtrlMode_AUTO),
			u32GetActivationDuration(enFANCtrlMode_AUTO, enLivetime),
			u32GetActivationDuration(enFANCtrlMode_AUTO, enRuntime) ));
	ETG_TRACE_USR2(("%26s: %30s=%d (%u, live time: %d, run time: %d)",       strIdentifier,
			" STAT_ACTIVE_COUNTER", u32GetActivationCounter(enFANCtrlMode_AUTO, enOperationtime),
			ETG_ENUM(HC_FANMODE, enFANCtrlMode_AUTO),
			u32GetActivationCounter(enFANCtrlMode_AUTO, enLivetime),
			u32GetActivationCounter(enFANCtrlMode_AUTO, enRuntime) ));
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_FAN_StatisticsComplete()
*
* DESCRIPTION: trace all attributes of FAN Statistics
*
* PARAMETER:   const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Statistics::vTraceInfo_FAN_StatisticsComplete(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceInfo_FAN_StatisticsComplete() entered."));

   if (OSAL_NULL == strIdentifier)
	   return;
   // trace fan actiation statistics depending of fan control mode
   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo) > _u16Idx; ++_u16Idx)
   {
	   tenFANCtrlMode _enMode = (tenFANCtrlMode)_u16Idx;
	   ETG_TRACE_USR2(("%26s: STAT_ACT_DURATION_%02d=%d ms (%u, live time: %d ms, run time: %d ms)",       strIdentifier,
			   _enMode, u32GetActivationDuration(_enMode, enOperationtime),
			   ETG_ENUM(HC_FANMODE, _enMode),
			   u32GetActivationDuration(_enMode, enLivetime),
			   u32GetActivationDuration(_enMode, enRuntime) ));
	   ETG_TRACE_USR2(("%26s:  STAT_ACT_COUNTER_%02d=%d (%u, live time: %d, run time: %d)",       strIdentifier,
			   _enMode, u32GetActivationCounter(_enMode, enOperationtime),
			   ETG_ENUM(HC_FANMODE, _enMode),
			   u32GetActivationCounter(_enMode, enLivetime),
			   u32GetActivationCounter(_enMode, enRuntime) ));
   }
   // trace fan control mode statistics
   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo) > _u16Idx; ++_u16Idx)
   {
	   tenFANCtrlMode _enMode = (tenFANCtrlMode)_u16Idx;
	   ETG_TRACE_USR2(("%26s: MODE_ACT_DURATION_%02d=%d ms (%u, live time: %d ms, run time: %d ms)",       strIdentifier,
			   _enMode, u32GetModeDuration(_enMode, enOperationtime),
			   ETG_ENUM(HC_FANMODE, _enMode),
			   u32GetModeDuration(_enMode, enLivetime),
			   u32GetModeDuration(_enMode, enRuntime) ));
	   ETG_TRACE_USR2(("%26s:  MODE_ACT_COUNTER_%02d=%d (%u, live time: %d, run time: %d)",       strIdentifier,
			   _enMode, u32GetModeActivationCounter(_enMode, enOperationtime),
			   ETG_ENUM(HC_FANMODE, _enMode),
			   u32GetModeActivationCounter(_enMode, enLivetime),
			   u32GetModeActivationCounter(_enMode, enRuntime) ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	u32GetActivationDuration()
*
* DESCRIPTION: 	get the activation duration [ms] of specified fan mode and
*               storage type
*
* PARAMETER: 	tenFANCtrlMode enMode
*               tenFANStatisticStorageType enStorageType
*
* RETURNVALUE: 	tU32		[ms]
*
*******************************************************************************/
tU32 hc_tclFAN_Statistics::u32GetActivationDuration(tenFANCtrlMode enMode, tenFANStatisticStorageType enStorageType) const
{
	ETG_TRACE_USR4(("u32GetActivationDuration(mode: %u) entered.", ETG_ENUM(HC_FANMODE, enMode)));

	tU32 _u32Return = 0;
	tU32 _au32ActivationDuration[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		if (enOperationtime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ActivationDurationArray_OperationTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32ActivationDuration, HC_C_U16_ARRAYELEMENTS(_au32ActivationDuration));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32ActivationDuration[enMode];
			}
		}
		else if (enLivetime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ActivationDurationArray_LiveTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32ActivationDuration, HC_C_U16_ARRAYELEMENTS(_au32ActivationDuration));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32ActivationDuration[enMode];
			}
		}
		else
		{
			// runtime
			_u32Return = m_atStatisticInfo[enMode].u32ActivationDuration;
		}
	}
	ETG_TRACE_USR3(("u32GetActivationDuration: activation duration %d ms for mode %u and storage type %u read with result %d.",
			_u32Return, ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType), _s32Result));

	return _u32Return;
}

/*******************************************************************************
*
* FUNCTION: 	s32IncreaseActivationDuration()
*
* DESCRIPTION: 	increase fan activation duration depending of specified fan
*               control mode for all storage types
*
* PARAMETER: 	tenFANCtrlMode enMode
*               u32TmeDiff
*
* RETURNVALUE: 	tS32         { >=0 - successful | else - failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Statistics::s32IncreaseActivationDuration(tenFANCtrlMode enMode, tU32 u32TmeDiff)
{
	ETG_TRACE_USR4(("s32IncreaseActivationDuration(mode: %u, timediff: %d ms) entered.", ETG_ENUM(HC_FANMODE, enMode), u32TmeDiff));

	tS32 _s32Return = 0;
	tU32 _au32ActivationDuration[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		// increase runtime
		m_atStatisticInfo[enMode].u32ActivationDuration += u32TmeDiff;
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
		// increase operation time
		dp_tclHeatCtrlDPSYSFAN_ActivationDurationArray_OperationTime _oDPElementOpTime;
		_s32Result = _oDPElementOpTime.s32GetData(_au32ActivationDuration, HC_C_U16_ARRAYELEMENTS(_au32ActivationDuration));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32ActivationDuration[enMode] += u32TmeDiff;
			_s32Result = _oDPElementOpTime.s32SetData(_au32ActivationDuration, HC_C_U16_ARRAYELEMENTS(_au32ActivationDuration));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseActivationDuration: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
				_s32Return -= 4;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseActivationDuration: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
			_s32Return -= 2;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
		// increase live time
		dp_tclHeatCtrlDPSYSFAN_ActivationDurationArray_LiveTime _oDPElementLiveTime;
		_s32Result = _oDPElementLiveTime.s32GetData(_au32ActivationDuration, HC_C_U16_ARRAYELEMENTS(_au32ActivationDuration));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32ActivationDuration[enMode] += u32TmeDiff;
			_s32Result = _oDPElementLiveTime.s32SetData(_au32ActivationDuration, HC_C_U16_ARRAYELEMENTS(_au32ActivationDuration));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseActivationDuration: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
				_s32Return -= 16;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseActivationDuration: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
			_s32Return -= 8;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
	}
	else
	{
		ETG_TRACE_ERR(("s32IncreaseActivationDuration: unspecified mode %u!", ETG_ENUM(HC_FANMODE, enMode)));
		_s32Return -= 1;
	}

	return _s32Return;
}

/*******************************************************************************
*
* FUNCTION: 	u32GetActivationCounter()
*
* DESCRIPTION: 	get the activation counter of specified fan mode and
*               storage type
*
* PARAMETER: 	tenFANCtrlMode enMode
*               tenFANStatisticStorageType enStorageType
*
* RETURNVALUE: 	tU32
*
*******************************************************************************/
tU32 hc_tclFAN_Statistics::u32GetActivationCounter(tenFANCtrlMode enMode, tenFANStatisticStorageType enStorageType) const
{
	ETG_TRACE_USR4(("u32GetActivationCounter(mode: %u, storage type: %u) entered.", ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType)));

	tU32 _u32Return = 0;
	tU32 _au32ActivationCounter[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		if (enOperationtime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ActivationCounterArray_OperationTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32ActivationCounter[enMode];
			}
		}
		else if (enLivetime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ActivationCounterArray_LiveTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32ActivationCounter[enMode];
			}
		}
		else
		{
			// runtime
			_u32Return = m_atStatisticInfo[enMode].u32ActivationCounter;
		}
	}
	ETG_TRACE_USR3(("u32GetActivationCounter: activation counter %d for mode %u and storage type %u read with result %d.",
			_u32Return, ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType), _s32Result));

	return _u32Return;
}

/*******************************************************************************
*
* FUNCTION: 	s32IncreaseActivationCounter()
*
* DESCRIPTION: 	increase fan activation counter depending of specified fan
*               control mode
*
* PARAMETER: 	tenFANCtrlMode enMode
*               tU32 u32Count
*
* RETURNVALUE: 	tS32         { >=0 - successful | else - failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Statistics::s32IncreaseActivationCounter(tenFANCtrlMode enMode, tU32 u32Count)
{
	ETG_TRACE_USR4(("s32IncreaseActivationCounter(mode: %u, count: %d) entered.", ETG_ENUM(HC_FANMODE, enMode), u32Count));

	tS32 _s32Return = 0;
	tU32 _au32ActivationCounter[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		// increase runtime
		m_atStatisticInfo[enMode].u32ActivationCounter += u32Count;
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
		// increase operation time
		dp_tclHeatCtrlDPSYSFAN_ActivationCounterArray_OperationTime _oDPElementOpTime;
		_s32Result = _oDPElementOpTime.s32GetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32ActivationCounter[enMode] += u32Count;
			_s32Result = _oDPElementOpTime.s32SetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseActivationCounter: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
				_s32Return -= 4;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseActivationCounter: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
			_s32Return -= 2;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
		// increase live time
		dp_tclHeatCtrlDPSYSFAN_ActivationCounterArray_LiveTime _oDPElementLiveTime;
		_s32Result = _oDPElementLiveTime.s32GetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32ActivationCounter[enMode] += u32Count;
			_s32Result = _oDPElementLiveTime.s32SetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseActivationCounter: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
				_s32Return -= 16;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseActivationCounter: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
			_s32Return -= 8;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
	}
	else
	{
		ETG_TRACE_ERR(("s32IncreaseActivationCounter: unspecified mode %u!", ETG_ENUM(HC_FANMODE, enMode)));
		_s32Return -= 1;
	}

	return _s32Return;
}

/*******************************************************************************
*
* FUNCTION: 	u32GetModeDuration()
*
* DESCRIPTION: 	get fan mode duration depending of specified fan control mode
*
* PARAMETER: 	tenFANCtrlMode enMode
*               tenFANStatisticStorageType enStorageType
*
* RETURNVALUE: 	tU32		[ms]
*
*******************************************************************************/
tU32 hc_tclFAN_Statistics::u32GetModeDuration(tenFANCtrlMode enMode, tenFANStatisticStorageType enStorageType) const
{
	ETG_TRACE_USR4(("u32GetModeDuration(mode: %u, storage type: %u) entered.",
			ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType)));

	tU32 _u32Return = 0;
	tU32 _au32Duration[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		if (enOperationtime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ControlModeDurationArray_OperationTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32Duration, HC_C_U16_ARRAYELEMENTS(_au32Duration));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32Duration[enMode];
			}
		}
		else if (enLivetime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ControlModeDurationArray_LiveTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32Duration, HC_C_U16_ARRAYELEMENTS(_au32Duration));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32Duration[enMode];
			}
		}
		else
		{
			// runtime
			_u32Return = m_atStatisticInfo[enMode].u32ModeDuration;
		}
	}
	ETG_TRACE_USR3(("u32GetModeDuration: fan mode duration %d ms for mode %u and storage type %u read with result %d.",
			_u32Return, ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType), _s32Result));

	return _u32Return;
}

/*******************************************************************************
*
* FUNCTION: 	s32IncreaseModeDuration()
*
* DESCRIPTION: 	increase fan mode duration depending of specified fan control mode
*
* PARAMETER: 	tenFANCtrlMode enMode
*               u32TmeDiff
*
* RETURNVALUE: 	tS32         { >=0 - successful | else - failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Statistics::s32IncreaseModeDuration(tenFANCtrlMode enMode, tU32 u32TmeDiff)
{
	ETG_TRACE_USR4(("s32IncreaseModeDuration(mode: %u, timediff: %d ms) entered.", ETG_ENUM(HC_FANMODE, enMode), u32TmeDiff));

	tS32 _s32Return = 0;
	tU32 _au32Duration[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		// increase runtime
		m_atStatisticInfo[enMode].u32ModeDuration += u32TmeDiff;
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
		// increase operation time
		dp_tclHeatCtrlDPSYSFAN_ControlModeDurationArray_OperationTime _oDPElementOpTime;
		_s32Result = _oDPElementOpTime.s32GetData(_au32Duration, HC_C_U16_ARRAYELEMENTS(_au32Duration));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32Duration[enMode] += u32TmeDiff;
			_s32Result = _oDPElementOpTime.s32SetData(_au32Duration, HC_C_U16_ARRAYELEMENTS(_au32Duration));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseModeDuration: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
				_s32Return -= 4;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseModeDuration: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
			_s32Return -= 2;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
		// increase live time
		dp_tclHeatCtrlDPSYSFAN_ControlModeDurationArray_LiveTime _oDPElementLiveTime;
		_s32Result = _oDPElementLiveTime.s32GetData(_au32Duration, HC_C_U16_ARRAYELEMENTS(_au32Duration));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32Duration[enMode] += u32TmeDiff;
			_s32Result = _oDPElementLiveTime.s32SetData(_au32Duration, HC_C_U16_ARRAYELEMENTS(_au32Duration));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseModeDuration: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
				_s32Return -= 16;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseModeDuration: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
			_s32Return -= 8;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
	}
	else
	{
		ETG_TRACE_ERR(("s32IncreaseModeDuration: unspecified mode %u!", ETG_ENUM(HC_FANMODE, enMode)));
		_s32Return -= 1;
	}

	return _s32Return;
}

/*******************************************************************************
*
* FUNCTION: 	u32GetModeActivationCounter()
*
* DESCRIPTION: 	fan mode activation counter depending of specified fan control mode
*
* PARAMETER: 	tenFANCtrlMode enMode
*               tenFANStatisticStorageType enStorageType
*
* RETURNVALUE: 	tU32
*
*******************************************************************************/
tU32 hc_tclFAN_Statistics::u32GetModeActivationCounter(tenFANCtrlMode enMode, tenFANStatisticStorageType enStorageType) const
{
	ETG_TRACE_USR4(("u32GetModeActivationCounter(mode: %u, storage type: %u) entered.",
			ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType)));

	tU32 _u32Return = 0;
	tU32 _au32Counter[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		if (enOperationtime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ControlModeCounterArray_OperationTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32Counter, HC_C_U16_ARRAYELEMENTS(_au32Counter));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32Counter[enMode];
			}
		}
		else if (enLivetime == enStorageType)
		{
			dp_tclHeatCtrlDPSYSFAN_ControlModeCounterArray_LiveTime _oDPElement;
			tS32 _s32Result = _oDPElement.s32GetData(_au32Counter, HC_C_U16_ARRAYELEMENTS(_au32Counter));
			if (DP_S32_NO_ERR <= _s32Result)
			{
				_u32Return = _au32Counter[enMode];
			}
		}
		else
		{
			// runtime
			_u32Return = m_atStatisticInfo[enMode].u32ModeActivationCounter;
		}
	}
	ETG_TRACE_USR3(("u32GetModeActivationCounter: fan mode activation counter %d for mode %u and storage type %u read with result %d.",
			_u32Return, ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType), _s32Result));

	return _u32Return;
}

/*******************************************************************************
*
* FUNCTION: 	s32IncreaseModeActivationCounter()
*
* DESCRIPTION: 	fan mode activation counter depending of specified fan control mode
*
* PARAMETER: 	tenFANCtrlMode enMode
*               tU32 u32Count
*
* RETURNVALUE: 	tS32         { >=0 - successful | else - failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Statistics::s32IncreaseModeActivationCounter(tenFANCtrlMode enMode, tU32 u32Count)
{
	ETG_TRACE_USR4(("s32IncreaseModeActivationCounter(mode: %u, count: %d) entered.", ETG_ENUM(HC_FANMODE, enMode), u32Count));

	tS32 _s32Return = 0;
	tU32 _au32ActivationCounter[HC_C_U16_ARRAYELEMENTS(m_atStatisticInfo)] = {0};
	tS32 _s32Result = DP_S32_NO_ERR;

	if (enFANCtrlMode_MAX > enMode)
	{
		// increase runtime
		m_atStatisticInfo[enMode].u32ModeActivationCounter += u32Count;
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
		// increase operation time
		dp_tclHeatCtrlDPSYSFAN_ControlModeCounterArray_OperationTime _oDPElementOpTime;
		_s32Result = _oDPElementOpTime.s32GetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32ActivationCounter[enMode] += u32Count;
			_s32Result = _oDPElementOpTime.s32SetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseModeActivationCounter: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
				_s32Return -= 4;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseModeActivationCounter: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enOperationtime), _s32Result));
			_s32Return -= 2;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_OPERATIONTIME
#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
		// increase live time
		dp_tclHeatCtrlDPSYSFAN_ControlModeCounterArray_LiveTime _oDPElementLiveTime;
		_s32Result = _oDPElementLiveTime.s32GetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
		if (DP_S32_NO_ERR <= _s32Result)
		{
			_au32ActivationCounter[enMode] += u32Count;
			_s32Result = _oDPElementLiveTime.s32SetData(_au32ActivationCounter, HC_C_U16_ARRAYELEMENTS(_au32ActivationCounter));
			if (DP_S32_NO_ERR > _s32Result)
			{
				ETG_TRACE_ERR(("s32IncreaseModeActivationCounter: writing for mode %u and storage type %u failed with %d!",
						ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
				_s32Return -= 16;
			}
		}
		else
		{
			ETG_TRACE_ERR(("s32IncreaseModeActivationCounter: reading for mode %u and storage type %u failed with %d!",
					ETG_ENUM(HC_FANMODE, enMode), ETG_CENUM(tenFANStatisticStorageType, enLivetime), _s32Result));
			_s32Return -= 8;
		}
#endif //#ifdef VARIANT_S_FTR_ENABLE_STATISTICS_STORING_LIVETIME
	}
	else
	{
		ETG_TRACE_ERR(("s32IncreaseModeActivationCounter: unspecified mode %u!", ETG_ENUM(HC_FANMODE, enMode)));
		_s32Return -= 1;
	}

	return _s32Return;
}

/*******************************************************************************
*
* FUNCTION: 	vOnUpdateFANStatistics()
*
* DESCRIPTION: 	update statistical information
*
* PARAMETER: 	tenFANCtrlMode enMode            new FAN control mode
*               tenFANPowerState enPwrState      new FAN power state
*               tU8 u8Speed                      new FAN speed
*               tU16 u16Current                  new FAN current
*
* RETURNVALUE:  none
*
*******************************************************************************/
tVoid hc_tclFAN_Statistics::vOnUpdateFANStatistics(tenFANCtrlMode enMode, tenFANPowerState enPwrState, tU8 u8Speed, tU16 u16Current)
{
	ETG_TRACE_USR1(("vOnUpdateFANStatistics(mode: %u, pwrstate: %u, speed: %d percent, current: %d mA) entered.",
			ETG_ENUM(HC_FANMODE, enMode), ETG_ENUM(HC_FANPWRSTATE, enPwrState), u8Speed, u16Current));

	OSAL_tMSecond _u32CurTimeStamp = OSAL_ClockGetElapsedTime();
	OSAL_tMSecond _u32TimeDiff = (OSAL_NULL != m_tLastInfo.u32TimeStamp)?_u32CurTimeStamp - m_tLastInfo.u32TimeStamp:OSAL_NULL;
	tS32 _s32Result = 0;

	ETG_TRACE_USR1(("vOnUpdateFANStatistics: last info - %d ms ... time stamp: %d ms mode: %u, pwrstate: %u, speed: %d percent, current: %d mA ... time diff: %d ms",
			_u32CurTimeStamp, m_tLastInfo.u32TimeStamp,
			ETG_ENUM(HC_FANMODE, m_tLastInfo.enFANCtrlMode), ETG_ENUM(HC_FANPWRSTATE, m_tLastInfo.enFANPowerState),
			m_tLastInfo.u8FANSpeed, m_tLastInfo.u16FANCurrent, _u32TimeDiff ));

	if (enFANCtrlMode_MAX > m_tLastInfo.enFANCtrlMode)
	{
		// update fan activation duration and activation counter
		if (TRUE == bIsFanActive(m_tLastInfo.u8FANSpeed))
		{
			_s32Result = s32IncreaseActivationDuration(m_tLastInfo.enFANCtrlMode, _u32TimeDiff);
			if (0 > _s32Result)
			{
				ETG_TRACE_ERR(("vOnUpdateFANStatistics: activation duration for mode %u could not increase with %d ms and returned with %d!",
						ETG_ENUM(HC_FANMODE, m_tLastInfo.enFANCtrlMode), _u32TimeDiff, _s32Result ));
			}
		}
		else
		{
			if (TRUE == bIsFanActive(u8Speed))
			{
				_s32Result = s32IncreaseActivationCounter(m_tLastInfo.enFANCtrlMode, 1);
				if (0 > _s32Result)
				{
					ETG_TRACE_ERR(("vOnUpdateFANStatistics: activation counter for mode %u could not increase and returned with %d!",
							ETG_ENUM(HC_FANMODE, m_tLastInfo.enFANCtrlMode), _s32Result ));
				}
			}
		}
		// update fan mode duration
		_s32Result = s32IncreaseModeDuration(m_tLastInfo.enFANCtrlMode, _u32TimeDiff);
		if (0 > _s32Result)
		{
			ETG_TRACE_ERR(("vOnUpdateFANStatistics: mode duration for mode %u could not increase and returned with %d!",
					ETG_ENUM(HC_FANMODE, m_tLastInfo.enFANCtrlMode), _s32Result ));
		}
		// update fan control mode activation counter
		if (enMode != m_tLastInfo.enFANCtrlMode)
		{
			_s32Result = s32IncreaseModeActivationCounter(enMode, 1);
			if (0 > _s32Result)
			{
				ETG_TRACE_ERR(("vOnUpdateFANStatistics: mode activation counter for mode %u could not increase and returned with %d!",
						ETG_ENUM(HC_FANMODE, enMode), _s32Result ));
			}
		}
	}
	else
	{
		ETG_TRACE_ERR(("vOnUpdateFANStatistics: unspecified fan control mode %u!", ETG_ENUM(HC_FANMODE, m_tLastInfo.enFANCtrlMode) ));
	}
	// set last info
	m_tLastInfo.u32TimeStamp = _u32CurTimeStamp;
	m_tLastInfo.enFANCtrlMode = enMode;
	m_tLastInfo.enFANPowerState = enPwrState;
	m_tLastInfo.u8FANSpeed = u8Speed;
	m_tLastInfo.u16FANCurrent = u16Current;
}
