/*******************************************************************************
* FILE:        hc_tclFAN_Speed.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*-------------------------------------------------------------------------------
*
* DESCRIPTION: class to control fan speed
*
*-------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 28.07.2015| CM-AI/EPB2 Bernard | init
*
*******************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include "hc_AppMain.h"
#include "hc_tclFAN_Speed.h"
#include "hc_tclFAN.h"
#include "Ihc_ClientThermal.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclFAN_Speed.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Speed()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Speed::hc_tclFAN_Speed()
{
   ETG_TRACE_USR4(("hc_tclFAN_Speed() entered."));
   // reference of own FAN
   m_poFAN = OSAL_NULL;
   // requested FAN speed
   m_u8FANSpeedReq = 0;
   // confirmed FAN speed
   m_u8FANSpeedAck = 0;
   // current FAN speed
   m_u8FANSpeedCur = 0;
   // configured upper step size
   m_u8ConfigUpperStepSize = HC_FANCONTROL_FANSPEED_STEPSIZE_UPPER;
   // configured lower step size
   m_u8ConfigLowerStepSize = HC_FANCONTROL_FANSPEED_STEPSIZE_LOWER;

   // reference to device interface
   m_poDevIF = OSAL_NULL;
   // write counts of unchanged speed
   m_u8DevIFSpeedWriteCount = 0;
}

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Speed()
*
* DESCRIPTION: constructor
*
* PARAMETER:   hc_tclFAN* poFAN
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Speed::hc_tclFAN_Speed(hc_tclFAN* poFAN)
{
   ETG_TRACE_USR4(("hc_tclFAN_Speed(fan: 0x%08x) entered.", poFAN ));
   // reference of own FAN
   m_poFAN = poFAN;
   // requested FAN speed
   m_u8FANSpeedReq = 0;
   // confirmed FAN speed
   m_u8FANSpeedAck = 0;
   // current FAN speed
   m_u8FANSpeedCur = 0;
   // configured upper step size
   m_u8ConfigUpperStepSize = HC_FANCONTROL_FANSPEED_STEPSIZE_UPPER;
   // configured lower step size
   m_u8ConfigLowerStepSize = HC_FANCONTROL_FANSPEED_STEPSIZE_LOWER;

   // reference to device interface
   m_poDevIF = OSAL_NULL;
   // write counts of unchanged speed
   m_u8DevIFSpeedWriteCount = 0;
}
/*******************************************************************************
*
* FUNCTION:    ~hc_tclFAN_Speed()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Speed::~hc_tclFAN_Speed()
{
   ETG_TRACE_USR4(("~hc_tclFAN_Speed() entered."));

   m_poDevIF = OSAL_NULL;
   m_poFAN = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences()
*
* DESCRIPTION: Function to get all reference needed by this class.
* 			   A reference should always be the Interface class of the object
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Speed::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));

   hc_tclAppMain* _poMain = hc_tclAppMain::theServer();
   if (OSAL_NULL != _poMain)
   {
	   m_poDevIF = dynamic_cast<I_hc_tclClientThermal*>(_poMain->getHandler("I_hc_tclClientThermal"));
   }

}

/*******************************************************************************
*
* FUNCTION:    vStartCommunication()
*
* DESCRIPTION: Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Speed::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   // reset own state informations
   // ... requested FAN speed
   m_u8FANSpeedReq = 0;
   // ... confirmed FAN speed
   m_u8FANSpeedAck = OSAL_C_U8_MAX;
   // ... current FAN speed
   m_u8FANSpeedCur = 0;
}

/*******************************************************************************
*
* FUNCTION:    vStopCommunication()
*
* DESCRIPTION: Function to stop all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Speed::vStopCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStopCommunication() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vMonitorFANSpeed()
*
* DESCRIPTION: supervision of FAN speed
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Speed::vMonitorFANSpeed(tVoid)
{
   ETG_TRACE_USR4(("vMonitorFANSpeed() entered."));

   if (TRUE == bGetFanSpeedAvail())
   {
	   // trigger FAN speed output stepper
	   vFANSpeedStepper();
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_FANSpeed()
*
* DESCRIPTION: trace attributes of FAN speed
*
* PARAMETER:   const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Speed::vTraceInfo_FANSpeed(const char* strIdentifier) const
{
   ETG_TRACE_USR4(("vTraceInfo_FANSpeed() entered."));

	ETG_TRACE_USR2(("%26s: <<< FAN Speed >>> ", strIdentifier ));

	ETG_TRACE_USR2(("%26s: %30s=%d [perc] ",strIdentifier, "       FAN_SPEED_CUR",
			u8GetFANSpeedCur() ));
	ETG_TRACE_USR2(("%26s: %30s=%d [perc] ",strIdentifier, "       FAN_SPEED_REQ",
			u8GetFANSpeedReq() ));
	ETG_TRACE_USR2(("%26s: %30s=%u / %u [perc] ",strIdentifier, "       FAN_SPEED_ACK",
			u8GetFANSpeedAck(), m_u8FANSpeedAck ));

	ETG_TRACE_USR2(("%26s: %30s=%d [perc] ",strIdentifier, " CFG_FAN_SPEED_UPPER",
			u8GetConfigUpperStepSize() ));
	ETG_TRACE_USR2(("%26s: %30s=%d [perc] ",strIdentifier, " CFG_FAN_SPEED_LOWER",
			u8GetConfigLowerStepSize() ));
}

/*******************************************************************************
*
* FUNCTION:    vFANSpeedOutputStepper()
*
* DESCRIPTION: output stepper of FAN speed
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Speed::vFANSpeedStepper(tVoid)
{
   ETG_TRACE_USR4(("vFANSpeedOutputStepper() entered."));

   tU8 _u8Speed = u8GetFANSpeedCur();
   // update current fan speed depending of requested fan speed
   if (u8GetFANSpeedReq() != u8GetFANSpeedCur())
   {
	   m_u8DevIFSpeedWriteCount = HC_FANCONTROL_FANSPEED_REWRITE_MAX_COUNT;
	   if (u8GetFANSpeedReq() < u8GetFANSpeedCur())
	   {
		   if (_u8Speed >= u8GetConfigLowerStepSize())
			   _u8Speed -= u8GetConfigLowerStepSize();
		   else
			   _u8Speed = 0;
		   if (_u8Speed < u8GetFANSpeedReq())
			   _u8Speed = u8GetFANSpeedReq();
	   }
	   else
	   {
		   if (_u8Speed <= (100 - u8GetConfigUpperStepSize()))
			   _u8Speed += u8GetConfigUpperStepSize();
		   else
			   _u8Speed = 100;
		   if (_u8Speed > u8GetFANSpeedReq())
			   _u8Speed = u8GetFANSpeedReq();
	   }
   }
   if (100 < _u8Speed)
   {
	   _u8Speed = 100;
   }
   // set device interface speed value
   if (0 < m_u8DevIFSpeedWriteCount)
   {
	   if (_u8Speed != u8SetDevIFSpeed(_u8Speed))
	   {
		   // new speed could not set
		   ETG_TRACE_FATAL(("vFANSpeedStepper: new device speed %u percent couldn't set!", _u8Speed ));
	   }
	   else
	   {
		   if (0 < m_u8DevIFSpeedWriteCount)
			   --m_u8DevIFSpeedWriteCount;
		   u8SetFANSpeedCur(_u8Speed);
	   }
   }
}

/*******************************************************************************
*
* FUNCTION:    bGetDevIFAvail()
*
* DESCRIPTION: device interface availability state
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclFAN_Speed::bGetDevIFAvail(tVoid) const
{
   ETG_TRACE_USR4(("bGetDevIFAvail() entered."));

   if (OSAL_NULL != m_poDevIF)
   {
	   return m_poDevIF->bIsServiceAvailable();
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u8SetDevIFSpeed()
*
* DESCRIPTION: write speed to device
*
* PARAMETER:   tU8 u8Speed               new speed
*
* RETURNVALUE: tU8                       written speed
*
*******************************************************************************/
tU8 hc_tclFAN_Speed::u8SetDevIFSpeed(tU8 u8Speed) const
{
   ETG_TRACE_USR4(("u8SetDevIFSpeed(speed: %u) entered.", u8Speed ));

   if (OSAL_NULL != m_poDevIF && OSAL_NULL != m_poFAN)
   {
	   return m_poDevIF->u8SetFANSpeed(m_poFAN->enGetFANID(), u8Speed);
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u8SetDevIFSpeed()
*
* DESCRIPTION: write speed to device
*
* PARAMETER:   tenThermalSensorID 		enSensorID
*              tU8 u8Speed              new speed
*
* RETURNVALUE: tU8                       written speed
*
*******************************************************************************/
tU8 hc_tclFAN_Speed::u8SetDevIFSpeed(tenThermalSensorID enSensorID, tU8 u8Speed) const
{
   ETG_TRACE_USR4(("u8SetDevIFSpeed(speed: %u) entered.", u8Speed ));

   if (OSAL_NULL != m_poDevIF)
   {
	   return m_poDevIF->u8SetFANSpeed(enSensorID, u8Speed);
   }
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u8GetDevIFSpeed()
*
* DESCRIPTION: read speed from device
*
* PARAMETER:   none
*
* RETURNVALUE: tU8                       current speed
*
*******************************************************************************/
tU8 hc_tclFAN_Speed::u8GetDevIFSpeed(tVoid) const
{
   ETG_TRACE_USR4(("u8GetDevIFSpeed() entered."));

   if (OSAL_NULL != m_poDevIF)
   {
	   return m_poDevIF->u8GetFANSpeed();
   }
   return 0;
}
