/************************************************************************
* FILE:        hc_tclFAN_Power.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: class to control fan power state
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 28.07.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include "hc_AppMain.h"
#include "hc_tclFAN_Power.h"
#include "hc_tclFAN.h"
#include "hc_tclFAN_Speed.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclFAN_Power.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Power()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Power::hc_tclFAN_Power()
{
   ETG_TRACE_USR4(("hc_tclFAN_Power() entered."));
   m_poFAN = OSAL_NULL;
   m_enFANPwrStateReq = enFANPowerState_INIT;
   m_enFANPwrStateAck = enFANPowerState_UNDEF;
   // device interface
   m_poDevIF = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Power()
*
* DESCRIPTION: constructor
*
* PARAMETER:   hc_tclFAN* poFAN                reference of own FAN
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Power::hc_tclFAN_Power(hc_tclFAN* poFAN)
{
   ETG_TRACE_USR4(("hc_tclFAN_Power() entered."));
   m_poFAN = poFAN;
   m_enFANPwrStateReq = enFANPowerState_INIT;
   m_enFANPwrStateAck = enFANPowerState_UNDEF;
   // device interface
   m_poDevIF = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclFAN_Power()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Power::~hc_tclFAN_Power()
{
   ETG_TRACE_USR4(("~hc_tclFAN_Power() entered."));
   m_poDevIF = OSAL_NULL;
   m_poFAN = OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION:    vGetReferences()
*
* DESCRIPTION: Function to get all reference needed by this class.
* 			   A reference should always be the Interface class of the object
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Power::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   // load reference of device interface
   if (OSAL_NULL != m_poFAN)
   {
	   m_poDevIF = m_poFAN->poGetSpeed();
   }
}

/*******************************************************************************
*
* FUNCTION:    vStartCommunication()
*
* DESCRIPTION: Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Power::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   // reset own state informations
   m_enFANPwrStateReq = enFANPowerState_INIT;
   m_enFANPwrStateAck = enFANPowerState_UNDEF;
}

/*******************************************************************************
*
* FUNCTION:    vStopCommunication()
*
* DESCRIPTION: Function to stop all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Power::vStopCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStopCommunication() entered."));
}

/*******************************************************************************
*
* FUNCTION:    bFanPowerAvail()
*
* DESCRIPTION: returns the availability of fan power
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclFAN_Power::bFanPowerAvail(tVoid) const
{
   ETG_TRACE_USR4(("bFanPowerAvail() entered." ));

   if (OSAL_NULL != m_poDevIF)
   {
	   return m_poDevIF->bGetFanSpeedAvail();
   }
   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    enSetPwrState()
*
* DESCRIPTION: set FAN power state and return the new requested state
*
* PARAMETER:   tenFANPowerState enState
*
* RETURNVALUE: tenFANPowerState
*
*******************************************************************************/
tenFANPowerState hc_tclFAN_Power::enSetPwrState(tenFANPowerState enState)
{
   ETG_TRACE_USR4(("enSetPwrState(state: %u) entered.", enState));

   if (enGetPwrStateReq() != enState)
   {
	   m_enFANPwrStateReq = enState;
   }

   return enGetPwrStateReq();
}

/*******************************************************************************
*
* FUNCTION:    vMonitor()
*
* DESCRIPTION: supervision of FAN power state
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Power::vMonitorFANPwrState(tVoid)
{
   ETG_TRACE_USR4(("vMonitorFANPwrState() entered."));
   // reading from device interface
   if (TRUE == bFanPowerAvail())
   {
	   if (OSAL_NULL != m_poDevIF)
	   {
		   (tVoid)enSetPwrStateAck(0 < m_poDevIF->u8GetFANSpeedAck());
	   }
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_FANPwrState()
*
* DESCRIPTION: trace attributes of FAN power state
*
* PARAMETER:   const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Power::vTraceInfo_FANPwrState(const char* strIdentifier) const
{
    ETG_TRACE_USR4(("vTraceInfo_FANPwrState() entered."));

	ETG_TRACE_USR2(("%26s: <<< FAN Power State >>> ", strIdentifier ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",    	strIdentifier, "       FAN_PWR_AVAIL",
			ETG_ENUM(HC_Bool, bFanPowerAvail()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",		strIdentifier, "   FAN_PWR_STATE_REQ",
			ETG_ENUM(HC_FANPWRSTATE, enGetPwrStateReq()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",		strIdentifier, "   FAN_PWR_STATE_ACK",
			ETG_ENUM(HC_FANPWRSTATE, enGetPwrStateAck()) ));
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ", 	strIdentifier, "       PWR_DEVICE_IF",
			m_poDevIF ));
}
