/*******************************************************************************
* FILE:        hc_tclFAN_Current.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*-------------------------------------------------------------------------------
*
* DESCRIPTION: class to measure fan current
*
*-------------------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 28.07.2015| CM-AI/EPB2 Bernard | init
*
*******************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/
#include "hc_AppMain.h"
#include "hc_tclFAN_Current.h"
#include "hc_tclFAN.h"

/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclFAN_Current.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Current()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Current::hc_tclFAN_Current()
{
   ETG_TRACE_USR4(("hc_tclFAN_Current() entered."));
   m_u16FANCurrent = 0;
   m_u16FANCurrentAve = 0;
   m_u16FANCurrentAveElements = 0;
   m_u8LastFanSpeed = 0;

   m_poFAN = OSAL_NULL;

   (tVoid)memset((tVoid*)m_strADCDevicePath, OSAL_NULL, sizeof(m_strADCDevicePath));
   (tVoid)strncpy(m_strADCDevicePath, OSAL_C_STRING_DEVICE_ADC_FAN_DIAG, strlen(OSAL_C_STRING_DEVICE_ADC_FAN_DIAG));
   m_hADCDevice = OSAL_ERROR;
   m_u32ADCErrorCode = OSAL_E_NOERROR;
   (tVoid)memset((tVoid*)m_strADCErrorText, OSAL_NULL, sizeof(m_strADCErrorText));
   (tVoid)strncpy(m_strADCErrorText, OSAL_coszErrorText(m_u32ADCErrorCode), sizeof(m_strADCErrorText)-1);
   m_u16ADCMeasureBuffer = 0;
   m_u16ADCVolt = 0;
   m_bADCAvailable = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    hc_tclFAN_Current()
*
* DESCRIPTION: constructor
*
* PARAMETER:   hc_tclFAN* poFAN				reference of own FAN
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Current::hc_tclFAN_Current(hc_tclFAN* poFAN)
{
   ETG_TRACE_USR4(("hc_tclFAN_Current(fan: 0x%08x) entered.", poFAN));
   m_u16FANCurrent = 0;
   m_u16FANCurrentAve = 0;
   m_u16FANCurrentAveElements = 0;
   m_u8LastFanSpeed = 0;

   m_poFAN = poFAN;

   (tVoid)memset((tVoid*)m_strADCDevicePath, OSAL_NULL, sizeof(m_strADCDevicePath));
   (tVoid)strncpy(m_strADCDevicePath, OSAL_C_STRING_DEVICE_ADC_FAN_DIAG, strlen(OSAL_C_STRING_DEVICE_ADC_FAN_DIAG));
   m_hADCDevice = OSAL_ERROR;
   m_u32ADCErrorCode = OSAL_E_NOERROR;
   (tVoid)memset((tVoid*)m_strADCErrorText, OSAL_NULL, sizeof(m_strADCErrorText));
   (tVoid)strncpy(m_strADCErrorText, OSAL_coszErrorText(m_u32ADCErrorCode), sizeof(m_strADCErrorText)-1);
   m_u16ADCMeasureBuffer = 0;
   m_u16ADCVolt = 0;
   m_bADCAvailable = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclFAN_Current()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclFAN_Current::~hc_tclFAN_Current()
{
   ETG_TRACE_USR4(("~hc_tclFAN_Current() entered."));
   m_poFAN = OSAL_NULL;
   (tVoid)bSetADCAvailable(FALSE);
   (tVoid)s32ADCClose();

}

/*******************************************************************************
*
* FUNCTION:    vGetReferences()
*
* DESCRIPTION: Function to get all reference needed by this class.
* 			   A reference should always be the Interface class of the object
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
}

/*******************************************************************************
*
* FUNCTION:    vStartCommunication()
*
* DESCRIPTION: Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   // reset state information
   m_u16FANCurrent = 0;
   m_u16FANCurrentAve = 0;
   m_u16FANCurrentAveElements = 0;
   m_u16ADCMeasureBuffer = 0;
   m_u16ADCVolt = 0;
   // open device
   if (OSAL_ERROR == s32ADCOpen())
   {
	   (tVoid)bSetADCAvailable(FALSE);
	   ETG_TRACE_FATAL(("vStartCommunication: error occurred during opening of ADC device %50s! (Error %d: %s)",
			   strGetADCDevicePath(), u32GetADCErrorCode(), strGetADCErrorText() ));
   }
   else
   {
	   (tVoid)bSetADCAvailable(TRUE);
   }
}

/*******************************************************************************
*
* FUNCTION:    vStopCommunication()
*
* DESCRIPTION: Function to stop all dynamic objects e.g. threads, ...
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vStopCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStopCommunication() entered."));

   // close device
   (tVoid)bSetADCAvailable(FALSE);
   (tVoid)s32ADCClose();
}

/*******************************************************************************
*
* FUNCTION:    vSetFANCurrentAve()
*
* DESCRIPTION: supervision of FAN current
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vSetFANCurrentAve(tU16 u16Current)
{
   ETG_TRACE_USR4(("vSetFANCurrentAve(current: %d mA) entered.", u16Current ));

   if (OSAL_NULL != m_poFAN && m_u8LastFanSpeed != m_poFAN->u8GetSpeed())
   {
	   m_u16FANCurrentAveElements = 0;
	   m_u8LastFanSpeed = m_poFAN->u8GetSpeed();
   }
   tU16 _u16AveElements = m_u16FANCurrentAveElements;
   tU32 _u32AveCurrent = (tU32)u16GetFANCurrentAve();
   if (HC_FANCONTROL_MAX_AVERAGE_ELEMENTS > _u16AveElements)
	   ++_u16AveElements;
   else if (HC_FANCONTROL_MAX_AVERAGE_ELEMENTS < _u16AveElements)
	   return;
   m_u16FANCurrentAve = (tU16)(((_u32AveCurrent * (_u16AveElements-1)) + u16Current) / _u16AveElements);
   m_u16FANCurrentAveElements = _u16AveElements;
}

/*******************************************************************************
*
* FUNCTION:    vMonitorFANCurrent()
*
* DESCRIPTION: supervision of FAN current
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vMonitorFANCurrent(tVoid)
{
   ETG_TRACE_USR4(("vMonitorFANCurrent() entered."));
   // read ADC FAN_DIAG
   if (TRUE == bGetADCAvailable())
   {
	   if (OSAL_ERROR == s32ADCRead())
	   {
		   ETG_TRACE_FATAL(("vMonitorFANCurrent: Error occurred during reading of ADC %50s!  (Error %d: %s)",
				   strGetADCDevicePath(), u32GetADCErrorCode(), strGetADCErrorText() ));
	   }
   }
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_FANCurrent()
*
* DESCRIPTION: trace attributes of FAN current
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vTraceInfo_FANCurrent(const char* strIdentifier) const
{
    ETG_TRACE_USR4(("vTraceInfo_FANCurrent() entered."));

	ETG_TRACE_USR2(("%26s: <<< FAN current >>> ", strIdentifier ));
	ETG_TRACE_USR2(("%26s: %30s=%d [mA] ",	strIdentifier, "         FAN_CURRENT",
			u16GetFANCurrent() ));
	ETG_TRACE_USR2(("%26s: %30s=%d [mA] ", 	strIdentifier, " FAN_CURRENT_AVERAGE",
			u16GetFANCurrentAve() ));

	ETG_TRACE_USR2(("%26s: %30s=0x%04x ", 	strIdentifier, "           ADC_VALUE",
			u16GetADCMeasureBuffer() ));
	ETG_TRACE_USR2(("%26s: %30s=%d [mV] ", 	strIdentifier,"         ADC_VOLTAGE",
			u16GetADCVoltage() ));
	ETG_TRACE_USR2(("%26s: %30s=%u ",    	strIdentifier, "    ADC_DEVICE_AVAIL",
			ETG_ENUM(HC_Bool, bGetADCAvailable()) ));
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ", 	strIdentifier, "          ADC_DEVICE",
			hGetADCDevice() ));
	ETG_TRACE_USR2(("%26s: %30s=%s ",     	strIdentifier, "      ADC_DEVICEPATH",
			strGetADCDevicePath() ));
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ", 	strIdentifier, "       ADC_ERRORCODE",
			u32GetADCErrorCode() ));
	ETG_TRACE_USR2(("%26s: %30s=%s ",     	strIdentifier, "       ADC_ERRORTEXT",
			strGetADCErrorText() ));
}

/*******************************************************************************
*
* FUNCTION:    s32ADCOpen()
*
* DESCRIPTION: open ADC interface device
*
* PARAMETER:   None
*
* RETURNVALUE: tS32		{OSAL_OK = successful | OSAL_ERROR = failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Current::s32ADCOpen(tVoid)
{
   ETG_TRACE_USR4(("vMonitor() entered."));

   tS32 _s32Return = OSAL_OK;

   if (OSAL_ERROR != hGetADCDevice())
	   (tVoid)s32ADCClose();

   m_hADCDevice = OSAL_IOOpen(strGetADCDevicePath(), OSAL_EN_READONLY);
   if (OSAL_ERROR != hGetADCDevice())
   {
	   vSetADCErrorCode(OSAL_E_NOERROR);
	   if (OSAL_ERROR == s32ADCRead())
	   {
		   ETG_TRACE_FATAL(("s32ADCOpen: ADC couldn't read!  (Error %d: %s)",
				   u32GetADCErrorCode(), strGetADCErrorText() ));
	   }
   }
   else
   {
	   _s32Return = OSAL_ERROR;
	   vSetADCErrorCode(OSAL_u32ErrorCode());

	   ETG_TRACE_FATAL(("s32ADCOpen: couldn't open ADC device %50s! (Error %d: %s)",
			   strGetADCDevicePath(), u32GetADCErrorCode(), strGetADCErrorText() ));
   }

   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32ADCRead()
*
* DESCRIPTION: read from ADC interface device
*
* PARAMETER:   None
*
* RETURNVALUE: tS32		{OSAL_OK = successful | OSAL_ERROR = failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Current::s32ADCRead(tVoid)
{
	ETG_TRACE_USR4(("s32ADCRead() entered."));

	tS32 _s32Return = OSAL_ERROR;

	if (OSAL_ERROR != hGetADCDevice())
	{
		if (OSAL_ERROR == OSAL_s32IORead(hGetADCDevice(), (tPS8)&m_u16ADCMeasureBuffer, sizeof(m_u16ADCMeasureBuffer)))
		{
			vSetADCErrorCode(OSAL_u32ErrorCode());

			ETG_TRACE_FATAL(( "s32ADCRead: ADC couldn't read !  (Error %d: %s)",
				   u32GetADCErrorCode(), strGetADCErrorText() ));
		}
		else
		{
			_s32Return = OSAL_OK;
			vSetADCErrorCode(OSAL_E_NOERROR);
			// convert to voltage and save it
			tU16 _u16ADCVoltage = u16SetADCVoltage(u16Conv2miliVolt(u16GetADCMeasureBuffer()));
			// convert to current and set new FAN current
			(tVoid)u16SetFANCurrent(u16Conv2miliAmp(_u16ADCVoltage));
			ETG_TRACE_USR3(( "s32ADCRead: Successful ADC value 0x%04x read (voltage: %d mV, current: %d mA, average current: %d mA)",
					u16GetADCMeasureBuffer(), u16GetADCVoltage(), u16GetFANCurrent(), u16GetFANCurrentAve() ));
		}
	}

	return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    s32ADCClose()
*
* DESCRIPTION: close ADC interface device
*
* PARAMETER:   None
*
* RETURNVALUE: tS32		{OSAL_OK = successful | OSAL_ERROR = failed}
*
*******************************************************************************/
tS32 hc_tclFAN_Current::s32ADCClose(tVoid)
{
   ETG_TRACE_USR4(("s32ADCClose() entered."));

   tS32 _s32Return = OSAL_OK;

   if ((OSAL_ERROR != hGetADCDevice()) && (OSAL_ERROR == OSAL_s32IOClose(hGetADCDevice())))
   {
	   _s32Return = OSAL_ERROR;
	   vSetADCErrorCode(OSAL_u32ErrorCode());

	   ETG_TRACE_FATAL(("s32GPIOClose: error occurred during closing of ADC device %50s! (Error %d: %s)",
			   strGetADCDevicePath(), u32GetADCErrorCode(), strGetADCErrorText() ));
   }
   else
   {
	   vSetADCErrorCode(OSAL_E_NOERROR);
   }
   m_hADCDevice = OSAL_ERROR;
   return _s32Return;
}

/*******************************************************************************
*
* FUNCTION:    vSetADCErrorCode()
*
* DESCRIPTION: set error code of last GPIO interface device access
*
* PARAMETER:   tU32 u32ErrorCode
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid hc_tclFAN_Current::vSetADCErrorCode(tU32 u32ErrorCode)
{
   ETG_TRACE_USR4(("vSetADCErrorCode(error code: 0x%08x) entered.", u32ErrorCode ));

   m_u32ADCErrorCode = u32ErrorCode;
   (tVoid)memset((tVoid*)m_strADCErrorText, OSAL_NULL, sizeof(m_strADCErrorText));
   (tVoid)strncpy(m_strADCErrorText, OSAL_coszErrorText(u32GetADCErrorCode()), sizeof(m_strADCErrorText)-1);
}
