/************************************************************************
* FILE:        hc_tclFANControl.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: class to control fan
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 22.04.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/
// datapool access
#ifndef DP_S_IMPORT_INTERFACE_FI
#define DP_S_IMPORT_INTERFACE_FI
#include "dp_fc_heatctrl_if.h"
#endif //#ifndef DP_S_IMPORT_INTERFACE_FI

#include "I_hc_tclSimuMode.h"
#include "I_hc_tclCfgMgr.h"
#include "hc_tclFANControl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclFANControl.cpp.trc.h"
#endif

/*******************************************************************************
| defines and macros (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| typedefs (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: global)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| variable definition (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function prototype (scope: module-local)
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| function implementation (scope: module-local)
|-----------------------------------------------------------------------------*/

/*************************************************************************
*
* FUNCTION:    hc_tclFANControl()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain* poMainAppl: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclFANControl::hc_tclFANControl(const hc_tclAppMain* poMainAppl)
: I_hc_tclFANControl(poMainAppl)
{
   ETG_TRACE_USR4(("hc_tclFANControl() entered."));
   // simulation mode
   m_poSimuMode = OSAL_NULL;
   // configuration manager
   m_poCfgMgr = OSAL_NULL;
   // configuration
   m_bEnable = FALSE;
   m_u8EnableDevel = OSAL_C_U8_MAX;
   // initial references of fan objects
   (tVoid)memset((tVoid*)m_apoFAN, OSAL_NULL, sizeof(m_apoFAN));
   m_apoFAN[enFANID_SYSTEM] = &m_oFAN;
   // sensor specific fan speeds
   (tVoid)memset((tVoid*)m_au8SensorFanSpeed, OSAL_NULL, sizeof(m_au8SensorFanSpeed));

   // trigger message fan update
   (tVoid)memset((tVoid*)&m_oTriggerFanUpdate, OSAL_NULL, sizeof(m_oTriggerFanUpdate));
   (tVoid)OSAL_szStringNCopy(m_oTriggerFanUpdate.strClassName, "I_hc_tclFANControl", sizeof(m_oTriggerFanUpdate.strClassName)-1);
   m_oTriggerFanUpdate.eCmd = hc_tclBaseIf::eTriggerFANUpdate;
   // update message fan state
   (tVoid)memset((tVoid*)&m_oUpdateFanState, OSAL_NULL, sizeof(m_oUpdateFanState));
   (tVoid)OSAL_szStringNCopy(m_oUpdateFanState.strClassName, "Ihc_tclServiceHeatIf", sizeof(m_oUpdateFanState.strClassName)-1);
   m_oUpdateFanState.eCmd = hc_tclBaseIf::eUpdateFanState;
   // timer creation
   if (OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)cb_vTimerFanUpdate, (tVoid*)this, &m_hTimerFanUpdate))
   {
	   ETG_TRACE_FATAL(("hc_tclFANControl: couldn't create timer to update fan info."));
	   m_hTimerFanUpdate = OSAL_NULL;
   }
   else
   {
	   (tVoid)OSAL_s32TimerSetTime(hGetTimer(), 0, 0);
   }
}


/*************************************************************************
*
* FUNCTION:    ~hc_tclFANControl()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclFANControl::~hc_tclFANControl()
{
   ETG_TRACE_USR4(("~hc_tclFANControl() entered."));
   if (OSAL_NULL != hGetTimer())
   {
	   (tVoid)OSAL_s32TimerSetTime(hGetTimer(), 0, 0);
	   (tVoid)OSAL_s32TimerDelete(hGetTimer());
   }
   (tVoid)memset((tVoid*)m_apoFAN, OSAL_NULL, sizeof(m_apoFAN));
   m_poCfgMgr = OSAL_NULL;
   m_poSimuMode = OSAL_NULL;
}

/*************************************************************************
*
* FUNCTION:    vHandleMessage
*
* DESCRIPTION: handle internal message
*
* PARAMETER:   hc_tclBaseIf::TMsg* pMsg : reference of received internal message
*
* RETURNVALUE: none
*
*************************************************************************/
tVoid hc_tclFANControl::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
	HC_NULL_POINTER_CHECK(pMsg);
	ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));

	tBool _bChanged = FALSE;

	switch (pMsg->eCmd)
	{
	case hc_tclBaseIf::eTriggerFANUpdate:
	   {
			// monitor fan
			for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_apoFAN) > _u16Idx; ++_u16Idx)
			{
				tenFANID _enFanID = (tenFANID)_u16Idx;
				if (OSAL_NULL != poGetFan(_enFanID))
				{
					tenFANCtrlMode _enMode = enGetFANCtrlMode(_enFanID);
					tenFANPowerState _enPwrState = enGetFANPwrState(_enFanID);
					tU8 _u8Speed = u8GetFANSpeed(_enFanID);
					tU16 _u16Current = u16GetFANCurrent(_enFanID);
					// update fan statistics
					poGetFan(_enFanID)->vOnUpdateFANStatistics(_enMode, _enPwrState, _u8Speed, _u16Current);
					// monitor running
					poGetFan(_enFanID)->vMonitorFAN();
					// current change
					tU16 _u16CurrentChange = (_u16Current > u16GetFANCurrent(_enFanID))
							?(_u16Current-u16GetFANCurrent(_enFanID))
									:(u16GetFANCurrent(_enFanID)-_u16Current);
					// update CCA service handler and trace output
					if (enGetFANCtrlMode(_enFanID) != _enMode
					   || enGetFANPwrState(_enFanID) != _enPwrState
					   || u8GetFANSpeed(_enFanID) != _u8Speed
					   || (HC_FANCONTROL_CURRENT_MIN_CHANGE <= _u16CurrentChange))
					{
						_bChanged = TRUE;
					}
				}
			}
			// restart timer
			if (OSAL_NULL != hGetTimer())
				(tVoid)OSAL_s32TimerSetTime(hGetTimer(), u32GetCycleTime(), 0);
	   }
	   break;

	case hc_tclBaseIf::eNewSensorFanSpeed:
	   {
		   // set new fan speed
		   for (tU16 _u16Idx=0; HC_C_U16_ARRAYELEMENTS(m_apoFAN) > _u16Idx; ++_u16Idx)
		   {
			   tenFANID _enFanID = (tenFANID)_u16Idx;
			   if (OSAL_NULL != poGetFan(_enFanID))
			   {
				   tenFANCtrlMode _enMode = enGetFANCtrlMode(_enFanID);
				   tenFANPowerState _enPwrState = enGetFANPwrState(_enFanID);
				   tU8 _u8Speed = u8GetFANSpeed(_enFanID);
				   tU16 _u16Current = u16GetFANCurrent(_enFanID);
				   // set new required fan speed of specified thermal sensor
				   (tVoid)poGetFan(_enFanID)->bSetReqFanControl(
						   pMsg->u.tFANSpeedReq.enSensorID,
						   enFANCtrlMode_AUTO, enFANPowerState_UNDEF,
						   pMsg->u.tFANSpeedReq.au8Speed[_enFanID], OSAL_C_U8_MAX);
					// current change
					tU16 _u16CurrentChange = (_u16Current > u16GetFANCurrent(_enFanID))
							?(_u16Current-u16GetFANCurrent(_enFanID))
									:(u16GetFANCurrent(_enFanID)-_u16Current);
				   if (enGetFANCtrlMode(_enFanID) != _enMode
					  || enGetFANPwrState(_enFanID) != _enPwrState
					  || u8GetFANSpeed(_enFanID) != _u8Speed
                      || (HC_FANCONTROL_CURRENT_MIN_CHANGE <= _u16CurrentChange))
				   {
					   _bChanged = TRUE;
				   }
			   }
		   }
	   }
	   break;

	case hc_tclBaseIf::eTriggerLoadSettings:
	   {
		   // stop fan control
		   vStopCommunication();
		   // start fan control
		   vStartCommunication();
	   }
	   break;

	default:
	   {
	   }
	   break;
	}
	if (TRUE == _bChanged)
	{
		vTraceInfo();
		if (OSAL_NULL != hc_tclAppMain::theServer())
			hc_tclAppMain::theServer()->vPostInternalMessage(&m_oUpdateFanState);
	}
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vHandleTraceMessage(const tUChar* puchData)
{
	ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
	HC_NULL_POINTER_CHECK(puchData);

	tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
	switch (u32MsgCode) {
	case FC_HEATCTRL_HC_GETFANSTATE:
	   {
		   vHandleTraceCmd_GetFanState(puchData);
	   }
	   break;

	case FC_HEATCTRL_HC_GETFANSTATISTICS:
	   {
		   vHandleTraceCmd_GetFanStatistics(puchData);
	   }
	   break;

	case FC_HEATCTRL_HC_SETFANREQ:
	   {
		   vHandleTraceCmd_SetFanReq(puchData);
	   }
	   break;

	case FC_HEATCTRL_HC_CONFIGFAN_ENABLE:
	   {
		   vHandleTraceCmd_ConfigFanEnable(puchData);
	   }
	   break;

	case FC_HEATCTRL_HC_CONFIGFAN_ENABLE_RESET:
	   {
		   vHandleTraceCmd_ConfigFanEnable_Reset(puchData);
	   }
	   break;

	default:
	   ETG_TRACE_USR4(("hc_tclFANControl::vHandleTraceMessage(): unhandled command %d.", u32MsgCode));
	   break;
	}
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   // reference of simulation mode interface
   m_poSimuMode = dynamic_cast<I_hc_tclSimuMode*>(_cpoMain->getHandler("I_hc_tclSimuMode"));
   m_poCfgMgr = dynamic_cast<I_hc_tclCfgMgr*>(_cpoMain->getHandler("I_hc_tclCfgMgr"));
   // fan references
   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_apoFAN) > _u16Idx; ++_u16Idx)
   {
	   if (OSAL_NULL != poGetFan((tenFANID)_u16Idx))
		   poGetFan((tenFANID)_u16Idx)->vGetReferences();
   }
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   // load configuration
   (tVoid)bLoadConfig();
   // check configuration
   if (TRUE == bGetEnable())
   {
	   ETG_TRACE_USR3(("vStartCommunication: enabled fan control."));
	   // fan start
	   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_apoFAN) > _u16Idx; ++_u16Idx)
	   {
		   if (OSAL_NULL != poGetFan((tenFANID)_u16Idx))
			   poGetFan((tenFANID)_u16Idx)->vStartCommunication();
	   }
	   // start timer
	   if (OSAL_NULL != hGetTimer())
	   {
		   (tVoid)OSAL_s32TimerSetTime(hGetTimer(), u32GetCycleTime(), 0);
	   }
   }
   else
   {
	   ETG_TRACE_USR3(("vStartCommunication: fan control disabled and will be stopped."));
	   vStopCommunication();
   }
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStopCommunication()
*
* DESCRIPTION: 	Function to stop all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vStopCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStopCommunication() entered."));
   // stop timer
   if (OSAL_NULL != hGetTimer())
   {
	   (tVoid)OSAL_s32TimerSetTime(hGetTimer(), 0, 0);
   }
   // stop fan
   for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_apoFAN) > _u16Idx; ++_u16Idx)
   {
	   if (OSAL_NULL != poGetFan((tenFANID)_u16Idx))
		   poGetFan((tenFANID)_u16Idx)->vStopCommunication();
   }
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	Trace information
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vTraceInfo()
{
	ETG_TRACE_USR4(("vTraceInfo() entered."));
	tChar _strIdentStart[30] = "HC.APPL.FAN_.CTRL_.START";
	tChar _strIdentAttr[30] = "HC.APPL.FAN_.CTRL_.ATTR_";
	tChar _strIdentEnd[30] = "HC.APPL.FAN_.CTRL_.END__";
	// trace starting of trace block
	ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
	vTraceInfo_FANCONTROL(_strIdentAttr);
	for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_apoFAN) > _u16Idx; ++_u16Idx)
	{
	   if (OSAL_NULL != poGetFan((tenFANID)_u16Idx))
		   poGetFan((tenFANID)_u16Idx)->vTraceInfo_FAN(_strIdentAttr);
	}
	// trace ending of trace block
	ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION: 	enGetFANCtrlMode()
*
* DESCRIPTION: 	get the control mode of specified fan
*
* PARAMETER: 	tenFANID enFanID
*
* RETURNVALUE: 	tenFANCtrlMode
*
*******************************************************************************/
tenFANCtrlMode hc_tclFANControl::enGetFANCtrlMode(tenFANID enFanID) const
{
	ETG_TRACE_USR4(("enGetFANCtrlMode(fan: %u) entered.", ETG_ENUM(HC_FANID, enFanID)));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->enGetControlMode();
	return enFANCtrlMode_UNDEF;
}

/*******************************************************************************
*
* FUNCTION: 	enGetFANPwrState()
*
* DESCRIPTION: 	get the power state of specified fan
*
* PARAMETER: 	tenFANID enFanID
*
* RETURNVALUE: 	tenFANPowerState
*
*******************************************************************************/
tenFANPowerState hc_tclFANControl::enGetFANPwrState(tenFANID enFanID) const
{
	ETG_TRACE_USR4(("enGetFANPwrState(fan: %u) entered.", ETG_ENUM(HC_FANID, enFanID)));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->enGetPowerState();
	return enFANPowerState_UNDEF;
}

/*******************************************************************************
*
* FUNCTION: 	u8GetFANSpeed()
*
* DESCRIPTION: 	get the speed of specified fan
*
* PARAMETER: 	tenFANID enFanID
*
* RETURNVALUE: 	tU8		[percent]
*
*******************************************************************************/
tU8 hc_tclFANControl::u8GetFANSpeed(tenFANID enFanID) const
{
	ETG_TRACE_USR4(("u8GetFANSpeed(fan: %u) entered.", ETG_ENUM(HC_FANID, enFanID)));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->u8GetSpeed();
	return 0;
}

/*******************************************************************************
*
* FUNCTION: 	u16GetFANCurrent()
*
* DESCRIPTION: 	get the current of specified fan
*
* PARAMETER: 	tenFANID enFanID
*
* RETURNVALUE: 	tU16		[mA]
*
*******************************************************************************/
tU16 hc_tclFANControl::u16GetFANCurrent(tenFANID enFanID) const
{
	ETG_TRACE_USR4(("u16GetFANCurrent(fan: %u) entered.", ETG_ENUM(HC_FANID, enFanID)));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->u16GetCurrent();
	return 0;
}

/*******************************************************************************
*
* FUNCTION: 	u32GetActivationDuration()
*
* DESCRIPTION: 	getting ActivationDuration in [ms] depending of fan control mode
*               and storage type for specified fan
*
* PARAMETER: 	tenFANID enFanID
*               tenFANCtrlMode enMode
*               tenFANStatisticStorageType enStorageType
*
* RETURNVALUE: 	tU32		[ms]
*
*******************************************************************************/
tU32 hc_tclFANControl::u32GetActivationDuration(tenFANID enFanID, tenFANCtrlMode enMode, tenFANStatisticStorageType enStorageType) const
{
	ETG_TRACE_USR4(("u32GetActivationDuration(fan: %u, mode: %u, storage type: %u) entered.",
			ETG_ENUM(HC_FANID, enFanID), ETG_ENUM(HC_FANID, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType) ));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->u32GetActivationDuration(enMode, enStorageType);
	return 0;
}

/*******************************************************************************
*
* FUNCTION: 	u32GetActivationCounter()
*
* DESCRIPTION: 	getting ActivationCounter depending of fan control mode and
*               storage type for specified fan
*
* PARAMETER: 	tenFANID enFanID
*               tenFANCtrlMode enMode
*
* RETURNVALUE: 	tU32
*
*******************************************************************************/
tU32 hc_tclFANControl::u32GetActivationCounter(tenFANID enFanID, tenFANCtrlMode enMode, tenFANStatisticStorageType enStorageType) const
{
	ETG_TRACE_USR4(("u32GetActivationCounter(fan: %u, mode: %u, storage type: %u) entered.",
			ETG_ENUM(HC_FANID, enFanID), ETG_ENUM(HC_FANID, enMode), ETG_CENUM(tenFANStatisticStorageType, enStorageType) ));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->u32GetActivationCounter(enMode, enStorageType);
	return 0;
}

/*******************************************************************************
*
* FUNCTION: 	bSetReqFanControl()
*
* DESCRIPTION: 	set a new control request of specified fan
*
* PARAMETER: 	tenFANID enFanID
*               tenFANCtrlModeEvent enEvent
*               tenFANPowerState enPwrState
*               tU8 u8Speed
*               tU8 u8SpeedStart			     (default: 255 - dontcare)
*
* RETURNVALUE: 	tBool    {TRUE = successful | else = failed}
*
*******************************************************************************/
tBool hc_tclFANControl::bSetReqFanControl(tenFANID enFanID,
		tenFANCtrlModeEvent enEvent, tenFANPowerState enPwrState,
		tU8 u8Speed, tU8 u8SpeedStart)
{
	ETG_TRACE_USR4(("bSetReqFanControl() entered."));
	if (OSAL_NULL != poGetFan(enFanID))
		return poGetFan(enFanID)->bSetReqFanControl(enEvent, enPwrState, u8Speed, u8SpeedStart);
	return FALSE;
}

/*******************************************************************************
*
* FUNCTION: 	bLoadConfig()
*
* DESCRIPTION: 	loads the fan control configuration information
*
* PARAMETER: 	None
*
* RETURNVALUE: 	tBool       {TRUE - enable | else - disable}
*
*******************************************************************************/
tBool hc_tclFANControl::bLoadConfig(tVoid)
{
   ETG_TRACE_USR4(("bLoadConfig() entered." ));

   // reset fan enable
   m_bEnable = FALSE;

   // load configured fan enabling
   if ((tU8)TRUE < m_u8EnableDevel)
   {
	   dp_tclHeatCtrlDPFan_Mounted_Develop _oFanMountedDevelop;
	   if ((tU8)TRUE < _oFanMountedDevelop.tGetData()) // OSAL_C_U8_MAX
	   {
		   // load KDS configuration value
		   if (OSAL_NULL != m_poCfgMgr && ((tU8)TRUE >= m_poCfgMgr->u8GetKDSFANEnable()))
		   {
			   m_bEnable = ((tU8)TRUE == m_poCfgMgr->u8GetKDSFANEnable());
			   ETG_TRACE_USR3(("bLoadConfig: loaded KDS configuration value %d, enable = %d ",
					   m_poCfgMgr->u8GetKDSFANEnable(), m_bEnable ));
		   }
		   else
		   {
			   // load registry value
			   dp_tclHeatCtrlDPFan_Mounted _oFanMounted;
			   m_bEnable = _oFanMounted.tGetData();
			   ETG_TRACE_USR3(("bLoadConfig: loaded registry configuration value %d, enable = %d ",
					   _oFanMounted.tGetData(), m_bEnable ));
		   }
	   }
	   else
	   {
		   // load persistent developer configuration value
		   m_bEnable = ((tU8)TRUE == _oFanMountedDevelop.tGetData());
		   ETG_TRACE_USR3(("bLoadConfig: loaded persistent developer configuration value %d, enable = %d ",
				   _oFanMountedDevelop.tGetData(), m_bEnable ));
	   }
   }
   else
   {
	   // load dynamically developer configuration value
	   m_bEnable = ((tU8)TRUE == m_u8EnableDevel);
	   ETG_TRACE_USR3(("bLoadConfig: loaded dynamically developer configuration value %d, enable = %d ",
			   m_u8EnableDevel, m_bEnable ));
   }
   return m_bEnable;
}

/*******************************************************************************
*
* FUNCTION: 	cb_vTimerFanUpdate()
*
* DESCRIPTION: 	timer callback to update fan info
*
* PARAMETER: 	tVoid* pArg
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
OSAL_tpfCallback hc_tclFANControl::cb_vTimerFanUpdate(tVoid* pArg)
{
   ETG_TRACE_USR4(("cb_vTimerFanUpdate(pArg 0x%08x) entered.", pArg ));

   hc_tclFANControl* _poFanControl = (hc_tclFANControl*)pArg;

   if (OSAL_NULL != _poFanControl && OSAL_NULL != hc_tclAppMain::theServer())
   {
	   hc_tclAppMain::theServer()->vPostInternalMessage(&_poFanControl->m_oTriggerFanUpdate);
   }

   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo_FANControl()
*
* DESCRIPTION: trace attributes of FANControl
*
* PARAMETER:   const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vTraceInfo_FANCONTROL(const char* strIdentifier) const
{
    ETG_TRACE_USR4(( "vTraceInfo_FANCTRL() entered" ));

    ETG_TRACE_USR2(("%26s: <<< FANControl >>> ", strIdentifier ));

	ETG_TRACE_USR2(("%26s: %30s=%u ",       strIdentifier, "      FANCTRL_ENABLE",
			ETG_ENUM(HC_Bool, bGetEnable()) ));
	ETG_TRACE_USR2(("%26s: %30s=0x%08x ",   strIdentifier, "       FANCTRL_TIMER",
			hGetTimer() ));
	ETG_TRACE_USR2(("%26s: %30s=%0d [ms] ", strIdentifier, "   FANCTRL_CYCLETIME",
			u32GetCycleTime() ));
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetFanState()
*
* DESCRIPTION: 	handler for trace command HC_GetFanState
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vHandleTraceCmd_GetFanState(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_GetFanState() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   ETG_TRACE_FATAL(("%26s: HC_GetFanState ",
		   "HC.APPL.SIMU.CMD__.START" ));

   vTraceInfo();

   ETG_TRACE_FATAL(("%26s: finished without errors",
		   "HC.APPL.FAN_.CMD__.RES__" ));
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetFanStatistics()
*
* DESCRIPTION: 	handler for trace command HC_GetFanStatistics
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vHandleTraceCmd_GetFanStatistics(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_GetFanStatistics() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   ETG_TRACE_FATAL(("%26s: HC_GetFanStatistics ",
		   "HC.APPL.SIMU.CMD__.START" ));

   hc_tclFAN* _poFAN = poGetFan(enFANID_SYSTEM);
   if (OSAL_NULL != _poFAN)
   {
	   _poFAN->vTraceInfo_FAN_StatisticsComplete("HC.APPL.FAN_.CTRL_.ATTR_");
   }

   ETG_TRACE_FATAL(("%26s: finished without errors",
		   "HC.APPL.FAN_.CMD__.RES__" ));
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetFanState()
*
* DESCRIPTION: 	handler for trace command HC_SetFanReq
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vHandleTraceCmd_SetFanReq(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_SetFanReq() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU8 _u8Event = puchData[3];
   tU8 _u8PwrState = puchData[4];
   tU8 _u8Speed = puchData[5];
   tU8 _u8SpeedStart = puchData[6];

   ETG_TRACE_FATAL(("%26s: HC_SetFanReq %u %u %d %d ",
		   "HC.APPL.SIMU.CMD__.START",
		   ETG_ENUM(HC_FANMODEEVENT, _u8Event),
		   ETG_ENUM(HC_FANPWRSTATE, _u8PwrState),
		   _u8Speed, _u8SpeedStart ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   if (TRUE == bSetReqFanControl((tenFANCtrlModeEvent)_u8Event, (tenFANPowerState)_u8PwrState, _u8Speed, _u8SpeedStart))
	   {
		   ETG_TRACE_FATAL(("%26s: finished without errors",
				   "HC.APPL.FAN_.CMD__.RES__" ));
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors",
				   "HC.APPL.FAN_.CMD__.RES__" ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.FAN_.CMD__.RES__" ));

   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigFanEnable()
*
* DESCRIPTION: 	handler for trace command HC_ConfigFan_Enable
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vHandleTraceCmd_ConfigFanEnable(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigFanEnable() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tBool _bPersist = (TRUE == puchData[3]);
   tBool _bEnable = (TRUE == puchData[4]);

   ETG_TRACE_FATAL(("%26s: HC_ConfigFan_Enable %u %u ",
		   "HC.APPL.SIMU.CMD__.START",
		   ETG_ENUM(HC_Persist, _bPersist),
		   ETG_ENUM(HC_Bool, _bEnable) ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   tBool _bEnableOld = bGetEnable();
	   m_u8EnableDevel = (tU8)_bEnable;
	   if (TRUE == _bPersist)
	   {
		   dp_tclHeatCtrlDPFan_Mounted_Develop _oFanMountedDevelop;
		   if (m_u8EnableDevel != _oFanMountedDevelop.tGetData())
		   {
			   _oFanMountedDevelop.vSetData(m_u8EnableDevel);
		   }
	   }
	   if (_bEnableOld != _bEnable)
	   {
		   // stop fan control
		   vStopCommunication();
		   // start fan control
		   vStartCommunication();
	   }
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.FAN_.CMD__.RES__" ));
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.FAN_.CMD__.RES__" ));

   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigFanEnable_Reset()
*
* DESCRIPTION: 	handler for trace command HC_ConfigFan_Enable
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclFANControl::vHandleTraceCmd_ConfigFanEnable_Reset(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigFanEnable() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   ETG_TRACE_FATAL(("%26s: vHandleTraceCmd_ConfigFanEnable_Reset ",
		   "HC.APPL.SIMU.CMD__.START" ));

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   m_u8EnableDevel = OSAL_C_U8_MAX;
	   dp_tclHeatCtrlDPFan_Mounted_Develop _oFanMountedDevelop;
	   if (OSAL_C_U8_MAX != _oFanMountedDevelop.tGetData())
	   {
		   _oFanMountedDevelop.vSetData(m_u8EnableDevel);
	   }
	   // stop fan control
	   vStopCommunication();
	   // start fan control
	   vStartCommunication();
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.FAN_.CMD__.RES__" ));
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.FAN_.CMD__.RES__" ));

   }
}

/*******************************************************************************
*
* FUNCTION: 	poGetFan()
*
* DESCRIPTION: 	retun reference of specified fan object
*
* PARAMETER:   	tenFANID enFanID
*
* RETURNVALUE: 	hc_tclFAN*
*
*******************************************************************************/
hc_tclFAN* hc_tclFANControl::poGetFan(tenFANID enFanID) const
{
   ETG_TRACE_USR4(("poGetFan(fan: %u) entered.", ETG_ENUM(HC_FANID, enFanID) ));
   if (HC_C_U16_ARRAYELEMENTS(m_apoFAN) > (tU16)enFanID)
	   return m_apoFAN[enFanID];
   return OSAL_NULL;
}
