/************************************************************************
* FILE:        hc_tclEmergencyFunctionControl.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: classes to control all emergency functions
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 11.03.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "I_hc_tclSimuMode.h"
#include "hc_tclEmergencyFunctionControl.h"
#include "hc_tclEmergencyFunction.h"
#include "hc_tclEmergencyFunctionSystemShutdown.h"
#include "hc_tclEmergencyFunctionBacklightOFF.h"
#include "hc_tclEmergencyFunctionAttenuation.h"
#include "hc_tclEmergencyFunctionCDDriveOFF.h"
#include "hc_tclEmergencyFunctionDVDDriveOFF.h"
#include "hc_tclEmergencyFunctionTVModuleOFF.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclEmergencyFunctionControl.cpp.trc.h"
#endif

hc_tclEmergencyFunction* hc_tclEmergencyFunctionControl::m_apoFunction[enEmergencyFunctionID_MAX];

/*************************************************************************
*
* FUNCTION:    hc_tclEmergencyFunctionControl()
*
* DESCRIPTION: constructor
*
* PARAMETER:   const hc_tclAppMain* poMainAppl: main - object of this application
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclEmergencyFunctionControl::hc_tclEmergencyFunctionControl(const hc_tclAppMain* poMainAppl)
: I_hc_tclEmergencyFunctionControl(poMainAppl)
{
   ETG_TRACE_USR4(("hc_tclEmergencyFunctionControl() entered."));
   // simulation mode
   m_poSimuMode = OSAL_NULL;
   // init timer handles
   (tVoid)memset((tVoid*)m_ahDelayTimer,OSAL_NULL,sizeof(m_ahDelayTimer));
   // prepare update message for own service handler
   (tVoid)memset((tVoid*)&m_msgUpdateFunctionState_SrvHdl,OSAL_NULL,sizeof(m_msgUpdateFunctionState_SrvHdl));
   (tVoid)OSAL_szStringNCopy(m_msgUpdateFunctionState_SrvHdl.strClassName,
		   "Ihc_tclServiceHeatIf", sizeof(m_msgUpdateFunctionState_SrvHdl.strClassName)-1);
   m_msgUpdateFunctionState_SrvHdl.eCmd = hc_tclBaseIf::eUpdateEmergencyFunctionState;
   m_msgUpdateFunctionState_SrvHdl.u.tEmergencyFunctionState.enFunctionID = enEmergencyFunctionID_MAX;
   m_msgUpdateFunctionState_SrvHdl.u.tEmergencyFunctionState.enFunctionState = enEmergencyFunctionState_MAX;
   // prepare update message for all service handler
   (tVoid)memset((tVoid*)m_amsgUpdateFunctionState_Master,OSAL_NULL,sizeof(m_amsgUpdateFunctionState_Master));
   for (tU8 _u8FctID = 0; (tU8)enEmergencyFunctionID_MAX > _u8FctID; ++_u8FctID)
   {
	   m_amsgUpdateFunctionState_Master[_u8FctID].eCmd = hc_tclBaseIf::eUpdateEmergencyFunctionState;
	   m_amsgUpdateFunctionState_Master[_u8FctID].u.tEmergencyFunctionState.enFunctionID = (tenEmergencyFunctionID)_u8FctID;
   }
}


/*************************************************************************
*
* FUNCTION:    ~hc_tclEmergencyFunctionControl()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*************************************************************************/
hc_tclEmergencyFunctionControl::~hc_tclEmergencyFunctionControl()
{
   ETG_TRACE_USR4(("~hc_tclEmergencyFunctionControl() entered."));
   m_poSimuMode = OSAL_NULL;
   for (tU8 _u8Index = 0; (tU8)enEmergencyFunctionID_MAX > _u8Index; ++_u8Index)
   {
	   // stop and delete delay timer of emergency function
	   if (OSAL_NULL != m_ahDelayTimer[_u8Index])
	   {
		   (tVoid)OSAL_s32TimerSetTime(m_ahDelayTimer[_u8Index], 0, 0);
		   (tVoid)OSAL_s32TimerDelete(m_ahDelayTimer[_u8Index]);
	   }
	   m_ahDelayTimer[_u8Index] = OSAL_NULL;
	   // destroy emergency function objects
	   if (OSAL_NULL != m_apoFunction[_u8Index])
	   {
		   OSAL_DELETE m_apoFunction[_u8Index];
	   }
	   m_apoFunction[_u8Index] = OSAL_NULL;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage()
*
* DESCRIPTION: 	handler for internal messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleMessage(hc_tclBaseIf::TMsg* pMsg)
{
   ETG_TRACE_USR4(("vHandleMessage() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)pMsg->eCmd), pMsg->u.u32Data));
   HC_NULL_POINTER_CHECK(pMsg);
   switch (pMsg->eCmd)
   {
   case hc_tclBaseIf::eNewEmergencyFunctionState:
	   vHandleMessage_NewEmergencyFunctionInfo(*pMsg);
	   break;
   default:
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceMessage()
*
* DESCRIPTION: 	handler for trace command messages
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleTraceMessage(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceMessage() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   tU32 u32MsgCode = (puchData[1]<<8) | puchData[2];
   switch (u32MsgCode)
   {
   case FC_HEATCTRL_HC_GETEMERGENCYFUNCTIONSTATE:
	   {
		   vHandleTraceCmd_GetEFState(puchData);
	   }
	   break;

   case FC_HEATCTRL_HC_CONFIGEMERGENCYFUNCTION_ENABLE:
	   {
		   vHandleTraceCmd_ConfigEFEnable(puchData);
	   }
	   break;

   case FC_HEATCTRL_HC_CONFIGEMERGENCYFUNCTION_ENABLE_RESET:
	   {
		   vHandleTraceCmd_ConfigEFEnableReset(puchData);
	   }
	   break;

   case FC_HEATCTRL_HC_CONFIGEMERGENCYFUNCTION_DELAYTIME:
	   {
		   vHandleTraceCmd_ConfigEFDelayTime(puchData);
	   }
	   break;

   case FC_HEATCTRL_HC_CONFIGEMERGENCYFUNCTION_DELAYTIME_RESET:
	   {
		   vHandleTraceCmd_ConfigEFDelayTimeReset(puchData);
	   }
	   break;

	default:
	   ETG_TRACE_USR4(("hc_tclEmergencyFunctionControl::vHandleTraceMessage(): unhandled command %d.", u32MsgCode));
	   break;
   }
}

/*******************************************************************************
*
* FUNCTION: 	vGetReferences(tVoid)
*
* DESCRIPTION: 	Function to get all reference needed by this class.
* 				A reference should always be the Interface class of the object
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vGetReferences(tVoid)
{
   ETG_TRACE_USR4(("vGetReferences() entered."));
   // reference of simulation mode interface
   m_poSimuMode = dynamic_cast<I_hc_tclSimuMode*>(_cpoMain->getHandler("I_hc_tclSimuMode"));
   // Instantiate all emergency function objects
   m_apoFunction[enEmergencyFunctionID_SystemShutdown] = new hc_tclEmergencyFunctionSystemShutdown();
   m_apoFunction[enEmergencyFunctionID_BacklightOFF] = new hc_tclEmergencyFunctionBacklightOFF();
   m_apoFunction[enEmergencyFunctionID_Attenuation] = new hc_tclEmergencyFunctionAttenuation();
   m_apoFunction[enEmergencyFunctionID_CDDriveOff] = new hc_tclEmergencyFunctionCDDriveOFF();
   m_apoFunction[enEmergencyFunctionID_DVDDriveOff] = new hc_tclEmergencyFunctionDVDDriveOFF();
   m_apoFunction[enEmergencyFunctionID_TVModuleOff] = new hc_tclEmergencyFunctionTVModuleOFF();
}

/*******************************************************************************
*
* FUNCTION: 	tVoid vStartCommunication()
*
* DESCRIPTION: 	Function to start all dynamic objects e.g. threads, ...
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vStartCommunication(tVoid)
{
   ETG_TRACE_USR4(("vStartCommunication() entered."));
   HC_NULL_POINTER_CHECK(hc_tclAppMain::theServer());
   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgUpdateFunctionState_SrvHdl);
   for (tU8 _u8Index = 0; (tU8)enEmergencyFunctionID_MAX > _u8Index; ++_u8Index)
   {
	   tenEmergencyFunctionID _enFunctionID = (tenEmergencyFunctionID)_u8Index;
	   hc_tclEmergencyFunction* _poFunction = poGetFunction(_enFunctionID);
	   if (OSAL_NULL != _poFunction)
	   {
		   m_amsgUpdateFunctionState_Master[_enFunctionID].u.tEmergencyFunctionState.enFunctionState = _poFunction->enGetFunctionState();
		   // send initial message to emergency function master
		   hc_tclAppMain::theServer()->vPostInternalMessage(&m_amsgUpdateFunctionState_Master[_enFunctionID]);
		   // create emergency function delay timer
		   if (OSAL_OK != OSAL_s32TimerCreate( (OSAL_tpfCallback)cb_vTimerEmergencyFunction,
				   (tVoid*)&m_amsgUpdateFunctionState_Master[_enFunctionID],
				   &m_ahDelayTimer[_enFunctionID]))
		   {
			   ETG_TRACE_FATAL(("vStartCommunication: couldn't create delay timer for emergency function %d.", _enFunctionID ));
		   }
		   else
			   (tVoid)OSAL_s32TimerSetTime(m_ahDelayTimer[_enFunctionID], 0, 0);
	   }
   }
}

/*******************************************************************************
*
* FUNCTION: 	vTraceInfo()
*
* DESCRIPTION: 	Trace information
*
* PARAMETER: 	None.
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vTraceInfo()
{
}

/*******************************************************************************
*
* FUNCTION: 	enGetFunctionState()
*
* DESCRIPTION: 	returns the current function state
*
* PARAMETER: 	tenEmergencyFunctionID enFunctionID
*
* RETURNVALUE: 	tenEmergencyFunctionID.
*
*******************************************************************************/
tenEmergencyFunctionState hc_tclEmergencyFunctionControl::enGetFunctionState(tenEmergencyFunctionID enFunctionID) const
{
	 if (enEmergencyFunctionID_MAX > enFunctionID && OSAL_NULL != m_apoFunction[enFunctionID])
		 return m_apoFunction[enFunctionID]->enGetFunctionState();
	 return enEmergencyFunctionState_DISABLED;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleMessage_NewEmergencyFunctionInfo()
*
* DESCRIPTION: 	handler for internal messages
*
* PARAMETER:   	hc_tclBaseIf::TMsg* pMsg: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleMessage_NewEmergencyFunctionInfo(hc_tclBaseIf::TMsg &Msg)
{
   ETG_TRACE_USR4(("vHandleMessage_NewEmergencyFunctionInfo() entered %u -> data: %d.", ETG_ENUM(HC_INT_MSG , (tU32)Msg.eCmd), Msg.u.u32Data));
   HC_NULL_POINTER_CHECK(hc_tclAppMain::theServer());
   tBool _bSrvHdlUpdateDone = FALSE;
   for (tU8 _u8Index = 0; (tU8)enEmergencyFunctionID_MAX > _u8Index; ++_u8Index)
   {
	   hc_tclEmergencyFunction* _poFunction = poGetFunction((tenEmergencyFunctionID)_u8Index);
	   tBool _bFunctionStateChanged = FALSE;
	   if (OSAL_NULL != _poFunction)
	   {
		   _poFunction->vHandleMessage_NewEmergencyFunctionInfo(Msg, _bFunctionStateChanged);
		   // start delay timer or inform directly the specified emergency function master
		   if (TRUE == _bFunctionStateChanged)
		   {
			   // trigger service handler update at first
			   if (FALSE == _bSrvHdlUpdateDone)
			   {
				   hc_tclAppMain::theServer()->vPostInternalMessage(&m_msgUpdateFunctionState_SrvHdl);
				   _bSrvHdlUpdateDone = TRUE;
			   }
			   m_amsgUpdateFunctionState_Master[_u8Index].u.tEmergencyFunctionState.enFunctionState = _poFunction->enGetFunctionState();
			   if (enEmergencyFunctionState_Active <= _poFunction->enGetFunctionState()
				   && enEmergencyFunctionState_ActiveLow >= _poFunction->enGetFunctionState())
			   {
				   if ((0 < _poFunction->u32GetConfigDelayTime()) && (OSAL_NULL != m_ahDelayTimer[_u8Index]))
					   // start delay timer of emergency function
					   (tVoid)OSAL_s32TimerSetTime(m_ahDelayTimer[_u8Index], _poFunction->u32GetConfigDelayTime(), 0);
				   else
					   // send emergency function state update to specified master
					   hc_tclAppMain::theServer()->vPostInternalMessage(&m_amsgUpdateFunctionState_Master[_u8Index]);
			   }
			   else
			   {
				   if ((0 < _poFunction->u32GetConfigDelayTime()) && (OSAL_NULL != m_ahDelayTimer[_u8Index]))
					   // stop delay timer of emergency function
					   (tVoid)OSAL_s32TimerSetTime(m_ahDelayTimer[_u8Index], 0, 0);
				   // send emergency function state update to specified master
				   hc_tclAppMain::theServer()->vPostInternalMessage(&m_amsgUpdateFunctionState_Master[_u8Index]);
			   }
		   }
	   }
   }
}

/*******************************************************************************
*
* FUNCTION: 	cb_vTimerEmergencyFunction()
*
* DESCRIPTION: 	delay timer callback to inform emergency function master
*
* PARAMETER: 	tVoid* pArg ... hc_tclBaseIf::TMsg*
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
OSAL_tpfCallback hc_tclEmergencyFunctionControl::cb_vTimerEmergencyFunction(tVoid* pArg)
{
   ETG_TRACE_USR4(("cb_vTimerEmergencyFunction(pArg 0x%08x) entered.", pArg ));

   hc_tclBaseIf::TMsg* _pMsg = (hc_tclBaseIf::TMsg*)pArg;

   if (OSAL_NULL != _pMsg && OSAL_NULL != hc_tclAppMain::theServer())
	   hc_tclAppMain::theServer()->vPostInternalMessage(_pMsg);

   return OSAL_NULL;
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_GetEFState()
*
* DESCRIPTION: 	handler for trace command HC_GetEmergencyFunctionState
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleTraceCmd_GetEFState(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_GetEFState() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];

   ETG_TRACE_FATAL(("%26s: HC_GetEmergencyFunctionState %u",
		   "HC.APPL.EF__.CMD__.START",
		   ETG_ENUM(HC_EFID, param1) ));

   tenEmergencyFunctionID _enFuctID = (tenEmergencyFunctionID)param1;
   hc_tclEmergencyFunction* _poFunct = poGetFunction(_enFuctID);
   if (enEmergencyFunctionID_MAX == _enFuctID || enEmergencyFunctionID_UNDEF == _enFuctID)
   {
	   for (tU8 _u8ID = (tU8)enEmergencyFunctionID_SystemShutdown; (tU8)enEmergencyFunctionID_MAX > _u8ID; ++_u8ID)
	   {
		   _enFuctID = (tenEmergencyFunctionID)_u8ID;
		   _poFunct = poGetFunction(_enFuctID);
		   if (OSAL_NULL != _poFunct)
			   _poFunct->vTraceInfo();
	   }
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.EF__.CMD__.RES__" ));
   }
   else if (OSAL_NULL != _poFunct)
   {
	   _poFunct->vTraceInfo();
	   ETG_TRACE_FATAL(("%26s: finished without errors",
			   "HC.APPL.EF__.CMD__.RES__" ));
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid function id %d",
			   "HC.APPL.EF__.CMD__.RES__",
			   param1 ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigEFEnable()
*
* DESCRIPTION: 	handler for trace command HC_ConfigEmergencyFunction_Enable
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleTraceCmd_ConfigEFEnable(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigEFEnable() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];
   tU8 param2 = puchData[4];
   tU8 param3 = puchData[5];

   ETG_TRACE_FATAL(("%26s: HC_ConfigEmergencyFunction_Enable %u %u %u ",
		   "HC.APPL.EF__.CMD__.START",
		   ETG_ENUM(HC_EFID, param1),
		   ETG_ENUM(HC_Persist, param2),
		   ETG_ENUM(HC_Bool, param3) ));

   tenEmergencyFunctionID _enFuctID = (tenEmergencyFunctionID)param1;
   hc_tclEmergencyFunction* _poFunct = poGetFunction(_enFuctID);

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   if (OSAL_NULL != _poFunct)
	   {
		   tBool _bResult = _poFunct->bSetConfigEnable((tBool)param3);
		   if (TRUE == (tBool)param2)
			   _poFunct->bSetConfigEnablePers((tBool)param3);
		   _poFunct->vTraceInfo();
		   if (TRUE == _bResult)
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid function id %d",
				   "HC.APPL.EF__.CMD__.RES__",
				   param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.EF__.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigEFEnableReset()
*
* DESCRIPTION: 	handler for trace command HC_ConfigEmergencyFunction_Enable_Reset
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleTraceCmd_ConfigEFEnableReset(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigEFEnableReset() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];

   ETG_TRACE_FATAL(("%26s: HC_ConfigEmergencyFunction_Enable_Reset %u ",
		   "HC.APPL.EF__.CMD__.START",
		   ETG_ENUM(HC_EFID, param1) ));

   tenEmergencyFunctionID _enFuctID = (tenEmergencyFunctionID)param1;
   hc_tclEmergencyFunction* _poFunct = poGetFunction(_enFuctID);

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   if (OSAL_NULL != _poFunct)
	   {
		   if (TRUE == _poFunct->bResetConfigEnablePers())
		   {
			   _poFunct->vTraceInfo();
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid function id %d",
				   "HC.APPL.EF__.CMD__.RES__",
				   param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.EF__.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigEFDelayTime()
*
* DESCRIPTION: 	handler for trace command HC_ConfigEmergencyFunction_DelayTime
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleTraceCmd_ConfigEFDelayTime(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigEFDelayTime() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];
   tU8 param2 = puchData[4];
   tU8 param3 = puchData[5];

   ETG_TRACE_FATAL(("%26s: HC_ConfigEmergencyFunction_DelayTime %u %u %d ",
		   "HC.APPL.EF__.CMD__.START",
		   ETG_ENUM(HC_EFID, param1),
		   ETG_ENUM(HC_Persist, param2),
		   param3 ));

   tenEmergencyFunctionID _enFuctID = (tenEmergencyFunctionID)param1;
   hc_tclEmergencyFunction* _poFunct = poGetFunction(_enFuctID);

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   if (OSAL_NULL != _poFunct)
	   {
		   tBool _bResult = _poFunct->bSetConfigDelayTime(param3);
		   if (TRUE == (tBool)param2)
			   _poFunct->bSetConfigDelayTimePers(param3);
		   _poFunct->vTraceInfo();
		   if (TRUE == _bResult)
		   {
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid function id %d",
				   "HC.APPL.EF__.CMD__.RES__",
				   param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.EF__.CMD__.RES__" ));
   }
}

/*******************************************************************************
*
* FUNCTION: 	vHandleTraceCmd_ConfigEFDelayTimeReset()
*
* DESCRIPTION: 	handler for trace command HC_ConfigEmergencyFunction_DelayTime_Reset
*
* PARAMETER:   	const tUChar* puchData: reference of received message
*
* RETURNVALUE: 	None.
*
*******************************************************************************/
tVoid hc_tclEmergencyFunctionControl::vHandleTraceCmd_ConfigEFDelayTimeReset(const tUChar* puchData)
{
   ETG_TRACE_USR4(("vHandleTraceCmd_ConfigEFDelayTimeReset() entered (data: 0x%08x).", puchData ));
   HC_NULL_POINTER_CHECK(puchData);

   //tU32 param1 = ((tU32)puchData[3]<<24) | ((tU32)puchData[4]<<16) | ((tU32)puchData[5] << 8) | (tU32)puchData[6];
   tU8 param1 = puchData[3];

   ETG_TRACE_FATAL(("%26s: HC_ConfigEmergencyFunction_DelayTime_Reset %u ",
		   "HC.APPL.EF__.CMD__.START",
		   ETG_ENUM(HC_EFID, param1) ));

   tenEmergencyFunctionID _enFuctID = (tenEmergencyFunctionID)param1;
   hc_tclEmergencyFunction* _poFunct = poGetFunction(_enFuctID);

   if (OSAL_NULL != m_poSimuMode && TRUE == m_poSimuMode->bGetSimuMode())
   {
	   if (OSAL_NULL != _poFunct)
	   {
		   if (TRUE == _poFunct->bResetConfigDelayTimePers())
		   {
			   _poFunct->vTraceInfo();
			   ETG_TRACE_FATAL(("%26s: finished without errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("%26s: finished with errors",
					   "HC.APPL.EF__.CMD__.RES__" ));
		   }
	   }
	   else
	   {
		   ETG_TRACE_FATAL(("%26s: finished with errors ... invalid function id %d",
				   "HC.APPL.EF__.CMD__.RES__",
				   param1 ));
	   }
   }
   else
   {
	   ETG_TRACE_FATAL(("%26s: finished with errors ... simulation mode must be active!",
			   "HC.APPL.EF__.CMD__.RES__" ));
   }
}
