/************************************************************************
* FILE:        hc_tclEmergencyFunction.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: base class of all emergency functions
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 11.03.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

#include "hc_tclEmergencyFunction.h"
#include "hc_tclThermalSensorControl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclEmergencyFunction.cpp.trc.h"
#endif

// names of emergency function IDs
const char* m_strEmergencyFunctionID[enEmergencyFunctionID_MAX] = {
		"UNDEF",
		"SystemShutdown",
		"BacklightOff",
		"Attenuation",
		"CDDriveOff",
		"DVDDriveOff",
		"TVModuleOff"
};

const char* strGetEmergencyFunctionID(tenEmergencyFunctionID enFunctionID)
{
	if (enEmergencyFunctionID_MAX > enFunctionID)
		return m_strEmergencyFunctionID[enFunctionID];
	return m_strEmergencyFunctionID[enEmergencyFunctionID_UNDEF];
};

// abbreviation of emergency function
const char* m_strEmergencyFunctionAbbreviation[enEmergencyFunctionID_MAX] = {
		"UNDEF",
		"SYSS_",
		"BLOFF",
		"ATTEN",
		"CDOFF",
		"DVOFF",
		"TVOFF"
};
const char* strGetEmergencyFunctionAbbreviation(tenEmergencyFunctionID enFunctionID)
{
	if (enEmergencyFunctionID_MAX > enFunctionID)
		return m_strEmergencyFunctionAbbreviation[enFunctionID];
	return m_strEmergencyFunctionAbbreviation[enEmergencyFunctionID_UNDEF];
};

// names of emergency function states
const char* m_strEmergencyFunctionState[enEmergencyFunctionState_MAX] = {
		"DISABLED",
		"NoResult",
		"Inactive",
		"Active",
		"ActiveHigh",
		"ActiveLow"
};

const char* strGetEmergencyFunctionState(tenEmergencyFunctionState enFunctionState)
{
	if (enEmergencyFunctionState_MAX > enFunctionState)
		return m_strEmergencyFunctionState[enFunctionState];
	return m_strEmergencyFunctionState[enEmergencyFunctionState_DISABLED];
}

/*******************************************************************************
*
* FUNCTION:    hc_tclEmergencyFunction()
*
* DESCRIPTION: constructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclEmergencyFunction::hc_tclEmergencyFunction()
: hc_tclEmergencyFunction_Config()
{
   ETG_TRACE_USR4(("hc_tclEmergencyFunction() entered."));

   m_enFunctionID = enEmergencyFunctionID_UNDEF;
   m_enFunctionState = enEmergencyFunctionState_DISABLED;
   (tVoid)memset((tVoid*)m_aenFunctionState, OSAL_NULL, sizeof(m_aenFunctionState));
}

/*******************************************************************************
*
* FUNCTION:    hc_tclEmergencyFunction()
*
* DESCRIPTION: constructor
*
* PARAMETER:   tenEmergencyFunctionID enFunctionID: own emergency function ID
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclEmergencyFunction::hc_tclEmergencyFunction(tenEmergencyFunctionID enFunctionID)
{
   ETG_TRACE_USR4(("hc_tclEmergencyFunction(id: %d) entered.", enFunctionID ));

   m_enFunctionID = enFunctionID;
   m_enFunctionState = enEmergencyFunctionState_DISABLED;
   (tVoid)memset((tVoid*)m_aenFunctionState, OSAL_NULL, sizeof(m_aenFunctionState));
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclEmergencyFunction()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclEmergencyFunction::~hc_tclEmergencyFunction()
{
   ETG_TRACE_USR4(("~hc_tclEmergencyFunction() entered."));
   (tVoid)memset((tVoid*)m_aenFunctionState, OSAL_NULL, sizeof(m_aenFunctionState));
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION: trace status information
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclEmergencyFunction::vTraceInfo()
{
   ETG_TRACE_USR4(("vTraceInfo() entered."));
   tChar _strIdentStart[30] = "";
   sprintf(_strIdentStart, "HC.APPL.EF__.%5s.START", strGetEmergencyFunctionAbbreviation(enGetFunctionID()));
   tChar _strIdentAttr[30] = "";
   sprintf(_strIdentAttr, "HC.APPL.EF__.%5s.ATTR_", strGetEmergencyFunctionAbbreviation(enGetFunctionID()));
   tChar _strIdentEnd[30] = "";
   sprintf(_strIdentEnd, "HC.APPL.EF__.%5s.END__", strGetEmergencyFunctionAbbreviation(enGetFunctionID()));

   // trace starting of trace block
   ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
   vTraceInfo_EF(_strIdentAttr);
   vTraceInfo_Config(_strIdentAttr);
   // trace ending of trace block
   ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION:    hc_tclEmergencyFunction()
*
* DESCRIPTION: emergency function state machine
*
* PARAMETER:   tEmergencyFunctionInfo_Internal &tEmergencyFunctionInfo
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclEmergencyFunction::vEmergencyFunction_SM(const tEmergencyFunctionInfo_Internal &tEmergencyFunctionInfo, tBool &bStateChanged)
{
   ETG_TRACE_USR4(("vEmergencyFunction_SM() entered."));
   if (FALSE == bGetConfigEnable())
	   return;
   tenEmergencyFunctionState _tFuncStateSensor_Old = enGetFunctionState(tEmergencyFunctionInfo.enSensorID);
   tenEmergencyFunctionState _tFuncStateSensor_New = tEmergencyFunctionInfo.aenFunctionState[enGetFunctionID()];
   if ((_tFuncStateSensor_Old != _tFuncStateSensor_New) && (TRUE == bSetFunctionState(tEmergencyFunctionInfo.enSensorID, _tFuncStateSensor_New)))
   {
	   tenEmergencyFunctionState _enFunctionState_New = enEmergencyFunctionState_DISABLED;
	   tU32 _u32ActiveSensorMask = 0;
	   for (tU8 _u8Index = 0; (tU8)enThermalSensorID_MAX > _u8Index; ++_u8Index)
	   {
		   if (enGetFunctionState((tenThermalSensorID)_u8Index) > _enFunctionState_New)
		   {
			   _enFunctionState_New = enGetFunctionState((tenThermalSensorID)_u8Index);
		   }
		   if (enGetFunctionState((tenThermalSensorID)_u8Index) >= enEmergencyFunctionState_Active
			   && enGetFunctionState((tenThermalSensorID)_u8Index) <= enEmergencyFunctionState_ActiveLow)
		   {
			   _u32ActiveSensorMask |= (1 << _u8Index);
		   }
	   }
	   if (enGetFunctionState() != _enFunctionState_New)
	   {
		   bSetFunctionState(_enFunctionState_New);
	   }
	   bStateChanged = TRUE;
	   // make EM_TRACE entry during activation
	   if (enGetFunctionState(tEmergencyFunctionInfo.enSensorID) >= enEmergencyFunctionState_Active
		   && enGetFunctionState(tEmergencyFunctionInfo.enSensorID) <= enEmergencyFunctionState_ActiveLow)
	   {
		   vMakeEMTRACEEmergencyFunction(tEmergencyFunctionInfo.enSensorID, _u32ActiveSensorMask);
	   }
   }
   ETG_TRACE_USR4(("vEmergencyFunction_SM(%d) exit.", bStateChanged ));
}

/*******************************************************************************
*
* FUNCTION:    vMakeEMTRACEEmergencyFunction()
*
* DESCRIPTION: handle new emergency function information
*
* PARAMETER:   tenThermalSensorID enSensorID
*              tU32 u32SensorMask
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclEmergencyFunction::vMakeEMTRACEEmergencyFunction(tenThermalSensorID enSensorID, tU32 u32SensorMask) const
{
   ETG_TRACE_USR4(("vMakeEMTRACEEmergencyFunction(sensor: %d) entered.", enSensorID ));

   hc_tclAppMain* _poApp = hc_tclAppMain::theServer();
   HC_NULL_POINTER_CHECK(_poApp);
   hc_tclThermalSensorControl* _poThermalSensorControl = dynamic_cast<hc_tclThermalSensorControl*>(_poApp->getHandler("I_hc_tclThermalSensorControl"));
   HC_NULL_POINTER_CHECK(_poThermalSensorControl);

   ETG_TRACE_ERRMEM(("HEATCTRL: emergency function %20s of sensor %20s with thermal state %15s and temperature %ddC is %20s! (sensor mask=0x%04x)",
				  strGetEmergencyFunctionID(enGetFunctionID()),
				  strGetThermalSensorID(enSensorID),
				  strGetThermalState(_poThermalSensorControl->enGetSensorState(enSensorID)),
				  _poThermalSensorControl->tGetTemperature(enSensorID),
				  strGetEmergencyFunctionState(enGetFunctionState()),
				  u32SensorMask ));
}

/*******************************************************************************
*
* FUNCTION:    vHandleMessage_NewEmergencyFunctionInfo()
*
* DESCRIPTION: handle new emergency function information
*
* PARAMETER:   hc_tclBaseIf::TMsg &Msg
*
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclEmergencyFunction::vHandleMessage_NewEmergencyFunctionInfo(hc_tclBaseIf::TMsg &Msg, tBool &bStateChanged)
{
   ETG_TRACE_USR4(("vHandleMessage_NewEmergencyFunctionInfo(sensor: %d) entered.", Msg.u.tEmergencyFunctionInfo.enSensorID ));
   vEmergencyFunction_SM(Msg.u.tEmergencyFunctionInfo, bStateChanged);
   if (TRUE == bStateChanged)
	   vTraceInfo();
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo_EF()
*
* DESCRIPTION: trace status information of emergency function
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: none
*
*******************************************************************************/
tVoid hc_tclEmergencyFunction::vTraceInfo_EF(const char* strIdentifier)
{
   ETG_TRACE_USR4(("vTraceInfo_EF() entered."));

   ETG_TRACE_USR2(("%26s: %30s=%25s ",
		   strIdentifier,
		   "                NAME",
		   strGetEmergencyFunctionID(enGetFunctionID()) ));
   ETG_TRACE_USR2(("%26s: %30s=%d ",
		   strIdentifier,
		   "                  ID",
		   enGetFunctionID() ));
   ETG_TRACE_USR2(("%26s: %30s=%25s ",
		   strIdentifier,
		   "               STATE",
		   strGetEmergencyFunctionState(enGetFunctionState()) ));
   ETG_TRACE_USR2(("%26s: %30s=%25s ",
		   strIdentifier,
		   "               STATE",
		   strGetEmergencyFunctionState(enGetFunctionState()) ));
   for (tU8 _u8Index = (tU8)enThermalSensorID_V850; (tU8)enThermalSensorID_MAX > _u8Index; ++_u8Index)
   {
	   tChar _strAttrName[20] = "";
	   (tVoid)sprintf(_strAttrName, "  TSEN_%5s_", strGetThermalSensorAbbreviation((tenThermalSensorID)_u8Index));

	   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ",
			   strIdentifier,
			   _strAttrName,
			   "___NAME",
			   strGetThermalSensorID((tenThermalSensorID)_u8Index) ));
	   ETG_TRACE_USR2(("%26s: %20s%10s=%25s ",
			   strIdentifier,
			   _strAttrName,
			   "EFSTATE",
			   strGetEmergencyFunctionState(enGetFunctionState((tenThermalSensorID)_u8Index)) ));
   }
}
