/************************************************************************
* FILE:        hc_tclCfgMgr_PDDTHM.cpp
* PROJECT:
* SW-COMPONENT:fc_heatctrl
*----------------------------------------------------------------------
*
* DESCRIPTION: implementation of common PDD thermalManagment class
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author             | Modification
* 24.06.2015| CM-AI/EPB2 Bernard | init
*
*************************************************************************/

/*******************************************************************************
| includes: system- and project- includes
|-----------------------------------------------------------------------------*/
// Basic OSAL includes
#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#define AHL_S_IMPORT_INTERFACE_GENERIC
#include "ahl_if.h"         // use Application Help Library

#define ETRACE_S_IMPORT_INTERFACE_GENERIC
#include "etrace_if.h"

/*******************************************************************************
| set own module id
|-----------------------------------------------------------------------------*/

/*******************************************************************************
| includes: needed interfaces from external components
|-----------------------------------------------------------------------------*/


/*******************************************************************************
| includes: internal and external interfaces from this component
|-----------------------------------------------------------------------------*/

#include "hc_tclCfgMgr_PDDTHM.h"
#include "hc_AppMain.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_FC_HEATCTRL_APPLICATION
#include "trcGenProj/Header/hc_tclCfgMgr_PDDTHM.cpp.trc.h"
#endif

const thermal_management_inc_fi::EModuleID m_enPDDModulID[enKDSDataSensorID_MAX] = {
		// enKDSDataSensorID_UNDEF = 0,
		(thermal_management_inc_fi::EModuleID)0,
		// enKDSDataSensorID_SCC,
		(thermal_management_inc_fi::EModuleID)0,
		// enKDSDataSensorID_Panel,
		thermal_management_inc_fi::eDisplay,
		// enKDSDataSensorID_CDDrive,
		thermal_management_inc_fi::eCDDRIVE,
		// enKDSDataSensorID_DVDDrive,
		thermal_management_inc_fi::eDVDDRIVE,
		// enKDSDataSensorID_GNSS,
		thermal_management_inc_fi::eGPS,
		// enKDSDataSensorID_Gyro,
		thermal_management_inc_fi::eGYRO,
		// enKDSDataSensorID_eMMC,
		thermal_management_inc_fi::eEMMC,
		// enKDSDataSensorID_Amplifier,
		thermal_management_inc_fi::ePower_Amp,
		// enKDSDataSensorID_ApplicationProcessor,
		thermal_management_inc_fi::eIMxIntel,
		// enKDSDataSensorID_HeatSink,
		thermal_management_inc_fi::eAmpHeatSink,
		// enKDSDataSensorID_Blueray,
		thermal_management_inc_fi::eBlueRayDrive,
		// enKDSDataSensorID_XMTuner,
		thermal_management_inc_fi::eXM,
		// enKDSDataSensorID_TVModule,
		thermal_management_inc_fi::eTV,
		// enKDSDataSensorID_TVModuleSlave,
		thermal_management_inc_fi::eTV_SLAVE,
		//enKDSDataSensorID_Panel2,
		thermal_management_inc_fi::eDISPLAY_2
};

const tenThermalSensorID m_enSensorID[enKDSDataSensorID_MAX] = {
		// enKDSDataSensorID_UNDEF = 0,
		enThermalSensorID_UNDEF,
		// enKDSDataSensorID_SCC,
		enThermalSensorID_V850,
		// enKDSDataSensorID_Panel,
		enThermalSensorID_Panel,
		// enKDSDataSensorID_CDDrive,
		enThermalSensorID_CDDRIVE,
		// enKDSDataSensorID_DVDDrive,
		enThermalSensorID_DVDDRIVE,
		// enKDSDataSensorID_GNSS,
		enThermalSensorID_GPS,
		// enKDSDataSensorID_Gyro,
		enThermalSensorID_GYRO,
		// enKDSDataSensorID_eMMC,
		enThermalSensorID_EMMC,
		// enKDSDataSensorID_Amplifier,
		enThermalSensorID_AMP,
		// enKDSDataSensorID_ApplicationProcessor,
		enThermalSensorID_IMX,
		// enKDSDataSensorID_HeatSink,
		enThermalSensorID_HEATSINK,
		// enKDSDataSensorID_Blueray,
		enThermalSensorID_BLUERAY,
		// enKDSDataSensorID_XMTuner,
		enThermalSensorID_XMModule,
		// enKDSDataSensorID_TVModule,
		enThermalSensorID_TVModule,
		// enKDSDataSensorID_TVModuleSlave,
		enThermalSensorID_TVModuleSlave,
		// enKDSDataSensorID_Panel2,
		enThermalSensorID_Panel2
};

/*******************************************************************************
*
* FUNCTION:    hc_tclCfgMgr_PDDThresholdBase()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr_PDDThresholdBase::hc_tclCfgMgr_PDDThresholdBase()
{
   ETG_TRACE_USR4(("hc_tclCfgMgr_PDDThresholdBase() entered."));
   m_bKDSLoaded = FALSE;
   m_bPDDUpdated = FALSE;
   m_s16KDSThreshold = 0;
   m_u8KDSHysteresis = 0;
   m_u16KDSTimeDelay = 0;
}

/*******************************************************************************
*
* FUNCTION:    bIsKDSConfigValid()
*
* DESCRIPTION: validate KDS configuration data
*
* PARAMETER:   tenKDSDataSensor enSensorIdx
* 			   tenKDSDataSensorThreshold enThresholdIdx
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDThresholdBase::bIsKDSConfigValid(tenKDSDataSensor enSensorIdx, tenKDSDataSensorThreshold enThresholdIdx) const
{
   ETG_TRACE_USR4(("bIsKDSConfigValid(sensor idx: %d, threshold idx: %d) entered.", enSensorIdx, enThresholdIdx ));

   tBool _bReturn = TRUE;

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsConfigChanged()
*
* DESCRIPTION: check for changes between KDs and PDD co0nfiguration data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDThresholdBase::bIsConfigChanged() const
{
   ETG_TRACE_USR4(("bIsConfigChanged() entered."));

   if (FALSE == bIsKDSLoaded())
	   return FALSE;
   else if (s16GetKDSThreshold() != s16GetPDDThreshold())
	   return TRUE;
   else if (u8GetKDSHysteresis() != u8GetPDDHysteresis())
	   return TRUE;
   else if (u16GetKDSTimeDelay() != u16GetPDDTimeDelay())
	   return TRUE;
   else if (FALSE == bIsPDDInitialized())
	   return TRUE;
   else
	   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u8Deserialize()
*
* DESCRIPTION: deserialize of raw KDS data
*
* PARAMETER:   const tU8 au8KDSBuffer[]		KDS configuration data buffer
*              const tU8 u8Size				size of buffer
*              const tU8 u8StartIndex		current start index
*
* RETURNVALUE: tU8		next start index
*
*******************************************************************************/
tU8 hc_tclCfgMgr_PDDThresholdBase::u8Deserialize(const tU8 au8KDSBuffer[],
		const tU8 u8Size, const tU8 u8StartIndex)
{
   ETG_TRACE_USR4(("u8Deserialize(buffer: 0x%08x, start index: %d) entered.",
		   au8KDSBuffer, u8StartIndex ));
   tU16 _u16NextStartIndex = (tU16)u8StartIndex + (tU16)(sizeof(m_s16KDSThreshold) + sizeof(m_u8KDSHysteresis) + sizeof(m_u16KDSTimeDelay));
   m_bKDSLoaded = FALSE;
   if ((tU16)u8Size < _u16NextStartIndex)
   {
	   ETG_TRACE_FATAL(("u8Deserialize: invalid size of buffer! (size: %d, start index: %d, next start index: %d)",
			   u8Size, u8StartIndex, _u16NextStartIndex ));
   }
   else
   {
	   m_s16KDSThreshold = (tS16)((au8KDSBuffer[u8StartIndex + 0] << 8) | au8KDSBuffer[u8StartIndex + 1]);
	   m_u8KDSHysteresis = au8KDSBuffer[u8StartIndex + 2];
	   m_u16KDSTimeDelay = (tU16)((au8KDSBuffer[u8StartIndex + 3] << 8) | au8KDSBuffer[u8StartIndex + 4]);
	   m_bKDSLoaded = TRUE;
   }
   return (tU8)_u16NextStartIndex;
}

/*******************************************************************************
*
* FUNCTION:    bUpdatePDD()
*
* DESCRIPTION: update PDD configuration data
*
* PARAMETER:   tBool bAnyCase			TRUE - ignore all flags, just do it
*
* RETURNVALUE: None
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDThresholdBase::bUpdatePDD(tBool bAnyCase)
{
   ETG_TRACE_USR4(("bUpdatePDD() entered." ));

   if ((TRUE == bAnyCase) || (TRUE == bIsConfigChanged()))
   {
	   m_bPDDUpdated = TRUE;
	   // Set datapool values
	   if (FALSE == bSetPDDThreshold())
	   {
		   m_bPDDUpdated = FALSE;
		   ETG_TRACE_FATAL(("bUpdatePDD: PDD Threshold Temperature couldn't set!" ));
	   }
	   if (FALSE == bSetPDDHysteresis())
	   {
		   m_bPDDUpdated = FALSE;
		   ETG_TRACE_FATAL(("bUpdatePDD: PDD Hysteresis couldn't set!" ));
	   }
	   if (FALSE == bSetPDDTimeDelay())
	   {
		   m_bPDDUpdated = FALSE;
		   ETG_TRACE_FATAL(("bUpdatePDD: PDD Time Delay couldn't set!" ));
	   }
   }
   else if (FALSE == bIsConfigChanged())
   {
	   m_bPDDUpdated = TRUE;
   }
   return m_bPDDUpdated;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION: trace configuration data
*
* PARAMETER:   tenKDSDataSensor enSensorIdx
* 			   tenKDSDataSensorThreshold enThresholdIdx
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid hc_tclCfgMgr_PDDThresholdBase::vTraceInfo(tenKDSDataSensor enSensorIdx,
		tenKDSDataSensorThreshold enThresholdIdx, const tChar strIdentAttr[]) const
{
	ETG_TRACE_USR4(("vTraceInfo(sensor idx: %d, threshold idx: %d) entered.",
			enSensorIdx, enThresholdIdx ));

	tChar _strAttrName[35] = "";
	tChar _strTresholdName[enKDSDataSensorThreshold_MAX][3] = {"CH", "H_", "L_", "CL"};
	// ... Temperature
	sprintf(_strAttrName, "      %2s_Temp_%02d_KDS", _strTresholdName[enThresholdIdx], enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentAttr,
			_strAttrName, s16GetKDSThreshold() ));
	sprintf(_strAttrName, "      %2s_Temp_%02d_PDD", _strTresholdName[enThresholdIdx], enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentAttr,
			_strAttrName, s16GetPDDThreshold() ));
	// ... Hysteresis
	sprintf(_strAttrName, "      %2s_Hyst_%02d_KDS", _strTresholdName[enThresholdIdx], enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dK] ", strIdentAttr,
			_strAttrName, u8GetKDSHysteresis() ));
	sprintf(_strAttrName, "      %2s_Hyst_%02d_PDD", _strTresholdName[enThresholdIdx], enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dK] ", strIdentAttr,
			_strAttrName, u8GetPDDHysteresis() ));
	// ... TimeDelay
	sprintf(_strAttrName, "     %2s_Delay_%02d_KDS", _strTresholdName[enThresholdIdx], enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [s] ", strIdentAttr,
			_strAttrName, u16GetKDSTimeDelay() ));
	sprintf(_strAttrName, "     %2s_Delay_%02d_PDD", _strTresholdName[enThresholdIdx], enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [s] ", strIdentAttr,
			_strAttrName, u16GetPDDTimeDelay() ));
}

//******************************************************************************
// helper class for PDD sensor data
//******************************************************************************
/*******************************************************************************
*
* FUNCTION:    hc_tclCfgMgr_PDDThresholdBase()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr_PDDSensorConfigBase::hc_tclCfgMgr_PDDSensorConfigBase()
{
   ETG_TRACE_USR4(("hc_tclCfgMgr_PDDSensorConfigBase() entered."));
   (tVoid)memset((tVoid*)m_apoThresholds,OSAL_NULL, sizeof(m_apoThresholds));
   m_bKDSLoaded = FALSE;
   m_u8KDSSensorID = 0;
   m_s16KDSMaxTemp = 0;
   m_s16KDSMinTemp = 0;
   m_bPDDUpdated = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclCfgMgr_PDDThresholdBase()
*
* DESCRIPTION: destructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr_PDDSensorConfigBase::~hc_tclCfgMgr_PDDSensorConfigBase()
{
   ETG_TRACE_USR4(("~hc_tclCfgMgr_PDDSensorConfigBase() entered."));
   (tVoid)memset((tVoid*)m_apoThresholds,OSAL_NULL, sizeof(m_apoThresholds));
}

/*******************************************************************************
*
* FUNCTION:    bIsKDSConfigValid()
*
* DESCRIPTION: validate KDS configuration data
*
* PARAMETER:   tenKDSDataSensor enSensorIdx
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDSensorConfigBase::bIsKDSConfigValid(tenKDSDataSensor enSensorIdx) const
{
   ETG_TRACE_USR4(("bIsKDSConfigValid(sensor idx: %d) entered.", enSensorIdx));

   tBool _bReturn = TRUE;
   if (0 < u8GetKDSModulID())
   {
	   // sorted temperature list from higher to lower
	   tS16 _as16Temp[(tU32)enKDSDataSensorThreshold_MAX * 2] = {0};
	   // collect all thresholds
	   for (tU8 _u8Idx = 0; (tU8)enKDSDataSensorThreshold_MAX > _u8Idx; ++ _u8Idx)
	   {
		   hc_tclCfgMgr_PDDThresholdBase* _poThreshold = poGetThreshold((tenKDSDataSensorThreshold)_u8Idx);
		   if (OSAL_NULL != _poThreshold)
		   {
			   if (_u8Idx < (tU8)enKDSDataSensorThreshold_Low)
			   {
				   _as16Temp[(_u8Idx*2)] = _poThreshold->s16GetKDSThreshold();
				   _as16Temp[(_u8Idx*2) + 1] = _poThreshold->s16GetKDSThreshold() - (tS16)_poThreshold->u8GetKDSHysteresis() ;
			   }
			   else
			   {
				   _as16Temp[(_u8Idx*2)] = _poThreshold->s16GetKDSThreshold() + (tS16)_poThreshold->u8GetKDSHysteresis() ;
				   _as16Temp[(_u8Idx*2) + 1] = _poThreshold->s16GetKDSThreshold();
			   }
			   // validate threshold
			   if (FALSE == _poThreshold->bIsKDSConfigValid(enSensorIdx, (tenKDSDataSensorThreshold)_u8Idx))
			   {
				   _bReturn = FALSE;
				   ETG_TRACE_FATAL(("bIsKDSConfigValid: invalid threshold %d for sensor %d!", 0, enSensorIdx ));
			   }
		   }
	   }
	   // checking
	   if (s16GetKDSMaxTemp() <= _as16Temp[0]) // MAX ?? CH
	   {
		   _bReturn = FALSE;
		   ETG_TRACE_FATAL(("bIsKDSConfigValid: invalid threshold %d for sensor %d!", 0, enSensorIdx ));
	   }
	   for (tU8 _u8Idx = 0; ((sizeof(_as16Temp)/sizeof(_as16Temp[0]))-1) > _u8Idx; ++_u8Idx )
	   {
		   if ( _as16Temp[_u8Idx] < _as16Temp[_u8Idx + 1])
		   {
			   _bReturn = FALSE;
			   ETG_TRACE_FATAL(("bIsKDSConfigValid: invalid threshold %d for sensor %d!", 0, enSensorIdx ));
		   }
	   }
	   if (_as16Temp[ ((sizeof(_as16Temp)/sizeof(_as16Temp[0]))-1) ] <= s16GetKDSMinTemp()) // CL ?? MIN
	   {
		   _bReturn = FALSE;
		   ETG_TRACE_FATAL(("bIsKDSConfigValid: invalid threshold %d for sensor %d!", ((sizeof(_as16Temp)/sizeof(_as16Temp[0]))-1), enSensorIdx ));
	   }
   }
   else
   {

   }

   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsConfigChanged()
*
* DESCRIPTION: check for changes between KDS and PDD co0nfiguration data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDSensorConfigBase::bIsConfigChanged() const
{
   ETG_TRACE_USR4(("bIsConfigChanged() entered."));

   tBool _bIsThresholdChanged = FALSE;
   for (tU8 _u8Index = 0; ((tU8)enKDSDataSensorThreshold_MAX > _u8Index) && (FALSE == _bIsThresholdChanged); ++_u8Index)
   {
	   if (OSAL_NULL != m_apoThresholds[_u8Index])
		   _bIsThresholdChanged = m_apoThresholds[_u8Index]->bIsConfigChanged();
   }
   if (FALSE == bIsKDSLoaded())
	   return FALSE;
   else if (u8GetKDSModulID() != u8GetPDDModulID())
	   return TRUE;
   else if (s16GetKDSMaxTemp() != s16GetPDDMaxTemp())
	   return TRUE;
   else if (s16GetKDSMinTemp() != s16GetPDDMinTemp())
	   return TRUE;
   else if (TRUE == _bIsThresholdChanged)
	   return TRUE;
   else if (FALSE == bIsPDDInitialized())
	   return TRUE;
   else
	   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u8GetKDSModulID()
*
* DESCRIPTION: returns the ModulID of specified SensorID
*
* PARAMETER:   None
*
* RETURNVALUE: tU8
*
*******************************************************************************/
tU8 hc_tclCfgMgr_PDDSensorConfigBase::u8GetKDSModulID() const
{
   ETG_TRACE_USR4(("u8GetKDSModulID() entered."));

   if (u8GetKDSSensorID() < (tU8)enKDSDataSensorID_MAX)
	   return (tU8)m_enPDDModulID[u8GetKDSSensorID()];
   return 0;
}

/*******************************************************************************
*
* FUNCTION:    u8Deserialize()
*
* DESCRIPTION: deserialize of raw KDS data
*
* PARAMETER:   const tU8 au8KDSBuffer[]		KDS configuration data buffer
*              const tU8 u8Size				size of buffer
*              const tU8 u8StartIndex		current start index
*
* RETURNVALUE: tU8		next start index
*
*******************************************************************************/
tU8 hc_tclCfgMgr_PDDSensorConfigBase::u8Deserialize(const tU8 au8KDSBuffer[],
		const tU8 u8Size, const tU8 u8StartIndex)
{
   ETG_TRACE_USR4(("u8Deserialize(buffer: 0x%08x, start index: %d) entered.",
		   au8KDSBuffer, u8StartIndex ));
   tU16 _u16NextStartIndex = (tU16)u8StartIndex + (tU16)(sizeof(m_u8KDSSensorID) + sizeof(m_s16KDSMaxTemp) + sizeof(m_s16KDSMinTemp));
   tBool _bKDSLoaded = TRUE;
   m_bKDSLoaded = FALSE;
   if ((tU16)u8Size < _u16NextStartIndex)
   {
	   ETG_TRACE_FATAL(("u8Deserialize: invalid size of buffer! (size: %d, start index: %d, next start index: %d)",
			   u8Size, u8StartIndex, _u16NextStartIndex ));
   }
   else
   {
	   m_u8KDSSensorID = au8KDSBuffer[u8StartIndex + 0];
	   m_s16KDSMaxTemp = (tS16)((au8KDSBuffer[u8StartIndex + 1] << 8) | au8KDSBuffer[u8StartIndex + 2]);
	   m_s16KDSMinTemp = (tS16)((au8KDSBuffer[u8StartIndex + 3] << 8) | au8KDSBuffer[u8StartIndex + 4]);
	   for (tU8 _u8Index = 0; ((tU8)enKDSDataSensorThreshold_MAX > _u8Index) && (_u16NextStartIndex <= (tU16)u8Size) && (TRUE == _bKDSLoaded); ++_u8Index)
	   {
		   if(OSAL_NULL != m_apoThresholds[_u8Index])
		   {
			   _u16NextStartIndex = (tU16)m_apoThresholds[_u8Index]->u8Deserialize(au8KDSBuffer, u8Size, (tU8)_u16NextStartIndex);
			   _bKDSLoaded = m_apoThresholds[_u8Index]->bIsKDSLoaded();
		   }
		   else
		   {
			   _bKDSLoaded = FALSE;
		   }
	   }
	   if (_u16NextStartIndex <= u8Size && TRUE == _bKDSLoaded)
		   m_bKDSLoaded = TRUE;
   }
   return (tU8)_u16NextStartIndex;
}

/*******************************************************************************
*
* FUNCTION:    bUpdatePDD()
*
* DESCRIPTION: update PDD configuration data
*
* PARAMETER:   tBool bAnyCase			TRUE - ignore all flags, just do it
*
* RETURNVALUE: None
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDSensorConfigBase::bUpdatePDD(tBool bAnyCase)
{
   ETG_TRACE_USR4(("vTraceInfo() entered." ));

   if ((TRUE == bAnyCase) || (TRUE == bIsConfigChanged()))
   {
	   m_bPDDUpdated = TRUE;
	   // Set datapool values
	   if (FALSE == bSetPDDModulID())
	   {
		   m_bPDDUpdated = FALSE;
		   ETG_TRACE_FATAL(("bUpdatePDD: PDD SensorID couldn't set!" ));
	   }
	   if (FALSE == bSetPDDMaxTemp())
	   {
		   m_bPDDUpdated = FALSE;
		   ETG_TRACE_FATAL(("bUpdatePDD: PDD Maximum Temperature couldn't set!" ));
	   }
	   if (FALSE == bSetPDDMinTemp())
	   {
		   m_bPDDUpdated = FALSE;
		   ETG_TRACE_FATAL(("bUpdatePDD: PDD Minimum Temperature couldn't set!" ));
	   }
	   for (tU8 _u8Idx = 0; (tU8)enKDSDataSensorThreshold_MAX > _u8Idx; ++_u8Idx)
	   {
		   if (OSAL_NULL != m_apoThresholds[_u8Idx])
		   {
			   if (FALSE == m_apoThresholds[_u8Idx]->bUpdatePDD(bAnyCase))
			   {
				   m_bPDDUpdated = FALSE;
				   ETG_TRACE_FATAL(("bUpdatePDD: error detected during setting of Threshold %d!", _u8Idx ));
			   }
		   }
		   else
		   {
			   ETG_TRACE_FATAL(("bUpdatePDD: no Threshold object reference for index %d !", _u8Idx ));
		   }
	   }
   }
   else if (FALSE == bIsConfigChanged())
   {
	   m_bPDDUpdated = TRUE;
   }
   return m_bPDDUpdated;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION: trace configuration data
*
* PARAMETER:   tenKDSDataSensor enSensorIdx
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid hc_tclCfgMgr_PDDSensorConfigBase::vTraceInfo(tenKDSDataSensor enSensorIdx, const tChar strIdentAttr[]) const
{
	ETG_TRACE_USR4(("vTraceInfo(sensor idx: %d) entered.", enSensorIdx ));

	tChar _strAttrName[35] = "";
	// ... SensorID
	sprintf(_strAttrName, "     SensorID_%02d_KDS", enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d / ModuleID=%d ", strIdentAttr,
			_strAttrName, u8GetKDSSensorID(), u8GetKDSModulID() ));
	sprintf(_strAttrName, "     SensorID_%02d_PDD", enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d / ModuleID=%d", strIdentAttr,
			_strAttrName, u8GetPDDModulID(), u8GetPDDModulID() ));
	// ... MaximumTemperature
	sprintf(_strAttrName, "      MaxTemp_%02d_KDS", enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentAttr,
			_strAttrName, s16GetKDSMaxTemp() ));
	sprintf(_strAttrName, "      MaxTemp_%02d_PDD", enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentAttr,
			_strAttrName, s16GetPDDMaxTemp() ));
	// ... MinimumTemperature
	sprintf(_strAttrName, "      MinTemp_%02d_KDS", enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentAttr,
			_strAttrName, s16GetKDSMinTemp() ));
	sprintf(_strAttrName, "      MinTemp_%02d_PDD", enSensorIdx);
	ETG_TRACE_USR2(("%26s: %30s=%d [dC] ", strIdentAttr,
			_strAttrName, s16GetPDDMinTemp() ));
	// trace threshold configuration data
	for (tU8 _u8Idx = 0; (tU8)enKDSDataSensorThreshold_MAX > _u8Idx; ++_u8Idx)
	{
	   if (OSAL_NULL != m_apoThresholds[_u8Idx])
	   {
		   m_apoThresholds[_u8Idx]->vTraceInfo(enSensorIdx, (tenKDSDataSensorThreshold)_u8Idx, strIdentAttr);
	   }
	}
}

//******************************************************************************
// class for PDD configuration of thermal management
//******************************************************************************

/*******************************************************************************
*
* FUNCTION:    hc_tclCfgMgr_PDDTHM()
*
* DESCRIPTION: constructor
*
* PARAMETER:   None
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr_PDDTHMBase::hc_tclCfgMgr_PDDTHMBase()
{
   ETG_TRACE_USR4(("hc_tclCfgMgr_PDDTHM() entered."));
   (tVoid)memset((tVoid*)m_apoSensors,OSAL_NULL, sizeof(m_apoSensors));
   m_u8Version = 0;
   m_bKDSLoaded = FALSE;
   m_bPDDUpdated = FALSE;
}

/*******************************************************************************
*
* FUNCTION:    ~hc_tclCfgMgr_PDDTHM()
*
* DESCRIPTION: destructor
*
* PARAMETER:   none
*
* RETURNVALUE: none
*
*******************************************************************************/
hc_tclCfgMgr_PDDTHMBase::~hc_tclCfgMgr_PDDTHMBase()
{
   ETG_TRACE_USR4(("~hc_tclCfgMgr_PDDTHM() entered."));
   (tVoid)memset((tVoid*)m_apoSensors,OSAL_NULL, sizeof(m_apoSensors));
}

/*******************************************************************************
*
* FUNCTION:    bIsKDSConfigValid()
*
* DESCRIPTION: validate KDS configuration data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDTHMBase::bIsKDSConfigValid() const
{
   ETG_TRACE_USR4(("bIsKDSConfigValid() entered."));

   tBool _bReturn = TRUE;
   // check all sensor data
   for (tU8 _u8Index = 0; ((tU8)enKDSDataSensor_MAX > _u8Index); ++_u8Index)
   {
	   if (OSAL_NULL != m_apoSensors[_u8Index])
	   {
		   if (FALSE == m_apoSensors[_u8Index]->bIsKDSConfigValid((tenKDSDataSensor)_u8Index))
		   {
			   _bReturn = FALSE;
			   ETG_TRACE_FATAL(("bIsKDSConfigValid: KDS configuration for sensor %d is invalid!", _u8Index ));
		   }
	   }
   }
   return _bReturn;
}

/*******************************************************************************
*
* FUNCTION:    bIsConfigChanged()
*
* DESCRIPTION: check for changes between KDS and PDD configuration data
*
* PARAMETER:   None
*
* RETURNVALUE: tBool
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDTHMBase::bIsConfigChanged() const
{
   ETG_TRACE_USR4(("bIsConfigChanged() entered."));

   tBool _bIsSensorDataChanged = FALSE;
   for (tU8 _u8Index = 0; ((tU8)enKDSDataSensor_MAX > _u8Index) && (FALSE == _bIsSensorDataChanged); ++_u8Index)
   {
	   if (OSAL_NULL != m_apoSensors[_u8Index])
		   _bIsSensorDataChanged = m_apoSensors[_u8Index]->bIsConfigChanged();
   }
   if (FALSE == bIsKDSLoaded())
	   return FALSE;
   else if (u16GetKDSVersion() != u16GetPDDVersion())
	   return TRUE;
   else if (TRUE == _bIsSensorDataChanged)
	   return TRUE;
   else if (FALSE == bIsPDDInitialized())
	   return TRUE;
   else
	   return FALSE;
}

/*******************************************************************************
*
* FUNCTION:    u8Deserialize()
*
* DESCRIPTION: deserialize of raw KDS data
*
* PARAMETER:   const tU8 au8KDSBuffer[]		KDS configuration data buffer
*              const tU8 u8Size				size of buffer
*              const tU8 u8StartIndex		current start index
*
* RETURNVALUE: tU8		next start index
*
*******************************************************************************/
tU8 hc_tclCfgMgr_PDDTHMBase::u8Deserialize(const tU8 au8KDSBuffer[],
		const tU8 u8Size, const tU8 u8StartIndex)
{
   ETG_TRACE_USR4(("u8Deserialize(buffer: 0x%08x, start index: %d) entered.",
		   au8KDSBuffer, u8StartIndex ));
   tU16 _u16NextStartIndex = (tU16)u8StartIndex + (tU16)(sizeof(m_u8Version));
   tBool _bKDSLoaded = TRUE;
   m_bKDSLoaded = FALSE;
   if ((tU16)u8Size < _u16NextStartIndex)
   {
	   ETG_TRACE_FATAL(("u8Deserialize: invalid size of buffer! (size: %d, start index: %d, next start index: %d)",
			   u8Size, u8StartIndex, _u16NextStartIndex ));
   }
   else
   {
	   m_u8Version = au8KDSBuffer[u8StartIndex + 0];
	   for (tU8 _u8Index = 0; ((tU8)enKDSDataSensor_MAX > _u8Index) && (_u16NextStartIndex <= (tU16)u8Size) && (TRUE == _bKDSLoaded); ++_u8Index)
	   {
		   if(OSAL_NULL != m_apoSensors[_u8Index])
		   {
			   _u16NextStartIndex = (tU16)m_apoSensors[_u8Index]->u8Deserialize(au8KDSBuffer, u8Size, (tU8)_u16NextStartIndex);
			   _bKDSLoaded = m_apoSensors[_u8Index]->bIsKDSLoaded();
		   }
		   else
		   {
			   _bKDSLoaded = FALSE;
		   }
	   }
	   if (_u16NextStartIndex <= u8Size && TRUE == _bKDSLoaded)
		   m_bKDSLoaded = TRUE;
   }
   return (tU8)_u16NextStartIndex;
}

/*******************************************************************************
*
* FUNCTION:    bUpdatePDD()
*
* DESCRIPTION: update PDD configuration data
*
* PARAMETER:   tBool bAnyCase			TRUE - ignore all flags, just do it
*
* RETURNVALUE: None
*
*******************************************************************************/
tBool hc_tclCfgMgr_PDDTHMBase::bUpdatePDD(tBool bAnyCase)
{
   ETG_TRACE_USR4(("vTraceInfo() entered." ));

   if (FALSE == bIsKDSConfigValid())
   {
	   ETG_TRACE_FATAL(("bUpdatePDD: no update done for invalid configuration!" ));
   }
   else
   {
	   if ((TRUE == bAnyCase) || (TRUE == bIsConfigChanged()))
	   {
		   m_bPDDUpdated = TRUE;
		   // Set datapool values
		   if (FALSE == bSetPDDVersion())
		   {
			   m_bPDDUpdated = FALSE;
			   ETG_TRACE_FATAL(("bUpdatePDD: PDD Version couldn't set!" ));
		   }
		   for (tU8 _u8Idx = 0; (tU8)enKDSDataSensor_MAX > _u8Idx; ++_u8Idx)
		   {
			   if (OSAL_NULL != m_apoSensors[_u8Idx])
			   {
				   if (FALSE == m_apoSensors[_u8Idx]->bUpdatePDD(bAnyCase))
				   {
					   m_bPDDUpdated = FALSE;
					   ETG_TRACE_FATAL(("bUpdatePDD: error detected during setting of Sensor %d!", _u8Idx ));
				   }
			   }
			   else
			   {
				   ETG_TRACE_FATAL(("bUpdatePDD: no Sensor object reference for index %d !", _u8Idx ));
			   }
		   }
		   tU8 _u8Reserved = u8GetPDDReserved();
		   if (FALSE == bSetPDDReserved())
		   {
			   m_bPDDUpdated = FALSE;
			   ETG_TRACE_FATAL(("bUpdatePDD: PDD Reserved couldn't set!" ));
		   }
		   // PDD commit to finalize saving process / synchronize PDD with V850
		   dp_tclSrvIf _oDpSrvIf;
		   tU32 _u32DpID = DP_U32_POOL_ID_THMDPPDD;
		   tS32 _s32ErrorCode = _oDpSrvIf.s32StorePoolNow(_u32DpID);
		   if (DP_S32_NO_ERR != _s32ErrorCode)
		   {
			   ETG_TRACE_FATAL(("bUpdatePDD: storing of PDD 0x%08x failed with error code 0x%08x!", _u32DpID, _s32ErrorCode ));
		   }
		   ETG_TRACE_ERRMEM(( "HEATCTRL: PDD-THM written (counts %d -> %d)",
				   _u8Reserved, u8GetPDDReserved() ));
	   }
	   else if (FALSE == bIsConfigChanged())
	   {
		   ETG_TRACE_FATAL(("bUpdatePDD: no update done for identical configuration!" ));
		   m_bPDDUpdated = TRUE;
	   }
   }
   return m_bPDDUpdated;
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION: trace configuration data
*
* PARAMETER:   None
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid hc_tclCfgMgr_PDDTHMBase::vTraceInfo(tVoid) const
{
	ETG_TRACE_USR4(("vTraceInfo() entered." ));

	tChar _strIdentStart[30] = "HC.APPL.CFG_.MGR__.START";
	tChar _strIdentAttr[30] = "HC.APPL.CFG_.MGR__.ATTR_";
	tChar _strIdentEnd[30] = "HC.APPL.CFG_.MGR__.END__";

	ETG_TRACE_USR2(("%026s: ", _strIdentStart ));
	vTraceInfo(_strIdentAttr);
	ETG_TRACE_USR2(("%026s: ", _strIdentEnd ));
}

/*******************************************************************************
*
* FUNCTION:    vTraceInfo()
*
* DESCRIPTION: trace configuration data
*
* PARAMETER:   	const char* strIdentifier		trace identifier
*                                               like "HC_APPL.TSEN.GYRO_.ATTR_"
*
* RETURNVALUE: None
*
*******************************************************************************/
tVoid hc_tclCfgMgr_PDDTHMBase::vTraceInfo(const char* strIdentifier) const
{
	ETG_TRACE_USR4(("vTraceInfo() entered." ));

	// status info
	// ... KDS data loaded
	ETG_TRACE_USR2(("%26s: <<< PDD-THM configuration status >>> ", strIdentifier ));
	ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier,
			"          KDS_loaded", ETG_ENUM(HC_Bool, bIsKDSLoaded()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier,
			"           KDS_valid", ETG_ENUM(HC_Bool, bIsKDSConfigValid()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier,
			"     PDD_initialized", ETG_ENUM(HC_Bool, bIsPDDInitialized()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier,
			"     KDS_PDD_changed", ETG_ENUM(HC_Bool, bIsConfigChanged()) ));
	ETG_TRACE_USR2(("%26s: %30s=%u ", strIdentifier,
			"         PDD_updated", ETG_ENUM(HC_Bool, bIsPDDUpdated()) ));
	// ... Version
	ETG_TRACE_USR2(("%26s: <<< PDD-THM configuration data >>> ", strIdentifier ));
	ETG_TRACE_USR2(("%26s: %30s=%d / %04d", strIdentifier,
			"         VERSION_KDS", u8GetKDSVersion(), u16GetKDSVersion() ));
	ETG_TRACE_USR2(("%26s: %30s=%d ", strIdentifier,
			"         VERSION_PDD", u16GetPDDVersion() ));
	// trace sensor configuration data
	for (tU8 _u8Idx = 0; (tU8)enKDSDataSensor_MAX > _u8Idx; ++_u8Idx)
	{
	   if (OSAL_NULL != m_apoSensors[_u8Idx])
		   m_apoSensors[_u8Idx]->vTraceInfo((tenKDSDataSensor)_u8Idx, strIdentifier);
	}
	ETG_TRACE_USR2(("%26s: %30s=%d ", strIdentifier,
			"        RESERVED_PDD", u8GetPDDReserved() ));
}

/*******************************************************************************
*
* FUNCTION:    enConvert2KDSSensorID()
*
* DESCRIPTION: convert PDD Sensor ID to KDS Sensor ID
*
* PARAMETER:   tU8 u8PDDSensorID
*
* RETURNVALUE: tenKDSDataSensorID
*
*******************************************************************************/
tenKDSDataSensorID hc_tclCfgMgr_PDDTHMBase::enConvert2KDSSensorID(tU8 u8PDDSensorID) const
{
	ETG_TRACE_USR4(("enConvert2KDSSensorID(PDD sensor id: %d) entered.", u8PDDSensorID ));
	tenKDSDataSensorID _enReturn = enKDSDataSensorID_UNDEF;
	if (0 < u8PDDSensorID)
	{
		for (tU16 _u16Idx = 0; HC_C_U16_ARRAYELEMENTS(m_enPDDModulID) > _u16Idx; ++_u16Idx)
		{
			if ((tU8)m_enPDDModulID[_u16Idx] == u8PDDSensorID)
			{
				_enReturn = (tenKDSDataSensorID)_u16Idx;
				break;
			}
		}
	}
	return _enReturn;
}

/*******************************************************************************
*
* FUNCTION:    enConvert2SensorID()
*
* DESCRIPTION: convert PDD Sensor ID to KDS Sensor ID
*
* PARAMETER:   tU8 u8PDDSensorID
*
* RETURNVALUE: tenKDSDataSensorID
*
*******************************************************************************/
tenThermalSensorID hc_tclCfgMgr_PDDTHMBase::enConvert2SensorID(tenKDSDataSensorID enKDSSensorID) const
{
	ETG_TRACE_USR4(("enConvert2SensorID(KDS sensor id: %d) entered.", enKDSSensorID ));
	if (HC_C_U16_ARRAYELEMENTS(m_enSensorID) > (tU16)enKDSSensorID)
		return m_enSensorID[enKDSSensorID];
	return enThermalSensorID_UNDEF;
}
